#include "Gnutella.h"
#include "QueryHit.h"
#include "Errors.h"
#include "Message.h"
#include "StringManipulation.h"

#include <cctype>
#include <stdio.h>

static const unsigned queryHitOffsetTTL = 17;
static const unsigned queryHitOffsetHops = 18;

Q_UINT8 QueryHit::ttl() const
{
      Q_ASSERT(_data.size() > queryHitOffsetTTL);
      return static_cast<Q_UINT8>(_data[queryHitOffsetTTL]);
}

QueryHit::QueryHit(Q_UINT8 ttl, Q_UINT8 hops, const std::string id, const std::string data)
: _id(id), _valid(true)
{
      // We adjust ttl and hops here (when building the packet)
      if (ttl > 0) ttl--;
      if (hops < 254) hops++;
      //! Ensure that _data is big enough to reply to ttl()
      toStringStartOfPacket(id, ttl, hops); 

  if( data.size() < 11 ) 
    { 
      qDebug("invalid initial size");
      _valid = false;
      return;
    }

  std::string qh(data.data(), data.size());

  Q_UINT16 port  = Message::str_uint16(qh.substr(1, 2).data());
  Q_UINT32 ip    = Message::str_uint32(qh.substr(3, 4).data());

  _address = Address(ip, port);
  _hits  = *((Q_UINT8*)qh.substr(0, 1).data());
  _speed = Message::str_uint32(qh.substr(7, 4).data());

  qh.erase(0, 11);

  if( qh.size() < 16 )
    {
      qDebug("error");
      _valid = false;
      return;
    }

  _id_result = qh.substr(qh.size() - 16);
  qh.erase(qh.size() - 16, 16);

  std::string::size_type idx1;
  std::string::size_type idx2;

  for(int hits_counter = 0; hits_counter < _hits; hits_counter++)
    {
      Q_UINT32 index = Message::str_uint32(qh.substr(0, 4).data());
      Q_UINT32 fsize = Message::str_uint32(qh.substr(4, 4).data());

      qh.erase(0, 8);

      idx1 = qh.find( std::string("\0", 1) );
      
      if(idx1 == std::string::npos) // delimiter not found => clear rest of query hit
	{
	  qh.erase();
	  break;
	}

      idx2 = qh.find( std::string("\0", 1), idx1 + 1 );

      if(idx2 == std::string::npos) // delimiter not found => clear rest of query hit
	{
	  qh.erase();
	  break;
	}

      std::string fname = qh.substr(0, idx1);

      std::string additional;
      std::string sha1;

      if(idx2 > (idx1 + 1))
	{
	  StringManipulation sm(qh.substr( idx1 + 1, idx2 - idx1 - 1 ));
	  std::vector<std::string> extra_data;
	  sm.split_to_vector( extra_data, 0x1c );

	  for( int l = 0; l < extra_data.size(); ++l )
	    if( extra_data[l].find( "bps" ) != std::string::npos )
	      additional = extra_data[0];
	    else
	      if( extra_data[l].find( "urn:sha1:" ) != std::string::npos )
		sha1 = extra_data[l].substr( extra_data[l].find( ":sha1:" ) + 6 );
	}

      qh.erase(0, idx2 + 1);
      _results.push_back(QueryResult(index, fsize, fname, additional, sha1));
    }


  if(qh.size() > 0)  // extended query hit
    {
      _trailer = qh;
      //cout << _trailer << endl;
    }
  else
    _trailer = "";

  Q_UINT32 datasize = data.size();
  _data += Message::uint_str(datasize);
  _data += data;
}

QueryHit::QueryHit(const std::string header_id, const Q_UINT8 ttl, const Q_UINT8 hops, Address& address, const Q_UINT32 speed, const std::string result_id, std::vector<QueryResult>& results)
{
  _id = header_id;
  _hits = results.size();
  _address = address;
  _speed = speed;
  _id_result = result_id;
  _results = results;
  _trailer.assign( std::string("QTEL\2\0\0", 7) );
  _valid = true;

  toString(_id, ttl, hops);
}

void QueryHit::toStringStartOfPacket(const std::string &id, Q_UINT8 ttl, Q_UINT8 hops)
{
      Q_ASSERT(id.length() == 16);
      _data = id;

      _data += static_cast<unsigned char>(CMD_QUERYHIT);
      _data += static_cast<unsigned char>(ttl);
      _data += static_cast<unsigned char>(hops);
}

void QueryHit::toString(const std::string &id, Q_UINT8 ttl, Q_UINT8 hops)
{
  toStringStartOfPacket(id, ttl, hops);

  std::string result;

  for(int i = 0; i < _results.size(); ++i)
    {
      result.append( Message::uint_str(_results[i]._index) );
      result.append( Message::uint_str(_results[i]._filesize) );
      result.append( _results[i]._filename );
      result += '\0';
      const QueryResult &r = _results[i];
      if (r._sha1.length()) {
	    result += "urn:sha1:";
	    result += r._sha1;
      }
      result += '\0';
    }

  result.append(_trailer);
  result.append(_id_result);

  Q_UINT32 payload = result.size() + 1+2+4+4;
  Q_UINT16 port = _address.port();
  Q_UINT32 ip = _address.longIP();

  _data.append( Message::uint_str(payload) );

  _data.append( Message::uint_str(_hits) );
  _data.append( Message::uint_str(port) );
  _data.append( Message::uint_str(ip) );
  _data.append( Message::uint_str(_speed) );

  _data.append( result );
}


bool QueryHit::isValid()
{
  return _valid;
}
