#include "RiffFile.h"

#include <fstream>

#if !defined(_WIN32)
using namespace std;
#endif

/***************************************************************************
	macros and constants
***************************************************************************/

// define REVERSE_ENDIANISM if the endianism of the host platform is not Intel
// (Intel is little-endian)
#ifdef REVERSE_ENDIANISM
 #define SWAP_32(int32) (  \
	((((DWORD) int32) & 0x000000FFL) << 24) +  \
	((((DWORD) int32) & 0x0000FF00L) << 8) +  \
	((((DWORD) int32) & 0x00FF0000L) >> 8) +  \
	((((DWORD) int32) & 0xFF000000L) >> 24))
#endif

struct TypeRecord {
	char* typeName;  // four-letter name
	char* realName;  // English name
};

const int numExtraTypes = 24;
const TypeRecord extraTypes[numExtraTypes] = {
	{ "DISP", "Display name" },
	{ "IARL", "Archival location" },
	{ "IART", "Artist" },
	{ "ICMS", "Commissioned" },
	{ "ICMT", "Comments" },
	{ "ICOP", "Copyright" },
	{ "ICRD", "Creation date" },
	{ "ICRP", "Cropped" },
	{ "IDIM", "Dimensions" },
	{ "IDPI", "Dots Per Inch" },
	{ "IENG", "Engineer" },
	{ "IGNR", "Genre" },
	{ "IKEY", "Keywords" },
	{ "ILGT", "Lightness" },
	{ "IMED", "Medium" },
	{ "INAM", "Name" },
	{ "IPLT", "Palette Setting" },
	{ "IPRD", "Product" },
	{ "ISBJ", "Subject" },
	{ "ISFT", "Software" },
	{ "ISHP", "Sharpness" },
	{ "ISRC", "Source" },
	{ "ISRF", "Source Form" },
	{ "ITCH", "Technician" },
};

RiffFile::RiffFile(const char *name)
{
	file.open(name, (std::ios::in|std::ios::binary));

	if( !rewind() ) {
		file.close( );
	}
}

RiffFile::~RiffFile()
{
	if( !file.bad() )
		file.close( );
}

bool RiffFile::rewind()
{
	// clear the chunk stack
	while( !chunks.empty() )
		chunks.pop();

	// rewind to the start of the file
	if( !file.seekg(0) )
		return false;

	// look for a valid RIFF header
	RiffChunk topChunk(*this);

	if( file.eof() || strcmp(topChunk.name, "RIFF") )
		return false;

	// found; push it on the stack, and leave the put pointer in the same
	//  place as the get pointer.
	formSize = topChunk.size;
	chunks.push(topChunk);
	return true;
}

bool RiffFile::push(const char* chunkType)
{
	// can't descend if we haven't started out yet.
	if( chunks.empty() )
		return false;

	// first, go to the start of the current chunk, if we're looking for
	//  a named chunk.
	if( chunkType )
		if( !file.seekg(chunks.top().start, ios::beg) )
			return false;

	// read chunks until one matches or we exhaust this chunk
	while( !file.eof() && file.tellg() < chunks.top().after )
	{
		RiffChunk chunk(*this);

		if( !file.eof() )
		{
			// see if the subchunk type matches
			if( !chunkType || strcmp(chunk.name, chunkType) == 0 )
			{
				// found; synchronize the put pointer, push
				// the chunk, and succeed
				chunks.push(chunk);
				return true;
			} else {
				// not found; go to the next one.
				if( !file.seekg(chunk.after, ios::beg) )
					return false;
			}
		}
	}

	// couldn't find it; synchronize the put pointer and return error.
	file.seekg(chunks.top().start, ios::beg);
	return false;
}

bool RiffFile::pop()
{
	// if we've only got the top level chunk (or not even that), then we can't
	// go up.
	if( chunks.size() < 2 )
		return false;

	// Position the get and put pointers at the end of the current subchunk.
	file.seekg(chunks.top().after, ios::beg);

	// Pop up the stack.
	chunks.pop();
	return true;
}

long RiffFile::chunkSize() const
{
	if( !chunks.empty() )
		return chunks.top().size;
	else
		return 0;
}

const char* RiffFile::chunkName() const
{
	if( !chunks.empty() )
		return chunks.top().name;
	else
		return 0;
}

const char* RiffFile::subType() const
{
	if( !chunks.empty() && chunks.top().subType[0] )
		return chunks.top().subType;
	else
		return 0;
}

bool RiffFile::getNextExtraItem(int *type, std::string& value)
{
	// if the current chunk is LIST/INFO, then try to read another subchunk.
	if( strcmp(chunkName(), "LIST") == 0 && strcmp(subType(), "INFO") == 0 )
	{
		if( push() )
		{
			if( readExtraItem(type, value) )
				return true;
			else
				// unrecognized type.  Continue on.
				return getNextExtraItem(type, value);
		} else {
			// got to the end of the LIST/INFO chunk. Pop back out
			// and continue looking.
			pop();
			return getNextExtraItem(type, value);
		}
	// we're not in a LIST/INFO chunk, so look for the next DISP or
	// LIST/INFO.
	} else {
		push();

		if( strcmp(chunkName(), "DISP") == 0 )
		{
			// DISP chunk: read and pop back out.
			return readExtraItem(type, value);
		} else if( strcmp(chunkName(), "LIST") == 0
				&& strcmp(subType(), "INFO") == 0 )
		{
			// LIST/INFO chunk: read first element
			return getNextExtraItem(type, value);
		} else {
			// Some other chunk.  Pop back out and move on.
			if( pop() )
				return getNextExtraItem(type, value);
			else
				return false;
		}
	}
}

// Reads extra data from the current chunk, and pops out of it.
bool RiffFile::readExtraItem(int *type, std::string& value)
{
	// see if it's one we recognize
	bool found = false;
	for( int i = 0; i < numExtraTypes; i++ )
	{
		if( strcmp(chunkName(), extraTypes[i].typeName) == 0 )
		{
//			type = extraTypes[i].realName;
//			type = extraTypes[i].typeName;
			*type = i;
			found = true;
		}
	}

	// DISP chunks skip four bytes before the display name starts.
	if( strcmp(chunkName(), "DISP") == 0 )
	{
		file.get();
		file.get();
		file.get();
		file.get();
	}

	// read the value, if we recognize the type
	if( found )
	{
		int c;
		value = "";
		while( (c = file.get()) != '\0' && c != EOF )
			value += char(c);
	}

	// whether we recognize it or not, pop back out.
	pop();
	return found;
}

/***************************************************************************
	member functions for RiffChunk
***************************************************************************/

RiffChunk::RiffChunk(RiffFile& parent)
{
	// read the chunk name
	parent.filep()->read(name, 4);
	name[4] = '\0';

	// read the chunk size
	parent.filep()->read((char*)&size, 4);

#ifdef REVERSE_ENDIANISM
	// reverse the endianism of the chunk size.
	size = SWAP_32(size);
#endif

	// if this is a RIFF or LIST chunk, read its subtype.
	if( strcmp(name, "RIFF") == 0 || strcmp(name, "LIST") == 0 )
	{
		parent.filep()->read(subType, 4);
		subType[4] = '\0';

		// subtract the subtype from the size of the data.
		size -= 4;
	} else
		*subType = '\0';

	// the chunk starts after the name and size.
	start = parent.filep()->tellg();

	// the next chunk starts after this one, but starts on a word boundary.
	after = start + size;
	if (after % 2)
		after++;
}

