#include <errno.h>
#include <iostream.h>
#include <qfile.h>
#include <sys/types.h>
#include <unistd.h>

#include "SHA1Computer.h"
#include "SHA1Storage.h"
#include "SharedFile.h"
#include "sha1.h"

using namespace std;

SHA1Computer::SHA1Computer()
{
#ifndef SHARP
      start();
#else
      // Since Qtopia does not support threads we can not compute the sha1
      // in the background
      run();
#endif
}

void SHA1Computer::queue(SharedFile *sf, const QString &fileName)
{
      char *f = new char[fileName.length() + 1];
      memcpy(f, fileName.data(), fileName.length());
      f[fileName.length()] = '\0';
#ifndef SHARP
      _input.lock();
      _queue.enqueue(f);
      _input.unlock();
#else
      _queue.enqueue(f);
#endif
}

static const char base32Digit[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZ234567";

static void convert_sha1_into_base32(const uint8_t *sha1_binary, char *sha1_base32)
{
      int i;
      const unsigned char *p = sha1_binary;
      char *q = sha1_base32;
      for(i = 4; i; i--) { /* 4 sequences of 8 base-32 digits */
	    *q++ = base32Digit[(p[0] & 0xf8) >> 3];
	    *q++ = base32Digit[(p[0] & 0x07) << 2 | (p[1] & 0xc0) >> 6]; p++;
	    *q++ = base32Digit[(p[0] & 0x3e) >> 1];
	    *q++ = base32Digit[(p[0] & 0x01) << 4 | (p[1] & 0xf0) >> 4]; p++;
	    *q++ = base32Digit[(p[0] & 0x0f) << 1 | (p[1] & 0x80) >> 7]; p++;
	    *q++ = base32Digit[(p[0] & 0x7c) >> 2];
	    *q++ = base32Digit[(p[0] & 0x03) << 3 | (p[1] & 0xe0) >> 5]; p++;
	    *q++ = base32Digit[(p[0] & 0x1f)]; p++;
      }
}

void SHA1Computer::processOne()
{
#ifndef SHARP
      _input.lock();
#endif
      char *f = _queue.dequeue();
#ifndef SHARP
      _input.unlock();
#endif
      if (!f) return;

      QFile file(f);
      if (!file.open(IO_Raw|IO_ReadOnly)) {
	    cerr << "Qtella: SHA1Computer::processOne: unable to open " << f
		 << " in order to compute SHA1 hash: " << endl;
	    return;
      }
      SHA1Context context;
      SHA1Reset(&context);
      for(;;) {
	    char buffer[4096];
	    int r = file.readBlock(buffer, 4096);
	    if (r < 0) {
		  perror("SHA1Computer::processOne: readBlock");
		  return;
	    }
	    if (!r) {
		  uint8_t hash[SHA1HashSize];
		  SHA1Result(&context, hash);
		  char hash_base32[SHA1_BASE32_SIZE + 1];
		  convert_sha1_into_base32(hash, hash_base32);
		  hash_base32[SHA1_BASE32_SIZE] = '\0';
		  addResult(f, hash_base32);
		  return;
	    }
	    int res = SHA1Input(&context, reinterpret_cast<uint8_t *>(&buffer), r);
	    if (res != shaSuccess) {
		  cerr <<  "error in SHA1Input" << endl;
		  return;
	    }
      }
}

void SHA1Computer::run()
{
  // no sha1 computation for zaurus
#ifndef SHARP
      for(;;) {
	    _input.lock();
	    bool empty = _queue.isEmpty();
	    _input.unlock();
	    if (!empty) processOne(); else sleep(1);
      }
#endif
}

QStack<SHA1FileNameHash> *SHA1Computer::getResults()
{
      if (!_result) return NULL;
#ifndef SHARP
      _output.lock();
#endif
      QStack<SHA1FileNameHash> *result = _result;
      _result = NULL;
#ifndef SHARP
      _output.unlock();
#endif
      return result;
}

void SHA1Computer::addResult(char *fileName, char *sha1)
{
      SHA1FileNameHash *r = new SHA1FileNameHash;
      r -> _fileName = fileName;
      memcpy(r -> _hash, sha1, SHA1_BASE32_SIZE);
      r -> _hash[SHA1_BASE32_SIZE] = '\0';

#ifndef SHARP
      _output.lock();
#endif
      if (!_result) _result = new QStack<SHA1FileNameHash>;
      _result -> push(r);
#ifndef SHARP
      _output.unlock();
#endif
}

SHA1Computer SHA1ComputerInstance;
