#include <iostream>
#include <qdir.h>
#include <qfile.h>
#include <qtextstream.h>
#include <qtimer.h>

#include "SHA1Computer.h"
#include "SHA1Storage.h"
#include "SharedFile.h"

using namespace std;

struct SHA1CacheFileLine {
      QString _sha1;
      uint32_t _fileSize;
      uint32_t _mtime;
      QString _fileName;
};

void SHA1Storage::writeLine(QTextStream &cache, const QString &hash, uint32_t size, uint32_t mtime, const QString &fileName)
{
      cache << hash << '\t' << size << '\t' << mtime << '\t' << fileName << '\n';
}

void SHA1Storage::create(QList<SHA1CacheFileLine> *image)
{
      QFile storage(_cacheFileName);
      if (!storage.open(IO_WriteOnly)) {
	    cerr << "Can't create SHA1 cache file " << _cacheFileName << endl;
	    return;
      }
      cerr << "Creating SHA1 cache file " << _cacheFileName << endl;
      QTextStream cacheFile(&storage);
      cacheFile << 
"#\n"
"# qtella SHA1 cache file\n"
"# This is file is automatically generated.\n"
"# Format is: SHA1 digest (base32 encoded)<TAB>file size<TAB>file mtime<TAB>file name\n"
"# Comment lines start with a sharp (#)\n"
"# \n"
	    "\n";
      if (!image) return;
      for(SHA1CacheFileLine *i = image -> first(); i; i = image -> next()) {
	    writeLine(cacheFile, i -> _sha1, i -> _fileSize, i -> _mtime, i -> _fileName);
      }
}

//!Retrieve data from the cache

SHA1Storage::SHA1Storage()
:_cache(9887), _resultCollector(NULL), _pendingSHA1Calculations(0)
{
      connect(&_resultCollector, SIGNAL(timeout()), this, SLOT(checkForAvailableResults()));

      _cacheFileName = QDir::homeDirPath() + "/.qtella/sha1_cache";
      _originOfTime.setTime_t(0);

      QFile storage(_cacheFileName);
      if (!storage.open(IO_ReadOnly)) {
	    create(NULL);
	    return;
      }

      QList<SHA1CacheFileLine> cacheImage;
      cacheImage.setAutoDelete(true);
      bool cacheTrashed = false;

      QTextStream cacheFile(&storage);
      for(;;) {
	    QString line = cacheFile.readLine();
	    if (line.isNull()) break;
	    if (line.isEmpty() || line.at(0) == '#') continue;
	    QTextStream decoder(&line, IO_ReadOnly);
	    QString sha1;
	    uint fileSize;
	    int mtime;
	    decoder >> sha1 >> fileSize >> mtime;
	    decoder.skipWhiteSpace();
	    QString fileName = decoder.read();
	    QFileInfo f(fileName);
	    int fileMtime = _originOfTime.secsTo(f.lastModified());
	    if (f.size() == fileSize && fileMtime == mtime) {
		  _cache.replace(fileName, new QString(sha1));
		  SHA1CacheFileLine *line = new SHA1CacheFileLine;
		  line -> _sha1 = sha1;
		  line -> _fileSize = fileSize;
		  line -> _mtime = mtime;
		  line -> _fileName = fileName;
		  cacheImage.append(line);
	    } else
		  cacheTrashed = true;
      }
      if (cacheTrashed) create(&cacheImage);
}

SHA1Storage SHA1StorageInstance;

/*!
  Return a pointer to the SHA1 of the file given by its fileName. Return NULL if unknown.
 */

QString *SHA1Storage::get(SharedFile *sf)
{ 
      QString fileName(sf->completeFilename().c_str());
      QString *result = _cache[fileName];
      if (!result) {
	    SHA1ComputerInstance.queue(sf, fileName);
	    _pendingFile.insert(fileName, sf);
	    _pendingSHA1Calculations++;
	    if (!_resultCollector.isActive()) _resultCollector.start(1000);
      }
      return result;
}

//! Timer pooling the SHA1Computer to get and process results.

void SHA1Storage::checkForAvailableResults()
{
      QStack<SHA1FileNameHash> *l = SHA1ComputerInstance.getResults();
      if (!l) return;
      
      QFile storage(_cacheFileName );
      if (!storage.open(IO_WriteOnly | IO_Append)) {
	    cerr << "qtella: Unable to open SHA1 cache file " << _cacheFileName ;
      }
      QTextStream cacheFile(&storage);

      while(!l -> isEmpty()) {
	    SHA1FileNameHash *h = l -> pop();
	    QString fileName(h -> _fileName);

	    QString *sha1 = new QString(h -> _hash);
	    _cache.replace(fileName, sha1);

	    QFile f(fileName);
	    if (!f.exists()) continue;
	    QFileInfo info(fileName);
	    writeLine(cacheFile, *sha1, f.size(), _originOfTime.secsTo(info.lastModified()), fileName);

	    SharedFile *sf = _pendingFile[fileName];
	    if (sf) sf -> _sha1 = sha1;
	    _pendingFile.remove(fileName);

	    if (!--_pendingSHA1Calculations) _resultCollector.stop();

	    delete h;
      }
      delete l;
}
