#include "Gnutella.h"
#include "Servent.h"
#include "Connections.h"
#include "QtellaSub.h"
#include "Logging.h"

#include "Ping.h"
#include "Host.h"
#include "Address.h"
#include "Message.h"
#include "PongCache.h"
#include "Bye.h"
#include "StringManipulation.h"
#include "WidgetConfig.h"
#include "WidgetConnections.h"

#include <string>
#include <vector>
#include <strstream>

#include <qsocket.h>
#include <qlistview.h>
#include <qspinbox.h>
#include <qhostaddress.h>
#include <qlineedit.h>
#include <qlistbox.h>

#undef _DEBUG
#undef _SHOW_MESSAGES

#ifdef _SHOW_MESSAGES
#include <iostream>
#endif

Servent::Servent(Connections* parent, std::string id, QListViewItem *item, QSocket* socket, const std::string user_agent, const std::string handshake)
  : _id(id), _item(item), _timeout(0), _input(0), _output(0), _state(Idle),
    _parent(parent), _direction(0), _last_ping_send(0), _bytes_in(0), _bytes_out(0), 
    _bandwidth_in(0), _bandwidth_out(0), _queries(0),
    _timeout_counter(0), _bandwidth_counter(0), _ping_counter(0), _user_agent(user_agent),
    _header(handshake)
{
  _bye = 200;

  if( _user_agent.empty() )
    _version = "0.4";
  else
    _version = "0.6";

  _last_input = time(NULL);
  _start = time(NULL);

  _connect_str = CONNECT_STR;
  _response_str = RESPONSE_STR;

  if(socket) _socket = socket; else _socket = new QSocket();

  QObject::connect(_socket, SIGNAL(error(int)), SLOT(slotError(int)));
  QObject::connect(_socket, SIGNAL(connected()), SLOT(slotConnect()));
  QObject::connect(_socket, SIGNAL(readyRead()), SLOT(slotReadyRead()));
  QObject::connect(_socket, SIGNAL(connectionClosed()), SLOT(slotClosed()));
  QObject::connect(_socket, SIGNAL(delayedCloseFinished()), SLOT(slotDelayedClosed()));

  if(socket)
    if(_socket->state() == QSocket::Connection)
      {
	if( _version == "0.4" ) 
	  sendMessageDirect(_response_str);

	Ping p(1, 0, _id);
	sendMessageDirect(p._data);  // send initial ping
	_parent->_statistic.out_ping_n++;
	_parent->_statistic.out_ping_b += p._data.size();
	setState(Connected);
      }
    else
      setState(Closed);

  _servant_number = Servent::getNextServentNumber();

  _timeout_timer = new QTimer(this);
  QObject::connect(_timeout_timer, SIGNAL(timeout()), SLOT(timeout()));
  _timeout_timer->start(1000);
}


Servent::~Servent()
{
  checkXTry();

  std::strstream str;
  str << "Destructor called for " << _address.toString() << ". Servent: <" << _user_agent 
      << "> Buffer size: " << _buffer.size() << " Tx: " << _output
      << " Rx: " << _input << std::ends;
  Logging::getInstance()->log( Logging::Info, str.str() );
  str.freeze(false);

  PongCache::getInstance()->removePong(_servant_number);
  if(_item) delete _item;
  if(_timeout_timer) delete _timeout_timer;

  QObject::disconnect(_socket, 0, 0, 0);
  if(_socket) 
    {
      Bye bye( _bye );
      sendMessageDirect( bye.toString() );

      delete _socket;
    }
}


void Servent::setByeCode( const int code )
{
  _bye = code;
}


const int Servent::getBufferSize()
{
  return _buffer.size();
}


int Servent::getNextServentNumber()
{
  static unsigned int number = 1;
  ++number;
  if(!number) number = 1;
  return number;
}


void Servent::connect(Address address)
{
  _address = address;
  _strip.erase();
  _strip = address.strIP();

  setState(Connecting);
  _timeout = 0;
  _socket->connectToHost(_strip.c_str(), address.port());
}



void Servent::slotDelayedClosed()
{
  setState(Closed);
}


void Servent::close()
{
  Bye bye( _bye );
  sendMessageDirect( bye.toString() );

  _socket->close();
  setState(Closed);
}



int Servent::state()
{
  return _state;
}



void Servent::setState(int state)
{
  if( (state == Closed) && (!_message_buffer.empty()) )
    _state = Closing;
  else
    _state = state;
}



void Servent::slotError(int err)
{
#ifdef _DEBUG
  qDebug( "entering Servent::slotError with error code %d", err );
#endif

  // check whether we've got the X-Try field
  checkXTry();

  slotReadyRead();

  std::strstream str;
  str << "Connection error. Servent: <" << _user_agent 
      << "> Buffer size: " << _buffer.size() << " Tx: " << _output 
      << " Rx: " << _input << std::ends;
  Logging::getInstance()->log( Logging::Error, str.str() );
  str.freeze(false);

  if(_socket->state() != QSocket::Idle) _socket->close();
  setState(Error);
  _timeout = 0;
}



void Servent::writeConnectString()
{
  _timeout = 0;

  QHostAddress ha = _socket->peerAddress();
  
  // protocol v0.6 implementation, first connect string
  std::string connect_string;

  if( QtellaSub::getInstance()->_use_opensourcep2p )
    connect_string = "OPENSOURCE";
  else
    connect_string = "GNUTELLA";

  connect_string += std::string(" CONNECT/0.6\r\nUser-Agent: Qtella ");
  connect_string += std::string(VERSION) + ADDITIONAL_VERSION_STR + "\r\n";
  connect_string += std::string("Remote-IP: ") + std::string(ha.toString().latin1()) + "\r\n";
  connect_string += std::string("X-Qtella-User: ") + 
    std::string( _parent->_parent->_widget_config->ui_lineedit_userid->text().latin1() ) + "\r\n";
  connect_string += std::string("X-Qtella-Group: ") + 
    std::string( _parent->_parent->_widget_config->ui_lineedit_groupid->text().latin1() ) + "\r\n";
#ifdef _DEFLATE
  connect_string += std::string("Accept-Encoding: deflate\r\n");
#endif
  connect_string += std::string("\r\n");
  sendMessageDirect(connect_string);

  // qDebug("<%s>", connect_string.c_str());
  // old protocol v0.4 implementation
  // sendMessageDirect(_connect_str);
}



void Servent::slotConnect()
{
  setState(Waiting);
  writeConnectString();
}



const void Servent::checkXTry()
{
#ifdef _DEBUG
  qDebug( "Servent::checkXTry()" );
#endif
  if( vect_find( _buffer, "X-Try:" ) == std::string::npos )
    return;

  // great, we've got IPs of other hosts
  std::string s;
  copy( _buffer.begin(), _buffer.end(), back_inserter( s ) );

#ifdef _DEBUG
  qDebug( "Got: <%s>", s.c_str() );
#endif

  std::string::size_type pos = s.find( "X-Try:" );
  s = s.substr( pos + 6 );
  pos = s.find_first_of( "0123456789" );
  std::string::size_type pos2 = s.find_first_of( "\n" );

  if( pos2 <= pos )
    return;

  s = s.substr( pos, pos2 - pos );
  s = s.substr( 0, s.find_last_of( "0123456789" ) + 1 );

#ifdef _DEBUG
  qDebug( "X-Try check: <%s>", s.c_str() );
#endif

  StringManipulation sm( s );
  std::vector< std::string > v;
  sm.split_to_vector( v, ',' );
  for( int i = 0; i < v.size(); ++i )
    _parent->_parent->_widget_connections->ui_listbox_hostlist->insertItem( v[i].c_str(), -1);
}


void Servent::slotClosed()
{
#ifdef _DEBUG
  qDebug( "entering Servent::slotClosed(); my number is %d", _servant_number );
#endif

  // check whether we've got the X-Try field
  checkXTry();

  slotReadyRead();

  std::strstream str;
  str << "Connection closed by foreign host. Servent: <" << _user_agent 
      << "> Buffer size: " << _buffer.size() << " Tx: " << _output
      << " Rx: " << _input << std::ends;
  Logging::getInstance()->log( Logging::Info, str.str() );
  str.freeze(false);

  setState(Closed);
}



void Servent::sendMessageDirect( const std::string& buffer )
{
  if(_socket->state() == QSocket::Connection)
    {
      _output++;
      _bytes_out += buffer.size();
      _socket->writeBlock(buffer.data(), buffer.size());
      _socket->flush();
    }
}



void Servent::sendMessage(std::string& buffer, bool cache)
{
  if( ((_send_buffer.size() < 512) && (_socket->bytesToWrite() < 1024))|| !cache )
    {
      _output++;
      _send_buffer.append(buffer);
    }
}


void Servent::limitVectors()
{
  while(_id_set.size() > 2000) _id_set.erase(_id_set.begin());
}


int Servent::getDirection()
{
  return _direction;
}


void Servent::setDirection(int i)
{
  _direction = i;
}


void Servent::slotReadyRead()
{
#ifdef _DEBUG
  qDebug("entering Servent::slotReadyRead");
#endif

  char   cbuff[4096+10];

  if( ! _socket->bytesAvailable() )
    {
#ifdef _DEBUG
      qDebug( "no bytes available" );
#endif
      return;
    }
#ifdef _DEBUG
  else
    qDebug( "%d bytes available.", _socket->bytesAvailable() );
#endif
    
  _last_input = time(NULL);

  int bytes = 0;
  do
    {
      int rd = _socket->readBlock(cbuff, 4096);
      bytes += rd;

      for( int i = 0; i < rd; ++i )
        {
          _buffer.push_back( cbuff[i] );
#ifdef _SHOW_MESSAGES
          std::cout << cbuff[i] << std::ends;
#endif
        }
    }
  while( _socket->bytesAvailable() );

#ifdef _SHOW_MESSAGES
  std::cout << std::endl;
#endif
#ifdef _DEBUG
  qDebug("Servent (%d) read %d bytes", _servant_number, bytes);
#endif



  if( state() == Waiting && vect_find( _buffer, "\r\n\r\n" ) != std::string::npos )
    {
#ifdef _DEBUG
      qDebug( "Found header. %d", _servant_number );
#endif
      checkXTry();

      std::string::size_type idx = vect_find( _buffer, "\r\n\r\n" );
      _header = vect_substr( _buffer, 0, idx );
      std::string b = vect_substr( _buffer, 0, _buffer.size() );
      
      std::vector< char >::iterator e = _buffer.begin();
      for( int i = 0; i < idx + 4; ++i)
        ++e;
      _buffer.erase( _buffer.begin(), e );
      
      
      if( b.substr(0, 9) != "GNUTELLA/" && b.substr(0, 11) != "OPENSOURCE/" )
        {
#ifdef _DEBUG
          qDebug("Servent error");
#endif
          slotError(-1);
          return;
        }
      
      b.erase( 0, b.find( "/" ) + 1 );
      
      // get version
      
#ifdef _DEBUG
      qDebug( "Retrieving version number. %d", _servant_number );
#endif
      
      idx = b.find(" ");
      if( idx == std::string::npos )
        {
#ifdef _DEBUG
          qDebug("Servent error");
#endif
          slotError(-1);
          return;
        }
      
      _version = b.substr(0, idx);
      
      b.erase(0, idx +1 );  // delete version number
      
      // get status code
      
      idx = b.find(" ");
      if( idx == std::string::npos )
        {
#ifdef _DEBUG
          qDebug("Servent error");
#endif
          slotError(-1);
          return;
        }
      
      std::string status = b.substr(0, idx);
      b.erase(0, idx + 1); // delete status code with space from buffer
      
      // get Remote-IP field
      idx = b.find("Remote-IP:");
      if( idx != std::string::npos )
        {
          idx = b.find_first_of("0123456789", idx);
          std::string::size_type idx2 = b.find_first_not_of("0123456789.", idx);
          if( idx2 != std::string::npos )
            {
              std::string ip = b.substr( idx, idx2 - idx );
#ifdef _DEBUG
              qDebug( "Found my IP: <%s>", ip.c_str() );
#endif
              QtellaSub::getInstance()->setReportedIP( ip );
            }
        }
      
      // get user agent information
      
      idx = b.find("User-Agent:");
      if( idx != std::string::npos )
        {
          b.erase(0, idx + 11);
          idx = b.find_first_not_of(" ");
          b.erase(0, idx);
          idx = b.find("\r\n");
          _user_agent = b.substr(0, idx);
        }
      
      if( status == "503")
        {
#ifdef _DEBUG
          qDebug("Servent: Busy <%s>", b.c_str());
#endif
          slotError(-1);
          return;
        }
      
      if( status != "200")
        {
#ifdef _DEBUG
          qDebug( "Servent: unknown status <%s>" );
#endif
          std::strstream str;
          str << _address.toString() << " Status: <" << status << ">" << " Servent: <" << _user_agent 
              << "> Buffer size: " << b.size() << " Tx: " << _output
              << " Rx: " << _input << std::ends;
          Logging::getInstance()->log( Logging::Error, str.str() );
          str.freeze(false);
          
#ifdef _DEBUG
          qDebug("Servent error");
#endif
          slotError(-1);
          return;
        }
	  
      // status code is 200 ok
	  
      idx = b.find("\r\n\r\n"); // find end of header
      b.erase(0, idx + 4);      // delete header from buffer
      
      setState(Connected);
      
      std::string r;
      if( QtellaSub::getInstance()->_use_opensourcep2p )
        r = "OPENSOURCE";
      else 
        r = "GNUTELLA";
      
      r += std::string("/0.6 200 OK\r\n\r\n");
      sendMessageDirect(r);
      
      int ttl = _parent->_parent->_widget_config->ui_spinbox_ttl->value();
      Ping p(1, 0, _id);
      sendMessageDirect(p._data);  // send initial ping
      _parent->_statistic.out_ping_n++;
      _parent->_statistic.out_ping_b += p._data.size();

#ifdef _DEBUG
      qDebug("Servent ok. %d", _servant_number );
#endif
      return;
    }

  else
    if( state() != Waiting )
    while( _buffer.size() >= HEADER_SIZE )
      {
        Q_UINT32 payload  = Message::str_uint32( _buffer, 19 );  // position 19, 4 bytes
        
        if( payload > 20000 )
          {
            _buffer.clear();
#ifdef _DEBUG
            qDebug("Servent error (payload is big)");
#endif
            slotError(-1);
            return;
          }
		
        if( _buffer.size() >= (HEADER_SIZE + payload) )
          {
            ++_input;
			
            _parent->incAllPackets();
			
            int limit = 40;
            if( time(NULL) - _last_search < 30 ) // if last search is not older than 30s 
              limit = 120; // set limit to 120 messages/s
			
            if( _message_buffer.size() < limit )
              {
                std::vector< char > v;
                for( int k = 0; k < HEADER_SIZE + payload; ++k )
                  v.push_back( _buffer[k] );
                _message_buffer.push_back( v );
              }
            else
              _parent->incIgnoredPackets();

#ifdef _DEBUG
            qDebug( "Removing message from buffer." );
#endif
            std::vector< char >::iterator pos = _buffer.begin();
            for( int i = 0; i < HEADER_SIZE + payload; ++i )
              ++pos;
            _buffer.erase( _buffer.begin(), pos );
            _bytes_in += HEADER_SIZE + payload;
          }
        else
          return;
      }
#ifdef _DEBUG
  qDebug("exit Servent::slotReadyRead");
#endif
}


void Servent::timeout()
{
  _timeout_counter++;

  if(_timeout_counter >= 5)
    if( (time(NULL) - _last_input) >= 120 ) 
      {
        close();
        _timeout_counter = 0;
      }


  if( (_socket->state() == QSocket::Connection) && (state() == Connected) &&
      (_send_buffer.size() > 0) )
    {
      _bytes_out += _send_buffer.size();
      _socket->writeBlock(_send_buffer.data(), _send_buffer.size());
      _send_buffer.erase();
    }


  _bandwidth_counter++;

  if(_bandwidth_counter >= 2)
    {
      _bandwidth_in = static_cast<double>(_bytes_in) / static_cast<double>(2048);
      _bandwidth_out = static_cast<double>(_bytes_out) / static_cast<double>(2048);
      _bytes_in = 0;
      _bytes_out = 0;
      _bandwidth_counter = 0;
    }


  _ping_counter++;
  
  if(_ping_counter >= 3)
    {
      if( !_last_accepted_ping_id.empty() )
        if( ((static_cast<unsigned char>(_last_accepted_ping_id[8]) == 0xff) && 
             (_last_accepted_ping_id[15] == 1)) ||
            ((time(NULL) - _last_ping_send >= 30) && (state() == Connected)) )
          {
            int ttl = _parent->_parent->_widget_config->ui_spinbox_ttl->value();
            Ping p(ttl, 0, _id);
            sendMessage(p._data, false);
			
            _parent->_statistic.out_ping_n++;
            _parent->_statistic.out_ping_b += p._data.size();
			
            _last_ping_send = time(NULL);
          }
      _ping_counter = 0;
    }
  
  
  _queries = 0;
}


long Servent::uptime()
{
  return (time(NULL) - _start);
}


const void Servent::updateLastSearch()
{
  _last_search = time(NULL);
}

#ifdef QTELLA_WIN
std::string::size_type Servent::vect_find( const std::vector<char>& v, const std::string& pattern )
{
// #ifdef _DEBUG
//   qDebug( "Servent:vect_find: <%d>, <%s> ", v.size(), pattern.c_str() );
// #endif
  if( v.size() < pattern.size() )
    return std::string::npos;
  for( std::string::size_type i = 0; i <= v.size() - pattern.size(); ++i )
    {
      bool a = true;
      for( std::string::size_type j = 0; j < pattern.size(); ++j )
        if( v[i + j] != pattern[j] )
          {
            a = false;
            break;
          }
      if( a ) 
        return i;
    }
  return std::string::npos;
}

std::string Servent::vect_substr( const std::vector<char>& v, const unsigned beg, const unsigned num )
{
  std::string s;
  for( int i = beg; i < beg + num; ++i )
    s = s + v[i];
  return s;
}
#endif

// std::string Servent::getFirstMessage()
// {
//   std::string s;
//   copy( _message_buffer.front().begin(), _message_buffer.front().end(), back_inserter( s ) );
//   return s;
// }

std::vector< char >& Servent::getFirstMessage()
{
  return _message_buffer.front();
}

const void Servent::removeFirstMessage()
{
  _message_buffer.pop_front();
}

const std::string& Servent::getUserID()
{
  return _user_id;
}

const std::string& Servent::getGroupID()
{
  return _group_id;
}

const std::string Servent::getHeaderField( const std::string& header, const std::string& field )
{
  std::string::size_type pos = header.find( field + ":" );
  if( pos == std::string::npos )
    return "";
  pos += field.size() + 1;
  std::string::size_type pos2 = header.find( "\r\n", pos );
  std::string s = header.substr( pos, pos2 - pos );
  s = s.substr( s.find_first_not_of( " " ) );
  while( ! s.empty() && s[s.size() - 1] == ' ' )
    s.erase( s.size() - 1 );
  return s;
}
