#include "Gnutella.h"
#include "Upload.h"
#include "UploadManager.h"
#include "SharedFile.h"
#include "QtellaSub.h"
#include "Connections.h"
#include "Push.h"
#include "StringManipulation.h"
#include "SharedViewItem.h"
#include "WidgetConfig.h"
#include "WidgetUploads.h"

#include "qsocket.h"
#include "qlistview.h"
#include "qfile.h"
#include "qdatetime.h"
#include "qspinbox.h"
#include "qhostaddress.h"
#include "qtimer.h"
#include "qpixmap.h"
#include "qimage.h"
#include "qcheckbox.h"

#include <iostream>
#include <iomanip>
#include <strstream>
#if !defined(_WIN32)
#include <unistd.h>
#endif

#undef _DEBUG

const std::string Upload::statestr[] = {"Busy", "Start", "Connected", "NotFound", 
					"Finished", "Error", "Abort", "Closed", "Connecting"};

Upload::Upload(UploadManager *parent, const std::string& data, QSocket* socket)
  : _parent(parent), _push(false), _state(Connecting), item(NULL), _socket(socket), _timer(NULL)
{
  _data.assign( data.data(), data.size() );

  connect(_socket, SIGNAL(connectionClosed()), SLOT(slotClosed()));
  initUpload();
  if(getState() != Connected) return;

  _timer = new QTimer(this);
  connect(_timer, SIGNAL(timeout()), SLOT(slotTimer()));
  _timer->start(1000);
 
  _time_start.start();
}


Upload::Upload(UploadManager *parent, Push push)
  : _parent(parent), _push(true), _state(Connecting), _push_data(push), item(NULL), _timer(NULL)
{
  _socket = new QSocket();
  connect(_socket, SIGNAL(connectionClosed()), SLOT(slotClosed()));
  connect(_socket, SIGNAL(connected()), SLOT(slotConnected()));
  connect(_socket, SIGNAL(readyRead()), SLOT(slotReadyRead()));
  _socket->connectToHost(_push_data._address.strIP().c_str(), _push_data._address.port());
}


Upload::~Upload()
{
  if(_timer) delete _timer;
  if(_socket) delete _socket;
  if(item) delete item;
}


int Upload::getState()
{
  return _state;
}


void Upload::slotClosed()
{
  if(_socket) delete _socket;
  if(_timer) delete _timer;

  _socket = NULL;
  _timer = NULL;
  setState(Closed);
}

void Upload::abortUpload()
{
  if(_timer) _timer->stop();
  if(_socket) _socket->close();
  _parent->_parent->connections->_statistic.loads_dabor++;
  setState(Abort);
}

void Upload::setState(int state)
{
  if( _state == Finished ) return;
  _state = state;
  if(item) item->setText(1, statestr[_state].c_str());
}



void Upload::slotConnected()  // connected by push
{
  std::string f = _parent->_parent->vSharedFiles[_push_data._index]->file;
  std::strstream str;
  str << "GIV " << _push_data._index << ":" << std::ends;
  std::string s = str.str();
  str.freeze(false);
  s += Connections::toHex(_parent->_parent->connections->_id) + "/" + f + "\n\n";
      
  _socket->writeBlock(s.data(), s.size());
}



void Upload::slotReadyRead()  // connected by push
{
  while(_socket->canReadLine())
    {
      std::string n( _socket->readLine().latin1() );
      _data += n;

      if(_data.find("\r\n\r\n") != std::string::npos)
	{
	  disconnect(_socket, 0, 0, 0);
	  initUpload();
	  if(getState() != Connected) return;
	  _timer = new QTimer(this);
	  connect(_timer, SIGNAL(timeout()), SLOT(slotTimer()));
	  _timer->start(1000);
	  _time_start.start();
	}
    }
}

static int hexToInt(char c)
{
      if (c >= '0' && c <= '9') return c - '0';
      else if (c >= 'a' && c <= 'f') return c - 'a' + 0xa;
      else if (c >= 'A' && c <= 'F') return c - 'A' + 0xa;
      else {
	    qDebug("%s: shouldn't be reached", __PRETTY_FUNCTION__);
	    return 0;
      }
}

static void unescape(const std::string &from, char *result)
{
      const char *p = from.data();
      char *q = result;
      for(int left = from.length(); left; left--, q++) {
	    if (*p == '%' && left > 2 && isxdigit(p[1]) && isxdigit(p[2])) {
		  int c = hexToInt(p[1]) << 4 | hexToInt(p[2]);
		  *q = c;
		  p += 3;
	    } else
		  *q = *p++;
      }
      *q = '\0';
}

std::string &operator+=(std::string &lhs, const QString &rhs) {
      return lhs.append(rhs.latin1(), rhs.length());
}

void Upload::send404()
{
      std::string s;
      s = "HTTP/1.0 404 NOT FOUND\r\nServer: Qtella "+std::string(VERSION) + ADDITIONAL_VERSION_STR + "\r\n\r\n";
#ifdef _DEBUG
      qDebug( "Sending 404: file not found" );
      std::cerr << "===\n" << s << "===" << std::endl;
#endif
      _socket->writeBlock(s.data(), s.size());
      _socket->close();
      setState(NotFound);
}

void Upload::initUpload()
{
#ifdef _DEBUG
  qDebug( "Upload::initUpload <%s, %d>", _data.c_str(), _data.size() );
#endif

  if( _data.size() < 4 )
    {
      _socket->close();
      setState(Closed);
      return;
    }

  StringManipulation sm(_data);

  if(_data.substr(0,3) != "GET")
    {
      setState(Closed);
      return;
    }

  std::string::size_type  idx = sm.to_lower().find("/get/");
  
  if(idx == std::string::npos)
    {
	  send404();
	  return;
    }

  idx +=5;
  std::string::size_type  idx2 = _data.find("/", idx);

  if(idx2 == std::string::npos)
    {
	  send404();
	  return;
    }

  std::string sindex = _data.substr(idx, idx2 - idx);

#ifdef _DEBUG
  qDebug( "Index: <%s>", sindex.c_str() );
#endif

  std::strstream a;
  a << sindex << std::ends;
  a >> index;

  
  idx = sm.to_lower().find(" http", idx2);

  if(idx == std::string::npos)
    {
      _socket->close();
      setState(Closed);
      return;
    }
  
  filename = _data.substr(idx2 + 1, idx - idx2 - 1);
#ifdef _DEBUG
  qDebug( "Filename/1: <%d:%s>", filename.length(), filename.c_str() );
#endif

  char buffer[filename.size() + 1];
  unescape(filename, buffer);
  filename = buffer;

#ifdef _DEBUG
  qDebug( "Filename/2: <%d:%s>", filename.length(), filename.c_str() );
#endif

  if( (idx = sm.to_lower().find("user-agent:")) != std::string::npos )
    {
      idx = _data.find_first_not_of(" ", idx+11);
      idx2 = _data.find("\r\n", idx);
      client = _data.substr(idx, idx2-idx);
    }
  else
    client = "";

#ifdef _DEBUG
  qDebug( "User-Agent: <%s>", client.c_str() );
#endif

  if( (idx = sm.to_lower().find("range:")) != std::string::npos )
    {
      idx = _data.find_first_of("1234567890", idx+6);
      idx2 = _data.find("-", idx);
      std::string srange_start = _data.substr(idx, idx2-idx);
      
      std::strstream c;
      c << srange_start << std::ends;
      c >> range_start;

      idx = _data.find("\r\n", idx2);
      std::string s = _data.substr(idx2, idx - idx2);

//       if( (idx = _data.find_first_of("0123456789", idx2+1)) != std::string::npos )
      if( (idx = s.find_first_of("0123456789")) != std::string::npos )
	{
	  idx2 = s.find_first_not_of("0123456789", idx);
	  std::string srange_end = s.substr(idx, idx2 - idx - 1);

	  std::strstream b;
	  b << srange_end << std::ends;
	  b >> range_end;
	}
      else range_end = 0;
    }
  else
    {
      range_start = 0;
      range_end = 0;
    }

#ifdef _DEBUG
  qDebug( "Range: <%d> - <%d>", range_start, range_end );
#endif

  if( index >= _parent->_parent->vSharedFiles.size() ) // invalid index
    {
#ifdef _DEBUG
      qDebug( "Invalid index! <%d> Shared files: %d", index, _parent->_parent->vSharedFiles.size() );
#endif
      send404();
      return;
    }

  if( _parent->_parent->vSharedFiles[index]->file == filename )
    {
      _parent->_parent->vSharedFiles[index]->requests++;
      std::strstream str;
      str << _parent->_parent->vSharedFiles[index]->requests << std::ends;
      _parent->_parent->vSharedFiles[index]->item->setText(2, str.str());
      str.freeze(false);
    }
  else  // invalid filename
    {
#ifdef _DEBUG
	  const std::string &f = _parent->_parent->vSharedFiles[index]->file;
	  qDebug( "Filename: <%d:%s>", f.length(), f.c_str() );
	  qDebug( "Filename: <%d:%s>", filename.length(), filename.c_str() );
	  qDebug( "Invalid filename!" );
#endif
	  send404();
	  return;
    }

  // get ip
  _remote_ip = static_cast<const char*>( _socket->peerAddress().toString() );
  
  if( _parent->numberUploads() >= _parent->_parent->_widget_config->ui_spinbox_maxuploads->value() ||
      _parent->numberUploadsFromHost( _remote_ip ) >= _parent->_parent->_widget_config->ui_max_up_per_user->value() )
    {
#ifdef _DEBUG
      qDebug( "Sending 503 Busy" );
#endif
      std::strstream str;
      std::string    s;
      str << "HTTP/1.0 503 Busy\r\nServer: Qtella " << VERSION << ADDITIONAL_VERSION_STR << "\r\n\r\n" << std::ends;
      s = str.str();
      str.freeze(false);
      _socket->writeBlock(s.c_str(), s.size());
      _socket->flush();
      _socket->close();
      _state = Busy;
      return;
    }

  // 
  std::string f = _parent->_parent->vSharedFiles[index]->directory;

  if(f.size() > 0) 
    if( f[f.size()-1] != '/' ) f += '/';
  f += _parent->_parent->vSharedFiles[index]->file;

  item = new QListViewItem( _parent->_parent->_widget_uploads->ui_listview_uploads, 
			    _parent->_parent->vSharedFiles[index]->file.c_str());
  item->setText( 2, _remote_ip.c_str() );

  _file.setName(f.c_str());

  if( ! _file.exists() )
    {
	  send404();
	  return;
    }

  setState(Connected);

  bool sendContentRange = range_start || range_end;

  if(range_end == 0) range_end = _file.size() - 1;

  _towrite = range_end - range_start + 1;

  std::string s = "HTTP/1.0 ";
  if (sendContentRange)
    s += "206 Partial Content\r\n";
  else
    s += "200 Ok\r\n";

  s += "Content-Length: ";
  std::strstream str;
  str << _towrite << std::ends;  // must be file.size() !!
  s += std::string(str.str());
  str.freeze(false);
  s += std::string("\r\nContent-Type: application/binary\r\nServer: Qtella "+std::string(VERSION)
				   + ADDITIONAL_VERSION_STR+"\r\n");

  if (sendContentRange) {
    std::strstream str;
    str << range_start << '-' << range_end << '/' << _file.size() << ends;
    s += "Content-Range: bytes ";
    s += str.str();
    s += "\r\n";
  }

  SharedFile &sf = *_parent->_parent->vSharedFiles[index];
  if (sf._sha1) {
	s += "X-Gnutella-Content-URN: urn:sha1:";
	s += *sf._sha1;
	s += "\r\n";
  }

  s += "\r\n";

#ifdef _DEBUG
  std::cerr << "===\n" << s << "===" << std::endl;
#endif
  _socket->writeBlock(s.c_str(), s.size());

  if( !_file.open(IO_ReadOnly|IO_Raw) )
    {
      setState(Error);
      _socket->close();
      return;
    }

  _file.at(range_start);
  _written = 0;
}



void Upload::slotTimer()
{
  char         cbuff[8192];
  unsigned int bytes = _parent->_parent->_widget_config->ui_spinbox_upbandwidth->value() * 1024;
  int          written = 0;

  if(_parent->numberUploads() > 0) bytes /= _parent->numberUploads();

  if(_socket->bytesToWrite() > bytes) bytes = 0;
  else
    bytes -= _socket->bytesToWrite();

  while(bytes > 0)
    {
      int r;
      int w;

      int toread = 8192;
      if(toread > bytes) toread = bytes;
      if(toread > _towrite) toread = _towrite;

      r = _file.readBlock(cbuff, toread);

      if(_socket->state() == QSocket::Connection) w = _socket->writeBlock(cbuff, r);
      bytes -= r;
      _towrite -= w;
      _written += w;
      written += w;
      _parent->_parent->connections->_statistic.loads_usize += w;

      if(w != r)
	{
	  _parent->_parent->connections->_statistic.loads_uabor++;
	  setState(Error);
	  _socket->close();
	  _file.close();
	  _timer->stop();
	  return;
	}

      if(_towrite == 0)
	{
	  _timer->stop();
	  _socket->close();
	  setState(Finished);
	  _parent->_parent->connections->_statistic.loads_usucc++;
	  _file.close();
	  
	  // final bandwidth
	  std::strstream str;
	  double b;
	  b = static_cast<double>((_written-range_start)*1000 / (_time_start.elapsed()*1024));
	  str.seekp(0, std::ios::beg);
	  str.setf(std::ios::fixed);
	  str << std::setprecision(1) << b << " KB/s" << std::ends;
	  item->setText(4, str.str());
	  str.freeze(false);
	  
	  // final time
	  QTime time_end = QTime();
	  time_end = time_end.addMSecs(_time_start.elapsed());
	  item->setText(5, (const char*)time_end.toString());

	  if( QtellaSub::getInstance()->_widget_config->ui_checkbox_up_progressvalue->isChecked() )
	    item->setText(3, "100%");

	  if( QtellaSub::getInstance()->_widget_config->ui_checkbox_up_showprogressbar->isChecked() )
	    {
	      QColor color(0x20, 0xff, 0x20);
	      QImage img(64, 11, 32, 16);
	      img.fill(qRgb(0xff, 0xff, 0xff));
	      int x;
	      int y;
	      for(x = 0; x < 64; x++)
		{
		  img.setPixel(x, 0, qRgb(0xaa, 0xaa, 0xaa));
		  img.setPixel(x, 10, qRgb(0xaa, 0xaa, 0xaa));
		}
	      for(y = 0; y < 11; y++)
		{
		  img.setPixel(0, y, qRgb(0xaa, 0xaa, 0xaa));
		  img.setPixel(63, y, qRgb(0xaa, 0xaa, 0xaa));
		}
	      QPixmap pict;
	      
	      for(y = 1; y <= 9; ++y)
		for(x = 1; x <= 62; ++x) img.setPixel(x, y, color.rgb());
	      
	      pict.convertFromImage(img);
	      item->setPixmap(3, pict);
	    }

	  return;
	}
    }

  // bandwidth
  std::strstream str;
  str.setf(std::ios::fixed);
  str << std::setprecision(1) << static_cast<double>(written/1024) << " KB/s" << std::ends;
  item->setText(4, str.str());
  str.freeze(false);
  _bandwidth = written / 1024;

  // progress
  if( QtellaSub::getInstance()->_widget_config->ui_checkbox_up_progressvalue->isChecked() )
    {
      str.seekp(0, std::ios::beg);
      if(range_end > 0)
	str << 100 * (range_end - _towrite) / range_end << "%" << std::ends;
      else
	str << "100%" << std::ends;
      //str << 100 * _written / (range_end+1) << "%" << std::ends;
      item->setText(3, str.str());
      str.freeze(false);
    }
  else item->setText(3, " ");

  if( QtellaSub::getInstance()->_widget_config->ui_checkbox_up_showprogressbar->isChecked() )
    {
      unsigned long l = 0;
      if( range_end > 0 ) l = 62 * (range_end - _towrite) / range_end;
      else l = 62;

      QColor color(0x20, 0x20, 0xff);
      if( _towrite == 0 ) color.setRgb(0x20, 0xff, 0x20);

      QImage img(64, 11, 32, 16);
      img.fill(qRgb(0xff, 0xff, 0xff));
      int x;
      int y;
      for(x = 0; x < 64; x++)
	{
	  img.setPixel(x, 0, qRgb(0xaa, 0xaa, 0xaa));
	  img.setPixel(x, 10, qRgb(0xaa, 0xaa, 0xaa));
	}
      for(y = 0; y < 11; y++)
	{
	  img.setPixel(0, y, qRgb(0xaa, 0xaa, 0xaa));
	  img.setPixel(63, y, qRgb(0xaa, 0xaa, 0xaa));
	}
      QPixmap pict;
			  
      for(y = 1; y <= 9; ++y)
	for(x = 1; x <= l; ++x) img.setPixel(x, y, color.rgb());
			  
      pict.convertFromImage(img);
      item->setPixmap(3, pict);
    }
  else
    {
      QPixmap pict;
      item->setPixmap(3, pict);
    }


  // time left
  double u = static_cast<double>( _written / (_time_start.elapsed()/1000) );
  unsigned long sectogo = static_cast<unsigned long>( _towrite / u );
  QTime time_togo = QTime();
  time_togo = time_togo.addSecs(sectogo);
  item->setText(5, (const char*)time_togo.toString());
}


double Upload::readBandwidth()
{
  return _bandwidth;
}


const std::string& Upload::getRemoteIP()
{
  return _remote_ip;
}
