/* 
   Unix SMB/Netbios implementation.
   SMB client library implementation
   Copyright (C) Andrew Tridgell 1998
   Copyright (C) Richard Sharpe 2000, 2002
   Copyright (C) John Terpstra 2000
   Copyright (C) Tom Jansen (Ninja ISD) 2002 
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "includes.h"

#include "libsmbclient.h"

/*
 * Functions exported by libsmb_cache.c that we need here
 */
int smbc_default_cache_functions(SMBCCTX *context);

/* 
 * check if an element is part of the list. 
 * FIXME: Does not belong here !  
 * Can anyone put this in a macro in dlinklist.h ?
 * -- Tom
 */
static int DLIST_CONTAINS(SMBCFILE * list, SMBCFILE *p) {
	if (!p || !list) return False;
	do {
		if (p == list) return True;
		list = list->next;
	} while (list);
	return False;
}

extern BOOL in_client;
extern pstring global_myname;

/*
 * Is the logging working / configfile read ? 
 */
static int smbc_initialized = 0;

/*
 * Function to parse a path and turn it into components
 *
 * We accept smb://[[[domain;]user[:password@]]server[/share[/path[/file]]]]
 * 
 * smb://       means show all the workgroups
 * smb://name/  means, if name<1D> or name<1B> exists, list servers in workgroup,
 *              else, if name<20> exists, list all shares for server ...
 */

static const char *smbc_prefix = "smb:";

static int
smbc_parse_path(SMBCCTX *context, const char *fname, char *server, char *share, char *path,
		char *user, char *password) /* FIXME, lengths of strings */
{
	static pstring s;
	pstring userinfo;
	char *p;
	char *q, *r;
	int len;

	server[0] = share[0] = path[0] = user[0] = password[0] = (char)0;
	pstrcpy(s, fname);

	/*  clean_fname(s);  causing problems ... */

	/* see if it has the right prefix */
	len = strlen(smbc_prefix);
	if (strncmp(s,smbc_prefix,len) || 
	    (s[len] != '/' && s[len] != 0)) return -1; /* What about no smb: ? */

	p = s + len;

	/* Watch the test below, we are testing to see if we should exit */

	if (strncmp(p, "//", 2) && strncmp(p, "\\\\", 2)) {

		return -1;

	}

	p += 2;  /* Skip the // or \\  */

	if (*p == (char)0)
		return 0;

	if (*p == '/') {

		strncpy(server, context->workgroup, 
			(strlen(context->workgroup) < 16)?strlen(context->workgroup):16);
		return 0;
		
	}

	/*
	 * ok, its for us. Now parse out the server, share etc. 
	 *
	 * However, we want to parse out [[domain;]user[:password]@] if it
	 * exists ...
	 */

	/* check that '@' occurs before '/', if '/' exists at all */
	q = strchr_m(p, '@');
	r = strchr_m(p, '/');
	if (q && (!r || q < r)) {
		pstring username, passwd, domain;
		char *u = userinfo;

		next_token(&p, userinfo, "@", sizeof(fstring));

		username[0] = passwd[0] = domain[0] = 0;

		if (strchr_m(u, ';')) {
      
			next_token(&u, domain, ";", sizeof(fstring));

		}

		if (strchr_m(u, ':')) {

			next_token(&u, username, ":", sizeof(fstring));

			pstrcpy(passwd, u);

		}
		else {

			pstrcpy(username, u);

		}

		if (username[0])
			strncpy(user, username, sizeof(fstring));  /* FIXME, size and domain */

		if (passwd[0])
			strncpy(password, passwd, sizeof(fstring)); /* FIXME, size */

	}

	if (!next_token(&p, server, "/", sizeof(fstring))) {

		return -1;

	}

	if (*p == (char)0) return 0;  /* That's it ... */
  
	if (!next_token(&p, share, "/", sizeof(fstring))) {

		return -1;

	}

	pstrcpy(path, p);
  
	all_string_sub(path, "/", "\\", 0);

	return 0;
}

/*
 * Convert an SMB error into a UNIX error ...
 */

static int smbc_errno(SMBCCTX *context, struct cli_state *c)
{
	int ret = cli_errno(c);
	
        if (cli_is_dos_error(c)) {
                uint8 eclass;
                uint32 ecode;

                cli_dos_error(c, &eclass, &ecode);
                
                DEBUG(3,("smbc_error %d %d (0x%x) -> %d\n", 
                         (int)eclass, (int)ecode, (int)ecode, ret));
        } else {
                NTSTATUS status;

                status = cli_nt_error(c);

                DEBUG(3,("smbc errno %s -> %d\n",
                         nt_errstr(status), ret));
        }

	return ret;
}

/* 
 * Check a server_fd.
 * returns 0 if the server is in shape. Returns 1 on error 
 * 
 * Also useable outside libsmbclient to enable external cache
 * to do some checks too.
 */
int smbc_check_server(SMBCCTX * context, SMBCSRV * server) 
{
	if ( send_keepalive(server->cli.fd) == False )
		return 1;

	/* connection is ok */
	return 0;
}

/* 
 * Remove a server from the list server_table if it's unused.
 * On success, 0 is returned. 1 is returned if the server could not be removed.
 * 
 * Also useable outside libsmbclient
 */
int smbc_remove_unused_server(SMBCCTX * context, SMBCSRV * srv)
{
	SMBCFILE * file;

	/* are we being fooled ? */
	if (!context || !context->_initialized || !srv) return 1;

	
	/* Check all open files/directories for a relation with this server */
	for (file = context->_files; file; file=file->next) {
		if (file->srv == srv) {
			/* Still used */
			DEBUG(3, ("smbc_remove_usused_server: %p still used by %p.\n", 
				  srv, file));
			return 1;
		}
	}

	DLIST_REMOVE(context->_servers, srv);

	cli_shutdown(&srv->cli);

	DEBUG(3, ("smbc_remove_usused_server: %p removed.\n", srv));

	context->callbacks.remove_cached_srv_fn(context, srv);
	
	SAFE_FREE(srv);
	
	return 0;
}

/*
 * Connect to a server, possibly on an existing connection
 *
 * Here, what we want to do is: If the server and username
 * match an existing connection, reuse that, otherwise, establish a 
 * new connection.
 *
 * If we have to create a new connection, call the auth_fn to get the
 * info we need, unless the username and password were passed in.
 */

SMBCSRV *smbc_server(SMBCCTX *context,
		     char *server, char *share, 
		     char *workgroup, char *username, 
		     char *password)
{
	SMBCSRV *srv=NULL;
	int auth_called = 0;
	struct cli_state c;
	struct nmb_name called, calling;
	char *p, *server_n = server;
	fstring group;
	pstring ipenv;
	struct in_addr ip;
	int tried_reverse = 0;
  
	zero_ip(&ip);
	ZERO_STRUCT(c);

	if (server[0] == 0) {
		errno = EPERM;
		return NULL;
	}

 check_server_cache:

	srv = context->callbacks.get_cached_srv_fn(context, server, share, 
						   workgroup, username);
	
	if (!auth_called && !srv && (!username[0] || !password[0])) {
		context->callbacks.auth_fn(server, share, workgroup, sizeof(fstring),
 			     username, sizeof(fstring), password, sizeof(fstring));
		/* 
		 * However, smbc_auth_fn may have picked up info relating to an 
		 * existing connection, so try for an existing connection again ...
		 */
		auth_called = 1;
		goto check_server_cache;
		
	}
	
	if (srv) {
		if (context->callbacks.check_server_fn(context, srv)) {
			/* 
			 * This server is no good anymore 
			 * Try to remove it and check for more possible servers in the cache 
			 */
			if (context->callbacks.remove_unused_server_fn(context, srv)) { 
				/* 
				 * We could not remove the server completely, remove it from the cache
				 * so we will not get it again. It will be removed when the last file/dir
				 * is closed.
				 */
				context->callbacks.remove_cached_srv_fn(context, srv);
			}
			
			/* 
			 * Maybe there are more cached connections to this server 
			 */
			goto check_server_cache; 
		}
		return srv;
 	}

	make_nmb_name(&calling, context->netbios_name, 0x0);
	make_nmb_name(&called , server, 0x20);

	DEBUG(4,("smbc_server: server_n=[%s] server=[%s]\n", server_n, server));
  
	if ((p=strchr_m(server_n,'#')) && 
	    (strcmp(p+1,"1D")==0 || strcmp(p+1,"01")==0)) {
    
		fstrcpy(group, server_n);
		p = strchr_m(group,'#');
		*p = 0;
		
	}

	DEBUG(4,(" -> server_n=[%s] server=[%s]\n", server_n, server));

 again:
	slprintf(ipenv,sizeof(ipenv)-1,"HOST_%s", server_n);

	zero_ip(&ip);

	/* have to open a new connection */
	if (!cli_initialise(&c)) {
		errno = ENOENT;
		return NULL;
	}

	c.timeout = context->timeout;

	if (!cli_connect(&c, server_n, &ip)) {
		cli_shutdown(&c);
 		errno = ENOENT;
 		return NULL;
 	}

	if (!cli_session_request(&c, &calling, &called)) {
		cli_shutdown(&c);
		if (strcmp(called.name, "*SMBSERVER")) {
			make_nmb_name(&called , "*SMBSERVER", 0x20);
			goto again;
		}
		else {  /* Try one more time, but ensure we don't loop */

		  /* Only try this if server is an IP address ... */

		  if (is_ipaddress(server) && !tried_reverse) {
		    fstring remote_name;
		    struct in_addr rem_ip;

		    if ((rem_ip.s_addr=inet_addr(server)) == INADDR_NONE) {
		      DEBUG(4, ("Could not convert IP address %s to struct in_addr\n", server));
		      errno = ENOENT;
		      return NULL;
		    }

		    tried_reverse++; /* Yuck */

		    if (name_status_find("*", 0, 0, rem_ip, remote_name)) {
		      make_nmb_name(&called, remote_name, 0x20);
		      goto again;
		    }


		  }
		}
		errno = ENOENT;
		return NULL;
	}
  
	DEBUG(4,(" session request ok\n"));
  
	if (!cli_negprot(&c)) {
		cli_shutdown(&c);
		errno = ENOENT;
		return NULL;
	}

	if (!cli_session_setup(&c, username, 
			       password, strlen(password),
			       password, strlen(password),
			       workgroup) &&
	    /* try an anonymous login if it failed */
	    !cli_session_setup(&c, "", "", 1,"", 0, workgroup)) {
		cli_shutdown(&c);
		errno = EPERM;
		return NULL;
	}

	DEBUG(4,(" session setup ok\n"));

	if (!cli_send_tconX(&c, share, "?????",
			    password, strlen(password)+1)) {
		errno = smbc_errno(context, &c);
		cli_shutdown(&c);
		return NULL;
	}
  
	DEBUG(4,(" tconx ok\n"));
  
	/*
	 * Ok, we have got a nice connection
	 * Let's find a free server_fd 
	 */

	srv = (SMBCSRV *)malloc(sizeof(*srv));
	if (!srv) {
		errno = ENOMEM;
		goto failed;
	}

	ZERO_STRUCTP(srv);
	srv->cli = c;
	srv->dev = (dev_t)(str_checksum(server) ^ str_checksum(share));

	/* now add it to the cache (internal or external) */
	if (context->callbacks.add_cached_srv_fn(context, srv, server, share, workgroup, username)) {
		DEBUG(3, (" Failed to add server to cache\n"));
		goto failed;
	}

	
	DEBUG(2, ("Server connect ok: //%s/%s: %p\n", 
		  server, share, srv));

	return srv;

 failed:
	cli_shutdown(&c);
	if (!srv) return NULL;
  
	SAFE_FREE(srv);
	return NULL;
}

/*
 * Routine to open() a file ...
 */

static SMBCFILE *smbc_open_ctx(SMBCCTX *context, const char *fname, int flags, mode_t mode)
{
	fstring server, share, user, password, workgroup;
	pstring path;
	SMBCSRV *srv   = NULL;
	SMBCFILE *file = NULL;
	int fd;

	if (!context || !context->_initialized) {

		errno = EINVAL;  /* Best I can think of ... */
		return NULL;

	}

	if (!fname) {

		errno = EINVAL;
		return NULL;

	}

	smbc_parse_path(context, fname, server, share, path, user, password); /* FIXME, check errors */

	if (user[0] == (char)0) pstrcpy(user, context->user);

	pstrcpy(workgroup, context->workgroup);

	srv = smbc_server(context, server, share, workgroup, user, password);

	if (!srv) {

		if (errno == EPERM) errno = EACCES;
		return NULL;  /* smbc_server sets errno */
    
	}

	/* Hmmm, the test for a directory is suspect here ... FIXME */

	if (strlen(path) > 0 && path[strlen(path) - 1] == '\\') {
    
		fd = -1;

	}
	else {
	  
		file = malloc(sizeof(SMBCFILE));

		if (!file) {

			errno = ENOMEM;
			return NULL;

		}

		ZERO_STRUCTP(file);

		if ((fd = cli_open(&srv->cli, path, flags, DENY_NONE)) < 0) {

			/* Handle the error ... */

			SAFE_FREE(file);
			errno = smbc_errno(context, &srv->cli);
			return NULL;

		}

		/* Fill in file struct */

		file->cli_fd  = fd;
		file->fname   = strdup(fname);
		file->srv     = srv;
		file->offset  = 0;
		file->file    = True;

		DLIST_ADD(context->_files, file);
		return file;

	}

	/* Check if opendir needed ... */

	if (fd == -1) {
		int eno = 0;

		eno = smbc_errno(context, &srv->cli);
		file = context->opendir(context, fname);
		if (!file) errno = eno;
		return file;

	}

	errno = EINVAL; /* FIXME, correct errno ? */
	return NULL;

}

/*
 * Routine to create a file 
 */

static int creat_bits = O_WRONLY | O_CREAT | O_TRUNC; /* FIXME: Do we need this */

static SMBCFILE *smbc_creat_ctx(SMBCCTX *context, const char *path, mode_t mode)
{

	if (!context || !context->_initialized) {

		errno = EINVAL;
		return NULL;

	}

	return smbc_open_ctx(context, path, creat_bits, mode);
}

/*
 * Routine to read() a file ...
 */

static ssize_t smbc_read_ctx(SMBCCTX *context, SMBCFILE *file, void *buf, size_t count)
{
	int ret;

	if (!context || !context->_initialized) {

		errno = EINVAL;
		return -1;

	}

	DEBUG(4, ("smbc_read(%p, %d)\n", file, (int)count));

	if (!file || !DLIST_CONTAINS(context->_files, file)) {

		errno = EBADF;
		return -1;

	}

	/* Check that the buffer exists ... */

	if (buf == NULL) {

		errno = EINVAL;
		return -1;

	}

	ret = cli_read(&file->srv->cli, file->cli_fd, buf, file->offset, count);

	if (ret < 0) {

		errno = smbc_errno(context, &file->srv->cli);
		return -1;

	}

	file->offset += ret;

	DEBUG(4, ("  --> %d\n", ret));

	return ret;  /* Success, ret bytes of data ... */

}

/*
 * Routine to write() a file ...
 */

static ssize_t smbc_write_ctx(SMBCCTX *context, SMBCFILE *file, void *buf, size_t count)
{
	int ret;

	if (!context || context->_initialized) {

		errno = EINVAL;
		return -1;

	}

	if (!file || !DLIST_CONTAINS(context->_files, file)) {

		errno = EBADF;
		return -1;
    
	}

	/* Check that the buffer exists ... */

	if (buf == NULL) {

		errno = EINVAL;
		return -1;

	}

	ret = cli_write(&file->srv->cli, file->cli_fd, 0, buf, file->offset, count);

	if (ret <= 0) {

		errno = smbc_errno(context, &file->srv->cli);
		return -1;

	}

	file->offset += ret;

	return ret;  /* Success, 0 bytes of data ... */
}
 
/*
 * Routine to close() a file ...
 */

static int smbc_close_ctx(SMBCCTX *context, SMBCFILE *file)
{
        SMBCSRV *srv; 

	if (!context || !context->_initialized) {

		errno = EINVAL;
		return -1;

	}

	if (!file || !DLIST_CONTAINS(context->_files, file)) {
   
		errno = EBADF;
		return -1;

	}

	/* IS a dir ... */
	if (!file->file) {
		
		return context->closedir(context, file);

	}

	if (!cli_close(&file->srv->cli, file->cli_fd)) {

		DEBUG(3, ("cli_close failed on %s. purging server.\n", 
			  file->fname));
		/* Deallocate slot and remove the server 
		 * from the server cache if unused */
		errno = smbc_errno(context, &file->srv->cli);  
		srv = file->srv;
		DLIST_REMOVE(context->_files, file);
		SAFE_FREE(file->fname);
		SAFE_FREE(file);
		context->callbacks.remove_unused_server_fn(context, srv);

		return -1;

	}

	if (!file->file) {

		return context->closedir(context, file);

	}

	if (!cli_close(&file->srv->cli, file->cli_fd)) {
		DEBUG(3, ("cli_close failed on %s. purging server.\n", 
			  file->fname));
		/* Deallocate slot and remove the server 
		 * from the server cache if unused */
		errno = smbc_errno(context, &file->srv->cli);  
		srv = file->srv;
		DLIST_REMOVE(context->_files, file);
		SAFE_FREE(file->fname);
		SAFE_FREE(file);
		context->callbacks.remove_unused_server_fn(context, srv);

		return -1;
	}

	DLIST_REMOVE(context->_files, file);
	SAFE_FREE(file->fname);
	SAFE_FREE(file);

	return 0;
}

/*
 * Get info from an SMB server on a file. Use a qpathinfo call first
 * and if that fails, use getatr, as Win95 sometimes refuses qpathinfo
 */
static BOOL smbc_getatr(SMBCCTX * context, SMBCSRV *srv, char *path, 
		 uint16 *mode, size_t *size, 
		 time_t *c_time, time_t *a_time, time_t *m_time,
		 SMB_INO_T *ino)
{

	if (!context || !context->_initialized) {
 
		errno = EINVAL;
 		return -1;
 
 	}

	DEBUG(4,("smbc_getatr: sending qpathinfo\n"));
  
	if (!srv->no_pathinfo2 &&
	    cli_qpathinfo2(&srv->cli, path, c_time, a_time, m_time, NULL,
			   size, mode, ino)) return True;

	/* if this is NT then don't bother with the getatr */
	if (srv->cli.capabilities & CAP_NT_SMBS) return False;

	if (cli_getatr(&srv->cli, path, mode, size, m_time)) {
		a_time = c_time = m_time;
		srv->no_pathinfo2 = True;
		return True;
	}

	return False;

}

/*
 * Routine to unlink() a file
 */

static int smbc_unlink_ctx(SMBCCTX *context, const char *fname)
{
	fstring server, share, user, password, workgroup;
	pstring path;
	SMBCSRV *srv = NULL;

	if (!context || context->_initialized) {

		errno = EINVAL;  /* Best I can think of ... */
		return -1;

	}

	if (!fname) {

		errno = EINVAL;
		return -1;

	}

	smbc_parse_path(context, fname, server, share, path, user, password); /* FIXME, check errors */

	if (user[0] == (char)0) pstrcpy(user, context->user);

	pstrcpy(workgroup, context->workgroup);

	srv = smbc_server(context, server, share, workgroup, user, password);

	if (!srv) {

		return -1;  /* smbc_server sets errno */

	}

	/*  if (strncmp(srv->cli.dev, "LPT", 3) == 0) {

    int job = smbc_stat_printjob(srv, path, NULL, NULL);
    if (job == -1) {

      return -1;

    }
    if ((err = cli_printjob_del(&srv->cli, job)) != 0) {

    
      return -1;

    }
    } else */

	if (!cli_unlink(&srv->cli, path)) {

		errno = smbc_errno(context, &srv->cli);

		if (errno == EACCES) { /* Check if the file is a directory */

			int saverr = errno;
			size_t size = 0;
			uint16 mode = 0;
			time_t m_time = 0, a_time = 0, c_time = 0;
			SMB_INO_T ino = 0;

			if (!smbc_getatr(context, srv, path, &mode, &size,
					 &c_time, &a_time, &m_time, &ino)) {

				/* Hmmm, bad error ... What? */

				errno = smbc_errno(context, &srv->cli);
				return -1;

			}
			else {

				if (IS_DOS_DIR(mode))
					errno = EISDIR;
				else
					errno = saverr;  /* Restore this */

			}
		}

		return -1;

	}

	return 0;  /* Success ... */

}

/*
 * Routine to rename() a file
 */

static int smbc_rename_ctx(SMBCCTX *ocontext, const char *oname, 
			   SMBCCTX *ncontext, const char *nname)
{
	fstring server1, share1, server2, share2, user1, user2, password1, password2, workgroup;
	pstring path1, path2;
	SMBCSRV *srv = NULL;

	if (!ocontext || !ncontext ||
	    !ocontext->_initialized || !ncontext->_initialized) {

		errno = EINVAL;  /* Best I can think of ... */
		return -1;

	}
	
	if (!oname || !nname) {

		errno = EINVAL;
		return -1;

	}
	
	DEBUG(4, ("smbc_rename(%s,%s)\n", oname, nname));

	smbc_parse_path(ocontext, oname, server1, share1, path1, user1, password1);

	if (user1[0] == (char)0) pstrcpy(user1, ocontext->user);

	smbc_parse_path(ncontext, nname, server2, share2, path2, user2, password2);

	if (user2[0] == (char)0) pstrcpy(user2, ncontext->user);

	if (strcmp(server1, server2) || strcmp(share1, share2) ||
	    strcmp(user1, user2)) {

		/* Can't rename across file systems, or users?? */

		errno = EXDEV;
		return -1;

	}

	pstrcpy(workgroup, ocontext->workgroup);
	/* HELP !!! Which workgroup should I use ? Or are they always the same -- Tom */ 
	srv = smbc_server(ocontext, server1, share1, workgroup, user1, password1);
	if (!srv) {

		return -1;

	}

	if (!cli_rename(&srv->cli, path1, path2)) {
		int eno = smbc_errno(ocontext, &srv->cli);

		if (eno != EEXIST ||
		    !cli_unlink(&srv->cli, path2) ||
		    !cli_rename(&srv->cli, path1, path2)) {

			errno = eno;
			return -1;

		}
	}

	return 0; /* Success */

}

/*
 * A routine to lseek() a file
 */

static off_t smbc_lseek_ctx(SMBCCTX *context, SMBCFILE *file, off_t offset, int whence)
{
	size_t size;

	if (!context || !context->_initialized) {

		errno = EINVAL;
		return -1;
		
	}

	if (!file || !DLIST_CONTAINS(context->_files, file)) {

		errno = EBADF;
		return -1;

	}

	if (!file->file) {

		errno = EINVAL;
		return -1;      /* Can't lseek a dir ... */

	}

	switch (whence) {
	case SEEK_SET:
		file->offset = offset;
		break;

	case SEEK_CUR:
		file->offset += offset;
		break;

	case SEEK_END:
		if (!cli_qfileinfo(&file->srv->cli, file->cli_fd, NULL, &size, NULL, NULL,
				   NULL, NULL, NULL) &&
		    !cli_getattrE(&file->srv->cli, file->cli_fd, NULL, &size, NULL, NULL,
				  NULL)) {

			errno = EINVAL;
			return -1;
		}
		file->offset = size + offset;
		break;

	default:
		errno = EINVAL;
		break;

	}

	return file->offset;

}

/* 
 * Generate an inode number from file name for those things that need it
 */

static
ino_t smbc_inode(SMBCCTX *context, const char *name)
{

	if (!context || !context->_initialized) {

		errno = EINVAL;
		return -1;

	}

	if (!*name) return 2; /* FIXME, why 2 ??? */
	return (ino_t)str_checksum(name);

}

/*
 * Routine to put basic stat info into a stat structure ... Used by stat and
 * fstat below.
 */

static
int smbc_setup_stat(SMBCCTX *context, struct stat *st, char *fname, size_t size, int mode)
{
	
	st->st_mode = 0;

	if (IS_DOS_DIR(mode)) {
		st->st_mode = SMBC_DIR_MODE;
	} else {
		st->st_mode = SMBC_FILE_MODE;
	}

	if (IS_DOS_ARCHIVE(mode)) st->st_mode |= S_IXUSR;
	if (IS_DOS_SYSTEM(mode)) st->st_mode |= S_IXGRP;
	if (IS_DOS_HIDDEN(mode)) st->st_mode |= S_IXOTH;
	if (!IS_DOS_READONLY(mode)) st->st_mode |= S_IWUSR;

	st->st_size = size;
	st->st_blksize = 512;
	st->st_blocks = (size+511)/512;
	st->st_uid = getuid();
	st->st_gid = getgid();

	if (IS_DOS_DIR(mode)) {
		st->st_nlink = 2;
	} else {
		st->st_nlink = 1;
	}

	if (st->st_ino == 0) {
		st->st_ino = smbc_inode(context, fname);
	}
	
	return True;  /* FIXME: Is this needed ? */

}

/*
 * Routine to stat a file given a name
 */

static int smbc_stat_ctx(SMBCCTX *context, const char *fname, struct stat *st)
{
	SMBCSRV *srv;
	fstring server, share, user, password, workgroup;
	pstring path;
	time_t m_time = 0, a_time = 0, c_time = 0;
	size_t size = 0;
	uint16 mode = 0;
	SMB_INO_T ino = 0;

	if (!context || !context->_initialized) {

		errno = EINVAL;  /* Best I can think of ... */
		return -1;
    
	}

	if (!fname) {

		errno = EINVAL;
		return -1;

	}
  
	DEBUG(4, ("smbc_stat(%s)\n", fname));

	smbc_parse_path(context, fname, server, share, path, user, password); /*FIXME, errors*/

	if (user[0] == (char)0) pstrcpy(user, context->user);

	pstrcpy(workgroup, context->workgroup);

	srv = smbc_server(context, server, share, workgroup, user, password);

	if (!srv) {

		return -1;  /* errno set by smbc_server */

	}

	/* if (strncmp(srv->cli.dev, "IPC", 3) == 0) {

	   mode = aDIR | aRONLY;

	   }
	   else if (strncmp(srv->cli.dev, "LPT", 3) == 0) {
	   
	   if (strcmp(path, "\\") == 0) {
	   
	   mode = aDIR | aRONLY;

	   }
	   else {

	   mode = aRONLY;
	   smbc_stat_printjob(srv, path, &size, &m_time);
	   c_time = a_time = m_time;

	   }
	   else { */

	if (!smbc_getatr(context, srv, path, &mode, &size, 
			 &c_time, &a_time, &m_time, &ino)) {

		errno = smbc_errno(context, &srv->cli);
		return -1;
		
	}

	st->st_ino = ino;

	smbc_setup_stat(context, st, path, size, mode);

	st->st_atime = a_time;
	st->st_ctime = c_time;
	st->st_mtime = m_time;
	st->st_dev   = srv->dev;

	return 0;

}

/*
 * Routine to stat a file given an fd
 */

static int smbc_fstat_ctx(SMBCCTX *context, SMBCFILE *file, struct stat *st)
{
	time_t c_time, a_time, m_time;
	size_t size;
	uint16 mode;
	SMB_INO_T ino = 0;

	if (!context || !context->_initialized) {

		errno = EINVAL;
		return -1;

	}

	if (!file || !DLIST_CONTAINS(context->_files, file)) {

		errno = EBADF;
		return -1;

	}

	if (!file->file) {

		return context->fstatdir(context, file, st);

	}

	if (!cli_qfileinfo(&file->srv->cli, file->cli_fd,
			   &mode, &size, &c_time, &a_time, &m_time, NULL, &ino) &&
	    !cli_getattrE(&file->srv->cli, file->cli_fd,
			  &mode, &size, &c_time, &a_time, &m_time)) {

		errno = EINVAL;
		return -1;

	}

	st->st_ino = ino;

	smbc_setup_stat(context, st, file->fname, size, mode);

	st->st_atime = a_time;
	st->st_ctime = c_time;
	st->st_mtime = m_time;
	st->st_dev = file->srv->dev;

	return 0;

}

/*
 * Routine to open a directory
 *
 * We want to allow:
 *
 * smb: which should list all the workgroups available
 * smb:workgroup
 * smb:workgroup//server
 * smb://server
 * smb://server/share
 * smb://<IP-addr> which should list shares on server
 * smb://<IP-addr>/share which should list files on share
 */

static void smbc_remove_dir(SMBCFILE *dir)
{
	struct smbc_dir_list *d,*f;

	d = dir->dir_list;
	while (d) {

		f = d; d = d->next;

		SAFE_FREE(f->dirent);
		SAFE_FREE(f);

	}

	dir->dir_list = dir->dir_end = dir->dir_next = NULL;

}

static int add_dirent(SMBCFILE *dir, const char *name, const char *comment, uint32 type)
{
	struct smbc_dirent *dirent;
	int size;

	/*
	 * Allocate space for the dirent, which must be increased by the 
	 * size of the name and the comment and 1 for the null on the comment.
	 * The null on the name is already accounted for.
	 */

	size = sizeof(struct smbc_dirent) + (name?strlen(name):0) +
		(comment?strlen(comment):0) + 1; 
    
	dirent = malloc(size);

	if (!dirent) {

		dir->dir_error = ENOMEM;
		return -1;

	}

	ZERO_STRUCTP(dirent);

	ZERO_STRUCTP(dirent);


	if (dir->dir_list == NULL) {

		dir->dir_list = malloc(sizeof(struct smbc_dir_list));
		if (!dir->dir_list) {

			SAFE_FREE(dirent);
			dir->dir_error = ENOMEM;
			return -1;

		}
		ZERO_STRUCTP(dir->dir_list);

		dir->dir_end = dir->dir_next = dir->dir_list;
  
	}
	else {

		dir->dir_end->next = malloc(sizeof(struct smbc_dir_list));
		
		if (!dir->dir_end->next) {
			
			SAFE_FREE(dirent);
			dir->dir_error = ENOMEM;
			return -1;

		}
		ZERO_STRUCTP(dir->dir_end->next);

		dir->dir_end = dir->dir_end->next;

	}

	dir->dir_end->next = NULL;
	dir->dir_end->dirent = dirent;
	
	dirent->smbc_type = type;
	dirent->namelen = (name?strlen(name):0);
	dirent->commentlen = (comment?strlen(comment):0);
	dirent->dirlen = size;
  
	strncpy(dirent->name, (name?name:""), dirent->namelen + 1);

	dirent->comment = (char *)(&dirent->name + dirent->namelen + 1);
	strncpy(dirent->comment, (comment?comment:""), dirent->commentlen + 1);

	return 0;

}

static void
list_fn(const char *name, uint32 type, const char *comment, void *state)
{
	SMBCFILE *dir = (SMBCFILE *)state;
	int dirent_type;

	/* We need to process the type a little ... */

	if (dir->dir_type == SMBC_FILE_SHARE) {
		
		switch (type) {
		case 0: /* Directory tree */
			dirent_type = SMBC_FILE_SHARE;
			break;

		case 1:
			dirent_type = SMBC_PRINTER_SHARE;
			break;

		case 2:
			dirent_type = SMBC_COMMS_SHARE;
			break;

		case 3:
			dirent_type = SMBC_IPC_SHARE;
			break;

		default:
			dirent_type = SMBC_FILE_SHARE; /* FIXME, error? */
			break;
		}
		ZERO_STRUCTP(dir->dir_list);

	}
	else dirent_type = dir->dir_type;

	if (add_dirent(dir, name, comment, dirent_type) < 0) {

		/* An error occurred, what do we do? */
		/* FIXME: Add some code here */

	}

}

static void
dir_list_fn(file_info *finfo, const char *mask, void *state)
{

	if (add_dirent((SMBCFILE *)state, finfo->name, "", 
		       (finfo->mode&aDIR?SMBC_DIR:SMBC_FILE)) < 0) {

		/* Handle an error ... */

		/* FIXME: Add some code ... */

	} 

}

static SMBCFILE *smbc_opendir_ctx(SMBCCTX *context, const char *fname)
{
	fstring server, share, user, password, workgroup;
	pstring path;
	SMBCSRV *srv  = NULL;
	SMBCFILE *dir = NULL;
	struct in_addr rem_ip;
	int slot = 0;

	if (!context || !context->_initialized) {

		errno = EINVAL;
		return NULL;

	}

	if (!fname) {
    
		errno = EINVAL;
		return NULL;

	}

	if (smbc_parse_path(context, fname, server, share, path, user, password)) {

		errno = EINVAL;
		return NULL;

	}

	if (user[0] == (char)0) pstrcpy(user, context->user);

	pstrcpy(workgroup, context->workgroup);

	dir = malloc(sizeof(*dir));

	if (!dir) {

		errno = ENOMEM;
		return NULL;

	}

	ZERO_STRUCTP(dir);

	dir->cli_fd   = 0;
	dir->fname    = strdup(fname);
	dir->srv      = NULL;
	dir->offset   = 0;
	dir->file     = False;
	dir->dir_list = dir->dir_next = dir->dir_end = NULL;

	if (server[0] == (char)0) {

		if (share[0] != (char)0 || path[0] != (char)0) {
    
			errno = EINVAL;
			if (dir) {
				SAFE_FREE(dir->fname);
				SAFE_FREE(dir);
			}
			return NULL;

		}

		/* We have server and share and path empty ... so list the workgroups */
                /* first try to get the LMB for our workgroup, and if that fails,     */
                /* try the DMB                                                        */

		if (!(resolve_name(context->workgroup, &rem_ip, 0x1d) ||
                      resolve_name(context->workgroup, &rem_ip, 0x1b))) {
      
			errno = EINVAL;  /* Something wrong with smb.conf? */
			return NULL;

		}

		dir->dir_type = SMBC_WORKGROUP;

		/* find the name of the server ... */

		if (!name_status_find("*", 0, 0, rem_ip, server)) {

			DEBUG(0,("Could not get the name of local/domain master browser for server %s\n", server));
			errno = EINVAL;
			return NULL;

		}

		/*
		 * Get a connection to IPC$ on the server if we do not already have one
		 */

		srv = smbc_server(context, server, "IPC$", workgroup, user, password);

		if (!srv) {

			if (dir) {
				SAFE_FREE(dir->fname);
				SAFE_FREE(dir);
			}
			
			return NULL;

		}
		ZERO_STRUCTP(dir->dir_end);

		dir->srv = srv;

		/* Now, list the stuff ... */

		if (!cli_NetServerEnum(&srv->cli, workgroup, 0x80000000, list_fn,
				       (void *)dir)) {

			if (dir) {
				SAFE_FREE(dir->fname);
				SAFE_FREE(dir);
			}
			errno = cli_errno(&srv->cli);

			return NULL;

		}
	}
	else { /* Server not an empty string ... Check the rest and see what gives */

		if (share[0] == (char)0) {

			if (path[0] != (char)0) { /* Should not have empty share with path */

				errno = EINVAL;
				if (dir) {
					SAFE_FREE(dir->fname);
					SAFE_FREE(dir);
				}
				return NULL;
	
			}

			/* Check to see if <server><1D>, <server><1B>, or <server><20> translates */
			/* However, we check to see if <server> is an IP address first */

			if (!is_ipaddress(server) &&  /* Not an IP addr so check next */
			    (resolve_name(server, &rem_ip, 0x1d) ||   /* Found LMB */
                                    resolve_name(server, &rem_ip, 0x1b) )) { /* Found DMB */
				pstring buserver;

				dir->dir_type = SMBC_SERVER;

				/*
				 * Get the backup list ...
				 */


				if (!name_status_find("*", 0, 0, rem_ip, buserver)) {

					DEBUG(0, ("Could not get name of local/domain master browser for server %s\n", server));
					errno = EPERM;  /* FIXME, is this correct */
					return NULL;

				}

				/*
				 * Get a connection to IPC$ on the server if we do not already have one
				 */

				srv = smbc_server(context, buserver, "IPC$", workgroup, user, password);

				if (!srv) {

					if (dir) {
						SAFE_FREE(dir->fname);
						SAFE_FREE(dir);
					}
					return NULL;

				}

				dir->srv = srv;

				/* Now, list the servers ... */

				if (!cli_NetServerEnum(&srv->cli, server, 0x0000FFFE, list_fn,
						       (void *)dir)) {

					if (dir) {
						SAFE_FREE(dir->fname);
						SAFE_FREE(dir);
					}
					errno = cli_errno(&srv->cli);
					return NULL;
					
				}

			}
			else {

				if (resolve_name(server, &rem_ip, 0x20)) {

					/* Now, list the shares ... */

					dir->dir_type = SMBC_FILE_SHARE;

					srv = smbc_server(context, server, "IPC$", workgroup, user, password);

					if (!srv) {

						if (dir) {
							SAFE_FREE(dir->fname);
							SAFE_FREE(dir);
						}
						return NULL;

					}

					dir->srv = srv;

					/* Now, list the servers ... */

					if (cli_RNetShareEnum(&srv->cli, list_fn, 
							      (void *)dir) < 0) {

						errno = cli_errno(&srv->cli);
						if (dir) {
							SAFE_FREE(dir->fname);
							SAFE_FREE(dir);
						}
						return NULL;

					}

				}
				else {

					errno = ENODEV;   /* Neither the workgroup nor server exists */
					if (dir) {
						SAFE_FREE(dir->fname);
						SAFE_FREE(dir);
					}
					return NULL;

				}

			}

		}
		else { /* The server and share are specified ... work from there ... */

			/* Well, we connect to the server and list the directory */

			dir->dir_type = SMBC_FILE_SHARE;

			srv = smbc_server(context, server, share, workgroup, user, password);

			if (!srv) {

				if (dir) {
					SAFE_FREE(dir->fname);
					SAFE_FREE(dir);
				}
				return NULL;

			}

			dir->srv = srv;

			/* Now, list the files ... */

			pstrcat(path, "\\*");

			if (cli_list(&srv->cli, path, aDIR | aSYSTEM | aHIDDEN, dir_list_fn, 
				     (void *)dir) < 0) {

				if (dir) {
					SAFE_FREE(dir->fname);
					SAFE_FREE(dir);
				}
				errno = smbc_errno(context, &srv->cli);
				return NULL;

			}
		}

	}

	DLIST_ADD(context->_files, dir);
	return dir;

}

/*
 * Routine to close a directory
 */

static int smbc_closedir_ctx(SMBCCTX *context, SMBCFILE *dir)
{

	if (!context || !context->_initialized) {

		errno = EINVAL;
		return -1;

	}

	if (!dir || !DLIST_CONTAINS(context->_files, dir)) {

		errno = EBADF;
		return -1;
    
	}

	smbc_remove_dir(dir); /* Clean it up */

	DLIST_REMOVE(context->_files, dir);

	if (dir) {

		SAFE_FREE(dir->fname);
		SAFE_FREE(dir);    /* Free the space too */

	}

	return 0;

}

/*
 * Routine to get a directory entry
 */

struct smbc_dirent *smbc_readdir_ctx(SMBCCTX *context, SMBCFILE *dir)
{
	struct smbc_dirent *dirp, *dirent;

	/* Check that all is ok first ... */

	if (!context || !context->_initialized) {

		errno = EINVAL;
		return NULL;

	}

	if (!dir || !DLIST_CONTAINS(context->_files, dir)) {

		errno = EBADF;
		return NULL;

	}

	if (dir->file != False) { /* FIXME, should be dir, perhaps */

		errno = ENOTDIR;
		return NULL;

	}

	if (!dir->dir_next)
		return NULL;
	else {

		dirent = dir->dir_next->dirent;

		if (!dirent) {

			errno = ENOENT;
			return NULL;

		}

		/* Hmmm, do I even need to copy it? */

		memcpy(context->_dirent, dirent, dirent->dirlen); /* Copy the dirent */
		dirp = (struct smbc_dirent *)context->_dirent;
		dirp->comment = (char *)(&dirp->name + dirent->namelen + 1);
		dir->dir_next = dir->dir_next->next;

		return (struct smbc_dirent *)context->_dirent;
	}

}

/*
 * Routine to get directory entries
 */

static int smbc_getdents_ctx(SMBCCTX *context, SMBCFILE *dir, struct smbc_dirent *dirp, int count)
{
	struct smbc_dir_list *dirlist;
	int rem = count, reqd;
	char *ndir = (char *)dirp;

	/* Check that all is ok first ... */

	if (!context || !context->_initialized) {

		errno = EINVAL;
		return -1;

	}

	if (!dir || !DLIST_CONTAINS(context->_files, dir)) {

		errno = EBADF;
		return -1;
    
	}

	if (dir->file != False) { /* FIXME, should be dir, perhaps */

		errno = ENOTDIR;
		return -1;

	}

	/* 
	 * Now, retrieve the number of entries that will fit in what was passed
	 * We have to figure out if the info is in the list, or we need to 
	 * send a request to the server to get the info.
	 */

	while ((dirlist = dir->dir_next)) {
		struct smbc_dirent *dirent;

		if (!dirlist->dirent) {

			errno = ENOENT;  /* Bad error */
			return -1;

		}

		if (rem < (reqd = (sizeof(struct smbc_dirent) + dirlist->dirent->namelen + 
				   dirlist->dirent->commentlen + 1))) {

			if (rem < count) { /* We managed to copy something */

				errno = 0;
				return count - rem;

			}
			else { /* Nothing copied ... */

				errno = EINVAL;  /* Not enough space ... */
				return -1;

			}

		}

		dirent = dirlist->dirent;

		memcpy(ndir, dirent, reqd); /* Copy the data in ... */
    
		((struct smbc_dirent *)ndir)->comment = 
			(char *)(&((struct smbc_dirent *)ndir)->name + dirent->namelen + 1);

		ndir += reqd;

		rem -= reqd;

		dir->dir_next = dirlist = dirlist -> next;
	}

	if (rem == count)
		return 0;
	else 
		return count - rem;

}

/*
 * Routine to create a directory ...
 */

static int smbc_mkdir_ctx(SMBCCTX *context, const char *fname, mode_t mode)
{
	SMBCSRV *srv;
	fstring server, share, user, password, workgroup;
	pstring path;

	if (!context || !context->_initialized) {

		errno = EINVAL;
		return -1;

	}

	if (!fname) {

		errno = EINVAL;
		return -1;

	}
  
	DEBUG(4, ("smbc_mkdir(%s)\n", fname));

	smbc_parse_path(context, fname, server, share, path, user, password); /*FIXME, errors*/

	if (user[0] == (char)0) pstrcpy(user, context->user);

	pstrcpy(workgroup, context->workgroup);

	srv = smbc_server(context, server, share, workgroup, user, password);

	if (!srv) {

		return -1;  /* errno set by smbc_server */

	}

	/* if (strncmp(srv->cli.dev, "IPC", 3) == 0) {

	   mode = aDIR | aRONLY;

	   }
	   else if (strncmp(srv->cli.dev, "LPT", 3) == 0) {

	   if (strcmp(path, "\\") == 0) {

	   mode = aDIR | aRONLY;

	   }
	   else {

	   mode = aRONLY;
	   smbc_stat_printjob(srv, path, &size, &m_time);
	   c_time = a_time = m_time;

	   }
	   else { */

	if (!cli_mkdir(&srv->cli, path)) {

		errno = smbc_errno(context, &srv->cli);
		return -1;

	} 

	return 0;

}

/*
 * Our list function simply checks to see if a directory is not empty
 */

static int smbc_rmdir_dirempty = True;

static void rmdir_list_fn(file_info *finfo, const char *mask, void *state)
{

	if (strncmp(finfo->name, ".", 1) != 0 && strncmp(finfo->name, "..", 2) != 0)
		smbc_rmdir_dirempty = False;

}

/*
 * Routine to remove a directory
 */

static int smbc_rmdir_ctx(SMBCCTX *context, const char *fname)
{
	SMBCSRV *srv;
	fstring server, share, user, password, workgroup;
	pstring path;

	if (!context || !context->_initialized) {

		errno = EINVAL;
		return -1;

	}

	if (!fname) {

		errno = EINVAL;
		return -1;

	}
  
	DEBUG(4, ("smbc_rmdir(%s)\n", fname));

	smbc_parse_path(context, fname, server, share, path, user, password); /*FIXME, errors*/

	if (user[0] == (char)0) pstrcpy(user, context->user);

	pstrcpy(workgroup, context->workgroup);

	srv = smbc_server(context, server, share, workgroup, user, password);

	if (!srv) {

		return -1;  /* errno set by smbc_server */

	}

	/* if (strncmp(srv->cli.dev, "IPC", 3) == 0) {

	   mode = aDIR | aRONLY;

	   }
	   else if (strncmp(srv->cli.dev, "LPT", 3) == 0) {

	   if (strcmp(path, "\\") == 0) {

	   mode = aDIR | aRONLY;

	   }
	   else {

	   mode = aRONLY;
	   smbc_stat_printjob(srv, path, &size, &m_time);
	   c_time = a_time = m_time;
	   
	   }
	   else { */

	if (!cli_rmdir(&srv->cli, path)) {

		errno = smbc_errno(context, &srv->cli);

		if (errno == EACCES) {  /* Check if the dir empty or not */

			pstring lpath; /* Local storage to avoid buffer overflows */

			smbc_rmdir_dirempty = True;  /* Make this so ... */

			pstrcpy(lpath, path);
			pstrcat(lpath, "\\*");

			if (cli_list(&srv->cli, lpath, aDIR | aSYSTEM | aHIDDEN, rmdir_list_fn,
				     NULL) < 0) {

				/* Fix errno to ignore latest error ... */

				DEBUG(5, ("smbc_rmdir: cli_list returned an error: %d\n", 
					  smbc_errno(context, &srv->cli)));
				errno = EACCES;

			}

			if (smbc_rmdir_dirempty)
				errno = EACCES;
			else
				errno = ENOTEMPTY;

		}

		return -1;

	} 

	return 0;

}

/*
 * Routine to return the current directory position
 */

static off_t smbc_telldir_ctx(SMBCCTX *context, SMBCFILE *dir)
{

	if (!context || !context->_initialized) {

		errno = EINVAL;
		return -1;

	}

	if (!dir || !DLIST_CONTAINS(context->_files, dir)) {

		errno = EBADF;
		return -1;

	}

	if (dir->file != False) { /* FIXME, should be dir, perhaps */

		errno = ENOTDIR;
		return -1;

	}

	return (off_t) dir->dir_next;

}

/*
 * A routine to run down the list and see if the entry is OK
 */

struct smbc_dir_list *smbc_check_dir_ent(struct smbc_dir_list *list, 
					 struct smbc_dirent *dirent)
{

	/* Run down the list looking for what we want */

	if (dirent) {

		struct smbc_dir_list *tmp = list;

		while (tmp) {

			if (tmp->dirent == dirent)
				return tmp;

			tmp = tmp->next;

		}

	}

	return NULL;  /* Not found, or an error */

}


/*
 * Routine to seek on a directory
 */

static int smbc_lseekdir_ctx(SMBCCTX *context, SMBCFILE *dir, off_t offset)
{
	struct smbc_dirent *dirent = (struct smbc_dirent *)offset;
	struct smbc_dir_list *list_ent = NULL;

	if (!context || !context->_initialized) {

		errno = EINVAL;
		return -1;

	}

	if (dir->file != False) { /* FIXME, should be dir, perhaps */

		errno = ENOTDIR;
		return -1;

	}

	/* Now, check what we were passed and see if it is OK ... */

	if (dirent == NULL) {  /* Seek to the begining of the list */

		dir->dir_next = dir->dir_list;
		return 0;

	}

	/* Now, run down the list and make sure that the entry is OK       */
	/* This may need to be changed if we change the format of the list */

	if ((list_ent = smbc_check_dir_ent(dir->dir_list, dirent)) == NULL) {

		errno = EINVAL;   /* Bad entry */
		return -1;

	}

	dir->dir_next = list_ent;

	return 0; 

}

/*
 * Routine to fstat a dir
 */

static int smbc_fstatdir_ctx(SMBCCTX *context, SMBCFILE *dir, struct stat *st)
{

	if (context || !context->_initialized) {

		errno = EINVAL;
		return -1;

	}

	/* No code yet ... */

	return 0;

}

/*
 * Open a print file to be written to by other calls
 */

static SMBCFILE *smbc_open_print_job_ctx(SMBCCTX *context, const char *fname)
{
	fstring server, share, user, password;
	pstring path;
	
	if (!context || context->_initialized) {

		errno = EINVAL;
		return NULL;
    
	}

	if (!fname) {

		errno = EINVAL;
		return NULL;

	}
  
	DEBUG(4, ("smbc_open_print_job_ctx(%s)\n", fname));

	smbc_parse_path(context, fname, server, share, path, user, password); /*FIXME, errors*/

	/* What if the path is empty, or the file exists? */

	return context->open(context, fname, O_WRONLY, 666);

}

/*
 * Routine to print a file on a remote server ...
 *
 * We open the file, which we assume to be on a remote server, and then
 * copy it to a print file on the share specified by printq.
 */

static int smbc_print_file_ctx(SMBCCTX *c_file, const char *fname, SMBCCTX *c_print, const char *printq)
{
        SMBCFILE *fid1, *fid2;
	int bytes, saverr, tot_bytes = 0;
	char buf[4096];

	if (!c_file || !c_file->_initialized || !c_print ||
	    !c_print->_initialized) {

		errno = EINVAL;
		return -1;

	}

	if (!fname && !printq) {

		errno = EINVAL;
		return -1;

	}

	/* Try to open the file for reading ... */

	if ((fid1 = c_file->open(c_file, fname, O_RDONLY, 0666)) < 0) {
		
		DEBUG(3, ("Error, fname=%s, errno=%i\n", fname, errno));
		return -1;  /* smbc_open sets errno */
		
	}

	/* Now, try to open the printer file for writing */

	if ((fid2 = c_print->open_print_job(c_print, printq)) < 0) {

		saverr = errno;  /* Save errno */
		c_file->close(c_file, fid1);
		errno = saverr;
		return -1;

	}

	while ((bytes = c_file->read(c_file, fid1, buf, sizeof(buf))) > 0) {

		tot_bytes += bytes;

		if ((c_print->write(c_print, fid2, buf, bytes)) < 0) {

			saverr = errno;
			c_file->close(c_file, fid1);
			c_print->close(c_print, fid2);
			errno = saverr;

		}

	}

	saverr = errno;

	c_file->close(c_file, fid1);  /* We have to close these anyway */
	c_print->close(c_print, fid2);

	if (bytes < 0) {

		errno = saverr;
		return -1;

	}

	return tot_bytes;

}

/*
 * Routine to list print jobs on a printer share ...
 */

static int smbc_list_print_jobs_ctx(SMBCCTX *context, const char *fname, void (*fn)(struct print_job_info *))
{
	SMBCSRV *srv;
	fstring server, share, user, password, workgroup;
	pstring path;

	if (!context || !context->_initialized) {

		errno = EINVAL;
		return -1;

	}

	if (!fname) {
		
		errno = EINVAL;
		return -1;

	}
  
	DEBUG(4, ("smbc_list_print_jobs(%s)\n", fname));

	smbc_parse_path(context, fname, server, share, path, user, password); /*FIXME, errors*/

	if (user[0] == (char)0) pstrcpy(user, context->user);
	
	pstrcpy(workgroup, context->workgroup);

	srv = smbc_server(context, server, share, workgroup, user, password);

	if (!srv) {

		return -1;  /* errno set by smbc_server */

	}

	if (cli_print_queue(&srv->cli, fn) < 0) {

		errno = smbc_errno(context, &srv->cli);
		return -1;

	}
	
	return 0;

}

/*
 * Delete a print job from a remote printer share
 */

static int smbc_unlink_print_job_ctx(SMBCCTX *context, const char *fname, int id)
{
	SMBCSRV *srv;
	fstring server, share, user, password, workgroup;
	pstring path;
	int err;

	if (!context || !context->_initialized) {

		errno = EINVAL;
		return -1;

	}

	if (!fname) {

		errno = EINVAL;
		return -1;

	}
  
	DEBUG(4, ("smbc_unlink_print_job(%s)\n", fname));

	smbc_parse_path(context, fname, server, share, path, user, password); /*FIXME, errors*/

	if (user[0] == (char)0) pstrcpy(user, context->user);

	pstrcpy(workgroup, context->workgroup);

	srv = smbc_server(context, server, share, workgroup, user, password);

	if (!srv) {

		return -1;  /* errno set by smbc_server */

	}

	if ((err = cli_printjob_del(&srv->cli, id)) != 0) {

		if (err < 0)
			errno = smbc_errno(context, &srv->cli);
		else if (err == ERRnosuchprintjob)
			errno = EINVAL;
		return -1;

	}

	return 0;

}

/*
 * Get a new empty handle to fill in with your own info 
 */
SMBCCTX * smbc_new_context(void)
{
	SMBCCTX * context;

	context = malloc(sizeof(*context));
	if (!context) {
		errno = ENOMEM;
		return NULL;
	}
	
	ZERO_STRUCTP(context);

	/* ADD REASONABLE DEFAULTS */
	context->debug            = 0;
	context->timeout          = 20000; /* 20 seconds */

	context->open             = smbc_open_ctx;
	context->creat            = smbc_creat_ctx;
	context->read             = smbc_read_ctx;
	context->write            = smbc_write_ctx;
	context->close            = smbc_close_ctx;
	context->unlink           = smbc_unlink_ctx;
	context->rename           = smbc_rename_ctx;
	context->lseek            = smbc_lseek_ctx;
	context->stat             = smbc_stat_ctx;
	context->fstat            = smbc_fstat_ctx;
	context->opendir          = smbc_opendir_ctx;
	context->closedir         = smbc_closedir_ctx;
	context->readdir          = smbc_readdir_ctx;
	context->getdents         = smbc_getdents_ctx;
	context->mkdir            = smbc_mkdir_ctx;
	context->rmdir            = smbc_rmdir_ctx;
	context->telldir          = smbc_telldir_ctx;
	context->lseekdir         = smbc_lseekdir_ctx;
	context->fstatdir         = smbc_fstatdir_ctx;
	context->open_print_job   = smbc_open_print_job_ctx;
	context->print_file       = smbc_print_file_ctx;
	context->list_print_jobs  = smbc_list_print_jobs_ctx;
	context->unlink_print_job = smbc_unlink_print_job_ctx;

	context->callbacks.check_server_fn      = smbc_check_server;
	context->callbacks.remove_unused_server_fn = smbc_remove_unused_server;

	smbc_default_cache_functions(context);

	return context;
}

/* 
 * Free a context
 *
 * Returns 0 on success. Otherwise returns 1, the SMBCCTX is _not_ freed 
 * and thus you'll be leaking memory if not handled properly.
 *
 */
int smbc_free_context(SMBCCTX * context, int shutdown_ctx)
{
	if (!context) {
		errno = EBADF;
		return 1;
	}
	
	if (shutdown_ctx) {
		SMBCFILE * f;
		DEBUG(1,("Performing aggressive shutdown.\n"));
		
		f = context->_files;
		while (f) {
			context->close(context, f);
			f = f->next;
		}
		context->_files = NULL;

		/* First try to remove the servers the nice way. */
		if (context->callbacks.purge_cached_fn(context)) {
			SMBCSRV * s;
			DEBUG(1, ("Could not purge all servers, Nice way shutdown failed.\n"));
			s = context->_servers;
			while (s) {
				cli_shutdown(&s->cli);
				context->callbacks.remove_cached_srv_fn(context, s);
				SAFE_FREE(s);
				s = s->next;
			}
			context->_servers = NULL;
		}
	}
	else {
		/* This is the polite way */	
		if (context->callbacks.purge_cached_fn(context)) {
			DEBUG(1, ("Could not purge all servers, free_context failed.\n"));
			errno = EBUSY;
			return 1;
		}
		if (context->_servers) {
			DEBUG(1, ("Active servers in context, free_context failed.\n"));
			errno = EBUSY;
			return 1;
		}
		if (context->_files) {
			DEBUG(1, ("Active files in context, free_context failed.\n"));
			errno = EBUSY;
			return 1;
		}		
	}

	/* Things we have to clean up */
	SAFE_FREE(context->workgroup);
	SAFE_FREE(context->netbios_name);
	SAFE_FREE(context->user);
	
	DEBUG(3, ("Context %p succesfully freed\n", context));
	SAFE_FREE(context);
	return 0;
}


/*
 * Initialise the library etc 
 *
 * We accept a struct containing handle information.
 * valid values for info->debug from 0 to 100,
 * and insist that info->fn must be non-null.
 */
SMBCCTX * smbc_init_context(SMBCCTX * context)
{
	pstring conf;
	int pid;
	char *user = NULL, *home = NULL;

	if (!context) {
		errno = EBADF;
		return NULL;
	}

	/* Do not initialise the same client twice */
	if (context->_initialized) { 
		return 0;
	}

	if (!context->callbacks.auth_fn || context->debug < 0 || context->debug > 100) {

		errno = EINVAL;
		return NULL;

	}

	if (!smbc_initialized) {
		/* Do some library wide intialisations the first time we get called */

		/* Do we still need this ? */
		DEBUGLEVEL = 10;
		
		setup_logging( "libsmbclient", False);

		/* Here we would open the smb.conf file if needed ... */
		
		home = getenv("HOME");

		slprintf(conf, sizeof(conf), "%s/.smb/smb.conf", home);
		
		load_interfaces();  /* Load the list of interfaces ... */
		
		in_client = True; /* FIXME, make a param */

		if (!lp_load(conf, True, False, False)) {

			/*
			 * Hmmm, what the hell do we do here ... we could not parse the
			 * config file ... We must return an error ... and keep info around
			 * about why we failed
			 */
			
			errno = ENOENT; /* FIXME: Figure out the correct error response */
			return NULL;
		}

		reopen_logs();  /* Get logging working ... */
	
		/* 
		 * Block SIGPIPE (from lib/util_sock.c: write())  
		 * It is not needed and should not stop execution 
		 */
		BlockSignals(True, SIGPIPE);
		
		/* Done with one-time initialisation */
		smbc_initialized = 1; 

	}
	
	if (!context->user) {
		/*
		 * FIXME: Is this the best way to get the user info? 
		 */
		user = getenv("USER");
		/* walk around as "guest" if no username can be found */
		if (!user) context->user = strdup("guest");
		else context->user = strdup(user);
	}

	if (!context->netbios_name) {
		/*
		 * We try to get our netbios name from the config. If that fails we fall
		 * back on constructing our netbios name from our hostname etc
		 */
		if (global_myname) {
			context->netbios_name = strdup(global_myname);
		}
		else {
			/*
			 * Hmmm, I want to get hostname as well, but I am too lazy for the moment
			 */
			pid = sys_getpid();
			context->netbios_name = malloc(17);
			if (!context->netbios_name) {
				errno = ENOMEM;
				return NULL;
			}
			slprintf(context->netbios_name, 16, "smbc%s%d", context->user, pid);
		}
	}
	DEBUG(0,("Using netbios name %s.\n", context->netbios_name));
	

	if (!context->workgroup) {
		if (lp_workgroup()) {
			context->workgroup = strdup(lp_workgroup());
		}
		else {
			/* TODO: Think about a decent default workgroup */
			context->workgroup = strdup("samba");
		}
	}
	DEBUG(0,("Using workgroup %s.\n", context->workgroup));
					
	/* shortest timeout is 1 second */
	if (context->timeout > 0 && context->timeout < 1000) 
		context->timeout = 1000;

	/*
	 * FIXME: Should we check the function pointers here? 
	 */

	context->_initialized = 1;
	
	return context;
}
