"======================================================================
|
|   LC_NUMERIC and LC_MONETARY support
|
|   $Revision: 1.95.1$
|   $Date: 2000/12/27 10:45:49$
|   $Author: pb$
|
 ======================================================================"


"======================================================================
|
| Copyright 1988-92, 1994-95, 1999, 2000 Free Software Foundation, Inc.
| Written by Paolo Bonzini.
|
| This file is part of the GNU Smalltalk class library.
|
| The GNU Smalltalk class library is free software; you can redistribute it
| and/or modify it under the terms of the GNU Lesser General Public License
| as published by the Free Software Foundation; either version 2.1, or (at
| your option) any later version.
| 
| The GNU Smalltalk class library is distributed in the hope that it will be
| useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
| MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser
| General Public License for more details.
| 
| You should have received a copy of the GNU Lesser General Public License
| along with the GNU Smalltalk class library; see the file COPYING.LESSER.
| If not, write to the Free Software Foundation, 59 Temple Place - Suite
| 330, Boston, MA 02111-1307, USA.  
|
 ======================================================================"

LcPrintFormats subclass: #LcNumeric
    instanceVariableNames: 'decimalPoint thousandsSep grouping'
    classVariableNames: ''
    poolDictionaries: ''
    category: 'i18n-Printing'!

LcNumeric comment: 
'Sending either #?, #printString: or #print:on: converts a Number to
a String according to the rules that are used in the given locale.'!

LcNumeric subclass: #LcMonetary
    instanceVariableNames: 'currencySymbol positiveSign negativeSign fracDigits pCsPrecedes pSepBySpace nCsPrecedes nSepBySpace pSignPosn nSignPosn'
    classVariableNames: ''
    poolDictionaries: ''
    category: 'i18n-Printing'!

LcMonetary comment: 
'Sending either #?, #printString: or #print:on: converts a Number to
a String according to the rules that are used in the given locale for
printing currency amounts.'!

LcMonetary subclass: #LcMonetaryISO
    instanceVariableNames: ''
    classVariableNames: ''
    poolDictionaries: ''
    category: 'i18n-Printing'!

LcMonetaryISO comment: 
'Sending either #?, #printString: or #print:on: converts a Number to
a String according to the ISO rules for printing currency amounts;
the locale specifies the currency symbol and the number of fractional
digits.'!

!LcNumeric class methodsFor: 'accessing'!

category
    ^#'LC_NUMERIC'!
    
bigEndianID
    ^#[16r96 16r06 16r17 16rDF]!

littleEndianID
    ^#[16rDF 16r17 16r06 16r96]! !

!LcNumeric methodsFor: 'printing'!

basicPrint: aNumber on: aStream
    | n nonLocalized stream decimal |
    self readData.
    nonLocalized := aNumber asBasicNumberType printString.
    decimal := nonLocalized indexOf: $. ifAbsent: [ nonLocalized size + 1 ].

    stream := ReadWriteStream on: nonLocalized.
    stream position: decimal; truncate; reset.
    self printIntegerPart: stream size: decimal - 1 on: aStream.
    self printFractionalPart: nonLocalized startingAt: decimal + 1 on: aStream!

print: aNumber on: aStream
    self basicPrint: aNumber on: aStream! !

!LcNumeric methodsFor: 'private'!

printIntegerPart: stream size: size on: aStream
    | groupings left |
    (thousandsSep isEmpty or: [ grouping isEmpty ])
	ifTrue: [ aStream nextPutAll: stream contents. ^self ].

    groupings := self computeGroupSizes: size.
    left := size.
    groupings reverseDo: [ :num |
	left := left - num.
	num timesRepeat: [ aStream nextPut: stream next ].
	left > 0 ifTrue: [ aStream nextPutAll: thousandsSep ].
    ]!

computeGroupSizes: size
    | howMany foundZero n left |
    howMany := self computeNumberOfGroups: size.
    left := size.

    howMany = 0 ifTrue: [ ^Array with: size ].

    ^(1 to: howMany) collect: [ :index || next |
	index > grouping size ifFalse: [ n := grouping at: index ].
	n = 255 ifTrue: [ n := left ].
	left := left - n.
	left < 0 ifTrue: [ left + n ] ifFalse: [ n ]
    ].
!

computeNumberOfGroups: size
    | left last n |
    left := size.
    n := 0.
    grouping isEmpty ifTrue: [ ^0 ].
    grouping do: [ :each |
	n := n + 1.
        each >= 255 ifTrue: [ ^n ].
	left < each ifTrue: [ ^n ].
	left := left - each.
	last := each.
    ].
    ^n + (left // last)!

printFractionalPart: string startingAt: decimal on: aStream
    decimal > string size ifTrue: [ ^self ].
    aStream nextPutAll: decimalPoint.
    aStream nextPutAll: (string copyFrom: decimal to: string size)! !

!LcNumeric methodsFor: 'reading'!

bePosix
    self setDefaults!

readDataFrom: f
    decimalPoint := self readStringFrom: f.
    thousandsSep := self readStringFrom: f.
    grouping := (self readStringFrom: f) asByteArray!

setDefaults
    decimalPoint isEmpty ifTrue: [ decimalPoint := '.' ].
    thousandsSep isEmpty ifTrue: [ thousandsSep := '' ].
    grouping isEmpty ifTrue: [ grouping := #[] ].
! !

!LcMonetary class methodsFor: 'accessing'!

category
    ^#'LC_MONETARY'!

bigEndianID
    ^#[16r96 16r06 16r17 16rDA]!

littleEndianID
    ^#[16rDA 16r17 16r06 16r96]! !

!LcMonetary methodsFor: 'printing'!

print: aNumber on: aStream
    self
	print: aNumber
	on: aStream
	currency: true
	parentheses: false!

print: aNumber on: aStream currency: currency parentheses: p
    | signChar signPos csPrecedes sepBySpace paren |
    self readData.
    paren := p.
    aNumber < 0
	ifTrue: [
	    csPrecedes := nCsPrecedes.
	    sepBySpace := nSepBySpace.
	    signPos := nSignPosn.
	    signChar := negativeSign.
	    signPos = 0 ifTrue: [ paren := true ].
	    paren ifTrue: [ signChar := '()' ].
	]
	ifFalse: [
	    csPrecedes := pCsPrecedes.
	    sepBySpace := pSepBySpace.
	    signPos := pSignPosn.
	    signChar := positiveSign.
	    
	    "Contrast paren = true with signPos = 0: the former
	     prints spaces, the latter prints nothing for positive
	     numbers!"
	    paren ifTrue: [ signChar := '  ' ].
	].

    "Set default values and tweak signPos if parentheses are needed"
    paren ifTrue: [ signPos := 0 ].

    paren ifTrue: [ aStream nextPut: (signChar at: 1) ].
    signPos = 1 ifTrue: [ aStream nextPutAll: signChar ].

    csPrecedes
	ifTrue: [
	    signPos = 3 ifTrue: [ aStream nextPutAll: signChar ].
	    currency ifTrue: [
		aStream nextPutAll: currencySymbol.
		sepBySpace ifTrue: [ aStream space ].
	    ].
	    signPos = 4 ifTrue: [ aStream nextPutAll: signChar ].
	].

    self basicPrint: aNumber abs on: aStream.
    
    csPrecedes
	ifFalse: [
	    signPos = 3 ifTrue: [ aStream nextPutAll: signChar ].
	    currency ifTrue: [
		sepBySpace ifTrue: [ aStream space ].
		aStream nextPutAll: currencySymbol.
	    ].
	    signPos = 4 ifTrue: [ aStream nextPutAll: signChar ].
	].

    signPos = 2 ifTrue: [ aStream nextPutAll: signChar ].
    paren ifTrue: [ aStream nextPutAll: (signChar at: 2) ].
! !

!LcMonetary methodsFor: 'private'!

printFractionalPart: string startingAt: decimal on: aStream
    | last zeros digits |
    fracDigits = 0 ifTrue: [ ^self ].
    fracDigits = 127 ifTrue: [ ^self ].
    last := decimal + fracDigits - 1.
    zeros := last - string size max: 0.
    aStream
	nextPutAll: decimalPoint;
	nextPutAll: (string copyFrom: decimal to: last - zeros);
	next: zeros put: $0.
! !

!LcMonetary methodsFor: 'reading'!

readDataFrom: f
    | intCurrSymbol intFracDigits |
    intCurrSymbol := self readStringFrom: f.
    currencySymbol  := self readStringFrom: f.

    super readDataFrom: f.
    positiveSign := self readStringFrom: f.
    negativeSign := self readStringFrom: f.
    intFracDigits := self readByteFrom: f.
    fracDigits := self readByteFrom: f.
    pCsPrecedes := self readByteFrom: f.
    pSepBySpace := self readByteFrom: f.
    nCsPrecedes := self readByteFrom: f.
    nSepBySpace := self readByteFrom: f.
    pSignPosn := self readByteFrom: f.
    pSignPosn := self readByteFrom: f.

    self intCurrSymbol: intCurrSymbol intFracDigits: intFracDigits!

intCurrSymbol: intCurrSymbol intFracDigits: intFracDigits!

bePosix
    currencySymbol  := ''.
    positiveSign := ''.
    negativeSign := ''.
    fracDigits := 255.
    pCsPrecedes := 255.
    pSepBySpace := 255.
    nCsPrecedes := 255.
    nSepBySpace := 255.
    pSignPosn := 255.
    pSignPosn := 255.
    super bePosix!

setDefaults
    super setDefaults.
    (positiveSign isEmpty and: [ negativeSign isEmpty ])
	ifTrue: [ negativeSign := $- ].

    pSignPosn = 255 ifTrue: [ pSignPosn := 4 ].
    nSignPosn = 255 ifTrue: [ nSignPosn := 4 ].
    pSepBySpace := pSepBySpace > 0.
    nSepBySpace := nSepBySpace > 0.
    pCsPrecedes := pCsPrecedes > 0.
    nCsPrecedes := nCsPrecedes > 0.
! !

!LcMonetaryISO methodsFor: 'choosing'!

intCurrSymbol: intCurrSymbol intFracDigits: intFracDigits
    currencySymbol := intCurrSymbol.
    fracDigits := intFracDigits!

setDefaults
    "These insights are from The GNU C library manual, chapter 19."

    "When you use int_curr_symbol, you never print an additional space,
    because int_curr_symbol itself contains the appropriate separator.
    The POSIX standard says that these two members apply to the
    int_curr_symbol as well as the currency_symbol. But an example in
    the ISO C standard clearly implies that they should apply only to
    the currency_symbol---that the int_curr_symbol contains any
    appropriate separator, so you should never print an additional
    space. Based on what we know now, we recommend you ignore these
    members when printing international currency symbols, and print
    no extra space."
    nSepBySpace := pSepBySpace := 0.
    
    "The POSIX standard says that these two members apply to the
    int_curr_symbol as well as the currency_symbol. The ISO C
    standard seems to imply that they should apply only to the
    currency_symbol---so the int_curr_symbol should always precede 
    the amount. We can only guess which of these (if either) matches
    the usual conventions for printing international currency symbols.
    Our guess is that they should always precede the amount. If we
    find out a reliable answer, we will put it here."
    nCsPrecedes := pCsPrecedes := 1.
    
    "The ISO standard doesn't say what you should do when the value
    is CHAR_MAX. We recommend you print the sign after the currency symbol. 

    It is not clear whether you should let these members apply to the
    international currency format or not. POSIX says you should, but
    intuition plus the examples in the ISO C standard suggest you
    should not. We hope that someone who knows well the conventions for
    formatting monetary quantities will tell us what we should recommend.
    
    [In the meanwhile, I'm following the ISO C approach of overriding the
    defaults when international formatting is requested.]"
    nSignPosn := pSignPosn := 4.

    super setDefaults! !


!Number methodsFor: 'coercion'!

asBasicNumberType
    ^self asFloat! !

!Integer methodsFor: 'coercion'!

asBasicNumberType
    ^self! !

!Float methodsFor: 'coercion'!

asBasicNumberType
    ^self! !

