/******************************** -*- C -*- ****************************
 *
 *	Input module: stream interface and Readline completion handling
 *
 *	$Revision: 1.95.1$
 *	$Date: 2000/12/27 10:45:49$
 *	$Author: pb$
 *
 ***********************************************************************/

/***********************************************************************
 *
 * Copyright 1988-92, 1994-95, 1999, 2000 Free Software Foundation, Inc.
 * Written by Paolo Bonzini.
 *
 * This file is part of GNU Smalltalk.
 *
 * GNU Smalltalk is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2, or (at your option) any later 
 * version.
 * 
 * GNU Smalltalk is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * GNU Smalltalk; see the file COPYING.  If not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
 *
 ***********************************************************************/

#include "gst.h"
#include "dict.h"
#include "input.h"
#include "lex.h"
#include "sysdep.h"
#include "alloc.h"

#ifdef STDC_HEADERS
# include <string.h>
# include <stdlib.h>
#endif /* STDC_HEADERS */

#ifdef HAVE_READLINE
# include <readline/readline.h>
# include <readline/history.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_IO_H
#include <io.h>
#endif

#include <stdio.h>

#define EMACS_PROCESS_MARKER	'\001' /* ^A as marker -- random choice */

typedef struct StringStreamStruct {
  char		*strBase;	/* base of asciz string */
  char		*str;		/* pointer into asciz string */
} StringStream;

typedef struct StreamStruct {
  StreamType	      type;
  int		      pushedBackChars[2]; /* the 2 buffered characters */
  int		      pushedBackCount;  /* number of chars pushed back */
  int		      line;
  int		      column;
  char		      *fileName;
  int		      fileOffset;
  OOP		      fileNameOOP;	/* the full path name for file */
  mst_Boolean	      prompt;
  union {
    FILE	*u_st_file;
    StringStream u_st_str;
  } st;
  struct StreamStruct *prevStream;
} *Stream;

#define st_file		st.u_st_file
#define st_str		st.u_st_str

static int                      myGetC();
static void 			lineStamp(), myClose();
static Stream			pushNewStream();

static Stream			inStream = nil;

/* Controls the use of the changes file, for recording source text.
   If nil, no recording */
char				*changeFileName = nil;

static FILE			*changeStr = nil;


/* Set by command line flag.  When true, Smalltalk creates FileSegment objects
 * even for files outside the kernel directory.
 */
mst_Boolean			storeNoSource = false;

/* If true, the normal execution information is supressed, and the prompt
 * is emitted with a special marker character ahead of it to let the process
 * filter know that the execution has completed.
 */
mst_Boolean	 	emacsProcess = false;


/***********************************************************************
 *
 *	Generic "stream" interface.  A stream is an abstraction for input and
 *	output.  It is most like common lisp streams.  Basically, these streams
 *	provide transparent reading from either a Smalltalk string, or a UNIX
 *	file.  They stack, and the previous stream can be restored by doing a
 *	"popStream" which optionally "closes" the current stream and goes back 
 *	to using the previous stream.
 *
 * 	The `readline()' interface:
 *
 * 		The behavior is like the Smalltalk-String interface.
 * 		The end-of-string or a NULL strBase-pointer decides
 * 		to read in a new line.  The prompt is still shown by
 * 		the readline() call.
 * 		
 ***********************************************************************/

void
popStream(closeIt)
     mst_Boolean closeIt;
{
  Stream	oldStream;

  oldStream = inStream;
  inStream = inStream->prevStream;

  if (closeIt && oldStream) {
    myClose(oldStream);
    xfree(oldStream);
  }
}

void
pushUNIXFile(file, fileName)
     FILE	*file;
     char	*fileName;
{
  Stream	newStream;

  newStream = pushNewStream(fileStreamType);
  newStream->st_file = file;
  newStream->fileName = fileName;
  newStream->prompt = isatty(fileno(file));
}

void
pushSmalltalkString(stringOOP)
     OOP	stringOOP;
{
  Stream	newStream;

  newStream = pushNewStream(stringStreamType);

  newStream->st_str.strBase = (char *)toCString(stringOOP);
  newStream->st_str.str = newStream->st_str.strBase;
  newStream->fileName = "a Smalltalk string";
  newStream->prompt = false;
}

void 
pushCString(string)
     char	*string;
{
  Stream	newStream;

  newStream = pushNewStream(stringStreamType);

  newStream->st_str.strBase = string;
  newStream->st_str.str = newStream->st_str.strBase;
  newStream->fileName = "a C string";
  newStream->prompt = false;
}

void
pushStdinString()
{
#ifdef HAVE_READLINE
  Stream	newStream;

  if (emacsProcess) {
#endif
    pushUNIXFile(stdin, "stdin");
    return;
#ifdef HAVE_READLINE
  }

  newStream = pushNewStream(readlineStreamType);

  newStream->st_str.strBase = 0;	/* force readline() but no free() */
  newStream->st_str.str = 0;
  newStream->fileName = "stdin";	/* that's where we get input from */
  newStream->prompt = false;		/* prompt is shown by readline() */
#endif /* HAVE_READLINE */
}

Stream
pushNewStream(type)
     StreamType type;
{
  Stream	newStream;

  newStream = (Stream)xmalloc(sizeof(struct StreamStruct));

  newStream->pushedBackCount = 0;
  newStream->line = 1;
  newStream->column = 0;
  newStream->fileOffset = 0;
  newStream->type = type;
  newStream->fileNameOOP = nilOOP;
  newStream->prevStream = inStream;
  inStream = newStream;

  return (newStream);
}


/*
 *	void setStreamInfo(line, fileName, fileOffset)
 *
 * Description
 *
 *	This function resets the file type information for the current stream.
 *	It is typically used by fileIn type methods when filing in a subsection
 *	of a real file via a temporary file what the real source of the text
 *	is.
 *
 * Inputs
 *
 *	line  : line number that the input starts on
 *	fileName: 
 *		The name of the file, only used for file type streams.
 *	fileOffset: 
 *		The byte offset from the start of the actual file where this
 *		given portion of code starts.
 *
 */
void
setStreamInfo(line, fileName, fileOffset)
     int	line, fileOffset;
     char	*fileName;
{
  inStream->line = line;
  if (inStream->type == fileStreamType) {
    inStream->fileName = fileName;
    inStream->fileOffset = fileOffset;
  }
}

int
myGetC(stream)
     Stream	stream;
{
  int		ic = 0;

  if (stream->type == stringStreamType) {
    ic = *stream->st_str.str++;
    return ((ic == '\0') ? EOF : ic);
  } else if (stream->type == fileStreamType) {
    ic = getc(stream->st_file);
    return (ic);
#ifdef HAVE_READLINE
  } else if (stream->type == readlineStreamType) {
    char *r_line;
    int r_len;

    if (stream->st_str.strBase) {
      ic = *stream->st_str.str++;
      if (ic) {
        return (ic);
      }
      /* If null, read a new line */
    }

    if(stream->st_str.strBase) {
      xfree(stream->st_str.strBase);
      stream->st_str.strBase = 0;
    }
    r_line = readline("st> ");
    if (!r_line) {
      /* return value of NULL indicates EOF */
      return (EOF);
    }
    if (*r_line) {
      /* add only non-empty lines. */
      add_history(r_line);
    }

    /* tack on the newline, not returned by readline() */
    r_len = strlen(r_line);
    r_line = xrealloc(r_line, (unsigned) (r_len + 2));
    if (!r_line) {
      errorf("Out of memory reallocating linebuffer space");
      stream->st_str.str = stream->st_str.strBase = 0;
      ic = '\n';			/* return a newline ... */
    } else {
      r_line[r_len] = '\n';
      r_line[r_len+1] = '\0';
      stream->st_str.str = stream->st_str.strBase = r_line;
      ic = *stream->st_str.str++;
    }
#endif /* HAVE_READLINE */
  } else {
    errorf("Bad stream type passed to myGetC");
    hadError = true;
  }
  return (ic);
}

#ifdef old_code /* Sat Mar 23 21:00:38 1991 */
/**/static void myUngetC(ic, stream)
/**/int	ic;
/**/Stream	stream;
/**/{
/**/  if (stream->type == stringStreamType) {
/**/    stream->st_str.str--;
/**/  } else if (stream->type == fileStreamType) {
/**/    ungetc(ic, stream->st_file);
/**/#ifdef HAVE_READLINE
/**/  } else if (stream->type == readlineStreamType) {
/**/    if (stream->st_str.str > stream->st_str.strBase) {
/**/      *--stream->st_str.str = ic;
/**/    } else {
/**/      errorf("Cannot unget character to readline stream");
/**/    }
/**/#endif /* HAVE_READLINE */
/**/  } else {
/**/    errorf("Bad stream type passed to myUngetC");
/**/    hadError = true;
/**/  }
/**/}
#endif /* old_code Sat Mar 23 21:00:38 1991 */

void
myClose(stream)
     Stream	stream;
{
  if (stream->type == stringStreamType) {
    xfree(stream->st_str.strBase);
  } else if (stream->type == fileStreamType) {
    fclose(stream->st_file);
#ifdef HAVE_READLINE
  } else if (stream->type == readlineStreamType) {
    if (stream->st_str.strBase) {
      xfree(stream->st_str.strBase);
      stream->st_str.strBase = 0;
    }
#endif /* HAVE_READLINE */
  } else {
    errorf("Bad stream type passed to myClose");
    hadError = true;
  }
}


StreamType
getCurStreamType()
{
  if (inStream) {
    return (inStream->type);
  } else {
    return (unknownStreamType);
  }
}

OOP
getCurString()
{
  if (inStream && inStream->type == stringStreamType) {
    return (stringNew(inStream->st_str.strBase));
  } else {
    return (nilOOP);
  }
}

OOP
getCurFileName()
{
  char		*fullFileName;

  if (inStream && inStream->type == fileStreamType) {
    if (inStream->fileNameOOP == nilOOP) {
      if (strcmp(inStream->fileName, "stdin") == 0) {
	fullFileName = strdup(inStream->fileName);
      } else {
	fullFileName = getFullFileName(inStream->fileName);
      }
      inStream->fileNameOOP = stringNew(fullFileName);
    }
    return (inStream->fileNameOOP);
  } else {
    return (nilOOP);
  }
}

#ifdef HAVE_READLINE
OOP
getCurReadline()
{
  if (inStream && inStream->type == readlineStreamType) {
    return (stringNew(inStream->st_str.strBase));
  } else {
    return (nilOOP);
  }
}
#endif /* HAVE_READLINE */


/*
 * Report an error to the user.  ### Will show position in the text of
 * the error at some point.
 */

#define ERROR_ARGS	arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8
void
errorf(str, ERROR_ARGS)
     char	*str;
     long	ERROR_ARGS;
{
  char		errStr[256];

  if (reportErrors) {
    fflush(stdout);
  }
  lineStamp();
  if (reportErrors) {
    fprintf(stderr, str, ERROR_ARGS);
    fprintf(stderr, "\n");
    fflush(stderr);
  } else {
    if (firstErrorStr == nil) {
      sprintf(errStr, str, ERROR_ARGS);
      firstErrorStr = strdup(errStr);
    }
  }
}
#undef ERROR_ARGS

/* ??? is void the right type */
void
yyerror(s)
     char	*s;
{
  errorf("%s", s);
}

void
lineStamp()
{
  if (reportErrors) {
    if (inStream) {
      if (inStream->fileName) {
	fprintf(stderr, "%s:", inStream->fileName);
      }
      fprintf(stderr, "%d: ", inStream->line);
    } else {
      fprintf(stderr, "gst: ");
    }
  } else {			/* called internally with error handling */
    if (inStream) {
      if (inStream->fileName) {
	if (firstErrorStr == NULL) {
	  firstErrorFile = strdup(inStream->fileName);
	}
      }
      if (firstErrorStr == NULL) {
	firstErrorLine = inStream->line;
      }
    } else {
      if (firstErrorStr == NULL) {
	firstErrorLine = -1;
      }
    }
  }
}


/* Return current method source string from open FileStream */
char *
getMethodSourceFromCurFile()
{
  char 		*buf;
  int		endPos, length, startPos;

  startPos = getMethodStartPos();
  endPos = getCurFilePos(); 
  /* The second -1 removes the terminating '!' */
  length = endPos  - startPos - 1;
  buf = (char *)xmalloc(length + 1);
  fseek(inStream->st_file, startPos, 0);
  fread(buf, length, 1, inStream->st_file);
  buf[length] = '\0';
  /* Restore old file position */
  fseek(inStream->st_file, endPos, 0); 
  return(buf);
}


int
getCurFilePos()
{
  /* ### not sure about this particular use -- the current file pos must be
   * from the change log, but currently the FileSegments we produce are from
   * the input file... so they lose.
   * The original reason for implementing the change log was to make recom-
   * piles of a method lying in a changed file work... but this way not even
   * recompiles of methods lying in an unchanged file do work!
   */
#ifdef not_working_for_now
/**/  if (changeStr) {
/**/    return (ftell(changeStr));
/**/  } else
#endif

  if (inStream && inStream->type == fileStreamType) {
    return (ftell(inStream->st_file) + inStream->fileOffset);
  } else {
    return (-1);
  }
}

void
initChangesStream()
{
  if (changeFileName) {
    changeStr = fopen(changeFileName, "a");
  }
}

/*
 *	void resetChangesFile()
 *
 * Description
 *
 *	Resets the changes file to be zero length.
 *
 */
void
resetChangesFile()
{
  if (!changeFileName) {
    return;
  }
  if (changeStr) {
    fclose(changeStr);
  }
  remove(changeFileName);
  if (changeStr) {
    initChangesStream();
  }
}


int
nextChar()
{
  register int		ic;

  if (inStream->pushedBackCount > 0) {
    ic = inStream->pushedBackChars[--inStream->pushedBackCount];
    return (ic);
  } else {
    if (inStream->column == 0 && inStream->prompt) {
      if (emacsProcess) {
	printf("%c", EMACS_PROCESS_MARKER);
      }
      printf("st> ");
    }
    ic = myGetC(inStream);

    if (changeStr) {
      if (ic == EOF) {
	fflush(changeStr);
      } else {
	fputc(ic, changeStr);
      }
    }

    if (ic == '\n') {		/* a new line that was not pushed back */
      inStream->line++;
      inStream->column = 0;
    } else {
      inStream->column++;
    }
    return (ic);
  }
}

/*
 *	static void unreadChar(ic)
 *
 * Description
 *
 *	Push character 'ic' back into the input queue.  Allows for two
 *	character pushback currently.  This solves the problem of lexing 3. and
 *	then finding out that what we should have lexed was 3 followed by . as
 *	a statement terminator.
 *
 * Inputs
 *
 *	ic    : character to push back into the input stream.
 *
 */
void
unreadChar(ic)
int	ic;
{
  inStream->pushedBackChars[inStream->pushedBackCount++] = ic;
}


/* Return true if current compilation stream is a FileStream associated
 * with a path relative to the kernel directory. 
 */ 
mst_Boolean
isKernelFile()
{
  char		*fullFileName, *fullFileNameTest;
  mst_Boolean	result;	

  if (storeNoSource) {
    return (true);
  }
  if ((strcmp(inStream->fileName, "stdin") == 0)
      || (getMethodStartPos() > getCurFilePos())) {
    return (true);
  }

  fullFileName = fullFileNameTest = getFullFileName(inStream->fileName);
#if defined(MSDOS) || defined(WIN32) || defined(__OS2__)
  if (fullFileName[1] == ':') {
    fullFileName += 2;		/* skip over "<drive>:"  */
  }
#endif
  result = (strstr(fullFileNameTest, KERNEL_PATH)  == fullFileNameTest);
  xfree(fullFileNameTest);

  return (result);
}


#ifdef HAVE_READLINE
/* Storage for the possible completions */
static char **pointers;
static int count;
static int freeItems;
static int completionsEnabled = 1;
static int sortedCount;

/* Internal functions */
static void merge();
static void addCompletion();
static int  compareStrings();

/* Readline callbacks */
static char *readlineQuoteFilename(), *readlineDequoteFilename(),
            *symbolGenerator(), **readlineMatchSymbols();

/* Find apostrophes and double them */
char *
readlineQuoteFilename (s, rtype, qcp)
     char *s;
     int rtype;
     char *qcp;
{
  char *base = alloca(strlen(s) * 2 + 2);
  char *p, *r;

  int quote;

  r = base;
  quote = *qcp;
  if (!quote) {
    quote = *rl_completer_quote_characters;
  }

  *r++ = quote;
  for (p = s; *p; ) {
    if (*p == quote) {
      *r++ = quote;
    }
    *r++ = *p++;
  }
  *r++ = 0;
  return (strdup(base));
}

/* Find double apostrophes and turn them to single ones */
char *
readlineDequoteFilename (s, qc)
     char *s;
     char qc;
{
  char *base = alloca(strlen(s) + 2);
  char *p, *r;

  if (!qc) {
    return strdup (s);
  }

  r = base;
  for (p = s; *p; ) {
    if (*p == qc) {
      p++;
    }
    *r++ = *p++;
  }
  *r++ = 0;
  return (strdup(base));
}

/* These two are not used, but are provided for additional flexibility. */
void
enableCompletion()
{
  completionsEnabled++;
}

void
disableCompletion()
{
  completionsEnabled--;
}

/* Enter an item in the list */
void
addCompletion(str, len)
     char *str;
     int len;
{
  char *s = xmalloc(len + 1);
  memcpy (s, str, len);
  s[len] = 0;

  if (!freeItems) {
    freeItems += 50;
    pointers = (char **) xrealloc (pointers, SIZEOF_CHAR_P * (count + 50));
  }

  freeItems--;
  pointers[count++] = s;
}

void
addSymbolCompletion(str, len)
     char *str;
     int len;
{
  char *base = str, *p = str;

  if (completionsEnabled < 1) {
    return;
  }

  while (len-- && *p) {
    if (*p++ == ':' && (base != p - 1)) {
      addCompletion (base, p - base);
      base = p;
    }
  }

  /* We enter globals in the table, too */
  if (base != p && !islower(*str)) {
    addCompletion (base, p - base);
  }
}

/* Merge the contents of a1 with the contents of a2,
 * storing the result in a2.  If a1 and a2 overlap,
 * reallocate must be true.
 */
void
merge (a1, count1, a2, count2, reallocate)
     char **a1, **a2;
     int count1, count2;
     mst_Boolean reallocate;
{
  char *source, *dest;

  /* Check if an array is empty */
  if (!count1) {
    return;
  }
  if (!count2) {
    memmove (a1, a2, count1 * SIZEOF_CHAR_P);
    return;
  }

  if (reallocate) {
    char **new = (char **) alloca (count1 * SIZEOF_CHAR_P);
    memcpy (new, a1, count1 * SIZEOF_CHAR_P);
    a1 = new;
  }

  source = a1[count1 - 1];
  dest = a2[count2 - 1];
  for(;;) {
    if (strcmp(source, dest) < 0) {
      /* Take it from the destination array */
      a2[count2 + count1 - 1] = dest;
      if (--count2 == 0) {
	/* Any leftovers from the source array? */
	memcpy (a1, a2, count1 * SIZEOF_CHAR_P);
	return;
      }

      dest = a2[count2 - 1];
    } else {
      /* Take it from the source array */
      a2[count2 + count1 - 1] = source;
      if (--count1 == 0) {
	return;
      }

      source = a1[count1 - 1];
    }
  }

}

/* Comparison function for qsort */
int
compareStrings(a, b)
    char **a;
    char **b;
{
  return strcmp (*a, *b);
}

/* Communication between symbolGenerator and readlineMatchSymbols */
static int matchesLeft, currentIndex;

char *
symbolGenerator(text, state)
     char *text;
     int state;
{
  if (matchesLeft == 0) {
    return (NULL);
  }

  /* Since we have to sort the array to perform the binary search, we
   * remove duplicates and avoid that readline resorts the result.
   */
  while (matchesLeft > 1 &&
	 strcmp(pointers[currentIndex], pointers[currentIndex + 1]) == 0) {
    currentIndex++;
    matchesLeft--;
  }
    
  matchesLeft--;
  return strdup(pointers[currentIndex++]);
}


char **
readlineMatchSymbols(text, start, end)
     char *text;
     int start, end;
{
  int low, high, middle, len;

  /* Check for strings (not matched) and for symbols (matched) */
  if (start != 0 && rl_line_buffer[start - 1] == '\'') {
    if (start == 1 || rl_line_buffer[start - 2] != '#') {
      return NULL;
    }
  }

  /* Prepare for binary searching.  We use qsort when necessary, and
   * merge the result, instead of doing expensive (quadratic) insertion
   * sorts.
   */
  if (sortedCount < count) {
    qsort (&pointers[sortedCount], count - sortedCount,
	   SIZEOF_CHAR_P, compareStrings);

    merge (&pointers[sortedCount], count - sortedCount,
	   pointers, sortedCount, true);

    sortedCount = count;
  }

  /* Initialize currentIndex and matchesLeft with two binary searches. */
  len = strlen(text);

  /* The first binary search gives the first matching item. */
  low = -1;
  high = count;
  while (low + 1 != high) {
    middle = (low + high) / 2;
    if (strncmp (pointers[middle], text, len) < 0)
      low = middle;
    else
      high = middle;
  }
  currentIndex = high;

  /* This binary search gives the first non-matching item instead */
  low = -1;
  high = count;
  while (low + 1 != high) {
    middle = (low + high) / 2;
    if (strncmp (pointers[middle], text, len) <= 0)
      low = middle;
    else
      high = middle;
  }

  matchesLeft = high - currentIndex;
  return matchesLeft ? completion_matches(text, symbolGenerator) : NULL;
}

void
initializeReadline()
{
  static char everything[255];
  int i;

  /* Allow conditional parsing of the ~/.inputrc file. */
  rl_readline_name = "Smalltalk";

  /* Always put filenames in quotes */
  for (i = 0; i < 255; i++) {
    everything[i] = i+1;
  }
  rl_filename_quote_characters = everything;
  rl_completer_quote_characters = "'\"";
  rl_basic_word_break_characters = "() [];+-=*<>~'?%/@|&#^\"\\";

  /* Consider binary selectors both word-breaking characters and 
   * candidates for completion */
  rl_special_prefixes = "+-=*<>~?%/@|&\\";

  /* Our rules for quoting are a bit different from the default */
  rl_filename_quoting_function = readlineQuoteFilename;
  rl_filename_dequoting_function = readlineDequoteFilename;

  /* Try to match a symbol before a filename */
  rl_attempted_completion_function = readlineMatchSymbols;

  /* Since we have to sort the array to perform the binary search,
   * remove duplicates and avoid that readline resorts the result.
   */
  rl_ignore_completion_duplicates = 0;

  if (count == 0) {
    addAllSymbolCompletions();
  }
}
     
#endif /* HAVE_READLINE */
