/******************************** -*- C -*- ****************************
 *
 *	Interpreter employing a bytecode->native code.
 *
 *	$Revision: 1.95.1$
 *	$Date: 2000/12/27 10:45:49$
 *	$Author: pb$
 *
 ***********************************************************************/

/***********************************************************************
 *
 * Copyright 1988-92, 1994-95, 1999, 2000 Free Software Foundation, Inc.
 * Written by Paolo Bonzini.
 *
 * This file is part of GNU Smalltalk.
 *
 * GNU Smalltalk is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2, or (at your option) any later
 * version.
 *
 * GNU Smalltalk is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * GNU Smalltalk; see the file COPYING.	 If not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 ***********************************************************************/

/*
 * This is basically how the interpreter works.
 *
 * The interpreter expects to be called in an environment where there
 * already exists a well-defined method context.  The instruction pointer,
 * stored in the global variable "nativeIP", and the stack pointer, stored 
 * in the global variable "sp", should be set up to point into the current
 * method and MethodContext.  Other global variables, such as "thisMethod",
 * self", "temporaries", etc. should also be setup; see the routine
 * prepareExecutionEnvironment for details.  `interpret' immediately calls
 * `runNativeCode' (coded with GNU lightning in xlat.c) and starts
 * executing compiled code.
 * 
 * Every now and then, the compiled code checks to see if any change in its
 * state is required, such as switching to a new process, dealing with an
 * asynchronous signal and printing out the byte codes that are being
 * executed, if that was requested by the user.  If so, it returns to
 * `interpret', telling where to restart execution.  `interpret' handles
 * the event and passes control back to the compiled code.  Control is
 * returned to `interpret' also when the private #__terminate method is
 * executed.
 *
 * When a message send occurs for the first time, an internal routine (either
 * doSend or doSuper) is invoked (these routines are coded with GNU lightning
 * in xlat.c); this routine calls lookupNativeIP, which looks for the
 * method in a cache like the one used by the standard interpreter, and if it
 * is not found, asks xlat.c to return the address; in turn, xlat.c
 * looks for the code into an hash table (which handles collisions unlike
 * lookupNativeIP's cache) before compiling it.  doSend then saves the
 * returned address so that it will be used the next time the message is
 * sent, then it jumps to the address.
 *
 * If no selector is found, doSend calls lookupNativeIP again, asking for the
 * address of the #doesNotUnderstand: method.  Note that lookupNativeIP had
 * modified the stack, pushing a Message object that embeds information about
 * the original selector and arguments.  If #doesNotUnderstand: is not
 * understood, a crash is extremely likely; things like this are however to be
 * expected, since you're really playing a bad game and going against some
 * logical things that the VM assumes for speed.
 *
 * After the first time, doSend is not called: instead, the previous address
 * is used to guess the likely address where the code to be called lies.  Of
 * course, the `guessed' address might be wrong, because the receiver's class
 * might have changed.  So the prolog of every compiled method checks for this
 * eventuality and, if so, performs a standard lookup by jumping back to
 * doSend.  Note that this cannot happen with doSuper, because super sends
 * always go to the same receiver (which is the `self' object).
 *
 * Methods can be marked as doing special things.  These are returning self,
 * returning an instance variable, return a literal object, or executing a
 * primitive method definition.	 In the first three cases, the method does
 * not create a context, and immediately jumps back to the caller.  In the
 * latter case, the method calls executePrimitiveOperation, examines its
 * return value, and does one of three things:
 * - jumping back to the caller
 * - creating a context and executing the Smalltalk code (failure)
 * - load a new nativeIP and jump there
 *
 * Primitive 81 (BlockClosure>>#value) is special-cased in that the new
 * nativeIP is cached in the same way that doSend caches the address of a
 * method.
 *
 * When a method returns, the context that called it is examined to restore
 * the interpreter's global variables to the state that they were in before
 * the callee was invoked.  After the state has been restored, the callee
 * jumps back to the caller, almost oblivious to the fact that, likely,
 * millions of instructions have been executed since control was left.
 *
 * Note that the microprocessor's CALL and RETURN instructions are not used.
 * Instead, once the runNativeCode function is called, control threads from
 * one method to another, without creating any stack frames, mantaining all
 * the information partly inside ContextPart objects and partly inside
 * InlineCache structures (see xlat.c).  This allows an extremely easy
 * implementation of non-local returns.
 */



static voidPtr lookupNativeIP();

#include "internal.h"

InternalFunc internalFuncs[] = {
  (InternalFunc) unwindContext,
  (InternalFunc) unwindMethod,
  (InternalFunc) activateNewContext,
  (InternalFunc) prepareContext,
  (InternalFunc) emptyContextStack,
  (InternalFunc) lookupNativeIP
};

#define getContextIP(ctx) 	((long) ((ctx)->returnIP) - 1)

#define getReturnIP(ipOffset) 	fromInt( ((ipOffset) - nativeBase) >> 1)

#define setThisMethod(method, ipOffset) {				\
  thisMethod = (method);						\
  nativeIP = nativeBase + (ipOffset);					\
}


voidPtr
lookupNativeIP(sendSelector, sendArgs, receiver, methodClass)
     OOP	sendSelector;
     int	sendArgs;
     OOP	receiver;
     OOP	methodClass;	/* the class in which to start the search */
{
  REGISTER(1, long		hashIndex);
  REGISTER(2, MethodCacheEntry	*methodData);
  REGISTER(3, OOP		receiverClass);

  /* hash the selector and the class of the receiver together using XOR.
   * Since both are addresses in the oopTable, and since oopTable entries
   * are 2 longs in size, shift over by 3 bits (4 on 64-bit architectures)
   * to remove the useless low order zeros. */

#ifdef PROFBLOCK
  ps.numMessageSends++;
#endif
  sampleCounter++;
  hashIndex = methodCacheHash(sendSelector, methodClass);
  methodData = &methodCache[hashIndex];

  if (methodData->selectorOOP != sendSelector
      || methodData->startingClassOOP != methodClass) {
    /* :-( cache miss )-: */
    if (!lookupMethod (sendSelector, methodData, sendArgs, methodClass)) {
      return (nil);
    } else {
      /* The next test cannot succeed */
      methodData->receiverClass = nil;
    }
  }

  receiverClass = isInt(receiver) ? smallIntegerClass : oopClass(receiver);
  if (methodData->receiverClass == receiverClass) {
    return (methodData->nativeCode);
  }

  methodData->receiverClass = receiverClass; 
  methodData->nativeCode = getNativeCode(methodData->methodOOP, receiverClass);
  return (methodData->nativeCode);
}

/* On entry to this routine, the stack should have the receiver and the
 * arguments pushed on the stack.  We need to get a new context, setup
 * things like the IP, SP, and Temporary pointers, and then return.   Note
 * that this routine DOES NOT invoke the interpreter; it merely sets up a
 * new context so that calling (or, more typically, returning to) the
 * interpreter will operate properly.  This kind of sending is for normal
 * messages only.  Things like sending a "value" message to a block context are
 * handled by primitives which do similar things, but they use information from
 * BlockClosure objects that we don't have available (nor need) here.
 */

void
sendMessageInternal(sendSelector, sendArgs, receiver, methodClass)
     OOP	sendSelector;
     int	sendArgs;
     OOP	receiver;
     OOP	methodClass;	/* the class in which to start the search */
{
  long				hashIndex;
  REGISTER(1, OOP		receiverClass);
  REGISTER(2, MethodCacheEntry	*methodData);
  REGISTER(3, MethodHeader	header);

  /* hash the selector and the class of the receiver together using XOR.
   * Since both are addresses in the oopTable, and since oopTable entries
   * are 2 longs in size, shift over by 3 bits (4 on 64-bit architectures)
   * to remove the useless low order zeros. */

#ifdef PROFBLOCK
  ps.numMessageSends++;
#endif
  sampleCounter++;
  hashIndex = methodCacheHash(sendSelector, methodClass);
  methodData = &methodCache[hashIndex];

  if (methodData->selectorOOP != sendSelector
      || methodData->startingClassOOP != methodClass) {
    /* :-( cache miss )-: */
    if (!lookupMethod (sendSelector, methodData, sendArgs, methodClass)) {
      sendMessage(doesNotUnderstandColonSymbol, 1, false);
      return;
    }
  }

  header = methodData->methodHeader;

#if 0
  if (header.headerFlag) {
    switch (header.headerFlag) {
    case 1:
      /* 1, return the receiver - self is already on the stack...so we leave it */
      return;

    case 2: {
      register long primIndex = header.primitiveIndex;
      /* 2, return instance variable */
      /* replace receiver with the returned instance variable */
      setStackTop(oopToObj(receiver)->data[primIndex]);
      return;
    }

    case 3: {
      /* 3, return literal constant */
      /* replace receiver with the returned literal constant */
      setStackTop( getMethodLiterals(methodData->methodOOP) [0] );
      return;
    }

    default: break;
    }
  }
#endif

  receiverClass = isInt(receiver) ? smallIntegerClass : oopClass(receiver);
  if (methodData->receiverClass != receiverClass) {
    methodData->receiverClass = receiverClass; 
    methodData->nativeCode = getNativeCode(methodData->methodOOP, receiverClass);
  }
  nativeIP = methodData->nativeCode;
}


void
sendMethod(methodOOP)
     OOP	methodOOP;
{
  OOP receiverClass;
  MethodHeader header;
  REGISTER(1, Method	        method);
  REGISTER(2, OOP               receiver);
  REGISTER(3, int               sendArgs);

#ifdef PROFBLOCK
  ps.numMessageSends++;
#endif
  sampleCounter++;

  method = (Method) oopToObj(methodOOP);
  header = method->header;
  sendArgs = header.numArgs;
  receiver = stackAt(sendArgs);

#if 0
  if (header.headerFlag) {
    switch (header.headerFlag) {
    case 1:
      /* 1, return the receiver - self is already on the stack...so we leave it */
      selfReturns++;
      return;

    case 2: {
      register long primIndex = header.primitiveIndex;
      /* 2, return instance variable */
      /* replace receiver with the returned instance variable */
      setStackTop(instanceVariable(receiver, primIndex));
      instVarReturns++;
      return;
    }

    case 3: {
      /* 3, return literal constant */
      /* replace receiver with the returned literal constant */
      setStackTop( getMethodLiterals(methodOOP) [0] );
      literalReturns++;
      return;
    }

    case 4:
      if (!primitiveTable[header.primitiveIndex].primFunc (
	header.primitiveIndex, sendArgs, methodOOP)) {

	/* primitive succeeded.	 Continue with the parent context */
	return;
      }
      /* primitive failed.  Invoke the normal method.  methodData
       * may be clobbered by a setjmp in executePrimitiveOperation */
      methodData = &methodCache[hashIndex];
      break;

    case 0:	/* only here so that the compiler skips a range check */
    case 5:
    case 6:
    case 7:
    default: break;
    }
  }
#endif

  receiverClass = isInt(receiver) ? smallIntegerClass : oopClass(receiver);
  nativeIP = getNativeCode(methodOOP, receiverClass);
}

/*
 *	static mst_Boolean sendBlockValue(numArgs)
 *
 * Description
 *
 *	This is the equivalent of sendMessage, but is for blocks.  The block
 *	context that is to the the receiver of the "value" message should be
 *	"numArgs" into the stack.  SP is set to the top of the arguments in the
 *	block context, which have been copied out of the caller's context.
 *
 * Inputs
 *
 *	numArgs:
 *		The number of arguments sent to the block.
 *
 * Outputs
 *
 *	true if failed, false if numArgs matches what the BlockClosure says.
 */
static mst_Boolean
sendBlockValue(numArgs)
     int	numArgs;
{
  OOP				closureOOP;
  REGISTER(2, BlockClosure	closure);
  REGISTER(3, BlockHeader	header);

  closureOOP = stackAt(numArgs);
  closure = (BlockClosure)oopToObj(closureOOP);
  header = ((Block)oopToObj(closure->block))->header;
  if(numArgs != header.numArgs) {
    /* check numArgs asap */
    return (true);
  }

  if (isInt(closure->receiver)) {
    nativeIP = getNativeCode(closure->block, smallIntegerClass);
  } else {
    nativeIP = getNativeCode(closure->block, oopClass(closure->receiver));
  }
  return(false);
}


void
validateMethodCacheEntries()
{
  register int		     i;
  register MethodCacheEntry *mc;
  for (i = 0; i < METHOD_CACHE_SIZE; i++) {
    mc = &methodCache[i];
    if (mc->selectorOOP && !isValidIP(mc->nativeCode)) {
      /* invalidate this entry */
      mc->selectorOOP = nil;
    }
  }
}


void
interpret()
{
  inInterpreter = true;
  for (; ;) {
    MethodContext thisContext;
    if (!nativeIP) {
      return;
    }

    nativeIP = runNativeCode(nativeIP);

    if (!exceptFlag) {
      return;
    }

    if (abortExecution) {
      OOP selectorOOP;
      selectorOOP = internString(abortExecution);
      abortExecution = nil;
      sendMessage(selectorOOP, 0, false);
    }

    if (asyncQueueIndex) {	/* deal with any async signals	*/
      register int	i;
      IntState		oldSigMask;
      oldSigMask = disableInterrupts(); /* block out everything! */
      for (i = 0; i < asyncQueueIndex; i++) {
	syncSignal(queuedAsyncSignals[i]);
      }
      asyncQueueIndex = 0;
      enableInterrupts(oldSigMask);
    }
    
    thisContext = (MethodContext) oopToObj(thisContextOOP);
    thisContext->returnIP = getReturnIP(nativeIP);
    if (!isNil(switchToProcess)) {
      changeProcessContext(switchToProcess);
    }

    exceptFlag = false;
  }

  inInterpreter = false;
}
