/******************************** -*- C -*- ****************************
 *
 *	Functions for byte code optimization & analysis
 *
 *	$Revision: 1.95.1$
 *	$Date: 2000/12/27 10:45:49$
 *	$Author: pb$
 *
 ***********************************************************************/

/***********************************************************************
 *
 * Copyright 1988-92, 1994-95, 1999, 2000 Free Software Foundation, Inc.
 * Written by Paolo Bonzini.
 *
 * This file is part of GNU Smalltalk.
 *
 * GNU Smalltalk is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2, or (at your option) any later
 * version.
 *
 * GNU Smalltalk is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * GNU Smalltalk; see the file COPYING.	 If not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 ***********************************************************************/


#include "gst.h"
#include "byte.h"
#include "comp.h"
#include "opt.h"
#include "input.h"
#include "interp.h"
#include "memzero.h"
#include "obstack.h"
#if STDC_HEADERS
#include <string.h> /* for memcpy */
#include <stdlib.h>
#endif /* STDC_HEADERS */
#include <stdio.h>

/* Define this to disable the peephole bytecode optimizer.  It works
 * well and increases a bit performance, so there's no reason to do that
 * unless you're debugging the compiler.
 */
/* #define NO_OPTIMIZE */

/* The JIT compiler prefers optimized bytecodes, because they are
 * more regular.
 */
#ifdef USE_JIT_TRANSLATION
#undef NO_OPTIMIZE
#endif

/* Define this to disable type inference for SmallIntegers. */
#define NO_TYPE_INFERENCE

int isPushTable[256] = {
1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1, /* 0 */
1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1, /* 16 */
1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1, /* 32 */
1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1, /* 48 */
1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1, /* 64 */
1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1, /* 80 */
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /* 96 */
1,1,1,1,1,1,1,1,0,0,0,0,0,0,0,0, /* 112 */
1,0,0,0,0,0,0,0,1,1,0,0,0,0,0,0, /* 128 */
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /* 144 */
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /* 160 */
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /* 176 */
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /* 192 */
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /* 208 */
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /* 224 */
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0}; /* 240 */

int isSendTable[256] = {
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /* 0 */
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /* 16 */
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /* 32 */
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /* 48 */
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /* 64 */
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /* 80 */
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /* 96 */
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /* 112 */
0,0,0,1,1,1,0,0,0,0,0,0,0,0,0,0, /* 128 */
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /* 144 */
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /* 160 */
1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1, /* 176 */
1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1, /* 192 */
1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1, /* 208 */
1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1, /* 224 */
1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1}; /* 240 */

int stackBalanceTable[256] = {
 1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  /* 0 */
 1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  /* 16 */
 1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  /* 32 */
 1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  /* 48 */
 1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  /* 64 */
 1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  1,  /* 80 */
-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,  /* 96 */
 1,  1,  1,  1,  1,  1,  1,  1,  0,  0,  0,  0,  0,  0,255,  0,  /* 112 */
 1,  0, -1,255,255,255,255, -1,  1,  1,255,  0,  0,  0,  0,  0,  /* 128 */
 0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1,  /* 144 */
 0,  0,  0,  0,  0,  0,  0,  0, -1, -1, -1, -1, -1, -1, -1, -1,  /* 160 */
-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,  /* 176 */
-1, -2,  0,  0, -1,  0, -1,  0, -1,  0, -1, -1,  0, -1,  0,  0,  /* 192 */
 0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  0,  /* 208 */
-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,  /* 224 */
-2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2, -2}; /* 240 */

static int  jumpOffsets[16] =  {
  -1022, -766, -510, -254, 2, 258, 514, 770,
      2,  258,  514,  770, 2, 258, 514, 770
};

static mst_Boolean		optimizeBasicBlock();

static int			compareBlocks();

static inline int		checkInnerBlock();

static int			makeDestinationTable();

static void			connectBasicBlocks(), analyzeBasicBlock(),
				unconstrainWrittenVariables(), 
				addEdge(), visitBackwardEdges(),
				visitForwardEdges();

static mst_Boolean		nextBasicBlock();

static int			findBlockHeading();

static inline long		mask();

static inline int		constraintSet(), constraintReset(),
				constraintSum(), constraintSubtract(),
				constraintMultiply(), constraintRemainder(),
				constraintDivision(), constraintBitAnd(),
				constraintBitOr(), constraintBitShift(),
				constraintCopy();

static inline void		constraintLess(), constraintLessEqual(),
				constraintGreater(), constraintGreaterEqual(),
				constraintEqual();
			
static void			constraintCopyBlock(), constraintMergeBlock();


/* This structure and the following one are used by the bytecode
 * peephole optimizer.
 *
 * This one, in particular, defines where basic blocks start in the non-
 * optimized bytecodes.	 byte is nothing more than an offset in those
 * bytecodes; id is used to pair jump bytecodes with their destinations:
 * in the initial scan, when we encounter a jump bytecode we fill two
 * BlockBoundaries -- one has positive id and represents the destination
 * of the jump, one has negative id (but with the same absolute value)
 * and represents the jump bytecode itself.
 */
typedef struct BlockBoundaryStruct {
  short byte;
  short id;
} BlockBoundary;

/* This structure defines how to fix the jumps after the optimized basic
 * blocks are put together.  Everything is done after the peephole pass
 * because this allows us to handle forward jumps and backward jumps in
 * the same way.
 * When single blocks are optimized, the sorted BlockBoundaries are
 * examined one at a time.  As we process blocks, we fill an array of
 * Jump structures with offsets in the optimized bytecode.  We fill a
 * single field at a time -- the id's sign in the BlockBoundary says
 * which field is to be filled, the absolute value gives which Jump
 * structure is to be filled.  In the end, BlockBoundaries whose id's
 * absolute value is the same are all paired.
 */
typedef struct JumpStruct {
  int	from;  /* where the jump bytecode lies */
  int	dest;  /* where the jump bytecode lands */
} Jump;


/* This structure contains the minimum and maximum value that a
 * variable can hold.  If both values are outside the possible range
 * for SmallIntegers mean that the variable does not hold a
 * SmallInteger.
 * If only one value holds a SmallInteger, it means either `< max'
 * or `> min' (respectively if max or min are in-range).
 */

typedef struct ConstraintStruct {
  long			  min, max;
  struct ConstraintStruct *source;
} Constraint;

static Constraint smallIntConstraint = { MIN_ST_INT, MAX_ST_INT, NULL };
static Constraint nonIntConstraint = { MIN_ST_INT-1, MAX_ST_INT+1, NULL };


/* This structure is used by the JIT compiler's bytecode analyzer
 * to store information about basic blocks and the control-flow graph.
 */
typedef struct BasicBlockStruct BasicBlock;
typedef struct GraphEdgeStruct GraphEdge;

struct BasicBlockStruct {
  Byte		*bp;				/* ptr to first bytecode */
  int		size;
  int		offset;				/* from start of bytecodes */
  GraphEdge	*forwardEdges;			/* in edges from before */
  GraphEdge	*backwardEdges;			/* in edges from after */
  int		numForwardEdges;		/* # of in edges from before */
  int		numBackwardEdges;		/* # of in edges from after */
  int		timesVisited;			/* flag for analyzeBasicBlock */
  int		active;				/* flag for graph searches */
  BasicBlock	*next, *satisfiedNext;		/* the out edges */
  Constraint	*constraints;
};

struct GraphEdgeStruct {
  BasicBlock	*block;
  GraphEdge	*next;
};

/* This structure maps BasicBlock structure to where they start in 
 * memory.  It is strictly needed for forward references only, but
 * actually we use it for backward references too.
 */
typedef struct BlockPointerStruct {
  Byte		*bp;
  BasicBlock	*block;
} BlockPointer;


int
isSimpleReturn (byteCodes)
     ByteCodes byteCodes;
{
  Byte	*bytes;
  long	byteCodeLen;

  if (byteCodes == nil) {
    return (0);
  }

  byteCodeLen = byteCodeLength(byteCodes);
  bytes = byteCodes->base;

  /* check for ^self */
  if (byteCodeLen == 1 && bytes[0] == (returnIndexed | receiverIndex)) {
    return (1);
  }

  /* check for ^instanceVariable */
  if (byteCodeLen == 2) {
    if ((bytes[0] & ~15) == pushReceiverVariable
	&& bytes[1] == returnContextStackTop) {
      return (((bytes[0] & 0x0F) << 8) | 2);
    }
  } else if (byteCodeLen == 3) {
    if (bytes[0] == pushIndexed
	&& (bytes[1] & locationMask) == receiverLocation
	&& bytes[2] == returnContextStackTop) {
      return (((bytes[1] & ~locationMask) << 8) | 2);
    }
  }

  /* check for ^firstLiteral */
  if (byteCodeLen == 1) {
    if (bytes[0] == (returnIndexed | trueIndex)) {
      addForcedObject(trueOOP);
      return (3);
    } else if (bytes[0] == (returnIndexed | falseIndex)) {
      addForcedObject(falseOOP);
      return (3);
    } else if (bytes[0] == (returnIndexed | nilIndex)) {
      addForcedObject(nilOOP);
      return (3);
    } else {
      return (0);
    }
  }

  if (byteCodeLen == 2) {
    if (bytes[1] != returnContextStackTop) {
      return (0);
    }

    if (bytes[0] == pushLitConstant) {
      return (3);
    } else if (bytes[0] == (pushSpecial | litMinusOneIndex)) {
      addForcedObject(fromInt(-1));
      return (3);
    } else if (bytes[0] == (pushSpecial | litZeroIndex)) {
      addForcedObject(fromInt(0));
      return (3);
    } else if (bytes[0] == (pushSpecial | litOneIndex)) {
      addForcedObject(fromInt(1));
      return (3);
    } else if (bytes[0] == (pushSpecial | litTwoIndex)) {
      addForcedObject(fromInt(2));
      return (3);
    }
  }

  return (0);
}

/*
 *	int checkKindOfBlock(bc, literals)
 *
 * Description
 *
 *	Look whether the block compiled in the given bytecodes can be
 *	optimized.  Answer a number that identifies the possibility to
 *	optimize it.
 *
 * Inputs
 *
 *	bc:
 *		The bytecodes to be examined.
 *	literals:
 *		The literals array for the block.
 *
 * Outputs
 *
 *	0:	The block is a clean block
 *	1:	The block is self-contained but accesses the receiver
 *	31:	The block contains a return from method or push thisContext
 *	else	The result is the number of outer contexts which the receiver
 *		needs access to, + 1.
 */
int
checkKindOfBlock(bc, literals)
     ByteCodes	bc;
     OOP	*literals;
{
  int		status, newStatus;
  Byte		*bp, *end;
  OOP		blockClosureOOP;

  status = 0;	/* clean block */
  for (bp = bc->base, end = bc->ptr; bp != end; bp += byteCodeSize(*bp)) {
    switch (*bp) {
      case 134:
        if ((bp[1] & 63) == popStoreIntoArray) {
          break;
        }
        /* operation on instance variables - fall through */

      case 0:   case 1:   case 2:   case 3:	/* push instance variable */
      case 4:   case 5:   case 6:   case 7:
      case 8:   case 9:   case 10:  case 11:
      case 12:  case 13:  case 14:  case 15:
      case 96:  case 97:  case 98:  case 99:	/* pop into instance var */
      case 100: case 101: case 102: case 103:
      case 112: case 120: case 140:		/* push self/return/set top */
	if (status == 0) {
	  status = 1;
	}
	break;

      case 32: case 33: case 34: case 35:	/* push literal constant */
      case 36: case 37: case 38: case 39:
      case 40: case 41: case 42: case 43:
      case 44: case 45: case 46: case 47:
      case 48: case 49: case 50: case 51:
      case 52: case 53: case 54: case 55:
      case 56: case 57: case 58: case 59:
      case 60: case 61: case 62: case 63:
	newStatus = checkInnerBlock(literals[*bp & 31]);
	if (newStatus > status) {
	  if (newStatus == 31) {
	    return (31);
	  }
	  status = newStatus;
	}
	break;

      case 124:					/* return from method */
        return (31);

      case 126:					/* big literal operations */
	if ((bp[1] & locationMask) != pushLiteral) {
	  continue;
	}
	blockClosureOOP = literals[((bp[1] & ~locationMask) << 6) | bp[2]];
	newStatus = checkInnerBlock(blockClosureOOP);
	if (newStatus > status) {
	  if (newStatus == 31) {
	    return (31);
	  }
	  status = newStatus;
	}
	break;

      case 137:					/* push this context */
        if (bp[1] != blockCopyColonSpecial) {
          return (31);
        }
        break;

      case 128: case 129: case 130: case 142:	/* 2-byte stack ops */
	if ((bp[1] & locationMask) == receiverLocation && status == 0) {
	  status = 1;
	} else if ((bp[1] & locationMask) == litConstLocation) {
	  newStatus = checkInnerBlock(literals[bp[1] & ~locationMask]);
	  if (newStatus > status) {
	    if (newStatus == 31) {
	      return (31);
	    }
	    status = newStatus;
	  }
	}
	break;

      case 138:					/* outer temp operation */
	if (status < (1 + bp[2])) {
	  status = 1 + bp[2];
	  if (status > 31) {			/* ouch! how deep!! */
	    return (31);
	  }
	}
	break;
    }
  }
  return (status);
}

int
checkInnerBlock(blockClosureOOP)
     OOP	blockClosureOOP;
{
  int		newStatus;
  BlockClosure	blockClosure;
  Block		block;

  if (!isClass(blockClosureOOP, blockClosureClass)) {
    return (0);
  }

  /* This case is the most complicated -- we must check the
   * cleanness of the inner block and adequately change the status.
   *
   * full block:	no way dude -- exit immediately
   * clean block:     	same for us
   * receiver access:	same for us
   * access to temps in the Xth context: from the perspective of the
   * 		    block being checked here, it is like an access to
   * 		    temps in the (X-1)th context
   * access to this block's temps: our outerContext can be nil either,
   *		    but for safety we won't be a clean block.
   */
  blockClosure = (BlockClosure) oopToObj(blockClosureOOP);
  block = (Block) oopToObj(blockClosure->block);
  newStatus = block->header.clean;
  switch(newStatus) {
    case 31: case 0: case 1:	return(newStatus);
    default:			return(newStatus - 1);
  }
}


/*
 *	ByteCodes optimizeByteCodes(byteCodes)
 *
 * Description
 *
 *	Divide the byte codes of a method in basic blocks, optimize each one,
 *	reunite the optimized blocks and return a new vector of byte codes
 *	that contains the stream of optimized byte codes.  Actual optimization
 *	of the basic blocks is optimizeBasicBlock's task; this function
 *	takes care of handling basic blocks and optimizing jumps (which span
 *	multiple basic blocks).
 *
 * Inputs
 *
 *	byteCodes:
 *		A vector of byte codes to be optimized.
 *
 * Outputs
 *
 *	*ANOTHER* vector of byte codes.
 */

int
compareBlocks (a, b)
     register BlockBoundary *a, *b;
{
  return (a->byte - b->byte);
}

ByteCodes
optimizeByteCodes (byteCodes)
     ByteCodes byteCodes;
{
  register BlockBoundary *blocks, *current;
  register Jump		 *jumps;
  register Byte		 *bp;
  Byte	  		 *end, *first;
  int			 num;

#ifdef NO_OPTIMIZE
  return (byteCodes);
#endif

  bp = byteCodes->base;
  end = byteCodes->ptr;
  blocks = alloca(sizeof(BlockBoundary) * (end - bp + 1));
  memset(blocks, 0, sizeof(BlockBoundary) * (end - bp + 1));

  /* 1) Split into basic blocks.  This part cheats so that the final fixup
   * also performs jump optimization. */
  for(current = blocks, num = 0; bp != end; bp += byteCodeSize(*bp)) {
    Byte	*dest = bp;
    mst_Boolean	 canOptimizeJump;
    do {
      canOptimizeJump = false;
      switch (*dest) {
	/* short jump */
	case jumpShort:
	  if (dest[2] == popStackTop) {
	    /* The bytecodes can only be those produced by #ifTrue:/#ifFalse:
	     *	    0: jump to 2
	     *	    1: push nil
	     *	    2: pop stack top
	     * This could not be optimized to a single pop, cause bytecodes 1
	     * and 2 lie in different basic blocks! So we rewrite it to a
	     * functionally equivalent but optimizable bytecode sequence. */
	    *dest = popStackTop;
	    break;
	  }
	  /* Fall through */

	case jumpShort|1: case jumpShort|2: case jumpShort|3:
	case jumpShort|4: case jumpShort|5: case jumpShort|6:
	case jumpShort|7:
	  /* If bp == dest, we could end up writing a 2-byte jump bytecode
	   * where space was only reserved for a 1-byte jump bytecode!
	   * But if we jump to a return, we can safely optimize -- returns
	   * are always one byte */
	  canOptimizeJump = (bp != dest);
	  dest += *dest;
	  dest -= 142;
	  canOptimizeJump |= (*dest >= 120 && *dest <= 125);
	  break;

	/* pop and short jump if false */
	case popJumpFalseShort  : case popJumpFalseShort|1:
	case popJumpFalseShort|2: case popJumpFalseShort|3:
	case popJumpFalseShort|4: case popJumpFalseShort|5:
	case popJumpFalseShort|6: case popJumpFalseShort|7:
	  /* UNCONDITIONAL jumps to CONDITIONAL jumps must not be touched! */
	  if (bp == dest) {
	    dest += *dest;
	    dest -= 150;
	  }
	  break;

	/* long jump, pop and long jump if true, pop and long jump if false */
	case jumpLong  : case jumpLong|1: case jumpLong|2:
	case jumpLong|3: case jumpLong|4: case jumpLong|5:
	case jumpLong|6: case jumpLong|7:
	  /* 2-byte unconditional jump, we can indeed optimize it */
	  canOptimizeJump = true;
	  dest += ((signed int)dest[1]) + jumpOffsets[*dest & 15];
	  break;

	case popJumpTrue   : case popJumpTrue |1:
	case popJumpTrue |2: case popJumpTrue |3:
	case popJumpFalse  : case popJumpFalse|1:
	case popJumpFalse|2: case popJumpFalse|3:
	  /* UNCONDITIONAL jumps to CONDITIONAL jumps must not be touched! */
	  if (bp == dest) {
	    dest += ((signed int)dest[1]) + jumpOffsets[*dest & 15];
	  }
	  break;

	case returnIndexed  :	   case returnIndexed|1:
	case returnIndexed|2:	   case returnIndexed|3:
	case returnMethodStackTop: case returnContextStackTop:
	  /* Return bytecodes - patch the original jump to return directly */
	  if (*bp >= jumpLong) {
	    bp[0] = nopBytecode;
	    bp[1] = *dest;		/* fill both bytes */
	  } else {
	    *bp = *dest;
	  }
	  /* This in fact eliminated the jump, don't split in basic blocks */
	  dest = bp;
	  break;
      }
    } while (canOptimizeJump);
    if (bp != dest) {
      current->byte = dest - byteCodes->base;
      current->id   = ++num;
      current++;
      current->byte = bp - byteCodes->base;
      current->id   = -num;
      current++;
    }
  }

  /* 2) Get the "real" block boundaries by sorting them according to where they
   * happen in the original bytecode.  Note that a simple bucket sort is not
   * enough because multiple jumps could end on the same bytecode, and the
   * same bytecode could be both the start and the destination of a jump! */
  qsort(blocks, current - blocks, sizeof(BlockBoundary), compareBlocks);

  /* 3) Optimize the single basic blocks, and reorganize into `jumps' the
   * data that was put in blocks */
  jumps = alloca(sizeof(Jump) * num);

  for (bp = byteCodes->base; blocks != current; blocks++) {
    first = bp;
    bp = byteCodes->base + blocks->byte;
    optimizeBasicBlock(first, bp);
    if (blocks->id > 0) {
      jumps[ blocks->id - 1].dest = currentByteCodeLength();
    } else {
      jumps[-blocks->id - 1].from = currentByteCodeLength();
    }
  }
  optimizeBasicBlock(bp, end);

  freeByteCodes(byteCodes);
  byteCodes = getByteCodes();

  /* 4) Fix the jumps so that they correctly point to the start of the same
   * basic block */
  for(; num--; jumps++) {
    short offset;

    bp = byteCodes->base + jumps->from;
    offset = jumps->dest - jumps->from - 2;
    if (offset == -1) {			/* jump to following bytecode do */
      if (*bp >= jumpLong) {		/* not exist - use other bytecodes */
	bp[1] = nopBytecode;		/* 2 byte jumps = nop+nop or pop+nop */
      }
      if (*bp & 8) {
	*bp = popStackTop;		/* pop stack top for conditional */
      } else {				/* jumps */
	*bp = nopBytecode;		/* nop for unconditional jumps */
      }
      continue;
    }
    switch (*bp & ~7) {
      /* short jumps */
      case jumpShort:		*bp = jumpShort		| offset; continue;
      case popJumpFalseShort:	*bp = popJumpFalseShort | offset; continue;

      /* long jumps */
      case jumpLong:		*bp = jumpLong | 4; break;
      case popJumpTrue:		*bp &= ~3;	    break;
    }
    *bp++  += offset >> 8;
    *bp	   =  offset & 255;
  }

  return (byteCodes);
}

/*
 *	static ByteCodes optimizeBasicBlock(from, to)
 *
 * Description
 *
 *	Scan the bytecodes between the given addresses. As they are
 *	scanned, they are overwritten with an optimized version; in
 *	the end, compileByteCodes() is used to append them to the
 *	stream of optimized bytecodes.
 *
 * Inputs
 *
 *	from:
 *		Pointer to the first bytecode to be optimized.
 *	to:
 *		Pointer past the last bytecode to be optimized.
 *
 * Outputs
 *
 *	Return whether at least a bytecode was generated.
 */

#define NEXT(size) {	\
  n = size;		\
  opt += size;		\
  continue;		\
}

#define REPLACE(size) {	\
  n = size;		\
  continue;		\
}

#define COPY {			\
  n = byteCodeSize(byte);	\
  opt++;			\
  if(n != 1) *opt++ = *bp++;	\
  if(n == 3) *opt++ = *bp++;	\
  continue;			\
}

#define BEGIN	     if(0) {
#define BYTECODE(n)  } else if(byte == (n)) {
#define RANGE(a, b)  } else if((unsigned char)(byte - (a)) < ((b) - (a))) {
#define EITHER(a, b) } else if(byte == (a) b) {
#define OR(b)  			|| byte == (b)
#define CONDITION(c) } else if(c) {
#define NO_MATCH     } else {
#define END	     }

mst_Boolean
optimizeBasicBlock (from, to)
     Byte *from;
     Byte *to;
{
  register Byte byte, *bp, *opt;
  int  n;

  bp = opt = from;
  n = 0;
  while (bp != to) {
    byte = *opt = *bp++;
    BEGIN
      RANGE(returnIndexed, returnContextStackTop)
	opt++;
	break; /* this `break' performs unreachable code elimination! */

      BYTECODE(nopBytecode)
	REPLACE(n);

      BYTECODE(notEqualSpecial)
	if (!n) NEXT(1);
	if (opt[-1] == (pushSpecial | nilIndex)) {  /* x ~= nil */
	  opt[-1] = notNilSpecial;
	  REPLACE(1);
	}
	NEXT(1);

      EITHER(sameObjectSpecial,
        OR(equalSpecial))
	if (!n) NEXT(1);
	if (opt[-1] == (pushSpecial | nilIndex)) {  /* x = nil, x == nil */
	  opt[-1] = isNilSpecial;
	  REPLACE(1);
	}
	NEXT(1);

      BYTECODE(popStackTop)
 	if (n) {
	  byte = opt[-n];
	  BEGIN
	    CONDITION(isPushByteCode(byte))   /* push/pop */
	      opt -= n;
	      NEXT(0);

	    BYTECODE(storeIndexed)   /* store/pop */
	      byte = *--opt;	 /* get data byte */
	      opt--;	  /* move to opcode */
	      if (byte < 8) {
		*opt = popReceiverVariable | byte;
		NEXT(1);
	      } else if (byte >= 64 && byte < 72) {
		*opt = popTemporaryVariable | (byte & 63);
		NEXT(1);
	      } else {
		*opt = popStoreIndexed;
		NEXT(2);
	      }

	    EITHER(bigLiteralsBytecode,
	      OR(bigInstanceBytecode)
	      OR(outerTempBytecode))
	      byte = opt[-2];	  /* get second byte */
	      if (byte < popStoreVariable) {
		opt -= n;    /* push/pop */
		NEXT(0);
	      } else if (byte >= storeVariable) {
		opt[-2] ^= (popStoreVariable ^ storeVariable);
		REPLACE(3);
	      }
	  END
	}

	if (bp != to && (*bp & ~3) == returnIndexed) {
	  *opt++ = *bp++;   /* pop/return */
	  break;    /* kill unreachable code */
	}
	NEXT(1);

      CONDITION(isPushByteCode(byte))	 /* push/push -> dup */
	if (!n) COPY;
	if(opt[-n] == *opt) {
	  if (n == 1) {
	    *opt = dupStackTop;
	    NEXT(1);
	  } else if (opt[-1] == *bp) {
	    *opt = dupStackTop;
	    bp++;
	    NEXT(1);
	  }
	}

	BEGIN			/* pop-store/push -> store */
	  RANGE(pushReceiverVariable, pushReceiverVariable | 7)
	    if (opt[-n] == (popReceiverVariable | (byte & 15))) {
	      opt[-1] = storeIndexed;
	      *opt++ = receiverLocation | (byte & 15);
	      REPLACE(2);
	    }
	    if(opt[-n] == popStackTop) {	/* pop/push -> replace */
	      opt--;
	      *opt++ = replaceIndexed;
	      *opt++ = receiverLocation | (byte & 15);
	      REPLACE(2);
	    }

	  RANGE(pushReceiverVariable | 8, pushReceiverVariable | 15)
	    if (opt[-n] == popStoreIndexed
		&& opt[-1] == (receiverLocation | (byte & 15))) {
	      opt[-2] = storeIndexed;
	      REPLACE(2);
	    }
	    if(opt[-n] == popStackTop) {
	      opt--;
	      *opt++ = replaceIndexed;
	      *opt++ = receiverLocation | (byte & 15);
	      REPLACE(2);
	    }

	  RANGE(pushTemporaryVariable, pushTemporaryVariable | 7)
	    if (opt[-n] == (popTemporaryVariable | (byte & 15))) {
	      opt[-1] = storeIndexed;
	      *opt++ = temporaryLocation | (byte & 15);
	      REPLACE(2);
	    }
	    if(opt[-n] == popStackTop) {
	      opt--;
	      *opt++ = replaceIndexed;
	      *opt++ = temporaryLocation | (byte & 15);
	      REPLACE(2);
	    }

	  RANGE(pushTemporaryVariable | 8, pushTemporaryVariable | 15)
	    if (opt[-n] == popStoreIndexed
		&& opt[-1] == (temporaryLocation | (byte & 15))) {
	      opt[-2] = storeIndexed;
	      REPLACE(2);
	    }
	    if(opt[-n] == popStackTop) {
	      opt--;
	      *opt++ = replaceIndexed;
	      *opt++ = temporaryLocation | (byte & 15);
	      REPLACE(2);
	    }

	  RANGE(pushLitVariable, pushLitVariable | 31)
	    if (opt[-n] == popStoreIndexed
		&& opt[-1] == (litVarLocation | (byte & 31))) {
	      opt[-2] = storeIndexed;
	      REPLACE(2);
	    }
	    if(opt[-n] == popStackTop) {
	      opt--;
	      *opt++ = replaceIndexed;
	      *opt++ = litVarLocation | (byte & 31);
	      REPLACE(2);
	    }

	  RANGE(pushLitConstant, pushLitConstant | 31)
	    if(opt[-n] == popStackTop) {
	      opt--;
	      *opt++ = replaceIndexed;
	      *opt++ = litConstLocation | (byte & 31);
	      REPLACE(2);
	    }

	  BYTECODE(pushSpecial | receiverIndex)
	    if (opt[-n] == popStackTop) {
	      opt[-1] = replaceSelf;
	      REPLACE(1);
	    }

	  BYTECODE(pushSpecial | litOneIndex)
	    if (opt[-n] == popStackTop) {
	      opt[-1] = replaceOne;
	      REPLACE(1);
	    }

	  BYTECODE(pushIndexed)
	    byte = *bp++;
	    if (opt[-n] == popStoreIndexed) {
	      opt[-2] = storeIndexed;
	      if (opt[-1] == byte) {
	        REPLACE(2);
	      } else {
	        *opt = replaceIndexed;
	      }
	    } else if (opt[-n] == popStackTop) {
	      *--opt = replaceIndexed;
	    }
	    opt[1] = byte;
	    NEXT(2);
	END;

	NEXT(1);	/* no match */

      EITHER(bigLiteralsBytecode,
	OR(bigInstanceBytecode)
	OR(outerTempBytecode))

	if (!n || opt[-n] != byte) COPY;

	byte = opt[-2];    /* get second byte */
	if (byte < popStoreVariable) {
	  if (byte == *bp && opt[-1] == bp[1]) {	/* push/push -> dup */
	    *opt = dupStackTop;
	    bp += 2;
	    NEXT(1);
	  }
	} else if (byte < storeVariable) {	/* pop-store/push -> store */
	  if ((byte & 63) == (*bp & 63) && opt[-1] == bp[1]) {
	    opt[-2] ^= (popStoreVariable ^ storeVariable);
	    bp += 2;
	    REPLACE(3);
	  }
        }

	opt++;
	*opt++ = *bp++;
	*opt++ = *bp++;
	REPLACE(3);

      NO_MATCH
        COPY;
    END;
  }
  compileByteCodes(from, opt);
  return (opt != from);
}


/*
 *	void computeStackPositions(byteCodes, size, base, pos)
 *
 * Description
 *
 *	Fill a table that says which stack slot is touched by each bytecode.
 *
 * Inputs
 *
 *	bp:
 *		A pointer to some byte codes for which the destination table
 *		must be computed.
 *	size:
 *		The size of the bytecodes
 *	base:
 *		A pointer to the array that will be used as the stack (an
 *		array of pointers); pos[x] == &base[0] if bp[x] writes in the
 *		first stack slot, pos[x] == &base[1] if bp[x] writes in the
 *		second stack slot, etc.
 *	pos:
 *		A pointer to the first location where the table will be
 *		written
 *
 * Outputs
 *
 *	pos is filled.
 */


void
computeStackPositions(bp, size, base, pos)
     Byte	*bp;
     int	size;
     voidPtr	*base;
     voidPtr	**pos;
{
  Byte		*end;
  int		balance, ofs;
  static int	stackOpBalanceTable[16] = {
     1, 1, -1, 0, /* 126     (push, push, pop/store, store)      */
     0, 0, 0,  0, /* unused 				         */
    -1, 1, -1, 0, /* 134     (pop/store, push, pop/store, store) */
   255, 1, -1, 0, /* 138     (invalid, push, pop/store, store)   */
  };
  
  memzero(pos, sizeof(voidPtr *) * size);

  pos[0] = base;
  for(end = bp + size; bp != end; 
      pos += byteCodeSize(*bp), bp += byteCodeSize(*bp)) {

    switch (*bp) {
      /* 3-byte stack operations */
      case bigLiteralsBytecode:
      case bigInstanceBytecode:
      case outerTempBytecode:
        balance = stackOpBalanceTable[(*bp - 126) | (bp[1] >> 6)];
        break;

      /* 1-byte sends */
      case send1ExtByte:
      case sendSuper1ExtByte:
        balance = -(bp[1] >> 5);
        break;

      /* 2-byte send */
      case send2ExtByte:
        balance = -(bp[1] & 31);
        break;

      /* Everything else */
      default:
        balance = stackBalanceTable[*bp];
	if (*bp >= jumpShort) {
	  if (*bp < jumpLong) {
	    /* short jumps */
	    ofs = (*bp & 7) + 2;
	    pos[ofs] = pos[0] + balance;
	  } else if (*bp < plusSpecial) {
	    /* long jumps */
	    ofs = ((signed int)bp[1]) + jumpOffsets[*bp & 15];
	    if (ofs > 0) {
	      pos[ofs] = pos[0] + balance;
	    }
	  }
	}
    }

    if (balance == 255) {
      errorf("Invalid bytecode encountered during bytecode analysis");
      balance = 0;
    }

    if (!pos[byteCodeSize(*bp)]) {
      pos[byteCodeSize(*bp)] = pos[0] + balance;
    }
  }
}


/*
 *	void analyzeBytecodes(methodOOP, size, dest, inIntTab, obstack)
 *
 * Description
 *
 *	Fill a table that says which bytecodes leave a SmallInteger as
 *	the stack-top
 *
 * Inputs
 *
 *	methodOOP:
 *		An OOP for the method that is being compiled (either
 *		a CompiledMethod or CompiledBlock).
 *	size:
 *		The size of the bytecodes
 *	dest:
 *		A pointer to char; on output, dest[i] is non-zero if and
 *		only if bp[i] is the destination of a jump. It is positive
 *		for a forward jump and negative for a backward jump.
 *	intTab:
 *		A table containing TOP_UNKNOWN for don't-knows,
 *		TOP_IS_INTEGER for is-a-SmallInteger and
 *		TOP_IS_NOT_INTEGER for isn't-a-SmallInteger.
 *	obstack:
 *		An obstack into which we allocate temporary data.
 *
 * Outputs
 *
 *	dest and intTab are filled.
 */
static int		stackDepth;
static Constraint	*selfConstraint;
static char		*intTab;
static OOP		classOOP;
static OOP		*literals;
static struct obstack	*dataObstack;

void
analyzeByteCodes(methodOOP, size, dest, inIntTab, obstack)
     OOP	 methodOOP;
     int	 size;
     char	 *dest, *inIntTab;
     struct obstack *obstack;
{
  Byte		*bp, *end;
  BasicBlock	*basicBlocks, *bb;
  BlockPointer	*blockHeadings;
  int		structSize, numJumps, numTemps, n;

  dataObstack = obstack;
  intTab = inIntTab;
  memzero(intTab, sizeof(char) * size);

  bp = getMethodByteCodes(methodOOP);
  numJumps = makeDestinationTable(bp, size, dest);

#ifdef NO_TYPE_INFERENCE
  return;
#endif

  classOOP = getMethodClass(methodOOP);
  literals = getMethodLiterals(methodOOP);

  if (oopClass(methodOOP) == compiledBlockClass) {
    stackDepth = getBlockHeader(methodOOP).depth;
    numTemps = getBlockHeader(methodOOP).numTemps;
  } else {
    stackDepth = getMethodHeader(methodOOP).stackDepth;
    numTemps = getMethodHeader(methodOOP).numTemps;
  }

  stackDepth = CTX_SIZE(stackDepth);

  selfConstraint = 
    (classOOP == smallIntegerClass) ? &smallIntConstraint : &nonIntConstraint,

  /* We have * 2 because a basic block has a single entry
   * and a single exit.
   */
  blockHeadings = (BlockPointer *) obstack_alloc(dataObstack, 
    sizeof(BlockPointer) * numJumps * 2 + 1);

  structSize = sizeof(BasicBlock) + sizeof(Constraint) * (stackDepth - 1);
  bb = basicBlocks = (BasicBlock *)obstack_alloc(dataObstack, structSize);
    
  for (end = bp + size, n = 0;;) {
    mst_Boolean fallsThrough;
    blockHeadings[n].bp = bp;
    blockHeadings[n].block = bb;
    fallsThrough = nextBasicBlock(bb, bp, dest);

    /* Go on, please ... */
    bp += bb->size;
    if (bp == end) {
      break;
    }
    dest += bb->size;
    n++;

    /* Make the previous block point to this one */
    bb->next = fallsThrough ? (BasicBlock *) bp : NULL;
    bb = (BasicBlock *) obstack_alloc(dataObstack, structSize);
  }

  connectBasicBlocks(basicBlocks, blockHeadings, n);

  /* Calls itself recursively, visiting the graph depth-first */
  constraintReset(basicBlocks->constraints, numTemps);
  analyzeBasicBlock(basicBlocks, NULL, 0, numTemps);

  obstack_free(dataObstack, basicBlocks);
}

/*
 *	int makeDestinationTable(byteCodes, size, destination)
 *
 * Description
 *
 *	Fill a table that says to which bytecodes a jump lands.
 *
 * Inputs
 *
 *	bp:
 *		A pointer to some byte codes for which the destination table
 *		must be computed.
 *	size:
 *		The size of the bytecodes
 *	dest:
 *		A pointer to char; on output, dest[i] is non-zero if and
 *		only if bp[i] is the destination of a jump. It is positive
 *		for a forward jump and negative for a backward jump.
 *
 * Outputs
 *
 *	dest is filled; the number of jumps is returned.
 */

int
makeDestinationTable(bp, size, dest)
     Byte	 *bp;
     int	  size;
     char	 *dest;
{
  Byte *end;
  int	n;
  
  memzero(dest, sizeof(char) * size);

  for(n = 0, end = bp + size; bp != end; 
      dest += byteCodeSize(*bp), bp += byteCodeSize(*bp)) {

    if (*bp >= jumpShort) {
      if (*bp < jumpLong) {
	/* short jumps */
	dest[(*bp & 7) + 2] = 1;
	n++;
      } else if (*bp < plusSpecial) {
	int ofs;
	ofs = ((signed int)bp[1]) + jumpOffsets[*bp & 15];
	/* long jumps */
	dest[ofs] = (ofs <= 0) ? -1 : 1;
	n++;
      }
    }
  }

  return (n);
}


/*
 *	void connectBasicBlocks(basicBlocks, blockHeadings, numHeadings)
 *
 * Description
 *
 *	Change the `next' and `codeNext' pointers in basicBlocks from pointers 
 *	to bytecodes to pointers to another BasicBlock structure.  Also
 *	fills in the `offset' field.
 *
 * Inputs
 *
 *	basicBlocks:
 *		A list of BasicBlock structures containing the CFG
 *		for the method.
 *	blockHeadings:
 *		An array that pairs the address of the starting bytecodes
 *		to the corresponding BasicBlock structures, sorted by
 *		the address of the starting bytecode.
 *	numHeadings:
 *		The size of the blockHeadings array
 *
 * Outputs
 *
 *	See Description
 */
int
findBlockHeading (ofs, blkPtr)
     register Byte	   *ofs;
     register BlockPointer *blkPtr;
{
  return (ofs - blkPtr->block->bp);
}

void
connectBasicBlocks(basicBlocks, blockHeadings, numHeadings)
     BlockPointer	 *blockHeadings;
     int		 numHeadings;
{
  register BlockPointer *heading, *thisOne;
  register BasicBlock	*successor, *thisBlock;
  register Byte		*base = blockHeadings[0].bp;
  int			left;
  
  for (thisOne = blockHeadings, left = numHeadings; left--; thisOne++) {
    thisBlock = thisOne->block;
    thisBlock->offset = thisBlock->bp - base;

    if (!thisBlock->next) {
      /* This block returns. */
      continue;
    }

    if ((Byte *) thisBlock->next == thisOne[1].bp) {
      /* Optimize the common case when the successor follows this
       * block in the bytecode stream */
      successor = thisOne[1].block;
    } else {
      heading = bsearch(thisBlock->next,
	blockHeadings, numHeadings, sizeof(BlockPointer),
	findBlockHeading);
      
      successor = heading->block;
    }
    thisBlock->next = successor;
    addEdge (thisBlock, successor);

    if (thisBlock->satisfiedNext) {
      /* Do another search if we have a conditional branch.  Conditional
       * jumps only go forward, so we can reduce the binary search range.
       * (actually, because of caching it might be faster to search
       * everything...)
       */
      heading = bsearch(thisBlock->satisfiedNext,
        thisOne + 1, left - 1, sizeof(BlockPointer),
        findBlockHeading);
      
      successor = heading->block;
      thisBlock->satisfiedNext = successor;
      addEdge (thisBlock, successor);
    }
  }
}

/*
 *	void nextBasicBlock(basicBlock, bp, dest)
 *
 * Description
 *
 *	Fill the basicBlock structure with information on the basic block
 *	whose start is pointed by bp.  All the fields but `next' and
 *	`offset' are filled.
 *
 * Inputs
 *
 *	basicBlock:
 *		A pointer to a BasicBlock structure to be filled.
 *	bp:
 *		A pointer to the first bytecode in the BasicBlock.
 *	dest:
 *		A pointer that parallels bp into the table prepared
 *		by makeDestinationTable.
 *
 * Outputs
 *
 *	See Description; in addition it returns whether execution 
 *	falls into the immediately following block at the end of
 *	this one.
 */
mst_Boolean
nextBasicBlock(basicBlock, bp, dest)
     register BasicBlock *basicBlock;
     register Byte	 *bp;
     register char	 *dest;
{
  mst_Boolean fallsThrough = true;

  basicBlock->bp = bp;
  basicBlock->satisfiedNext = NULL;
  basicBlock->numBackwardEdges = basicBlock->numForwardEdges = 0;
  basicBlock->active = 0;

  for (;;) {
    /* Go on until we find a branch bytecode, a return bytecode,
     * or a jump destination. */
    if ((*bp >= returnIndexed && *bp <= returnContextStackTop)
      || (*bp >= jumpShort && *bp < plusSpecial)
      || (*dest && bp != basicBlock->bp)) {
      break;
    }
    
    dest += byteCodeSize(*bp);
    bp += byteCodeSize(*bp);
  }

  /* If we got here because a jump lands on bp, we are done --
   * that bytecode actually belongs to the *next* basic block.
   * Else, we must increment bp to include the jump or return
   * in the basic block, and fill the `next' or `satisfiedNext'
   * field for unconditional or conditional jumps, respectively. */
  if (!*dest || bp == basicBlock->bp) {
    if (*bp >= jumpShort && *bp < plusSpecial) {
      int ofs;

      ofs = *bp < jumpLong
	? (*bp & 7) + 2
	: ((signed int)bp[1]) + jumpOffsets[*bp & 15];

      if ((*bp & 8) != (jumpShort & 8)) {
        basicBlock->satisfiedNext = (BasicBlock *) (bp + ofs);
        fallsThrough = true;
      } else {
        basicBlock->next = (BasicBlock *) (bp + ofs);
        fallsThrough = false;
      }
    } else {
      /* A return */
      fallsThrough = false;
    }

    /* Include the jump or return in the basic block */
    bp += byteCodeSize(*bp);
  }

  basicBlock->size = bp - basicBlock->bp;
  return (fallsThrough);
}

/* This does the actual work for analyzeBasicBlock, walking through the 
 * block's bytecodes.  This however does not know about loops, nor does
 * it know about multiple predecessors.  The algorithm is as follows:
 * - trace the execution of the bytecodes as long as they're not jumps
 *   or boolean operations.  Sends to self are inspected for primitives
 *   that return SmallIntegers.
 *
 * - on a boolean operation, check if it is followed by a jump, and if
 *   both LHS and RHS are known to be SmallIntegers.  In this case,
 *   move our own output constraints to the both successor BasicBlocks,
 *   merging either the result of the boolean operation, or its inverse,
 *   with the current constraints about the LHS or RHS (and its source,
 *   and its source's source, etc...)
 *
 *	(Simple) example:
 *		i <= 1000
 *
 *	Stack contains:
 *		i		MIN_ST_INT .. MAX_ST_INT   <----.
 *		...						| source
 *		LHS		MIN_ST_INT .. MAX_ST_INT -------'
 *		RHS		1000	   .. 1000
 *
 *	The `true' basic block will have i constrained to MIN_ST_INT .. 1000,
 *	the `false' basic block will have i constrained to 1001 .. MAX_ST_INT.
 *
 *	(Complex) example:
 *		i <= j
 *
 *	Stack contains:
 *		j		1          .. 400   <--------------.
 *		i		MIN_ST_INT .. 250   <-----------.  |
 *		...						|  |
 *		LHS		MIN_ST_INT .. 250   ------------'  |
 *		RHS		1   	   .. 400   ---------------'
 *
 *	The `true' basic block will have no gain, because its constraints
 *	will still be 1..400 and MIN_ST_INT..250.  But imposing i.min > j.min
 *	and j.max < i.max (see constraintLessEqual) in the `false' basic
 *	block, constrains i to 2 .. 250 and j to 1 .. 249.
 *
 * - on an unconditional jump, go on evaluating the destination.
 *   That means, move our own output constraints to the successor
 *   BasicBlock and recurse.
 *
 * - on a jump that was not preceded by a boolean, evaluate both
 *   branches, moving our own output constraints to both successor
 *   BasicBlocks and recursing.
 *
 * Actually, the recursion part is done in analyzeBasicBlock, not here.
 */
void
propagateConstraints(basicBlock, intTab)
     BasicBlock	 *basicBlock;
     char	 *intTab;
{
  Constraint *sp, *temps;
  Byte *bp, *end, *intTabPtr;
  Byte b, inverseCondition;
  int n, spOffset, top;

  for (bp = basicBlock->bp, end = bp + basicBlock->size,
       sp = temps = basicBlock->constraints,
       intTabPtr = intTab + basicBlock->offset,
       top = TOP_UNKNOWN;
       
       bp < end;
       bp += byteCodeSize(b),
       intTabPtr += byteCodeSize(b)) {

    *intTabPtr = top;
    b = *bp;
    switch (b) {
      case 0: case 1: case 2: case 3:		/* push inst var */
      case 4: case 5: case 6: case 7:
      case 8: case 9: case 10: case 11:
      case 12: case 13: case 14: case 15:
	top = constraintReset(sp++, 1);
	continue;

      case 16: case 17: case 18: case 19:
      case 20: case 21: case 22: case 23:
      case 24: case 25: case 26: case 27:
      case 28: case 29: case 30: case 31: 
        top = constraintCopy(temps[b - 16], sp++);
	continue;

      case 32: case 33: case 34: case 35:	/* push literal */
      case 36: case 37: case 38: case 39:
      case 40: case 41: case 42: case 43:
      case 44: case 45: case 46: case 47:
      case 48: case 49: case 50: case 51: 
      case 52: case 53: case 54: case 55:
      case 56: case 57: case 58: case 59:
      case 60: case 61: case 62: case 63:
        top = constraintSet(literals[b - 32], sp++);
	continue;

      case 64: case 65: case 66: case 67:	/* push global */
      case 68: case 69: case 70: case 71:
      case 72: case 73: case 74: case 75:
      case 76: case 77: case 78: case 79:
      case 80: case 81: case 82: case 83:
      case 84: case 85: case 86: case 87:
      case 88: case 89: case 90: case 91: 
      case 92: case 93: case 94: case 95:
	top = constraintReset(sp++, 1);
	continue;

      case 96: case 97: case 98: case 99:	/* pop inst var */
      case 100: case 101: case 102: case 103:
        sp--;
        top = TOP_UNKNOWN;
        continue;

      case 104: case 105: case 106: case 107:	/* pop temp var */
      case 108: case 109: case 110: case 111:
        top = constraintCopy(--sp, temps[b - 104]);
        continue;

      case 112:					/* push self */
        top = constraintCopy(selfConstraint, sp++);
        continue;

      case 113: case 114: case 115:		/* push true/false/nil */
        top = constraintReset(sp++, 1);
        continue;

      case 116: case 117: case 118: case 119:	/* push -1..2 */
        top = constraintSet(fromInt(b - 117), sp++);
        continue;

      case 120: case 121: case 122: case 123:		/* returns */
      case 124: case 125:
        continue;

      case 126:		/* big literals */
        n = (bp[1] * 256 + bp[2]) & 16383;
        switch (bp[1] & operationMask) {
          case pushLiteral:	 top = constraintSet(literals[n], sp++); break;
          case pushVariable:	 top = constraintReset(sp++, 1); break;
          case popStoreVariable: top = TOP_UNKNOWN; sp--; break;
          case storeVariable:    break;
        }
        continue;

      case 128:		/* push */
	n = bp[1] & 63;
	switch(bp[1] & locationMask) {
	  case receiverLocation:  top = constraintReset(sp++, 1); break;
	  case temporaryLocation: top = constraintCopy(temps[n], sp++); break;
	  case litConstLocation:  top = constraintSet(literals[n], sp++); break;
	  case litVarLocation:    top = constraintReset(sp++, 1); break;
	}
	continue;

      case 129:		/* store */
	n = bp[1] & 63;
	if((bp[1] & locationMask) == temporaryLocation) {
	  top = constraintCopy(sp - 1, temps[n]);
	}
	continue;

      case 130:			/* pop/store */
	n = bp[1] & 63;
	--sp;
	if((bp[1] & locationMask) == temporaryLocation) {
	  top = constraintCopy(sp, temps[n]);
	}
	continue;

      case 131:			/* 2-byte send */
      case 133:
        sp -= (bp[1] >> 5);
        top = constraintReset(sp - 1, 1);
        continue;

      case 132:			/* 3-byte send */
        sp -= (bp[1] & 31);
        top = constraintReset(sp - 1, 1);
        continue;

      case 134:			/* big instance var */
        switch (bp[1] & operationMask) {
          case popStoreIntoArray: top = TOP_UNKNOWN; sp--; break;
          case pushVariable:	  top = constraintReset(sp++, 1); break;
          case popStoreVariable:  top = TOP_UNKNOWN; sp--; break;
          case storeVariable:     break;
        }
        continue;

      case 135:		/* pop */
        sp--;
        top = TOP_UNKNOWN;
        continue;

      case 136:		/* dup */
        top = constraintCopy(sp - 1, sp);
        sp++;
        continue;

      case 137:		/* push thisContext */
        top = constraintReset(sp++, 1);
        continue;

      case 138:
        switch (bp[1] & operationMask) {
          case 0:		  break;
          case pushVariable:	  top = constraintReset(sp++, 1); break;
          case popStoreVariable:  top = TOP_UNKNOWN; sp--; break;
          case storeVariable:     break;
        }
        continue;

      case 139:		/* nop */
        continue;

      case 140:		/* top = self */
        top = constraintCopy(selfConstraint, sp);
        continue;

      case 141:		/* top = 1 */
        top = constraintSet(fromInt(1), sp);
        continue;

      case 142:		/* set top */
	n = bp[1] & 63;
	switch(bp[1] & locationMask) {
	  case receiverLocation:  top = constraintReset(sp, 1); break;
	  case temporaryLocation: top = constraintCopy(temps[n], sp); break;
	  case litConstLocation:  top = constraintSet(literals[n], sp); break;
	  case litVarLocation:    top = constraintReset(sp, 1); break;
	}
	continue;

      case 143:		/* exit */
        continue;

      case 152: case 153: case 154: case 155:	/* conditional jump */
      case 156: case 157: case 158: case 159:
      case 168: case 169: case 170: case 171:
      case 172: case 173: case 174: case 175:
        sp--;
        analyzeBasicBlock(basicBlock->satisfiedNext, basicBlock,
          0, sp - basicBlock->constraints);
        /* fall through */

      case 144: case 145: case 146: case 147:	/* unconditional jump */
      case 148: case 149: case 150: case 151:
      case 160: case 161: case 162: case 163:
      case 164: case 165: case 166: case 167:
        analyzeBasicBlock(basicBlock->next, basicBlock,
          0, sp - basicBlock->constraints);

        top = TOP_UNKNOWN;
        continue;

      case 176:	/* plus */
        sp--;
        top = constraintSum(sp - 1, sp, sp - 1);
        continue;

      case 177:	/* minus */
        sp--;
        top = constraintSubtract(sp - 1, sp, sp - 1);
        continue;

      case 184:	/* times */
        sp--;
        top = constraintMultiply(sp - 1, sp, sp - 1);
        continue;

      case 186:	/* remainder */
        sp--;
        top = constraintRemainder(sp - 1, sp, sp - 1);
        continue;

      case 188:	/* shift */
        sp--;
        top = constraintBitShift(sp - 1, sp, sp - 1);
        continue;

      case 189: /* int division */
        sp--;
        top = constraintDivision(sp - 1, sp, sp - 1);
        continue;

      case 190:	/* and */
        sp--;
        top = constraintBitAnd(sp - 1, sp, sp - 1);
        continue;

      case 191:	/* or */
        sp--;
        top = constraintBitOr(sp - 1, sp, sp - 1);
        continue;

      case 194: case 195: case 197: case 199:	/* miscellaneous 1-arg */
      case 201: case 204: case 206: case 207:

      case 208: case 209: case 210: case 211:	/* 1-byte sends, 0 arg */
      case 212: case 213: case 214: case 215:
      case 216: case 217: case 218: case 219:
      case 220: case 221: case 222: case 223:
        top = constraintReset(sp - 1, 1);
        continue;


      case 185: case 192: case 196: case 198:  /* miscellaneous 1-argument */
      case 200: case 202: case 203: case 205:

      case 224: case 225: case 226: case 227:	/* 1-byte sends, 1 arg */
      case 228: case 229: case 230: case 231:
      case 232: case 233: case 234: case 235:
      case 236: case 237: case 238: case 239:
        sp -= 1;
        top = constraintReset(sp - 1, 1);
        continue;

      case 193:					/* #at:put: */

      case 240: case 241: case 242: case 243:	/* 1-byte sends, 2 arg */
      case 244: case 245: case 246: case 247:
      case 248: case 249: case 250: case 251:
      case 252: case 253: case 254: case 255:
        sp -= 2;
        top = constraintReset(sp - 1, 1);
        continue;
    }

    /* Here we have a relational operator */
    if ((bp[1] < 152 || bp[1] > 159) &&		/* not a 1-byte conditional jump */
        (bp[1] < 168 || bp[1] > 175)) {		/* not a 2-byte conditional jump */
      sp -= 1;
      top = constraintReset(sp - 1, 1);
      continue;
    }

    /* We can make the constraints more strict */
    spOffset = sp - basicBlock->constraints;
    inverseCondition = (b == 182 || b == 183) ? b ^ 1 : b ^ 3;
    analyzeBasicBlock(basicBlock->satisfiedNext, basicBlock, b, spOffset - 1);
    analyzeBasicBlock(basicBlock->next, basicBlock, inverseCondition, spOffset - 1);

    /* We just processed the jump -- the basic block ends here.  Update
     * the intTab for the jump's bytecode. */
    intTabPtr[1] = TOP_IS_NOT_INTEGER;
    break;
  }
}

/* The most complicated thing to do here is handling loops.  We scan
 * the loop body to search for written variables and, if they were
 * SmallIntegers, we widen their constraints at the loop entry point,
 * setting the worst possible range (either MIN_ST_INT .. MAX_ST_INT,
 * <= MAX_ST_INT, or >= MIN_ST_INT).
 *
 * We then perform our normal chores, which will pass a reduced
 * constraint to the loop (for example to MIN_ST_INT .. 1000 for a
 * `1 to: 1000 do: [ :i | ... ]' loop).  When we re-enter the
 * basic block, we merge our in-constraint with the loop's out-constraint
 * (in this case, we will merge 1 .. 1 with MIN_ST_INT .. 1001 --
 * 1001 because the loop body increments i) and propagate constraints
 * without doing any recursion in the loop.
 *
 * What if the loop decrements i instead of incrementing? The same
 * holds, because we re-enter the basic-block with a <= 999 constraint.
 */
void
analyzeBasicBlock(basicBlock, enterFrom, condition, topOfStack)
     BasicBlock	 *basicBlock, *enterFrom;
     Byte	 condition;
     int	 topOfStack;
{
  if (!basicBlock)
    return;

  basicBlock->timesVisited++;

  if (enterFrom) {
    /* First, work from the back */
    if (basicBlock->numForwardEdges) {
      if (basicBlock->timesVisited == 0) {
	constraintMergeBlock (enterFrom->constraints, basicBlock->constraints,
	  condition, topOfStack);
      } else {
	constraintCopyBlock (enterFrom->constraints, basicBlock->constraints,
	  condition, topOfStack);
      }

      if (--basicBlock->numForwardEdges) {
	/* Only do the actual analysis the last time we're through */
	return;
      }

      basicBlock->timesVisited = -basicBlock->numBackwardEdges;
    }

    if (basicBlock->numBackwardEdges) {
      if (basicBlock->timesVisited == -basicBlock->numBackwardEdges) {
	/* Unconstrain the variables on the first pass */
	visitBackwardEdges(basicBlock, unconstrainWrittenVariables, basicBlock);
      
      } else {
	if (basicBlock->timesVisited < 0) {
	  /* Add any information on the changes made by the loop body */
	  constraintMergeBlock (enterFrom->constraints, basicBlock->constraints,
	    condition, topOfStack);

	  /* HOW DO I PROPAGATE THE CONSTRAINTS IN THE CONDITION WITHOUT
	   * DOING RECURSION IN THE LOOP BODY???
	   * idea:
	   *    - visit the backward edges, find the last basic block
	   *	  containing a backward jump to basicBlock.
	   *    - visit the whole loop (see unconstrainWrittenVariable)
	   *	  and find the last basic block that jumps over the
	   *	  basic block we had found.
	   *	- somehow make propagateConstraints recurse until that
	   *	  basic block and no further.  To do so, we could
	   *	  change restartAnalysisFor into, say,
	   *	  `restrictAnalysisRange'.
	   */
	}
	return;
      }
    }
  }

  propagateConstraints(basicBlock, intTab);
}

/* Walk the CFG down, depth-first, from `loopBlock', which initially
 * should be the same as entryPoint. */
void
unconstrainWrittenVariables(loopBlock, entryPoint)
     BasicBlock *loopBlock;
     BasicBlock	*entryPoint;
{
  register Byte		*bp, *end;
  register Constraint	*c;

  for (bp = loopBlock->bp, end = bp + loopBlock->size;
       bp < end;
       bp += byteCodeSize(*bp)) {
    
    if ((*bp & ~7) == popTemporaryVariable) {
      c = &entryPoint->constraints[*bp & 7];

    } else if ((*bp == storeIndexed || *bp == popStoreIndexed) &&
    		(bp[1] & locationMask) == temporaryLocation) {
      c = &entryPoint->constraints[bp[1] & ~locationMask];
    } else {
      continue;
    }

    /* Found a `store-into-temporary' bytecode.  If the variable was
     * integer, instead of unconstraining we set the worst possible
     * conditions.  This is because if they do bitwise operations
     * the result cannot in any case exceed MAX_ST_INT or MIN_ST_INT
     * (no carries can occur); instead if they do most arithmetic
     * operations (+, - *) an overflow will occur.
     *
     * ### This needs to be reviewed later though.
     */
    if (c->min >= MIN_ST_INT) {
      c->min = MIN_ST_INT;
    }
    if (c->max <= MAX_ST_INT) {
      c->max = MAX_ST_INT;
    }
  }

  /* Assumes no jumps to the middle of a loop -- always true
   * because Smalltalk has no goto. */
  if (loopBlock != entryPoint) {
    visitForwardEdges(loopBlock, unconstrainWrittenVariables, entryPoint);
  }
}


/* Functions to operate on Constraint operands follow */

#define EXIT_IF_OUT_OF_RANGE				\
  if (a->min < MIN_ST_INT || a->max > MAX_ST_INT	\
      || b->min < MIN_ST_INT || b->max > MAX_ST_INT) {	\
    return;						\
  }

#define CHECK_OUT_OF_RANGE_STRICT			\
  if (a->min < MIN_ST_INT || a->max > MAX_ST_INT	\
      || b->min < MIN_ST_INT || b->max > MAX_ST_INT) {	\
    result->min = MIN_ST_INT-1;				\
    result->max = MAX_ST_INT+1;				\
    return TOP_UNKNOWN;					\
  }

#define CHECK_OUT_OF_RANGE					\
  if ((a->min < MIN_ST_INT && a->max > MAX_ST_INT)		\
      || (b->min < MIN_ST_INT && b->max > MAX_ST_INT)) {	\
    result->min = MIN_ST_INT-1;					\
    result->max = MAX_ST_INT+1;					\
    return TOP_UNKNOWN;						\
  }

#define DO_RETURN						\
  return (result->min < MIN_ST_INT || result->max > MAX_ST_INT)	\
    ? TOP_UNKNOWN : TOP_IS_INTEGER				\

int
constraintCopy(a, result)
     Constraint *a, *result;
{
  result->min = a->min;
  result->max = a->max;
  result->source = a;

  DO_RETURN;
}

int
constraintSet(oop, result)
     OOP oop;
     Constraint *result;
{
  result->source = NULL;
  if (isInt(oop)) {
    result->min = MIN_ST_INT-1;
    result->max = MAX_ST_INT+1;
    return TOP_IS_NOT_INTEGER;
  } else {
    result->min = result->max = toInt(oop);
    return TOP_IS_INTEGER;
  }
}

int
constraintReset(a, n)
     Constraint *a;
     int	n;
{
  for (; n--; a++) {
    a->min = MIN_ST_INT-1;
    a->max = MAX_ST_INT+1;
    a->source = NULL;
  }

  return TOP_UNKNOWN;
}

int
constraintSum(a, b, result)
     Constraint *a, *b, *result;
{
  register long min, max;
  
  result->source = NULL;
  CHECK_OUT_OF_RANGE;

  min = a->min + b->min;
  result->min = min < MIN_ST_INT || a->min < MIN_ST_INT || b->min < MIN_ST_INT
    ? MIN_ST_INT-1 : min;

  max = a->max + b->max;
  result->max = max > MAX_ST_INT || a->max > MAX_ST_INT || b->max > MAX_ST_INT
    ? MAX_ST_INT+1 : min;

  DO_RETURN;
}

int
constraintSubtract(a, b, result)
     Constraint *a, *b, *result;
{
  register long min, max;

  result->source = NULL;
  CHECK_OUT_OF_RANGE;

  min = a->min - b->max;
  result->min = min < MIN_ST_INT || a->min < MIN_ST_INT || b->max > MAX_ST_INT
    ? MIN_ST_INT-1 : min;

  max = a->max - b->min;
  result->max = max > MAX_ST_INT || a->max > MAX_ST_INT || b->min < MIN_ST_INT
    ? MAX_ST_INT+1 : min;

  DO_RETURN;
}

int
constraintMultiply(a, b, result)
     Constraint *a, *b, *result;
{
  register long r1, r2, r3, r4;
  register long m12, m34, M12, M34;

  result->source = NULL;
  CHECK_OUT_OF_RANGE_STRICT;

  r1 = mulWithCheck(a->min, b->min);
  r2 = mulWithCheck(a->min, b->max);
  m12 = (r1 < r2) ? r1 : r2; M12 = r1 ^ r2 ^ m12;

  r3 = mulWithCheck(a->max, b->min);
  r4 = mulWithCheck(a->max, b->max);
  m34 = (r3 < r4) ? r3 : r4; M34 = r3 ^ r4 ^ m34;

  result->min = (m12 < m34) ? m12 : m34;
  result->max = (M12 > M34) ? M12 : M34;
  DO_RETURN;
}

int
constraintRemainder(a, b, result)
     register Constraint *a, *b, *result;
{
  result->source = NULL;

  /* It does not matter if `a' is a LargeInteger as long as
   * b is a SmallInteger. */
  if ((a->min < MIN_ST_INT && a->max > MAX_ST_INT)
      || (b->min < MIN_ST_INT || b->max > MAX_ST_INT)) {
    result->min = MIN_ST_INT-1;
    result->max = MAX_ST_INT+1;
    return TOP_UNKNOWN;
  }

  result->min = (b->min < 0) ? (b->min + 1) : 0;
  result->max = (b->max > 0) ? (b->max - 1) : 0;
  DO_RETURN;
}

int
constraintDivision(a, b, result)
     Constraint *a, *b, *result;
{
  register long r1, r2, r3, r4;
  register long m12, m34, M12, M34;

  result->source = NULL;
  CHECK_OUT_OF_RANGE_STRICT;

/* Subtract 1 from the result if the result is negative (to
 * round towards -oo rather than 0).
 */
#define DIVISION(a, b)	((a) / (b) - ( ((a) ^ (b)) < 0) )

  r1 = DIVISION(a->min, b->min);
  r2 = DIVISION(a->max, b->max);
  m12 = (r1 < r2) ? r1 : r2; M12 = r1 ^ r2 ^ m12;

  r3 = DIVISION(a->min, b->max);
  r4 = DIVISION(a->max, b->min);
  m34 = (r3 < r4) ? r3 : r4; M34 = r3 ^ r4 ^ m34;

  result->min = (m12 < m34) ? m12 : m34;
  result->max = (M12 > M34) ? M12 : M34;
  DO_RETURN;
}

long
mask(x)
     register long x;
{
  /* If x+1 is a power of two, we have
   *     x    00000000....00111111...11
   *     x+1  00000000....01000000...00
   * x & x+1  00000000....00000000...00
   */
  while (x & (x+1))
    x |= x >> 1;

  return x;
}

/* The functions for bit-wise & and | only give results rounded
 * to the next power of two (or to the next power of two - 1).
 * This is fine because these functions usually extract bit
 * contents, so they're operands are likely to be constrained
 * by powers of two, either.
 */
int
constraintBitAnd(a, b, result)
     Constraint *a, *b, *result;
{
  /* A good compiler will analyze life-times and overlap
   * r2 to a1, r12 to r1, r3 to b1, r4, b2, r34 to r3,
   * getting by with five registers (see below) */
  register long r1, r2, r3, r4, r12, r34;
  register long a1, a2, b1, b2;

  result->source = NULL;
  CHECK_OUT_OF_RANGE_STRICT;

  /* The minimum is obtained when the result is negative
   * (that's why we use 0 if a value is positive) and
   * there are as many zeroes as possible on the right
   * (that's why we use ~ on the mask).
   */
  a1 = a->min; a2 = a->max; b1 = b->min; b2 = b->max;

  a1 = a1 < 0 ? ~mask(~a1) : 0;
  a2 = a2 < 0 ? ~mask(~a2) : 0;
  b1 = b1 < 0 ? ~mask(~b1) : 0;
  b2 = b2 < 0 ? ~mask(~b2) : 0;

  r1 = a1 & b1; r2 = a1 & b2; r12 = (r1 < r2) ? r1 : r2;
  r3 = a2 & b1; r4 = a2 & b2; r34 = (r3 < r4) ? r3 : r4;

  /* The maximum is obtained when the result is positive
   * and as little zero bits as possible are cleared
   * (if a value must be negative, anding -1 will give the
   * biggest result, because we don't risk zeroing any bit).
   */
  a1 = a->min; a2 = a->max; b1 = b->min; b2 = b->max;

  a1 = a1 > 0 ? mask(a1) : -1;
  a2 = a2 > 0 ? mask(a2) : -1;
  b1 = b1 > 0 ? mask(b1) : -1;
  b2 = b2 > 0 ? mask(b2) : -1;

  result->min = (r12 < r34) ? r12 : r34;

  r1 = a1 & b1; r2 = a1 & b2; r12 = (r1 > r2) ? r1 : r2;
  r3 = a2 & b1; r4 = a2 & b2; r34 = (r3 > r4) ? r3 : r4;
  result->max = (r12 > r34) ? r12 : r34;

  DO_RETURN;
}

int
constraintBitOr(a, b, result)
     Constraint *a, *b, *result;
{
  register long min, max;

  result->source = NULL;
  CHECK_OUT_OF_RANGE_STRICT;

  min = a->min < b->min ? a->min : b->min;
  max = a->max > b->max ? a->max : b->max;

  result->min = min < 0 ? ~mask(~min) : 0;
  result->max = max > 0 ?  mask( max) : 0;

  DO_RETURN;
}

int
constraintBitShift(a, b, result)
     register Constraint *a, *b, *result;
{
  register long r1, r2, r3, r4;
  register long m12, m34, M12, M34;

#define MIN_LONG	(-1L ^ (~0UL >> 1))

  result->min = MIN_ST_INT-1;
  result->max = MAX_ST_INT+1;
  result->source = NULL;
  if ((a->min < MIN_ST_INT || a->max > MAX_ST_INT)
      || (b->min <= -SIZEOF_LONG || b->max >= SIZEOF_LONG)) {
    return TOP_UNKNOWN;
  }
  if (b->max > 0 && (
	(a->min << b->max >> b->max != a->min)
	|| (a->max << b->max >> b->max != a->max))) {
    return TOP_UNKNOWN;
  }

  r1 = r3 = a->min; r2 = r4 = a->max;

  if (b->max > 0) {
    r1 <<= b->max; r2 <<= b->max;
  } else {
    r1 >>= -b->max; r2 >>= -b->max;
  }
  m12 = (r1 < r2) ? r1 : r2; M12 = r1 ^ r2 ^ m12;

  if (b->min > 0) {
    r3 <<= b->min; r4 <<= b->min;
  } else {
    r3 >>= -b->min; r4 >>= -b->min;
  }
  m34 = (r3 < r4) ? r3 : r4; M34 = r1 ^ r2 ^ m34;

  result->min = (m12 < m34) ? m12 : m34;
  result->max = (M12 > M34) ? M12 : M34;
  DO_RETURN;
}


void
constraintLess(a, b)
     register Constraint *a, *b;
{
  register Constraint *copy;
  EXIT_IF_OUT_OF_RANGE;

  /* a 239 .. 500
   * b 1   .. 250		     a < b -->    239 .. 249    240 .. 250 */
  
  a->max = min (a->max, b->max - 1);	/* impose a->max < b->max */
  b->min = max (b->min, a->min + 1);	/* impose b->min > a->min */

  /* Propagate... */
  for (; copy = a->source; copy->max = a->max, a = copy);
  for (; copy = b->source; copy->min = b->min, b = copy);
}

void
constraintLessEqual(a, b)
     register Constraint *a, *b;
{
  register Constraint *copy;
  EXIT_IF_OUT_OF_RANGE;

  /* a 239 .. 500
   * b 1   .. 250		     a <= b -->    239 .. 250    239 .. 250 */
  
  a->max = min (a->max, b->max);	/* impose a->max <= b->max */
  b->min = max (b->min, a->min);	/* impose b->min >= a->min */

  /* Propagate... */
  for (; copy = a->source; copy->max = a->max, a = copy);
  for (; copy = b->source; copy->min = b->min, b = copy);
}

void
constraintGreater(a, b)
     register Constraint *a, *b;
{
  register Constraint *copy;
  EXIT_IF_OUT_OF_RANGE;

  /* Same as constraintLess, but with reversed a's and b's. */
  b->max = min (b->max, a->max - 1);
  a->min = max (a->min, b->min + 1);

  /* Propagate... */
  for (; copy = b->source; copy->max = b->max, b = copy);
  for (; copy = a->source; copy->min = a->min, a = copy);
}

void
constraintGreaterEqual(a, b)
     register Constraint *a, *b;
{
  register Constraint *copy;
  EXIT_IF_OUT_OF_RANGE;

  /* Same as constraintLessEqual, but with reversed a's and b's. */
  b->max = min (b->max, a->max);
  a->min = max (a->min, b->min);

  /* Propagate... */
  for (; copy = b->source; copy->max = b->max, b = copy);
  for (; copy = a->source; copy->min = a->min, a = copy);
}

void
constraintEqual(a, b)
     register Constraint *a, *b;
{
  register Constraint *copy;
  EXIT_IF_OUT_OF_RANGE;

  /* Same as constraintLessEqual+constraintGreaterEqual */
  a->max = b->max = min (b->max, a->max);
  b->min = a->min = max (a->min, b->min);

  /* Propagate... */
  for (; copy = b->source; copy->max = b->max, copy->min = b->min, b = copy);
  for (; copy = a->source; copy->max = a->max, copy->min = a->min, a = copy);
}

void
constraintCopyBlock(a, b, condition, n)
     register Constraint *a, *b;
     int		 condition;
     register int	 n;
{
  register long		delta;

  for (delta = (char *)b - (char *)a; n--; a++, b++) {
    b->min = a->min;
    b->max = a->max;
    b->source = (Constraint *) (a->source ? ((char *)a->source) + delta : NULL);
  }

  switch(condition) {
    case lessThanSpecial:
      constraintLess(b - 2, b - 1);
      break;

    case greaterThanSpecial:
      constraintGreater(b - 2, b - 1);
      break;

    case lessEqualSpecial:
      constraintLessEqual(b - 2, b);
      break;

    case greaterEqualSpecial:
      constraintGreaterEqual(b - 2, b);
      break;

    case equalSpecial:
      constraintEqual(b - 2, b);
      break;

    case notEqualSpecial:
      break;
  }
}

void
constraintMergeBlock(a, b, condition, n)
     register Constraint *a, *b;
     int		 condition;
     register int	 n;
{
  register long	 delta;
  Constraint	 *copy;

  if (condition) {
    copy = (Constraint *) alloca(n * sizeof(Constraint));
    constraintCopyBlock(a, copy, condition, n);
    a = copy;
  }

  for (delta = (char *)b - (char *)a; n--; a++, b++) {
    b->min = min(a->min, b->min);
    b->max = max(a->max, b->max);
    
    if (a->source) {
      register Constraint *source = (Constraint *) ( ((char *)a->source) + delta);
      if (source != b->source) {
        b->source = NULL;
      }
    }
  }
}


void
addEdge(from, to)
     BasicBlock	*from, *to;
{
  GraphEdge	*e = obstack_alloc(dataObstack, sizeof(GraphEdge));
  
  e->block = to;
  if (from->bp < to->bp) {
    e->next = to->forwardEdges;
    to->forwardEdges = e;
  } else {
    e->next = to->backwardEdges;
    to->backwardEdges = e;
  }
}

void
visitBackwardEdges(basicBlock, func, extra)
     BasicBlock	*basicBlock, *extra;
     void	(*func)();
{
  GraphEdge	*e;
  
  for (e = basicBlock->backwardEdges; e; e = e->next) {
    func(basicBlock, extra);
  }
}

void
visitForwardEdges(basicBlock, func, extra)
     BasicBlock	*basicBlock, *extra;
     void	(*func)();
{
  GraphEdge	*e;
  
  for (e = basicBlock->forwardEdges; e; e = e->next) {
    func(basicBlock, extra);
  }
}

