/******************************** -*- C -*- ****************************
 *
 *	Byte code interpreter primitives include file
 *
 *	$Revision: 1.95.1$
 *	$Date: 2000/12/27 10:45:49$
 *	$Author: pb$
 *
 ***********************************************************************/

/***********************************************************************
 *
 * Copyright 1988-92, 1994-95, 1999, 2000 Free Software Foundation, Inc.
 * Written by Steve Byrne.
 *
 * This file is part of GNU Smalltalk.
 *
 * GNU Smalltalk is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2, or (at your option) any later 
 * version.
 * 
 * GNU Smalltalk is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * GNU Smalltalk; see the file COPYING.  If not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
 *
 ***********************************************************************/

#ifdef PROFBLOCK
#define COUNT_PRIM	primitives[primitive]++
#else
#define COUNT_PRIM
#endif

#ifdef OPTIMIZE
#define receiverIsInt(x)		(true)
#define receiverIsClass(x, y)		(true)
#define receiverIsAKindOf(x, y)		(true)
#define PRIM_CHECKS_RECEIVER		PRIM_SUCCEED
#else
#define receiverIsInt(x)		isInt((x))
#define receiverIsClass(x, y)		isClass((x), (y))
#define receiverIsAKindOf(x, y)		isAKindOf((x), (y))
#define PRIM_CHECKS_RECEIVER		(PRIM_SUCCEED | PRIM_FAIL)
#endif

#ifdef USE_JIT_TRANSLATION
#define PRIM_FAILED			return ((long) -1)
#define PRIM_SUCCEEDED			return ((long) 0)
#define PRIM_SUCCEEDED_RELOAD_IP	return ((long) nativeIP)
#else
#define PRIM_FAILED			return (true)
#define PRIM_SUCCEEDED			return (false)
#define PRIM_SUCCEEDED_RELOAD_IP	return (false)
#endif

#define DEF_PRIM(name)                                          \
static long							\
executePrimitive_##name(primitive, numArgs, methodOOP)		\
     int	  primitive;					\
     volatile int numArgs;					\
     OOP	  methodOOP;					\
{								\
  COUNT_PRIM;							\
  primitivesExecuted++;						\
  lastPrimitive = primitive;

#define PRIM_ATTR(name, attributes)                             \
}								\
 								\
static long							\
attrPrimitive_##name(primitive)					\
     int	  primitive;					\
{								\
  return (attributes);						\
}

#define intBinOp(op, noOverflow)                \
    register OOP	oop1;			\
    register OOP	oop2;			\
    oop2 = popOOP();				\
    oop1 = popOOP();				\
    if (receiverIsInt(oop1) && isInt(oop2)) {	\
      register long iarg1, iarg2;		\
      iarg1 = toInt(oop1);			\
      iarg2 = toInt(oop2);			\
						\
      op;                                       \
      if(noOverflow || !INT_OVERFLOW(iarg1)) {	\
        pushInt(iarg1);			        \
        PRIM_SUCCEEDED;				\
      }                                         \
    }						\
    unPop(2);					\
    PRIM_FAILED

#define boolBinOp(operator)					\
    register OOP	oop1;					\
    register OOP	oop2;					\
    oop2 = popOOP();						\
    oop1 = popOOP();						\
    if (receiverIsInt(oop1) && isInt(oop2)) {			\
      pushBoolean( ((long)oop1) operator ((long)oop2) );	\
      PRIM_SUCCEEDED;						\
    }								\
    unPop(2);							\
    PRIM_FAILED


/* SmallInteger + arg  */
DEF_PRIM(1)
{
  intBinOp(iarg1 += iarg2, false);
}
PRIM_ATTR(1, PRIM_SUCCEED | PRIM_FAIL)

/* SmallInteger - arg */
DEF_PRIM(2)
{
  intBinOp(iarg1 -= iarg2, false);
}
PRIM_ATTR(2, PRIM_SUCCEED | PRIM_FAIL)

/* SmallInteger < arg */
DEF_PRIM(3)
{
  boolBinOp(<);
}
PRIM_ATTR(3, PRIM_SUCCEED | PRIM_FAIL)

/* SmallInteger > arg */
DEF_PRIM(4)
{
  boolBinOp(>);
}
PRIM_ATTR(4, PRIM_SUCCEED | PRIM_FAIL)

/* SmallInteger <= arg */
DEF_PRIM(5)
{
  boolBinOp(<=);
}
PRIM_ATTR(5, PRIM_SUCCEED | PRIM_FAIL)

/* SmallInteger >= arg */
DEF_PRIM(6)
{
  boolBinOp(>=);
}
PRIM_ATTR(6, PRIM_SUCCEED | PRIM_FAIL)

/* SmallInteger =, == arg */
DEF_PRIM(7)
{
  boolBinOp(==);
}
PRIM_ATTR(7, PRIM_SUCCEED | PRIM_FAIL)

/* SmallInteger ~=, ~~ arg */
DEF_PRIM(8)
{
  boolBinOp(!=);
}
PRIM_ATTR(8, PRIM_SUCCEED | PRIM_FAIL)

/* SmallInteger * arg */
DEF_PRIM(9)
{
  intBinOp(iarg1 = mulWithCheck(iarg1, iarg2), false);
}
PRIM_ATTR(9, PRIM_SUCCEED | PRIM_FAIL)

/* SmallInteger / arg */
/* quotient as exact as possible */
DEF_PRIM(10)
{
    register OOP	oop1;
    register OOP	oop2;

    oop2 = popOOP();
    oop1 = popOOP();
    if (receiverIsInt(oop1) && isInt(oop2) && oop2 != fromInt(0)) {
      register long iarg1, iarg2, result;
      iarg1 = toInt(oop1);
      iarg2 = toInt(oop2);
      result = iarg1 / iarg2;
      iarg2 *= result;
      if (iarg1 == iarg2) {
	pushInt(result);
	PRIM_SUCCEEDED;
      }
    }
    unPop(2);
    PRIM_FAILED;
}
PRIM_ATTR(10, PRIM_SUCCEED | PRIM_FAIL)

/* SmallInteger \\ arg */
/* remainder truncated towards -infinity */
DEF_PRIM(11)
{
    register OOP	oop1;
    register OOP	oop2;

    oop2 = popOOP();
    oop1 = popOOP();
    if (receiverIsInt(oop1) && isInt(oop2) && oop2 != fromInt(0)) {
      register long iarg1, iarg2, result;
      iarg1 = toInt(oop1);
      iarg2 = toInt(oop2);
      result = iarg1 % iarg2;
      pushInt(result && ((result ^ iarg2) < 0) ? result + iarg2 : result);
      PRIM_SUCCEEDED;
    }
    unPop(2);
    PRIM_FAILED;
}
PRIM_ATTR(11, PRIM_SUCCEED | PRIM_FAIL)			

/* SmallInteger // arg */
/* quotient truncated towards -infinity */
DEF_PRIM(12)
{
    register OOP	oop1;
    register OOP	oop2;

    oop2 = popOOP();
    oop1 = popOOP();
    if (receiverIsInt(oop1) && isInt(oop2) && oop2 != fromInt(0)) {
      register long iarg1, iarg2;
      iarg1 = toInt(oop1);
      iarg2 = toInt(oop2);
      if (iarg2 < 0) {		/* make the divisor positive */
        iarg1 = -iarg1;
        iarg2 = -iarg2;
      }
      if (iarg1 < 0) {		/* differing signs => use black magic */
        pushInt(- ((iarg2 - 1 - iarg1) / iarg2));
      } else {
        pushInt(iarg1 / iarg2);
      }
      PRIM_SUCCEEDED;
    }
    unPop(2);
    PRIM_FAILED;
}
PRIM_ATTR(12, PRIM_SUCCEED | PRIM_FAIL)

/* SmallInteger quo: arg */
/* quotient truncated towards 0 */
DEF_PRIM(13)
{
    register OOP	oop1;
    register OOP	oop2;

    oop2 = popOOP();
    oop1 = popOOP();
    if (receiverIsInt(oop1) && isInt(oop2)) {
      register long iarg1, iarg2;
      iarg1 = toInt(oop1);
      iarg2 = toInt(oop2);
      if (iarg2 != 0) {
	pushInt(iarg1 / iarg2);
	PRIM_SUCCEEDED;
      }
    }
    unPop(2);
    PRIM_FAILED;
}
PRIM_ATTR(13, PRIM_SUCCEED | PRIM_FAIL)

/* SmallInteger bitAnd: arg */
DEF_PRIM(14)
{
  intBinOp(iarg1 &= iarg2, true);
}
PRIM_ATTR(14, PRIM_SUCCEED | PRIM_FAIL)

/* SmallInteger bitOr: arg */	
DEF_PRIM(15)
{
  intBinOp(iarg1 |= iarg2, true);
}
PRIM_ATTR(15, PRIM_SUCCEED | PRIM_FAIL)

/* SmallInteger bitXor: arg */	
DEF_PRIM(16)
{
  intBinOp(iarg1 ^= iarg2, true);
}
PRIM_ATTR(16, PRIM_SUCCEED | PRIM_FAIL)

/* SmallInteger bitShift: arg */
DEF_PRIM(17)
{
    register OOP	oop1;
    register OOP	oop2;

    oop2 = popOOP();
    oop1 = popOOP();
    if (receiverIsInt(oop1) && isInt(oop2)) {
      register long iarg1;
      register long iarg2;
      iarg1 = toInt(oop1);
      iarg2 = toInt(oop2);
      if (iarg2 < 0) {
	pushInt(iarg1 >> -iarg2);
	PRIM_SUCCEEDED;
      }
      if (iarg2 < ST_INT_SIZE) {
	register long result = iarg1 << iarg2;
	if ((result >> iarg2) == iarg1) {
	  pushInt(result);
	  PRIM_SUCCEEDED;
	}
      }
    }
    unPop(2);
    PRIM_FAILED;
}
PRIM_ATTR(17, PRIM_SUCCEED | PRIM_FAIL)


/* SmallInteger asFloat */
DEF_PRIM(40)
{
    register OOP oop1;

    oop1 = popOOP();
    if (receiverIsInt(oop1)) {
      pushOOP(floatNew((double)toInt(oop1)));
      PRIM_SUCCEEDED;
    }
    unPop(1);
    PRIM_FAILED;
}
PRIM_ATTR(40, PRIM_CHECKS_RECEIVER)

DEF_PRIM(41_50)
{
    register double farg2;
    OOP	oop1;
    OOP	oop2;

    oop2 = popOOP();
    oop1 = popOOP();
    if (isClass(oop2, floatClass)) {
      farg2 = floatOOPValue(oop2);
    } else if (isInt(oop2)) {
      farg2 = (float) toInt(oop2);
    } else {
      unPop(2);
      PRIM_FAILED;
    }

    if (receiverIsClass(oop1, floatClass)) {
      register double farg1;
      farg1 = floatOOPValue(oop1);
      switch (primitive) {
        case 41: pushOOP(floatNew(farg1 + farg2));	break;
        case 42: pushOOP(floatNew(farg1 - farg2));	break;
        case 43: pushBoolean(farg1 < farg2); 		break;
        case 44: pushBoolean(farg1 > farg2);		break;
        case 45: pushBoolean(farg1 <= farg2);		break;
        case 46: pushBoolean(farg1 >= farg2);		break;
        case 47: pushBoolean(farg1 == farg2);		break;
        case 48: pushBoolean(farg1 != farg2);		break;
        case 49: pushOOP(floatNew(farg1 * farg2));	break;
        case 50:
	  if (farg2 != 0.0) {
	    pushOOP(floatNew(farg1 / farg2));
	  } else {
            unPop(2);
            PRIM_FAILED;
          }
      }
      PRIM_SUCCEEDED;
    }

    unPop(2);
    PRIM_FAILED;
}
PRIM_ATTR(41_50, PRIM_SUCCEED | PRIM_FAIL)

/* Float truncated */
DEF_PRIM(51)
{
    register OOP oop1;

    oop1 = popOOP();
    if (receiverIsClass(oop1, floatClass)) {
      double oopValue = floatOOPValue(oop1);
      if (oopValue >= 0.0 && oopValue <= MAX_ST_INT) {
	pushInt((long)(oopValue + 0.000000000000005));
	PRIM_SUCCEEDED;
      } else if (oopValue < 0.0 && oopValue >= MIN_ST_INT) {
	pushInt((long)(oopValue - 0.000000000000005));
	PRIM_SUCCEEDED;
      }
    }
    unPop(1);
    PRIM_FAILED;
}
PRIM_ATTR(51, PRIM_SUCCEED | PRIM_FAIL)

/* Float fractionPart */
DEF_PRIM(52)
{
    register OOP oop1;

    oop1 = popOOP();
    if (receiverIsClass(oop1, floatClass)) {
      register double farg1;
      double fdummy;
      farg1 = floatOOPValue(oop1);
      if (farg1 < 0.0) {
        farg1 = -modf(-farg1, &fdummy);
        if (farg1 < -0.999999999999995) farg1 = 0.0;
      } else {
        farg1 = modf(farg1, &fdummy);
        if (farg1 > 0.999999999999995) farg1 = 0.0;
      }

      pushOOP(floatNew(farg1));
      PRIM_SUCCEEDED;
    } 
    unPop(1);
    PRIM_FAILED;
}
PRIM_ATTR(52, PRIM_CHECKS_RECEIVER)

/* Float exponent */
DEF_PRIM(53)
{
    register OOP oop1;

    oop1 = popOOP();
    if (receiverIsClass(oop1, floatClass)) {
      register double farg1;
      int intArg1;

      farg1 = floatOOPValue(oop1);
      if (farg1 == 0.0) {
	intArg1 = 1;
      } else {
	frexp(floatOOPValue(oop1), &intArg1);
	intArg1--;
      }
      pushInt(intArg1);
      PRIM_SUCCEEDED;
    }
    unPop(1);
    PRIM_FAILED;
}
PRIM_ATTR(53, PRIM_CHECKS_RECEIVER | PRIM_RETURN_SMALL_SMALLINTEGER)

/* Float timesTwoPower: */
DEF_PRIM(54)
{
    OOP oop1;
    OOP oop2;

    oop2 = popOOP();
    oop1 = popOOP();
    if (receiverIsClass(oop1, floatClass) && isInt(oop2)) {
      register double farg1;
      register long iarg2;
      farg1 = floatOOPValue(oop1);
      iarg2 = toInt(oop2);
      pushOOP(floatNew(ldexp(farg1, iarg2)));
      PRIM_SUCCEEDED;
    }
    unPop(2);
    PRIM_FAILED;
}
PRIM_ATTR(54, PRIM_SUCCEED | PRIM_FAIL)



/* Object at:, Object basicAt: */
DEF_PRIM(60)
{
    OOP oop1;
    OOP oop2;

    oop2 = popOOP();
    oop1 = stackTop();
    if (isInt(oop2)) {
      register long arg2;
      arg2 = toInt(oop2);
      oop1 = indexOOP(oop1, arg2);
      if (oop1) {
        setStackTop(oop1);
        PRIM_SUCCEEDED;
      }
    }
    unPop(1);
    PRIM_FAILED;
}
PRIM_ATTR(60, PRIM_SUCCEED | PRIM_FAIL)

/* Object at:put:, Object basicAt:put: */
DEF_PRIM(61)
{
    OOP oop1;
    OOP oop2;
    OOP oop3;

    oop3 = popOOP();
    oop2 = popOOP();
    oop1 = stackTop();
    if (isInt(oop2) && !oopIsReadOnly(oop1)) {
      register long arg2;
      arg2 = toInt(oop2);
      if (indexOOPPut(oop1, arg2, oop3)) {
	setStackTop(oop3);
	PRIM_SUCCEEDED;
      }
    }

    unPop(2);
    PRIM_FAILED;
}
PRIM_ATTR(61, PRIM_SUCCEED | PRIM_FAIL)

/* Object basicSize; Object size; String size */
DEF_PRIM(62)
{
    OOP oop1;

    oop1 = popOOP();
    pushInt(numIndexableFields(oop1));
    PRIM_SUCCEEDED;
}
PRIM_ATTR(62, PRIM_SUCCEED | PRIM_RETURN_SMALL_SMALLINTEGER)

/* String at:; String basicAt: */
DEF_PRIM(63)
{
    OOP oop1;
    OOP oop2;

    oop2 = popOOP();
    oop1 = stackTop();
    if (isInt(oop2)) {
      register long arg2;
      arg2 = toInt(oop2);
      oop1 = indexStringOOP(oop1, arg2);
      if (oop1) {
        setStackTop(oop1);
        PRIM_SUCCEEDED;
      }
    }

    unPop(1);
    PRIM_FAILED;
}
PRIM_ATTR(63, PRIM_SUCCEED | PRIM_FAIL)

/* String basicAt:put:; String at:put: */
DEF_PRIM(64)
{
    OOP oop1;
    OOP oop2;
    OOP oop3;

    oop3 = popOOP();
    oop2 = popOOP();
    oop1 = stackTop();
    if (isInt(oop2) && isClass(oop3, charClass) && !oopIsReadOnly(oop1)) {
      register long arg2;
      arg2 = toInt(oop2);
      if (indexStringOOPPut(oop1, arg2, oop3)) {
	setStackTop(oop3);
	PRIM_SUCCEEDED;
      }
    }

    unPop(2);
    PRIM_FAILED;
}
PRIM_ATTR(64, PRIM_SUCCEED | PRIM_FAIL)


/* CompiledBlock numArgs:numTemps:bytecodes:depth:literals: */
DEF_PRIM(68)
{
    OOP *literals = oopToObj(popOOP())->data;
    int depth = toInt(popOOP());
    OOP bytecodesOOP = popOOP();
    int numTemps = toInt(popOOP());
    int numArgs = toInt(popOOP());
    ByteCodes bytecodes = extractByteCodes(bytecodesOOP);

    OOP block = blockNew(numArgs, numTemps, bytecodes, depth, literals);
    setStackTop(block);
    PRIM_SUCCEEDED;
}
PRIM_ATTR(68, PRIM_SUCCEED)

/* CompiledMethod literals:numArgs:numTemps:primitive:bytecodes:depth: */
DEF_PRIM(69)
{
    int depth = toInt(popOOP());
    OOP bytecodesOOP = popOOP();
    int primitive = toInt(popOOP());
    int numTemps = toInt(popOOP());
    int numArgs = toInt(popOOP());
    OOP literals = popOOP();
    ByteCodes bytecodes = extractByteCodes(bytecodesOOP);

    OOP method = makeNewMethod(primitive, numArgs, numTemps, depth, literals,
				 bytecodes, nilOOP, nilOOP);
    setStackTop(method);
    PRIM_SUCCEEDED;
}
PRIM_ATTR(69, PRIM_SUCCEED)

/* Behavior basicNew; Behavior new; */
DEF_PRIM(70)
{
    OOP oop1;

    oop1 = stackTop();
    if (isOOP(oop1)) {
      if (!classIsIndexable(oop1)) {
	oop1 = allocOOP(instantiate(oop1));
	setStackTop(oop1);
	PRIM_SUCCEEDED;
      }
    }
    PRIM_FAILED;
}
PRIM_ATTR(70, PRIM_SUCCEED | PRIM_FAIL)

/* Behavior new:; Behavior basicNew: */
DEF_PRIM(71)
{
    OOP oop1;
    OOP oop2;

    oop2 = popOOP();
    oop1 = stackTop();
    if (isOOP(oop1) && isInt(oop2)) {
      if (classIsIndexable(oop1)) {
	register long arg2;
	arg2 = toInt(oop2);
	if (arg2 >= 0) {
	  oop1 = instantiateOOPWith(oop1, arg2);
	  setStackTop(oop1);
	  PRIM_SUCCEEDED;
	}
      }
    }

    unPop(1);
    PRIM_FAILED;
}
PRIM_ATTR(71, PRIM_SUCCEED | PRIM_FAIL)

/* Object become: */
DEF_PRIM(72)
{
    OOP oop1;
    OOP oop2;

    oop2 = popOOP();
    oop1 = stackTop();
    if (isOOP(oop1) && !oopIsReadOnly(oop1)) {
      if (isOOP(oop2) && !oopIsReadOnly(oop2)) {
        swapObjects(oop1, oop2);
        /* ??? maybe we should do `self = oop2' here? */
        PRIM_SUCCEEDED;
      }
    }
    unPop(1);
    PRIM_FAILED;
}
PRIM_ATTR(72, PRIM_SUCCEED | PRIM_FAIL)

/* Object instVarAt: */
DEF_PRIM(73)
{
    OOP oop1;
    OOP oop2;

    oop2 = popOOP();
    oop1 = stackTop();
    if (isInt(oop2)) {
      register long arg2;
      arg2 = toInt(oop2);
      if (checkBoundsOf(oop1, arg2)) {
	setStackTop(instVarAt(oop1, arg2));
	PRIM_SUCCEEDED;
      }
    }
    unPop(1);
    PRIM_FAILED;
}
PRIM_ATTR(73, PRIM_SUCCEED | PRIM_FAIL)

/* Object instVarAt:put: */
DEF_PRIM(74)
{
    OOP oop1;
    OOP oop2;
    OOP oop3;

    oop3 = popOOP();
    oop2 = popOOP();
    oop1 = stackTop();
    if (isInt(oop2)) {
      register long arg2;
      arg2 = toInt(oop2);
      if (checkBoundsOf(oop1, arg2)) {
	if (instVarAtPut(oop1, arg2, oop3)) {
	  PRIM_SUCCEEDED;
	}
      }
    }
    unPop(2);
    PRIM_FAILED;
}
PRIM_ATTR(74, PRIM_SUCCEED | PRIM_FAIL)

/* Object asOop; Object hash; Symbol hash */
DEF_PRIM(75)
{
    OOP oop1;

    oop1 = popOOP();
    if (isOOP(oop1)) {
      pushInt(oopIndex(oop1));
      PRIM_SUCCEEDED;
    }
    unPop(1);
    PRIM_FAILED;
}
PRIM_ATTR(75, PRIM_SUCCEED | PRIM_FAIL)

/* SmallInteger asObject; SmallInteger asObjectNoFail */
DEF_PRIM(76)
{
    OOP oop1;
    long arg1;
    oop1 = stackTop();
    arg1 = toInt(oop1);
    if (oopIndexValid(arg1)) {
      setStackTop(oopAt(arg1));
      PRIM_SUCCEEDED;
    }

    PRIM_FAILED;
}
PRIM_ATTR(76, PRIM_SUCCEED | PRIM_FAIL)

/* Behavior someInstance */
DEF_PRIM(77)
{
    OOP oop1;
    OOP oop2;

    oop1 = stackTop();
    for (oop2 = allOopsTable; oop2 < &oopTable[oopTableSize]; oop2++) {
      if (oopValid(oop2) && oop1 == oopClass(oop2)) {
	setStackTop(oop2);
	PRIM_SUCCEEDED;
      }
    }
    PRIM_FAILED;
}
PRIM_ATTR(77, PRIM_SUCCEED | PRIM_FAIL)

/* Object nextInstance */
DEF_PRIM(78)
{
    OOP oop1;

    oop1 = stackTop();
    if (oopClass(oop1) == charClass) {
      /* Characters are one after another - at the end there is nilOOP */
      oop1++;
      if (charClass == oopClass(oop1)) {
	setStackTop(oop1);
	PRIM_SUCCEEDED;
      }
    } else if (isOOP(oop1) && oop1 >= oopTable) {
      /* There is just one copy of all other builtin objects,
       * so fail for a builtin */
      register OOP classOOP = oopClass(oop1);
      for (++oop1; oop1 <= lastUsedOOP; oop1++) {
	if (oopValid(oop1) && classOOP == oopClass(oop1)) {
	  setStackTop(oop1);
	  PRIM_SUCCEEDED;
	}
      }
    }
    PRIM_FAILED;
}
PRIM_ATTR(78, PRIM_SUCCEED | PRIM_FAIL)



/* BlockClosure blockCopy: */
DEF_PRIM(80)
{
    register BlockClosure closure, original;
    OOP oop1;
    OOP oop2;

    oop2 = popOOP();
    oop1 = stackTop();
    closure = (BlockClosure)newInstance(blockClosureClass);
    original = (BlockClosure)oopToObj(oop1);
    closure->outerContext = oop2;
    closure->block = original->block;
    closure->receiver = self;
    setStackTop(allocOOP(closure));
    PRIM_SUCCEEDED;
}
PRIM_ATTR(80, PRIM_SUCCEED)

/* BlockClosure value
   BlockClosure value:
   BlockClosure value:value:
   BlockClosure value:value:value: */
DEF_PRIM(81)
{
    if(sendBlockValue(numArgs)) {
      PRIM_FAILED;
    } else {
      PRIM_SUCCEEDED_RELOAD_IP;
    }
}
PRIM_ATTR(81, PRIM_FAIL | PRIM_RELOAD_IP | PRIM_CACHE_NEW_IP)

/* BlockContext valueWithArguments: */
DEF_PRIM(82)
{
    OOP oop1;
    OOP oop2;

    oop2 = popOOP();
    oop1 = stackTop();
    if (isClass(oop2, arrayClass)) {
      int i;
      numArgs = numIndexableFields(oop2);
      for (i = 1; i <= numArgs; i++) {
	pushOOP(arrayAt(oop2, i));
      }
      if(sendBlockValue(numArgs)) {
        popNOOPs(numArgs);
        pushOOP(oop2);
        PRIM_FAILED;
      } else {
	PRIM_SUCCEEDED_RELOAD_IP;
      }
    }
    unPop(1);
    PRIM_FAILED;
}
PRIM_ATTR(82, PRIM_FAIL | PRIM_RELOAD_IP)

/* Object perform:
 * Object perform:with:
 * Object perform:with:with:
 * Object perform:with:with:with: */
DEF_PRIM(83)
{
    OOP oop1;
    OOP oopVec[4];
    int i;

    /* pop off the arguments (if any) */
    numArgs--;
    for (i = 0; i < numArgs; i++) {
      oopVec[i] = popOOP();
    }
    oop1 = popOOP();		/* the selector */
    if (isClass(oop1, symbolClass)
        && checkSendCorrectness(stackTop(), oop1, numArgs)) {

      /* push the args back onto the stack */
      while(--i >= 0) {
	pushOOP(oopVec[i]);
      }
      sendMessage(oop1, numArgs, false);
      PRIM_SUCCEEDED_RELOAD_IP;
    }

    if (isClass(oop1, compiledMethodClass)) {
      Method        method;
      MethodHeader  header;

      method = (Method) oopToObj(oop1);
      header = method->header;
      if (header.numArgs == numArgs) {
	/* push the args back onto the stack */
	while(--i >= 0) {
	  pushOOP(oopVec[i]);
	}
	sendMethod(oop1);
	PRIM_SUCCEEDED_RELOAD_IP;
      }
    }

    unPop(numArgs + 1);
    PRIM_FAILED;
}
PRIM_ATTR(83, PRIM_FAIL | PRIM_RELOAD_IP)		

/* Object perform:withArguments: */
DEF_PRIM(84)
{
    OOP oop1;
    OOP oop2;

    oop2 = popOOP();
    oop1 = popOOP();
    numArgs = numIndexableFields(oop2);
    if (isClass(oop2, arrayClass) && isClass(oop1, symbolClass)
        && checkSendCorrectness(stackTop(), oop1, numArgs)) {

      int i;
      for (i = 1; i <= numArgs; i++) {
	pushOOP(arrayAt(oop2, i));
      }
      sendMessage(oop1, numArgs, false);
      PRIM_SUCCEEDED_RELOAD_IP;
    }

    if (isClass(oop1, compiledMethodClass)) {
      Method        method;
      MethodHeader  header;

      method = (Method) oopToObj(oop1);
      header = method->header;
      if (header.numArgs == numArgs) {
	int i;
	for (i = 1; i <= numArgs; i++) {
	  pushOOP(arrayAt(oop2, i));
	}
	sendMethod(oop1);
	PRIM_SUCCEEDED_RELOAD_IP;
      }
    }

    unPop(2);
    PRIM_FAILED;
}
PRIM_ATTR(84, PRIM_FAIL | PRIM_RELOAD_IP)


/* Semaphore signal */
DEF_PRIM(85)
{
    OOP oop1;
    IntState	oldSigMask;

    oop1 = stackTop();
    oldSigMask = disableInterrupts(); /* block out everything! */
    syncSignal(oop1);
    enableInterrupts(oldSigMask);
    PRIM_SUCCEEDED;
}
PRIM_ATTR(85, PRIM_SUCCEED)

/* Semaphore wait */
DEF_PRIM(86)
{
    OOP oop1;
    Semaphore	sem;
    IntState	oldSigMask;

    oop1 = stackTop();
    oldSigMask = disableInterrupts(); /* block out everything! */
    sem = (Semaphore)oopToObj(oop1);
    if (toInt(sem->signals) <= 0) {
      /* have to suspend, move this to the end of the list */
      addLastLink(oop1, getActiveProcess());
      activeProcessYield();
    } else {
      sem->signals = decrInt(sem->signals);
    }
    enableInterrupts(oldSigMask);
    PRIM_SUCCEEDED;
}
PRIM_ATTR(86, PRIM_SUCCEED)		

/* Process resume */
DEF_PRIM(87)
{
    OOP oop1;

    oop1 = stackTop();
    if (resumeProcess(oop1)) {
      PRIM_SUCCEEDED;
    } else {
      PRIM_FAILED;
    }
}
PRIM_ATTR(87, PRIM_SUCCEED | PRIM_FAIL)

/* Process yield */
DEF_PRIM(88)
{
    OOP oop1;

    usleep(1000);			/* exhibit good behavior */
    oop1 = stackTop();
    if (oop1 == getActiveProcess()) {
      setStackTop(nilOOP);		/* this is our return value */
      if (isProcessReady(oop1)) {
        sleepProcess(oop1);		/* move to the end of the list */
      }
      activeProcessYield();
    }
    PRIM_SUCCEEDED;
}
PRIM_ATTR(88, PRIM_SUCCEED)


/* Behavior flushCache */
DEF_PRIM(89)
{
    invalidateMethodCache();
    PRIM_SUCCEEDED;
}
PRIM_ATTR(89, PRIM_SUCCEED)

/* CompiledCode discardTranslation */
DEF_PRIM(90)
{
#ifdef USE_JIT_TRANSLATION
    discardNativeCode(stackTop());
#endif
    PRIM_SUCCEEDED;
}
PRIM_ATTR(90, PRIM_SUCCEED)

/* Object changeClassTo: */
DEF_PRIM(91)
{
    OOP oop1;

    oop1 = popOOP();
    if (isAKindOf(oopClass(oop1), behaviorClass)) {
      mst_Object obj;
      obj = oopToObj(stackTop());
      obj->objClass = oop1;
      PRIM_SUCCEEDED;
    }
    unPop(1);			/* trying to do Bad Things */
    PRIM_FAILED;
}
PRIM_ATTR(91, PRIM_SUCCEED | PRIM_FAIL)

/* Time class timezoneBias */
DEF_PRIM(96)
{
    setStackTopInt(currentTimeZoneBias());
    PRIM_SUCCEEDED;
}
PRIM_ATTR(96, PRIM_SUCCEED)

/* Time class timezone */
DEF_PRIM(97)
{
    OOP oop1;
    char *result;

    result = currentTimeZoneName();
    oop1 = stringNew(result);
    setStackTop(oop1);
    xfree(result);
    PRIM_SUCCEEDED;
}
PRIM_ATTR(97, PRIM_SUCCEED)

/* Time class secondClock
 *  -- note: this primitive has different
 *     semantics from those defined in the
 *     book.  This primitive returns the
 *     seconds since/to Jan 1, 2000 00:00:00
 *     instead of Jan 1,1901.
 */
DEF_PRIM(98)
{
    /* 10957 = days between 1970 and 2000 */
    setStackTopInt(getTime() - 86400*10957);  
    PRIM_SUCCEEDED;
}
PRIM_ATTR(98, PRIM_SUCCEED)

/* Time class millisecondClock
 * -- Note: the semantics of this primitive
 *    are different than those described in
 *    the book.  This primitive returns the
 *    number of milliseconds since midnight
 *    today. */
DEF_PRIM(99)
{
    setStackTopInt(getMilliTime() % (24*60*60*1000));
    PRIM_SUCCEEDED;
}
PRIM_ATTR(99, PRIM_SUCCEED)

/* Processor signal: semaphore
 *           atMilliseconds: deltaMilliseconds
 */
DEF_PRIM(100)
{
    OOP oop1;
    OOP oop2;

    oop2 = popOOP();
    oop1 = popOOP();
    if (isInt(oop2)) {
      register long arg2;
      arg2 = toInt(oop2);
      if (arg2 <= 0) {
	IntState oldSigMask;
	oldSigMask = disableInterrupts(); /* block out everything! */
        timeoutSem = nilOOP;
	syncSignal(oop1);
	enableInterrupts(oldSigMask);
      } else {
        timeoutSem = oop1;
        signalAfter(arg2, timeoutHandler);
        PRIM_SUCCEEDED;
      }
    }

    unPop(2);
    PRIM_FAILED;
}
PRIM_ATTR(100, PRIM_SUCCEED | PRIM_FAIL)

/* Processor isTimeoutProgrammed */
DEF_PRIM(101)
{
    setStackTopBoolean(!isNil(timeoutSem));
    PRIM_SUCCEEDED;
}
PRIM_ATTR(101, PRIM_SUCCEED)



/* String hash */
DEF_PRIM(104)
{
    unsigned long hash;
    long	  spec;
    Byte	  *base;
    OOP		  oop1;

    oop1 = stackTop();
    spec = oopInstanceSpec(oop1);
#ifndef OPTIMIZE
    if (!(spec & ((~0 << ISP_NUMFIXEDFIELDS) | ISP_ISPOINTERS | ISP_ISWORDS))) {
#endif
      base = stringOOPChars(oop1);
      hash = hashString(base, numIndexableFields(oop1));
      setStackTopInt(hash);
      PRIM_SUCCEEDED;
#ifndef OPTIMIZE
    }
#endif
    PRIM_FAILED;
}
PRIM_ATTR(104, PRIM_CHECKS_RECEIVER)

/* ByteArray primReplaceFrom:to:with:startingAt
 * ByteArray replaceFrom:to:withString:startingAt:
 * String replaceFrom:to:withByteArray:startingAt:
 * String primReplaceFrom:to:with:startingAt:*/
DEF_PRIM(105)
{
    OOP		srcIndexOOP, srcOOP, dstEndIndexOOP, dstStartIndexOOP, dstOOP;
    int		dstEndIndex, dstStartIndex, srcIndex, dstLen, srcLen,
   		dstRangeLen;
    long	spec;
    Byte	*dstBase, *srcBase;

    srcIndexOOP = popOOP();
    srcOOP = popOOP();
    dstEndIndexOOP = popOOP();
    dstStartIndexOOP = popOOP();
    if (isInt(srcIndexOOP) && isInt(dstStartIndexOOP)
	&& isInt(dstEndIndexOOP) && !isInt(srcOOP)) {
      spec = oopInstanceSpec(srcOOP);
      if ( !(spec & (ISP_ISWORDS | ISP_ISPOINTERS)) ) {
	/* dstEnd is inclusive: (1 to: 1) has length 1 */
	dstEndIndex = toInt(dstEndIndexOOP);
	dstStartIndex = toInt(dstStartIndexOOP);
	srcIndex = toInt(srcIndexOOP);
	dstOOP = stackTop();
	dstLen = numIndexableFields(dstOOP);
	srcLen = numIndexableFields(srcOOP);
	dstRangeLen = dstEndIndex - dstStartIndex + 1;
	if ((dstRangeLen >= 0 && dstEndIndex <= dstLen
	       && dstStartIndex > 0)) {
	  if (dstRangeLen > 0) { /* don't do it unless somethings to copy */
	    if ((srcIndex <= srcLen) && (srcIndex > 0)
		&& (srcIndex + dstRangeLen - 1 <= srcLen)) {
	      /* do the copy */
	      dstBase = stringOOPChars(dstOOP);
	      srcBase = stringOOPChars(srcOOP);
	      memcpy(&dstBase[dstStartIndex-1], &srcBase[srcIndex-1],
		     dstRangeLen);
	    }
	  }
	  PRIM_SUCCEEDED;
	}
      }
    }
	      	
    unPop(4);
    PRIM_FAILED;
}
PRIM_ATTR(105, PRIM_SUCCEED | PRIM_FAIL)



/* Object ==, Character = */
DEF_PRIM(110)
{
    OOP oop1;
    OOP oop2;

    oop2 = popOOP();
    oop1 = popOOP();
    pushBoolean(oop1 == oop2);
    PRIM_SUCCEEDED;
}
PRIM_ATTR(110, PRIM_SUCCEED)		

/* Object class */
DEF_PRIM(111)
{
    OOP oop1;

    oop1 = popOOP();
    if (isInt(oop1)) {
      pushOOP(smallIntegerClass);
    } else {
      pushOOP(oopClass(oop1));
    }
    PRIM_SUCCEEDED;
}
PRIM_ATTR(111, PRIM_SUCCEED)		


    /* 112 - 116 currently not in use */

/* quitPrimitive: status
 * Always fail because if it succeeds we don't return */
DEF_PRIM(117)
{
    OOP oop1;
    long arg1;

    oop1 = stackTop();
    if (isInt(oop1)) {
      arg1 = toInt(oop1);
      exit(arg1);
    }
    PRIM_FAILED;
}
PRIM_ATTR(117, PRIM_FAIL)

    /* 118 - 127 currently not in use */


/* ------- GNU Smalltalk specific primitives begin here -------------------- */

/* Dictionary at: */
DEF_PRIM(128)
{
    OOP oop1;
    OOP oop2;

    oop2 = popOOP();
    oop1 = stackTop();
    setStackTop(dictionaryAt(oop1, oop2));
    PRIM_SUCCEEDED;
}
PRIM_ATTR(128, PRIM_SUCCEED)		

/* Dictionary at: put: */
DEF_PRIM(129)
{
    OOP oop1;
    OOP oop2;
    OOP oop3;

    oop3 = popOOP();
    oop2 = popOOP();
    oop1 = stackTop();
    dictionaryAtPut(oop1, oop2, oop3);
    setStackTop(oop3);
    PRIM_SUCCEEDED;
}
PRIM_ATTR(129, PRIM_SUCCEED)		

    /* This is not defined in terms of #error: in a .st file because some of
     * the required functionality may not be present when it gets first
     * invoked, say during the loading of the first kernel files.  We'll
     * redefine it later.
     */
/* Object doesNotUnderstand:
 * Object error: */
DEF_PRIM(130_131)
{
    OOP oop1;
    OOP oop2;

    oop2 = popOOP();
    oop1 = stackTop();
    printObject(oop1);
    if (primitive == 130) {
      printf(" did not understand selector '");
      printSymbol(messageSelector(oop2));
      printf("'\n\n");
    } else {
      printf(" error: ");
      printString(oop2);
      printf("\n\n");
    }
    showBacktrace();
    showStackContents();
    /* cannot halt - not a problem, since this is only temporary.
       The worst thing that can happen is that the image is not
       rebuilt correctly, but they'll understand it because it will
       result in loads of error messages. */

    PRIM_SUCCEEDED;
}
PRIM_ATTR(130_131, PRIM_SUCCEED)



/* Character class value: */
DEF_PRIM(132)
{
    OOP oop1;
    OOP oop2;

    oop2 = popOOP();
    oop1 = stackTop();
    if (isInt(oop2)) {
      register long arg2;
      arg2 = toInt(oop2);
      if (arg2 >= 0 && arg2 <= 255) {
	setStackTop(charOOPAt(arg2));
	PRIM_SUCCEEDED;
      }
    }
    unPop(1);
    PRIM_FAILED;
}
PRIM_ATTR(132, PRIM_SUCCEED | PRIM_FAIL)

/* Character asciiValue */
DEF_PRIM(133)
{
    OOP oop1;

    oop1 = stackTop();
    setStackTopInt(charOOPValue(oop1));
    PRIM_SUCCEEDED;
}
PRIM_ATTR(133, PRIM_SUCCEED)

/* Symbol class intern: aString */
DEF_PRIM(134)
{
    OOP oop2;

    oop2 = stackTop();	/* keeps this guy referenced while being interned */
    if (isClass(oop2, stringClass)) {
      OOP internedString;
      internedString = internStringOOP(oop2);
      (void)popOOP();
      
      setStackTop(internedString);
      PRIM_SUCCEEDED;
    }
    unPop(1);
    PRIM_FAILED;
}
PRIM_ATTR(134, PRIM_SUCCEED | PRIM_FAIL)

/* Dictionary new */
DEF_PRIM(135)
{
    setStackTop(dictionaryNew());
    PRIM_SUCCEEDED;
}
PRIM_ATTR(135, PRIM_SUCCEED)		


    
/* Memory addressOfOOP: oop */
DEF_PRIM(138)
{
    OOP oop1;
    OOP oop2;

    oop2 = popOOP();
    oop1 = popOOP();
    if (isOOP(oop2)) {
      pushOOP(fromCULong((long)oop2));
      PRIM_SUCCEEDED;
    }
    unPop(2);
    PRIM_FAILED;
}
PRIM_ATTR(138, PRIM_SUCCEED | PRIM_FAIL)

/* Memory addressOf: oop */
DEF_PRIM(139)
{
    OOP oop1;
    OOP oop2;

    oop2 = popOOP();
    oop1 = popOOP();
    if (isOOP(oop2)) {
      pushOOP(fromCULong((long)oopToObj(oop2)));
      PRIM_SUCCEEDED;
    }
    unPop(2);
    PRIM_FAILED;
}
PRIM_ATTR(139, PRIM_SUCCEED | PRIM_FAIL)		


/* SystemDictionary backtrace */
DEF_PRIM(140)
{
    showBacktrace();
    PRIM_SUCCEEDED;
}
PRIM_ATTR(140, PRIM_SUCCEED)

/* SystemDictionary getTraceFlag: anIndex */
DEF_PRIM(141)
{
    OOP oop1;
    OOP oop2;

    oop2 = popOOP();
    oop1 = popOOP();
    if (isInt(oop2)) {
      register long arg2;
      mst_Boolean *boolAddr;
      arg2 = toInt(oop2);
      boolAddr = boolAddrIndex(arg2);
      if (boolAddr != NULL) {
	oop1 = *boolAddr ? trueOOP : falseOOP;
	pushOOP(oop1);
	PRIM_SUCCEEDED;
      }
    }

    unPop(2);
    PRIM_FAILED;
}
PRIM_ATTR(141, PRIM_SUCCEED | PRIM_FAIL)		

/* SystemDictionary setTraceFlag: anIndex to: aBoolean */
DEF_PRIM(142)
{
    OOP oop1;
    OOP oop2;

    oop2 = popOOP();
    oop1 = popOOP();
    if (isInt(oop1)) {
      mst_Boolean *boolAddr;
      long arg1 = toInt(oop1);
      boolAddr = boolAddrIndex(arg1);
      if (boolAddr != NULL) {
	*boolAddr = (oop2 == trueOOP);
	setExceptFlag(true);
	PRIM_SUCCEEDED;
      }
    }
    
    unPop(2);
    PRIM_FAILED;
}
PRIM_ATTR(142, PRIM_SUCCEED | PRIM_FAIL)		


/* CObject class alloc: nBytes */
DEF_PRIM(144)
{
    OOP oop1;
    OOP oop2;

    oop2 = popOOP();
    oop1 = stackTop();
    if (isInt(oop2)) {
      register long arg2;
      arg2 = toInt(oop2);
      setStackTop(allocCObject(oop1, arg2));
      PRIM_SUCCEEDED;
    }
    unPop(1);
    PRIM_FAILED;
}
PRIM_ATTR(144, PRIM_SUCCEED | PRIM_FAIL)

/* Memory type: aType at: anAddress */
DEF_PRIM(145)
{
    OOP oop1;
    OOP oop2;
    OOP oop3;

    oop3 = popOOP();
    oop2 = popOOP();
    oop1 = popOOP();
    if (isCLong(oop3) && isInt(oop2)) {
      register long arg1, arg2;
      arg1 = toInt(oop2);
      arg2 = toCLong(oop3);
      switch (arg1) {
      case 0:			/* char */
	/* may want to use Character instead? */
	pushOOP(charOOPAt(*(char *)arg2));
	PRIM_SUCCEEDED;
      case 1:			/* unsigned char */
	pushOOP(charOOPAt(*(unsigned char *)arg2));
	PRIM_SUCCEEDED;
      case 2:			/* short */
	pushInt(*(short *)arg2);
	PRIM_SUCCEEDED;
      case 3:			/* unsigned short */
	pushInt(*(unsigned short *)arg2);
	PRIM_SUCCEEDED;
      case 4:			/* long */
	pushOOP(fromCLong(*(long *)arg2));
	PRIM_SUCCEEDED;
      case 5:			/* unsigned long */
	pushOOP(fromCULong(*(unsigned long *)arg2));
	PRIM_SUCCEEDED;
      case 6:			/* float */
	pushOOP(floatNew(*(float *)arg2));
	PRIM_SUCCEEDED;
      case 7:			/* double */
	pushOOP(floatNew(*(double *)arg2));
	PRIM_SUCCEEDED;
      case 8:			/* string */
	if (*(char **)arg2) {
	  pushOOP(stringNew(*(char **)arg2));
	} else {
	  pushOOP(nilOOP);
	}
	PRIM_SUCCEEDED;
      case 9:			/* OOP */
	pushOOP(*(OOP *)arg2);
	PRIM_SUCCEEDED;
      case 10:			/* int */
	pushOOP(fromCInt(*(int *)arg2));
	PRIM_SUCCEEDED;
      case 11:			/* unsigned int */
	pushOOP(fromCUInt(*(unsigned int *)arg2));
	PRIM_SUCCEEDED;
      }
    }

    unPop(3);
    PRIM_FAILED;
}
PRIM_ATTR(145, PRIM_SUCCEED | PRIM_FAIL)

/* Memory type: aType at: anAddress put: aValue */
DEF_PRIM(146)
{
    OOP oop4;
    OOP oop3;
    OOP oop2;

    oop4 = popOOP();
    oop3 = popOOP();
    oop2 = popOOP();
    /* don't pop the receiver */
    if (isCLong(oop3) && isInt(oop2)) {
      register long arg1, arg2;
      arg1 = toInt(oop2);
      arg2 = toCLong(oop3);
      switch (arg1) {
      case 0:			/* char */
      case 1:			/* unsigned char */
	/* may want to use Character instead? */
	if (isClass(oop4, charClass)) {
	  *(char *)arg2 = charOOPValue(oop4);
	  PRIM_SUCCEEDED;
	} else if (isInt(oop4)) {
	  *(char *)arg2 = (char)toInt(oop4);
	  PRIM_SUCCEEDED;
	}
	break;
      case 2:			/* short */
      case 3:			/* unsigned short */
	if (isInt(oop4)) {
	  *(short *)arg2 = (short)toInt(oop4);
	  PRIM_SUCCEEDED;
	}
	break;
      case 4:			/* long */
      case 5:			/* unsigned long */
	if (isCLong(oop4)) {
	  *(long *)arg2 = toCLong(oop4);
	  PRIM_SUCCEEDED;
	}
	break;
      case 6:			/* float */
	if (isClass(oop4, floatClass)) {
	  *(float *)arg2 = (float)floatOOPValue(oop4);
	  PRIM_SUCCEEDED;
	}
	break;
      case 7:			/* double */
	if (isClass(oop4, floatClass)) {
	  *(double *)arg2 = floatOOPValue(oop4);
	  PRIM_SUCCEEDED;
	}
	break;
      case 8:			/* string */
	if (isClass(oop4, stringClass) || isClass(oop4, symbolClass)) {
	  /* Char* cast on the right side needed because toCString returns Byte * */
	  *(char **)arg2 = (char *)toCString(oop4);
	  PRIM_SUCCEEDED;
	}
	break;
      case 9:			/* OOP */
	*(OOP *)arg2 = oop4;
	PRIM_SUCCEEDED;
      case 10:			/* int */
      case 11:			/* unsigned int */
	if (isCInt(oop4)) {
	  *(int *)arg2 = toCInt(oop4);
	  PRIM_SUCCEEDED;
	}
	break;
      }
    }

    unPop(3);
    PRIM_FAILED;
}
PRIM_ATTR(146, PRIM_SUCCEED | PRIM_FAIL)


    /* HOLE -- 147/149  */
    

/* methodsFor: category */
DEF_PRIM(150)
{
    setCompilationCategory(popOOP());
    setCompilationClass(stackTop());
    displayCompilationTrace("Compiling");
    compileCode = true;
    PRIM_SUCCEEDED;
}
PRIM_ATTR(150, PRIM_SUCCEED)

/* methodsFor: category ifTrue: condition */
DEF_PRIM(151)
{
    OOP oop2;

    oop2 = popOOP();
    setCompilationCategory(popOOP());
    setCompilationClass(stackTop());
    if (oop2 == trueOOP) {
      displayCompilationTrace("Conditionally compiling");
    } else {
      skipCompilation = true;
      displayCompilationTrace("Conditionally skipping");
    }
    compileCode = true;
    PRIM_SUCCEEDED;
}
PRIM_ATTR(151, PRIM_SUCCEED)		


/* ProcessorScheduler signal: aSemaphore onInterrupt: anInteger */
DEF_PRIM(152)
{
    OOP oop1;
    OOP oop2;

    oop2 = popOOP();
    oop1 = popOOP();
    if (isInt(oop2)) {
      register long arg2;
      arg2 = toInt(oop2);
      semIntVec[arg2] = oop1;
      setSignalHandler(arg2, semIntHandler);
      /* should probably package up the old interrupt state here for return
       * so that it can be undone */
      PRIM_SUCCEEDED;
    }

    unPop(2);
    PRIM_FAILED;
}
PRIM_ATTR(152, PRIM_SUCCEED | PRIM_FAIL)


/* SystemDictionary spaceGrowRate */
DEF_PRIM(153)
{
    setStackTop(floatNew((double) spaceGrowRate));
    PRIM_SUCCEEDED;
}
PRIM_ATTR(153, PRIM_SUCCEED)

/* SystemDictionary spaceGrowRate: */
DEF_PRIM(154)
{
    long	arg1;
    OOP oop1;

    oop1 = popOOP();
    if (isClass(oop1, floatClass)) {
      arg1 = (int) floatOOPValue(oop1);
    } else if (isInt(oop1)) {
      arg1 = toInt(oop1);
    } else {
      unPop(1);
      PRIM_FAILED;
    }

    if (arg1 > 0 && arg1 <= 500) {
      spaceGrowRate = arg1;
      PRIM_SUCCEEDED;
    }

    unPop(1);
    PRIM_FAILED;
}
PRIM_ATTR(154, PRIM_SUCCEED | PRIM_FAIL)
    
/* SystemDictionary growThresholdPercent */
DEF_PRIM(155)
{
    setStackTop(floatNew((double) growThresholdPercent));
    PRIM_SUCCEEDED;
}
PRIM_ATTR(155, PRIM_SUCCEED)

/* SystemDictionary growThresholdPercent: */
DEF_PRIM(156)
{
    OOP 	oop1;
    long	arg1;

    oop1 = popOOP();
    if (isClass(oop1, floatClass)) {
      arg1 = (int) floatOOPValue(oop1);
    } else if (isInt(oop1)) {
      arg1 = toInt(oop1);
    } else {
      unPop(1);
      PRIM_FAILED;
    }
    if (arg1 > 0 && arg1 < 100) {
      growThresholdPercent = arg1;
      PRIM_SUCCEEDED;
    }

    unPop(1);
    PRIM_FAILED;
}
PRIM_ATTR(156, PRIM_SUCCEED | PRIM_FAIL)

/* SystemDictionary growTo: numBytes */
DEF_PRIM(157)
{
    OOP oop1;
    long	arg1;

    oop1 = popOOP();
    if (isInt(oop1)) {
      arg1 = toInt(oop1);
      if (growTo(arg1)) {
	PRIM_SUCCEEDED;
      }
    }

    unPop(1);
    PRIM_FAILED;
}
PRIM_ATTR(157, PRIM_SUCCEED | PRIM_FAIL)


/* CObject class alloc: nbytes type: aType */
DEF_PRIM(158)
{
    OOP oop1;
    OOP oop2;

    oop1 = popOOP();
    oop2 = popOOP();
    if (isInt(oop2)) {
      register long arg2;
      arg2 = toInt(oop2);
      arg2 = (long)xmalloc(arg2);

      setStackTop(cObjectNewTyped(arg2, oop1));
      PRIM_SUCCEEDED;
    }
    unPop(2);
    PRIM_FAILED;
}
PRIM_ATTR(158, PRIM_SUCCEED | PRIM_FAIL)		


/* exp */
DEF_PRIM(160)
{
    OOP oop1;

    oop1 = stackTop();
    if (receiverIsClass(oop1, floatClass)) {
      double farg1 = floatOOPValue(oop1);
      setStackTop(floatNew(exp(farg1)));
      PRIM_SUCCEEDED;
    }
    PRIM_FAILED;
}
PRIM_ATTR(160, PRIM_CHECKS_RECEIVER)

/* ln */
DEF_PRIM(161)
{
    OOP oop1;

    oop1 = stackTop();
    if (receiverIsClass(oop1, floatClass)) {
      double farg1 = floatOOPValue(oop1);
      if (isNaN(farg1) || farg1 > 0.0) {
	setStackTop(floatNew(log(farg1)));
	PRIM_SUCCEEDED;
      }
    }
    PRIM_FAILED;
}
PRIM_ATTR(161, PRIM_SUCCEED | PRIM_FAIL)

/* SystemDictionary resetStatistics */
DEF_PRIM(162)
{
#ifdef PROFBLOCK
    ps.numThisContexts = 0;
    ps.numMethodAllocs = 0;
    ps.numMethodFrees = 0;
    ps.numMessageSends = 0;
    ps.numValues = 0;
    ps.numCacheCollisions = 0;
    ps.stackSizeSum = 0;
    ps.stackDepth = 0;
    ps.maxStackDepth = 0;
    ps.numMinorGCs = 0;
    ps.numMajorGCs = 0;
    initByteCodeCounter();
#endif
    PRIM_SUCCEEDED;
}
PRIM_ATTR(162, PRIM_SUCCEED)

/* SystemDictionary printStatistics */
DEF_PRIM(163)
{
#ifdef PROFBLOCK
    printf("%d thisContexts, of %d contexts used %d were lifo\n",
	   ps.numThisContexts, ps.numMethodAllocs, ps.numMethodFrees);
    printf("%d message sends, %d were value: methods\n",
	   ps.numMessageSends, ps.numValues);
    printf("%d method cache collisions\n", ps.numCacheCollisions);
    printf("%d maxDepth\n", ps.maxStackDepth);
    printf("%d gc's (%d minor, %d major)\n",
    	   ps.numMinorGCs + ps.numMajorGCs, ps.numMinorGCs, ps.numMajorGCs);

    printByteCodeCounts();
#else
    printf("VM profiling not active\n");
#endif
    PRIM_SUCCEEDED;
}
PRIM_ATTR(163, PRIM_SUCCEED)

/* raisedTo: aNumber -- receiver ** aNumber */
DEF_PRIM(164)
{
    OOP oop1;
    OOP oop2;

    oop2 = popOOP();
    oop1 = stackTop();
    if (receiverIsClass(oop1, floatClass) && isClass(oop2, floatClass)) {
      double farg1, farg2;
      farg1 = floatOOPValue(oop1);
      farg2 = floatOOPValue(oop2);
      if (farg1 > 0.0 || (farg1 = 0.0 && farg2 > 0.0)) {
        setStackTop(floatNew(pow(farg1, farg2)));
        PRIM_SUCCEEDED;
      }
    }
    unPop(1);
    PRIM_FAILED;
}
PRIM_ATTR(164, PRIM_SUCCEED | PRIM_FAIL)

/* CObject free */
DEF_PRIM(165)
{
    OOP oop1;

    oop1 = stackTop();
    if (receiverIsAKindOf(oopClass(oop1), cObjectClass)) {
      freeCObject(oop1);	/* free allocated space */
      setStackTop(nilOOP);
      PRIM_SUCCEEDED;
    }
    PRIM_FAILED;
}
PRIM_ATTR(165, PRIM_CHECKS_RECEIVER)

/* sqrt */
DEF_PRIM(166)
{
    OOP oop1;

    oop1 = stackTop();
    if (receiverIsClass(oop1, floatClass)) {
      double farg1 = floatOOPValue(oop1);
      if (farg1 < 0.0) {
	PRIM_FAILED;
      }
      setStackTop(floatNew(sqrt(farg1)));
      PRIM_SUCCEEDED;
    }
    PRIM_FAILED;
}
PRIM_ATTR(166, PRIM_SUCCEED | PRIM_FAIL)

  /* >>>>>> 167: HOLE <<<<<< */

/* ceiling, floor */
DEF_PRIM(168_169)
{
    OOP oop1;

    oop1 = stackTop();
    if (receiverIsClass(oop1, floatClass)) {
      double farg1 = floatOOPValue(oop1);
      if ((farg1 > MIN_ST_INT) && farg1 < MAX_ST_INT) {
	switch (primitive) {
	case 168: setStackTopInt((long)ceil(farg1));	PRIM_SUCCEEDED;
	case 169: setStackTopInt((long)floor(farg1));	PRIM_SUCCEEDED;
        }
      }
    }
    PRIM_FAILED;
}
PRIM_ATTR(168_169, PRIM_SUCCEED | PRIM_FAIL)

/* Behavior basicNewInFixedSpace */
DEF_PRIM(170)
{
    OOP oop1;

    oop1 = stackTop();
    if (isOOP(oop1)) {
      if (!classIsIndexable(oop1)) {
	oop1 = allocOOP(instantiate(oop1));
	makeFixedOOP(oop1);
	setStackTop(oop1);
	PRIM_SUCCEEDED;
      }
    }
    PRIM_FAILED;
}
PRIM_ATTR(170, PRIM_SUCCEED | PRIM_FAIL)

/* Behavior basicNewInFixedSpace: */
DEF_PRIM(171)
{
    OOP oop1;
    OOP oop2;

    oop2 = popOOP();
    oop1 = stackTop();
    if (isOOP(oop1) && isInt(oop2)) {
      if (classIsIndexable(oop1)) {
	register long arg2;
	arg2 = toInt(oop2);
	if (arg2 >= 0) {
	  oop1 = instantiateOOPWith(oop1, arg2);
	  makeFixedOOP(oop1);
	  setStackTop(oop1);
	  PRIM_SUCCEEDED;
	}
      }
    }

    unPop(1);
    PRIM_FAILED;
}
PRIM_ATTR(171, PRIM_SUCCEED | PRIM_FAIL)

DEF_PRIM(172)
{
    OOP oop1;

    oop1 = stackTop();
    if (isOOP(oop1) && !oopIsReadOnly(oop1)) {
      if (makeFixedOOP(oop1)) {
	PRIM_SUCCEEDED;
      }
    }
    PRIM_FAILED;
}
PRIM_ATTR(172, PRIM_SUCCEED | PRIM_FAIL)

/* tan */
DEF_PRIM(176)
{
    OOP oop1;

    oop1 = stackTop();
    if (receiverIsClass(oop1, floatClass)) {
      double farg1 = floatOOPValue(oop1);
      farg1 = tan(farg1);
      if (!isFinite(farg1) || (farg1 > -1.0e8 && farg1 < 1.0e8)) {
        /* Already infinite, or in a decent range - ok */
        setStackTop(floatNew(farg1));
	PRIM_SUCCEEDED;
      } else {
        /* Huge, push infinity (either positive or negative) */
        setStackTop(floatNew(farg1 / 0.0));
	PRIM_SUCCEEDED;
      }
    }
    PRIM_FAILED;
}
PRIM_ATTR(176, PRIM_SUCCEED | PRIM_FAIL)

/* sin, cos, arcTan */
DEF_PRIM(177_179)
{
    OOP oop1;

    oop1 = stackTop();
    if (receiverIsClass(oop1, floatClass)) {
      double farg1 = floatOOPValue(oop1);
      switch (primitive) {
      case 177: setStackTop(floatNew(sin(farg1)));	PRIM_SUCCEEDED;
      case 178: setStackTop(floatNew(cos(farg1)));	PRIM_SUCCEEDED;
      case 179: setStackTop(floatNew(atan(farg1)));	PRIM_SUCCEEDED;
      }
    }
    PRIM_FAILED;
}
PRIM_ATTR(177_179, PRIM_CHECKS_RECEIVER)

/* arcSin, arcCos */
DEF_PRIM(180_181)
{
    OOP oop1;

    oop1 = stackTop();
    if (receiverIsClass(oop1, floatClass)) {
      double farg1 = floatOOPValue(oop1);
      if (isNaN(farg1)) {
        PRIM_SUCCEEDED;		/* return NaN */
      } else if (farg1 < -1.0 || farg1 > 1.0) {
        PRIM_FAILED;
      }
      switch (primitive) {
      case 180: setStackTop(floatNew(asin(farg1)));	PRIM_SUCCEEDED;
      case 181: setStackTop(floatNew(acos(farg1)));	PRIM_SUCCEEDED;
      }
    }
    PRIM_FAILED;
}
PRIM_ATTR(180_181, PRIM_SUCCEED | PRIM_FAIL)


/* CObject at: byteoffset type: aType */
DEF_PRIM(182_183)
{
    OOP oop1;
    OOP oop2;
    OOP oop3;

    oop3 = popOOP();
    oop2 = popOOP();
    oop1 = popOOP();
    if (isInt(oop2)) {
      char*  addr;
      register long arg2;
      arg2 = toInt(oop2);
      if (primitive == 182) {
	addr = *(char **)cObjectValue(oop1);
	if (addr == 0) {
	  pushOOP(nilOOP);
	  PRIM_SUCCEEDED;
	} 
      } else {
	addr = cObjectValue(oop1);
      }

      addr += arg2;		/* compute effective address */
      if (isInt(oop3)) {	/* int type spec means a scalar type */
	long arg3;
	arg3 = toInt(oop3);
	
	switch (arg3) {
	case 0:
	  pushOOP(charOOPAt(*(Byte *)addr));
	  PRIM_SUCCEEDED;

	case 1:
	  pushOOP(charOOPAt(*(Byte *)addr));
	  PRIM_SUCCEEDED;

	case 2:
	  pushInt(*(short *)addr);
	  PRIM_SUCCEEDED;

	case 3:
	  pushInt(*(unsigned short *)addr);
	  PRIM_SUCCEEDED;

	case 4:
	  pushOOP(fromCLong(*(long *)addr));
	  PRIM_SUCCEEDED;

	case 5:
	  pushOOP(fromCULong(*(unsigned long *)addr));
	  PRIM_SUCCEEDED;

	case 6:
	  pushOOP(floatNew(*(float *)addr));
	  PRIM_SUCCEEDED;

	case 7:
	  pushOOP(floatNew(*(double *)addr));
	  PRIM_SUCCEEDED;

	case 8:
	  {
	    char **strAddr;
	    strAddr = (char **)addr;
	    if (*strAddr) {
	      pushOOP(stringNew(*strAddr));
	      PRIM_SUCCEEDED;
	    } else {
	      pushOOP(nilOOP);
	      PRIM_SUCCEEDED;
	    }
	  }
	case 9:
	  pushOOP(*(OOP *)addr);
	  PRIM_SUCCEEDED;

	case 10:
	  pushOOP(fromCInt(*(int *)addr));
	  PRIM_SUCCEEDED;

	case 11:
	  pushOOP(fromCUInt(*(unsigned int *)addr));
	  PRIM_SUCCEEDED;
	} 

      } else {			/* non int type means use the type as the 
				   type of the effective address */
	/* It's an oddball case, but it does seem possible that
	 * oop3 could get GC'ed out of existence before it gets used,
	 * since oop3 is not on the stack, and if cObjectNewTyped could
	 * cause a GC
	 */
	IncPtr		incPtr;

	incPtr = incSavePointer();
	incAddOOP(oop3);
	setStackTop(cObjectNewTyped(addr, oop3));
	incRestorePointer(incPtr);
	PRIM_SUCCEEDED;
      }
    }
    unPop(3);
    PRIM_FAILED;
}
PRIM_ATTR(182_183, PRIM_SUCCEED | PRIM_FAIL)

/* set the value; 184 derefs first *
 * CObject at: byteOffset put: aValue type: aType */
DEF_PRIM(184_185)
{
      /* I don't think that this deals properly with setting the pointer
	 value as opposed to setting the pointed-to value. */
    OOP oop1;
    OOP oop2;
    OOP oop3;
    OOP oop4;

    oop4 = popOOP();
    oop3 = popOOP();
    oop2 = popOOP();
    oop1 = stackTop();
      
    if (isInt(oop2)) {
      char*  addr;
      register long arg2;
      arg2 = toInt(oop2);
      if (primitive == 184) {
	addr = *(char **)cObjectValue(oop1);
      } else {
	addr = cObjectValue(oop1);
      }

      addr += arg2;		/* compute effective address */
      if (isInt(oop4)) {	/* int type spec means a scalar type */
	long arg4;
	arg4 = toInt(oop4);
	switch (arg4) {
	case 0:			/* char */
	case 1:			/* uchar */
	  if (isClass(oop3, charClass)) {
	    *addr = charOOPValue(oop3);
	    PRIM_SUCCEEDED;
	  } else if (isInt(oop3)) {
	    *(char *)addr = (char) toInt(oop3);
	    PRIM_SUCCEEDED;
	  }
	  break;

	case 2:			/* short */
	case 3:			/* ushort */
	  if (isInt(oop3)) {
	    *(short *)addr = (short)toInt(oop3);
	    PRIM_SUCCEEDED;
	  }
	  break;

	case 4:			/* long */
	case 5:			/* ulong */
	  if (isCLong(oop3)) {
	    *(long *)addr = (long)toCLong(oop3);
	    PRIM_SUCCEEDED;
	  }
	  break;

	case 6:
	  {
	    float *floatAddr;
	    floatAddr = (float *)addr;
	    if (isInt(oop3)) {
	      *floatAddr = (float)toInt(oop3);
	      PRIM_SUCCEEDED;
	    } else if (isClass(oop3, floatClass)) {
	      *floatAddr = (float)floatOOPValue(oop3);
	      PRIM_SUCCEEDED;
	    } 
	  }
	  break;

	case 7:			/* double */
	  {
	    double *doubleAddr;
	    doubleAddr = (double *)addr;
	    if (isInt(oop3)) {
	      *doubleAddr = toInt(oop3);
	      PRIM_SUCCEEDED;
	    } else if (isClass(oop3, floatClass)) {
	      *doubleAddr = floatOOPValue(oop3);
	      PRIM_SUCCEEDED;
	    }
	  }
	  break;

	case 8:			/* string */
	  {			/* note that this does not allow for
				 * replacemnt in place */
				/* to replace in place, use replaceFrom: */
	    char **strAddr;
	    strAddr = (char **)addr;
	    if (oop3 == nilOOP) {
	      *strAddr = (char *)0;
	      PRIM_SUCCEEDED;
	    } else {
	      if (isAKindOf(oopClass(oop3), stringClass)) {
		*strAddr = (char *)toCString(oop3);
		PRIM_SUCCEEDED;
	      }
	    }
	  }

	case 9:
	  *(OOP *)addr = oop3;
	  PRIM_SUCCEEDED;

	case 10:			/* int */
	case 11:			/* uint */
	  if (isCInt(oop3)) {
	    *(int *)addr = (int)toCInt(oop3);
	    PRIM_SUCCEEDED;
	  }
	  break;

	}
      } else {		/* non int type means use the type as the 
				   type of the effective address */
	*(voidPtr *)addr = cObjectValue(oop3);    /* IS THIS RIGHT?!? */
	PRIM_SUCCEEDED;
      }
    }

    unPop(3);
    PRIM_FAILED;
}
PRIM_ATTR(184_185, PRIM_SUCCEED | PRIM_FAIL)

/* CString replaceWith: aString */
DEF_PRIM(188)
{
    OOP oop1;
    OOP oop2;

    oop2 = popOOP();
    oop1 = stackTop();
    
    /* assumes the receiver is already pointing at an area of memory that is
     * the correct size; does not (re)allocate receiver's string at all.
     */

    if (isClass(oop2, stringClass) || isClass(oop2, byteArrayClass)) {
      unsigned long srcLen;
      Byte	*dstBase, *srcBase;
      srcBase = stringOOPChars(oop2);
      srcLen = numIndexableFields(oop2);

      dstBase = *(Byte **)cObjectValue(oop1);
      memcpy(dstBase, srcBase, srcLen);
      dstBase[srcLen] = '\0';	/* since it's a CString type, we NUL term it */
      PRIM_SUCCEEDED;
    }
    unPop(1);
    PRIM_FAILED;
}
PRIM_ATTR(188, PRIM_SUCCEED | PRIM_FAIL)

  /* >>>>>> 189-199: HOLE <<<<<< */

/* ByteArray class fromCdata: aCObject size: anInteger
 * ??? Are these used at all? */
DEF_PRIM(200)
{
    OOP oop1;
    OOP oop2;
    OOP oop3;

    oop3 = popOOP();
    oop2 = popOOP();
    oop1 = stackTop();
    if (isInt(oop3)) {
      long arg3 = toInt(oop3);
      OOP byteArrayOOP = byteArrayNew(cObjectValue(oop2), arg3);
      setStackTop(byteArrayOOP);
      PRIM_SUCCEEDED;
    }
    unPop(2);
    PRIM_FAILED;
}
PRIM_ATTR(200, PRIM_SUCCEED | PRIM_FAIL)

/* String class fromCdata: aCObject size: anInteger */
DEF_PRIM(201)
{
    OOP oop1;
    OOP oop2;
    OOP oop3;

    oop3 = popOOP();
    oop2 = popOOP();
    oop1 = stackTop();
    if (isInt(oop3)) {
      long arg3 = toInt(oop3);
      OOP stringOOP = countedStringNew(cObjectValue(oop2), arg3);
      setStackTop(stringOOP);
      PRIM_SUCCEEDED;
    }
    unPop(2);
    PRIM_FAILED;
}
PRIM_ATTR(201, PRIM_SUCCEED | PRIM_FAIL)

/* String asCdata: aCType
 * ByteArray asCdata: aCType */
DEF_PRIM(203_204)
{
    voidPtr data;
    int     size;
    OOP oop1;
    OOP oop2;

    oop2 = popOOP();
    oop1 = stackTop();
#ifndef OPTIMIZE
    if ((isClass(oop1, stringClass) && primitive == 203)
          || (isClass(oop1, byteArrayClass) && primitive == 204)) {
#endif
      if(isAKindOf(oopClass(oop2), cTypeClass)) {
        size = numIndexableFields(oop1);
        data = xmalloc(size);
        if (data) {
	  memcpy(data, oopToObj(oop1)->data, size);
	  setStackTop(cObjectNewTyped(data, oop2));
	  PRIM_SUCCEEDED;
	}
      }
#ifndef OPTIMIZE
    }
#endif
    unPop(1);
    PRIM_FAILED;
}
PRIM_ATTR(203_204, PRIM_CHECKS_RECEIVER)		


  /* >>>>>> 205-229: HOLE <<<<<< */


/* SystemDictionary monitor: aBoolean */
DEF_PRIM(230)
{
    OOP oop1;

    oop1 = popOOP();
#ifdef USE_MONCONTROL    
    moncontrol(oop1 == trueOOP);
#endif /* USE_MONCONTROL */
    PRIM_SUCCEEDED;
}
PRIM_ATTR(230, PRIM_SUCCEED)


/* SystemDictionary byteCodeCounter */
DEF_PRIM(231)
{
    setStackTopInt(byteCodeCounter);
    PRIM_SUCCEEDED;
}
PRIM_ATTR(231, PRIM_SUCCEED)

/* SystemDictionary debug */
DEF_PRIM(232)
{
    debug();			/* used to allow gdb to stop based on
				 * Smalltalk execution paths.
				 */
    PRIM_SUCCEEDED;
}
PRIM_ATTR(232, PRIM_SUCCEED)


/* Object isReadOnly */
DEF_PRIM(233)
{
    OOP oop1;

    oop1 = stackTop();
    setStackTopBoolean(oopIsReadOnly(oop1));
    PRIM_SUCCEEDED;
}
PRIM_ATTR(233, PRIM_SUCCEED)

/* Object makeReadOnly: */
DEF_PRIM(234)
{
    OOP oop1;
    OOP oop2;

    oop2 = popOOP();
    oop1 = stackTop();
    if (oop2 == trueOOP) {
      makeOOPReadOnly(oop1, true);
      PRIM_SUCCEEDED;
    } else if (oop2 == falseOOP) {
      makeOOPReadOnly(oop1, false);
      PRIM_SUCCEEDED;
    } else {
      unPop(1);
      PRIM_FAILED;
    }
}
PRIM_ATTR(234, PRIM_SUCCEED | PRIM_FAIL)


/* Behavior compileString: aString */
DEF_PRIM(235)
{
    OOP oop1;
    OOP oop2;

    oop2 = popOOP();
    oop1 = popOOP();
    if (isClass(oop2, stringClass)) {
      compileCode = true;	/* tell the lexer we do internal compiles */
      pushSmalltalkString(oop2);
      setCompilationClass(oop1);
      parseStream();
      popStream(false);		/* don't close a String! */
      pushOOP(latestCompiledMethod);
      PRIM_SUCCEEDED;
    }
    unPop(2);
    PRIM_FAILED;
}
PRIM_ATTR(235, PRIM_SUCCEED | PRIM_FAIL)

/* Behavior compileString: aString ifError: aBlock */
DEF_PRIM(236)
{
    OOP oop1;
    OOP oop2;
    OOP oop3;

    oop3 = popOOP();
    oop2 = popOOP();
    oop1 = popOOP();
    if (isClass(oop2, stringClass) && isClass(oop3, blockClosureClass)) {
      mst_Boolean	oldReportErrors = reportErrors;

      if (oldReportErrors) {
	/* only clear out these guys on first transition */
	firstErrorStr = firstErrorFile = NULL;
      }
      reportErrors = false;
      compileCode = true;	/* tell the lexer we do internal compiles */
      pushSmalltalkString(oop2);
      setCompilationClass(oop1);
      parseStream();
      popStream(false);		/* don't close a String! */
      if (firstErrorStr != NULL) {
	pushOOP(oop3);		/* block context */
	if (firstErrorFile != NULL) {
	  pushOOP(stringNew(firstErrorFile));
	  xfree(firstErrorFile);
	} else {
	  pushOOP(nil);
	}
	pushInt(firstErrorLine);
	pushOOP(stringNew(firstErrorStr));
	xfree(firstErrorStr);
	firstErrorStr = firstErrorFile = NULL;
        reportErrors = oldReportErrors;
	if(sendBlockValue(3)) {
	  PRIM_FAILED;
	} else {
	  PRIM_SUCCEEDED_RELOAD_IP;
	}
      } else {
        reportErrors = oldReportErrors;
	pushOOP(latestCompiledMethod);
      }
      PRIM_SUCCEEDED;
    }
    unPop(3);
    PRIM_FAILED;
}
PRIM_ATTR(236, PRIM_FAIL | PRIM_SUCCEED | PRIM_RELOAD_IP)		



/* FileStream fileIn */
DEF_PRIM(247)
{
    char	*fileName;
    FileStream	fileStream;
    FILE	*file;
    OOP		oop1;

    oop1 = stackTop();
    fileStream = (FileStream)oopToObj(oop1);
    if (isNil(fileStream->file)) {
      PRIM_FAILED;
    }
    file = (FILE *)cObjectValue(fileStream->file);
    fileName = toCString(fileStream->name);
    if (fileIsReadable(fileName)) {
      pushUNIXFile(file, fileName);
      useUndeclared++;
      parseStream();
      useUndeclared--;
      popStream(false);		/* we didn't open it, so we don't close it */
      PRIM_SUCCEEDED;
    }
    xfree(fileName);
    PRIM_FAILED;
}
PRIM_ATTR(247, PRIM_SUCCEED | PRIM_FAIL)		

/* FileStream fileInLine: lineNum
 *            fileName: aString
 *	      at: charPosInt
 */
DEF_PRIM(248)
{
    char	*fileName, *realFileName;
    FileStream	fileStream;
    FILE	*file;
    OOP 	oop1, oop2, oop3, oop4;

    oop4 = popOOP();
    oop3 = popOOP();
    oop2 = popOOP();
    oop1 = stackTop();
    fileStream = (FileStream)oopToObj(oop1);
    if (isNil(fileStream->file)) {
      PRIM_FAILED;
    }
    file = (FILE *)cObjectValue(fileStream->file);
    fileName = toCString(fileStream->name);
    realFileName = nil;
    if (fileIsReadable(fileName)) {
      if (isInt(oop2)
	  && (isNil(oop3) || (isClass(oop3, stringClass) && isInt(oop4)))) {
	register long arg2;
	long arg4;
	arg2 = toInt(oop2);
	if (!isNil(oop3)) {
	  arg4 = toInt(oop4);
	  realFileName = toCString(oop3);
	} else {
	  arg4 = 0;
	}

	pushUNIXFile(file, fileName);
	setStreamInfo(arg2, realFileName, arg4);
	parseStream();
	popStream(false);		/* we didn't open it, so we don't close it */
	PRIM_SUCCEEDED;
      }
    }
    xfree(fileName);
    if (realFileName) {
      xfree(realFileName);
    }
    unPop(3);
    PRIM_FAILED;
}
PRIM_ATTR(248, PRIM_SUCCEED | PRIM_FAIL)


/* Behavior makeDescriptorFor: funcNameString
            returning: returnTypeSymbol
	    withArgs: argsArray */
DEF_PRIM(249)
{
    OOP oop1;
    OOP oop2;
    OOP oop3;
    OOP oop4;

    oop4 = popOOP();
    oop3 = popOOP();
    oop2 = popOOP();
    oop1 = popOOP();

    if (isClass(oop2, stringClass)
	&& (isClass(oop3, symbolClass)
	    || isAKindOf(oopClass(oop3), cTypeClass))
	&& (isClass(oop4, arrayClass)
	    || isClass(oop4, undefinedObjectClass))) {
      pushOOP(makeDescriptor(oop2, oop3, oop4));
      PRIM_SUCCEEDED;
    }
    unPop(4);
    PRIM_FAILED;
}
PRIM_ATTR(249, PRIM_SUCCEED | PRIM_FAIL)

  /* >>>>>>>>>>>>>> HOLE: 250 <<<<<<<<<<<<< */

/* Object snapshot: aString */
DEF_PRIM(251)
{
    char	*fileName;
    OOP 	oop2;

    oop2 = popOOP();
    if (isClass(oop2, stringClass)) {
      fileName = toCString(oop2);
      saveToFile(fileName);
      xfree(fileName);
      PRIM_SUCCEEDED;
    }
    unPop(1);
    PRIM_FAILED;
}
PRIM_ATTR(251, PRIM_SUCCEED | PRIM_FAIL)		

/* Object basicPrint */
DEF_PRIM(252)
{
    printf("Object: ");
    printObject(stackTop());
    PRIM_SUCCEEDED;
}
PRIM_ATTR(252, PRIM_SUCCEED)

/* pb 1998 Nov 9 - added 253 for weak objects */
/* Object makeWeak */
DEF_PRIM(253)
{
    makeOOPWeak(stackTop());
    PRIM_SUCCEEDED;
}
PRIM_ATTR(253, PRIM_SUCCEED)


/* FileStream>>#fileOp..., variadic */
DEF_PRIM(254)
{
    char	*fileName, *fileMode;
    FileStream	fileStream;
    FILE	*file;
    OOP		oop1, oopVec[4];
    int		i, ch;
    long	arg1;

    /* Ordering of file operations must match that used in FileSegment.st */
    enum {
      openFilePrim,
      closeFilePrim,
      getCharPrim,
      putCharPrim,
      seekPrim,
      tellPrim,
      eofPrim,
      popenFilePrim,
      sizePrim,
      putCharsPrim,
      getCharsPrim,
      flushPrim,
      getBytePrim
    };


    for (i = numArgs; --i >= 0; ) {
      oopVec[i] = popOOP();
    }
    oop1 = stackTop();
    if (!isInt(oopVec[0])) {
      unPop(numArgs);
      PRIM_FAILED;
    }

    arg1 = toInt(oopVec[0]);
    if (arg1 == ENUM_INT(openFilePrim) || arg1 == ENUM_INT(popenFilePrim)) {
      /* open: fileName[1] mode: mode[2] or
       * popen: command[1] dir: direction[2] */
      fileName = toCString(oopVec[1]);
      fileMode = toCString(oopVec[2]);
      if (arg1 == ENUM_INT(openFilePrim)) {
	file = openFile((char *)fileName, (char *)fileMode);
      } else {
	file = openPipe(fileName, fileMode);
      }
      if (file == NULL) {
	xfree(fileName);
	xfree(fileMode);
	unPop(numArgs);
	PRIM_FAILED;
      }
      setFileStreamFile(oop1, cObjectNew(file), oopVec[1],
			  arg1 == ENUM_INT(popenFilePrim));
      xfree(fileName);
      xfree(fileMode);
      PRIM_SUCCEEDED;
    }

    fileStream = (FileStream)oopToObj(oop1);
    if (isNil(fileStream->file)) {
      unPop(numArgs);
      PRIM_FAILED;
    }
    file = (FILE *)cObjectValue(fileStream->file);
    switch (arg1) {

    case closeFilePrim:	/* FileStream close */
      if (fileStream->isPipe == trueOOP) {
	setStackTopInt(closePipe(file));
      } else {
	setStackTopInt(fclose(file));
      }
      PRIM_SUCCEEDED;
	
    case getCharPrim:   /* FileStream next */
      ch = getc(file);
      if (ch == EOF) {	/* cause nil to be returned */
	setStackTop(nilOOP);
      } else {
	setStackTop(charOOPAt(ch));
      }
      PRIM_SUCCEEDED;
	  
    case putCharPrim:	/* FileStream nextPut: aChar */
      if (isClass(oopVec[1], charClass)) {
	ch = charOOPValue(oopVec[1]);
      } else if (isInt(oopVec[1])) {
	ch = toInt(oopVec[1]);
	if (ch < -128 || ch > 255) {
	  break;
	} else if (ch < 0) {
	  ch += 256;
	}
      } else {
	break;
      }
      fputc(ch, file);
      PRIM_SUCCEEDED;

    case seekPrim:		/* FileStream position: position */
      if (fseek(file, toInt(oopVec[1]), 0) == EOF) {
	break;
      } else {
        PRIM_SUCCEEDED;
      }

    case tellPrim:		/* FileStream position */
      setStackTopInt(ftell(file));
      if (toInt(stackTop()) == EOF) {
	break;
      } else {
	PRIM_SUCCEEDED;
      }

    case eofPrim: {		/* FileStream atEnd */ 
      mst_Boolean atEof;
      ch = getc(file);
      atEof = feof(file);
      setStackTopBoolean(atEof);
      ungetc(ch, file);
      PRIM_SUCCEEDED;
    }

    case sizePrim:
      {
	struct stat statBuf;
	fflush(file);
	if(fstat(fileno(file), &statBuf) < 0) {
	  break;
	}
	setStackTopInt(statBuf.st_size);
	PRIM_SUCCEEDED;
      }

    case putCharsPrim:	/* only works for strings currently */
      if (isAKindOf(oopClass(oopVec[1]), stringClass)) {
	fwrite(stringOOPChars(oopVec[1]), numIndexableFields(oopVec[1]), 1, file);
	PRIM_SUCCEEDED;
      } else {
	break;
      }

    case getCharsPrim:	/* only works for strings */
      if (isInt(oopVec[1])) {
	OOP stringOOP;
	register long arg2;
	arg2 = toInt(oopVec[1]);
	stringOOP = newString(arg2);
	if (fread(stringOOPChars(stringOOP), arg2, 1, file) == 0) {
	  break;
	}
	setStackTop(stringOOP);
	PRIM_SUCCEEDED;
      }

    case flushPrim:
      fflush(file);
      PRIM_SUCCEEDED;

    case getBytePrim:
      ch = getc(file);
      if (ch == EOF) {	/* cause nil to be returned */
	setStackTop(nilOOP);
      } else {
	setStackTopInt(ch);
      }
      PRIM_SUCCEEDED;
    }

    unPop(numArgs);
    PRIM_FAILED;
}
PRIM_ATTR(254, PRIM_SUCCEED | PRIM_FAIL)


/* C callout primitive. */
DEF_PRIM(255)
{
    mst_Boolean	oldInCCode;
    mst_Boolean result;
#ifdef STACK_JMPBUFS
    InterpJmpBuf	localJmpBuf;
#endif

#ifndef USE_JIT_TRANSLATION
    /* Fail if we were put in the caches used by #at:, #at:put:, or #size. */
    if (methodOOP == nil) {
      return (true);
    }
#endif

#ifdef STACK_JMPBUFS
    localJmpBuf.old = cCalloutJmpBuf;
    cCalloutJmpBuf = &localJmpBuf;
#else
    cCalloutJmpBuf->old = cCalloutJmpBuf;
#endif

    inInterpreter = false;
    oldInCCode = inCCode;
    inCCode = true;
    if (setjmp(cCalloutJmpBuf->jmpBuf) == 0) {
      result = invokeCRoutine(numArgs, methodOOP);
    }
    inCCode = oldInCCode;
    inInterpreter = !inCCode;
    cCalloutJmpBuf = cCalloutJmpBuf->old;
    if (result) {
      PRIM_SUCCEEDED;
    } else {
      PRIM_FAILED;
    }
}
PRIM_ATTR(255, PRIM_SUCCEED | PRIM_FAIL)


/* pb 1998 Nov 28 - added 256-257 for finalization */
DEF_PRIM(256)
{
    markOOPToFinalize(stackTop(), true);
    PRIM_SUCCEEDED;
}
PRIM_ATTR(256, PRIM_SUCCEED)
    
DEF_PRIM(257)
{
    markOOPToFinalize(stackTop(), false);
    PRIM_SUCCEEDED;
}
PRIM_ATTR(257, PRIM_SUCCEED)

/* Namespace current: aNamespace */
DEF_PRIM(258)
{
    /* Always fail */
    currentNamespace = stackTop();
    PRIM_FAILED;
}
PRIM_ATTR(258, PRIM_FAIL)

/* brd Sun Oct 24 15:36:15 PDT 1993 */
/* added primitives 256-264 to support the browser */

DEF_PRIM(260)
{
    gcFlip();
    PRIM_SUCCEEDED;
}
PRIM_ATTR(260, PRIM_SUCCEED) /* SystemDictionary compact */

  /* >>>>>> 261-263: HOLE <<<<<< */
  
DEF_PRIM(264)
{
    OOP oop1;

    oop1 = popOOP();
    enableGC = (oop1 == trueOOP);
    PRIM_SUCCEEDED;
}
PRIM_ATTR(264, PRIM_SUCCEED) /* SystemDictionary enableGC: aBoolean */

DEF_PRIM(HOLE)
{
    if (primitive >= 128) {
      errorf("Unhandled primitive operation %d", primitive);
    }
    unPop(numArgs);
    PRIM_FAILED;
}
PRIM_ATTR(HOLE, PRIM_FAIL)


#undef intBinOp
#undef boolBinOp
#undef DEF_PRIM
#undef COUNT_PRIM
#undef PRIM_ATTR
#undef DEF_PRIM

#define PRIM_ENTRY(name)  { executePrimitive_##name, attrPrimitive_##name }

#define HOLE_32 \
PRIM_ENTRY(HOLE), PRIM_ENTRY(HOLE), PRIM_ENTRY(HOLE), PRIM_ENTRY(HOLE), \
PRIM_ENTRY(HOLE), PRIM_ENTRY(HOLE), PRIM_ENTRY(HOLE), PRIM_ENTRY(HOLE), \
PRIM_ENTRY(HOLE), PRIM_ENTRY(HOLE), PRIM_ENTRY(HOLE), PRIM_ENTRY(HOLE), \
PRIM_ENTRY(HOLE), PRIM_ENTRY(HOLE), PRIM_ENTRY(HOLE), PRIM_ENTRY(HOLE), \
PRIM_ENTRY(HOLE), PRIM_ENTRY(HOLE), PRIM_ENTRY(HOLE), PRIM_ENTRY(HOLE), \
PRIM_ENTRY(HOLE), PRIM_ENTRY(HOLE), PRIM_ENTRY(HOLE), PRIM_ENTRY(HOLE), \
PRIM_ENTRY(HOLE), PRIM_ENTRY(HOLE), PRIM_ENTRY(HOLE), PRIM_ENTRY(HOLE), \
PRIM_ENTRY(HOLE), PRIM_ENTRY(HOLE), PRIM_ENTRY(HOLE), PRIM_ENTRY(HOLE)  \

PrimTableEntry primitiveTable[1024] = {
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(1), 
  PRIM_ENTRY(2), 
  PRIM_ENTRY(3), 
  PRIM_ENTRY(4), 
  PRIM_ENTRY(5), 		/* 5 */
  PRIM_ENTRY(6), 
  PRIM_ENTRY(7), 
  PRIM_ENTRY(8), 
  PRIM_ENTRY(9), 
  PRIM_ENTRY(10),		/* 10 */	     
  PRIM_ENTRY(11),			
  PRIM_ENTRY(12),
  PRIM_ENTRY(13),
  PRIM_ENTRY(14), 
  PRIM_ENTRY(15), 		/* 15 */
  PRIM_ENTRY(16), 
  PRIM_ENTRY(17),			        	
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),		/* 20 */
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),		/* 25 */
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),		/* 30 */
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),		/* 35 */
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(40),		/* 40 */
  PRIM_ENTRY(41_50),
  PRIM_ENTRY(41_50),
  PRIM_ENTRY(41_50),
  PRIM_ENTRY(41_50),
  PRIM_ENTRY(41_50),		/* 45 */
  PRIM_ENTRY(41_50),
  PRIM_ENTRY(41_50),
  PRIM_ENTRY(41_50),
  PRIM_ENTRY(41_50),
  PRIM_ENTRY(41_50),		/* 50 */
  PRIM_ENTRY(51),
  PRIM_ENTRY(52),
  PRIM_ENTRY(53),
  PRIM_ENTRY(54),
  PRIM_ENTRY(HOLE),		/* 55 */
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(60),		/* 60 */
  PRIM_ENTRY(61),
  PRIM_ENTRY(62),
  PRIM_ENTRY(63),
  PRIM_ENTRY(64),
  PRIM_ENTRY(HOLE),		/* 65 */
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(68),
  PRIM_ENTRY(69),
  PRIM_ENTRY(70),		/* 70 */
  PRIM_ENTRY(71),
  PRIM_ENTRY(72),
  PRIM_ENTRY(73),
  PRIM_ENTRY(74),
  PRIM_ENTRY(75),		/* 75 */
  PRIM_ENTRY(76),
  PRIM_ENTRY(77),
  PRIM_ENTRY(78),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(80),		/* 80 */
  PRIM_ENTRY(81),
  PRIM_ENTRY(82),
  PRIM_ENTRY(83),
  PRIM_ENTRY(84),
  PRIM_ENTRY(85),		/* 85 */
  PRIM_ENTRY(86),
  PRIM_ENTRY(87),
  PRIM_ENTRY(88),
  PRIM_ENTRY(89),
  PRIM_ENTRY(90),		/* 90 */
  PRIM_ENTRY(91),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),		/* 95 */
  PRIM_ENTRY(96),
  PRIM_ENTRY(97),
  PRIM_ENTRY(98),
  PRIM_ENTRY(99),
  PRIM_ENTRY(100),		/* 100 */
  PRIM_ENTRY(101),	
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(104),			
  PRIM_ENTRY(105),		/* 105 */
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(110),		/* 110 */
  PRIM_ENTRY(111),			
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),			
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),		/* 115 */
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(117),			
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),		/* 120 */
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),		/* 125 */
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(128),		
  PRIM_ENTRY(129),		
  PRIM_ENTRY(130_131),	/* 130 */	
  PRIM_ENTRY(130_131),		
  PRIM_ENTRY(132),	
  PRIM_ENTRY(133),	
  PRIM_ENTRY(134),	
  PRIM_ENTRY(135),		/* 135 */
  PRIM_ENTRY(HOLE),	
  PRIM_ENTRY(HOLE),	
  PRIM_ENTRY(138),	
  PRIM_ENTRY(139),	
  PRIM_ENTRY(140),		/* 140 */
  PRIM_ENTRY(141),	
  PRIM_ENTRY(142),	
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(144),	
  PRIM_ENTRY(145),		/* 145 */
  PRIM_ENTRY(146),	
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(150),		/* 150 */
  PRIM_ENTRY(151),	
  PRIM_ENTRY(152),	
  PRIM_ENTRY(153),	
  PRIM_ENTRY(154),	
  PRIM_ENTRY(155),		/* 155 */
  PRIM_ENTRY(156),	
  PRIM_ENTRY(157),	
  PRIM_ENTRY(158),	
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(160),		/* 160 */
  PRIM_ENTRY(161),	
  PRIM_ENTRY(162),	
  PRIM_ENTRY(163),	
  PRIM_ENTRY(164),	
  PRIM_ENTRY(165),		/* 165 */
  PRIM_ENTRY(166),	
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(168_169),
  PRIM_ENTRY(168_169),
  PRIM_ENTRY(170),		/* 170 */
  PRIM_ENTRY(171),
  PRIM_ENTRY(172),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),		/* 175 */
  PRIM_ENTRY(176),	
  PRIM_ENTRY(177_179),
  PRIM_ENTRY(177_179),
  PRIM_ENTRY(177_179),
  PRIM_ENTRY(180_181),	/* 180 */
  PRIM_ENTRY(180_181),
  PRIM_ENTRY(182_183),
  PRIM_ENTRY(182_183),
  PRIM_ENTRY(184_185),
  PRIM_ENTRY(184_185),	/* 185 */
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(188),		
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),		/* 190 */	
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),		/* 195 */
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(200),		/* 200 */
  PRIM_ENTRY(201),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(203_204),
  PRIM_ENTRY(203_204),
  PRIM_ENTRY(HOLE),		/* 205 */
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),		/* 210 */
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),		/* 215 */
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),		/* 220 */
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),		/* 225 */
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(230),		/* 230 */
  PRIM_ENTRY(231),
  PRIM_ENTRY(232),
  PRIM_ENTRY(233),
  PRIM_ENTRY(234),
  PRIM_ENTRY(235),		/* 235 */
  PRIM_ENTRY(236),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),		/* 240 */
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),		/* 245 */
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(247),	
  PRIM_ENTRY(248),	
  PRIM_ENTRY(249),
  PRIM_ENTRY(HOLE),		/* 250 */
  PRIM_ENTRY(251),	
  PRIM_ENTRY(252),	
  PRIM_ENTRY(253),	
  PRIM_ENTRY(254),	
  PRIM_ENTRY(255),		/* 255 */
  PRIM_ENTRY(256),
  PRIM_ENTRY(257),
  PRIM_ENTRY(258),	
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(260), 		/* 260 */
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE),
  PRIM_ENTRY(264), 

  PRIM_ENTRY(HOLE), PRIM_ENTRY(HOLE), PRIM_ENTRY(HOLE), PRIM_ENTRY(HOLE), PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE), PRIM_ENTRY(HOLE), PRIM_ENTRY(HOLE), PRIM_ENTRY(HOLE), PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE), PRIM_ENTRY(HOLE), PRIM_ENTRY(HOLE), PRIM_ENTRY(HOLE), PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE), PRIM_ENTRY(HOLE), PRIM_ENTRY(HOLE), PRIM_ENTRY(HOLE), PRIM_ENTRY(HOLE),
  PRIM_ENTRY(HOLE), PRIM_ENTRY(HOLE), PRIM_ENTRY(HOLE),

  HOLE_32, HOLE_32, HOLE_32, HOLE_32, HOLE_32, HOLE_32, HOLE_32,
  HOLE_32, HOLE_32, HOLE_32, HOLE_32, HOLE_32, HOLE_32, HOLE_32, HOLE_32,
  HOLE_32, HOLE_32, HOLE_32, HOLE_32, HOLE_32, HOLE_32, HOLE_32, HOLE_32,
};

#undef PRIM_ENTRY


/*
 *	static long executePrimitive_XXX(primitive, numArgs, methodOOP)
 *	long executePrimitiveOperation(primitive, numArgs, methodOOP)
 *
 * Description
 *
 *	These routines provide the definitions of the primitive methods in the
 *	GNU Smalltalk system.  They normally remove the arguments to the
 *	primitive methods from the stack, but if the primitive fails, the
 *	arguments are put back onto the stack and the routine returns true
 *	(-1 for the JIT), indicating failure to invoke the primitive.
 *
 * Inputs
 *
 *	primitive: 
 *		A C int that indicates the number of the primitive to invoke.
 *		Must be > 0.
 *	numArgs: 
 *		The number of arguments that the primitive has.
 *	methodOOP: 
 *		The OOP for the currently executing method.  This allows
 *		primitives to poke around in the method itself, to get at
 *		pieces that they need.  Normally, this is only used by the C
 *		callout routine to get at the compiled-in descriptor for the
 *		called C function.
 *
 * Outputs
 *
 *	False if the execution of the primitive operation succeeded, true if it
 *	failed for some reason.
 */
long executePrimitiveOperation(primitive, numArgs, methodOOP)
     int	  primitive;
     volatile int numArgs;
     OOP	  methodOOP;
{
  return primitiveTable[primitive].primFunc (primitive, numArgs, methodOOP);
}

long getPrimitiveAttributes(primitive)
     int	  primitive;
{
  return primitiveTable[primitive].attrFunc (primitive);
}

PrimitiveFunc getPrimitiveAddress(primitive)
     int	  primitive;
{
  return primitiveTable[primitive].primFunc;
}

