/******************************** -*- C -*- ****************************
 *
 *	Translator to native code.
 *
 *	$Revision: 1.95.1$
 *	$Date: 2000/12/27 10:45:49$
 *	$Author: pb$
 *
 ***********************************************************************/

/***********************************************************************
 *
 * Copyright 1988-92, 1994-95, 1999, 2000 Free Software Foundation, Inc.
 * Written by Paolo Bonzini.
 *
 * This file is part of GNU Smalltalk.
 *
 * GNU Smalltalk is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2, or (at your option) any later 
 * version.
 * 
 * GNU Smalltalk is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * GNU Smalltalk; see the file COPYING.  If not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
 *
 ***********************************************************************/

#include "gst.h"

#ifdef USE_JIT_TRANSLATION
#include "alloc.h"
#include "byte.h"
#include "comp.h"
#include "xlat.h"
#include "internal.h"
#include "interp.h"
#include "memzero.h"
#include "oop.h"
#include "opt.h"
#include "sysdep.h"
#include "obstack.h"
#include "lightning.h"
#include <math.h>

#ifdef __GNUC__
#warning .---------------------------------------
#warning | do not worry if you get lots of
#warning | 'value computed is not used' warnings
#warning `---------------------------------------
#endif

/* This file implements GNU Smalltalk's just-in-time compiler to native code.
 * It is inspired by techniques shown in Ian Piumarta's PhD thesis "Delayed
 * code generation in a Smalltalk-80 compiler" (available online at
 * http://www-sor.inria.fr/~piumarta), with quite a few modifications:
 *  - we target a RISC architecture (GNU lightning) instead of the CISC 
 *    Motorola 68020 architecture used in the thesis.
 *  - we use inline caching (only discussed briefly in the thesis)
 *  - block implementation is radically different
 *  - we don't work directly on a parse tree.  Rather, we recreate a tree
 *    structure from bytecodes that constitute a statement.
 *  - a few parts that were left back in the thesis (doesNotUnderstand:,
 *    non local returns, etc.) are fully implemented
 *
 * The aim of the code in this file is to generate pretty good code, as fast
 * as possible, and without requiring too much memory and information
 * (Self's 64MB requirement comes to mind...).  Nothing less, nothing
 * more.  All bottlenecks and inefficiencies should be due to the generic
 * nature of GNU lightning's architecture and to interp.c, not to the compiler.
 */


/* These two small structures are used to store information on labels and
 * forward references.
 */
typedef struct LabelUseStruct {
  jit_insn 		*addr;	/* addr of client insn */
  struct LabelUseStruct *next;	/* next label use or 0 */
} LabelUse;

typedef struct LabelStruct {
  jit_insn 	*addr;		/* defined address of label or 0 (forward) */
  LabelUse 	*uses;		/* list of uses while forward */
} Label;

/* And these one simplifies the handling of special selectors */
typedef struct SpecialSelectorStruct {
  OOP		  *selectorOOP;
  int		  numArgs;
  int		  operation;
} SpecialSelector;

/* This structure represents an n-tree. Children of a node are connected by
 * a linked list. It is probably the most important for the operation of the
 * translator.
 *
 * The translator operates on blocks of bytecodes that constitute a
 * statement, and represents what the block does on a stack.  The stack, 
 * however, does not contain results of the execution, but rather CodeTrees
 * that store how the value in that stack slot was computed; these CodeTrees
 * are built by dcd_* functions.  When a statement boundary is found (i.e.
 * a result on the stack is discarded, a jump is encountered, or a jump
 * destination is reached), the CodeTrees that are currently on the stack
 * are walked (calling recursively the gen_* functions depending on the
 * contents of the 'operation' field), resulting in the generation of
 * native code.
 */
typedef struct CodeTreeStruct {
  struct CodeTreeStruct	*child, *next;
  int			operation;
  voidPtr		data;
  Label			*jumpDest;
  Byte			*bp;
} CodeTree, *CodeStackElement, **CodeStackPointer;

/* This structure represents a message send.  A sequence of InlineCache
 * objects is initialized (as the CodeTree is constructed) and positioned
 * in the inlineCachesObstack.  Inline caches have two roles:
 * a) avoiding data to be stored in the methodsTableObstack, therefore
 *    making it easier to guess the size of the produced native code,
 *    and in the future allowing one to compile a single method without
 *    discarding all the work already done.
 * b) improving execution speed by lowering the number of MethodDictionary
 *    lookups to be done.
 * c) a pointer to an InlineCache is also used for the 'data' field in
 *    message send CodeTrees.
 */
typedef struct InlineCacheStruct {
  OOP			selector;
  jit_insn		*cachedIP;
  jit_insn		*returnIP;
  Byte			numArgs;
  Byte			more;		/* a boolean actually */
  Byte			isSuper;	/* this too */
  char			pad;
} InlineCache;

/* This structure forms a list that remembers which message sends were
 * inlined directly into the instruction flow.  The list is walked
 * by emitDeferredSends after the last bytecode has been compiled,
 * and recovery code that performs real message sends is written.
 */
typedef struct DeferredSendStruct {
  CodeTree			*tree;
  Label				*trueDest;
  Label				*falseDest;
  Label				*address;
  int				reg0, reg1;
  OOP				oop;
  struct DeferredSendStruct	*next;
} DeferredSend;


/* This makes up a hash table of translations */

#define MAX_BYTES_PER_BYTECODE	(100)
#define HASH_TABLE_SIZE		(16384)
#define METHOD_HEADER_SIZE 	(sizeof(MethodEntry) - sizeof(jit_insn))

/* Here is where the dynamically compiled stuff goes */
static MethodEntry 		*methodsTable[HASH_TABLE_SIZE],
				*released, *discarded;

/* Current status of the translator at the method level */
static MethodEntry		*current;
static struct obstack		auxDataObstack;
static InlineCache		*currInlineCache;
static DeferredSend		*deferredHead;
static Label			**labels, **thisLabel;
static Byte			*byteCodes;
static char			*intTab;
static OOP			*literals;
static OOP			methodClass;
static CodeStackElement		tStack[MAX_DEPTH];
static CodeStackPointer		tSP;

/* Current status of the code generator */
static mst_Boolean		selfCached, recVarCached;
static int			spDelta, selfClassCheck, stackCached;

/* These are pieces of native code that are used by the run-time. */
static jit_insn			*doSendCode, *doSuperCode, *nonBooleanCode;
voidPtr				(*runNativeCode)(), (*returnFromNativeCode)();

/* These OOPs are only needed to simplify things a bit */
static OOP minusOneOOP = fromInt(-1);
static OOP zeroOOP = fromInt(0);
static OOP oneOOP = fromInt(1);
static OOP twoOOP = fromInt(2);


/* Kinds of functions used in function tables */
typedef void		(*EmitFunc)();
typedef mst_Boolean	(*DecodeFunc)();

/* Constants used in the reconstruction of the parse tree (operation field)
 *
 * .--------------.---------------. .--------------.-----------.--------------.
 * | bits 14-15   |   bits 12-13  |.|   bits 6-8   | bits 3-5  |  bits 0-2    |
 * |--------------|---------------|.|--------------|-----------|--------------|
 * | class check  |  class check  |.| jump, pop &  | operation | suboperation |
 * | SmallInteger |  BlockClosure |.| return flags |           |              |
 * '--------------'---------------' '--------------'-----------'--------------'
 *                                 \
 *                                  \__ 3 unused bits
 */

/* operations 				   value of tree->data		*/
#define TREE_OP			00070
#define TREE_SEND		00000	/* points to an InlineCache	*/
#define TREE_STORE		00010   /* see below			*/
#define TREE_PUSH		00020   /* see below			*/
#define TREE_ALT_PUSH		00030   /* see below			*/
#define TREE_SET_TOP		00040   /* see below			*/
#define TREE_NOP		00050   /* unused			*/

/* suboperations for TREE_SEND */
#define TREE_SUBOP		00007
#define TREE_NORMAL		00000
#define TREE_BINARY_INT		00001
#define TREE_BINARY_BOOL	00003	/* 2 skipped - reserved to LIT_CONST */
#define TREE_BLOCKCOPY		00004
#define TREE_CLASS		00005
#define TREE_NIL_CHECK		00006
#define TREE_POP_INTO_ARRAY	00007	/* tree->data = index		*/

/* stack suboperations 			   value of tree->data		*/
#define TREE_REC_VAR		00000	/* variable number		*/
#define TREE_TEMP		00001	/* variable number		*/
#define TREE_LIT_CONST		00002   /* literal to be pushed		*/
#define TREE_LIT_VAR		00003	/* An Association object	*/
#define TREE_DUP		00004	/* unused			*/
#define TREE_SELF		00005	/* unused			*/
#define TREE_OUTER_TEMP		00006	/* unused			*/
#define TREE_THIS_CONTEXT	00007	/* unused			*/

/* suboperations for TREE_NOP */
#define TREE_ALREADY_EMITTED	00000
#define TREE_TWO_EXTRAS		00001

/* extra operations */
#define TREE_EXTRA		00700
#define TREE_EXTRA_NONE		00000
#define TREE_EXTRA_JMP_TRUE	00100
#define TREE_EXTRA_JMP_FALSE	00200
#define TREE_EXTRA_JMP_ALWAYS	00300
#define TREE_EXTRA_RETURN	00400
#define TREE_EXTRA_METHOD_RET	00500
#define TREE_EXTRA_POP		00600

/* class check flags */
#define TREE_CLASS_CHECKS	0x0F000L
#define TREE_IS_BLOCK		0x01000L
#define TREE_IS_NOT_BLOCK	0x02000L
#define TREE_IS_INTEGER		0x04000L
#define TREE_IS_NOT_INTEGER	0x08000L

/* testing macros */
#define NOT_INTEGER(tree) ( (tree)->operation & TREE_IS_NOT_INTEGER)
#define NOT_BLOCK(tree)	  ( (tree)->operation & TREE_IS_NOT_BLOCK)
#define IS_INTEGER(tree)  ( (tree)->operation & TREE_IS_INTEGER)
#define IS_BLOCK(tree)	  ( (tree)->operation & TREE_IS_BLOCK)
#define IS_PUSH(tree)	  ( ((tree)->operation & TREE_OP) == TREE_PUSH)
#define IS_SEND(tree)	  ( ((tree)->operation & TREE_OP) == TREE_SEND)
#define IS_STORE(tree)	  ( ((tree)->operation & TREE_OP) == TREE_STORE)
#define IS_SET_TOP(tree)  ( ((tree)->operation & TREE_OP) == TREE_SET_TOP)
#define IS_LITERAL(tree)  ( ((tree)->operation & TREE_SUBOP) == TREE_LIT_CONST)



/* Strength reduction */
static inline int		analyzeFactor();
static inline void		analyzeDividend();

/* Label handling */
static inline Label 		*lblNew();
static inline jit_insn		*lblGet();
static inline void 		lblUse();
static inline mst_Boolean	lblDefine();

/* Inlining (deferred sends) */
static void			deferSend();
static inline Label		*lastDeferredSend();
static inline void		emitDeferredSends();
static inline void		finishDeferredSend();

/* Method hash table handling */
static inline void		newMethodEntry();
static inline jit_insn		*finishMethodEntry();

/* CodeTree handling */
static inline CodeTree		*pushTreeNode(), *pushTreeNodeOOP(),
				*popTreeNode(), *pushSendNode();

static inline void		emitCode(), emitCodeTree(), setTopNodeExtra();

/* Prolog generation & overall initialization */
static inline mst_Boolean	emitMethodProlog(), emitBlockProlog();

static inline void		emitInterruptCheck(), generateRunTimeCode(),
				translateMethod(), emitPrimitive();

/* Code generation functions */
static void			gen_send(), gen_binaryInt(),
				gen_popIntoArray(), gen_binaryBool(),
				gen_blockCopy(), gen_fetchClass(),
				gen_nilCheck(), gen_storeRecVar(),
				gen_storeTemp(), gen_storeLitVar(),
				gen_storeOuter(), gen_pushRecVar(),
				gen_pushTemp(), gen_pushLitConst(),
				gen_pushLitVar(), gen_dupTop(), gen_pushSelf(),
				gen_pushOuter(), gen_pushContext(),
				gen_topRecVar(), gen_topTemp(), gen_topSelf(),
				gen_topOuter(), gen_altRecVar(), gen_altTemp(),
				gen_altLitConst(), gen_altLitVar(),
				gen_getTop(), gen_altSelf(), gen_altOuter(),
				gen_altContext(), gen_topLitConst(),
				gen_topLitVar(), gen_topContext(),
				gen_nothing(), gen_twoExtras(), 
				gen_illegal();

/* CodeTree building functions */
static mst_Boolean		dcd_pushRecVar(), dcd_pushTemp(),
				dcd_pushLit(), dcd_pushVar(), dcd_stRecVar(),
				dcd_stTemp(), dcd_pushSelf(),
				dcd_pushSpecial(), dcd_retSelf(),
				dcd_retSpecial(), dcd_explicitRet(),
				dcd_retStackTop(), dcd_bigLiteralOp(),
				dcd_pushIdxVal(), dcd_bigInstanceOp(),
				dcd_storeStackTop(), dcd_popStoreTop(),
				dcd_sendShort(), dcd_sendLong(),
				dcd_supSendShort(), dcd_popStack(),
				dcd_dupStack(), dcd_pushContext(),
				dcd_outerTempOp(), dcd_nop(), dcd_topSelf(),
				dcd_topOne(), dcd_topIndexedVal(),
				dcd_endExecution(), dcd_shJmp(),
				dcd_shJmpFalse(), dcd_longJmp(),
				dcd_popJmpTrue(), dcd_popJmpFalse(),
				dcd_sendSpecial(), dcd_sendPacked(),
				dcd_illegal();


/* Function table for the code generator */
static EmitFunc emitOperationFuncs[96] = {
  gen_send,	      gen_binaryInt,	  gen_illegal,	      gen_binaryBool,
  gen_blockCopy,      gen_fetchClass,     gen_nilCheck,	      gen_popIntoArray,

  gen_storeRecVar,    gen_storeTemp,	  gen_illegal,	      gen_storeLitVar,
  gen_illegal,	      gen_illegal,	  gen_storeOuter,     gen_illegal,

  gen_pushRecVar,     gen_pushTemp,	  gen_pushLitConst,   gen_pushLitVar,
  gen_dupTop,	      gen_pushSelf,	  gen_pushOuter,      gen_pushContext,

  gen_altRecVar,      gen_altTemp,	  gen_altLitConst,    gen_altLitVar,
  gen_getTop,	      gen_altSelf,	  gen_altOuter,       gen_altContext,

  gen_topRecVar,      gen_topTemp,	  gen_topLitConst,    gen_topLitVar,
  gen_illegal,	      gen_topSelf,	  gen_topOuter,       gen_topContext,

  gen_nothing,	      gen_twoExtras,	  gen_illegal,	      gen_illegal,
  gen_illegal,	      gen_illegal,	  gen_illegal,	      gen_illegal
};

/* Functions used in the reconstruction of the parse true */
static DecodeFunc decodeBytecodeFuncs[256] = {
  dcd_pushRecVar,   dcd_pushRecVar,	dcd_pushRecVar,	  dcd_pushRecVar,	/* 0 */
  dcd_pushRecVar,   dcd_pushRecVar,	dcd_pushRecVar,	  dcd_pushRecVar,	/* 4 */
  dcd_pushRecVar,   dcd_pushRecVar,	dcd_pushRecVar,	  dcd_pushRecVar,	/* 8 */
  dcd_pushRecVar,   dcd_pushRecVar,	dcd_pushRecVar,	  dcd_pushRecVar,	/* 12 */
  dcd_pushTemp,     dcd_pushTemp,	dcd_pushTemp,	  dcd_pushTemp,		/* 16 */
  dcd_pushTemp,     dcd_pushTemp,	dcd_pushTemp,	  dcd_pushTemp,		/* 20 */
  dcd_pushTemp,     dcd_pushTemp,	dcd_pushTemp,	  dcd_pushTemp,		/* 24 */
  dcd_pushTemp,     dcd_pushTemp,	dcd_pushTemp,	  dcd_pushTemp,		/* 28 */
  dcd_pushLit,      dcd_pushLit,	dcd_pushLit,	  dcd_pushLit,		/* 32 */
  dcd_pushLit,      dcd_pushLit,	dcd_pushLit,	  dcd_pushLit,		/* 36 */
  dcd_pushLit,      dcd_pushLit,	dcd_pushLit,	  dcd_pushLit,		/* 40 */
  dcd_pushLit,      dcd_pushLit,	dcd_pushLit,	  dcd_pushLit,		/* 44 */
  dcd_pushLit,      dcd_pushLit,	dcd_pushLit,	  dcd_pushLit,		/* 48 */
  dcd_pushLit,      dcd_pushLit,	dcd_pushLit,	  dcd_pushLit,		/* 52 */
  dcd_pushLit,      dcd_pushLit,	dcd_pushLit,	  dcd_pushLit,		/* 56 */
  dcd_pushLit,      dcd_pushLit,	dcd_pushLit,	  dcd_pushLit,		/* 60 */
  dcd_pushVar,      dcd_pushVar,	dcd_pushVar,	  dcd_pushVar,		/* 64 */
  dcd_pushVar,      dcd_pushVar,	dcd_pushVar,	  dcd_pushVar,		/* 68 */
  dcd_pushVar,      dcd_pushVar,	dcd_pushVar,	  dcd_pushVar,		/* 72 */
  dcd_pushVar,      dcd_pushVar,	dcd_pushVar,	  dcd_pushVar,		/* 76 */
  dcd_pushVar,      dcd_pushVar,	dcd_pushVar,	  dcd_pushVar,		/* 80 */
  dcd_pushVar,      dcd_pushVar,	dcd_pushVar,	  dcd_pushVar,		/* 84 */
  dcd_pushVar,      dcd_pushVar,	dcd_pushVar,	  dcd_pushVar,		/* 88 */
  dcd_pushVar,      dcd_pushVar,	dcd_pushVar,	  dcd_pushVar,		/* 92 */
  dcd_stRecVar,     dcd_stRecVar,	dcd_stRecVar,	  dcd_stRecVar,		/* 96 */
  dcd_stRecVar,     dcd_stRecVar,	dcd_stRecVar,	  dcd_stRecVar,		/* 100 */
  dcd_stTemp,	    dcd_stTemp,		dcd_stTemp,	  dcd_stTemp,		/* 104 */
  dcd_stTemp,	    dcd_stTemp,		dcd_stTemp,	  dcd_stTemp,		/* 108 */
  dcd_pushSelf,     dcd_pushSpecial,	dcd_pushSpecial,  dcd_pushSpecial,	/* 112 */
  dcd_pushSpecial,  dcd_pushSpecial,	dcd_pushSpecial,  dcd_pushSpecial,	/* 116 */
  dcd_retSelf,      dcd_retSpecial,	dcd_retSpecial,	  dcd_retSpecial,	/* 120 */
  dcd_explicitRet,  dcd_retStackTop,	dcd_bigLiteralOp, dcd_illegal,		/* 124 */
  dcd_pushIdxVal,   dcd_storeStackTop,	dcd_popStoreTop,  dcd_sendShort,	/* 128 */
  dcd_sendLong,     dcd_supSendShort,	dcd_bigInstanceOp,dcd_popStack,	        /* 132 */
  dcd_dupStack,     dcd_pushContext,	dcd_outerTempOp,  dcd_nop,		/* 136 */
  dcd_topSelf,      dcd_topOne,		dcd_topIndexedVal,dcd_endExecution,	/* 140 */
  dcd_shJmp,	    dcd_shJmp,		dcd_shJmp,	  dcd_shJmp,		/* 144 */
  dcd_shJmp,	    dcd_shJmp,		dcd_shJmp,	  dcd_shJmp,		/* 148 */
  dcd_shJmpFalse,   dcd_shJmpFalse,	dcd_shJmpFalse,   dcd_shJmpFalse,	/* 152 */
  dcd_shJmpFalse,   dcd_shJmpFalse,	dcd_shJmpFalse,   dcd_shJmpFalse,	/* 156 */
  dcd_longJmp,      dcd_longJmp,	dcd_longJmp,	  dcd_longJmp,		/* 160 */
  dcd_longJmp,      dcd_longJmp,	dcd_longJmp,	  dcd_longJmp,		/* 164 */
  dcd_popJmpTrue,   dcd_popJmpTrue,	dcd_popJmpTrue,	  dcd_popJmpTrue,	/* 168 */
  dcd_popJmpFalse,  dcd_popJmpFalse,	dcd_popJmpFalse,  dcd_popJmpFalse,	/* 172 */
  dcd_sendSpecial,  dcd_sendSpecial,	dcd_sendSpecial,  dcd_sendSpecial,	/* 176 */
  dcd_sendSpecial,  dcd_sendSpecial,	dcd_sendSpecial,  dcd_sendSpecial,	/* 180 */
  dcd_sendSpecial,  dcd_sendSpecial,	dcd_sendSpecial,  dcd_sendSpecial,	/* 184 */
  dcd_sendSpecial,  dcd_sendSpecial,	dcd_sendSpecial,  dcd_sendSpecial,	/* 188 */
  dcd_sendSpecial,  dcd_sendSpecial,	dcd_sendSpecial,  dcd_sendSpecial,	/* 192 */
  dcd_sendSpecial,  dcd_sendSpecial,	dcd_sendSpecial,  dcd_sendSpecial,	/* 196 */
  dcd_sendSpecial,  dcd_sendSpecial,	dcd_sendSpecial,  dcd_sendSpecial,	/* 200 */
  dcd_sendSpecial,  dcd_sendSpecial,	dcd_sendSpecial,  dcd_sendSpecial,	/* 204 */
  dcd_sendPacked,   dcd_sendPacked,	dcd_sendPacked,	  dcd_sendPacked,	/* 208 */
  dcd_sendPacked,   dcd_sendPacked,	dcd_sendPacked,	  dcd_sendPacked,	/* 212 */
  dcd_sendPacked,   dcd_sendPacked,	dcd_sendPacked,	  dcd_sendPacked,	/* 216 */
  dcd_sendPacked,   dcd_sendPacked,	dcd_sendPacked,	  dcd_sendPacked,	/* 220 */
  dcd_sendPacked,   dcd_sendPacked,	dcd_sendPacked,	  dcd_sendPacked,	/* 224 */
  dcd_sendPacked,   dcd_sendPacked,	dcd_sendPacked,	  dcd_sendPacked,	/* 228 */
  dcd_sendPacked,   dcd_sendPacked,	dcd_sendPacked,	  dcd_sendPacked,	/* 232 */
  dcd_sendPacked,   dcd_sendPacked,	dcd_sendPacked,	  dcd_sendPacked,	/* 236 */
  dcd_sendPacked,   dcd_sendPacked,	dcd_sendPacked,	  dcd_sendPacked,	/* 240 */
  dcd_sendPacked,   dcd_sendPacked,	dcd_sendPacked,	  dcd_sendPacked,	/* 244 */
  dcd_sendPacked,   dcd_sendPacked,	dcd_sendPacked,	  dcd_sendPacked,	/* 248 */
  dcd_sendPacked,   dcd_sendPacked,	dcd_sendPacked,	  dcd_sendPacked	/* 252 */
};

static SpecialSelector specialSendBytecodes[32] = {
  { &plusSymbol,		1,  TREE_SEND | TREE_BINARY_INT },
  { &minusSymbol,		1,  TREE_SEND | TREE_BINARY_INT },
  { &lessThanSymbol,		1,  TREE_SEND | TREE_BINARY_BOOL },
  { &greaterThanSymbol,		1,  TREE_SEND | TREE_BINARY_BOOL },
  { &lessEqualSymbol,		1,  TREE_SEND | TREE_BINARY_BOOL },
  { &greaterEqualSymbol,	1,  TREE_SEND | TREE_BINARY_BOOL },
  { &equalSymbol,		1,  TREE_SEND | TREE_BINARY_BOOL },
  { &notEqualSymbol,		1,  TREE_SEND | TREE_BINARY_BOOL },
  { &timesSymbol,		1,  TREE_SEND | TREE_BINARY_INT },
  { &divideSymbol,		1,  TREE_SEND | TREE_NORMAL },
  { &remainderSymbol,		1,  TREE_SEND | TREE_NORMAL /* TREE_BINARY_INT */ },
  { &atSignSymbol,		1,  TREE_SEND | TREE_NORMAL },
  { &bitShiftColonSymbol,	1,  TREE_SEND | TREE_NORMAL /* TREE_BINARY_INT */ },
  { &integerDivideSymbol,	1,  TREE_SEND | TREE_BINARY_INT },
  { &bitAndColonSymbol,		1,  TREE_SEND | TREE_BINARY_INT },
  { &bitOrColonSymbol,		1,  TREE_SEND | TREE_BINARY_INT },
  { &atColonSymbol,		1,  TREE_SEND | TREE_NORMAL },
  { &atColonPutColonSymbol,	2,  TREE_SEND | TREE_NORMAL },
  { &sizeSymbol,		0,  TREE_SEND | TREE_NORMAL },
  { &nextSymbol,		0,  TREE_SEND | TREE_NORMAL },
  { &nextPutColonSymbol,	1,  TREE_SEND | TREE_NORMAL },
  { &atEndSymbol,		0,  TREE_SEND | TREE_NORMAL },
  { &sameObjectSymbol,		1,  TREE_SEND | TREE_BINARY_BOOL },
  { &classSymbol,		0,  TREE_SEND | TREE_CLASS  },
  { &blockCopyColonSymbol,	1,  TREE_SEND | TREE_BLOCKCOPY },
  { &valueSymbol,		0,  TREE_SEND | TREE_NORMAL },
  { &valueColonSymbol,		1,  TREE_SEND | TREE_NORMAL },
  { &doColonSymbol,		1,  TREE_SEND | TREE_NORMAL },
  { &newSymbol,			0,  TREE_SEND | TREE_NORMAL },
  { &newColonSymbol,		1,  TREE_SEND | TREE_NORMAL },
  { &isNilSymbol,		0,  TREE_SEND | TREE_NIL_CHECK },
  { &notNilSymbol,		0,  TREE_SEND | TREE_NIL_CHECK }
};


/* Functions for managing the translated methods' hash table */

void
newMethodEntry(methodOOP, receiverClass, size)
     OOP methodOOP;
     OOP receiverClass;
     int size;
{
  current = (MethodEntry *) xmalloc(MAX_BYTES_PER_BYTECODE * (size + 2));
  current->methodOOP = methodOOP;
  current->receiverClass = receiverClass;
  current->inlineCaches = NULL;
  methodOOP->flags |= F_XLAT;

  obstack_init (&auxDataObstack);
  jit_set_ip (current->nativeCode);

  /* We need to compile a dummy prolog, which we'll overwrite, to make
   * GNU lightning's status consistent with that when the trampolineCode
   * was written. */
  jit_prolog(1);
  jit_set_ip (current->nativeCode);
}

jit_insn *
finishMethodEntry()
{
  unsigned int	hashEntry;
  char		*codePtr;

  /* Shrink the method, and store it into the hash table */
  codePtr = (char *) jit_get_label();
  jit_flush_code(current->nativeCode, codePtr);

  current = (MethodEntry *) xrealloc(current, codePtr - ((char *) current));

  hashEntry = oopIndex(current->methodOOP) % HASH_TABLE_SIZE;
  current->next = methodsTable[hashEntry];
  methodsTable[hashEntry] = current;

  obstack_free (&auxDataObstack, NULL);
  return (jit_insn *) (current->nativeCode);
}


/* Functions for managing the CodeTree */

CodeTree *
pushTreeNode(bp, firstChild, operation, data)
     Byte	*bp;
     CodeTree	*firstChild;
     int	operation;
     voidPtr	data;
{
  CodeTree *node = obstack_alloc(&auxDataObstack, sizeof(CodeTree));
  
  node->child = firstChild;
  node->next = NULL;
  node->operation = operation;
  node->data = data;
  node->bp = bp;
  *tSP++ = node;
  return (node);
}

CodeTree *
pushTreeNodeOOP(bp, firstChild, operation, literal)
     Byte	*bp;
     CodeTree	*firstChild;
     int	operation;
     OOP	literal;
{
  int classCheck;
  if (isInt(literal)) {
    classCheck = TREE_IS_NOT_BLOCK | TREE_IS_INTEGER;
  } else if (oopClass(literal) == blockClosureClass) {
    classCheck = TREE_IS_BLOCK | TREE_IS_NOT_INTEGER;
  } else {
    classCheck = TREE_IS_NOT_BLOCK | TREE_IS_NOT_INTEGER;
  }

  return pushTreeNode(bp, firstChild, operation | classCheck, literal);
}

CodeTree *
popTreeNode(linkedChild)
     CodeTree	*linkedChild;
{
  if (tSP <= tStack) {
    /* Stack underflow (note that it can be legal in a few cases, such as
     * for return stack top bytecodes) */
    return (NULL);
  } else {
    CodeTree *node = *--tSP;
    node->next = linkedChild;
    return (node);
  }
}

void
setTopNodeExtra(extra, jumpOffset)
     int	extra;
     int	jumpOffset;
{
  CodeTree *node;
  if (tSP <= tStack) {
    /* Stack is currently empty -- generate the code directly */
    if (extra != TREE_EXTRA_JMP_ALWAYS) {
      errorf("Stack underflow in JIT compilation");
      printObject(current->receiverClass);
      if (methodClass != current->receiverClass) {
        printf("(");
        printObject(methodClass);
        printf(")");
      }
      printf(">>");
      printObject(getMethodInfo(current->methodOOP)->selector);
      debug();
      exit(1);
    }
    node = alloca(sizeof(CodeTree));
  
    node->child = node->next = NULL;
    node->operation = TREE_NOP | TREE_ALREADY_EMITTED | extra;
    node->jumpDest = thisLabel[jumpOffset];
    emitCodeTree(node);
    return;
  }

  node = tSP[-1];
  if (node->operation & TREE_EXTRA) {
    /* More than one extra operation -- add a fake node */
    node = obstack_alloc(&auxDataObstack, sizeof(CodeTree));
  
    node->child = NULL;
    node->next = *tSP;
    node->operation = TREE_NOP | TREE_TWO_EXTRAS;
    tSP[-1] = node;
  }

  node->operation |= extra;
  node->jumpDest = thisLabel[jumpOffset];
}

CodeTree *
pushSendNode(bp, selector, numArgs, super, operation)
     Byte	 *bp;
     OOP	 selector;
     int	 numArgs;
     mst_Boolean super;
     int	 operation;
{
  CodeTree	*args, *node;

  if (*intTab == TOP_IS_INTEGER) {
    operation |= TREE_IS_NOT_BLOCK | TREE_IS_INTEGER;
  } else if (*intTab == TOP_IS_NOT_INTEGER) {
    operation |= TREE_IS_NOT_INTEGER;
  }

  currInlineCache->numArgs  = numArgs;
  currInlineCache->selector = selector;
  currInlineCache->cachedIP = super ? doSuperCode : doSendCode;
  currInlineCache->isSuper  = super;
  currInlineCache->more	    = true;

  /* Remember that we must pop an extra node for the receiver! */
  for (args = popTreeNode(NULL); numArgs--; ) {
    args = popTreeNode(args);
  }
  node = pushTreeNode(bp, args, operation, (voidPtr) currInlineCache++);
  return (node);
}

void
emitCodeTree(tree)
     CodeTree *tree;
{
  register int operation;

  operation = tree->operation & (TREE_OP | TREE_SUBOP);
  emitOperationFuncs[operation] (tree);
}

void
emitCode()
{
  register CodeTree **pTree, *tree;

  for (pTree = tStack; pTree < tSP; pTree++) {
    tree = *pTree;
    emitCodeTree(tree);
  }

  recVarCached = false;
  stackCached = -1;
  selfCached = false;
}


/* A couple of commodities for strength reduction */

int analyzeFactor(x)
     int x;
{
  int a;
  int b, c;

  a = x & (x-1);    /* clear lowest bit */
  a &= a-1;         /* again */

  if (a) {	    /* more than two bits are set to 1 */
    return 0;       /* don't attempt strength reduction */
  }

  for (b = 0; (x & 1) == 0; b++, x >>= 1);
  if (x == 1) {
    return b;	    /* a single bit was set */
  }

  for (c = b + 1; (x & 2) == 0; c++, x >>= 1);
  return b | (c << 8);
}

void
analyzeDividend(imm, shift, adjust, factor)
     int imm;
     int *shift;
     int *adjust;
     unsigned long *factor;
{
  int x, b, r;
  double f;

  *adjust = 0;

  /* compute floor(log2 imm) */
  for (r = 0, x = imm >> 1; x; r++, x >>= 1);

  if (!(imm & (imm - 1))) {
    /* x is a power of two */
    *shift = r;
    *factor = 0;
    return;
  }

  r += 31;
  f = ldexp( ((double) 1.0) / imm, r);
  b = (unsigned long) floor(f);

  if ((f - (double) b) < 0.5) {
    /* round f down to nearest integer, compute ((x + 1) * f) >> r */
    ++*adjust;
  } else {
    /* round f up to nearest integer, compute (x * f) >> r */
    ++b;
  }
  
  /* Try to shift less bits */
  while ((r >= 32) && ((b & 1) == 0)) {
    r--;
    b >>= 1;
  }

  *factor = b;
  *shift = r - 32;
}


/* Functions for managing Labels and forward references */

Label *
lblNew()
{
  Label *label = obstack_alloc(&auxDataObstack, sizeof(Label));
  label->addr = NULL;
  label->uses = NULL;
#ifdef DEBUG_LABELS
  printf("Defined reference at %p\n", label);
#endif
  return label;
}

jit_insn *
lblGet(label)
     Label *label;
{
  return label->addr ? label->addr : jit_forward();
}

void
lblUse(label, result)
     Label *label;
     jit_insn *result;
{
  if (!label->addr) {
    /* forward reference */
    LabelUse *use = obstack_alloc(&auxDataObstack, sizeof(LabelUse));
#ifdef DEBUG_LABELS
    printf("Forward reference at %p to %p\n", result, label);
#endif
    use->addr	  = result;
    use->next	  = label->uses;
    label->uses	  = use;
  } else {
#ifdef DEBUG_LABELS
    printf("Backward reference at %p to %p (%p)\n", result, label->addr, label);
#endif
  }
}

mst_Boolean
lblDefine(label)
     Label *label;
{
  LabelUse *use = label->uses;
  mst_Boolean used;

  jit_align(2);
  label->addr = jit_get_label();
  used = (use != NULL);

#ifdef DEBUG_LABELS
  printf("Defined label at %p (%p)\n", label->addr, label);
#endif
  while (use) {
    LabelUse *next= use->next;
#ifdef DEBUG_LABELS
    printf("Resolving forward reference at %p\n", use->addr);
#endif
    jit_patch(use->addr);
    use = next;
  }

  return (used);
}

void
finishDeferredSend()
{
  if (!deferredHead->trueDest) {
    deferredHead->trueDest = lblNew();
    lblDefine(deferredHead->trueDest);
    if (!deferredHead->falseDest) {
      deferredHead->falseDest = deferredHead->trueDest;
    }

  } else if (!deferredHead->falseDest) {
    deferredHead->falseDest = lblNew();
    lblDefine(deferredHead->falseDest);
  }
}

Label *
lastDeferredSend()
{
  return deferredHead->address;
}

void
deferSend(tree, isBool, address, reg0, reg1, oop)
     CodeTree		*tree;
     mst_Boolean	isBool;
     jit_insn		*address;
     int		reg0, reg1;
     OOP		oop;
{
  DeferredSend	*ds = obstack_alloc(&auxDataObstack, sizeof(DeferredSend));

  if (isBool) {
    switch (tree->operation & TREE_EXTRA) {
      case TREE_EXTRA_NONE:
      case TREE_EXTRA_POP:
      case TREE_EXTRA_RETURN:
      case TREE_EXTRA_METHOD_RET:
      case TREE_EXTRA_JMP_ALWAYS:
        isBool = false;
    }
  }

  ds->next = deferredHead;
  ds->tree = tree;
  ds->reg0 = reg0;
  ds->reg1 = reg1;
  ds->oop = oop;
  ds->address = lblNew();

  if (address) {
    lblUse(ds->address, address);
  }

  if (isBool) {
    if ((tree->operation & TREE_EXTRA) == TREE_EXTRA_JMP_TRUE) {
      ds->trueDest = tree->jumpDest;
      ds->falseDest = NULL;
    } else {
      ds->falseDest = tree->jumpDest;
      ds->trueDest = NULL;
    }
  } else {
    ds->trueDest = ds->falseDest = NULL;
  }

  deferredHead = ds;
}

/* Register usage:
 *   R0		scratch
 *   R1		cached address of 1st instance variable
 *   R2		scratch
 *   V0		stack top
 *   V1		cache address of 1st temporary or an outer context
 *		(also) pointer to the InlineCache upon a send
 *   V2		stack pointer
 */


/* Common pieces of code for generating stack operations */

/* Save the old stack top if it was cached in V0 */
#define BEFORE_PUSH do {						\
  spDelta += SIZEOF_LONG;						\
  if (spDelta > 0) {							\
    jit_stxi_p(spDelta, JIT_V2, JIT_V0);				\
  }									\
} while(0)

/* Generate code to evaluate the value to be replaced.  Generate
 * a `pop' by decrementing V2 unless the stack top is cached in V0 -- in
 * this case we can simply overwrite it.
 */
#define BEFORE_SET_TOP do {						\
  if (tree->child) {							\
    emitCodeTree(tree->child);						\
  }									\
  if (spDelta < 0) {							\
    jit_subi_p(JIT_V2, JIT_V2, SIZEOF_LONG);	/* pop stack top */	\
    spDelta += SIZEOF_LONG;						\
  }									\
} while(0)

/* Generate code to evaluate the value to be stored, and have it loaded
 * in V0. */
#define BEFORE_STORE do {						\
  emitCodeTree(tree->child);						\
  if (spDelta < 0) {							\
    jit_ldr_p(JIT_V0, JIT_V2);						\
    jit_subi_p(JIT_V2, JIT_V2, SIZEOF_LONG);	/* pop stack top */	\
    spDelta += SIZEOF_LONG;						\
  }									\
} while(0)


/* Common pieces of code for generating & caching addresses */

#define STACK_INDEX(bp)	   ((long) ((bp)[1] & ~locationMask) )

#define TEMP_OFS(tree)	   (SIZEOF_LONG * ((long) ((tree)->data)) )
#define REC_VAR_OFS(tree)  jit_ptr_field(mst_Object, data[(long) ((tree)->data)])
#define STACK_OFS(tree)	   (jit_ptr_field(BlockContext, contextStack) + \
				SIZEOF_LONG * STACK_INDEX((tree)->bp))


/* Cache the address of the first instance variable in R1 */
#define CACHE_REC_VAR do {					\
  if (!recVarCached) {			/* in R1 */			\
    if (!selfCached) {			/* in V0 */			\
      jit_ldi_p(JIT_R1, &self);						\
      jit_ldxi_p(JIT_R1, JIT_R1, jit_ptr_field(OOP, object));		\
    } else {								\
      jit_ldxi_p(JIT_R1, JIT_V0, jit_ptr_field(OOP, object));		\
    }									\
    recVarCached = true;						\
  }									\
} while(0)

/* Cache the address of the first temporary variable in V1 */
#define CACHE_TEMP do {							\
  if (stackCached != 0) {		/* in V1 */			\
    jit_ldi_p(JIT_V1, &temporaries);					\
    stackCached = 0;							\
  }									\
} while(0)

#define CACHE_NOTHING do {						\
  recVarCached = false;							\
  stackCached = -1;							\
  selfCached = false;							\
} while(0)

/* Cache into V1 the address of the outer context specified by the CodeTree.
 * If the address of another outer context (whose depth is lower) is currently
 * cached, avoid walking the list of outer contexts from the start.  This
 * optimizes bytecode sequences such as
 *
 *		push outer variable, n = 1, index = 2
 *		store outer variable, n = 2, index = 0
 *
 * Remember that stackCached = 0 means `cache the address of the first
 * temporary' (the address of the Context's first *indexed* instance variable),
 * while stackCached > 0 means `cache the address of the n-th outer context'
 * (i.e. the address of the Context's first *fixed* instance variable).
 * Although confusing, this was done because the VM provides the address of
 * the first indexed instance variable for thisContext into the `temporaries'
 * variable.
 */
#define CACHE_OUTER_CONTEXT do {					   \
  int scopes;								   \
  scopes = tree->bp[2];							   \
  if (stackCached <= 0 || stackCached > scopes) {			   \
    jit_ldi_p(JIT_V1, &thisContextOOP);					   \
    jit_ldxi_p(JIT_V1, JIT_V1, jit_ptr_field(OOP, object));		   \
    stackCached = scopes;						   \
  } else {								   \
    scopes -= stackCached;						   \
    stackCached += scopes;						   \
  }									   \
  while (scopes--) {							   \
    jit_ldxi_p(JIT_V1, JIT_V1, jit_ptr_field(BlockContext, outerContext)); \
    jit_ldxi_p(JIT_V1, JIT_V1, jit_ptr_field(OOP, object));		   \
  }									   \
} while(0)


/* Miscellaneous pieces of code */

/* Push the children on the stack -- needed for sends */
#define PUSH_CHILDREN do {						\
  CodeTree *child;							\
									\
  /* Emit code to generate the receiver and the arguments */		\
  for(child = tree->child; child; child = child->next) {		\
    emitCodeTree(child);						\
  }									\
} while(0)

/* Remember that the stack top is *not* cached in V0, and import V2 (the
 * stack pointer) from the sp variable. */
#define IMPORT_SP do {							\
  jit_ldi_p(JIT_V2, &sp);						\
  spDelta = -SIZEOF_LONG;						\
} while(0)

/* Export V2 (the stack pointer) into the sp variable; the top of the
 * stack is assured to be in *sp, not in V0. */
#define EXPORT_SP do {							\
  if (spDelta >= 0) {							\
    spDelta += SIZEOF_LONG;						\
    jit_stxi_p(spDelta, JIT_V2, JIT_V0);				\
    jit_addi_p(JIT_V2, JIT_V2, spDelta);				\
    jit_sti_p(&sp, JIT_V2);						\
    spDelta = -SIZEOF_LONG;						\
  }									\
} while(0)

/* Export V2 (the stack pointer) into the sp variable; the top of the
 * stack is assured to be in *sp AND in V0. */
#define CACHE_STACK_TOP do {						\
  if (spDelta < 0) {							\
    jit_ldr_p(JIT_V0, JIT_V2);						\
  } else {								\
    EXPORT_SP;								\
  }									\
} while(0)

/* Export V2 (the stack pointer) into the sp variable, without
 * saving the old stack top if it was cached in V0.  */
#define POP_EXPORT_SP do {						\
  if (spDelta) {							\
    jit_addi_p(JIT_V2, JIT_V2, spDelta);				\
  }									\
  jit_sti_p(&sp, JIT_V2);						\
  jit_ldr_p(JIT_V0, JIT_V2);						\
  spDelta = -SIZEOF_LONG;						\
} while(0)

/* Do a conditional jump to tree->jumpDest if the top of the stack
 * is successOOP, or to nonBooleanCode if it is anything but failOOP. */
#define CONDITIONAL_JUMP(successOOP, failOOP) do {			\
  jit_insn *addr;							\
  									\
  /* Save the value of the top of the stack */				\
  if (spDelta < 0) {							\
    jit_ldr_p(JIT_R0, JIT_V2);						\
  } else {								\
    jit_movr_p(JIT_R0, JIT_V0);						\
  }									\
  POP_EXPORT_SP;							\
									\
  addr = lblGet(tree->jumpDest);					\
  addr = jit_beqi_p(addr, JIT_R0, successOOP);				\
  lblUse(tree->jumpDest, addr);						\
  jit_bnei_p(nonBooleanCode, JIT_R0, failOOP);				\
									\
  CACHE_NOTHING;							\
} while(0)

#define UNWIND_THROUGH(func) do {					\
  emitInterruptCheck(JIT_NOREG);					\
  CACHE_STACK_TOP;							\
  jit_calli(func);							\
  IMPORT_SP;								\
  jit_ldi_p(JIT_R0, &nativeIP);						\
  jit_str_p(JIT_V2, JIT_V0);						\
  jit_jmpr(JIT_R0);							\
} while(0)


/* Pieces of code for inlining */

/* Don't inline if doing a send to super */
#define DONT_INLINE_SUPER do {						\
  if(ic->isSuper) {							\
    gen_send(tree);							\
    return;								\
  }									\
} while(0)

/* Don't attempt to inline an arithmetic operation if one of its
 * argument is known not to be a SmallInteger.
 */
#define DONT_INLINE_NONINTEGER						\
  if (NOT_INTEGER(tree->child) || NOT_INTEGER(tree->child->next)) {	\
    gen_send(tree);							\
    return;								\
  }

/* Create a `true' or `false' oop if the value is required `as is'; else
 * compile a `jump if true' or `jump if false' native opcode.  This is
 * the equivalent of the `jump lookahead' option in the bytecode interpreter.
 */
#define INLINED_CONDITIONAL do {					\
  jit_insn *addr;							\
  									\
  switch (tree->operation & TREE_EXTRA) {				\
    case TREE_EXTRA_NONE:						\
    case TREE_EXTRA_POP:						\
    case TREE_EXTRA_RETURN:						\
    case TREE_EXTRA_METHOD_RET:						\
    case TREE_EXTRA_JMP_ALWAYS:						\
      FALSE_SET(JIT_R0);						\
      jit_lshi_i(JIT_R0, JIT_R0, LONG_SHIFT+1);				\
      jit_addi_p(JIT_V0, JIT_R0, trueOOP);				\
      break;								\
									\
    case TREE_EXTRA_JMP_TRUE:						\
    case TREE_EXTRA_JMP_FALSE:						\
      if (spDelta) {							\
	jit_addi_p(JIT_V2, JIT_V2, spDelta);				\
      }									\
      spDelta = -SIZEOF_LONG;						\
      addr = lblGet(tree->jumpDest);					\
      if ((tree->operation & TREE_EXTRA) == TREE_EXTRA_JMP_TRUE) { \
        TRUE_BRANCH(addr);						\
      } else {								\
        FALSE_BRANCH(addr);						\
      }									\
      lblUse(tree->jumpDest, addr);					\
									\
      /* Change the CodeTree's operation to TREE_ALREADY_EMITTED */	\
      tree->operation &= TREE_CLASS_CHECKS;				\
      tree->operation |= TREE_NOP | TREE_ALREADY_EMITTED;		\
      break;								\
  }									\
} while(0)


/* Generate code for the only argument, and get the argument in V0.
 * Think twice about it, it is the same as the code needed to compile
 * a store!
 */
#define GET_UNARY_ARG 		BEFORE_STORE

/* Load the two arguments of an inlined binary message, optimizing the
 * common case when the second one is a literal (a == 5, a + 2).
 * reg0 and reg1 will contain the registers in which the arguments have
 * been loaded.
 */
#define GET_BINARY_ARGS do {						\
  CodeTree *second = tree->child->next;					\
									\
  emitCodeTree(tree->child);						\
  oop = nil;								\
  reg0 = JIT_V0;							\
  reg1 = JIT_V1;							\
  if (IS_LITERAL(second)) {						\
    if (spDelta < 0) {							\
      jit_ldr_p(JIT_V0, JIT_V2);					\
    }									\
    reg1 = JIT_NOREG;							\
    oop = (OOP) second->data;						\
  } else if (IS_PUSH(second)) {						\
    if (spDelta < 0) {							\
      jit_ldr_p(JIT_V0, JIT_V2);					\
      jit_addi_p(JIT_V2, JIT_V2, spDelta);				\
      spDelta = 0;							\
    }									\
    /* Load the second operand into V1 */				\
    second->operation ^= TREE_PUSH ^ TREE_ALT_PUSH;			\
    emitCodeTree(second);						\
  } else {								\
    emitCodeTree(second);						\
    if (spDelta < 0) {							\
      /* We load the 2nd argument and then the 1st */			\
      jit_ldr_p(JIT_V1, JIT_V2);					\
      jit_ldxi_p(JIT_V0, JIT_V2, -SIZEOF_LONG);				\
    } else {								\
      /* We load the 1st argument; the 2nd is already in V0 */		\
      /* ### isn't this superfluous??? */				\
      jit_ldxi_p(JIT_V1, JIT_V2, spDelta);				\
      reg0 = JIT_V1;							\
      reg1 = JIT_V0;							\
    }									\
    /* "Pop" the 2nd argument */					\
    spDelta -= SIZEOF_LONG;						\
  }									\
  									\
  if (spDelta) {							\
    jit_addi_p(JIT_V2, JIT_V2, spDelta);				\
    spDelta = 0;							\
  }									\
  CACHE_NOTHING;							\
} while(0)

/* Jump out of the instruction flow (to a send whose compilation is
 * deferred to after we compiled the method bytecodes) if one or both
 * arguments are not SmallIntegers.
 */
#define ENSURE_INT_ARGS(isBool, overflow) do {				\
  jit_insn	*classCheck;						\
									\
  if (IS_INTEGER(tree->child) && IS_INTEGER(tree->child->next)) {	\
    if (isBool || IS_INTEGER(tree)) {					\
      /* No need to do class checks & deferred sends */			\
      overflow = nil;							\
      break;								\
    }									\
    classCheck = NULL;							\
  } else if (IS_INTEGER(tree->child)) {					\
    classCheck = jit_bmci_ul(jit_forward(), reg1, 1);			\
  } else if (IS_INTEGER(tree->child->next)) {				\
    classCheck = jit_bmci_ul(jit_forward(), reg0, 1);			\
  } else {								\
    jit_andr_ul(JIT_R2, JIT_V0, JIT_V1);				\
    classCheck = jit_bmci_ul(jit_forward(), JIT_R2, 1);			\
  }									\
									\
  deferSend(tree, isBool, classCheck, reg0, reg1, oop);			\
  overflow = lastDeferredSend();					\
} while(0)

/* These are used to simplify the inlining code, as they group the
 * `second operand is a literal' and `second operand is a register'
 * cases in a single statement. */
#define EXPAND_(what)		what
#define IMM_OR_REG(opcode, a)					 \
	((reg1 != JIT_NOREG) 					 \
		? EXPAND_(jit_##opcode##r_l(a, reg0, reg1))	 \
		: EXPAND_(jit_##opcode##i_l(a, reg0, (long) oop)))



/* Message sends */
void
gen_send(tree)
     CodeTree	*tree;
{
  InlineCache *ic = (InlineCache *) tree->data;

  PUSH_CHILDREN;
  jit_movi_p(JIT_V1, ic);
  EXPORT_SP;

  jit_movi_ul(JIT_V0, tree->bp - byteCodes);
  jit_ldxi_p(JIT_R1, JIT_V1, jit_field(InlineCache, cachedIP));
  jit_sti_ul(&ip, JIT_V0);
  if (ic->isSuper) {
    jit_movi_p(JIT_V0, methodClass);
  }
  jit_jmpr(JIT_R1);
  jit_align(2);

  ic->returnIP = jit_get_label();
  IMPORT_SP;
  CACHE_NOTHING;

  gen_nothing(tree);
}

void
gen_binaryInt(tree)
     CodeTree	*tree;
{
  InlineCache *ic = (InlineCache *) tree->data;
  Label *overflow;
  int reg0, reg1;
  OOP oop;
  long imm;
  jit_insn *addr;

  DONT_INLINE_SUPER;
  DONT_INLINE_NONINTEGER;
  GET_BINARY_ARGS;
  ENSURE_INT_ARGS(false, overflow);

  imm = (long) oop;

  /* Now generate the code for the inlined operation.  Don't touch reg0/reg1
   * until we are sure that no overflow happens! */
  switch(*tree->bp) {
    case plusSpecial:
      if (reg1 == JIT_NOREG) {
	imm--;				/* strip tag bit */
	if (imm == 0) {
	  if (reg0 != JIT_V0) {
	    jit_movr_l(JIT_V0, reg0);
	  }
	  break;
	}

	if (overflow) {
	  jit_movr_l(JIT_R0, reg0);
	  addr = lblGet(overflow);
	  addr = jit_boaddi_l(addr, JIT_R0, imm);
	  lblUse(overflow, addr);
	  jit_movr_l(JIT_V0, JIT_R0);
	} else {
	  jit_addi_l(JIT_V0, reg0, imm);
	}

      } else {
	jit_subi_l(JIT_R0, reg0, 1);	  /* remove the tag bit */
	if (overflow) {
	  addr = lblGet(overflow);
	  addr = jit_boaddr_l(addr, JIT_R0, reg1);
	  lblUse(overflow, addr);
	  jit_movr_l(JIT_V0, JIT_R0);
	} else {
	  jit_addr_l(JIT_V0, reg0, reg1);
	}
      }
      break;



    case minusSpecial:
      if (reg1 == JIT_NOREG) {
	imm--;				/* strip tag bit */
	if (imm == 0) {
	  if (reg0 != JIT_V0) {
	    jit_movr_l(JIT_V0, reg0);
	  }
	  break;
	}

	if (overflow) {
	  jit_movr_l(JIT_R0, reg0);
	  addr = lblGet(overflow);
	  addr = jit_bosubi_l(addr, JIT_R0, imm);
	  lblUse(overflow, addr);
	  jit_movr_l(JIT_V0, JIT_R0);
	} else {
	  jit_subi_l(JIT_V0, reg0, imm);
	}

      } else {
	if (overflow) {
	  jit_movr_l(JIT_R0, reg0);
	  addr = lblGet(overflow);
	  addr = jit_bosubr_l(addr, JIT_R0, reg1);
	  lblUse(overflow, addr);
	  jit_addi_l(JIT_V0, JIT_R0, 1);	  /* add back the tag bit */
	} else {
	  jit_subr_l(JIT_V0, reg0, reg1);
	  jit_addi_l(JIT_V0, JIT_V0, 1);	  /* add back the tag bit */
	}
      }
      break;



    case timesSpecial:
      if (reg1 == JIT_NOREG) {
	jit_insn *addr1, *addr2;
	int reduce;

	imm >>= 1;
	if (imm == 0) {
	  jit_movi_p(JIT_V0, fromInt(0));
	  break;
	} else if (imm == 1) {
	  if (reg0 != JIT_V0) jit_movr_p(JIT_V0, reg0);
	  break;
	} else if (imm == -1) {
	  if (overflow) {
	    addr = lblGet(overflow);
	    addr = jit_beqi_l(addr, reg0, fromInt(MIN_ST_INT));
	    lblUse(overflow, addr);
	  }
	  jit_rsbi_l(JIT_V0, reg0, 2);
	  break;
	}

	if (overflow) {
	  addr = lblGet(overflow);
	  if (imm < 0) {
	    addr1 = jit_blti_p(addr, reg0, fromInt(MIN_ST_INT / -imm));
	    addr2 = jit_bgti_p(addr, reg0, fromInt(MAX_ST_INT / -imm));
	  } else {
	    addr1 = jit_blti_p(addr, reg0, fromInt(MIN_ST_INT / imm));
	    addr2 = jit_bgti_p(addr, reg0, fromInt(MAX_ST_INT / imm));
	  }
	  lblUse(overflow, addr1);
	  lblUse(overflow, addr2);
	}

	/* Do some strength reduction... */
	reduce = analyzeFactor(imm);
	if (reduce == 0) {
	  jit_muli_l(JIT_V0, reg0, imm);
	} else if ((reduce & 0x00FF00) == 0) {
	  jit_lshi_l(JIT_V0, reg0, reduce);
	} else if (reduce & 255) {
	  jit_lshi_l(JIT_R0, reg0, reduce & 255);
	  jit_lshi_l(JIT_V0, reg0, reduce >> 8);
	  jit_addr_l(JIT_V0, JIT_V0, JIT_R0);
	} else {
	  jit_lshi_l(JIT_R0, reg0, reduce >> 8);
	  jit_addr_l(JIT_V0, reg0, JIT_R0);
	}

	/* remove the excess due to the tag bit:
	 *   ((x-1) / 2 * imm) * 2 + 1 = x * imm - imm + 1 = (x*imm) - (imm-1)
	 */
	jit_subi_l(JIT_V0, reg0, imm - 1);

      } else {
	jit_rshi_l(JIT_R1, reg0, 1);
	jit_rshi_l(JIT_R0, reg1, 1);
	jit_mulr_l(JIT_R2, JIT_R0, JIT_R1);
	if (overflow) {
	  jit_hmulr_l(JIT_R0, JIT_R0, JIT_R1);	/* compute high bits */
	
	  /* check for sensible bits of the result in R0, and in bits 30-31 of R2 */
	  jit_rshi_i(JIT_R1, JIT_R0, SIZEOF_LONG * 8 - 1);
	  addr = lblGet(overflow);
	  addr = jit_bner_l(addr, JIT_R0, JIT_R1);
	  lblUse(overflow, addr);

	  jit_xorr_i(JIT_R1, JIT_R0, JIT_R2);
	  addr = lblGet(overflow);
	  addr = jit_bmsi_l(addr, JIT_R1, 3 << (SIZEOF_LONG * 8 - 2));
	  lblUse(overflow, addr);
	}

	jit_addr_l(JIT_V0, JIT_R2, JIT_R2);
	jit_ori_l(JIT_V0, JIT_V0, 1);
      }
      break;



    case integerDivideSpecial:
      if (reg1 == JIT_NOREG) {
	int shift;
	mst_Boolean adjust;
	unsigned long factor;

        imm >>= 1;
	if (imm == 0) {
	  addr = lblGet(overflow);
	  addr = jit_jmpi(addr);
	  lblUse(overflow, addr);
	  break;
	} else if (imm == 1) {
	  if (reg0 != JIT_V0) jit_movr_p(JIT_V0, reg0);
	  break;
	} else if (imm == -1) {
	  if (overflow) {
	    addr = lblGet(overflow);
	    addr = jit_beqi_l(addr, reg0, fromInt(MIN_ST_INT));
	    lblUse(overflow, addr);
	  }
	  jit_rsbi_l(JIT_V0, reg0, 2);
	  break;
	}

	jit_rshi_l(reg0, reg0, 1);

	if (imm < 0) {
	  jit_negr_l(reg0, reg0);
	  imm = -imm;
	}

	/* Fix the sign of the result: reg0 = imm - self - 1 if reg0 < 0
	 * All these instructions are no-ops if reg0 > 0, because R0=R1=0
	 */
	jit_rshi_l(JIT_R0, reg0, 8 * SIZEOF_LONG - 1);
	jit_andi_l(JIT_R1, JIT_R0, imm - 1);	/* if reg0 < 0, reg0 is... */
	jit_subr_l(reg0, reg0, JIT_R1);		/* self - imm + 1 */
	jit_xorr_l(reg0, reg0, JIT_R0);		/* imm - self - 2 */
	jit_subr_l(reg0, reg0, JIT_R0);		/* imm - self - 1 */

	/* Do some strength reduction... */
	analyzeDividend(imm, &shift, &adjust, &factor);
	
	if (adjust) {
	  /* If adjust is false, we have to sum 1 here, and the carry
	   * after the multiplication. */
	  jit_movi_l(JIT_R1, 0);
	  jit_addci_l(reg0, reg0, 1);
	  jit_addxi_l(JIT_R1, JIT_R1, 0);
	}

	shift--;				   /* for the tag bit */
	if (factor) {
	  jit_hmuli_l(reg0, reg0, factor);
	}
	if (shift < 0) {
	  jit_lshi_l(reg0, reg0, -shift);
	} else if (shift > 0) {
	  jit_rshi_l(reg0, reg0, shift);
	}
	if (adjust) {
	  jit_subr_l(reg0, reg0, JIT_R1);
	}

	/* negate the result if the signs were different */
	jit_xorr_l(reg0, reg0, JIT_R0);
	jit_subr_l(reg0, reg0, JIT_R0);

	/* now add the tag bit */
	jit_ori_l(JIT_V0, reg0, 1);

      } else {
	if (overflow) {
	  addr = lblGet(overflow);
	  addr = jit_beqi_p(addr, reg1, fromInt(0));
	  lblUse(overflow, addr);
	}

	jit_rshi_l(reg1, reg1, 1);
	jit_rshi_l(reg0, reg0, 1);

	/* Make the divisor positive */
	jit_rshi_l(JIT_R0, reg1, 8 * SIZEOF_LONG - 1);
	jit_xorr_l(reg0, reg0, JIT_R0);
	jit_xorr_l(reg1, reg1, JIT_R0);
	jit_subr_l(reg0, reg0, JIT_R0);
	jit_subr_l(reg1, reg1, JIT_R0);

	/* Fix the result if signs differ: reg0 -= reg1-1 */
	jit_rshi_l(JIT_R1, reg0, 8 * SIZEOF_LONG - 1);
	jit_subi_l(JIT_R0, reg1, 1);
	jit_andr_l(JIT_R0, JIT_R0, JIT_R1);	/* if reg0 < 0, reg0 is... */
	jit_subr_l(reg0, reg0, JIT_R0);		/* self - imm + 1 */
	jit_xorr_l(reg0, reg0, JIT_R1);		/* imm - self - 2 */
	jit_subr_l(reg0, reg0, JIT_R1);		/* imm - self - 1 */

	/* divide, then negate the result if the signs were different */
	jit_divr_l(JIT_R0, reg0, reg1);
	jit_xorr_l(JIT_R0, JIT_R0, JIT_R1);
	jit_subr_l(JIT_R0, JIT_R0, JIT_R1);

	/* add the tag bit */
	jit_addr_l(JIT_V0, JIT_R0, JIT_R0);
	jit_ori_l(JIT_V0, JIT_V0, 1);
      }
      break;



    case remainderSpecial:
    case bitShiftColonSpecial:
	/* not yet */
	addr = lblGet(overflow);
	addr = jit_jmpi(addr);
	lblUse(overflow, addr);
	break;

    case bitAndColonSpecial:	IMM_OR_REG(and, JIT_V0); break;
    case bitOrColonSpecial:	IMM_OR_REG(or, JIT_V0);	 break;
  }

  EXPORT_SP;
  if (overflow) {
    finishDeferredSend();
  }
  gen_nothing(tree);
}

void
gen_binaryBool(tree)
     CodeTree	*tree;
{
  InlineCache *ic = (InlineCache *) tree->data;
  Label *deferredSend;
  int reg0, reg1;
  OOP oop;

  DONT_INLINE_SUPER;
  if (*tree->bp != sameObjectSpecial) {
    DONT_INLINE_NONINTEGER;
  }
  GET_BINARY_ARGS;
  if (*tree->bp != sameObjectSpecial) {
    ENSURE_INT_ARGS(true, deferredSend);
  } else {
    deferredSend = nil;
  }

#define TRUE_BRANCH(addr)						\
  switch(*tree->bp) {							\
    case lessThanSpecial:	addr = IMM_OR_REG(blt, addr); break;	\
    case greaterThanSpecial:	addr = IMM_OR_REG(bgt, addr); break;	\
    case lessEqualSpecial:	addr = IMM_OR_REG(ble, addr); break;	\
    case greaterEqualSpecial:	addr = IMM_OR_REG(bge, addr); break;	\
    case sameObjectSpecial:						\
    case equalSpecial:		addr = IMM_OR_REG(beq, addr); break;	\
    case notEqualSpecial:	addr = IMM_OR_REG(bne, addr); break;	\
  }

#define FALSE_BRANCH(addr)						\
  switch(*tree->bp) {							\
    case lessThanSpecial:	addr = IMM_OR_REG(bge, addr); break;	\
    case greaterThanSpecial:	addr = IMM_OR_REG(ble, addr); break;	\
    case lessEqualSpecial:	addr = IMM_OR_REG(bgt, addr); break;	\
    case greaterEqualSpecial:	addr = IMM_OR_REG(blt, addr); break;	\
    case sameObjectSpecial:						\
    case equalSpecial:		addr = IMM_OR_REG(bne, addr); break;	\
    case notEqualSpecial:	addr = IMM_OR_REG(beq, addr); break;	\
  }

#define FALSE_SET(reg)							\
  switch(*tree->bp) {							\
    case lessThanSpecial:	IMM_OR_REG(ge, reg); break;		\
    case greaterThanSpecial:	IMM_OR_REG(le, reg); break;		\
    case lessEqualSpecial:	IMM_OR_REG(gt, reg); break;		\
    case greaterEqualSpecial:	IMM_OR_REG(lt, reg); break;		\
    case sameObjectSpecial:						\
    case equalSpecial:		IMM_OR_REG(ne, reg); break;		\
    case notEqualSpecial:	IMM_OR_REG(eq, reg); break;		\
  }

  INLINED_CONDITIONAL;
#undef TRUE_BRANCH
#undef FALSE_BRANCH
#undef FALSE_SET

  EXPORT_SP;
  if (deferredSend) {
    finishDeferredSend();
  }
  gen_nothing(tree);
}

void
gen_blockCopy(tree)
     CodeTree	*tree;
{
  InlineCache	*ic = (InlineCache *) tree->data;
  PrimitiveFunc	pf = getPrimitiveAddress(80);
  
  DONT_INLINE_SUPER;
  if (IS_BLOCK(tree->child)) {
    tree->operation |= TREE_IS_BLOCK | TREE_IS_NOT_INTEGER;
    PUSH_CHILDREN;

    EXPORT_SP;
    jit_prepare  (3);
    jit_movi_p   (JIT_R0, nil);
    jit_movi_i   (JIT_R1, 1);
    jit_movi_i   (JIT_R2, 80);
    jit_pusharg_p(JIT_R0);
    jit_pusharg_i(JIT_R1);
    jit_pusharg_i(JIT_R2);
    jit_finish   (pf);
    IMPORT_SP;

    recVarCached = false;
    stackCached = -1;
    selfCached = false;
    gen_nothing(tree);
  } else {
    gen_send(tree);
  }
}

void
gen_fetchClass(tree)
     CodeTree	*tree;
{
  InlineCache *ic = (InlineCache *) tree->data;

  DONT_INLINE_SUPER;
  GET_UNARY_ARG;
  if (IS_INTEGER(tree->child)) {
    jit_movi_p  (JIT_V0, smallIntegerClass);
  } else if (NOT_INTEGER(tree->child)) {
    jit_ldxi_p  (JIT_V0, JIT_V0, jit_ptr_field(OOP, object));
    jit_ldxi_p  (JIT_V0, JIT_V0, jit_ptr_field(mst_Object, objClass));
  } else {
    jit_insn *jmp;
    jit_movi_p  (JIT_R0, smallIntegerClass);
jmp=jit_bmsi_ul (jit_forward(), JIT_V0, 1);
    jit_ldxi_p  (JIT_R0, JIT_V0, jit_ptr_field(OOP, object));
    jit_ldxi_p  (JIT_R0, JIT_R0, jit_ptr_field(mst_Object, objClass));
    jit_patch	(jmp);
    jit_movr_p  (JIT_V0, JIT_R0);
  }

  selfCached = false;
  gen_nothing(tree);
}

void
gen_nilCheck(tree)
     CodeTree	*tree;
{
  InlineCache *ic = (InlineCache *) tree->data;
  mst_Boolean compileIsNil = *tree->bp == isNilSpecial;
  
  DONT_INLINE_SUPER;
  GET_UNARY_ARG;

#define TRUE_BRANCH(addr)   addr = compileIsNil ? jit_beqi_p((addr), JIT_V0, nilOOP) \
					  	: jit_bnei_p((addr), JIT_V0, nilOOP)
#define FALSE_BRANCH(addr)  addr = compileIsNil ? jit_bnei_p((addr), JIT_V0, nilOOP) \
					  	: jit_beqi_p((addr), JIT_V0, nilOOP)
#define FALSE_SET(reg)		   compileIsNil ? jit_nei_p ((reg),  JIT_V0, nilOOP) \
						: jit_eqi_p ((reg),  JIT_V0, nilOOP)
  INLINED_CONDITIONAL;
#undef TRUE_BRANCH
#undef FALSE_BRANCH
#undef FALSE_SET

  gen_nothing(tree);
}

void
gen_popIntoArray(tree)
     CodeTree	*tree;
{
  mst_Boolean	useCachedR0;
  CodeTree	*array, *value;
  int		index;

  array = tree->child;
  value = array->next;
  index = (int) tree->data;
  useCachedR0 = (array->operation & (TREE_OP | TREE_SUBOP))
    == (TREE_SEND | TREE_POP_INTO_ARRAY);

  /* This code is the same as GET_BINARY_ARGS, but it forces the 
   * first parameter in V0 and the second in V1. This is because
   * the bytecode leaves the first parameter in the stack top
   */

  emitCodeTree(array);
  if (IS_PUSH(value)) {
    if (spDelta < 0) {
      jit_ldr_p(JIT_V0, JIT_V2);
      jit_addi_p(JIT_V2, JIT_V2, spDelta);
      spDelta = 0;
    }
    /* Load the value operand into V1 */
    value->operation ^= TREE_PUSH ^ TREE_ALT_PUSH;
    emitCodeTree(value);
  } else {
    emitCodeTree(value);
    if (spDelta < 0) {
      /* We load the 2nd argument and then the 1st */
      jit_ldr_p(JIT_V1, JIT_V2);
      jit_ldxi_p(JIT_V0, JIT_V2, -SIZEOF_LONG);
      jit_addi_p(JIT_V2, JIT_V2, spDelta);
    } else {
      /* The 2nd argument is already in V0, move it in V1 */
      /* ### isn't this superfluous??? */
      jit_movr_p(JIT_V1, JIT_V0);
      jit_ldxi_p(JIT_V0, JIT_V2, spDelta);
    }

    /* "Pop" the 2nd argument */
    spDelta -= SIZEOF_LONG;
    useCachedR0 = false;
  }

  if (spDelta) {
    jit_addi_p(JIT_V2, JIT_V2, spDelta);
    spDelta = 0;
  }

  if (!useCachedR0) {
    /* Dereference the OOP into R0 */
    jit_ldxi_p(JIT_R0, JIT_V0, jit_ptr_field(OOP, object));
  }

  jit_stxi_p(jit_ptr_field(mst_Object, data[index]), JIT_R0, JIT_V1);
  gen_nothing(tree);
}


/* Stores */
void
gen_storeRecVar(tree)
     CodeTree	*tree;
{
  BEFORE_STORE;
  CACHE_REC_VAR;

  jit_stxi_p(REC_VAR_OFS(tree), JIT_R1, JIT_V0);
  gen_nothing(tree);
}

void
gen_storeTemp(tree)
     CodeTree	*tree;
{
  BEFORE_STORE;
  CACHE_TEMP;

  jit_stxi_p(TEMP_OFS(tree), JIT_V1, JIT_V0);
  gen_nothing(tree);
}

void
gen_storeLitVar(tree)
     CodeTree	*tree;
{
  char *assocOOP = ((char *) tree->data) + jit_ptr_field(OOP, object);
  BEFORE_STORE;

  jit_ldi_p(JIT_R0, assocOOP);
  jit_stxi_p(jit_ptr_field(Association, value), JIT_R0, JIT_V0);
  gen_nothing(tree);
}

void
gen_storeOuter(tree)
     CodeTree	*tree;
{
  BEFORE_STORE;
  CACHE_OUTER_CONTEXT;

  jit_stxi_p(STACK_OFS(tree), JIT_V1, JIT_V0);
  gen_nothing(tree);
}

/* Pushes */
void
gen_pushRecVar(tree)
     CodeTree	*tree;
{
  BEFORE_PUSH;
  CACHE_REC_VAR;

  jit_ldxi_p(JIT_V0, JIT_R1, REC_VAR_OFS(tree));
  selfCached = false;
  gen_nothing(tree);
}

void
gen_pushTemp(tree)
     CodeTree	*tree;
{
  BEFORE_PUSH;
  CACHE_TEMP;

  jit_ldxi_p(JIT_V0, JIT_V1, TEMP_OFS(tree));
  selfCached = false;
  gen_nothing(tree);
}

void
gen_pushLitConst(tree)
     CodeTree	*tree;
{
  BEFORE_PUSH;

  jit_movi_p(JIT_V0, tree->data);
  selfCached = false;
  gen_nothing(tree);
}

void
gen_pushLitVar(tree)
     CodeTree	*tree;
{
  char *assocOOP = ((char *) tree->data) + jit_ptr_field(OOP, object);
  BEFORE_PUSH;

  jit_ldi_p(JIT_V0, assocOOP);
  jit_ldxi_p(JIT_V0, JIT_V0, jit_ptr_field(Association, value));
  selfCached = false;
  gen_nothing(tree);
}

void
gen_dupTop(tree)
     CodeTree	*tree;
{
  if (spDelta < 0) {
    jit_ldr_p(JIT_V0, JIT_V2);
  }

  BEFORE_PUSH;
  gen_nothing(tree);
}

void
gen_pushSelf(tree)
     CodeTree	*tree;
{
  BEFORE_PUSH;

  if (!selfCached) {
    jit_ldi_p(JIT_V0, &self);
  }
  selfCached = true;
  gen_nothing(tree);
}

void
gen_pushOuter(tree)
     CodeTree	*tree;
{
  BEFORE_PUSH;
  CACHE_OUTER_CONTEXT;

  jit_ldxi_p(JIT_V0, JIT_V1, STACK_OFS(tree));
  selfCached = false;
  gen_nothing(tree);
}

void
gen_pushContext(tree)
     CodeTree	*tree;
{
  BEFORE_PUSH;

  jit_calli(PTR_EMPTY_CONTEXT_STACK);
  jit_ldi_p(JIT_V0, &thisContextOOP);
  selfCached = false;
  gen_nothing(tree);
}

/* Moves to V1 (alternative push) */
void
gen_altRecVar(tree)
     CodeTree	*tree;
{
  CACHE_REC_VAR;

  jit_ldxi_p(JIT_V1, JIT_R1, REC_VAR_OFS(tree));
  stackCached = -1;
  gen_nothing(tree);
}

void
gen_altTemp(tree)
     CodeTree	*tree;
{
  CACHE_TEMP;

  jit_ldxi_p(JIT_V1, JIT_V1, TEMP_OFS(tree));
  stackCached = -1;
  gen_nothing(tree);
}

void
gen_altLitConst(tree)
     CodeTree	*tree;
{
  jit_movi_p(JIT_V1, tree->data);
  stackCached = -1;
  gen_nothing(tree);
}

void
gen_altLitVar(tree)
     CodeTree	*tree;
{
  char *assocOOP = ((char *) tree->data) + jit_ptr_field(OOP, object);

  jit_ldi_p(JIT_V1, assocOOP);
  jit_ldxi_p(JIT_V1, JIT_V1, jit_ptr_field(Association, value));
  stackCached = -1;
  gen_nothing(tree);
}

void
gen_getTop(tree)
     CodeTree	*tree;
{
  if (spDelta < 0) {
    jit_ldr_p(JIT_V1, JIT_V2);
  } else {
    jit_movr_p(JIT_V1, JIT_V0);
  }

  stackCached = -1;
  gen_nothing(tree);
}

void
gen_altSelf(tree)
     CodeTree	*tree;
{
  if (!selfCached) {
    jit_ldi_p(JIT_V1, &self);
  }
  stackCached = -1;
  gen_nothing(tree);
}

void
gen_altOuter(tree)
     CodeTree	*tree;
{
  CACHE_OUTER_CONTEXT;

  jit_ldxi_p(JIT_V1, JIT_V1, STACK_OFS(tree));
  stackCached = -1;
  gen_nothing(tree);
}

void
gen_altContext(tree)
     CodeTree	*tree;
{
  jit_calli(PTR_EMPTY_CONTEXT_STACK);
  jit_ldi_p(JIT_V1, &thisContextOOP);
  stackCached = -1;
  gen_nothing(tree);
}

/* Set top */
void
gen_topRecVar(tree)
     CodeTree	*tree;
{
  BEFORE_SET_TOP;
  CACHE_REC_VAR;

  jit_ldxi_p(JIT_V0, JIT_R1, REC_VAR_OFS(tree));
  selfCached = false;
  gen_nothing(tree);
}

void
gen_topTemp(tree)
     CodeTree	*tree;
{
  BEFORE_SET_TOP;
  CACHE_TEMP;

  jit_ldxi_p(JIT_V0, JIT_V1, TEMP_OFS(tree));
  selfCached = false;
  gen_nothing(tree);
}

void
gen_topSelf(tree)
     CodeTree	*tree;
{
  BEFORE_SET_TOP;

  if (!selfCached) {
    jit_ldi_p(JIT_V0, &self);
  }
  selfCached = true;
  gen_nothing(tree);
}

void
gen_topOuter(tree)
     CodeTree	*tree;
{
  int index;
  BEFORE_SET_TOP;
  CACHE_OUTER_CONTEXT;
  index = ((Byte *) tree->data) [0];

  jit_ldxi_p(JIT_V0, JIT_V1, STACK_OFS(tree));
  selfCached = false;
  gen_nothing(tree);
}

void
gen_topLitConst(tree)
     CodeTree	*tree;
{
  BEFORE_SET_TOP;

  jit_movi_p(JIT_V0, tree->data);
  selfCached = false;
  gen_nothing(tree);
}

void
gen_topLitVar(tree)
     CodeTree	*tree;
{
  char *assocOOP = ((char *) tree->data) + jit_ptr_field(OOP, object);

  BEFORE_SET_TOP;

  jit_ldi_p(JIT_V0, assocOOP);
  jit_ldxi_p(JIT_V0, JIT_V0, jit_ptr_field(Association, value));
  selfCached = false;
  gen_nothing(tree);
}

void
gen_topContext(tree)
     CodeTree	*tree;
{
  BEFORE_SET_TOP;

  jit_calli(PTR_EMPTY_CONTEXT_STACK);
  jit_ldi_p(JIT_V0, &thisContextOOP);
  selfCached = false;
  gen_nothing(tree);
}

void
gen_illegal(tree)
     CodeTree	*tree;
{
  printf("Illegal operation %o in the code tree", tree->operation);
  exit(1);
}

/* Handling of extra operations */
void
gen_nothing(tree)
     CodeTree	*tree;
{
  switch (tree->operation & TREE_EXTRA) {
    case TREE_EXTRA_NONE:
      break;

    case TREE_EXTRA_POP:
      POP_EXPORT_SP;
      break;

    case TREE_EXTRA_RETURN:
      UNWIND_THROUGH(PTR_UNWIND_CONTEXT);
      break;

    case TREE_EXTRA_METHOD_RET:
      UNWIND_THROUGH(PTR_UNWIND_METHOD);
      break;

    case TREE_EXTRA_JMP_ALWAYS: {
      jit_insn *addr;
      
      CACHE_STACK_TOP;
      addr = lblGet(tree->jumpDest);
      addr = jit_jmpi(addr);
      lblUse(tree->jumpDest, addr);
      break;
    }

    case TREE_EXTRA_JMP_TRUE:
      CONDITIONAL_JUMP(trueOOP, falseOOP);
      break;

    case TREE_EXTRA_JMP_FALSE:
      CONDITIONAL_JUMP(falseOOP, trueOOP);
      break;
  }

  /* Change the CodeTree's operation field to TREE_ALREADY_EMITTED,
   * and null the extra op. field */
  tree->operation &= TREE_CLASS_CHECKS;
  tree->operation |= TREE_NOP | TREE_ALREADY_EMITTED;
}

void
gen_twoExtras(tree)
     CodeTree	*tree;
{
  emitCodeTree(tree->next);	/* emit the code for the real node */
  gen_nothing(tree);		/* and then the second extra */
}


/* Initialization and other code generation (prologs, interrupt checks) */

void
emitDeferredSends(ds)
     DeferredSend	*ds;
{
  jit_insn	*addr;
  CodeTree	*tree;
  InlineCache	*ic;

  if (!ds) {
    return;
  }

  emitDeferredSends(ds->next);

  tree = ds->tree;
  ic = (InlineCache *) tree->data;

  lblDefine(ds->address);
  if (ds->reg1 == JIT_NOREG) {
    jit_movi_p(JIT_R0, ds->oop);
    ds->reg1 = JIT_R0;
  }

  jit_stxi_p(SIZEOF_LONG * 1, JIT_V2, ds->reg0);
  jit_stxi_p(SIZEOF_LONG * 2, JIT_V2, ds->reg1);
  jit_addi_p(JIT_V2, JIT_V2, SIZEOF_LONG * 2);

  jit_movi_p(JIT_V1, ic);
  jit_sti_p(&sp, JIT_V2);
  jit_movi_ul(JIT_V0, tree->bp - byteCodes);
  jit_ldxi_p(JIT_R1, JIT_V1, jit_field(InlineCache, cachedIP));
  jit_sti_ul(&ip, JIT_V0);
  if (ic->isSuper) {
    jit_movi_p(JIT_V0, methodClass);
  }
  jit_jmpr(JIT_R1);
  jit_align(2);

  ic->returnIP = jit_get_label();

  IMPORT_SP;
  if (ds->trueDest == ds->falseDest) {
    /* This was an arithmetic deferred send. */
    jit_ldr_p(JIT_V0, JIT_V2);
    addr = lblGet(ds->trueDest);
    addr = jit_jmpi(addr);
    lblUse(ds->trueDest, addr);

  } else {
    /* This was a boolean deferred send. */
    jit_ldr_p(JIT_R0, JIT_V2);
    jit_subi_p(JIT_V2, JIT_V2, SIZEOF_LONG);
    jit_ldr_p(JIT_V0, JIT_V2);

    addr = lblGet(ds->trueDest);
    addr = jit_beqi_p(addr, JIT_R0, trueOOP);
    lblUse(ds->trueDest, addr);

    addr = lblGet(ds->falseDest);
    addr = jit_beqi_p(addr, JIT_R0, falseOOP);
    lblUse(ds->falseDest, addr);
    jit_jmpi(nonBooleanCode);
  }
}

void
emitInterruptCheck(restartReg)
     int restartReg;
{
  jit_insn *jmp, *begin;

    jit_align(2);
    begin = jit_get_label();

    jit_ldi_i(JIT_R2, &exceptFlag);
jmp=jit_beqi_i(jit_forward(), JIT_R2, 0);
    if (restartReg == JIT_NOREG) {
      jit_movi_p(JIT_RET, begin);
    } else {
      jit_movr_p(JIT_RET, restartReg);
    }
    jit_ret();
    jit_patch(jmp);
}

void
emitPrimitive(primitive, numArgs)
     int primitive, numArgs;
{
  /* primitive */
  jit_insn	*fail, *succeed;
  PrimitiveFunc	pf = getPrimitiveAddress(primitive);
  long		attr = getPrimitiveAttributes(primitive);

  if (attr & PRIM_INLINED) {
    errorf ("Primitive inlining not done yet");
  } else {
    jit_prepare  (3);
    jit_movi_p   (JIT_R0, current->methodOOP);
    jit_movi_p   (JIT_R1, numArgs);
    jit_pusharg_p(JIT_R0);
    jit_movi_p   (JIT_R2, primitive);
    jit_pusharg_i(JIT_R1);
    jit_pusharg_i(JIT_R2);
    jit_finish   (pf);
    jit_retval   (JIT_R0);
  }

  fail = ((attr & PRIM_FAIL) && (attr & (PRIM_SUCCEED | PRIM_RELOAD_IP)))
    ? jit_beqi_i  (jit_forward(), JIT_R0, -1)
    : nil;

  if (attr & PRIM_RELOAD_IP) {
    succeed = (attr & PRIM_SUCCEED)
      ? jit_beqi_i  (jit_forward(), JIT_R0, 0)
      : nil;

    if (attr & PRIM_CACHE_NEW_IP) {
      /* BlockClosure>>#value -- cache the value */
      jit_stxi_p(jit_field(InlineCache, cachedIP), JIT_V1, JIT_R0);
    }
    jit_movr_p  (JIT_V2, JIT_R0);

    if (succeed) {
      jit_patch   (succeed);
    }
  }
  if (attr & (PRIM_SUCCEED | PRIM_RELOAD_IP)) {
    emitInterruptCheck(JIT_V2);
    jit_jmpr    (JIT_V2);
  }
  if (fail) {
    jit_patch   (fail);
  }
}

void
emitContextSetup(numArgs, numTemps)
     int numArgs, numTemps;
{
  if (numArgs > 3 || numTemps > 3) {
    /* Call through a loop written in C */
    jit_movi_i    (JIT_V1, numTemps);
    jit_prepare   (3);
    jit_pusharg_p (JIT_V1);			/* numTemps */
    jit_pusharg_p (JIT_V2);			/* numArgs */
    jit_pusharg_p (JIT_R0);			/* newContext */
    jit_finish    (PTR_PREPARE_CONTEXT);
    IMPORT_SP;
    return;
  }

  /* Generate unrolled code to set up the frame -- this is done for
   * about 95% of the methods.
   */
  if (numArgs || numTemps) {
    int ofs;

    IMPORT_SP;
    switch (numArgs) {
      case 3: jit_ldxi_p(JIT_V0, JIT_V2, -2 * SIZEOF_LONG);
      case 2: jit_ldxi_p(JIT_R2, JIT_V2, -1 * SIZEOF_LONG);
      case 1: jit_ldr_p (JIT_R1, JIT_V2			 );
      case 0: break;
    }
    if (numTemps) {
      jit_movi_p(JIT_V1, nilOOP);
    }

    jit_addi_p  (JIT_V2, JIT_R0, jit_ptr_field(MethodContext, contextStack));
    jit_sti_p	(&temporaries, JIT_V2);
    ofs = 0;
    switch (numArgs) {
      case 3: jit_stxi_p(ofs, JIT_V2, JIT_V0); ofs += SIZEOF_LONG;
      case 2: jit_stxi_p(ofs, JIT_V2, JIT_R2); ofs += SIZEOF_LONG;
      case 1: jit_stxi_p(ofs, JIT_V2, JIT_R1); ofs += SIZEOF_LONG;
      case 0: break;
    }
    switch (numTemps) {
      case 3: jit_stxi_p(ofs, JIT_V2, JIT_V1); ofs += SIZEOF_LONG;
      case 2: jit_stxi_p(ofs, JIT_V2, JIT_V1); ofs += SIZEOF_LONG;
      case 1: jit_stxi_p(ofs, JIT_V2, JIT_V1); ofs += SIZEOF_LONG;
      case 0: break;
    }

    jit_addi_p  (JIT_V2, JIT_V2, ofs - SIZEOF_LONG);
  } else {
    jit_addi_p  (JIT_V2, JIT_R0, jit_ptr_field(MethodContext, contextStack[-1]));
  }
  jit_sti_p	(&sp, JIT_V2);
}

mst_Boolean
emitMethodProlog(method, receiverClass, hasSends)
     Method	 method;
     OOP	 receiverClass;
     mst_Boolean hasSends;
{
  MethodHeader header;
  int	       flag;
  
  header = method->header;
  flag = header.headerFlag;

  jit_ldxi_p      (JIT_V0, JIT_V2, SIZEOF_LONG * -header.numArgs);
  if (receiverClass == smallIntegerClass) {
    jit_bmci_ul   (doSendCode, JIT_V0, 1);
  } else {
    jit_bmsi_ul   (doSendCode, JIT_V0, 1);
    jit_ldxi_p    (JIT_R2, JIT_V0, jit_ptr_field(OOP, object));
    jit_ldxi_p    (JIT_R1, JIT_R2, jit_ptr_field(mst_Object, objClass));
    jit_bnei_p    (doSendCode, JIT_R1, receiverClass);
  }

  if (flag & 3) {
    jit_ldxi_p    (JIT_V1, JIT_V1, jit_field(InlineCache, returnIP));
    
    /* 1 is return self - nothing to do */
    if (flag > 1) {
      if (flag == 2) {
	/* return inst. var */
	int ofs = jit_ptr_field(mst_Object, data[header.primitiveIndex]);
	jit_ldxi_p    (JIT_R2, JIT_R2, ofs);   /* Remember? R2 is self->object */

      } else {
	/* return literal */
	OOP literal = oopToObj(method->literals)->data[0];
	jit_movi_p    (JIT_R2, literal);
      }

      jit_str_p     (JIT_V2, JIT_R2);	     /* Make it the stack top */
    }

    jit_jmpr      (JIT_V1);
    return (true);
  }

  jit_ldxi_p   (JIT_V2, JIT_V1, jit_field(InlineCache, returnIP));
  if (flag == 4) {
    emitPrimitive(header.primitiveIndex, header.numArgs);
  }

  /* Save the return IP */
  jit_ldi_p     (JIT_R0, &thisContextOOP);
  jit_subi_p    (JIT_V2, JIT_V2, doSendCode);
  jit_ldxi_p    (JIT_R0, JIT_R0, jit_ptr_field(OOP, object));
  jit_addi_p    (JIT_V2, JIT_V2, 1);
  jit_stxi_p    (jit_ptr_field(MethodContext, returnIP), JIT_R0, JIT_V2);

  /* prepare new state */
  jit_movi_i    (JIT_R0, header.stackDepth);
  jit_movi_i    (JIT_V2, header.numArgs);
  jit_prepare   (2);
  jit_pusharg_p (JIT_V2);
  jit_pusharg_p (JIT_R0);
  jit_finish    (PTR_ACTIVATE_NEW_CONTEXT);
  jit_retval    (JIT_R0);

  /* Remember? V0 was loaded with self for the inline cache test */
  jit_sti_p     (&self, JIT_V0);

  /* Set the new context's flags, and thisMethod */
  jit_movi_p    (JIT_V0, current->methodOOP);
  jit_movi_p    (JIT_V1, fromInt(0));
  jit_sti_p     (&thisMethod, JIT_V0);
  jit_stxi_p    (jit_ptr_field(MethodContext, flags), JIT_R0, JIT_V1);

  /* Move the arguments and nil the temporaries */
  emitContextSetup(header.numArgs, header.numTemps);
  return (false);
}

mst_Boolean
emitBlockProlog(block, receiverClass)
     Block	block;
     OOP	receiverClass;
{
  BlockHeader header;
  jit_insn    *x;

  header = block->header;

  /* Check if the number of arguments matches ours */
  jit_ldxi_uc   (JIT_R2, JIT_V1, jit_field(InlineCache, numArgs));
x=jit_beqi_ui   (jit_forward(), JIT_R2, header.numArgs);

  /* If they don't, check if we came here because somebody
   * called sendBlockValue.  In this case, the number of arguments
   * is surely valid and the inline cache's numArgs is bogus.
   * This handles #valueWithArguments:, #compileString:ifError:
   * and other primitives in which sendBlockValue is used.
   */
  jit_ldi_p	(JIT_R2, &nativeIP);
  jit_bnei_p	(doSendCode, JIT_R2, current->nativeCode);
  jit_patch	(x);

  /* Check if a block evaluation was indeed requested, and if the
   * BlockClosure really points to this CompiledBlock */
  jit_ldxi_p    (JIT_R1, JIT_V2, SIZEOF_LONG * -header.numArgs);
  jit_bmsi_ul   (doSendCode, JIT_R1, 1);
  jit_ldxi_p    (JIT_R1, JIT_R1, jit_ptr_field(OOP, object));
  jit_ldxi_p    (JIT_R0, JIT_R1, jit_ptr_field(mst_Object, objClass));
  jit_ldxi_p    (JIT_R2, JIT_R1, jit_ptr_field(BlockClosure, block));
  jit_bnei_p    (doSendCode, JIT_R0, blockClosureClass);
  jit_bnei_p    (doSendCode, JIT_R2, current->methodOOP);

  /* Now, the standard class check.  Always load self, but don't
   * check the receiver's class for clean blocks. */
  jit_ldxi_p    (JIT_V0, JIT_R1, jit_ptr_field(BlockClosure, receiver));
  if (block->header.clean != 0) {
    if (receiverClass == smallIntegerClass) {
      jit_bmci_ul   (doSendCode, JIT_V0, 1);
    } else {
      jit_bmsi_ul   (doSendCode, JIT_V0, 1);
      jit_ldxi_p    (JIT_R0, JIT_V0, jit_ptr_field(OOP, object));
      jit_ldxi_p    (JIT_R0, JIT_R0, jit_ptr_field(mst_Object, objClass));
      jit_bnei_p    (doSendCode, JIT_R0, receiverClass);
    }
  }

  /* All tests passed.  Now save the return IP */
  jit_ldxi_p    (JIT_V2, JIT_V1, jit_field(InlineCache, returnIP));
  jit_ldi_p     (JIT_R0, &thisContextOOP);
  jit_subi_p    (JIT_V2, JIT_V2, doSendCode);
  jit_ldxi_p    (JIT_R0, JIT_R0, jit_ptr_field(OOP, object));
  jit_addi_p	(JIT_V2, JIT_V2, 1);
  jit_stxi_p    (jit_ptr_field(MethodContext, returnIP), JIT_R0, JIT_V2);

  /* Get the outer context in a callee-preserved register... */
  jit_ldxi_p    (JIT_V1, JIT_R1, jit_ptr_field(BlockClosure, outerContext));

  /* prepare new state */
  jit_movi_i    (JIT_R0, header.depth);
  jit_movi_i    (JIT_V2, header.numArgs);
  jit_prepare   (2);
  jit_pusharg_p (JIT_V2);
  jit_pusharg_p (JIT_R0);
  jit_finish    (PTR_ACTIVATE_NEW_CONTEXT);
  jit_retval    (JIT_R0);

  /* Remember? V0 was loaded with self for the inline cache test.
   * Also initialize thisMethod and the pointer to the outerContext. */
  jit_movi_p    (JIT_R1, current->methodOOP);
  jit_sti_p     (&self, JIT_V0);
  jit_sti_p     (&thisMethod, JIT_R1);
  jit_stxi_p    (jit_ptr_field(BlockContext, outerContext), JIT_R0, JIT_V1);

  /* Move the arguments and nil the temporaries */
  emitContextSetup(header.numArgs, header.numTemps);

  return (false);
}


/* Code tree creation functions */

mst_Boolean dcd_pushRecVar(b, bp)
     Byte	b, *bp;
{
  pushTreeNode(bp,
    NULL,
    TREE_PUSH | TREE_REC_VAR,
    (voidPtr) (unsigned long) (b - pushReceiverVariable));

  return (false);
}

mst_Boolean dcd_pushTemp(b, bp)
     Byte	b, *bp;
{
  pushTreeNode(bp,
    NULL,
    TREE_PUSH | TREE_TEMP,
    (voidPtr) (unsigned long) (b - pushTemporaryVariable));

  return (false);
}

mst_Boolean dcd_pushLit(b, bp)
     Byte	b, *bp;
{
  pushTreeNodeOOP(bp,
    NULL,
    TREE_PUSH | TREE_LIT_CONST,
    literals[b - pushLitConstant]);

  return (false);
}

mst_Boolean dcd_pushVar(b, bp)
     Byte	b, *bp;
{
  pushTreeNodeOOP(bp,
    NULL,
    TREE_PUSH | TREE_LIT_VAR,
    literals[b - pushLitVariable]);

  return (false);
}

mst_Boolean dcd_stRecVar(b, bp)
     Byte	b, *bp;
{
  pushTreeNode(bp,
    popTreeNode(NULL),
    TREE_STORE | TREE_REC_VAR | TREE_EXTRA_POP,
    (voidPtr) (unsigned long) (b - popReceiverVariable));

  return (true);
}

mst_Boolean dcd_stTemp(b, bp)
     Byte	b, *bp;
{
  pushTreeNode(bp,
    popTreeNode(NULL),
    TREE_STORE | TREE_TEMP | TREE_EXTRA_POP,
    (voidPtr) (unsigned long) (b - popTemporaryVariable));

  return (true);
}

mst_Boolean dcd_pushSelf(b, bp)
     Byte	b, *bp;
{
  pushTreeNode(bp,
    NULL,
    TREE_PUSH | TREE_SELF,
    NULL);

  return (false);
}

mst_Boolean dcd_pushSpecial(b, bp)
     Byte	b, *bp;
{
  static OOP *specialOOPs[] = {
    &trueOOP, &falseOOP, &nilOOP,
    &minusOneOOP, &zeroOOP, &oneOOP, &twoOOP
  };

  pushTreeNodeOOP(bp,
    NULL,
    TREE_PUSH | TREE_LIT_CONST,
    *specialOOPs[b - pushSpecial - 1]);

  return (false);
}

mst_Boolean dcd_retSelf(b, bp)
     Byte	b, *bp;
{
  pushTreeNode(bp,
    popTreeNode(NULL),
    TREE_SET_TOP | TREE_SELF | TREE_EXTRA_RETURN | selfClassCheck,
    NULL);

  return (true);
}

mst_Boolean dcd_retSpecial(b, bp)
     Byte	b, *bp;
{
  static OOP *specialOOPs[] = { &trueOOP, &falseOOP, &nilOOP };

  pushTreeNodeOOP(bp,
    popTreeNode(NULL),
    TREE_SET_TOP | TREE_LIT_CONST | TREE_EXTRA_RETURN,
    *specialOOPs[b - returnIndexed - 1]);

  return (true);
}

mst_Boolean dcd_explicitRet(b, bp)
     Byte	b, *bp;
{
  setTopNodeExtra(TREE_EXTRA_METHOD_RET, 0);
  return (true);
}

mst_Boolean dcd_retStackTop(b, bp)
     Byte	b, *bp;
{
  setTopNodeExtra(TREE_EXTRA_RETURN, 0);
  return (true);
}

mst_Boolean dcd_bigLiteralOp(b, bp)
     Byte	b, *bp;
{
  unsigned int num;
  num = bp[1];
  num <<= 8;
  num |= bp[2];

  switch(num >> 14) {
    case 0:
      pushTreeNodeOOP(bp,
        NULL,
        TREE_PUSH | TREE_LIT_CONST,
        literals[num & 16383]);

      return (false);

    case 1:
      pushTreeNodeOOP(bp,
        NULL,
        TREE_PUSH | TREE_LIT_VAR,
        literals[num & 16383]);

      return (false);

    case 2:
      pushTreeNodeOOP(bp,
        popTreeNode(NULL),
        TREE_STORE | TREE_LIT_VAR | TREE_EXTRA_POP,
        literals[num & 16383]);

      return (true);

    case 3: default:
      pushTreeNodeOOP(bp,
        popTreeNode(NULL),
        TREE_STORE | TREE_LIT_VAR,
        literals[num & 16383]);

      return (false);
  }
}

mst_Boolean dcd_illegal(b, bp)
     Byte	b, *bp;
{
  errorf("Invalid bytecode %d found!", b);
  exit(1);
}

mst_Boolean dcd_bigInstanceOp(b, bp)
     Byte	b, *bp;
{
  unsigned int num;
  num = bp[1];
  num <<= 8;
  num |= bp[2];

  switch(num >> 14) {
    case 0:
      pushTreeNode(bp,
	popTreeNode(popTreeNode(NULL)),
	TREE_SEND | TREE_POP_INTO_ARRAY,
	(unsigned long) (num & 16383));

      return (false);

    case 1:
      pushTreeNode(bp,
        NULL,
        TREE_PUSH | TREE_REC_VAR,
	(unsigned long) (num & 16383));

      return (false);

    case 2:
      pushTreeNode(bp,
        popTreeNode(NULL),
        TREE_STORE | TREE_REC_VAR | TREE_EXTRA_POP,
	(unsigned long) (num & 16383));

      return (true);

    case 3: default:
      pushTreeNode(bp,
        popTreeNode(NULL),
        TREE_STORE | TREE_REC_VAR,
	(unsigned long) (num & 16383));

      return (false);
  }
}

mst_Boolean dcd_pushIdxVal(b, bp)
     Byte	b, *bp;
{
  Byte		ival2;
  ival2 = bp[1];
  
  switch (ival2 >> 6) {
    case 0:
      pushTreeNode(bp,
        NULL,
        TREE_PUSH | TREE_REC_VAR,
        (voidPtr) (unsigned long) (ival2 & 63));

      break;
    case 1:
      pushTreeNode(bp,
        NULL,
        TREE_PUSH | TREE_TEMP,
        (voidPtr) (unsigned long) (ival2 & 63));

      break;
    case 2:
      pushTreeNodeOOP(bp,
        NULL,
        TREE_PUSH | TREE_LIT_CONST,
        literals[(unsigned long) (ival2 & 63)]);

      break;
    case 3: default:
      pushTreeNodeOOP(bp,
        NULL,
        TREE_PUSH | TREE_LIT_VAR,
        literals[(unsigned long) (ival2 & 63)]);

      break;
  }
  return (false);
}

mst_Boolean dcd_storeStackTop(b, bp)
     Byte	b, *bp;
{
  Byte		ival2;
  ival2 = bp[1];
  
  switch (ival2 >> 6) {
    case 0:
      pushTreeNode(bp,
        popTreeNode(NULL),
        TREE_STORE | TREE_REC_VAR,
        (voidPtr) (unsigned long) (ival2 & 63));

      break;
    case 1:
      pushTreeNode(bp,
        popTreeNode(NULL),
        TREE_STORE | TREE_TEMP,
        (voidPtr) (unsigned long) (ival2 & 63));

      break;
    case 2:
      pushTreeNodeOOP(bp,
        popTreeNode(NULL),
        TREE_STORE | TREE_LIT_CONST,
        literals[(unsigned long) (ival2 & 63)]);

      break;
    case 3: default:
      pushTreeNodeOOP(bp,
        popTreeNode(NULL),
        TREE_STORE | TREE_LIT_VAR,
        literals[(unsigned long) (ival2 & 63)]);

      break;
  }
  return (false);
}

mst_Boolean dcd_popStoreTop(b, bp)
     Byte	b, *bp;
{
  Byte		ival2;
  ival2 = bp[1];
  
  switch (ival2 >> 6) {
    case 0:
      pushTreeNode(bp,
        popTreeNode(NULL),
        TREE_STORE | TREE_REC_VAR | TREE_EXTRA_POP,
        (voidPtr) (unsigned long) (ival2 & 63));

      break;
    case 1:
      pushTreeNode(bp,
        popTreeNode(NULL),
        TREE_STORE | TREE_TEMP | TREE_EXTRA_POP,
        (voidPtr) (unsigned long) (ival2 & 63));

      break;
    case 2:
      pushTreeNodeOOP(bp,
        popTreeNode(NULL),
        TREE_STORE | TREE_LIT_CONST | TREE_EXTRA_POP,
        literals[(unsigned long) (ival2 & 63)]);

      break;
    case 3: default:
      pushTreeNodeOOP(bp,
        popTreeNode(NULL),
        TREE_STORE | TREE_LIT_VAR | TREE_EXTRA_POP,
        literals[(unsigned long) (ival2 & 63)]);

      break;
  }
  return (true);
}

mst_Boolean dcd_sendShort(b, bp)
     Byte	b, *bp;
{
  Byte ival2 = bp[1];
  pushSendNode(bp, literals[ival2 & 31], ival2 >> 5, false, TREE_SEND);

  return (false);
}

mst_Boolean dcd_sendLong(b, bp)
     Byte	b, *bp;
{
  int ival2 = bp[1];
  int ival3 = bp[2];
  ival3 |= (ival2 & 192) << 2;
  pushSendNode(bp, literals[ival3], ival2 & 31, ival2 & 32, TREE_SEND);

  return (false);
}

mst_Boolean dcd_supSendShort(b, bp)
     Byte	b, *bp;
{
  Byte ival2 = bp[1];
  pushSendNode(bp, literals[ival2 & 31], ival2 >> 5, true, TREE_SEND);

  return (false);
}

mst_Boolean dcd_popStack(b, bp)
     Byte	b, *bp;
{
  setTopNodeExtra(TREE_EXTRA_POP, 0);
  return (true);
}

mst_Boolean dcd_dupStack(b, bp)
     Byte	b, *bp;
{
  pushTreeNode(bp,
    NULL,
    TREE_PUSH | TREE_DUP,
    NULL);

  return (false);
}

mst_Boolean dcd_pushContext(b, bp)
     Byte	b, *bp;
{
  pushTreeNode(bp,
    NULL,
    TREE_PUSH | TREE_THIS_CONTEXT,
    NULL);

  return (false);
}

mst_Boolean dcd_outerTempOp(b, bp)
     Byte	b, *bp;
{
  Byte op = bp[1] >> 6;

  switch(op) {
    case 0:
      abort();

    case 1:
      pushTreeNode(bp,
        NULL,
        TREE_PUSH | TREE_OUTER_TEMP,
        NULL);

      return (false);

    case 2:
      pushTreeNode(bp,
        popTreeNode(NULL),
        TREE_STORE | TREE_OUTER_TEMP | TREE_EXTRA_POP,
        NULL);

      return (true);

    case 3: default:
      pushTreeNode(bp,
        popTreeNode(NULL),
        TREE_STORE | TREE_OUTER_TEMP,
        NULL);

      return (false);
  }
}

mst_Boolean dcd_nop(b, bp)
     Byte	b, *bp;
{
  return (false);
}

mst_Boolean dcd_topSelf(b, bp)
     Byte	b, *bp;
{
  pushTreeNode(bp,
    popTreeNode(NULL),
    TREE_SET_TOP | TREE_SELF | selfClassCheck,
    NULL);

  return (true);
}

mst_Boolean dcd_topOne(b, bp)
     Byte	b, *bp;
{
  pushTreeNodeOOP(bp,
    popTreeNode(NULL),
    TREE_SET_TOP | TREE_LIT_CONST,
    fromInt(1));

  return (true);
}

mst_Boolean dcd_topIndexedVal(b, bp)
     Byte	b, *bp;
{
  Byte		ival2;
  ival2 = bp[1];
  
  switch (ival2 >> 6) {
    case 0:
      pushTreeNode(bp,
        popTreeNode(NULL),
        TREE_SET_TOP | TREE_REC_VAR,
        (voidPtr) (unsigned long) (ival2 & 63));

      break;
    case 1:
      pushTreeNode(bp,
        popTreeNode(NULL),
        TREE_SET_TOP | TREE_TEMP,
        (voidPtr) (unsigned long) (ival2 & 63));

      break;
    case 2:
      pushTreeNodeOOP(bp,
        popTreeNode(NULL),
        TREE_SET_TOP | TREE_LIT_CONST,
        literals[(unsigned long) (ival2 & 63)]);

      break;
    case 3: default:
      pushTreeNodeOOP(bp,
        popTreeNode(NULL),
        TREE_SET_TOP | TREE_LIT_VAR,
        literals[(unsigned long) (ival2 & 63)]);

      break;
  }
  return (true);
}

mst_Boolean dcd_endExecution(b, bp)
     Byte	b, *bp;
{
  errorf("End of execution bytecode ignored\n");
  return (false);
}

mst_Boolean dcd_shJmp(b, bp)
     Byte	b, *bp;
{
  setTopNodeExtra(TREE_EXTRA_JMP_ALWAYS, (b & 7) + 2);
  return (true);
}

mst_Boolean dcd_shJmpFalse(b, bp)
     Byte	b, *bp;
{
  setTopNodeExtra(TREE_EXTRA_JMP_FALSE, (b & 7) + 2);
  return (true);
}

mst_Boolean dcd_longJmp(b, bp)
     Byte	b, *bp;
{
  int ofs;
  ofs = (b & 7) << 8;
  ofs += bp[1] - 1022;
  setTopNodeExtra(TREE_EXTRA_JMP_ALWAYS, ofs);
  return (true);
}

mst_Boolean dcd_popJmpTrue(b, bp)
     Byte	b, *bp;
{
  int ofs;
  ofs = (b & 3) << 8;
  ofs += bp[1] + 2;
  setTopNodeExtra(TREE_EXTRA_JMP_TRUE, ofs);
  return (true);
}

mst_Boolean dcd_popJmpFalse(b, bp)
     Byte	b, *bp;
{
  int ofs;
  ofs = (b & 3) << 8;
  ofs += bp[1] + 2;
  setTopNodeExtra(TREE_EXTRA_JMP_FALSE, ofs);
  return (true);
}

mst_Boolean dcd_sendSpecial(b, bp)
     Byte	b, *bp;
{
  SpecialSelector *info = &specialSendBytecodes[b - plusSpecial];
  pushSendNode(bp, *info->selectorOOP, info->numArgs, false, info->operation);

  return (false);
}

mst_Boolean dcd_sendPacked(b, bp)
     Byte	b, *bp;
{
  int numArgs = (b - sendNoArg) >> 4;
  pushSendNode(bp, literals[b & 15], numArgs, false, TREE_SEND);

  return (false);
}




/* Main translator loop */
void
translateMethod(methodOOP, receiverClass, size)
     OOP	 methodOOP;
     OOP	 receiverClass;
     int	 size;
{
  register Byte	   *end, *bp, b;
  int		   inlineCacheCount;
  char		   *destinations;
  CodeStackPointer *stackPos;
  int		   i;

  recVarCached = selfCached = false;
  stackCached = -1;
  spDelta = -SIZEOF_LONG;
  deferredHead = nil;
  methodClass = getMethodClass(methodOOP);
  byteCodes = getMethodByteCodes(methodOOP);
  literals = getMethodLiterals(methodOOP);
  end = byteCodes + size;

  if (receiverClass == smallIntegerClass) {
    selfClassCheck = TREE_IS_INTEGER | TREE_IS_NOT_BLOCK;
  } else if (receiverClass == blockClosureClass) {
    selfClassCheck = TREE_IS_NOT_INTEGER | TREE_IS_BLOCK;
  } else {
    selfClassCheck = TREE_IS_NOT_INTEGER | TREE_IS_NOT_BLOCK;
  }

  /* Count the space for the inline caches */
  for (inlineCacheCount = 0, bp = byteCodes; bp < end; bp += byteCodeSize(b)) {
    b = *bp;
    inlineCacheCount += isSendByteCode(b);
  }

  /* Emit the prolog of the compiled code. */
  jit_ldi_p(JIT_V2, &sp);
  if (oopClass(methodOOP) == compiledBlockClass) {
    if (emitBlockProlog((Block) oopToObj(methodOOP), receiverClass)) {
      return;
    }
  } else {
    if (emitMethodProlog((Method) oopToObj(methodOOP), receiverClass)) {
      return;
    }
  }

  if (inlineCacheCount) {
    current->inlineCaches = currInlineCache = (InlineCache *) xmalloc(
      inlineCacheCount * sizeof(InlineCache));
  }

  stackPos = alloca(size * sizeof(CodeStackPointer *));
  labels = alloca(size * sizeof(Label *));
  destinations = (char *) labels;
  intTab = (char *) alloca(size);

  computeStackPositions(byteCodes, size, tStack, stackPos);
  analyzeByteCodes(methodOOP, size, 
    destinations, intTab,
    auxDataObstack);

  /* Create labels for bytecodes on which a jump lands */
  for (i = size; --i >= 0; ) {
    labels[i] = destinations[i] ? lblNew() : NULL;
  }

  /* Now, go through the main translation loop */
  for (bp = byteCodes, thisLabel = labels; bp < end;
       bp += byteCodeSize(b), thisLabel += byteCodeSize(b),
       stackPos += byteCodeSize(b), intTab += byteCodeSize(b)) {
    register mst_Boolean endStatement;

    /* Updating the tSP in pushTreeNode/popTreeNode is not enough,
     * because if two basic blocks are mutually exclusive the
     * SP at the second block's entrance must be the same as
     * the SP at the first block's entrance, even if the blocks
     * have a non-zero stack balance. */
    tSP = *stackPos;

    if (*thisLabel) {
      /* A basic block ends here. Compile it. */
      emitCode();
      CACHE_STACK_TOP;

      /* If the label was not used yet, it will be used for a backward
       * jump.  A backward jump could be used to code an infinite loop
       * such as `[] repeat', so we test exceptFlag here. */
      if (!lblDefine(*thisLabel)) {
	emitInterruptCheck(JIT_NOREG);
      }
    }

    b = *bp;
    endStatement = decodeBytecodeFuncs[b] (b, bp);
    if (endStatement) {
      /* The current statement has ended. Compile it. */
      emitCode();
    }
  }

  emitCode();
  emitDeferredSends(deferredHead);
  currInlineCache[-1].more = false;
}


/* External interfacing */

char *
getMethodBase()
{
  /* Return anything that is guaranteed to be a SmallInteger
   * far from a method's code. */
  if (!doSendCode) {
    generateRunTimeCode();
    memzero (methodsTable, sizeof(methodsTable));
  }

  return (doSendCode);
}

voidPtr
getNativeCode(methodOOP, receiverClass)
     OOP	 methodOOP;
     OOP	 receiverClass;
{
  register MethodEntry  *method, *prev;
  register unsigned int hashEntry;
  int			size;

  if (isNil(methodOOP)) {
    return (nil);
  }

  hashEntry = oopIndex(methodOOP) % HASH_TABLE_SIZE;
  if (method = methodsTable[hashEntry]) {
    if (method->methodOOP == methodOOP
	&& method->receiverClass == receiverClass) {
      return method->nativeCode;
    }

    for (prev = method; method = method->next; prev = method) {
      if (method->methodOOP != methodOOP
	  || method->receiverClass != receiverClass) {
	continue;
      }
      prev->next = method->next;
      method->next = methodsTable[hashEntry];
      methodsTable[hashEntry] = method;
      return method->nativeCode;
    }
  }

  size = numIndexableFields(methodOOP);
  newMethodEntry (methodOOP, receiverClass, size);
  translateMethod (methodOOP, receiverClass, size);
  return (finishMethodEntry (methodOOP));
}

void
walkAndResetCaches(head)
     MethodEntry **head;
{
  register InlineCache	*ic;
  register jit_insn	*lookupIP;
  register MethodEntry	*method;

  for (method = *head; method; method = method->next) {
    ic = method->inlineCaches;
    if (!ic) {
      continue;
    }
    do {
      lookupIP = ic->isSuper ? doSuperCode : doSendCode;
      if (ic->cachedIP != lookupIP && !isValidIP(ic->cachedIP)) {
	ic->cachedIP = lookupIP;
      }
    } while ((ic++)->more);
  }
}

void
freeReleasedNativeCode()
{
  MethodEntry		**hashEntry, *method;

  if (!released) {
    return;
  }

  for (hashEntry = methodsTable;
       hashEntry < &methodsTable[HASH_TABLE_SIZE];
       hashEntry++) {
    walkAndResetCaches(hashEntry);
  }

  walkAndResetCaches(&discarded);
  validateMethodCacheEntries();

  /* now free the list */
  while (method = released) {
    released = released->next;
    xfree(method);
  }
}

void
walkAndRemoveMethod(methodOOP, ptrNext)
     register MethodEntry  **ptrNext;
     OOP		   methodOOP;
{
  register MethodEntry  *method;

  while (method = *ptrNext) {
    if (method->methodOOP != methodOOP) {
      /* Move ptrNext forward */
      ptrNext = &(method->next);
      continue;
    }

    /* Adjust the list */
    *ptrNext = method->next;
    method->next = released;
    released = method;

    /* Mark the method as freed */
    if (method->inlineCaches) {
      xfree(method->inlineCaches);
    }
    method->receiverClass = nil;
    method->inlineCaches = nil;
  }

  /* Terminate the list */
  *ptrNext = NULL;
}

void
releaseNativeCode(methodOOP)
     OOP	 methodOOP;
{
  register unsigned int hashEntry;

  hashEntry = oopIndex(methodOOP) % HASH_TABLE_SIZE;
  walkAndRemoveMethod(methodOOP, &methodsTable[hashEntry]);
  methodOOP->flags &= ~F_XLAT;

  if (methodOOP->flags & F_XLAT_DISCARDED) {
    walkAndRemoveMethod(methodOOP, &discarded);
    methodOOP->flags &= ~F_XLAT_DISCARDED;
  }
}

void
discardNativeCode(methodOOP)
     OOP	 methodOOP;
{
  register MethodEntry  *method, **ptrNext;
  register unsigned int hashEntry;

  methodOOP->flags |= F_XLAT_DISCARDED;
  hashEntry = oopIndex(methodOOP) % HASH_TABLE_SIZE;
  ptrNext = &methodsTable[hashEntry];

  while (method = *ptrNext) {
    if (method->methodOOP != methodOOP) {
      /* Move ptrNext forward */
      ptrNext = &(method->next);
      continue;
    }

    /* Move to the `discarded' list */
    *ptrNext = method->next;
    method->next = discarded;
    discarded = method;
  }

  /* Terminate the list */
  *ptrNext = NULL;
}


void
generateRunTimeCode()
{
  voidPtr area = xmalloc(10000);
  jit_insn *doesNotUnderstandCode;

  runNativeCode = jit_set_ip(area).pptr;
  {
    static InlineCache ic;
    static int arg;

    jit_prolog	(1);
arg=jit_arg_p	();
    jit_getarg_p(JIT_R0, arg);
    jit_movi_p	(JIT_V1, &ic);
    jit_ldi_p	(JIT_V2, &sp);
    jit_jmpr	(JIT_R0);
    jit_align	(2);

    ic.returnIP = jit_get_label();
    jit_ret	();
  }

  /* send #doesNotUnderstand:  If the method is not understood, the stack
   * is changed to the format needed by #doesNotUnderstand: in lookupNativeIP;
   * no inline caching must take place because we have modify the stack
   * each time they try to send the message. */
  jit_align(2);
  doesNotUnderstandCode = jit_get_label();
  jit_prolog(1);
  jit_set_ip(doesNotUnderstandCode);
  {
    jit_ldi_p     (JIT_V2, &sp);	/* changed by lookupMethod!! */
    jit_movi_l	  (JIT_R2, 1);
    jit_ldi_p	  (JIT_R0, &doesNotUnderstandColonSymbol);
    jit_ldxi_p    (JIT_R1, JIT_V2, -SIZEOF_LONG);
    jit_prepare   (4);
    jit_pusharg_p (JIT_V0);		/* methodClass */
    jit_pusharg_p (JIT_R1);		/* receiver */
    jit_pusharg_i (JIT_R2);		/* numArgs */
    jit_pusharg_p (JIT_R0);		/* selector */
    jit_finish    (PTR_LOOKUP_NATIVE_IP);
    jit_retval    (JIT_R0);

    /* Could crash if again #doesNotUnderstand: -- probably better than
     * an infinite loop. */
    jit_jmpr      (JIT_R0);
  }

  jit_align(2);
  doSendCode = jit_get_label();
  jit_prolog(1);
  jit_set_ip(doSendCode);
  {
    jit_insn	  *jmp;

    /* load other parameters into R0/R2 */
    jit_ldxi_uc   (JIT_R2, JIT_V1, jit_field(InlineCache, numArgs));
    jit_ldxi_p    (JIT_R0, JIT_V1, jit_field(InlineCache, selector));

    /* load self into R1 */
    jit_lshi_l	  (JIT_R1, JIT_R2, LONG_SHIFT);
    jit_negr_l	  (JIT_R1, JIT_R1);
    jit_ldxr_l    (JIT_R1, JIT_V2, JIT_R1);

    /* method class */
    jit_movi_p	  (JIT_V0, smallIntegerClass);
jmp=jit_bmsi_l	  (jit_forward(), JIT_R1, 1);
    jit_ldxi_p    (JIT_V0, JIT_R1, jit_ptr_field(OOP, object));
    jit_ldxi_p    (JIT_V0, JIT_V0, jit_ptr_field(mst_Object, objClass));
    jit_patch	  (jmp);

    jit_prepare   (4);
    jit_pusharg_p (JIT_V0);		/* methodClass */
    jit_pusharg_p (JIT_R1);		/* receiver */
    jit_pusharg_i (JIT_R2);		/* numArgs */
    jit_pusharg_p (JIT_R0);		/* selector */
    jit_finish    (PTR_LOOKUP_NATIVE_IP);
    jit_retval    (JIT_R0);

    /* store the address in the inline cache if not #doesNotUnderstand: */
    jit_beqi_l	  (doesNotUnderstandCode, JIT_R0, 0);
    jit_stxi_p    (jit_field(InlineCache, cachedIP), JIT_V1, JIT_R0);
    jit_jmpr      (JIT_R0);
  }

  jit_align(2);
  doSuperCode = jit_get_label();
  jit_prolog(1);
  jit_set_ip(doSuperCode);
  {
    /* load other args into R1/R2 */
    jit_ldi_l     (JIT_R1, &self);
    jit_ldxi_uc   (JIT_R2, JIT_V1, jit_field(InlineCache, numArgs));
    jit_ldxi_p    (JIT_R0, JIT_V1, jit_field(InlineCache, selector));

    /* the method class is the superclass of the current method's methodClass */
    jit_ldxi_p    (JIT_V0, JIT_V0, jit_ptr_field(OOP, object));
    jit_ldxi_p    (JIT_V0, JIT_V0, jit_ptr_field(Class, superClass));

    jit_prepare   (4);
    jit_pusharg_p (JIT_V0);		/* methodClass */
    jit_pusharg_p (JIT_R1);		/* receiver */
    jit_pusharg_i (JIT_R2);		/* numArgs */
    jit_pusharg_p (JIT_R0);		/* selector */
    jit_finish    (PTR_LOOKUP_NATIVE_IP);
    jit_retval    (JIT_R0);

    /* store the address in the inline cache if not #doesNotUnderstand: */
    jit_beqi_l	  (doesNotUnderstandCode, JIT_R0, 0);
    jit_stxi_p    (jit_field(InlineCache, cachedIP), JIT_V1, JIT_R0);
    jit_jmpr      (JIT_R0);
  }

  jit_align(2);
  nonBooleanCode = jit_get_label();
  jit_prolog(1);
  jit_set_ip(nonBooleanCode);
  {
    static char	    methodName[] = "mustBeBoolean";
    extern char	    *abortExecution;

    jit_ldi_p	  (JIT_V2, &sp);		 /* push R0 on the */
    jit_stxi_p	  (SIZEOF_LONG, JIT_V2, JIT_R0); /* Smalltalk stack */
    jit_addi_p	  (JIT_V2, JIT_V2, SIZEOF_LONG);
    jit_movi_p	  (JIT_R1, methodName);
    jit_sti_p	  (&sp, JIT_V2);		/* update SP */
    jit_sti_p	  (&abortExecution, JIT_R1);
    jit_ret	  ();
  }

  jit_align(2);
  returnFromNativeCode = jit_get_ip().pptr;
  jit_prolog(1);
  jit_set_ip(returnFromNativeCode);
  {
    jit_movi_i(JIT_RET, 0);
    jit_ret();
  }
}
#endif /* USE_JIT_TRANSLATION */
