"======================================================================
|
|   Smalltalk TCP/IP sockets - IPAddress class
|
|   $Revision: 1.95.1$
|   $Date: 2000/12/27 10:45:49$
|   $Author: pb$
|
 ======================================================================"


"======================================================================
|
| Copyright 1988-92, 1994-95, 1999, 2000 Free Software Foundation, Inc.
| Written by Steve Byrne and Paolo Bonzini.
|
| This file is part of the GNU Smalltalk class library.
|
| The GNU Smalltalk class library is free software; you can redistribute it
| and/or modify it under the terms of the GNU Lesser General Public License
| as published by the Free Software Foundation; either version 2.1, or (at
| your option) any later version.
| 
| The GNU Smalltalk class library is distributed in the hope that it will be
| useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
| MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser
| General Public License for more details.
| 
| You should have received a copy of the GNU Lesser General Public License
| along with the GNU Smalltalk class library; see the file COPYING.LESSER.
| If not, write to the Free Software Foundation, 59 Temple Place - Suite
| 330, Boston, MA 02111-1307, USA.  
|
 ======================================================================"


Object subclass: #SocketAddress
       instanceVariableNames: ''
       classVariableNames: 'name '
       poolDictionaries: ''
       category: 'Sockets-Protocols'
! 

SocketAddress class instanceVariableNames:
    'anyLocalAddress localHostName loopbackHost unknownAddress cache'
!

SocketAddress subclass: #IPAddress
       instanceVariableNames: 'address'
       classVariableNames: ''
       poolDictionaries: ''
       category: 'Sockets-Protocols'
! 

CStruct newStruct: #CSockAddrStruct declaration: #( 
	(sinFamily short)
	(sinPort (array byte 2))
	(sinAddr (array byte 4))
	(sinZero (array byte 8))
)!

!SocketAddress class methodsFor: 'initialization'!

flush
    "Flush the cached IP addresses."
    localHostName := self primLocalName.
    anyLocalAddress := self createLocalAddress.
    unknownAddress := self createUnknownAddress.
    loopbackHost := self createLoopbackHost.
    loopbackHost name: localHostName.

    cache := Dictionary new.
!

createLocalAddress
    self subclassResponsibility
!

createUnknownAddress
    self subclassResponsibility
!

createLoopbackHost
    ^self createLocalAddress
!

onStartup
    self allSubclassesDo: [ :each | each flush ]
! !

!SocketAddress class methodsFor: 'accessing'!

anyLocalAddress
    "Answer an IPAddress representing a local address."
    ^anyLocalAddress
!

at: host cache: aBlock
    ^cache at: host ifAbsentPut: aBlock
!

localHostName
    "Answer the name of the local machine."
    ^localHostName
!

loopbackHost
    "Answer an IPAddress representing the local machine (127.0.0.1)."
    ^loopbackHost		"127.0.0.1"
!

unknownAddress
    "Answer an IPAddress representing an unknown machine (0.0.0.0)."
    ^unknownAddress		"0.0.0.0"
! !

!SocketAddress class methodsFor: 'host name lookup'!

allByName: aString
    "Answer all the IP addresses that refer to the the given host.  If
     a digit address is passed in aString, the result is an array
     containing the single passed address.  If the host could not be
     resolved to an IP address, answer nil."
    | host |
    host := aString asLowercase.
    (self isDigitAddress: host) ifTrue: [
	^self class at: host cache: [ Array with: (self fromString: host) ]
    ].

    ^self class at: host cache: [
	| result index addresses addr |
	result := self lookupAllHostAddr: host.

	result address = 0 ifTrue: [
	    addresses := WriteStream on: (Array new: 1).
	    result := result castTo: CByteType.
	    index := 0.
	    [   addr := self
		    with: (result at: index) value
		    with: (result at: index + 1) value
		    with: (result at: index + 2) value
		    with: (result at: index + 3) value.

		index := index + 4.
		addr = unknownAddress
	    ]   whileFalse: [ addresses nextPut: addr ].
	    result free.
	    addresses := addresses contents.
	].
	addresses
    ]
!

byName: aString
    "Answer a single IP address that refer to the the given host.  If
     a digit address is passed in aString, the result is the same as
     using #fromString:.  If the host could not be resolved to an IP
     address, answer nil."
    | all |
    aString isEmpty ifTrue: [ ^loopbackHost ].
    all := self allByName: aString.
    ^all isNil ifTrue: [ nil ] ifFalse: [ all anyOne ]
! !

!SocketAddress class methodsFor: 'abstract'!

lookupAllHostAddr: name
    self subclassResponsibility
!

primName: address
    self subclassResponsibility
!

primLocalName
    self subclassResponsibility
!

fromSockAddr: aByteArray port: portAdaptor
    "Private - Answer a new IPAddress from a ByteArray containing a
     C sockaddr structure.  The portAdaptor's value is changed
     to contain the port that the structure refers to."

    self subclassResponsibility
! !

!SocketAddress methodsFor: 'accessing'!

= anIPAddress
    "Answer whether the receiver and anIPAddress represent
     the same machine.  The host name is not checked because
     an IPAddress created before a DNS is activated is named
     after its numbers-and-dots notation, while the same IPAddress,
     created when a DNS is active, is named after its resolved name."
    ^(self class == anIPAddress class) and: [
	self asByteArray = anIPAddress asByteArray
    ]
!

hash
    "Answer an hash value for the receiver"
    ^self asByteArray hash
!

name
    "Answer the host name (or the digit notation if the DNS could not
    resolve the address).  If the DNS answers a different IP address
    for the same name, the second response is not cached and the digit
    notation is also returned (somebody's likely playing strange jokes
    with your DNS)."

    | addresses |
    name isNil ifFalse: [ ^name ].
    name := self class primName: self asByteArray.

    "No DNS active..."
    name isNil ifTrue: [ ^name := self printString ].

    addresses := self class at: name cache: [ Array with: self ].

    addresses do: [ :each |
	each getName isNil ifTrue: [ each name: name ].
	(each = self and: [ each getName ~= name ]) ifTrue: [
	    "Seems like someone's joking with the DNS server
	     and changed this host's IP address even though the
	     name stays the same. Don't cache the name and don't
	     even give away an alphanumeric name"
	    ^name := self printString
	].
    ].
    ^name
!

asByteArray
    self subclassResponsibility
! !

!SocketAddress methodsFor: 'private'!

getName
    ^name
!

name: newName
    name := newName
! !

!IPAddress class methodsFor: 'initialization'!

createLocalAddress
    | localAddrBytes |
    self
	primAnyLocalAddress: localHostName
	in: (localAddrBytes := ByteArray new: 4).
    ^IPAddress fromBytes: localAddrBytes.
!

createLoopbackHost
    ^IPAddress fromBytes: #[127 0 0 1].
!

createUnknownAddress
    ^(IPAddress fromBytes: #[0 0 0 0])
	name: '0.0.0.0';
	yourself
! !

!IPAddress class methodsFor: 'constants'!

addressSize
    "Answer the size of an IPv4 address."
    ^4
!

version
    "Answer the version of IP that the receiver implements."
    ^4
! !

!IPAddress class methodsFor: 'instance creation'!

fromBytes: aByteArray
    "Answer a new IPAddress from a ByteArray containing the bytes
     in the same order as the digit form: 131.175.6.2 would be
     represented as #[131 175 6 2]."
    ^self basicNew
	address: (aByteArray copy makeReadOnly: true)
!

fromSockAddr: aByteArray port: portAdaptor
    "Private - Answer a new IPAddress from a ByteArray containing a
     C sockaddr_in structure.  The portAdaptor's value is changed
     to contain the port that the structure refers to."
    | s |
    s := CSockAddrStruct sizeof.
    portAdaptor value:
	(aByteArray at: s - 13) * 256 + (aByteArray at: s - 12).
    
    ^self fromBytes: (aByteArray copyFrom: s - 11 to: s - 8)
!

fromString: aString
    "Answer a new IPAddress from a String containing the requested
     address in digit form.  Hexadecimal forms are not allowed.
     
     An Internet host address is a number containing four bytes of data.
     These are divided into two parts, a network number and a local
     network address number within that network. The network number
     consists of the first one, two or three bytes; the rest of the
     bytes are the local address. 

     Network numbers are registered with the Network Information Center
     (NIC), and are divided into three classes--A, B, and C. The local
     network address numbers of individual machines are registered with
     the administrator of the particular network. 

     Class A networks have single-byte numbers in the range 0 to 127. There
     are only a small number of Class A networks, but they can each support
     a very large number of hosts (several millions). Medium-sized Class B
     networks have two-byte network numbers, with the first byte in the range
     128 to 191; they support several thousands of host, but are almost
     exhausted. Class C networks are the smallest and the most commonly
     available; they have three-byte network numbers, with the first byte
     in the range 192-223. Class D (multicast, 224.0.0.0 to 239.255.255.255)
     and E (research, 240.0.0.0 to 255.255.255.255) also have three-byte
     network numbers.
     
     Thus, the first 1, 2, or 3 bytes of an Internet address specifies a
     network. The remaining bytes of the Internet address specify the address
     within that network.  The Class A network 0 is reserved for broadcast to
     all networks. In addition, the host number 0 within each network is
     reserved for broadcast to all hosts in that network.  The Class A network
     127 is reserved for loopback; you can always use the Internet address
     `127.0.0.1' to refer to the host machine (this is answered by the
     #loopbackHost class method).

     Since a single machine can be a member of multiple networks, it can have
     multiple Internet host addresses. However, there is never supposed to be
     more than one machine with the same host address. 

     There are four forms of the standard numbers-and-dots notation for
     Internet addresses: a.b.c.d specifies all four bytes of the address
     individually; a.b.c interprets as a 2-byte quantity, which is useful for
     specifying host addresses in a Class B network with network address number
     a.b; a.b intrprets the last part of the address as a 3-byte quantity,
     which is useful for specifying host addresses in a Class A network with
     network address number a. 

     If only one part is given, this corresponds directly to the host address
     number."

    | substrings |
    substrings := aString substrings: $. .
    substrings := substrings collect: [ :each | each asInteger ].
    ^self fromArray: substrings
!

fromArray: parts
    "Answer a new IPAddress from an array of numbers; the numbers
     are to be thought as the dot-separated numbers in the standard
     numbers-and-dots notation for IPv4 addresses."

    | result last |
    result := ByteArray new: 4.

    "e.g. 2 parts (a.b): byte 1 are taken from a and b; byte 
     4 and 3 are bits 0-7 and 8-15 of c respectively; byte 2 is
     whatever remains (bits 16-23 is the string is well-formed).
     Handling (result at: parts size) specially simplifies
     error checking."

    1 to: parts size - 1 do: [ :i |
	result at: i put: (parts at: i) asInteger
    ].
    last := (parts at: parts size) asInteger.
    result size to: parts size + 1 by: -1 do: [ :i |
	result at: i put: last \\ 256.
	last := last // 256.
    ].
    result at: parts size put: last.
    ^self fromBytes: result
!

new
    self shouldNotImplement
!

with: b1 with: b2 with: b3 with: b4
    "Answer a new IPAddress whose bytes (from most-significant
     to least-significant) are in the parameters."
    ^self basicNew address:
	((ByteArray with: b1 with: b2 with: b3 with: b4) makeReadOnly: true)
! !

!IPAddress class methodsFor: 'private'!

isDigitAddress: aString
    "Answer whether the receiver is a valid address in a.b.c.d form."
    | dots |
    dots := 0.
    (aString substrings: $.) do: [ :part |
	dots := dots + 1.
	(part allSatisfy: [ :each | each isDigit ])
	    ifFalse: [ ^false ].

	part asInteger > 255 ifTrue: [ ^false ].
    ].
    ^dots = 4
! !


!IPAddress methodsFor: 'accessing'!

asByteArray
    "Answer a read-only ByteArray of size four containing the
     receiver's bytes in network order (big-endian)"
    ^address
!

addressClass
    "Answer the `address class' of the receiver (see
    IPAddress class>>#fromString:)"
    | net |
    net := address at: 1.
    net < 128 ifTrue: [ ^$A ].
    net < 192 ifTrue: [ ^$B ].
    net < 224 ifTrue: [ ^$C ].
    ^net < 240 ifTrue: [ $D ] ifFalse: [ $E ]
!

host
    "Answer an host number for the receiver; this is given by
     the last three bytes for class A addresses, by the last
     two bytes for class B addresses, else by the last byte."
    | net |
    net := address at: 1.
    net < 128 ifTrue: [
	^(address at: 4) + ((address at: 3) * 256) + ((address at: 2) * 65536)
    ].
    net < 192 ifTrue: [
	^(address at: 4) + ((address at: 3) * 256)
    ].
    ^address at: 4
!

network
    "Answer a network number for the receiver; this is given by the
     first three bytes for class C/D/E addresses, by the first two
     bytes for class B addresses, else by the first byte."
    | net |
    net := address at: 1.
    net < 128 ifTrue: [ ^net ].
    net < 192 ifTrue: [	^net * 256 + (address at: 2) ].
    ^net * 65536 + ((address at: 2) * 256) + (address at: 2)
!

subnet
    "Answer an host number for the receiver; this is 0 for class A
     addresses, while it is given by the last byte of the network
     number for class B/C/D/E addresses."
    | net |
    net := address at: 1.
    net < 128 ifTrue: [ ^(address at: 2) ].
    net < 192 ifTrue: [	^(address at: 3) ].
    ^0
!

isMulticast
    "Answer whether the receiver reprensents an address reserved for
     multicast datagram connections"
    ^(address at: 1) between: 224 and: 239
    "^self addressClass == $D"
! !

!IPAddress methodsFor: 'printing'!

printOn: aStream
    address
	do: [ :each | each printOn: aStream ]
	separatedBy: [ aStream nextPut: $. ]
! !

!IPAddress methodsFor: 'private'!

address: aByteArray
    address := aByteArray
!

port: port
    "Return a ByteArray containing a struct sockaddr for the given port
     on the IP address represented by the receiver. Family = AF_INET."

    (port < 0) | (port > 16rFFFF)
	ifTrue: [ self error: 'port out of range' ].

    ^(ByteArray new: CSockAddrStruct sizeof)

	"Write sin_addr"
	replaceFrom: CSockAddrStruct sizeof - 11
	to: CSockAddrStruct sizeof - 8
	with: address
	startingAt: 1;

	"Write sin_family = AF_INET in host order"
	shortAt: 1 put: self class afInet;

	"Write sin_port in network order (big endian)"
	at: CSockAddrStruct sizeof - 13 put: port // 256;
	at: CSockAddrStruct sizeof - 12 put: (port bitAnd: 255);

	"ouf..."
	yourself
! !
