"======================================================================
|
|   ANSI Protocol system implementation
|
|   This file is in the public domain.
|
 ======================================================================"

Object subclass: #ClassOrganizer
	instanceVariableNames: 'protocolNames '
	classVariableNames: 'ClassesDictionary '
	poolDictionaries: ''
	category: 'Kernel-Classes'!

Object subclass: #MsgParmSpec
	instanceVariableNames: 'parmName parmProtocols parmAliasingAttribute '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'System-Protocols'!

Object subclass: #MsgReturnSpec
	instanceVariableNames: 'returnValueProtocols returnValueAliasingAttribute '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'System-Protocols'!

MsgReturnSpec subclass: #MsgReturnRuleSpec
	instanceVariableNames: 'ruleSourceCode ruleBlock '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'System-Protocols'!

Object subclass: #ProtocolANYSpec
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'System-Protocols'!

Object subclass: #ProtocolMsgSpec
	instanceVariableNames: 'selector parameterSpecifications returnValueSpecifications specSections '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'System-Protocols'!

ProtocolANYSpec subclass: #ProtocolSpec
	instanceVariableNames: 'name conformsTo description messageSpecifications '
	classVariableNames: 'DefaultConvTable FixNum OperatorTable Protocols UnaryConvTable UndefinedConformsToNames '
	poolDictionaries: ''
	category: 'System-Protocols'!

!MsgParmSpec comment: '
1.1.2.2 Parameter Specification
	A parameter specification is defined by a parameter name, a parameter interface definition, and a parameter aliasing attribute.
	A parameter specification places constraints on the parameter in terms of protocol conformance, and provides information concerning how the parameter is used by implementations of the message. The parameter name is the name of a formal parameter and is used to identify the parameter with a parameter specification, and to refer to the parameter in textual descriptions.
A parameter interface definition is defined as either:

	* A single protocol name <P>.
	* A logical OR of two or more protocols, written as <P1> | <P2> | ... | <Pn>

	The parameter interface definition identifies the behavioral assumptions the message makes concerning the parameter. A client must supply an appropriate actual parameter. An OR of protocols means that the parameter must conform to at least one of the protocols in the disjunction. This is required to describe cases where a message accepts objects with diverse behavior and tests their behavior by sending messages in order to determine the action to be taken. Note that this is different from the case where a message accepts objects with diverse behavior, but only makes use of common shared behavior. In the latter case, the message is not really dealing with diverse cases of behavior.
	When a message specifies that a given formal parameter must conform to a protocol <P>, it is making a commitment to use only behavior which is defined in <P> in the message implementation. In this sense, the conformance statement is a maximal behavioral requirement-at most all of the behavior described by <P> will be used, and no more.
	Aliasing information (for example, whether a parameter is stored, or whether a returned value is new or returned state) is specified to avoid having implementors use defensive programming techniques which result in unnecessary object creation and copying, incurring a performance penalty.
	We differentiate between incidental aliasing and essential aliasing, both for parameters and for return values. Essential aliasing forms a critical part of the behavior of the interface, and as such it must be specified by the interface designer. Incidental aliasing should not be specified since it is a side effect of implementation choices, and is not fundamental to the specified functionality of the interface.
	Essential aliasing of parameters is described using a parameter aliasing attribute:

	captured	The receiver always retains a reference to the parameter,
				directly or indirectly, as a result of this message.

	uncaptured	The receiver never retains a reference to the parameter,
				directly or indirectly, as a result of this message.

	unspecified	It is unspecified as to whether or not a reference is retained
				as a result of this message i.e. either case may occur.
'!

MsgReturnSpec comment: '
1.1.2.3 Return value specification

	A return value specification is defined by a return value protocol and a return value aliasing attribute. Whereas the parameter description is prescriptive in that it states requirements to which the parameters must conform, the return value information is descriptive in that it provides information about the result being returned. Whereas a protocol makes a conformance requirement statement about parameters, it makes a conformance commitment concerning the return value. The specification guarantees that the return value will conform to the specified protocol.
	A message specification may have multiple distinct return value specifications. Conversely, a single return value specification may describe multiple return values if the return value specification applies to all such values. Multiple return value specifications are required for cases where a message is defined to return objects conforming to different protocols, on a case-specific basis. These are conveniently described with separate conformance statements and aliasing annotations. In order to establish correspondence between sets of return value specifications, we do not permit two distinct return value specifications which promise conformance to the same protocol.
	If a message specification has no return value specification (that is, the return value is  not specified), then it is not prepared to guarantee anything about the behavior of the returned object. In this case we denote the return value as UNSPECIFIED. This can be used to separate procedural messages from functional messages; to allow for inconsequential differences in implementations; or to allow conforming implementations which return different results but are otherwise operationally equivalent.
	In order to relate return values through conformance, we define the return value interface definition for a message specification to be the single return value protocol, or the logical OR of the protocols in each distinct return value specification.
	Information concerning retained references to return values (by the message receiver) is described using a return value aliasing attribute, which is one of the following identifiers:

	state	The receiver retains a reference (direct or indirect) to the
			returned object after the method returns i.e. the object is
			returned state.

	new	The object is newly created in the method invocation and no
			reference (direct or indirect) is retained by the receiver after
			the method returns.

	unspecified	No information is provided as to the origin or retained
			references to the object (Note this is different from saying that
			the return value itself is UNSPECIFIED.  Here we are committing
			that the return value conforms to some protocol, but making no
			commitment about the aliasing behavior).

	Note that we do not attempt to describe the aliasing of the state variables of the return value itself-the attribute applies only to the first level returned object. The implication is that second and subsequent level aliasing of the return value is always unspecified. An exception occurs in the case where the returned state is an object which the client originally gave the service provider for safekeeping. This occurs with element retrieval in collections, for example. In such cases only the client knows the implications of modifying second level state of the return value.
'!

MsgReturnRuleSpec comment: '
1.1 Protocol: <protocolMessageReturnRuleSpec>

Conforms To
	<abstractProtocolMessageReturnSpec>

Description
	This implements the behavior for determining the protocol of the value returned from a message send.  It contains Smalltalk source code of a block that when evaluated will execute the rule that returns the message return value protocol. 

Instance Variables:
	ruleSourceCode		<readableString>		The rule Smalltalk source code block that when evaluated will returns the message return value protocol.

Class Variables:
	none.
'!

ProtocolANYSpec comment: '
ProtocolANYSpec is a singleton instance of a special protocol to which all other protocols conform.
	The protocol <ANY> places no restrictions on a parameter definition since it allows all possible parameters.
	Protocol <ANY> may be thought of as a protocol which specifies no behavior.
'!

ProtocolMsgSpec comment: '
1.1.2 Message Specification

	A message specification describes an individual message in the context of a particular protocol. It is defined by a message selector, a behavioral description, a set of parameter specifications, and a set of return value specifications. A specification for a particular message may appear in arbitrarily many protocols, and no two message specifications in the same protocol may have the same message selector. A message specification has no meaning outside of the context of a protocol.
A selector names a message, and is denoted as such with a preceding # symbol. For example, #at:put: is a 2-parameter message selector. We distinguish a message from a method as follows:

	A selector, together with its parameters, is a message.
	A selector, together with a receiver object, identifies a method, the
	unique implementation of the message.

	Just as a method is uniquely identified by a receiver and selector, a message specification is uniquely identified by a protocol and selector pair (<P>, s) where P is a protocol name and s is a selector.

1.1.2.1 Behavioral Description

	The behavior of a message is described with English text, using a definitional style wherever possible. Basic operations are described in terms of their effects on the abstract state of the object (using terms described in the glossary). These form the building blocks for specifying the behavior of more complex messages, which may be described in terms of the basic messages.
	Words which are glossary entries are always in italics, and words which are formal parameter names are always in a fixed-pitch font. This eliminates confusion between a specific use of a word as defined in the glossary and normal English usage.

1.1.4 Protocol Specification Conventions

1.1.4.1 Naming

	A protocol''s name has its initial letter capitalized if there is a global name defined in the standard that is conformant to the protocol. For instance, <OrderedCollection> has its first letter capitalized but <puttableStream> does not.
	Protocols that are required to be implemented as class objects in Smalltalk implementations end with the word "class". Protocols that are typically implemented as class objects, but are not required to be so, end with either the word "factory", if they are used to create new objects, or the word "discriminator".

1.1.4.2 Message Lists

	Each protocol includes a list of the message selectors defined or refined by the protocol.  If a message is refined by the protocol it is shown in italics in this list.

1.1.4.3 Message Definitions

	Message definitions have these elements:

		A header that shows the message pattern. The message pattern is preceded by the word "Message:" or for refinements of messages defined in other protocols, "Message Refinement:".

		A synopsis, which is a short and informal description of what the message does, under the heading "Synopsis".

		A more rigorous definition of the message. The heading for this section, "Definition:", is followed by the name of the defining protocol. For refinements, the text of the inherited definition is merely copied.

		For each inherited refinement and the current protocol''s refinement, a refinement section showing how the method is refined. The heading for this section, "Refinement:", is followed by the name of the refining protocol.

		A list of the parameters of the message under the heading "Parameters", what their required protocol conformance is, and whether they are captured by the receiver of the message. Each parameter is listed on a separate line using the format:

		parameterName	<parametersProtocol>	captured/uncaptured/unspecified

If there are no parameters, this element is omitted.

		A description of the return value, under the heading "Return Value", in the form:

			<returnValueProtocol>	state/new/unspecified
		or
			UNSPECIFIED

		A list of errors that define erroneous conditions for the message under the heading "Errors".

For example,

Message:	canAcceptSalaryIncrease: amount
Synopsis
Determine whether the receiver can accept the given salary increase.
Definition: <Employee>
This message determines whether the receiver is allowed to receive the given salary increase.
It answers true if the elevated salary is acceptable and false if not.
Parameters
amount	<scaledDecimal>		uncaptured
Return Value
<boolean>	unspecified
Errors
none
or,
Message Refinement:	canAcceptSalaryIncrease: amount
Synopsis
Determine whether the receiver can accept the given salary increase.
Definition: <Person>
This message determines whether the receiver is allowed to receive the given salary increase.
It answers true if the elevated salary is acceptable and false if not.
Refinement: <Employee>
This refines the inherited message by checking the amount against known consistency rules for an employee object.
Parameters
amount	<scaledDecimal>		uncaptured
Return Value
<boolean>	unspecified
Errors
none
In the second example, the message is a refinement of the definition from protocol <Person> and is refined in <Employee>.
'!

ProtocolSpec comment: '
This is the starting point for the Protocol Model Protocols documentation.

???Note: This is an exparimental implementation of the proposed ANSI Smalltalk standard document type protocols.  Along with modifications to the SUnits testing framework it is primarily used to programatically test the return values of ANSI Smalltalk standard messages.
	The class and method comments may not be acurate and are rudimentary (class especially) as I''m not sure protocol object are that good of an idea.
	It should not conflict (I hope) with Dolphin Smalltalk protocol implementation.  I tried to keep it compatible with the spirit of their implementation.  I borrowed some ideas from it: holding only the ptotocol name in confoming class descriptions so the protocol objects could be stripped from an application(I hope).
	I tried to use message names similar to their implementation.
	It needs refactoring to consolidate some common behavior (ProtocolMsgSpec & ProtocolANYSpec, and MsgReturnSpec & MsgReturnRuleSpec).  Also it needs to consistantly accept, return, and print readableString protocol names (Symbol or String) always bracketed by < & >.

1. Protocol Model Protocols

	The graphs below shows the conformance relationships between the Protocol Model protocols.

			<Object>
				|
	----------------------------------------------------------------
	|				|				|
	|- <protocol>		<protocolManager>		<classProtocolDesc>
	|- <abstractProtocolMessageSpec>
	|				|
	|		------------------------
	|		|			|
	|	<protocolMessageSpec>	<protocolANYSpec>
	|
	|- <protocolMessageParmSpec>
	| -<abstractProtocolMessageReturnSpec>
					|
			---------------------------------
			|				|
	<protocolMessageReturnSpec>	<protocolMessageReturnRuleSpec>

			<???Object???>
				|
	---------------------------------------------------------
	|							|
	|- <protocol factory>			<protocolManager factory>
	|- <abstractProtocolMessageSpec factory>
	|				|
	|		-----------------------------------------
	|		|					|
	|	<protocolMessageSpec factory>	<protocolANYSpec factory>
	|
	|- <protocolMessageParmSpec factory>
	|- <abstractProtocolMessageReturnSpec factory>
					|
			---------------------------------
			|				|
<protocolMessageReturnSpec factory>	<protocolMessageReturnRuleSpec factory>

List of All Protocol Model Protocols & Short Desc.:
	<protocolManager factory>		creating protocol manager.
	<protocolManager>			protocol management ops.
	<protocol factory>			creating protocols.
	<protocol>				protocol ops.
	<protocolMessageSpec factory>		creating protocol messages.
	<abstractProtocolMessageSpec factory>
	<abstractProtocolMessageSpec>
	<protocolMessageSpec>			protocol message ops.
	<protocolANYSpec factory>
	<protocolANYSpec>
	<protocolMessageParmSpec factory>	creating protocol msg parm specs.
	<protocolMessageParmSpec>		protocol message parameter spec ops.
	<abstractProtocolMessageReturnSpec factory>
	<abstractProtocolMessageReturnSpec>
	<protocolMessageReturnSpec factory>	creating protocol msg return specs.
	<protocolMessageReturnSpec>		protocol message return value spec. ops.
	<protocolMessageReturnRuleSpec factory>	creating protocol msg return value rule specs.
	<protocolMessageReturnRuleSpec>		protocol msg return rule spec ops.
	<classProtocolDesc>			class protocol description msg return spec ops.

1.1 Protocol: <protocolManager>

Conforms To
	<Object>

Description
	The protocol manager is a singleton instance that manages all the protocols in the system.

Instance Variables:
	none.

Class Variables:

	Protocols					<Set>		The set of <protocol>s.
	UndefinedConformsToNames	<Set>		The set of protocol name <Symbol>s of <protocol>s not yet created but referenced by paramaters or return values.
	DefaultConvTable			<Dictionary>	A two dimensional table for determining the protocol type a message with operand and receiver should return.

1.2 Protocol: <protocol>

Conforms To
	<Object>

Description
	
A protocol is a behavioral description that is a named semantic interface, defined by a set of message specifications.  There may be only a single instance of each protocol.  Protocols are independent of implementation inheritance relationships, and are intended to clearly specify essential aspects of behavior while leaving incidental aspects unspecified.  The fact that something is explicitly unspecified may be important information to both implementors and application developers.

Instance Variables:

name conformsTo description messageSpecifications 

	name					<Symbol>		The <protocol> case-sensative unique name #todo (always bracketed by < & >).
	conformsTo				<Set>			The list of protocol names to which it conforms.
	description				<readableString>	The <protocol> description.
	messageSpecifications	<Set>			Its list of message specifications  <protocolMessageSpec>s (message selector, description, parameters, & return values, etc.).

Class Variables:
	none.

???Note: currently class ProtocolSpec acts as the protocol manager and instances are protocols.

Tables:
	The Default Result Type.

						Default Conversion Table:
				(simplified for a single Float type) 

		   opera|nd	<integer>	<scaledDec>	<Fraction>	<Float>(e|d|q)
	--receiver---|-------------------------------------------------------
	<integer>	|	<integer>	<scaledDec>	<Fraction>	<Float>e
	<scaledDec>	|	<scaledDec>	<scaledDec>	<Fraction>	<Float>e
	<Fraction>	|	<Fraction>	<Fraction>	<Fraction>	<Float>e
	<Float>(e|d|q)	|	<Float>e		<Float>e		<Float>e		<Float>e

	The Unary Result Type

						Unary Conversion Table:

	receiver		|	result
	-------------------|------------------------
	<integer>		|	<rational>
	<Fraction>		|	<rational>
	<scaledDecimal>	|	<scaledDecimal>
	<Float>			|	<Float>
'!


!Object methodsFor: 'constants' stamp: 'rej 5/24/2000 17:08'!
protocolManager
	"Answer the singleton instance of the protocol manager."
		#'ACSProS'.

	^ProtocolSpec
! !


!ClassDescription methodsFor: 'private' stamp: 'RAH 4/25/2000 19:48'!
organization
	^ClassOrganizer at: self!

protocolNames: protocolNameList 
	"Private - Set the names of protocols to which the receiver's instances directly conform to protocolNameList. 
	Note: This excludes inherited protocols. protocolNameList must be a <collection> of <symbol>s."
	#'ACSProS'.
	(protocolNameList isKindOf: Set)
		ifFalse: [self error: 'Protocol name list not a Set.'].
	protocolNameList isEmpty
		ifTrue: [self organization protocolNamesOrNil: nil]
		ifFalse: [self organization protocolNamesOrNil: protocolNameList]! !

!ClassDescription methodsFor: 'protocols' stamp: 'RAH 4/25/2000 19:48'!
addProtocolNamed: protocolName 
	"Add the protocol named, protocolName, to the list of protocols to which the receiver conforms."
	| tmpProtocolNames |
	#'ACSProS'.
	tmpProtocolNames := self protocolNames.
	tmpProtocolNames add: protocolName.
	self protocolNames: tmpProtocolNames! !

!ClassDescription methodsFor: 'protocols' stamp: 'RAH 4/25/2000 19:48'!
conformsToProtocolNamed: protocolName 
	"Answer whether the receiver conforms to the protocol named protocolName.
	Note: This includes inherited protocols."
	#'ACSProS'.
	^ (self protocolNames includes: protocolName)
		or: [self superclass notNil and: [self superclass conformsToProtocolNamed: protocolName]]! !

!ClassDescription methodsFor: 'protocols' stamp: 'RAH 4/25/2000 19:48'!
definedAsProtocolNames
	"Answer the names of protocols to which the receiver's instances directly conform.  If it has no protocols, check super classes until protocols are found 
	Note: This excludes inherited protocols. 
	200/03/04 Harmon, R. Added."
	| tmpList |
	#'ACSProS'.
	tmpList := self organization protocolNamesOrNil.
	tmpList isNil ifTrue: [^ self superclass definedAsProtocolNames].
	^ tmpList! !

!ClassDescription methodsFor: 'protocols' stamp: 'RAH 4/25/2000 19:48'!
protocolNames
	"Answer the names of protocols to which the receiver's instances directly conform. 
	Note: This excludes inherited protocols."
	| tmpList |
	#'ACSProS'.
	tmpList := self organization protocolNamesOrNil.
	tmpList isNil ifTrue: [^ self protocolManager defaultProtocolNameCollection].
	^ tmpList! !

!ClassDescription methodsFor: 'protocols' stamp: 'RAH 4/25/2000 19:48'!
protocols
	"Answer the protocols to which the receiver's instances directly conform. 
	Note: This excludes inherited protocols."
	#'ACSProS'.

	^ self protocolManager protocolsInNameList: self protocolNames! !

!ClassDescription methodsFor: 'protocols' stamp: 'RAH 4/25/2000 19:48'!
removeProtocolNamed: protocolName 
	"Remove the protocol named, protocolName, from the list of protocols to which the receiver conforms."
	#'ACSProS'.

	self removeProtocolNamed: protocolName ifAbsent: [self protocolManager errorProtocolNotFound: protocolName]! !

!ClassDescription methodsFor: 'protocols' stamp: 'RAH 4/25/2000 19:48'!
removeProtocolNamed: protocolName ifAbsent: notFoundBlock 
	"Remove the protocol named, protocolName, from the list of protocols to which the receiver conforms.  Evaluate notFoundBlock if not found."
	| tmpProtocolNames |
	#'ACSProS'.
	tmpProtocolNames := self protocolNames.
	tmpProtocolNames isEmpty ifTrue: [^ notFoundBlock value].
	tmpProtocolNames remove: protocolName ifAbsent: [^ notFoundBlock value].
	self protocolNames: tmpProtocolNames! !

!ClassOrganizer class methodsFor: 'private' stamp: 'RAH 4/25/2000 19:48'!
at: aClass
	ClassesDictionary isNil
		ifTrue: [ ClassesDictionary := LookupTable new ].

	^ClassesDictionary at: aClass ifAbsentPut: [ self new ]!

!ClassOrganizer methodsFor: 'private' stamp: 'RAH 4/25/2000 19:48'!
protocolNamesOrNil
	"Private - Answer the list of names of the protocols implemented directly by the receiver's instances (i.e. excluding inherited protocols), or nil if none."
	#'ACSProS'.

	^ protocolNames! !

!ClassOrganizer methodsFor: 'private' stamp: 'RAH 4/25/2000 19:48'!
protocolNamesOrNil: protocolNamesOrNil 
	"Private - Set the names of the protocols implemented directly by the receiver's instances (i.e. excluding inherited protocols) to protocolNamesOrNil, or nil if none."
	#'ACSProS'.

	protocolNames := protocolNamesOrNil! !


!MsgParmSpec methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
parmAliasingAttribute
	"Answer the protocol message parameter specification parameter aliasing attribute."

	^ parmAliasingAttribute! !

!MsgParmSpec methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
parmName
	"Answer the protocol message parameter specification parameter name."

	^ parmName! !

!MsgParmSpec methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
parmProtocolNames
	"Answer the protocol names this protocol message parameter specification conforms to. 
	Note: The protocol names is a <Set> of <symbol>s."

	^ parmProtocols! !

!MsgParmSpec methodsFor: 'private' stamp: 'RAH 4/25/2000 19:48'!
setParmName: name protocolNames: protocolNames aliasing: aliasingAttribute 
	"Private - ."
	parmName := name.
	parmProtocols := protocolNames.
	parmAliasingAttribute := aliasingAttribute! !

!MsgParmSpec methodsFor: 'printing' stamp: 'RAH 4/25/2000 19:48'!
printOn: targetStream 
	"Append to targetStream a text representation of the receiver as a developer would want to see it (inspector, etc)."

	targetStream nextPutAll: self class name;
	 nextPut: $(;
	 nextPutAll: self parmName;
	 space.
	self parmProtocolNames do: [:protocolName | targetStream nextPut: $<;
		 nextPutAll: protocolName;
		 nextPut: $>]
		separatedBy: [targetStream nextPutAll: '|'].
	targetStream space; nextPutAll: self parmAliasingAttribute; nextPut: $)! !

!MsgParmSpec methodsFor: 'displaying' stamp: 'RAH 4/25/2000 19:48'!
displayOn: targetStream 
	"Append the receiver to targetStream in a format that a user would 
	want to see."
	#todo."??? chg from developer sees to user sees???"

	self printOn: targetStream! !

!MsgParmSpec methodsFor: 'displaying' stamp: 'RAH 4/25/2000 19:48'!
displayString
	"Answer a text representation of the receiver as a user would want to see it (program browser, etc)."

	^ String streamContents: [:stream | self displayOn: stream]! !

!MsgParmSpec methodsFor: 'storing' stamp: 'RAH 4/25/2000 19:48'!
storeSIFOn: targetStream 
	"Append to targetStream, a <puttableStream>, the ASCII representation of the receiver in SIF from which the receiver can be rebuilt but NOT reinstantiated via evaluate."
	#todo."??? Find better way to keep SIF knowledge out of model ???"
	targetStream nextPutAll: '#('.
	parmName asString printOn: targetStream.
	targetStream space; nextPut: $'.
	parmProtocols do: [:protocolName | targetStream nextPutAll: protocolName asString]
		separatedBy: [targetStream space].
	targetStream nextPut: $'.
	targetStream nextPutAll: ' #'.
	parmAliasingAttribute asString printOn: targetStream.
	targetStream nextPutAll: ')'! !

!MsgParmSpec methodsFor: 'storing' stamp: 'RAH 4/25/2000 19:48'!
storeSIFString
	"Answer a <readableString>, a representation of the receiver in SIF from which the receiver can be rebuilt but NOT reinstantiated via evaluate."
	| aStream |
	aStream := WriteStream on: (String new: 50).
	self storeOn: aStream.
	^ aStream contents! !


!MsgParmSpec class methodsFor: 'private' stamp: 'RAH 4/25/2000 19:48'!
privateNewParmName: nameIn protocolNames: protocolNames aliasing: aliasingAttribute 
	"Private -"
	| protocolNamesTmp |
	(nameIn isKindOf: String)
		ifFalse: [self error: 'parameter name not a String.'].
	protocolNamesTmp := self privateValidProtocolNames: protocolNames ifError: [^ self error: 'Protocol msg.parameter protocol names not a <collection> of <symbol>s.'].
	^ super new
		setParmName: nameIn
		protocolNames: protocolNamesTmp
		aliasing: aliasingAttribute! !

!MsgParmSpec class methodsFor: 'private' stamp: 'RAH 4/25/2000 19:48'!
privateValidProtocolNames: protocolNamesIn ifError: errorBlock 
	"Private -"
	| protocolNamesTmp |
	(protocolNamesIn isKindOf: Collection)
		ifFalse: [^ errorBlock value].
	protocolNamesTmp := self protocolManager defaultProtocolNameCollection.
	protocolNamesIn
		do: 
			[:protocolName | 
			(protocolName isKindOf: Symbol)
				ifFalse: [^ errorBlock value].
			protocolNamesTmp add: protocolName].
	^ protocolNamesTmp! !

!MsgParmSpec class methodsFor: 'instance creation' stamp: 'RAH 4/25/2000 19:48'!
new
	"Raise an exception as this is an inappropriate message."

	^ self shouldNotImplement! !

!MsgParmSpec class methodsFor: 'instance creation' stamp: 'RAH 4/25/2000 19:48'!
newCapturedParmName: nameIn protocolNames: protocolNames 
	"Answer a new protocol message parameter specification for parameter named, nameIn, conforming to the protocols named, protocolNames, and with a captured aliasing attribute. 
	Note: protocolNames must be a <collection> of <symbol>s."

	^ self
		privateNewParmName: nameIn
		protocolNames: protocolNames
		aliasing: self parmAliasingAttributeCaptured! !

!MsgParmSpec class methodsFor: 'instance creation' stamp: 'RAH 4/25/2000 19:48'!
newParmName: nameIn protocolNames: protocolNames aliasing: attributeIn 
	"Answer a new protocol message parameter specification for parameter named, nameIn, conforming to the protocols named, protocolNames, and with an aliasing attribute, attributeIn. 
	Note: protocolNames must be a <collection> of <symbol>s."

	(self aliasingAttributes includes: attributeIn)
		ifFalse: [self error: 'Protocol msg. parameter aliasing attribute not valid.'].
	^ self
		privateNewParmName: nameIn
		protocolNames: protocolNames
		aliasing: attributeIn! !

!MsgParmSpec class methodsFor: 'instance creation' stamp: 'RAH 4/25/2000 19:48'!
newUncapturedParmName: nameIn protocolNames: protocolNames 
	"Answer a new protocol message parameter specification for parameter named, nameIn, conforming to the protocols named, protocolNames, and with a uncaptured aliasing attribute. 
	Note: protocolNames must be a <collection> of <symbol>s."

	^ self
		privateNewParmName: nameIn
		protocolNames: protocolNames
		aliasing: self parmAliasingAttributeUncaptured! !

!MsgParmSpec class methodsFor: 'instance creation' stamp: 'RAH 4/25/2000 19:48'!
newUnspecifiedParmName: nameIn protocolNames: protocolNames 
	"Answer a new protocol message parameter specification for parameter named, nameIn, conforming to the protocols named, protocolNames, and with a unspecified aliasing attribute. 
	Note: protocolNames must be a <collection> of <symbol>s."

	^ self
		privateNewParmName: nameIn
		protocolNames: protocolNames
		aliasing: self parmAliasingAttributeUnspecified! !

!MsgParmSpec class methodsFor: 'constants-aliasing attributes' stamp: 'RAH 4/25/2000 19:48'!
aliasingAttributes
	"Answer a list of protocol message parameter aliasing attribute constants.
	Note: The list is a <Set> of <symbol>s."

	^ Set
		with: self parmAliasingAttributeCaptured
		with: self parmAliasingAttributeUncaptured
		with: self parmAliasingAttributeUnspecified! !

!MsgParmSpec class methodsFor: 'constants-aliasing attributes' stamp: 'RAH 4/25/2000 19:48'!
parmAliasingAttributeCaptured
	"Answer the captured protocol message parameter specification parameter aliasing attribute constant."

	^ #'captured'! !

!MsgParmSpec class methodsFor: 'constants-aliasing attributes' stamp: 'RAH 4/25/2000 19:48'!
parmAliasingAttributeUncaptured
	"Answer the uncaptured protocol message parameter specification parameter aliasing attribute constant."

	^ #'uncaptured'! !

!MsgParmSpec class methodsFor: 'constants-aliasing attributes' stamp: 'RAH 4/25/2000 19:48'!
parmAliasingAttributeUnspecified
	"Answer the unspecified protocol message parameter specification parameter aliasing attribute constant."

	^ #'unspecified'! !


!MsgReturnSpec methodsFor: 'displaying' stamp: 'RAH 4/25/2000 19:48'!
displayOn: targetStream 
	"Append the receiver to targetStream in a format that a user would want to see."

	#todo."??? chg from developer sees to user sees???"
	self printOn: targetStream! !

!MsgReturnSpec methodsFor: 'displaying' stamp: 'RAH 4/25/2000 19:48'!
displayString
	"Answer a text representation of the receiver as a user would want to see it (program browser, etc)."

	^ String streamContents: [:stream | self displayOn: stream]! !

!MsgReturnSpec methodsFor: 'storing' stamp: 'RAH 4/25/2000 19:48'!
storeSIFOn: targetStream 
	"Append to targetStream, a <puttableStream>, the ASCII representation of the receiver in SIF from which the receiver can be rebuilt but NOT reinstantiated via evaluate."

	#todo."??? Find better way to keep SIF knowledge out of model ???"
	targetStream nextPutAll: '#('.
	targetStream space; nextPut: $'.
	returnValueProtocols do: [:protocolName | targetStream nextPutAll: protocolName asString]
		separatedBy: [', ' printOn: targetStream].
	targetStream nextPut: $'.
	targetStream nextPutAll: ' #'.
	returnValueAliasingAttribute asString printOn: targetStream.
	targetStream nextPutAll: ')'! !

!MsgReturnSpec methodsFor: 'storing' stamp: 'RAH 4/25/2000 19:48'!
storeSIFString
	"Answer a <readableString>, a representation of the receiver in SIF from which the receiver can be rebuilt but NOT reinstantiated via evaluate."
	| aStream |
	aStream := WriteStream on: (String new: 50).
	self storeOn: aStream.
	^ aStream contents! !

!MsgReturnSpec methodsFor: 'private' stamp: 'RAH 4/25/2000 19:48'!
setProtocolNames: protocolNames aliasing: aliasingAttribute 
	"Private - ."

	returnValueProtocols := protocolNames.
	returnValueAliasingAttribute := aliasingAttribute! !

!MsgReturnSpec methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
returnValueAliasingAttribute
	"Answer the protocol message return value aliasing attribute."

	^ returnValueAliasingAttribute! !

!MsgReturnSpec methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
returnValueProtocolNames
	"Answer the protocol names the protocol message return value conforms to."

	^ returnValueProtocols! !

!MsgReturnSpec methodsFor: 'printing' stamp: 'RAH 4/25/2000 19:48'!
printOn: targetStream 
	"Append to targetStream a text representation of the receiver as a developer would want to see it (inspector, etc)."

	targetStream nextPutAll: self class name;
	 nextPut: $(.
	self returnValueProtocolNames do: [:protocolName |
		targetStream nextPut: $<;
		 nextPutAll: protocolName;
		 nextPut: $>]
		separatedBy: [targetStream nextPutAll: '|'].
	targetStream space; nextPutAll: self returnValueAliasingAttribute; nextPut: $)! !

!MsgReturnSpec methodsFor: 'testing' stamp: 'RAH 4/25/2000 19:48'!
isConformingReturnClass: returnClass 
	"Answer true if the class, returnClass, of the result of sending a message conforms to the receiver, else false."

	self returnValueProtocolNames
		do: [:protocolName | (returnClass conformsToProtocolNamed: protocolName)
				ifTrue: [^ true]].
	^ false! !


!MsgReturnRuleSpec methodsFor: 'displaying' stamp: 'RAH 4/25/2000 19:48'!
displayOn: targetStream 
	"Append the receiver to targetStream in a format that a user would want to see."

	#todo."??? chg from developer sees to user sees???"
	self printOn: targetStream! !

!MsgReturnRuleSpec methodsFor: 'displaying' stamp: 'RAH 4/25/2000 19:48'!
displayString
	"Answer a text representation of the receiver as a user would want to see it (program browser, etc)."

	^ String streamContents: [:stream | self displayOn: stream]! !

!MsgReturnRuleSpec methodsFor: 'accessing' stamp: 'rej 5/27/2000 16:01'!
returnProtocolName: receiver

	^self ruleBlock value: receiver
! !

!MsgReturnRuleSpec methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
returnValueAliasingAttribute
	"Signal an error as the receiver specifies no behavior."

	self error: 'Protocol msg. return value rule specifies no aliasing attribute.'! !

!MsgReturnRuleSpec methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
returnValueProtocolNames
	"Signal an error as the receiver specifies no behavior."

	self error: 'Protocol msg. return value rule specifies no protocol.'! !

!MsgReturnRuleSpec methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
returnValueRuleBlockSource
	"Answer the rule block source code that when evaluated with appropiate values answers the protocol message return value conforms-to protocol name."

	^ ruleSourceCode! !

!MsgReturnRuleSpec methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
returnValueRuleBlockSource: blockSource 
	"Set the rule block source code that when evaluated with appropiate values answers the protocol message return value conforms-to protocol name."

	ruleSourceCode := blockSource! !

!MsgReturnRuleSpec methodsFor: 'accessing' stamp: 'rej 5/27/2000 16:02'!
ruleBlock

	self fixup.
	ruleBlock isNil ifTrue: [self fixup.  ruleBlock := Compiler evaluate: ruleSourceCode].
	^ruleBlock ! !

!MsgReturnRuleSpec methodsFor: 'private' stamp: 'rej 5/27/2000 16:03'!
fixup

	(ruleSourceCode includes: $^) ifFalse: [^self].
	ruleSourceCode := ruleSourceCode select: [:c | c ~~ $^].
	ruleBlock := nil
! !

!MsgReturnRuleSpec methodsFor: 'private' stamp: 'RAH 4/25/2000 19:48'!
newRetValRuleSourceCode: ruleBlockSource 
	"Private - ."

	ruleSourceCode := ruleBlockSource! !

!MsgReturnRuleSpec methodsFor: 'printing' stamp: 'RAH 4/25/2000 19:48'!
printOn: targetStream 
	"Append to targetStream a text representation of the receiver as a developer would want to see it (inspector, etc)."

	targetStream nextPutAll: self class name;
	 nextPut: $(;
	 nextPutAll: self returnValueRuleBlockSource;
	 nextPut: $)! !

!MsgReturnRuleSpec methodsFor: 'testing' stamp: 'rej 5/24/2000 16:43'!
isConformingReturnClass: returnClass ruleReceiver: receiver
	"Answer true if the class, returnClass, of the result of sending a message conforms to the receiver, else false."
	 | returnProtocolName |
#todo. "??? bug does not allow return in block ???"
	returnProtocolName := self ruleBlock value: receiver.
	^returnClass conformsToProtocolNamed: returnProtocolName.
! !

!MsgReturnRuleSpec methodsFor: 'testing' stamp: 'rej 5/24/2000 16:42'!
isConformingReturnClass: returnClass ruleReceiver: receiver operand: operand
	"Answer true if the class, returnClass, of the result of sending a message conforms to the receiver, else false."
	| returnProtocolName |
#todo. "??? bug does not allow return in block ???"
	returnProtocolName := self ruleBlock value: receiver value: operand.
	^returnClass conformsToProtocolNamed: returnProtocolName.
! !


!MsgReturnSpec class methodsFor: 'instance creation' stamp: 'RAH 4/25/2000 19:48'!
new
	"Raise an exception as this is an inappropriate message."

	^ self shouldNotImplement! !

!MsgReturnSpec class methodsFor: 'instance creation' stamp: 'RAH 4/25/2000 19:48'!
newRetValNewProtocolNames: protocolNames 
	"Answer a new return value specification indicating a protocol message's return value has a new aliasing attribute and conforms to the protocols named, protocolNames. 
	Note: protocolNames must be a <collection> of <symbol>s."

	^ self privateNewRetValProtocolNames: protocolNames aliasing: self returnValueAliasingAttributeNew! !

!MsgReturnSpec class methodsFor: 'instance creation' stamp: 'RAH 4/25/2000 19:48'!
newRetValProtocolNames: protocolNames aliasing: aliasingAttribute 
	"Answer a new return value specification indicating a protocol message's return value conforms to the protocols named, protocolNames, and has an aliasing attribute, aliasingAttribute. 
	Note: protocolNames must be a <collection> of <symbol>s."

	^ self privateNewRetValProtocolNames: protocolNames aliasing: aliasingAttribute! !

!MsgReturnSpec class methodsFor: 'instance creation' stamp: 'RAH 4/25/2000 19:48'!
newRetValStateProtocolNames: protocolNames 
	"Answer a new return value specification indicating a protocol message's return value has a state aliasing attribute and conforms to the protocols named, protocolNames. 
	Note: protocolNames must be a <collection> of <symbol>s."

	^ self privateNewRetValProtocolNames: protocolNames aliasing: self returnValueAliasingAttributeState! !

!MsgReturnSpec class methodsFor: 'instance creation' stamp: 'RAH 4/25/2000 19:48'!
newRetValUnspecifiedProtocolNames: protocolNames 
	"Answer a new return value specification indicating a protocol message's return value has a unspecified aliasing attribute and conforms to the protocols named, protocolNames."

	^ self privateNewRetValProtocolNames: protocolNames aliasing: self returnValueAliasingAttributeUnspecified! !

!MsgReturnSpec class methodsFor: 'private' stamp: 'RAH 4/25/2000 19:48'!
privateNewRetValProtocolNames: protocolNames aliasing: aliasingAttribute 
	"Private -"
	| protocolNamesTmp |
	(self aliasingAttributes includes: aliasingAttribute)
		ifFalse: [self error: 'Protocol msg. return value aliasing attribute not valid.'].
	protocolNamesTmp := self privateValidProtocolNames: protocolNames ifError: [^ self error: 'Protocol msg. return value protocol names not a <collection> of <symbol>s.'].
	^ super new setProtocolNames: protocolNamesTmp aliasing: aliasingAttribute! !

!MsgReturnSpec class methodsFor: 'private' stamp: 'RAH 4/25/2000 19:48'!
privateValidProtocolNames: protocolNamesIn ifError: errorBlock 
	"Private -"
	| protocolNamesTmp |
	(protocolNamesIn isKindOf: Collection)
		ifFalse: [^ errorBlock value].
	protocolNamesTmp := self protocolManager defaultProtocolNameCollection.
	protocolNamesIn
		do: 
			[:protocolName | 
			(protocolName isKindOf: Symbol)
				ifFalse: [^ errorBlock value].
			protocolNamesTmp add: protocolName].
	^ protocolNamesTmp! !

!MsgReturnSpec class methodsFor: 'constants-aliasing attributes' stamp: 'RAH 4/25/2000 19:48'!
aliasingAttributes
	"Answer a list of protocol message return value aliasing attribute constants.
	Note: The list is a <Set> of <symbol>s."
	^Set
		with: self returnValueAliasingAttributeNew
		with: self returnValueAliasingAttributeState
		with: self returnValueAliasingAttributeUnspecified
		! !

!MsgReturnSpec class methodsFor: 'constants-aliasing attributes' stamp: 'RAH 4/25/2000 19:48'!
returnValueAliasingAttributeNew
	"Answer a protocol message return value new aliasing attribute constant."

	^ #'new'! !

!MsgReturnSpec class methodsFor: 'constants-aliasing attributes' stamp: 'RAH 4/25/2000 19:48'!
returnValueAliasingAttributeState
	"Answer a protocol message return value state aliasing attribute constant."

	^ #'state'! !

!MsgReturnSpec class methodsFor: 'constants-aliasing attributes' stamp: 'RAH 4/25/2000 19:48'!
returnValueAliasingAttributeUnspecified
	"Answer a protocol message return value unspecified aliasing attribute constant."

	^ #'unspecified'! !


!MsgReturnRuleSpec class methodsFor: 'private' stamp: 'RAH 4/25/2000 19:48'!
privateNewRetValRuleSourceCode: ruleBlockSource 
	"Private - ."

	(ruleBlockSource isKindOf: String)
		ifFalse: [self error: 'Protocol msg. return value rule block source not a String.'].
	^ self basicNew newRetValRuleSourceCode: ruleBlockSource;
	 yourself! !

!MsgReturnRuleSpec class methodsFor: 'instance creation' stamp: 'RAH 4/25/2000 19:48'!
newRetValRuleSourceCode: ruleBlockSource 
	"Answer a new return value specification representing a protocol message's return value conforms-to protocol determined by evaluating the rule, ruleBlockSource, with appropiate values."

	^ self privateNewRetValRuleSourceCode: ruleBlockSource! !


!PositionableStream methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
nextToken
	"Answer a <readableString>, the next token in the receiver's element stream, delimited by elements which answer true to #isSeparator. Answer a nil if there are no more token in the receiver."
	"99/12/02 Harmon, R. A. 	Fixed error & changed to return nil if no
							more tokens to conform to other impls."
	| startPos len |
	#'ACSProS'.
	self skipSeparators.
	self atEnd ifTrue: [^ nil].
	startPos := self position.
	len := 0.
	[self atEnd]
		whileFalse: 
			[self next isSeparator
				ifTrue: 
					[self position: startPos.
					^ self next: len].
			len := len + 1].
	self position: startPos.
	^ self next: len! !


!ProtocolANYSpec methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
addUndefinedProtocolNames

	^ self! !

!ProtocolANYSpec methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
allConformsToProtocolNames
	"Answer the names of all protocols to which the receiver conforms including super protocols."
	| tmpList |
	tmpList := self conformsToProtocolNames.
	self conformsToProtocolNames do: [:aProtocollName | tmpList addAll: (self protocolManager protocolNamed: aProtocollName) allConformsToProtocolNames].
	^ tmpList! !

!ProtocolANYSpec methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
allMessageSelectors
	"Answer all of selectors which make up the receiver's protocol and all protocols to which the receiver conforms."
	| tmpList |
	tmpList := self messageSelectors.
	self allConformsToProtocolNames do: [:aProtocollName | tmpList addAll: (self protocolManager protocolNamed: aProtocollName) messageSelectors].
	^ tmpList! !

!ProtocolANYSpec methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
conformingBehaviors
	"Answer all classes in class then metaclass hierarchy order (i.e. superclasses first) as all conform to the receiver. 
	Note: Return value is a <OrderedCollection> of  class or metaclass objects."
	| answer |
	answer := OrderedCollection new: 10.
	Smalltalk allClasses do: [:class | 
		answer addLast: class.
		answer addLast: class class].
	^ answer! !

!ProtocolANYSpec methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
conformsToMessageSelectors
	"Answer all of selectors which make up all protocols to which the receiver conforms."
	| tmpList |
	tmpList := self protocolManager defaultMessageSpecificationCollection.
	self allConformsToProtocolNames do: [:aProtocollName | tmpList addAll: (self protocolManager protocolNamed: aProtocollName) messageSelectors].
	^ tmpList! !

!ProtocolANYSpec methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
conformsToProtocolNames
	"Answer an empty list of protocol names to which the receiver conforms."

	^ self protocolManager defaultConformsToCollection! !

!ProtocolANYSpec methodsFor: 'accessing' stamp: 'RAH 6/22/2000 11:45'!
messageOrNilAtSelector: selector
	"Answer nil <ANY> protocol by definition can't have any messages."
	"2000/06/23 Harmon, R. Added to fix bug when TestCaseANSI >>
			#assertSend: is sent with a selector not defined in the
			target protocol or any of its inherited protocols."

	^nil
! !

!ProtocolANYSpec methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
messageSelectors
	"Answer an empty list of of selectors which make up the receiver's protocol."

	^ self protocolManager defaultMessageSpecificationCollection! !

!ProtocolANYSpec methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
messages
	"Answer an empty list of of message specifications of the receiver."

	^ self protocolManager defaultMessageSpecificationCollection! !

!ProtocolANYSpec methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
protocolDescription
	"Answer a description of the receiver."

	^ 'A protocol to which all other protocols conform.' copy! !

!ProtocolANYSpec methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
protocolName
	"Answer the name of the receiver."

	^ self protocolManager protocolANYName! !

!ProtocolANYSpec methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
selectorsInBehavior: classOrMetaclass 
	"Answer an empty list of selectors ofcorresponding messages as all classes and metaclasses conform to the receiver but it specifies no behavior."

	^ Set new! !

!ProtocolANYSpec methodsFor: 'comparing' stamp: 'RAH 4/25/2000 19:48'!
<= comperand 
	"Answer whether the receiver is less than or equal to comperand. 
	Note: This is to allow protocols to be sorted with the default sort block."

	(comperand isKindOf: self protocolManager protocol)
		| (comperand isKindOf: self protocolManager protocolANY) ifFalse: [self error: 'Comperand not a ProtocolSpec.'].
	^ self protocolName asLowercase <= comperand protocolName asLowercase! !

!ProtocolANYSpec methodsFor: 'comparing' stamp: 'RAH 4/25/2000 19:48'!
= comperand 
	"Answer whether the receiver is considered equal (contains same elements) to comperand."

	^ (comperand isKindOf: self protocolManager protocol)
		and: [self protocolName == comperand protocolName]! !

!ProtocolANYSpec methodsFor: 'comparing' stamp: 'RAH 4/25/2000 19:48'!
hash
	"Answer the hash value for the receiver."

	^ self protocolName hash! !

!ProtocolANYSpec methodsFor: 'displaying' stamp: 'RAH 4/25/2000 19:48'!
displayOn: targetStream 
	"Append the receiver to targetStream in a format that a user would want to see."

	targetStream nextPut: $<;
	 nextPutAll: self protocolName;
	 nextPut: $>! !

!ProtocolANYSpec methodsFor: 'displaying' stamp: 'RAH 4/25/2000 19:48'!
displayString
	"Answer a text representation of the receiver as a user would want to see it (program browser, etc)."

	^ String streamContents: [:stream | self displayOn: stream]! !

!ProtocolANYSpec methodsFor: 'renaming' stamp: 'RAH 4/25/2000 19:48'!
renameToProtocolName: unused 
	"Signal an error as the receiver can not be renamed."

	self error: 'Protocol <' , self protocolName , '> can not be renamed.'! !

!ProtocolANYSpec methodsFor: 'removing' stamp: 'RAH 4/25/2000 19:48'!
removeAllSelectors: unused 
	"Signal an error as the receiver specifies no behavior."

	self error: 'Protocol <' , self protocolName , '> specifies no behavior.'! !

!ProtocolANYSpec methodsFor: 'removing' stamp: 'RAH 4/25/2000 19:48'!
removeSelector: unused 
	"Signal an error as the receiver specifies no behavior."

	self error: 'Protocol <' , self protocolName , '> specifies no behavior.'! !

!ProtocolANYSpec methodsFor: 'removing' stamp: 'RAH 4/25/2000 19:48'!
removeSelector: unused1 ifAbsent: unused2 
	"Signal an error as the receiver specifies no behavior."

	self error: 'Protocol <' , self protocolName , '> specifies no behavior.'! !

!ProtocolANYSpec methodsFor: 'printing' stamp: 'RAH 4/25/2000 19:48'!
printOn: targetStream 
	"Append to targetStream a text representation of the receiver as a developer would want to see it (inspector, etc)."

	targetStream nextPut: $<;
	 nextPutAll: self protocolName;
	 nextPut: $>;
	 nextPut: $(.
	self messageSelectors do: [:selector | targetStream nextPutAll: selector]
		separatedBy: [targetStream nextPutAll: ', '].
	targetStream nextPut: $)! !

!ProtocolANYSpec methodsFor: 'filing-ANSI SIF' stamp: 'RAH 4/25/2000 19:48'!
fileOutOnSIFFiler: programFiler 
	"Do nothing as the receiver is created by protocol initialization."

	^ self! !

!ProtocolANYSpec methodsFor: 'searching' stamp: 'RAH 4/25/2000 19:48'!
includesSelector: unused 
	"Answer false as the receiver specifies no behavior."

	^ false! !

!ProtocolANYSpec methodsFor: 'work misc.' stamp: 'RAH 4/25/2000 19:48'!
wrkAllConformsToMessageSelectorsTo: aDict visited: visitedProtocols 
	self messageSelectors do: [:aMessageSelector |
		(aDict includesKey: aMessageSelector)
			ifFalse: [aDict at: aMessageSelector put: self protocolName]].
	visitedProtocols add: self protocolName.
	self conformsToProtocolNames do: [:aProtoName |
		(visitedProtocols includes: aProtoName)
			ifFalse: [(self protocolManager protocolNamed: aProtoName)
				wrkAllConformsToMessageSelectorsTo: aDict
				visited: visitedProtocols]]! !


!ProtocolANYSpec class methodsFor: 'private' stamp: 'RAH 4/25/2000 19:48'!
privateNewProtocolANY
	"Private -"
	| newProtocol |
	newProtocol := self basicNew.
	^ newProtocol! !

!ProtocolANYSpec class methodsFor: 'instance creation' stamp: 'RAH 4/25/2000 19:48'!
new
	"Raise an exception as this is an inappropriate message."

	^ self shouldNotImplement! !


!ProtocolMsgSpec methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
allReferredToProtocolNames
	"Answer a list of protocol names referred to by the receiver."
	| referredToNames protocolName |
	referredToNames := Set new.
	protocolName := self definedInProtocolName.
	protocolName notNil ifTrue: [referredToNames add: protocolName].
	protocolName := self refinedInProtocolName.
	protocolName isNil ifFalse: [referredToNames add: protocolName].
	self specForEachParmList do: [:msgSpecParm | referredToNames addAll: msgSpecParm parmProtocolNames].
	self specForEachReturnValueList
		do: [:msgSpecReturn | (msgSpecReturn isKindOf: self protocolManager protocolMsgReturnValueRuleSpec)
				ifFalse: [referredToNames addAll: msgSpecReturn returnValueProtocolNames]].
	^ referredToNames! !

!ProtocolMsgSpec methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
definedInProtocolName
	"Answer the protocol name in which the receiver is defined, or nil."

	#todo."??? should this be the proto is component of or Definition: sec proto ???"
	specSections isNil ifTrue: [^ nil].
	^ specSections at: #'DefinedIn' ifAbsent: []! !

!ProtocolMsgSpec methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
messageDefinition
	"Answer the definition of the receiver, or an empty string."

	#todo."??? should this be the proto is component of or Definition: sec proto ???"
	specSections isNil ifTrue: [^ String new].
	^ specSections at: #'Definition' ifAbsent: [String new]! !

!ProtocolMsgSpec methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
messageErrors
	"Answer the errors of the receiver, or an empty string."

	#todo."??? should this be the proto is component of or Definition: sec proto ???"
	specSections isNil ifTrue: [^ String new].
	^ specSections at: #'Errors' ifAbsent: [String new]! !

!ProtocolMsgSpec methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
messagePattern
	"Answer the message pattern of the receiver."
	| aStream colonCnt parmNames |
	parmNames := (self specForEachParmList collect: [:msgParmSpec | msgParmSpec parmName]) asArray.
	(selector includes: $:)
		ifFalse: 
			[parmNames size = 0 ifTrue: [^ selector asString].
			parmNames size = 1 ifTrue: [^ selector asString , ' ' , (parmNames at: 1)].
			self error: 'Mis-matched parms & selector.'].
	aStream := WriteStream on: (String new: 200).
	colonCnt := 0.
	selector
		do: [:char | char = $:
				ifTrue: 
					[colonCnt := colonCnt + 1.
					aStream nextPutAll: ': '.
					aStream nextPutAll: (parmNames at: colonCnt).
					colonCnt = parmNames size ifFalse: [aStream space]]
				ifFalse: [aStream nextPut: char]].
	^ aStream contents! !

!ProtocolMsgSpec methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
messageRefinement
	"Answer the refinement of the receiver, or an empty string."

	#todo."??? should this be the proto is component of or Definition: sec proto ???"
	specSections isNil ifTrue: [^ String new].
	^ specSections at: #'Refinement' ifAbsent: [String new]! !

!ProtocolMsgSpec methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
messageSelector
	"Answer the selector of the receiver."

	^ selector! !

!ProtocolMsgSpec methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
messageSynopsis
	"Answer the synopsis of the receiver, or an empty string."

	#todo."??? should this be the proto is component of or Definition: sec proto ???"
	specSections isNil ifTrue: [^ String new].
	^ specSections at: #'Synopsis' ifAbsent: [String new]! !

!ProtocolMsgSpec methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
refinedInProtocolName
	"Answer the protocol name in which the receiver is refined, or nil."

	#todo."??? should this be the proto is component of or Refinement: sec proto ???"
	specSections isNil ifTrue: [^ nil].
	^ specSections at: #'RefinedIn' ifAbsent: []! !

!ProtocolMsgSpec methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
specForEachParmList
	"Answer the specification for each message parameter list of the receiver."

	parameterSpecifications isNil ifTrue: [^ self class defaultParameterSpecificationCollection].
	^ parameterSpecifications! !

!ProtocolMsgSpec methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
specForEachReturnValueList
	"Answer the specification for each message return value list of the receiver."

	returnValueSpecifications isNil ifTrue: [^ self class defaultReturnValueSpecificationCollection].
	^ returnValueSpecifications! !

!ProtocolMsgSpec methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
specSections
	"Answer the specification sections of the receiver. 
	Note: specSections must be a <Dictionary> of <symbol> keys and <readableString> values.  Keys are: #'Synopsis' #'DefinedIn' #'Definition' #'RefinedIn' #'Refinement' #'Errors'."

	specSections isNil ifTrue: [^ self protocolManager defaultSpecSectionsCollection].
	^ specSections! !

!ProtocolMsgSpec methodsFor: 'comparing' stamp: 'RAH 4/25/2000 19:48'!
<= comperand 
	"Answer whether the receiver's message selector is less than or equal to comperand's message selector. 
	Note: This is to allow protocol message selectors to be sorted with the default sort block."

	(comperand isKindOf: self protocolManager protocolMsgSpec)
		ifFalse: [self error: 'Comperand not a ProtocolSpec.'].
	^ self messageSelector <= comperand messageSelector! !

!ProtocolMsgSpec methodsFor: 'comparing' stamp: 'RAH 4/25/2000 19:48'!
= comperand 
	"Answer whether the receiver is considered equal (contains same elements) to comperand.  They are equal if both are instances of the same class and have the same message selector."

	#todo."I'm not sure this tests effectively for the same elements?????"
	^ (comperand isKindOf: self protocolManager protocolMsgSpec)
		and: [self messageSelector == comperand messageSelector]! !

!ProtocolMsgSpec methodsFor: 'comparing' stamp: 'RAH 4/25/2000 19:48'!
hash
	"Answer the hash value for the receiver."

	#todo."I'm not sure this tests effectively for the same elements?????"
	^ self messageSelector hash! !

!ProtocolMsgSpec methodsFor: 'testing' stamp: 'RAH 4/25/2000 19:48'!
hasParms
	"Answer true if receiver has parameter specifications, else false."

	^ parameterSpecifications notNil! !

!ProtocolMsgSpec methodsFor: 'testing' stamp: 'RAH 4/25/2000 19:48'!
hasReturnValue
	"Answer true if receiver has return value specifications, else false."

	^ returnValueSpecifications notNil! !

!ProtocolMsgSpec methodsFor: 'testing' stamp: 'RAH 4/25/2000 19:48'!
isConformingReturn: returnObject 
	"Answer true if the result, returnObject, of sending the receiver conforms to the specified return value, else false."
	| returnClass |
	#todo."??? is no return value an error or compliant ???"
	self hasReturnValue ifFalse: [^ true].
	self isReturnValueSpecByRule ifTrue: [^ false].
	returnClass := returnObject class.
	self specForEachReturnValueList
		do: [:returnSpec | (returnSpec isConformingReturnClass: returnClass)
				ifTrue: [^ true]].
	^ false! !

!ProtocolMsgSpec methodsFor: 'testing' stamp: 'RAH 4/25/2000 19:48'!
isConformingReturn: returnObject opRECEIVER: receiver conformTo: protocolName selector: msgSelector 
	"Answer true if the result, returnObject, of sending the receiver conforms to the protocol in which it is used, or any protocol that conforms to that protocol, else false."

	#todo."??? Figure out how to do this test ???"
	^ self isConformingReturn: returnObject! !

!ProtocolMsgSpec methodsFor: 'testing' stamp: 'RAH 4/25/2000 19:48'!
isConformingReturn: returnObject ruleReceiver: receiver 
	"Answer true if the result, returnObject, of sending the receiver conforms to the specified return value, else false."

	#todo."??? is no return value an error or compliant ???"
	self hasReturnValue ifFalse: [^ true].
	self isReturnValueSpecByRule ifFalse: [^ false].
	^ self specForEachReturnValueList asArray first isConformingReturnClass: returnObject class ruleReceiver: receiver! !

!ProtocolMsgSpec methodsFor: 'testing' stamp: 'RAH 4/25/2000 19:48'!
isConformingReturn: returnObject ruleReceiver: receiver operand: operand 
	"Answer true if the result, returnObject, of sending the receiver conforms to the specified return value, else false."

	#todo."??? is no return value an error or compliant ???"
	self hasReturnValue ifFalse: [^ true].
	self isReturnValueSpecByRule ifFalse: [^ false].
	^ self specForEachReturnValueList asArray first
		isConformingReturnClass: returnObject class
		ruleReceiver: receiver
		operand: operand! !

!ProtocolMsgSpec methodsFor: 'testing' stamp: 'rej 5/24/2000 17:06'!
isReturnValueSpecByRule
	"Answer true if the receiver return value protocol is detirmined by a rule, else false."

	returnValueSpecifications isNil ifTrue: [
		^false. 
	].
	^returnValueSpecifications anyOne isKindOf: (self protocolManager protocolMsgReturnValueRuleSpec)
! !

!ProtocolMsgSpec methodsFor: 'filing-ANSI SIF' stamp: 'RAH 4/25/2000 19:48'!
fileOutOnSIFFiler: programFiler protocol: protocolName 
	"File out the receiver definition and its message definitions on ANSI SIF filer, programFiler."
	| parmString returnIsRuleSw returnOrRule tmpStream |
	#todo. "??? Add annotations ???"
	parmString := '#()'.
	self hasParms
		ifTrue: 
			[tmpStream := WriteStream on: (String new: 200).
			tmpStream nextPutAll: '#( '.
			self specForEachParmList do: [:msgParmSpec | msgParmSpec storeSIFOn: tmpStream]
				separatedBy: [tmpStream space].
			tmpStream nextPutAll: ' )'.
			parmString := tmpStream contents].
	returnIsRuleSw := false.
	returnOrRule := '#()'.
	self hasReturnValue
		ifTrue: [self isReturnValueSpecByRule
				ifTrue: 
					[returnIsRuleSw := true.
					returnOrRule := self specForEachReturnValueList asArray first returnValueRuleBlockSource]
				ifFalse: 
					[tmpStream := WriteStream on: (String new: 200).
					tmpStream nextPutAll: '#( '.
					self specForEachReturnValueList do: [:msgReturnSpec | msgReturnSpec storeSIFOn: tmpStream]
						separatedBy: [tmpStream space].
					tmpStream nextPutAll: ' )'.
					returnOrRule := tmpStream contents]].
	programFiler
		fileOutProtocol: protocolName
		message: self messagePattern
		synopsis: self messageSynopsis
		definedIn: self definedInProtocolName
		definition: self messageDefinition
		refinedIn: self refinedInProtocolName
		refinement: self messageRefinement
		parameters: parmString
		returnIsRule: returnIsRuleSw
		returnValuesOrRule: returnOrRule
		errors: self messageErrors
		annotations: Dictionary new! !

!ProtocolMsgSpec methodsFor: 'displaying' stamp: 'RAH 4/25/2000 19:48'!
displayOn: targetStream 
	"Append the receiver to targetStream in a format that a user would want to see."

	#todo."??? chg from developer sees to user sees???"
	self printOn: targetStream! !

!ProtocolMsgSpec methodsFor: 'displaying' stamp: 'RAH 4/25/2000 19:48'!
displayString
	"Answer a text representation of the receiver as a user would want to see it (program browser, etc)."

	^ String streamContents: [:stream | self displayOn: stream]! !

!ProtocolMsgSpec methodsFor: 'printing' stamp: 'RAH 4/25/2000 19:48'!
printOn: targetStream 
	"Append to targetStream a text representation of the receiver as a developer would want to see it (inspector, etc)."

	#todo."??? fix ???"
	targetStream nextPutAll: self class name;
	 nextPut: $(;
	 nextPutAll: self messageSelector;
	 nextPut: $(.
	self specForEachParmList do: [:parmSpec | targetStream nextPutAll: parmSpec parmName]
		separatedBy: [targetStream space].
	targetStream nextPutAll: ') '.
	"	self specForEachReturnValueList 
	do: [ :returnSpec | targetStream nextPutAll: returnSpec parmName ] 
	separatedBy: [targetStream space].
	"
	targetStream nextPut: $)! !

!ProtocolMsgSpec methodsFor: 'private' stamp: 'RAH 4/25/2000 19:48'!
setSelector: selectorIn specSections: specSectionsIn specForEachParmList: parmSpecsIn specForEachReturnValueList: returnValueSpecsIn 
	"Private -  
	Note: Assumes all parms have been checked for validity."

	selector := selectorIn.
	specSections := specSectionsIn.
	parameterSpecifications := parmSpecsIn.
	returnValueSpecifications := returnValueSpecsIn! !


!ProtocolMsgSpec class methodsFor: 'default' stamp: 'RAH 4/25/2000 19:48'!
defaultParameterSpecificationCollection
	"Private - Answer a <Set>, the default parameter specification collection object."

	^ Set new! !

!ProtocolMsgSpec class methodsFor: 'default' stamp: 'RAH 4/25/2000 19:48'!
defaultReturnValueSpecificationCollection
	"Private - Answer a <Set>, the default return value specification collection object."

	^ Set new! !

!ProtocolMsgSpec class methodsFor: 'private' stamp: 'RAH 4/25/2000 19:48'!
privateMessagePatternParmListOrNil: parmSpecsIn selector: selectorIn 
	"Private - Answer the message pattern of the receiver."
	| aStream colonCnt parmNames parmSpecsTmp |
	parmSpecsIn isNil
		ifTrue: [parmSpecsTmp := Set new]
		ifFalse: [parmSpecsTmp := parmSpecsIn].
	parmNames := (parmSpecsTmp collect: [:msgParmSpec | msgParmSpec parmName]) asArray.
	(selectorIn includes: $:)
		ifFalse: 
			[parmNames size = 0 ifTrue: [^ selectorIn asString].
			parmNames size = 1 ifTrue: [^ selectorIn asString , ' ' , (parmNames at: 1)].
			self error: 'Mis-matched parms & selectorIn.'].
	aStream := WriteStream on: (String new: 200).
	colonCnt := 0.
	selectorIn do: [:char |
		char = $:
			ifTrue: 
				[colonCnt := colonCnt + 1.
				aStream nextPutAll: ': '.
				aStream nextPutAll: (parmNames at: colonCnt).
				colonCnt = parmNames size ifFalse: [aStream space]]
			ifFalse: [aStream nextPut: char]].
	^ aStream contents! !

!ProtocolMsgSpec class methodsFor: 'private' stamp: 'RAH 4/25/2000 19:48'!
privateNewSelector: selectorIn specSectionsOrNil: specSectionsIn specForEachParmOrListOrNil: parmSpecsIn specForEachReturnValueOrListOrNil: retValSpecsIn 
	"Private -"
	| newProtocolMsgSpec specSectionsTmp parmSpecsTmp retValSpecsTmp |
	(selectorIn isKindOf: Symbol)
		ifFalse: [self error: 'Protocol msg. spec. selector not a Symbol.'].
	specSectionsTmp := self privateValidSpecSectionsOrNil: specSectionsIn ifError: [^ self error: 'Protocol msg. spec. spec. sections not a Dictionary.'].
	parmSpecsTmp := self
				privateValidParmOrListOrNil: parmSpecsIn
				selector: selectorIn
				ifError: [^ self error: 'Protocol msg. spec. parm not a Collection of ProtocolMsgSpec or nil.'].
	retValSpecsTmp := self privateValidReturnValueOrListOrNil: retValSpecsIn ifError: [^ self error: 'Protocol ret. val. spec. not a Collection of MsgReturnSpec or nil.'].
	newProtocolMsgSpec := super basicNew.
	newProtocolMsgSpec
		setSelector: selectorIn
		specSections: specSectionsTmp
		specForEachParmList: parmSpecsTmp
		specForEachReturnValueList: retValSpecsTmp.
	^ newProtocolMsgSpec! !

!ProtocolMsgSpec class methodsFor: 'private' stamp: 'RAH 4/25/2000 19:48'!
privateValidParmOrListOrNil: parmSpecsIn selector: selectorIn ifError: errorBlock 
	"Private -"
	| parmSpecsTmp colonCnt |
	(parmSpecsIn isKindOf: self protocolManager protocolMsgParmSpec)
		ifTrue: [self privateMessagePatternParmListOrNil: (Set with: parmSpecsIn)
				selector: selectorIn]
		ifFalse: [self privateMessagePatternParmListOrNil: parmSpecsIn selector: selectorIn].
	parmSpecsIn isNil ifTrue: [^ nil].
	(parmSpecsIn isKindOf: self protocolManager protocolMsgParmSpec)
		ifTrue: 
			[parmSpecsTmp := self defaultParameterSpecificationCollection.
			parmSpecsTmp add: parmSpecsIn.
			^ parmSpecsTmp].
	(parmSpecsIn isKindOf: Collection)
		ifFalse: [^ errorBlock value].
	parmSpecsIn isEmpty ifTrue: [^ nil].
	colonCnt := (selectorIn select: [:char | char = $:]) size.
	colonCnt > 0
		ifTrue: [colonCnt = parmSpecsIn size ifFalse: [self error: 'Protocol msg. spec. number of parms do not match selector.']]
		ifFalse: [parmSpecsIn size = 0 | (parmSpecsIn size = 1) ifFalse: [self error: 'Protocol msg. spec. number of parms do not match selector.']].
	parmSpecsTmp := self defaultParameterSpecificationCollection.
	parmSpecsIn
		do: 
			[:parmSpec | 
			(parmSpec isKindOf: self protocolManager protocolMsgParmSpec)
				ifFalse: [^ errorBlock value].
			parmSpecsTmp add: parmSpec].
	^ parmSpecsTmp! !

!ProtocolMsgSpec class methodsFor: 'private' stamp: 'RAH 4/25/2000 19:48'!
privateValidReturnValueOrListOrNil: retValSpecsIn ifError: errorBlock 
	"Private -"
	| retValSpecsTmp |
	retValSpecsIn isNil ifTrue: [^ nil].
	(retValSpecsIn isKindOf: self protocolManager protocolMsgReturnValueSpec)
		ifTrue: 
			[retValSpecsTmp := self defaultReturnValueSpecificationCollection.
			retValSpecsTmp add: retValSpecsIn.
			^ retValSpecsTmp].
	(retValSpecsIn isKindOf: Collection)
		ifFalse: [^ errorBlock value].
	retValSpecsIn isEmpty ifTrue: [^ nil].
	retValSpecsTmp := self defaultReturnValueSpecificationCollection.
	retValSpecsIn
		do: 
			[:rvSpec | 
			(rvSpec isKindOf: self protocolManager protocolMsgReturnValueSpec)
				ifFalse: [^ errorBlock value].
			retValSpecsTmp add: rvSpec].
	^ retValSpecsTmp! !

!ProtocolMsgSpec class methodsFor: 'private' stamp: 'RAH 4/25/2000 19:48'!
privateValidSpecSectionsOrNil: specSectionsIn ifError: errorBlock 
	"Private -"
	specSectionsIn isNil ifTrue: [^ nil].
	(specSectionsIn isKindOf: Dictionary)
		ifFalse: [^ errorBlock value].
	specSectionsIn isEmpty ifTrue: [^ nil].
	^ specSectionsIn! !

!ProtocolMsgSpec class methodsFor: 'instance creation' stamp: 'RAH 4/25/2000 19:48'!
new
	"Raise an exception as this is an inappropriate message."

	^ self shouldNotImplement! !

!ProtocolMsgSpec class methodsFor: 'instance creation' stamp: 'RAH 4/25/2000 19:48'!
newSelector: selector specSections: specSections specsForEachParm: parmSpecs specsForEachReturnValue: retValSpecs 
	"Answer a new protocol message specification with selector, selector, specSections, specSections, a list of specifications for each parameter, parmSpecs, and a list of specifications for each return value, retValSpecs.
	Note: specSections must be a <Dictionary> of <symbol> keys and <readableString> values, parmSpecs must be a <collection> of <protocolMessageParmSpec>s, retValSpecs, a <collection> of <protocolMessageReturnSpec>s."

	^ self
		privateNewSelector: selector
		specSectionsOrNil: specSections
		specForEachParmOrListOrNil: parmSpecs
		specForEachReturnValueOrListOrNil: retValSpecs! !


!ProtocolSpec methodsFor: 'comparing' stamp: 'RAH 4/25/2000 19:48'!
<= comperand 
	"Answer whether the receiver is less than or equal to comperand. 
	Note: This is to allow protocols to be sorted with the default sort block."

	(comperand isKindOf: self protocolManager protocol)
		| (comperand isKindOf: self protocolManager protocolANY) ifFalse: [self error: 'Comperand not a ProtocolSpec.'].
	^ self protocolName asLowercase <= comperand protocolName asLowercase! !

!ProtocolSpec methodsFor: 'comparing' stamp: 'RAH 4/25/2000 19:48'!
= comperand 
	"Answer whether the receiver is considered equal (contains same elements) to comperand."

	#todo."I'm not sure this makes any sense (= is ==) because if they have the same name they have to be the same object.  If it makes sense, shouldn't this test if contains same selectors?????"
	^ (comperand isKindOf: self protocolManager protocol)
		and: [self protocolName == comperand protocolName]! !

!ProtocolSpec methodsFor: 'comparing' stamp: 'RAH 4/25/2000 19:48'!
hash
	"Answer the hash value for the receiver."

	^ self protocolName hash! !

!ProtocolSpec methodsFor: 'renaming' stamp: 'RAH 4/25/2000 19:48'!
renameToProtocolName: newName 
	"Rename the receiver protocol to have the new name, newName and update any conforming class or metaclass."
	| conformingList |
	(self protocolManager includesProtocolNamed: newName)
		ifTrue: [^ self error: 'Duplicate protocol name: "' , newName , '".'].
	conformingList := self conformingBehaviors.
	conformingList do: [:classOrMetaclass | classOrMetaclass removeProtocolNamed: self protocolName].
	self setProtocolName: newName.
	self protocolManager privateRehashProtocols.
	conformingList do: [:classOrMetaclass | classOrMetaclass addProtocolNamed: newName]! !

!ProtocolSpec methodsFor: 'private' stamp: 'RAH 4/25/2000 19:48'!
addUndefinedProtocolNames
	"Private - ."

	messageSpecifications isNil ifTrue: [^ self].
	messageSpecifications do: [:msgSpec | self addUndefinedProtocolNamesInMsgSpec: msgSpec]! !

!ProtocolSpec methodsFor: 'private' stamp: 'RAH 4/25/2000 19:48'!
addUndefinedProtocolNamesInMsgSpec: aProtocolMsgSpec 
	"Private - ."

	aProtocolMsgSpec allReferredToProtocolNames do: [:protoName |
		(self protocolManager includesProtocolNamed: protoName asSymbol)
			ifFalse: [self protocolManager addUndefinedProtocolName: protoName asSymbol]]! !

!ProtocolSpec methodsFor: 'private' stamp: 'RAH 4/25/2000 19:48'!
errorSelectorNotFound: selector 
	"Private -"
	self error: 'Protocol ' , self printString , ' message "' , selector , '" not found'! !

!ProtocolSpec methodsFor: 'private' stamp: 'RAH 4/25/2000 19:48'!
newProtocolName: protocolName conformsToProtocolNames: conformsToList
	"Private - ."

	name := protocolName.
	conformsTo := conformsToList! !

!ProtocolSpec methodsFor: 'private' stamp: 'RAH 4/25/2000 19:48'!
setProtocolName: protocolName 
	"Private - ."

	name := protocolName! !

!ProtocolSpec methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
conformingBehaviors
	"Answer all the classes which conform to the receiver in class hierarchy order (i.e. superclasses first). 
	Note: Return value is a <OrderedCollection> of  class or metaclass objects."
	| answer |
	answer := OrderedCollection new: 10.
	Smalltalk allClasses
		do: 
			[:class | 
			(class conformsToProtocolNamed: self protocolName)
				ifTrue: [answer addLast: class].
			(class class conformsToProtocolNamed: self protocolName)
				ifTrue: [answer addLast: class class]].
	^ answer! !

!ProtocolSpec methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
conformsToProtocolNames
	"Answer the protocol names to which the receiver conforms."

	^ conformsTo! !

!ProtocolSpec methodsFor: 'accessing' stamp: 'rej 5/24/2000 04:47'!
inheritedMessageOrNilAtSelector: selector
	| msg |
	conformsTo do: [:protocolName | msg := (ProtocolSpec protocolNamed: protocolName)
			messageOrNilAtSelector: selector.
			msg notNil ifTrue: [^msg]].
	^nil
! !

!ProtocolSpec methodsFor: 'accessing' stamp: 'rej 5/23/2000 22:57'!
messageAtSelector: selector
	"Answer the message spec. at selector."
	| msg |
	msg := self messageOrNilAtSelector: selector.
	msg notNil ifTrue: [^msg].
	self error: 'Protocol message spec. at selector: "', selector, '" not found.'
! !

!ProtocolSpec methodsFor: 'accessing' stamp: 'rej 5/24/2000 04:52'!
messageOrNilAtSelector: selector

	messageSpecifications isNil ifTrue: [
		^self inheritedMessageOrNilAtSelector: selector].
	^messageSpecifications 
		detect: [ :protocol | protocol messageSelector = selector]
		ifNone: [^self inheritedMessageOrNilAtSelector: selector].
! !

!ProtocolSpec methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
messageSelectors
	"Answer the list of selectors which make up the receiver's protocol. 
	Note: Return value is a <Set> of <symbol>s,"

	messageSpecifications isNil ifTrue: [
		^ self protocolManager defaultMessageSpecificationCollection].
	^ messageSpecifications collect: [:msgSpec | msgSpec messageSelector]! !

!ProtocolSpec methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
messages
	"Answer a list of message specifications of the receiver. 
	Note: Return value is a <Set> of <protocolMessageSpec>s,"

	messageSpecifications isNil ifTrue: [
		^ self protocolManager defaultMessageSpecificationCollection].
	^ messageSpecifications! !

!ProtocolSpec methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
protocolDescription
	"Answer a description of the receiver."

	description isNil ifTrue: [
		^ self protocolManager defaultEmptyDescription].
	^ description! !

!ProtocolSpec methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
protocolDescription: newDescription 
	"Set the receiver's description to newDescription. 
	Note: If newDescription is empty then description is set to nil."

	(newDescription isKindOf: String)
		ifFalse: [self error: 'Protocol description not a String.'].
	newDescription isEmpty
		ifTrue: [description := nil]
		ifFalse: [description := newDescription]! !

!ProtocolSpec methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
protocolName
	"Answer the name of the receiver."

	^ name! !

!ProtocolSpec methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
selectorsInBehavior: classOrMetaclass 
	"If the class or metaclass, classOrMetaclass, conforms to the receiver answer all the selectors in which have corresponding messages, else an empty list. 
	Note: Return value is a <Set> of <symbol>s,"

	(classOrMetaclass conformsToProtocolNamed: self protocolName)
		ifTrue: [^ (classOrMetaclass selectors select: [:selector | self includesSelector: selector]) asSet].
	^ Set new! !

!ProtocolSpec methodsFor: 'adding' stamp: 'RAH 4/25/2000 19:48'!
addAllMessages: protocolMsgSpecList 
	"Add if not already present all the protocol messages in the list, protocolMsgSpecList, to the set of messages included in the receiver's protocol.
	Note: protocolMsgSpecList must be a <collection> of <protocolMessageSpec>s,"

	protocolMsgSpecList do: [:msgSpec | self addMessage: msgSpec]! !

!ProtocolSpec methodsFor: 'adding' stamp: 'RAH 4/25/2000 19:48'!
addMessage: aProtocolMsgSpec 
	"Add if not already present the protocol message, aProtocolMsgSpec, to the set of messages included in the receiver's protocol."

	(aProtocolMsgSpec isKindOf: self protocolManager protocolMsgSpec)
		ifFalse: [self error: 'Protocol message not a ProtocolMsgSpec.'].
	(self includesSelector: aProtocolMsgSpec messageSelector)
		ifFalse: 
			[messageSpecifications isNil ifTrue: [messageSpecifications := self protocolManager defaultMessageSpecificationCollection].
			messageSpecifications add: aProtocolMsgSpec.
			self addUndefinedProtocolNamesInMsgSpec: aProtocolMsgSpec]! !

!ProtocolSpec methodsFor: 'filing-ANSI SIF' stamp: 'RAH 4/25/2000 19:48'!
fileOutOnSIFFiler: programFiler 
	"File out the receiver definition and its message definitions on ANSI SIF filer, programFiler."

	#todo."??? Add annotations ???"
	programFiler
		fileOutProtocolDefinitionOf: self protocolName
		conformsToProtocolNames: self conformsToProtocolNames
		description: self protocolDescription
		annotations: Dictionary new.
	self messages asSortedCollection do: [:messageSpec | messageSpec fileOutOnSIFFiler: programFiler protocol: self protocolName]! !

!ProtocolSpec methodsFor: 'removing' stamp: 'RAH 4/25/2000 19:48'!
removeAllSelectors: selectorList 
	"After removing from the receiver all the messages with selectors, selectorList, answer them. 
	Note: selectorList must be a <collection> of  <symbol>s."
	| messageSpecs |
	messageSpecs := Set new.
	selectorList do: [:selector | messageSpecs add: (self removeSelector: selector)].
	^ messageSpecs! !

!ProtocolSpec methodsFor: 'removing' stamp: 'RAH 4/25/2000 19:48'!
removeSelector: selector 
	"Answer the message with selector, selector, after removing it from the receiver."

	^ self removeSelector: selector
		ifAbsent: [self errorSelectorNotFound: selector]! !

!ProtocolSpec methodsFor: 'removing' stamp: 'RAH 4/25/2000 19:48'!
removeSelector: selector ifAbsent: notFoundBlock 
	"Answer the message with selector, selector, after removing it from the receiver."
	| aProtocolMsgSpec |
	aProtocolMsgSpec := messageSpecifications
		detect: [:msgSpec | msgSpec messageSelector = selector]
		ifNone: [^ notFoundBlock value].
	^ messageSpecifications remove: aProtocolMsgSpec! !

!ProtocolSpec methodsFor: 'printing' stamp: 'RAH 4/25/2000 19:48'!
printOn: targetStream 
	"Append to targetStream a text representation of the receiver as a developer would want to see it (inspector, etc)."

	targetStream nextPut: $<;
	 nextPutAll: self protocolName;
	 nextPut: $>;
	 nextPut: $(.
	self messageSelectors do: [:selector | targetStream nextPutAll: selector]
		separatedBy: [targetStream nextPutAll: ', '].
	targetStream nextPut: $)! !

!ProtocolSpec methodsFor: 'displaying' stamp: 'RAH 4/25/2000 19:48'!
displayOn: targetStream 
	"Append the receiver to targetStream in a format that a user would want to see."

	targetStream nextPut: $<;
	 nextPutAll: self protocolName;
	 nextPut: $>! !

!ProtocolSpec methodsFor: 'displaying' stamp: 'RAH 4/25/2000 19:48'!
displayString
	"Answer a text representation of the receiver as a user would want to see it (program browser, etc)."

	^ String streamContents: [:stream | self displayOn: stream]! !

!ProtocolSpec methodsFor: 'zhold' stamp: 'RAH 4/25/2000 19:48'!
zremoveClass: classOrMetaclass 
	"Remove the receiver's name from the class or the metaclass, classOrMetaclass, list of protocol names. 	??not that great an idea???"

	classOrMetaclass removeProtocolNamed: self protocolName! !

!ProtocolSpec methodsFor: 'searching' stamp: 'RAH 4/25/2000 19:48'!
includesSelector: selector 
	"Answer whether the receiver includes the selector, selector."

	^ self messageSelectors includes: selector! !


!ProtocolSpec class methodsFor: 'default' stamp: 'RAH 4/25/2000 19:48'!
defaultConformsToCollection
	"Private - Answer an <Set>, the default conformsTo collection."

	^ Set new! !

!ProtocolSpec class methodsFor: 'default' stamp: 'RAH 4/25/2000 19:48'!
defaultEmptyDescription
	"Private - Answer an <String>, the default empty description."

	^ String new! !

!ProtocolSpec class methodsFor: 'default' stamp: 'RAH 4/25/2000 19:48'!
defaultMessageSpecificationCollection
	"Private - Answer an <Set>, the default messageSpecification collection."

	^ Set new! !

!ProtocolSpec class methodsFor: 'default' stamp: 'RAH 4/25/2000 19:48'!
defaultProtocolCollection
	"Private - Answer an <Set>, the default protocol collection."

	^ Set new! !

!ProtocolSpec class methodsFor: 'default' stamp: 'RAH 4/25/2000 19:48'!
defaultProtocolNameCollection
	"Private - Answer an <Set>, the default protocol name collection."

	^ Set new! !

!ProtocolSpec class methodsFor: 'default' stamp: 'RAH 4/25/2000 19:48'!
defaultSpecSectionsCollection
	"Private - Answer an <Dictionary>, the default SpecSections collection."

	^ Dictionary new! !

!ProtocolSpec class methodsFor: 'constants-etc.' stamp: 'RAH 4/25/2000 19:48'!
protocolANYName
	"Answer the protocol <ANY> name."

	^ #'ANY'! !

!ProtocolSpec class methodsFor: 'filing-ANSI SIF' stamp: 'RAH 4/25/2000 19:48'!
fileOutAllProtocolsSIFFiler: programFiler 
	"File out all protocol definitions on ANSI SIF filer, programFiler."
	| allProtos |
	allProtos := self allProtocols asSortedCollection.
	allProtos remove: (self protocolNamed: self protocolANYName)
		ifAbsent: [].
	self fileOutSIFAllProtocolsDescOnFiler: programFiler.
	allProtos do: [:protocol | protocol fileOutOnSIFFiler: programFiler]! !

!ProtocolSpec class methodsFor: 'filing-ANSI SIF' stamp: 'RAH 4/25/2000 19:48'!
fileOutSIFAllProtocolsToFileLocator: pathNameExt 
	"File out all protocol definitions in ANSI SIF to fileLocator, pathNameExt. 
	Note: Protocols are NOT defined in SIF, thus only a reader with macro enhancement will install protocols from the file.  All other conforming readers will ignore the contents as comments."
	| aFileStream programFiler |
	aFileStream := FileStream write: pathNameExt.
	
	[programFiler := self protocolManager newWriterOn: aFileStream.
	"SIF"
	self fileOutAllProtocolsSIFFiler: programFiler]
		ensure: [aFileStream close]! !

!ProtocolSpec class methodsFor: 'constants-classes' stamp: 'RAH 4/25/2000 19:48'!
protocol
	"Answer the protocol class object."

	^ Smalltalk at: #'ProtocolSpec'! !

!ProtocolSpec class methodsFor: 'constants-classes' stamp: 'RAH 4/25/2000 19:48'!
protocolANY
	"Answer the protocol <ANY> class object."

	^ Smalltalk at: #'ProtocolANYSpec'! !

!ProtocolSpec class methodsFor: 'constants-classes' stamp: 'RAH 4/25/2000 19:48'!
protocolMsgParmSpec
	"Answer the protocol message parameter specification class object."

	^ Smalltalk at: #'MsgParmSpec'! !

!ProtocolSpec class methodsFor: 'constants-classes' stamp: 'RAH 4/25/2000 19:48'!
protocolMsgReturnValueRuleSpec
	"Answer the protocol message return value rule specification class object."

	^ Smalltalk at: #'MsgReturnRuleSpec'! !

!ProtocolSpec class methodsFor: 'constants-classes' stamp: 'RAH 4/25/2000 19:48'!
protocolMsgReturnValueSpec
	"Answer the protocol message return value specification class object."

	^ Smalltalk at: #'MsgReturnSpec'! !

!ProtocolSpec class methodsFor: 'constants-classes' stamp: 'RAH 4/25/2000 19:48'!
protocolMsgSpec
	"Answer the protocol message specification class object."

	^ Smalltalk at: #'ProtocolMsgSpec'! !

!ProtocolSpec class methodsFor: 'initializing' stamp: 'RAH 4/25/2000 19:48'!
initialize
	"Class initialization.  Example: 
	 
	ProtocolSpec initialize
	"

	self initializeProtocols.
	self initializeDefaultConversionTable.
	self initializeUnaryConversionTable.
	self initializeOperatorTable! !

!ProtocolSpec class methodsFor: 'initializing' stamp: 'RAH 4/25/2000 19:48'!
initializeDefaultConversionTable
	"Discard all existing protocols.  Example: 
	 
	self protocolManager protocol initializeDefaultConversionTable
	"
	| tmp |
	DefaultConvTable := Dictionary new.
	tmp := Dictionary new.
	tmp at: #'integer' put: #'integer'.
	tmp at: #'scaledDecimal' put: #'scaledDecimal'.
	tmp at: #'Fraction' put: #'Fraction'.
	tmp at: #'Float' put: #'Float'.
	DefaultConvTable at: #'integer' put: tmp.
	tmp := Dictionary new.
	tmp at: #'integer' put: #'scaledDecimal'.
	tmp at: #'scaledDecimal' put: #'scaledDecimal'.
	tmp at: #'Fraction' put: #'Fraction'.
	tmp at: #'Float' put: #'Float'.
	DefaultConvTable at: #'scaledDecimal' put: tmp.
	tmp := Dictionary new.
	tmp at: #'integer' put: #'Fraction'.
	tmp at: #'scaledDecimal' put: #'Fraction'.
	tmp at: #'Fraction' put: #'Fraction'.
	tmp at: #'Float' put: #'Float'.
	DefaultConvTable at: #'Fraction' put: tmp.
	tmp := Dictionary new.
	tmp at: #'integer' put: #'Float'.
	tmp at: #'scaledDecimal' put: #'Float'.
	tmp at: #'Fraction' put: #'Float'.
	tmp at: #'Float' put: #'Float'.
	DefaultConvTable at: #'Float' put: tmp! !

!ProtocolSpec class methodsFor: 'initializing' stamp: 'RAH 4/25/2000 19:48'!
initializeOperatorTable
	"Discard all existing protocols.  Example: 
	 
	self protocolManager initializeOperatorTable
	"

	OperatorTable := Dictionary new.
	OperatorTable at: #'=' put: 'equalityOp'.
	OperatorTable at: #'==' put: 'identityOp'.
	OperatorTable at: #'~=' put: 'notEqualityOp'.
	OperatorTable at: #'~~' put: 'notIdentityOp'.
	OperatorTable at: #'&' put: 'andOp'.
	OperatorTable at: #'|' put: 'orOp'.
	OperatorTable at: #'<' put: 'lessThanOp'.
	OperatorTable at: #'<=' put: 'lessThanOrEqualToOp'.
	OperatorTable at: #'>' put: 'greaterThanOp'.
	OperatorTable at: #'>=' put: 'greaterThanOrEqualToOp'.
	OperatorTable at: #'*' put: 'multiplyOp'.
	OperatorTable at: #'+' put: 'addOp'.
	OperatorTable at: #'-' put: 'subtractOp'.
	OperatorTable at: #'/' put: 'divideOp'.
	OperatorTable at: #'//' put: 'integerDivideOp'.
	OperatorTable at: #'\\' put: 'remainderIntegerDivideOp'.
	OperatorTable at: #',' put: 'concatenateOp'! !

!ProtocolSpec class methodsFor: 'initializing' stamp: 'RAH 4/25/2000 19:48'!
initializeProtocols
	"Discard all existing protocols.  Example: 
	 
	self protocolManager protocol initializeProtocols
	"
	| nameList |
	(Protocols notNil and: [Protocols size > 1])
		ifTrue: 
			[nameList := String new.
			"Not just protocol <ANY>"
			(Protocols asArray copyFrom: 1 to: (3 min: Protocols size))
				do: [:protocol | nameList := nameList , protocol protocolName]
				separatedBy: [nameList := nameList , ', '].
			Protocols size > 3 ifTrue: [nameList := nameList , ' ...'].

			(self respondsTo: #portFunc) ifTrue: [
				(self portFunc promptYesNo: 'You are about to lose protocols (' , nameList , ').  Do it?')
					ifFalse: ["Do NOT discard existing protocols."
						^ self]]].

	UndefinedConformsToNames := self defaultProtocolNameCollection.
	Protocols := self defaultProtocolCollection.
	Protocols add: (Smalltalk at: #'ProtocolANYSpec') privateNewProtocolANY! !

!ProtocolSpec class methodsFor: 'initializing' stamp: 'RAH 4/25/2000 19:48'!
initializeUnaryConversionTable
	"Discard all existing protocols.  Example: 
	 
	self protocolManager protocol initializeDefaultConversionTable
	"

	UnaryConvTable := Dictionary new.
	UnaryConvTable at: #'integer' put: #'rational'.
	UnaryConvTable at: #'Fraction' put: #'rational'.
	UnaryConvTable at: #'rational' put: #'rational'.
	UnaryConvTable at: #'scaledDecimal' put: #'scaledDecimal'.
	UnaryConvTable at: #'Float' put: #'Float'! !

!ProtocolSpec class methodsFor: 'instance creation' stamp: 'RAH 4/25/2000 19:48'!
new
	"ProtocolSpecs must be unique for any particular name, and must be instantiated with the #name: method."

	^ self shouldNotImplement! !

!ProtocolSpec class methodsFor: 'instance creation' stamp: 'RAH 4/25/2000 19:48'!
newMessagePattern: messagePattern forProtocolNamed: protocolName synopsis: messageSynopsis definedIn: definedInProtocolName definition: messageDefinition refinedIn: refinedInProtocolName refinement: messageRefinement parameters: parmList returnRule: returnValueRule errors: messageErrors 
	"Create a new protocol message specification with message, messagePattern, etc. to protocol named protocolName."
	| newParmSpecs newReturnSpecs secDict |
	newParmSpecs := self parametersFromList: parmList.
	newReturnSpecs := Set with: (self protocolManager protocolMsgReturnValueRuleSpec newRetValRuleSourceCode: returnValueRule).
	secDict := self
				specSectionsFromSynopsis: messageSynopsis
				definedIn: definedInProtocolName
				definition: messageDefinition
				refinedIn: refinedInProtocolName
				refinement: messageRefinement
				errors: messageErrors.
	(self protocolNamed: protocolName)
		addMessage: (self protocolMsgSpec
				newSelector: (String methodSelector: messagePattern , ' ') asSymbol
				specSections: secDict
				specsForEachParm: newParmSpecs
				specsForEachReturnValue: newReturnSpecs)! !

!ProtocolSpec class methodsFor: 'instance creation' stamp: 'RAH 4/25/2000 19:48'!
newMessagePattern: messagePattern forProtocolNamed: protocolName synopsis: messageSynopsis definedIn: definedInProtocolName definition: messageDefinition refinedIn: refinedInProtocolName refinement: messageRefinement parameters: parmList returnValues: returnValuesList errors: messageErrors 
	"Create a new protocol message specification with message, messagePattern, etc. to protocol named protocolName."
	| newParmSpecs newReturnSpec secDict |
	newParmSpecs := self parametersFromList: parmList.
	newReturnSpec := returnValuesList collect: [:anArray | self protocolManager protocolMsgReturnValueSpec newRetValProtocolNames: (Set with: (anArray at: 1) asSymbol)
					aliasing: (anArray at: 2)].
	secDict := self
				specSectionsFromSynopsis: messageSynopsis
				definedIn: definedInProtocolName
				definition: messageDefinition
				refinedIn: refinedInProtocolName
				refinement: messageRefinement
				errors: messageErrors.
	(self protocolNamed: protocolName)
		addMessage: (self protocolMsgSpec
				newSelector: (String methodSelector: messagePattern , ' ') asSymbol
				specSections: secDict
				specsForEachParm: newParmSpecs
				specsForEachReturnValue: newReturnSpec)! !

!ProtocolSpec class methodsFor: 'instance creation' stamp: 'RAH 4/25/2000 19:48'!
newProtocolNamed: protocolName conformsToProtocolNames: conformsToList 
	"Answer a new protocol with the specified name, protocolName that conforms to the protocols named in conformsToList."

	^ self privateNewProtocolNamed: protocolName conformsToProtocolNames: conformsToList! !

!ProtocolSpec class methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
allProtocolNames
	"Answer the names of all Protocols in the system."

	^ (Protocols collect: [:protocol | protocol protocolName]) asSet! !

!ProtocolSpec class methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
allProtocolNamesSorted
	"Answer the names of all Protocols in the system sorted ignoring case."

	^ (Protocols collect: [:protocol | protocol protocolName]) asSortedCollection sortBlock: [:x :y | x asLowercase <= y asLowercase]! !

!ProtocolSpec class methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
allProtocols
	"Answer all Protocols in the system."

	^ Protocols! !

!ProtocolSpec class methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
defaultReturnProtocolNameReceiver: receiver operand: operand 
	"Answer the return value conforms-to protocol name of a message performing arithmetic, arithmetic progressions, and conversion on numerical quantities involving receiver and operand.
	Note: The result type of most numeric opeations is based upon the operand type.  The Default Result Type for all operand types except <Fraction> is the type to which the operands have been converted according to the Default ConversionTable.  If the converted operand type is <Fraction> the Default Result Type is <rational>.  In all cases where the type of the return value differs from the default result type  it is noted in the operation's description."
	| receiverProto operandProto convertedOperandType |
	receiverProto := self instanceProtocol: receiver.
	operandProto := self instanceProtocol: operand.
	convertedOperandType := (DefaultConvTable at: receiverProto)
				at: operandProto.
	convertedOperandType = #'Fraction' ifTrue: [^ #'rational'].
	^ convertedOperandType! !

!ProtocolSpec class methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
instanceProtocol: number 
	"Answer the conforms-to protocol name of number."

	#(integer Float Fraction scaledDecimal )
		do: [:protoName | (number class conformsToProtocolNamed: protoName)
				ifTrue: [^ protoName]].
	self error: 'Instance does not conforms to any numeric protocol.'! !

!ProtocolSpec class methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
protocolNamed: protocolName 
	"Answer the protocol named protocolName."

	(protocolName isKindOf: Symbol)
		ifFalse: [self error: 'Protocol name not a Symbol.'].
	^ Protocols detect: [:protocol | protocol protocolName = protocolName]
		ifNone: [self error: 'Protocol named: "' , protocolName , '" not found.']! !

!ProtocolSpec class methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
protocolsInNameList: protocolNameList 
	"Answer the list of protocols named in protocolNameList."

	^ (protocolNameList collect: [:protocolName | self protocolNamed: protocolName]) asSet! !

!ProtocolSpec class methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
unaryReturnProtocolNameReceiver: receiver 
	"Answer the return value conforms-to protocol name of a unary message performing arithmetic, arithmetic progressions, and conversion on numerical quantities sent to receiver."
	| receiverProto |
	receiverProto := self instanceProtocol: receiver.
	^ UnaryConvTable at: receiverProto! !

!ProtocolSpec class methodsFor: 'accessing' stamp: 'RAH 4/25/2000 19:48'!
undefinedConformsToProtocolNames
	"Answer the undefined conforms-to protocol names in the system."

	^ UndefinedConformsToNames! !

!ProtocolSpec class methodsFor: 'private' stamp: 'RAH 4/25/2000 19:48'!
addUndefinedProtocolName: protocolName 
	"Private - ."

	(protocolName isKindOf: Symbol)
		ifFalse: [self error: 'Protocol name not a Symbol.'].
	UndefinedConformsToNames add: protocolName! !

!ProtocolSpec class methodsFor: 'private' stamp: 'RAH 4/25/2000 19:48'!
errorProtocolNotFound: protocolName 
	"Private -"

	self error: 'Protocol <' , protocolName , '> not found'! !

!ProtocolSpec class methodsFor: 'private' stamp: 'RAH 4/25/2000 19:48'!
fileOutSIFAllProtocolsDescOnFiler: programFiler 
	"Private - File out an all-protocols program description in ANSI SIF format on the programFiler."
	| annotations |
	annotations := Dictionary new.
	annotations at: 'createdByApp' put: self portFunc dialectNameVersionString.
	annotations at: 'createdDateTime' put: self portFunc currentDateTimeString.
	programFiler
		fileOutProgramDescName: 'AllProto'
		prerequisiteProgramNames: (Set with: 'ACSProS')
		programAnnotations: annotations! !

!ProtocolSpec class methodsFor: 'private' stamp: 'RAH 4/25/2000 19:48'!
parametersFromList: parmList 
	"Private - Create a new protocol message specification with message, messagePattern, etc. to protocol named protocolName."
	| newParmSpecs names |
	newParmSpecs := parmList
				collect: 
					[:anArray | 
					names := (anArray at: 2) substrings collect: [:nameString | name asSymbol].
					self protocolManager protocolMsgParmSpec
						newParmName: (anArray at: 1)
						protocolNames: names
						aliasing: (anArray at: 3)].
	^ newParmSpecs! !

!ProtocolSpec class methodsFor: 'private' stamp: 'RAH 4/25/2000 19:48'!
privateNewProtocolNamed: protocolName conformsToProtocolNames: conforms 
	"Private -"
	| newProtocol conformsTmp |
	(self includesProtocolNamed: protocolName)
		ifTrue: [^ self error: 'Protocol named "' , protocolName , '" already exists.'].
	conformsTmp := self privateValidConformsToProtocolNames: conforms ifError: [^ self error: 'Protocol conforms-to list not a <collection> of existing protocol name <symbol>s.'].
	(protocolName isKindOf: Symbol)
		ifFalse: [self error: 'Protocol name not a Symbol.'].
	newProtocol := self basicNew.
	newProtocol newProtocolName: protocolName conformsToProtocolNames: conformsTmp.
	Protocols add: newProtocol.
	(UndefinedConformsToNames includes: protocolName)
		ifTrue: [UndefinedConformsToNames remove: protocolName].
	^ newProtocol! !

!ProtocolSpec class methodsFor: 'private' stamp: 'RAH 4/25/2000 19:48'!
privateRehashProtocols
	"Private -"

	Protocols rehash! !

!ProtocolSpec class methodsFor: 'private' stamp: 'RAH 4/25/2000 19:48'!
privateValidConformsToProtocolNames: protocolNamesIn ifError: errorBlock 
	"Private -"
	| protocolNamesTmp |
	(protocolNamesIn isKindOf: Collection)
		ifFalse: [^ errorBlock value].
	protocolNamesIn isEmpty ifTrue: [^ errorBlock value].
	protocolNamesIn isEmpty ifTrue: [^ errorBlock value].
	protocolNamesTmp := self protocolManager defaultProtocolNameCollection.
	protocolNamesIn
		do: 
			[:protocolName | 
			(protocolName isKindOf: Symbol)
				ifFalse: [^ errorBlock value].
			(self protocolManager includesProtocolNamed: protocolName)
				ifFalse: [self protocolManager addUndefinedProtocolName: protocolName].
			protocolNamesTmp add: protocolName].
	^ protocolNamesTmp! !

!ProtocolSpec class methodsFor: 'private' stamp: 'RAH 4/25/2000 19:48'!
specSectionsFromSynopsis: messageSynopsis definedIn: definedInProtocolName definition: messageDefinition refinedIn: refinedInProtocolName refinement: messageRefinement errors: messageErrors 
	"Private - Create a new specification sections with parms and return it."
	| secDict |
	secDict := self protocolManager defaultSpecSectionsCollection.
	definedInProtocolName isEmpty ifFalse: [secDict at: #'DefinedIn' put: definedInProtocolName].
	messageDefinition isEmpty ifFalse: [secDict at: #'Definition' put: messageDefinition].
	messageErrors isEmpty ifFalse: [secDict at: #'Errors' put: messageErrors].
	refinedInProtocolName isEmpty ifFalse: [secDict at: #'RefinedIn' put: refinedInProtocolName].
	messageRefinement isEmpty ifFalse: [secDict at: #'Refinement' put: messageRefinement].
	messageSynopsis isEmpty ifFalse: [secDict at: #'Synopsis' put: messageSynopsis].
	^ secDict! !

!ProtocolSpec class methodsFor: 'renaming' stamp: 'RAH 4/25/2000 19:48'!
renameProtocolNamed: oldName to: newName 
	"Rename the protocol named oldName to have the new name, newName. and update any conforming class or metaclass."
	| targetProtocol |
	targetProtocol := self protocolNamed: oldName.
	targetProtocol renameToProtocolName: newName! !

!ProtocolSpec class methodsFor: 'searching' stamp: 'RAH 4/25/2000 19:48'!
includesProtocolNamed: protocolName 
	"Answer whether the named protocol exists."

	(protocolName isKindOf: Symbol)
		ifFalse: [self error: 'Protocol name not a Symbol.'].
	Protocols detect: [:protocol | protocol protocolName = protocolName]
		ifNone: [^ false].
	^ true! !

!ProtocolSpec class methodsFor: 'removing' stamp: 'RAH 4/25/2000 19:48'!
purgeUnused
	"Answer a list of protocols which are not currently implemented by any class or metaclass in the system after removing them. 
	Note: This might take a while.  Example: 
	 
	1 protocolManager purgeUnused
	"
	#todo."Fix??? don't forget to leave <ANY> even though no class explicity conforms?? "
	self notYetImplemented! !

!ProtocolSpec class methodsFor: 'removing' stamp: 'RAH 4/25/2000 19:48'!
removeProtocolNamed: protocolName 
	"Remove the protocol named, protocolName, from the system. 
	Note: Protocol <ANY> can not be removed."
	| conformingList targetProtocol |
	protocolName = self protocolManager protocolANYName ifTrue: [self error: 'Protocol <' , protocolName , '> can not be removed.'].
	targetProtocol := self protocolNamed: protocolName.
	#todo."fix not to use conformingBehaviors (includes subclasses) but classDescription protocolNames includes: ???"
	conformingList := targetProtocol conformingBehaviors.
	conformingList
		do: [:classOrMetaclass | (classOrMetaclass protocolNames includes: protocolName)
				ifTrue: [classOrMetaclass removeProtocolNamed: protocolName]].
	Protocols remove: targetProtocol ifAbsent: [self error: 'Protocol named: "' , protocolName , '" not found.']! !

!ProtocolSpec class methodsFor: 'work misc.' stamp: 'RAH 4/25/2000 19:48'!
wrkAllRuleReturnValueList
	"Private - Answers a Dictionary of protocols containing messages with return value specified by a rule.  The key is the protocol name and the value is a Dictionarys of message selector keys and  return value spec. set with one MsgReturnRuleSpec containing the rule source code.  Example use: 
	 
	(FileStream readOnlyFileNamed: 
	'C:\Dev\ANSI\ANSIGood\Squeak\AProtos.st.chg' 
	) fileIn. 
	1 protocolManager wrkAllRuleReturnValueList
	"
	| protocolsWithRuleList msgSelRuleCodeList |
	protocolsWithRuleList := Dictionary new.
	self protocolManager allProtocols do: [:protocol | protocol messages
			do: [:msg | msg isReturnValueSpecByRule
					ifTrue: 
						[msgSelRuleCodeList := protocolsWithRuleList at: protocol protocolName ifAbsent: [protocolsWithRuleList at: protocol protocolName put: Dictionary new].
						msgSelRuleCodeList at: msg messageSelector put: msg specForEachReturnValueList]]].
	^ protocolsWithRuleList! !

!ProtocolSpec class methodsFor: 'work misc.' stamp: 'RAH 4/25/2000 19:48'!
wrkAssocGlobalGenUnitTestsForProtocolGroup: protocolGroupName protocolGlobalsClassSideGenTests: protocolGlobalsClassSideGenTests 
	"Private - Generate unit test stub classes for protocol group named, protocolGroupName.  Parameter protocolGlobalsClassSideGenTests array has the protocol name, correspnding global classes, instance or class side protocol ind, and generate unit test ind.  It will generate the methods specified by protocol and conforms-To protocols in the test class for each protocol global class (or protocol inheritance tree leaf).
	The test method naming convention I use is: 
	 
	instance side protocolmethods:	testInstXselector	testInstXselectorX 
	class side protocol methods:		testClsXselector	testClsXselectorX 
	conformsTo protocol methods:	testConToXselector testConToXselectorX 
	operator (+, ==, /, etc) methods:	testInstXadditionOp	testInstXfixOp1 
	 
	The operator test methods testInstXfixOp1 have to be hand editted to change test method selector to testInstXoperatorNameOp.  Example use: 
	
	"
	| assocGlobalClasses genUnitTestSw isClassSideProtocolSw protocolName |
	protocolGlobalsClassSideGenTests
		do: 
			[:parmArray | 
			protocolName := parmArray at: 1.
			assocGlobalClasses := parmArray at: 2.
			isClassSideProtocolSw := (parmArray at: 3)
						= 'class'.
			genUnitTestSw := (parmArray at: 4)
						= 'UT'.
			self
				wrkAssocProtocolNamed: protocolName
				toClassesNamed: assocGlobalClasses
				isClassSideProtocol: isClassSideProtocolSw.
			genUnitTestSw
				ifTrue: [self
						wrkGenTestClassForProtocol: protocolName
						inProtocolGroupNamed: protocolGroupName
						isClassSideProtocol: isClassSideProtocolSw]]! !

!ProtocolSpec class methodsFor: 'work misc.' stamp: 'RAH 4/25/2000 19:48'!
wrkAssocProtocolNamed: protocolName toClassesNamed: assocGlobalClassesArray isClassSideProtocol: isClassSideProtocolSw 
	"Private - Assign a protocol to a list of ANSI class globals."
	| classSymbol |
	assocGlobalClassesArray
		do: 
			[:className | 
			classSymbol := className asSymbol.
			isClassSideProtocolSw
				ifTrue: [(Smalltalk at: classSymbol) class addProtocolNamed: protocolName]
				ifFalse: [(Smalltalk at: classSymbol)
						addProtocolNamed: protocolName]]! !

!ProtocolSpec class methodsFor: 'work misc.' stamp: 'RAH 4/25/2000 19:48'!
wrkChkTestsForProtocol: protocolName inProtocolGroupNamed: protocolGroupName isClassSideProtocol: isClassSideProtocolSw 
	"Private - Generate test suite stub methods for the directly specified messages and all conforms-To protocols of protocol named, protocolName, in protocol group named, protocolGroupName.  Parameter isClassSideProtocolSw indicates an instance or class side protocol. 
	The test method naming convention I use is: 
	 
	instance side protocol methods:	testInstXselector	testInstXselectorX 
	class side protocol methods:		testClsXselector	testClsXselectorX 
	conformsTo protocol methods:	testConToXselector testConToXselectorX 
	operator (+, ==, /, etc) methods:	testInstXadditionOp	testInstXfixOp1 
	 
	The operator test methods testInstXfixOp1 have to be hand editted to change test method selector to testInstXoperatorNameOp.  Example use:
	"
	| protocol testSelNameSymbol protocolNameSymbolTmp sourceTmp aDict visitedProtocols testClass s1 s2 |
	protocol := 1 protocolManager protocolNamed: protocolName asSymbol.
	FixNum := 0.
	testClass := self wrkTestClassForProtocol: protocolName inProtocolGroupNamed: protocolGroupName.
	"Build a message selector->protocol dictionary of the directly specified and all conforms-To protocols messages"
	aDict := Dictionary new.
	visitedProtocols := Set new.
	protocol wrkAllConformsToMessageSelectorsTo: aDict visited: visitedProtocols.
	"Generate test suite methods"
	aDict
		keysAndValuesDo: 
			[:msgSel :aProtocolName | 
			testSelNameSymbol := (self wrkTestMethdodNameFrom: msgSel) asSymbol.
			(testClass includesSelector: testSelNameSymbol)
				ifTrue: 
					[sourceTmp := (testClass sourceMethodAt: testSelNameSymbol) asString.
					s1 := (sourceTmp indexOf: $<)
								+ 1.
					s2 := (sourceTmp indexOf: $>)
								- 1.
					protocolNameSymbolTmp := (sourceTmp copyFrom: s1 to: s2) asSymbol.
					self halt.
					protocolNameSymbolTmp = aProtocolName ifFalse: [self halt]]]! !

!ProtocolSpec class methodsFor: 'work misc.' stamp: 'RAH 4/25/2000 19:48'!
wrkGenTestClassForProtocol: protocolName inProtocolGroupNamed: protocolGroupName isClassSideProtocol: isClassSideProtocolSw 
	"Private - Generate test suite stub methods for the directly specified messages and all conforms-To protocols of protocol named, protocolName, in protocol group named, protocolGroupName.  Parameter isClassSideProtocolSw indicates an instance or class side protocol. 
	The test method naming convention I use is: 
	 
	instance side protocol methods:	testInstXselector	testInstXselectorX 
	class side protocol methods:		testClsXselector	testClsXselectorX 
	conformsTo protocol methods:	testConToXselector testConToXselectorX 
	operator (+, ==, /, etc) methods:	testInstXadditionOp	testInstXfixOp1 
	 
	The operator test methods testInstXfixOp1 have to be hand editted to change test method selector to testInstXoperatorNameOp.  Example use:
	"
	| protocol testSelName commentTmp aDict visitedProtocols testClass |
	protocol := 1 protocolManager protocolNamed: protocolName asSymbol.
	FixNum := 0.
	testClass := self wrkTestClassForProtocol: protocolName inProtocolGroupNamed: protocolGroupName.
	"Build a message selector->protocol dictionary of the directly specified and all conforms-To protocols messages"
	aDict := Dictionary new.
	visitedProtocols := Set new.
	protocol wrkAllConformsToMessageSelectorsTo: aDict visited: visitedProtocols.
	"Generate test suite methods"
	aDict
		keysAndValuesDo: 
			[:msgSel :aProtocolName | 
			testSelName := self wrkTestMethdodNameFrom: msgSel.
			commentTmp := '" <' , aProtocolName , '>#' , msgSel , ' "'.
			testClass compile: testSelName , '
	' , commentTmp , '
	#''' , protocolGroupName , '''.
'.
			testClass organization classify: testSelName asSymbol under: 'testing']! !

!ProtocolSpec class methodsFor: 'work misc.' stamp: 'RAH 4/25/2000 19:48'!
wrkGenerateMethdodsInTestClassNamed: testClassName forProtocolNamed: protocolName inProtocolGroupNamed: protocolGroupName isClassSideProtocol: isClassSideProtocolSw 
	"Private - Generate test suite stub methods in test class named, testClassName, for protocol named, protocolName, in protocol group named, protocolGroupName, that is an instance side protocol (isClassSideProtocolSw = false) or class side protocol (isClassSideProtocolSw = true). 
	I name instance side protocol test methods %testInstXselector% & %testInstXselectorX% for Class >>#selector & Class >>#selector:. 
	I name class side protocol test methods %testClsXselector% &  %testClsXselectorX% for Class >>#selector & Class >>#selector:. 
	I name operator (+, ==, /, etc) test methods %testInstXfixOp1% Class >>#selector & Class >>#+.   I number these fixes sequentially starting with 1. 
	The operator (+, ==, /, etc) test methods %testInstXfixOp1% have to be hand editted to change test method selector to %testInstXadditionOp% (identityOp, divisionOp, etc). 
	Example use: 
	 
	1 protocolManager 
	wrkGenerateMethdodsInTestClassNamed: 'CharacterFactoryProtocolTest' 
	forProtocolNamed: #'Character factory' 
	inProtocolGroupNamed: 'Numeric' 
	isClassSideProtocol: true. 
	1 protocolManager 
	wrkGenerateMethdodsInTestClassNamed: 'CharacterProtocolTest' 
	forProtocolNamed: #'Character' 
	inProtocolGroupNamed: 'Numeric' 
	isClassSideProtocol: false.
	"
	| protocol testSelName instOrClass classOrMetaclassObj fixNum commentTmp opTable |
	opTable := Dictionary new.
	opTable at: #'=' put: 'equalityOp'.
	opTable at: #'==' put: 'identityOp'.
	opTable at: #'~=' put: 'notEqualityOp'.
	opTable at: #'~~' put: 'notIdentityOp'.
	opTable at: #'&' put: 'andOp'.
	opTable at: #'|' put: 'orOp'.
	opTable at: #'<' put: 'lessThanOp'.
	opTable at: #'<=' put: 'lessThanOrEqualToOp'.
	opTable at: #'>' put: 'greaterThanOp'.
	opTable at: #'>=' put: 'greaterThanOrEqualToOp'.
	opTable at: #'*' put: 'multiplyOp'.
	opTable at: #'+' put: 'addOp'.
	opTable at: #'-' put: 'subtractOp'.
	opTable at: #'/' put: 'divideOp'.
	opTable at: #'//' put: 'integerDivideOp'.
	opTable at: #'\\' put: 'remainderIntegerDivideOp'.
	protocol := 1 protocolManager protocolNamed: protocolName.
	"Generate test suite class method >>#suite :"
	classOrMetaclassObj := (Smalltalk at: testClassName asSymbol) class.
	classOrMetaclassObj compile: 'suite
	| testSuite |
	testSuite := TestSuite new.
	self selectors do: [ :selector |
		(selector indexOfSubCollection: ''test'') = 1 ifTrue: [
			testSuite addTest: (self selector: selector)
	]	].
	^testSuite
'.
	classOrMetaclassObj organization classify: #'suite' under: 'instance creation'.
	"Generate test suite instance methods >>#testInstXetc or >>#testClsXetc :"
	isClassSideProtocolSw
		ifTrue: [instOrClass := 'Cls']
		ifFalse: [instOrClass := 'Inst'].
	fixNum := 0.
	protocol messageSelectors
		do: 
			[:msgSel | 
			msgSel isInfix
				ifTrue: [(opTable includesKey: msgSel)
						ifTrue: [testSelName := 'test' , instOrClass , 'X' , (opTable at: msgSel)]
						ifFalse: 
							[fixNum := fixNum + 1.
							testSelName := 'test' , instOrClass , 'XfixOp' , fixNum printString]]
				ifFalse: [testSelName := 'test' , instOrClass , 'X' , (msgSel asString
									collect: [:char | char = $:
											ifTrue: [$X]
											ifFalse: [char]])].
			commentTmp := '" ' , msgSel , ' "'.
			classOrMetaclassObj := Smalltalk at: testClassName asSymbol.
			classOrMetaclassObj compile: testSelName , '
	' , commentTmp , '
	#''' , protocolGroupName , '''.
'.
			classOrMetaclassObj organization classify: testSelName asSymbol under: 'testing']! !

!ProtocolSpec class methodsFor: 'work misc.' stamp: 'RAH 6/19/2000 15:18'!
wrkTestClassForProtocol: protocolName inProtocolGroupNamed: protocolGroupName 
	"Private - Answer a unit test class for protocol named, protocolName, in group named, protocolGroupName."
	| testClassName tmp classObj |
	testClassName := String new.
	tmp := protocolName substrings.
	tmp
		do: 
			[:protocolNameParts | 
			protocolNameParts at: 1 put: (protocolNameParts at: 1) asUppercase.
			testClassName := testClassName , protocolNameParts].
	testClassName := (testClassName , 'ANSITest') asSymbol.
	classObj := [(Smalltalk at: #'TestCaseProtocol')
				subclass: testClassName
				instanceVariableNames: ''
				classVariableNames: ''
				poolDictionaries: ''
				category: 'Tests-ANSI-' , protocolGroupName]
				on: Error
				do: 
					[:except | 
					except messageText: 'Error compiling class definition of : "' , testClassName , '" - ' , except description.
					^ except resignalAs: Warning].
	^ classObj! !

!ProtocolSpec class methodsFor: 'work misc.' stamp: 'RAH 4/25/2000 19:48'!
wrkTestMethdodNameFrom: messageSelector 
	"Private - Answer a generated test suite stub method name from messageSelector. fixNum which may be incremented if a test methdod name is generated that must be fixed up by hand."
	| testSelName |
	messageSelector isInfix
		ifTrue: [(OperatorTable includesKey: messageSelector)
				ifTrue: [testSelName := 'testX' , (OperatorTable at: messageSelector)]
				ifFalse: 
					[FixNum := FixNum + 1.
					testSelName := 'testXfixOp' , FixNum printString]]
		ifFalse: [testSelName := 'testX' , (messageSelector asString
							collect: [:char | char = $:
									ifTrue: [$X]
									ifFalse: [char]])].
	^ testSelName! !

!ProtocolSpec class methodsFor: 'examples' stamp: 'RAH 4/25/2000 19:48'!
xCommonTasks
	"The following are tasks comonly performed:"
	"Initialize the list of protocol objects:"
	1 protocolManager initializeProtocols.
	"Remove protocol objects not currently assigned to any class or metaclass 
	(Answers a list.  This might take a while.):"
	1 protocolManager purgeUnused.
	"Remove protocol object:"
	1 protocolManager removeProtocolNamed: #'classDescription'.
	"Answer the sorted list of protocol name symbols:"
	1 protocolManager allProtocolNamesSorted.
	"Answer the list of protocol name symbols:"
	1 protocolManager allProtocolNames.
	"Answer the list of undefined conforms-to protocol name symbols:"
	1 protocolManager undefinedConformsToProtocolNames.
	"Answer the list of conforming classes or metaclasses:"
	(1 protocolManager protocolNamed: #'integer') conformingBehaviors.
	"Answer the list of conforms-to protocol name symbols:"
	(self protocolManager protocolNamed: 'Object class' asSymbol) allConformsToProtocolNames.
	"-> Set (ANY Object instantiator classDescription )"
	"Answer the list of all of selectors which make up the protocol 
	and all protocols to which the it conforms:"
	(self protocolManager protocolNamed: 'Object class' asSymbol) allMessageSelectors.
	"Answer the list of all of selectors which make up the protocol:"
	(self protocolManager protocolNamed: 'Object class' asSymbol) conformsToMessageSelectors.
	1 protocolManager includesProtocolNamed: #'Object'.
	"-> true"
	"Assign or query classes or metaclasses:"
	(Smalltalk at: #'Symbol')
		addProtocolNamed: #'Object'.
	ExceptionSet removeProtocolNamed: #'exceptionSet'.
	ExceptionSet protocolNames.
	"-> Set (exceptionSet )"
	ExceptionSet class protocolNames.
	"->  Set ()"
	ExceptionSet conformsToProtocolNamed: #'Object'.
	"-> true"
	ExceptionSet class conformsToProtocolNamed: #'Object'.
	"-> true"
	Symbol conformsToProtocolNamed: #'Object'.
	"-> true"
	true class protocolNames.
	"-> Set ()"
	true class conformsToProtocolNamed: #'boolean'! !

!ProtocolSpec class methodsFor: 'examples' stamp: 'RAH 4/25/2000 19:48'!
xGlobalsImplementAllMsgsOfProtocolNamed: protoName 
	"Answer the a Dictionary containing the conforming global class or metaclass name and corresponding missing selector symbol of protocol named, protoName.  Example: 
	 
	1 protocolManager 
	xGlobalsImplementAllMsgsOfProtocolNamed: #'Character factory'
	"
	| protocol missingMsg conformingBehaviorObjs |
	protocol := 1 protocolManager protocolNamed: protoName.
	conformingBehaviorObjs := (1 protocolManager protocolNamed: 'Character factory' asSymbol) conformingBehaviors.
	missingMsg := Dictionary new.
	protocol messageSelectors do: [:msgSel | conformingBehaviorObjs
			do: [:classOrMetaclassObj | (classOrMetaclassObj includesSelector: msgSel)
					ifFalse: [missingMsg at: classOrMetaclassObj name put: msgSel]]].
	^ missingMsg! !

!ProtocolSpec class methodsFor: 'examples' stamp: 'RAH 4/25/2000 19:48'!
xMessagesAndReturnValuesOfProtocolNamed: protoName 
	"Answer the a string containing the messages and their corresponding return values of protocol named, protoName.  Example: 
	 
	1 protocolManager 
	xMessagesAndReturnValuesOfProtocolNamed: #'instantiator'
	"
	| protocol aStream |
	aStream := WriteStream on: (String new: 500).
	aStream cr.
	protocol := 1 protocolManager protocolNamed: protoName.
	protocol messages asSortedCollection
		do: 
			[:msg | 
			msg messageSelector printOn: aStream.
			aStream nextPutAll: '->'.
			msg specForEachReturnValueList printOn: aStream.
			aStream cr].
	aStream contents! !

!ProtocolSpec class methodsFor: 'examples' stamp: 'RAH 4/25/2000 19:48'!
xMsgSpecListOfSelector: selector inProtocolNamed: protoName 
	"Answer the list of msg specs for msg with selector, selector, in protocol named, protoName.  Example: 
	 
	1 protocolManager 
	xMsgSpecListOfSelector: #',' 
	inProtocolNamed: #'exceptionSelector'
	"
	| msgSpec proto |
	proto := self protocolManager protocolNamed: protoName.
	msgSpec := proto messageAtSelector: selector.
	^ msgSpec specForEachReturnValueList! !

!ProtocolSpec class methodsFor: 'examples' stamp: 'RAH 4/25/2000 19:48'!
xRemoveMsgSpecSelectors: selectorList fromProtocolNamed: protoName 
	"Answer the list of msg specs with selectors, selectorList, after removing the msg specs from protocol named, protoName.  Example: 
	1 protocolManager 
	xRemoveMsgSpecSelectors: #(#'name') 
	fromProtocolNamed: #'classDescription'
	"
	| proto |
	proto := self protocolManager protocolNamed: protoName.
	^ proto removeAllSelectors: selectorList! !

!ProtocolSpec class methodsFor: 'examples' stamp: 'RAH 4/25/2000 19:48'!
xRuleSourceOfMsgSelector: selector inProtocolNamed: protoName 
	"Answer the return value rule block source of msg with selector, selector, in protocol named, protoName.  Example: 
	 
	1 protocolManager 
	xRuleSourceOfMsgSelector: #'+' 
	inProtocolNamed: #'number'
	"
	| msgSpec proto |
	proto := self protocolManager protocolNamed: protoName.
	msgSpec := proto messageAtSelector: selector.
	^ msgSpec specForEachReturnValueList asArray first returnValueRuleBlockSource! !


!String class methodsFor: 'utility' stamp: 'RAH 4/25/2000 19:48'!
methodSelector: methodDefinitionString 
	"Private - Answer a <readableString>, the method selector extracted from the instance or class method definition methodDefinitionString if found, or <nil> if not found. 
	Note: This is not bullet proof.
	99/12/02 Harmon, R. A. 	Fixed error. 
	99/12/03 Harmon, R. A. 	Fixed %Definition:% accepted as keyword error."
	| sourceStream selectorStream token |
	#'ACSProS'.
	sourceStream := ReadStream on: methodDefinitionString trimSeparators.
	sourceStream contents isEmpty ifTrue: [^ nil].
	token := sourceStream nextToken.
	token isNil ifTrue: [^ nil].
	token last = $: ifFalse: ["Binary or unary selector."
		^ token].
	selectorStream := WriteStream on: (String new: 20).
	[(token isNil or: [token isEmpty]) not and: [token last = $: & ((token includes: $") not & (token includes: $') not & (token includes: $|) not)]]
		whileTrue: 
			[selectorStream nextPutAll: token.
			sourceStream nextToken.
			"Get keyword."
			token := sourceStream nextToken].
	^ selectorStream contents! !


ProtocolSpec initialize!
