"======================================================================
|
|   Smalltalk code pretty-printer
|
|   $Revision: 1.95.1$
|   $Date: 2000/12/27 10:45:49$
|   $Author: pb$
|
 ======================================================================"


"======================================================================
|
| Copyright 1988-92, 1994-95, 1999, 2000 Free Software Foundation, Inc.
| Written by Steve Byrne and Paolo Bonzini.
|
| This file is part of GNU Smalltalk.
|
| GNU Smalltalk is free software; you can redistribute it and/or modify it
| under the terms of the GNU General Public License as published by the Free
| Software Foundation; either version 2, or (at your option) any later version.
| 
| GNU Smalltalk is distributed in the hope that it will be useful, but WITHOUT
| ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
| FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
| details.
| 
| You should have received a copy of the GNU General Public License along with
| GNU Smalltalk; see the file COPYING.  If not, write to the Free Software
| Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
|
 ======================================================================"


Stream subclass: #IndentedStream
	   instanceVariableNames: 'stream level amount tabAmount column'
	   classVariableNames: 'Nl Tab'
	   poolDictionaries: ''
	   category: 'Examples-Useful'
!

!IndentedStream class methodsFor: 'instance creation'!

initialize
    Nl := Character nl.
    Tab := Character tab!

!IndentedStream class methodsFor: 'instance creation'!

on: aStream
    self initialize.
    ^self new initialize: aStream
! !

!IndentedStream methodsFor: 'writing'!

nextPut: aChar
    aChar == Nl ifTrue: [ column := 0. stream nl. ^self ].

    column == 0 ifTrue: [ self indentLine ].
    stream nextPut: aChar.
    column := aChar == Tab ifTrue: [ self nextTab ] ifFalse: [ column + 1 ].
!

nextPutAll: aString
    | stringSize index start ch |
    index := 1.
    stringSize := aString size.

    [ index <= stringSize ] whileTrue: [
	"Either the first character, or a newline, or a tab"
	self nextPut: (aString at: index).
	index := index + 1.
	start := index.
	  
	[   index > stringSize or: [
		ch := aString at: index.
		(ch = Nl) | (ch = Tab) 
	    ]
	] whileFalse: [ index := index + 1 ].

	column == 0 ifTrue: [ self indentLine ].
	stream nextPutAll: (aString copyFrom: start to: index - 1).
	column := column + (index - start)
    ].
!

tabTo: col
    | nextTab |
    nextTab := self nextTab.
    [ nextTab < col ] whileTrue: 
	[ stream nextPut: Tab. column := nextTab. nextTab := nextTab + 8 ].
    
    [ column < col ] whileTrue: 
	[ stream nextPut: $ . column := column + 1 ].
! !



!IndentedStream methodsFor: 'accessing'!

column
    ^column
!

level
    ^level
!

level: anInteger
    level := anInteger.
!

indentAmount
    ^amount
!

indentAmount: anInteger
    amount := anInteger.
!

popLevel
    self level >= 1 ifTrue: [ self level: self level - 1 ]
!

pushLevel
    self level: self level + 1
!

tabAmount
    ^tabAmount
!

tabAmount: anInteger
    tabAmount := anInteger.
! !



!IndentedStream methodsFor: 'private'!

initialize: aStream
    stream := aStream.
    column := 0.
    self
	indentAmount: 4;
	tabAmount: 8;
	level: 0
!

indentLine
    self tabTo: amount * level.
!

nextTab
    ^(column + tabAmount) truncateTo: tabAmount.
! !



STParseNodeVisitor subclass: #STFormatter
       instanceVariableNames: 'output selector state stack scopes savedComments putSpace '
       classVariableNames: 'IndentAmount'
       poolDictionaries: 'VMOtherConstants VMByteCodeNames'
       category: 'System-Compiler'
!

STFormatter comment:
'I provide the ability to pretty-print a Smalltalk parse tree.  My code
is horrible, full of hacks to get spacing and indentation right.  But I
work.

Known limitations:
- comments are not very well printed
- too many () with binary selectors
- code has no white lines between complex expressions
'!

!STFormatter class methodsFor: 'accessing'!

indentAmount
    IndentAmount isNil ifTrue: [ IndentAmount := 4 ].
    ^IndentAmount!

indentAmount: amount
    IndentAmount := amount! !

!STFormatter class methodsFor: 'parsing'!

reconstruct: aParseNode to: outputStream
    self new
	initialize: outputStream;
	visit: aParseNode!

print: aStream to: outputStream
    self new
	initialize: outputStream;
	visitMethodDefinition: aStream! !


!STFormatter methodsFor: 'disabled operations'!

visitDoitCode: aStream
    self shouldNotImplement! !


!STFormatter methodsFor: 'IndentedStream operation'!

popLevel
    output popLevel
!

pushLevel
    output pushLevel
!

nextPut: aCharacter
    | isNotSpace |
    isNotSpace := ((aCharacter == $ ) | (aCharacter value == 10)) not.
    putSpace & isNotSpace ifTrue: [ output space ].
    output nextPut: aCharacter.
    putSpace := isNotSpace.
!

nextPutAll: aString
    putSpace ifTrue: [ output space ].
    output nextPutAll: aString.
    putSpace := true.
!

reconstruct: anObject
    putSpace ifTrue: [ output space ].
    anObject reconstructOn: output.
    putSpace := true
!

space
    output nextPut: $ .
    putSpace := false.
!

nl
    output nextPut: Character nl.
    putSpace := false.
! !


!STFormatter methodsFor: 'private'!

checkEndOfBlockArgs
    state == #blockArgs ifTrue: [ self nextPut: $| ]
!

checkExcessiveLineLength
    output column > 40 ifTrue: [ self pushLevel; nl ]
!

outputSelectorPart: theSelector
    | pos size now |
    size := theSelector key size.
    now := theSelector value.
    pos := theSelector key indexOf: $: startingAt: now ifAbsent: [ size ].

    self nextPutAll: (theSelector key copyFrom: now to: pos).
    theSelector value: pos + 1.
    ^pos < size
!

prepareSelector: aSymbol
    ^aSymbol -> 1
!

initialize: aStream
    putSpace := false.
    scopes := OrderedCollection new.
    output := IndentedStream on: aStream.
    output indentAmount: self class indentAmount! !


!STFormatter methodsFor: 'comments'!

flushComments
    savedComments isNil ifFalse: [
	savedComments do: [ :each | self flushComment: each ] ].
    savedComments := OrderedCollection new.
!

flushComment: aString
    | paragraph stream endParagraph ch |
    output nextPut: $".
    (aString includes: Character nl)
	ifFalse: [
	    output column > 40
		ifTrue: [ self nl ]
		ifFalse: [ output tabTo: 40. putSpace := false ].

	    output nextPutAll: aString; nextPut: $".
	    self nl.
	    ^self
	].

    paragraph := ReadWriteStream on: String new.
    stream := ReadStream on: aString.
    [   paragraph nextPutAll: stream nextLine trimSeparators; space.
	endParagraph := stream atEnd or: [
	    (ch := stream peek) == Character cr
	    | (ch == Character nl)
	].
	endParagraph ifTrue: [
	    "The `skip: -1' eliminates the last space"
	    paragraph skip: -1; truncate; reset.
	    self format: paragraph.
	    stream atEnd ifTrue: [ output nextPut: $". self nl. ^self ].
	    stream peekFor: ch.
	    paragraph reset.
	].
    ] repeat.
!

format: aStream
    | start end ch |
    start := 1.
    [
	[
	    aStream atEnd ifTrue: [ ^self ].
	    ch := aStream next.
	    ch isSeparator ]
		whileTrue: [ output nextPut: ch ].

	start := aStream position - 1.
	[ aStream atEnd or: [ (ch := aStream next) isSeparator ] ] whileFalse.
	end := aStream position - 1.
	output column + end - start > 80 ifTrue: [ output nl ].
	output nextPutAll: (aStream copyFrom: start to: end).
    ] repeat
!

scopeEnter: startChar
    | savedState |
    self flushComments.
    savedState := output level -> stack.
    scopes add: savedState.

    stack := OrderedCollection new.
    savedState value isNil ifTrue: [ ^self ].
    self checkExcessiveLineLength.
    self nextPut: startChar.
    state := #block.
!

scopeLeave: endChar
    | savedState |
    self flushComments.
    savedState := scopes removeLast.
    stack := savedState value.
    scopes isEmpty ifTrue: [ ^self ].
    self checkEndOfBlockArgs.
    self nextPut: endChar.
    output level: savedState key.
!

! !


!STFormatter methodsFor: 'visiting particular nodes'!

scopeEnter
    self scopeEnter: $[
!

scopeLeave
    self scopeLeave: $]
!

argument: aSymbol
    | more |
    self flushComments.
    state == #selector ifFalse: [
        self nextPut: $:.
        putSpace := false.
        self nextPutAll: aSymbol.
        state := #blockArgs.
        ^self ].

    more := self outputSelectorPart: selector.
    self nextPutAll: aSymbol.

    more ifFalse: [ self nl; pushLevel ]
!

assignment: assignedNode
    self flushComments.
    stack isEmpty
	ifFalse: [ self nextPut: $(. putSpace := false ].

    self visit: assignedNode.
    self nextPutAll: ':='.
!

beginMessage: aMessageNode
    | needsParen |
    self flushComments.
    needsParen := stack isEmpty not and: [
	aMessageNode parenthesizeInside: stack last value ].

    stack add: needsParen -> aMessageNode.
    aMessageNode argumentCount > 0 ifTrue: [
	"Binary and keyword selectors might wrap to the next line"
	self checkExcessiveLineLength.
    ].

    needsParen ifTrue: [ self nextPut: $(. putSpace := false ].
!

braceArray: nodeCollection
    self
	scopeEnter: ${;
	sequence: nodeCollection;
	scopeLeave: $}
!

comment: aString
     savedComments add: aString
!

endMessage: aMessageNode
    self flushComments.
    stack removeLast key ifTrue: [ output nextPut: $) ].
    state := #receiver.
!

endOfStatement
    "Use `output' to force no spaces"
    output nextPut: $. .
    self nl.
    self flushComments.
    state := #receiver!

message: aMessageSendNode
    | mySelector |
    state == #message ifTrue: [ output nextPut: $; ].
    self flushComments.
    
    aMessageSendNode argumentCount = 0 ifTrue: [
	self nextPutAll: aMessageSendNode selector.
	state := #message.
	^self
    ].

    mySelector := self prepareSelector: aMessageSendNode selector.

    aMessageSendNode argumentCount > 1 ifTrue: [ self pushLevel ].
    aMessageSendNode allExpressionsDo: [ :argument |
	aMessageSendNode argumentCount > 1 ifTrue: [ self nl ].
	self outputSelectorPart: mySelector.
	self visit: argument.
    ].
    aMessageSendNode argumentCount > 1 ifTrue: [ self popLevel ].
    state := #message.
!

messageReceiver: anExpressionNode
    self flushComments.
    state := #receiver.
    self visit: anExpressionNode.
!

primitive: anIntegerOrNil
    self flushComments.
    state == #temporaries ifTrue: [ self nextPut: $|; nl ].
    state := #receiver.
    (anIntegerOrNil isNil or: [ anIntegerOrNil == 0 ])
	ifTrue: [ ^self ].

    self nextPutAll: '<primitive: '; print: anIntegerOrNil; nextPut: $>.
!

literal: anObject
    self flushComments.
    self checkExcessiveLineLength.
    state := #receiver.
    self reconstruct: anObject
!

return: aNode
    self flushComments.
    self nextPut: $^.
    putSpace := false.
    self visit: aNode
!

selector: aSymbol
    self flushComments.
    state := #selector.

    "For unary selectors, there are no calls to #argument:, so display the
     selector here. Other selectors are prepared to be split into parts."
    aSymbol asSymbol numArgs = 0
	ifTrue: [ self nextPutAll: aSymbol; pushLevel; nl ]
	ifFalse: [ selector := self prepareSelector: aSymbol ].
!

sequence: aCollection
    self flushComments.
    state == #temporaries
	ifTrue: [ self nextPut: $|; nl ]
	ifFalse: [ self checkEndOfBlockArgs ].

    (scopes size > 1) & (aCollection size > 1)
	ifTrue: [ self pushLevel; nl ].

    super sequence: aCollection!

temporary: aSymbol
    self flushComments.
    self checkEndOfBlockArgs.
    state == #temporaries ifFalse: [ self nextPut: $| ].
    state := #temporaries.
    self nextPutAll: aSymbol.
!

variable: aSymbol
    self flushComments.
    state := #receiver.
    self checkExcessiveLineLength.
    self nextPutAll: aSymbol
!

variableBinding: anOrderedCollection
    self flushComments.
    state := #receiver.
    self checkExcessiveLineLength.
    self nextPutAll: '#{'.
    anOrderedCollection
	do: [ :each | self nextPutAll: each ]
	separatedBy: [ self nextPut: $. ].

    self nextPut: $}.
! !


!Object methodsFor: 'storing'!

reconstructOn: aStream
    "Print the receiver as it appears in Smalltalk source.  The difference
     with #storeOn: is apparent, for example, for arrays, which print as
     #(a b c) rather than ((Array basicNew: 3) ...; yourself)."
    self storeOn: aStream!

innerReconstructOn: aStream
    "Print the receiver as it appears as an element of a literal array."
    self storeOn: aStream! !

!Array methodsFor: 'storing'!

reconstructOn: aStream
    aStream nextPut: $#.
    self innerReconstructOn: aStream!

innerReconstructOn: aStream
    aStream nextPut: $(.
    self
	do: [ :each | each innerReconstructOn: aStream ]
	separatedBy: [ aStream space ].

    aStream nextPut: $)! !

!ByteArray methodsFor: 'storing'!

reconstructOn: aStream
    aStream nextPut: $#.
    self innerReconstructOn: aStream!

innerReconstructOn: aStream
    aStream nextPut: $[.
    self
	do: [ :each | each printOn: aStream ]
	separatedBy: [ aStream space ].

    aStream nextPut: $]! !

!Symbol methodsFor: 'storing'!

innerReconstructOn: aStream
    self isSimpleSymbol
	ifTrue: [ aStream nextPutAll: self ]
	ifFalse: [ aStream print: self ]! !

!Association methodsFor: 'storing'!

reconstructOn: aStream
    "This is used for global variables -- just output the key."
    aStream nextPutAll: self key! !


!CompiledMethod methodsFor: 'formatting'!

prettyPrintedSourceString
    "Pretty print the method's source code (which is *not* decompiled) and
     return it."
    ^String streamContents: [ :stream | self prettyPrintOn: stream ]!

prettyPrintOn: aStream
    "Pretty print the method's source code (which is *not* decompiled) to
     aStream."
    STFormatter
	print: self methodSourceString readStream
	to: aStream! !


"This is just another cool example of usage of IndentedStream..."
IndentedStream subclass: #CStream
       instanceVariableNames: ''
       classVariableNames: ''
       poolDictionaries: ''
       category: 'Examples-Cool'
!

CStream comment: '
I emit indented C code.  The model that I have is that most of my 
operations emit things without the newline, leaving the newline and
consequently the indenting policy up to a few routines which make
all the indentation decisions.' !

!CStream class methodsFor: 'examples'!

example
    | s |
    s := self on: Transcript.
    s emitBlock: 
	[ s emitDecl: 'int' var: 'foo'; nl; nl.
    
	  s emitIf: 'foo > 3' then: 
	      [ s emitStmt: 'foo++'; nl.
		s emitIf: 'foo < 5' then:
		    [ s emitStmt: 'foo <<= 2'; nl.
		      s emitIf: 'foo & 0xff' then: [
			 s emitStmt: 'foo *= 5'; nl.
			 ].
		    ].
		s emitStmt: 'foo = sin(foo)'; 
		 emitLineComment: 'perhaps a bit unrealistic'; nl.
	      ].
	]
! !

!CStream methodsFor: 'cool hacks'!

nextPutBlock: aBlock
    aBlock value
!

!CStream methodsFor: 'accessing'!

emitBlock: aBlock
    self
	openBrace;
	nextPutBlock: aBlock;
	closeBrace
!

emitIf: expr then: aBlock
    self
	nextPutAll: 'if (';
	nextPutAll: expr;
	nextPutAll: ') ';
	emitBlock: aBlock.
!

emitElseIf: expr then: aBlock
    self
	nextPutAll: ' else ';
	emitIf: expr then: aBlock
!

emitElse: elseBlock
    self
	nextPutAll: ' else ';
	emitBlock: elseBlock.
!

emitWhile: expr do: aBlock
    self
	nextPutAll: 'while (';
	nextPutAll: expr;
	nextPutAll: ') ';
	emitBlock: aBlock.
!

emitWhile: expr
    self
	nextPutAll: 'while (';
	nextPutAll: expr;
	nextPutAll: '); '.
!

emitDo: aBlock while: expr
    self
	nextPutAll: 'do ';
	emitBlock: aBlock;
	nextPutAll: ' while (';
	nextPutAll: expr;
	nextPutAll: '); '.
!

emitFor: initExpr test: testExpr do: aBlock after: afterExpr
    self
	nextPutAll: 'for (';
	nextPutAll: initExpr;
	nextPutAll: '; ';
	nextPutAll: testExpr;
	nextPutAll: '; ';
	nextPutAll: afterExpr;
	nextPutAll: ') ';
	emitBlock: aBlock.
!

emitFor: initExpr test: testExpr after: afterExpr
    self
	nextPutAll: 'for (';
	nextPutAll: initExpr;
	nextPutAll: '; ';
	nextPutAll: testExpr;
	nextPutAll: '; ';
	nextPutAll: afterExpr;
	nextPutAll: '); '.
!

emitDecl: type var: var
    self nextPutAll: type; tabTo: 20; nextPutAll: var; nextPut: $;.
!

emitStmt: stmt
    self nextPutAll: stmt; nextPut: $; .
!

emitLineComment: text
    self tabTo: 40; emitComment: text.
!

emitComment: text
    self nextPutAll: '/* '; nextPutAll: text; nextPutAll: ' */'.
!

openBrace
    self nextPut: ${; nl.
    self pushLevel.
!

closeBrace
    self popLevel.
    self nextPut: $}; nl.
! !
