"======================================================================
|
|   LC_MESSAGES support (GNU gettext MO files)
|
|   $Revision: 1.95.1$
|   $Date: 2000/12/27 10:45:49$
|   $Author: pb$
|
 ======================================================================"


"======================================================================
|
| Copyright 1988-92, 1994-95, 1999, 2000 Free Software Foundation, Inc.
| Written by Paolo Bonzini.
|
| This file is part of the GNU Smalltalk class library.
|
| The GNU Smalltalk class library is free software; you can redistribute it
| and/or modify it under the terms of the GNU Lesser General Public License
| as published by the Free Software Foundation; either version 2.1, or (at
| your option) any later version.
|
| The GNU Smalltalk class library is distributed in the hope that it will be
| useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
| MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU Lesser
| General Public License for more details.
|
| You should have received a copy of the GNU Lesser General Public License
| along with the GNU Smalltalk class library; see the file COPYING.LESSER.
| If not, write to the Free Software Foundation, 59 Temple Place - Suite
| 330, Boston, MA 02111-1307, USA.
|
 ======================================================================"


Locale subclass: #LcMessages
    instanceVariableNames: ''
    classVariableNames: ''
    poolDictionaries: ''
    category: 'i18n-Messages'!

LcMessages comment:
'This object is a factory of LcMessagesDomain objects'!

Object subclass: #LcMessagesDomain
    instanceVariableNames: ''
    classVariableNames: ''
    poolDictionaries: ''
    category: 'i18n-Messages'!

LcMessagesDomain comment:
'This object is an abstract superclass for message domains (catalogs).
It contains methods to create instances of its subclasses, but they are
commonly used only by LcMessages.

Translations are accessed using either #at: or the shortcut binary
messages `?''.	This way, common idioms to access translated strings
will be

     string := NLS? ''abc''.
     string := self? ''abc''.

(in the first case NLS is a class variable, in the second the receiver
implements #? through delegation) which is only five or six characters
longer than the traditional

     string := ''abc''.

(cfr. the _("abc") idiom used by GNU gettext)'!

LcMessagesDomain subclass: #LcMessagesTerritoryDomain
    instanceVariableNames: 'primary secondary'
    classVariableNames: ''
    poolDictionaries: ''
    category: 'i18n-Messages'!

LcMessagesTerritoryDomain comment:
'This object asks for strings to a primary domain (e.g. it_IT)
and a secondary one (e.g. it).'!

LcMessagesDomain subclass: #LcMessagesDummyDomain
    instanceVariableNames: ''
    classVariableNames: ''
    poolDictionaries: ''
    category: 'i18n-Messages'!

LcMessagesDummyDomain comment:
'This object does no attempt to translate strings, returning
instead the same string passed as an argument to #?.'!

LcMessagesDomain subclass: #LcMessagesCatalog
    instanceVariableNames: 'file lastString lastCache'
    classVariableNames: ''
    poolDictionaries: ''
    category: 'i18n-Messages'!

LcMessagesCatalog comment:
'This object is an abstract superclass of objects that retrieve
translated strings from a file.	 It caches the last translated
string for speed.'!

LcMessagesCatalog subclass: #LcMessagesMoFileVersion0
    instanceVariableNames: 'cache original translated firstCharMap emptyGroup'
    classVariableNames: ''
    poolDictionaries: ''
    category: 'i18n-Messages'!

LcMessagesMoFileVersion0 comment:
'This object is an concrete class that retrieves translated strings
from a GNU gettext MO file.  The class method #fileFormatDescription
contains an explanation of the file format.'!

FileSegment subclass: #FileStreamSegment
       instanceVariableNames: ''
       classVariableNames: ''
       poolDictionaries: ''
       category: 'i18n-Messages'
!

FileStreamSegment comment:
'Unlike FileSegment, this object assumes that the `file'' instance
variable is a FileStream, not a file name.'!


!LcMessages class methodsFor: 'accessing'!

category
    ^#'LC_MESSAGES'!

!LcMessages methodsFor: 'accessing'!

territoryDirectory
    ^super territoryDirectory, '/LC_MESSAGES'!

languageDirectory
    ^super languageDirectory, '/LC_MESSAGES'! !

!LcMessages methodsFor: 'opening MO files'!

? aString
    ^self territoryDomain: aString!

domain: aString
    ^self domain: aString directory: self languageDirectory!

dummyDomain
    ^LcMessagesDummyDomain new!

dummyDomain: aString
    ^LcMessagesDummyDomain new!

territoryDomain: aString
    self isPosixLocale
	ifTrue: [ ^self dummyDomain: aString ].

    ^LcMessagesTerritoryDomain
	primary: (self domain: aString directory: self territoryDirectory)
	secondary: (self domain: aString directory: self languageDirectory)!

domain: aString directory: dirName
    self isPosixLocale
	ifTrue: [ ^self dummyDomain: aString ].

    ^LcMessagesDomain on: (Directory append: aString to: dirName)! !


!LcMessagesDomain class methodsFor: 'opening MO files'!

on: aFileName
    | stream found |
    found := false.
    (File exists: aFileName) ifTrue: [
	stream := FileStream open: aFileName mode: FileStream read.
	found := [
	    stream
		littleEndianMagicNumber: #[ 16rDE 16r12 16r04 16r95 ]
		bigEndianMagicNumber:    #[ 16r95 16r04 16r12 16rDE ].

	    stream nextLong = 0
	]   on: ExError
	    do: [ :ex | ex return: false ].

	found
	    ifTrue: [ stream reset ]
	    ifFalse: [ stream close ]
    ].

    ^found
	ifTrue: [ LcMessagesMoFileVersion0 new initialize: stream ]
	ifFalse: [ LcMessagesDummyDomain new ]! !

!LcMessagesDomain methodsFor: 'flushing the cache'!

flush
    "Flush any cached translations and reset the state of
     the receiver."
! !

!LcMessagesDomain methodsFor: 'querying'!

at: aString
    "Answer the translation of `aString', or answer aString itself
     if none is available."
    ^self? aString!

at: aString put: anotherString
    self shouldNotImplement!

? aString
    "Answer the translation of `aString', or answer aString itself
     if none is available."
    self subclassResponsibility!

translatorInformation
    "Answer information on the translation, or nil if there is none.
     This information is stored as the `translation' of an empty string."

    | info |
    info := self? ''.
    info isEmpty ifTrue: [ info := nil ].
    ^info
! !


!LcMessagesTerritoryDomain class methodsFor: 'instance creation'!

primary: domain1 secondary: domain2
    ^self new primary: domain1 secondary: domain2! !

!LcMessagesTerritoryDomain methodsFor: 'querying'!

? aString
    | primaryTranslation |
    ^(primaryTranslation := primary? aString) == aString
	ifTrue: [ secondary? aString ]
	ifFalse: [ primary? aString ]! !

!LcMessagesTerritoryDomain methodsFor: 'private'!

primary: domain1 secondary: domain2
    primary := domain1.
    secondary := domain2! !

!LcMessagesDummyDomain methodsFor: 'querying'!

? aString
    "Answer the translation of `aString', or answer aString itself
     if none is available (which always happens in this class)."
    ^aString! !


!LcMessagesCatalog methodsFor: 'flushing the cache'!

flush
    lastString := lastCache := nil! !

!LcMessagesCatalog methodsFor: 'querying'!

? aString
    "Answer the translation of `aString', or answer aString itself
     if none is available."
    aString == lastString ifFalse: [
	lastString := aString.
	lastCache := self translate: aString
    ].
    ^lastCache! !

!LcMessagesCatalog methodsFor: 'abstract'!

translate: aString
    "Answer the translation of `aString', or answer aString itself
     if none is available.  This overrides the caching operated by #?."
    self subclassResponsibility! !

!LcMessagesCatalog methodsFor: 'private'!

file
    ^file!

initialize: stream
    file := stream.
    self flush! !


!LcMessagesMoFileVersion0 class methodsFor: 'documentation'!

fileFormatDescription

"The Format of GNU MO Files (excerpt of the GNU gettext manual)
==============================================================

   The format of the generated MO files is best described by a picture,
which appears below.

   The first two words serve the identification of the file.  The magic
number will always signal GNU MO files.	 The number is stored in the
byte order of the generating machine, so the magic number really is two
numbers: `0x950412de' and `0xde120495'.	 The second word describes the
current revision of the file format.  For now the revision is 0.  This
might change in future versions, and ensures that the readers of MO
files can distinguish new formats from old ones, so that both can be
handled correctly.  The version is kept separate from the magic number,
instead of using different magic numbers for different formats, mainly
because `/etc/magic' is not updated often.  It might be better to have
magic separated from internal format version identification.

   Follow a number of pointers to later tables in the file, allowing
for the extension of the prefix part of MO files without having to
recompile programs reading them.  This might become useful for later
inserting a few flag bits, indication about the charset used, new
tables, or other things.

   Then, at offset O and offset T in the picture, two tables of string
descriptors can be found.  In both tables, each string descriptor uses
two 32 bits integers, one for the string length, another for the offset
of the string in the MO file, counting in bytes from the start of the
file.  The first table contains descriptors for the original strings,
and is sorted so the original strings are in increasing lexicographical
order.	The second table contains descriptors for the translated
strings, and is parallel to the first table: to find the corresponding
translation one has to access the array slot in the second array with
the same index.

   Having the original strings sorted enables the use of simple binary
search, for when the MO file does not contain an hashing table, or for
when it is not practical to use the hashing table provided in the MO
file.  This also has another advantage, as the empty string in a PO
file GNU `gettext' is usually *translated* into some system information
attached to that particular MO file, and the empty string necessarily
becomes the first in both the original and translated tables, making
the system information very easy to find.

   The size S of the hash table can be zero.  In this case, the hash
table itself is not contained in the MO file.  Some people might prefer
this because a precomputed hashing table takes disk space, and does not
win *that* much speed.	The hash table contains indices to the sorted
array of strings in the MO file.  Conflict resolution is done by double
hashing.  The precise hashing algorithm used is fairly dependent of GNU
`gettext' code, and is not documented here.

   As for the strings themselves, they follow the hash file, and each
is terminated with a <NUL>, and this <NUL> is not counted in the length
which appears in the string descriptor.	 The `msgfmt' program has an
option selecting the alignment for MO file strings.  With this option,
each string is separately aligned so it starts at an offset which is a
multiple of the alignment value.  On some RISC machines, a correct
alignment will speed things up.

   Nothing prevents a MO file from having embedded <NUL>s in strings.
However, the program interface currently used already presumes that
strings are <NUL> terminated, so embedded <NUL>s are somewhat useless.
But MO file format is general enough so other interfaces would be later
possible, if for example, we ever want to implement wide characters
right in MO files, where <NUL> bytes may accidently appear.

   This particular issue has been strongly debated in the GNU `gettext'
development forum, and it is expectable that MO file format will evolve
or change over time.  It is even possible that many formats may later
be supported concurrently.  But surely, we have to start somewhere, and
the MO file format described here is a good start.  Nothing is cast in
concrete, and the format may later evolve fairly easily, so we should
feel comfortable with the current approach.

	     byte
		  +------------------------------------------+
	       0  | magic number = 0x950412de		     |
		  |					     |
	       4  | file format revision = 0		     |
		  |					     |
	       8  | number of strings			     |	== N
		  |					     |
	      12  | offset of table with original strings    |	== O
		  |					     |
	      16  | offset of table with translation strings |	== T
		  |					     |
	      20  | size of hashing table		     |	== S
		  |					     |
	      24  | offset of hashing table		     |	== H
		  |					     |
		  .					     .
		  .    (possibly more entries later)	     .
		  .					     .
		  |					     |
	       O  | length & offset 0th string	----------------.
	   O + 8  | length & offset 1st string	------------------.
		   ...					  ...	| |
     O + ((N-1)*8)| length & offset (N-1)th string	     |	| |
		  |					     |	| |
	       T  | length & offset 0th translation  ---------------.
	   T + 8  | length & offset 1st translation  -----------------.
		   ...					  ...	| | | |
     T + ((N-1)*8)| length & offset (N-1)th translation	     |	| | | |
		  |					     |	| | | |
	       H  | start hash table			     |	| | | |
		   ...					  ...	| | | |
       H + S * 4  | end hash table			     |	| | | |
		  |					     |	| | | |
		  | NUL terminated 0th string  <----------------' | | |
		  |					     |	  | | |
		  | NUL terminated 1st string  <------------------' | |
		  |					     |	    | |
		   ...					  ...	    | |
		  |					     |	    | |
		  | NUL terminated 0th translation  <---------------' |
		  |					     |	      |
		  | NUL terminated 1st translation  <-----------------'
		  |					     |
		   ...					  ...
		  |					     |
		  +------------------------------------------+

Locating Message Catalog Files
------------------------------

   Because many different languages for many different packages have to
be stored we need some way to add these information to file message
catalog files.	The way usually used in Unix environments is have this
encoding in the file name.  This is also done here.  The directory name
given in `bindtextdomain's second argument (or the default directory),
followed by the value and name of the locale and the domain name are
concatenated:

     DIR_NAME/LOCALE/LC_CATEGORY/DOMAIN_NAME.mo

   The default value for DIR_NAME is system specific.  For the GNU
library, and for packages adhering to its conventions, it's:
     /usr/local/share/locale

LOCALE is the value of the locale whose name is this `LC_CATEGORY'.
For `gettext' and `dgettext' this locale is always `LC_MESSAGES'.
"! !

!LcMessagesMoFileVersion0 methodsFor: 'flushing the cache'!

flush
    | n oOfs tOfs sOfs hOfs |
    super flush.
    cache := LookupTable new.

    self file position: 8.
    n	 := self file nextLong.
    oOfs := self file nextLong.
    tOfs := self file nextLong.
    sOfs := self file nextLong.
    hOfs := self file nextLong.
    original := self readSegmentTable: oOfs size: n.
    translated := self readSegmentTable: tOfs size: n.
! !

!LcMessagesMoFileVersion0 methodsFor: 'private'!

initialize: stream
    super initialize: stream.
    self getFirstChars.
!

getFirstChars
    "This implementation does a limited form of bucketing
     to supply the speed lost by not implementing hashing. This
     method prepares a table that subdivides strings according
     to their first character."
    | lastIndex lastFirst |
    firstCharMap := Array new: 256.
    original doWithIndex: [ :segment :n |
	| interval first |
	segment size = 0
	    ifTrue: [ emptyGroup := n to: n ]
	    ifFalse: [
		"Read first character of the string"
		self file position: segment filePos.
		first := self file nextByte + 1.
		interval := firstCharMap at: first.
		interval isNil ifTrue: [
		    firstCharMap at: first put: n
		]
	    ]
    ].

    firstCharMap doWithIndex: [ :thisFirst :index |
	thisFirst notNil ifTrue: [
	    "Store an Interval at the lastIndex-th position"
	    lastIndex notNil ifTrue: [
		firstCharMap
		    at: lastIndex
		    put: (lastFirst to: thisFirst - 1).
	    ].

	    lastIndex := index.
	    lastFirst := thisFirst.
	]
    ].

    "Finish the last position too"
    lastIndex notNil ifTrue: [
	firstCharMap
	    at: lastIndex
	    put: (lastFirst to: original size).
    ]!

readSegmentTable: offset size: n
    "Answer a table of n FileStreamSegments loaded from the
     MO file, starting at the requested offset."
    self file position: offset.
    ^(1 to: n) collect: [ :unused |
	| size |
	size := self file nextLong.
	FileStreamSegment
	    on: self file
	    startingAt: self file nextLong
	    for: size
    ]
!

translate: aString
    "Translate aString, answer the translation"
    | value group n |
    value := cache at: aString ifAbsent: [ nil ].
    value isNil ifFalse: [ ^value ].

    group := aString isEmpty
	ifTrue: [ emptyGroup ]
	ifFalse: [ firstCharMap at: (aString at: 1) value + 1 ].

    group isNil ifTrue: [ ^aString ].

    n := self binarySearch: aString from: group first to: group last.
    ^n isNil
	ifTrue: [ cache at: aString put: aString ]
	ifFalse: [ cache at: aString put: (translated at: n) asString ]
!

binarySearch: aString from: low to: high
    "Do a binary search on `original', searching for aString"
    | i j mid originalString result |
    i := low.
    j := high.

    [ i > j ] whileFalse: [
	mid := (i + j + 1) // 2.
	originalString := original at: mid.
	originalString isString ifFalse: [
	    originalString become: originalString asString
	].
	result := aString compareTo: originalString.

	result = 0 ifTrue: [ ^mid ].
	result < 0
	    ifTrue: [ j := mid - 1]
	    ifFalse: [ i := mid + 1 ].
    ].
    ^nil
! !


!FileStreamSegment methodsFor: 'basic'!

withFileDo: aBlock
    ^aBlock value: self getFile
!

fileName
    "Answer the name of the file containing the segment"
    ^self getFile name
! !
