"======================================================================
|
|   LC_TIME support
|
|   $Revision: 1.95.1$
|   $Date: 2000/12/27 10:45:49$
|   $Author: pb$
|
 ======================================================================"


"======================================================================
|
| Copyright 1988-92, 1994-95, 1999, 2000 Free Software Foundation, Inc.
| Written by Paolo Bonzini.
|
| This file is part of the GNU Smalltalk class library.
|
| The GNU Smalltalk class library is free software; you can redistribute it
| and/or modify it under the terms of the GNU Lesser General Public License
| as published by the Free Software Foundation; either version 2.1, or (at
| your option) any later version.
|
| The GNU Smalltalk class library is distributed in the hope that it will be
| useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
| MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the GNU Lesser
| General Public License for more details.
|
| You should have received a copy of the GNU Lesser General Public License
| along with the GNU Smalltalk class library; see the file COPYING.LESSER.
| If not, write to the Free Software Foundation, 59 Temple Place - Suite
| 330, Boston, MA 02111-1307, USA.
|
 ======================================================================"

LcPrintFormats subclass: #LcTime
    instanceVariableNames: 'abday day abmon mon amPm dtFmt dFmt tFmt tFmtAmPm eras eraDTFmt eraDFmt eraTFmt altDigits '
    classVariableNames: ''
    poolDictionaries: ''
    category: 'i18n-Printing'!

Object subclass: #LcEra
    instanceVariableNames: 'first last forward offset name format'
    classVariableNames: ''
    poolDictionaries: ''
    category: 'i18n-Printing'!

LcTime comment:
'Sending either #?, #printString: or #print:on: converts a Date or Time
to a String according to the rules that are used in the given locale.'!

!LcTime class methodsFor: 'accessing'!

category
    ^#'LC_TIME'!

bigEndianID
    ^#[16r96 16r06 16r17 16rDC]!

littleEndianID
    ^#[16rDC 16r17 16r06 16r96]! !

!LcTime methodsFor: 'printing'!

eraPrint: aDateOrTimeOrArray on: aStream
    self readData.
    ^self
	print: aDateOrTimeOrArray
	on: aStream
	ifFull: eraDTFmt
	ifDate: eraDFmt
	ifTime: eraTFmt!

print: aDateOrTimeOrArray on: aStream
    self readData.
    ^self
	print: aDateOrTimeOrArray
	on: aStream
	ifFull: dtFmt
	ifDate: dFmt
	ifTime: tFmt!

print: aDate time: aTime format: aString on: aStream
    | what |
    self readData.
    what := Array with: aDate with: aTime.
    self strftime: what format: aString readStream on: aStream!

print: aDateOrTimeOrArray on: aStream ifFull: fullFmt ifDate: dateFmt ifTime: timeFmt
    | what format |
    format := fullFmt.
    what := aDateOrTimeOrArray.
    aDateOrTimeOrArray class == Date ifTrue: [
	what := Array with: aDateOrTimeOrArray with: nil.
	format := dateFmt
    ].
    aDateOrTimeOrArray class == Time ifTrue: [
	what := Array with: nil with: aDateOrTimeOrArray.
	format := timeFmt
    ].
    ^self strftime: what format: format readStream on: aStream! !

!LcTime methodsFor: 'tests'!

allFormatsExample
    ^'
%%a   %tabbreviated weekday	      %a
%%A   %tweekday			      %A
%%b   %tabbreviated month	      %b
%%B   %tmonth			      %B
%%c   %tdate & time		      %c
%%C   %tcentury			      %C
%%d   %tday of the month	      %d
%%D   %tdate (US)		      %D
%%e   %tday of the month	      %e
%%g   %tyear for the ISO week	      %g
%%G   %tyear for the ISO week	      %G
%%h   %tabbreviated month	      %h
%%H   %thours			      %H
%%I   %thours (AM/PM)		      %I
%%j   %tday of the year		      %j
%%k   %thours			      %k
%%l   %thours (AM/PM)		      %l
%%m   %tmonth			      %m
%%M   %tminutes			      %M
%%p   %tAM/PM			      %p
%%P   %tlowercase AM/PM		      %P
%%r   %tAM/PM time		      %r
%%R   %ttime (US)		      %R
%%s   %ttime_t			      %s
%%S   %tseconds			      %S
%%T   %ttime (US)		      %T
%%u   %tday of the week		      %u
%%U   %tweek number starting at Sun   %U
%%V   %tweek number starting at Thu   %V
%%w   %tday of the week, Sunday=0     %w
%%W   %tweek number starting at Mon   %W
%%x   %tdate			      %x
%%X   %ttime			      %X
%%y   %tyear (2-digit)		      %y
%%Y   %tyear (4-digit)		      %Y%n'! !

!LcTime methodsFor: 'private'!

strftime: timestamp format: formatStream on: aStream
    | d t |
    d := timestamp at: 1.
    t := timestamp at: 2.
    [ formatStream atEnd ] whileFalse: [
	aStream nextPutAll: (formatStream upTo: $%).
	self
	    strftimeField: timestamp
	    format: formatStream
	    on: aStream
	    date: d
	    time: t
    ]!

strftimeField: timestamp format: formatStream on: aStream date: d time: t
    "OUCH! This methods is 300+ lines long... but we need re-entrancy, and
     I don't want to create a separate object to print a particular
     date & time pair."

    | pad padTo ch output fmt era case invertCase modifier dow |
    pad := nil.
    padTo := 1.
    case := #yourself.
    invertCase := false.
    [
	ch := formatStream next.
	ch == $_ ifTrue: [ pad := $ . true ] ifFalse: [
	    ch == $- ifTrue: [ pad := nil. true ] ifFalse: [
		ch == $0 ifTrue: [ pad := $0. true ] ifFalse: [
		    ch == $^ ifTrue: [ case := #asUppercase. true ] ifFalse: [
			ch == $# ifTrue: [ invertCase := true ]; yourself
	]]]]
    ] whileTrue.

    modifier := nil.
    ch == $E ifTrue: [ modifier := $E ].
    ch == $O ifTrue: [ modifier := $O ].

    modifier isNil ifFalse: [ ch := formatStream next ].

    ch == $% ifTrue: [
	output := '%'.
    ].

    "Abbreviated weekday"
    ch == $a ifTrue: [
	dow := (d days + 2) \\ 7 + 1.
	output := abday at: dow.
	invertCase ifTrue: [ case := #asUppercase ].
    ].

    "Weekday"
    ch == $A ifTrue: [
	dow := (d days + 2) \\ 7 + 1.
	output := day at: dow.
	invertCase ifTrue: [ case := #asUppercase ].
    ].

    "Abbreviated month"
    (ch == $b or: [ ch == $h ]) ifTrue: [
	output := abmon at: d month.
	invertCase ifTrue: [ case := #asUppercase ].
    ].

    "Month"
    ch == $B ifTrue: [
	output := mon at: d month.
	invertCase ifTrue: [ case := #asUppercase ].
    ].

    "Full date"
    ch == $c ifTrue: [
	fmt := (modifier == $E)
	    ifTrue: [ modifier := nil. eraDTFmt ]
	    ifFalse: [ dtFmt ].

	output := String streamContents: [ :stream |
	    self strftime: timestamp format: fmt readStream on: stream
	]
    ].

    "Century"
    ch == $C ifTrue: [
	output := d year // 100.
	(modifier == $E)
	    ifTrue: [ modifier := nil. output := (self era: d year) name, output ]
    ].

    "Day of month"
    ch == $d ifTrue: [
	pad isNil ifTrue: [ pad := $0 ].
	padTo := 2.
	output := d day.
    ].

    "Date (month/day/year)"
    ch == $D ifTrue: [
	output := String streamContents: [ :stream |
	    self strftime: timestamp format: '%m/%d/%y' readStream on: stream
	]
    ].

    "Day of month"
    ch == $e ifTrue: [
	pad isNil ifTrue: [ pad := $  ].
	padTo := 2.
	output := d day.
    ].

    "Hours"
    ch == $H ifTrue: [
	pad isNil ifTrue: [ pad := $0 ].
	padTo := 2.
	output := t hours.
    ].

    "Hours (12-hours format)"
    ch == $I ifTrue: [
	pad isNil ifTrue: [ pad := $0 ].
	padTo := 2.
	output := t hours \\ 12.
	output = 0 ifTrue: [ output := 12 ].
    ].

    "Day of year"
    ch == $j ifTrue: [
	pad isNil ifTrue: [ pad := $0 ].
	padTo := 3.
	output := d dayOfYear
    ].

    "Hours"
    ch == $k ifTrue: [
	pad isNil ifTrue: [ pad := $  ].
	padTo := 2.
	output := t hours.
    ].

    "Hours (12-hours format)"
    ch == $l ifTrue: [
	pad isNil ifTrue: [ pad := $  ].
	padTo := 2.
	output := t hours \\ 12.
	output = 0 ifTrue: [ output := 12 ].
    ].

    "Month"
    ch == $m ifTrue: [
	pad isNil ifTrue: [ pad := $0 ].
	padTo := 2.
	output := d month
    ].

    "Minutes"
    ch == $M ifTrue: [
	pad isNil ifTrue: [ pad := $0 ].
	padTo := 2.
	output := t minutes.
    ].

    "Newline"
    ch == $n ifTrue: [
	output := Character nl asString
    ].

    "AM/PM"
    ch == $p ifTrue: [
	output := amPm at: (t hours // 12) + 1.
	invertCase ifTrue: [ case := #asLowercase ].
    ].

    "Lowercase AM/PM"
    ch == $P ifTrue: [
	output := amPm at: (t hours // 12) + 1.
	case := #asLowercase
    ].

    "AM/PM time"
    ch == $r ifTrue: [
	output := String streamContents: [ :stream |
	    self strftime: timestamp format: tFmtAmPm readStream on: stream
	]
    ].

    "Hours:Minutes time"
    ch == $R ifTrue: [
	output := String streamContents: [ :stream |
	    self strftime: timestamp format: '%H:%M' readStream on: stream
	]
    ].

    "Seconds since 1/1/1970"
    ch == $s ifTrue: [
	output := Date newDay: 1 monthIndex: 1 year: 1970.
	output := d isNil ifTrue: [ 0 ] ifFalse: [ (d subtractDate: output) * 86400 ].
	output := t isNil ifTrue: [ output ] ifFalse: [ output + t asSeconds ].
    ].

    "Seconds since 1/1/1970"
    ch == $S ifTrue: [
	pad isNil ifTrue: [ pad := $0 ].
	padTo := 2.
	output := t seconds.
    ].

    "Tab"
    ch == $t ifTrue: [
	output := Character tab asString
    ].

    "Hours:Minutes:Seconds time"
    ch == $T ifTrue: [
	output := String streamContents: [ :stream |
	    self strftime: timestamp format: '%H:%M:%S' readStream on: stream
	]
    ].

    "Day of week, 1=Monday, 7=Sunday"
    ch == $u ifTrue: [
	output := d dayOfWeek.
    ].

    "Week, first day=Sunday, 0 if before first Sunday"
    ch == $U ifTrue: [
	pad isNil ifTrue: [ pad := $0 ].
	padTo := 2.
	output := d weekStartingAt: 7.
    ].

    "Week, first day=Thursday, 52 or 53 if before first Thursday"
    ch == $V ifTrue: [
	pad isNil ifTrue: [ pad := $0 ].
	padTo := 2.
	output := d weekStartingAt: 4.
	output = 0 ifTrue: [
	    output := (d subtractDays: d day) weekStartingAt: 4
	].
    ].

    "Day of week, Sunday=0, Saturday=6"
    ch == $w ifTrue: [
	output := d dayOfWeek \\ 7.
    ].

    "Week, first day=Monday, 0 if before first Monday"
    ch == $W ifTrue: [
	pad isNil ifTrue: [ pad := $0 ].
	padTo := 2.
	output := d weekStartingAt: 1.
    ].

    "Date"
    ch == $x ifTrue: [
	fmt := (modifier == $E)
	    ifTrue: [ modifier := nil. eraDFmt ]
	    ifFalse: [ dFmt ].

	output := String streamContents: [ :stream |
	    self strftime: timestamp format: fmt readStream on: stream
	]
    ].

    "Time"
    ch == $X ifTrue: [
	fmt := (modifier == $X)
	    ifTrue: [ modifier := nil. eraTFmt ]
	    ifFalse: [ tFmt ].

	output := String streamContents: [ :stream |
	    self strftime: timestamp format: fmt readStream on: stream
	]
    ].

    "Current year or (if `g' and before first Thursday of the year) previous
     year; 2 digits."
    (ch == $y or: [ ch == $g ]) ifTrue: [
	output := d year.
	ch == $g ifTrue: [
	    (d weekStartingAt: 4) = 0 ifTrue: [ output := output - 1 ].
	].
	modifier == $E ifFalse: [
	    pad isNil ifTrue: [ pad := $0 ].
	    padTo := 2.
	    output := output \\ 100.
	].
	modifier == $E ifTrue: [
	    modifier := nil.
	    era := self era: d.
	    era forward ifFalse: [ output := output negated ].
	    output := era offset + output
	].
    ].

    "Current year or (if `g' and before first Thursday of the year) previous
     year; 4 digits."
    (ch == $Y or: [ ch == $G ]) ifTrue: [
	output := d year.
	ch == $G ifTrue: [
	    (d weekStartingAt: 4) = 0 ifTrue: [ output := output - 1 ].
	].
	modifier == $E ifFalse: [
	    pad isNil ifTrue: [ pad := $  ].
	    padTo := 4.
	].
	modifier == $E ifTrue: [
	    modifier := nil.
	    era := self era: d.
	    output := String streamContents: [ :stream |
		self strftime: timestamp format: era format readStream on: stream
	    ]
	]
    ].
    ch == $Z ifTrue: [ output := '' ].

    output isNil ifTrue: [
	output := ch asString.
	case := #yourself
    ].

    (output isInteger and: [ modifier == $O ]) ifTrue: [
	modifier := nil.
	output < altDigits size ifTrue: [
	    output := altDigits at: output + 1
	]
    ].
    modifier isNil ifFalse: [
	self error: 'invalid modifier specified'
    ].
    output isInteger ifTrue: [
	output := output printString.
	pad isNil ifFalse: [
	    ch := $0.
	    padTo - output size timesRepeat: [
		(output at: 1) == $-
		    ifTrue: [ output at: 1 put: $0. ch := $- ].

		aStream nextPut: ch.
		ch := $0
	    ]
	].
	case := #yourself.
    ].

    output := output perform: case.
    aStream nextPutAll: output
!

era: date
    | era low mid high |
    low := 1.
    high := eras size.
    [ low < high ] whileTrue: [
	mid := (low + high) // 2.
	era := eras at: mid.
	(era isBefore: date)
	    ifTrue: [ low := mid + 1 ]
	    ifFalse: [
		(era isAfter: date) ifFalse: [ ^era ].
		high := mid - 1
	    ]
    ]
! !

!LcTime methodsFor: 'reading'!

bePosix
    abday := #('Sun' 'Mon' 'Tue' 'Wed' 'Thu' 'Fri' 'Sat').
    day := #('Sunday' 'Monday' 'Tuesday' 'Wednesday' 'Thursday' 
	'Friday' 'Saturday').

    abmon := #('Jan' 'Feb' 'Mar' 'Apr' 'May' 'Jun' 'Jul' 'Aug' 
	'Sep' 'Oct' 'Nov' 'Dec').

    mon := #('January' 'February' 'March' 'April' 'May' 'June' 'July' 
	'August' 'September' 'October' 'November' 'December').

    amPm := #('AM' 'PM').

    dtFmt := '%a %b %e %H:%M:%S %Y'.
    dFmt := '%m/%d/%y'.
    tFmt := '%H:%M:%S'.
    tFmtAmPm := '%I:%M:%S'.
    self setDefaults!

readDataFrom: f
    | numAltDigits numEras save |
    abday := self readStringArrayFrom: f size: 7.
    day := self readStringArrayFrom: f size: 7.
    abmon := self readStringArrayFrom: f size: 12.
    mon := self readStringArrayFrom: f size: 12.
    amPm := self readStringArrayFrom: f size: 2.
    dtFmt := self readStringFrom: f.
    dFmt := self readStringFrom: f.
    tFmt := self readStringFrom: f.
    tFmtAmPm := self readStringFrom: f.

    (self hasOptionalData: f) ifFalse: [ ^self ].

    "Skip old-style era data"
    self skipEntry: f.
    self skipEntry: f.
    eraDFmt := self readStringFrom: f.
    save := self entryNumber: f.
    self skipEntry: f.
    eraDTFmt := self readStringFrom: f.
    eraTFmt := self readStringFrom: f.

    numAltDigits := self readWordFrom: f.
    numEras := self readWordFrom: f.
    eras := self bigEndianFirst: [
	| fs |
	fs := self getFileStream: f.
	(1 to: numEras) collect: [ :each | (LcEra new) readDataFrom: fs ]
    ]  on: f.

    self move: f toEntry: save.
    altDigits := self readStringArrayFrom: f size: numAltDigits!

setDefaults
    eras isNil ifTrue: [ eras := #() ].
    eras isEmpty ifFalse: [ ^self ].
    eraDFmt := dFmt.
    eraDTFmt := dtFmt.
    eraTFmt := tFmt.
    altDigits := #()! !

!LcEra methodsFor: 'accessing'!

first
    ^first!

last
    ^last!

forward
    ^forward!

offset
    ^offset!

name
    ^name!

format
    ^format!
    
isBefore: aDate
    ^last < aDate!

isAfter: aDate
    ^aDate < first! !

!LcEra methodsFor: 'reading'!

readFrom: fs
    | y m d l |
    forward := fs nextLong = $+ value.
    offset := fs nextLong.

    y := fs nextLong.
    m := fs nextLong.
    d := fs nextLong.
    first := Date newDay: d monthIndex: m year: y.

    y := fs nextLong.
    m := fs nextLong.
    d := fs nextLong.
    last := Date newDay: d monthIndex: m year: y.

    name := fs upTo: 0 asCharacter.
    l := name size.
    format := fs upTo: 0 asCharacter.
    l := (l + format size) \\ 4.
    l \\ 4 < 3 ifTrue: [ fs next ].
    l \\ 4 < 2 ifTrue: [ fs next ].
    l \\ 4 < 1 ifTrue: [ fs next ].
    ^self
! !

!Date methodsFor: 'calculations'!

weekStartingAt: startDay
    | yday wday weekDayJan1 first |
    yday := self dayOfYear - 1.			"January 1st = 0"
    wday := self dayOfWeek.
    weekDayJan1 := (wday - yday) \\ 7.		"week day for January 1st"
    first := (startDay - weekDayJan1) \\ 7.	"day of year for first startDay"

    ^(yday - first) // 7 + 1! !
