"======================================================================
|
|   ANSI exception handling classes
|
|   $Revision: 1.95.1$
|   $Date: 2000/12/27 10:45:49$
|   $Author: pb$
|
 ======================================================================"


"======================================================================
|
| Copyright 1988-92, 1994-95, 1999, 2000 Free Software Foundation, Inc.
| Written by Paolo Bonzini.
|
| This file is part of the GNU Smalltalk class library.
|
| The GNU Smalltalk class library is free software; you can redistribute it
| and/or modify it under the terms of the GNU Lesser General Public License
| as published by the Free Software Foundation; either version 2.1, or (at
| your option) any later version.
| 
| The GNU Smalltalk class library is distributed in the hope that it will be
| useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
| MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser
| General Public License for more details.
| 
| You should have received a copy of the GNU Lesser General Public License
| along with the GNU Smalltalk class library; see the file COPYING.LESSER.
| If not, write to the Free Software Foundation, 59 Temple Place - Suite
| 330, Boston, MA 02111-1307, USA.  
|
 ======================================================================"

Signal subclass: #Exception
    instanceVariableNames: 'creator'
    classVariableNames: ''
    poolDictionaries: ''
    category: 'Language-Exceptions'!

Exception class instanceVariableNames: 'coreException'.

Exception subclass: #Error
    instanceVariableNames: ''
    classVariableNames: ''
    poolDictionaries: ''
    category: 'Language-Exceptions'.

Exception subclass: #Notification
    instanceVariableNames: ''
    classVariableNames: ''
    poolDictionaries: ''
    category: 'Language-Exceptions'!

Notification subclass: #Warning
    instanceVariableNames: ''
    classVariableNames: ''
    poolDictionaries: ''
    category: 'Language-Exceptions'.

Error subclass: #UserBreak
    instanceVariableNames: ''
    classVariableNames: ''
    poolDictionaries: ''
    category: 'Language-Exceptions'.

Error subclass: #Halt
    instanceVariableNames: ''
    classVariableNames: ''
    poolDictionaries: ''
    category: 'Language-Exceptions'!

Halt subclass: #ArithmeticError
    instanceVariableNames: ''
    classVariableNames: ''
    poolDictionaries: ''
    category: 'Language-Exceptions'.

Halt subclass: #MessageNotUnderstood
    instanceVariableNames: 'message receiver'
    classVariableNames: ''
    poolDictionaries: ''
    category: 'Language-Exceptions'!

ArithmeticError subclass: #ZeroDivide
    instanceVariableNames: 'dividend'
    classVariableNames: ''
    poolDictionaries: ''
    category: 'Language-Exceptions'!

Exception comment:
'An Exception defines the characteristics of an exceptional event
in a different way than CoreExceptions.  Instead of creating an
hierarchy of objects and setting attributes of the objects, you
create an hierarchy of classes and override methods in those classes;
instances of those classes are passed to the handlers instead of
instances of the common class Signal.

Internally, Exception and every subclass of it hold onto a
CoreException, so the two mechanisms are actually interchangeable.'.

Error comment:
'Error represents a fatal error.  Instances of it are not resumable.'.

Halt comment:
'Halt represents a resumable error, usually a bug.'.

Notification comment:
'Notification represents a resumable, exceptional yet non-erroneous,
situation.  Signaling a notification in absence of an handler simply
returns nil.'.

MessageNotUnderstood comment:
'MessageNotUnderstood represents an error during message lookup. Signaling
it is the default action of the #doesNotUnderstand: handler'.

ArithmeticError comment:
'An ArithmeticError exception is raised by numeric classes when a program
tries to do something wrong, such as extracting the square root of a
negative number.'.

ZeroDivide comment:
'A ZeroDivide exception is raised by numeric classes when a program tries
to divide by zero.  Information on the dividend is available to the
handler.'.

Warning comment:
'Warning represents an `important'' but resumable error.'!


!Exception class methodsFor: 'instance creation'!

new
    "Create an instance of the receiver, which you will be able to
     signal later."

    | ctx creator |
    ctx := thisContext parentContext.
    [ (creator := ctx receiver) == self ] whileTrue: [
	ctx := ctx parentContext ].

    ^self basicNew
	initialize: creator
!

signal
    "Create an instance of the receiver, give it default attributes,
     and signal it immediately."

    ^self new signal
!

signal: messageText
    "Create an instance of the receiver, set its message text,
     and signal it immediately."

    ^self new
	messageText: messageText;
	signal
! !


!Exception class methodsFor: 'creating ExceptionCollections'!

, aTrappableEvent
    "Answer an ExceptionCollection containing all the exceptions in the
     receiver and all the exceptions in aTrappableEvent"

    ^ExceptionSet new
	add: self coreException;
	add: aTrappableEvent;
	yourself
! !


!Exception class methodsFor: 'initialization'!

initialize
    "Initialize the `links' between the core exception handling system
     and the ANSI exception handling system."

    "Usually, a coreException has no defined `resumable' state, because
     Exception overrides Signal>>#isResumable and gives an answer without
     asking the core exception.  For backwards compatibility, however, we
     must give a state to these (in case someone uses `on: ExAll do: ...')."

    (ExAll := CoreException basicNew)
	defaultHandler: [ :sig | self primError: sig messageText ];
	signalClass: Signal;
	isResumable: true;
	yourself.

    (coreException := ExAll newChild)
	defaultHandler: [ :sig | sig defaultAction ];
	signalClass: self;
	isResumable: true;
	yourself.

    (ExError := Error coreException)
	isResumable: false.

    (ExDoesNotUnderstand := MessageNotUnderstood coreException)
	isResumable: true.

    (ExHalt := Halt coreException)
	isResumable: true.

    (ExUserBreak := UserBreak coreException)
	isResumable: false.
! !


!Exception class methodsFor: 'interoperability with TrappableEvents'!

allExceptionsDo: aBlock
    "Private - Pass the coreException to aBlock"
    
    aBlock value: self coreException
!

coreException
    "Private - Return the coreException which represents instances of
     the receiver"

    coreException isNil ifFalse: [ ^coreException ].

    ^coreException := self superclass coreException newChild
	defaultHandler: [ :sig | sig defaultAction ];
	signalClass: self;
	yourself
!

whenSignalledIn: onDoBlock do: handlerBlock exitBlock: exitBlock
    "Private - Create an ExceptionHandler from the arguments and register it"

    self coreException
	whenSignalledIn: onDoBlock
	do: handlerBlock
	exitBlock: exitBlock
! !


!Exception class methodsFor: 'comparison'!

handles: anException
    "Answer whether the receiver handles `anException'."

    ^(anException isKindOf: Exception)
	ifTrue: [ anException isKindOf: self ]
	ifFalse: [ self coreException handles: anException ]
! !


!Exception methodsFor: 'comparison'!

= anObject
    "Answer whether the receiver is equal to anObject.  This is true if
     either the receiver or its coreException are the same object as anObject."

    "This definition is needed to make #handles: work for ExceptionCollections.
     `(Error, Warning) handles: Error new' must work even if the
     ExceptionCollections contains the coreExceptions associated to Error
     and Warning (see Exception class>>#,)."
    ^self == anObject or: [ self exception == anObject ]
!

hash
    "Answer an hash value for the receiver."
    ^self exception hash
! !


!Signal methodsFor: 'private'!

!Exception methodsFor: 'private'!

asAnsiException
    ^self
!
	        
initialize: anObject
    "Initialize the receiver's instance variables."
    
    creator := anObject.
    self
	initArguments: #();
	initException: self class coreException;
	messageText: self description
! !


!Exception methodsFor: 'exception description'!

description
    "Answer a textual description of the exception."
    ^'An exception has occurred'
!

isResumable
    "Answer true.  Exceptions are by default resumable."
    ^true
!

defaultAction
    "Execute the default action that is attached to the receiver."
    self creator primError: self messageText
! !


!Exception methodsFor: 'exception signaling'!

signal
    "Raise the exceptional event represented by the receiver"
    self exception instantiateNextHandler: self.
    ^self activateHandler: self isResumable
!

signal: messageText
    "Raise the exceptional event represented by the receiver, setting
     its message text to messageText."
    ^self messageText: messageText; signal
! !


!Exception methodsFor: 'private - copying'!

copyFrom: aSignal
    "Private - Initialize from another instance of Signal"

    (aSignal isKindOf: Exception) ifTrue: [ self initialize: aSignal creator ].
    super copyFrom: aSignal.
!

creator
    ^creator
! !


!Notification methodsFor: 'exception description'!

description
    "Answer a textual description of the exception."
    ^'An exceptional condition has occurred, but it is not to be considered
an error.'!

isResumable
    "Answer true.  Notification exceptions are by default resumable."
    ^true
!

defaultAction
    "Do the default action for notifications, which is to resume execution
     of the context which signaled the exception."
    self resume: nil
! !


!Warning methodsFor: 'exception description'!

description
    "Answer a textual description of the exception."
    ^'An exceptional condition has occurred.  It is reported to the user
even though it is not to be considered an error.'
! !


!Error methodsFor: 'exception description'!

description
    "Answer a textual description of the exception."
    ^'An exceptional condition has occurred, and has prevented normal
continuation of processing.'
!

isResumable
    "Answer false.  Error exceptions are by default unresumable; subclasses
     can override this method if desired."
    ^false
! !


!Halt methodsFor: 'description'!

description
    "Answer a textual description of the exception."
    ^'#halt was sent.'
!

isResumable
    "Answer true.  #halt exceptions are by default resumable."
    ^true
! !



!ArithmeticError methodsFor: 'description'!

description
    "Answer a textual description of the exception."
    ^'The program attempted to do an impossible arithmetic operation'
! !


!ZeroDivide class methodsFor: 'instance creation'!

dividend: aNumber
    "Create a new ZeroDivide object remembering that the dividend was 
     aNumber."
    ^super new dividend: aNumber
!

new
    "Create a new ZeroDivide object; the dividend is conventionally
     set to zero."
    ^super new dividend: 0
! !

!ZeroDivide methodsFor: 'accessing'!

dividend
    "Answer the number that was being divided by zero"
    ^dividend
! !

!ZeroDivide methodsFor: 'private'!

dividend: aNumber
    dividend := aNumber
! !

!ZeroDivide methodsFor: 'description'!

description
    "Answer a textual description of the exception."
    ^'The program attempted to divide a number by zero'
! !


!MessageNotUnderstood methodsFor: 'accessing'!

message
    "Answer the message that wasn't understood"
    ^message
!

receiver
    "Answer the object to whom the message send was directed"
    ^receiver
! !

!MessageNotUnderstood methodsFor: 'private'!

message: aMessage receiver: anObject
    message := aMessage.
    receiver := anObject.

    self messageText: ('did not understand ', message selector printString)
! !

!MessageNotUnderstood methodsFor: 'description'!

description
    "Answer a textual description of the exception."
    ^'The program sent a message which was not understood by the receiver.'
! !


!UserBreak methodsFor: 'description'!

description
    "Answer a textual description of the exception."
    ^'interrupted!!!'
! !


!Number methodsFor: 'error raising' "ifTrue: false"!

arithmeticError: msg
    "Raise an ArithmeticError exception having msg as its message text."
    ^ArithmeticError new signal: msg
!

zeroDivide
    "Raise a division-by-zero (ZeroDivide) exception whose dividend
     is the receiver."
    ^(ZeroDivide dividend: self) signal
! !


!Object methodsFor: 'error raising' "ifTrue: false"!

doesNotUnderstand: aMessage
    "Called by the system when a selector was not found. message is a
     Message containing information on the receiver"

    ^MessageNotUnderstood new
	message: aMessage receiver: self;
	signal
!

error: message
    "Display a walkback for the receiver, with the given error message.
     Signal an `Error' exception (you can trap it the old way too, with
     `ExError'"
    ^Error new
	signal: message
!

halt: message
    "Display a walkback for the receiver, with the given error message.
     Signal an `Halt' exception (you can trap it the old way too, with
     `ExHalt'"
    ^Halt new
	signal: message
!

userInterrupt
    "Display a walkback for the receiver, signalling UserBreak (or
     ExUserBreak if you prefer)."
    ^UserBreak new
	signal
! !


Exception initialize!