/******************************** -*- C -*- ****************************
 *
 *	Byte code array utility routines.
 *
 *	$Revision: 1.95.1$
 *	$Date: 2000/12/27 10:45:49$
 *	$Author: pb$
 *
 ***********************************************************************/

/***********************************************************************
 *
 * Copyright 1988-92, 1994-95, 1999, 2000 Free Software Foundation, Inc.
 * Written by Steve Byrne.
 *
 * This file is part of GNU Smalltalk.
 *
 * GNU Smalltalk is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2, or (at your option) any later
 * version.
 *
 * GNU Smalltalk is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * GNU Smalltalk; see the file COPYING.	 If not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 ***********************************************************************/


#include "gst.h"
#include "alloc.h"
#include "byte.h"
#include "dict.h"
#include "input.h"
#if STDC_HEADERS
#include <string.h> /* for memcpy */
#include <stdlib.h>
#endif /* STDC_HEADERS */
#include <stdio.h>

#define BYTE_CODE_CHUNK_SIZE  64

int byteCodeSizeTable[256] = {
1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1, /* 0 */
1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1, /* 16 */
1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1, /* 32 */
1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1, /* 48 */
1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1, /* 64 */
1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1, /* 80 */
1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1, /* 96 */
1,1,1,1,1,1,1,1,1,1,1,1,1,1,3,2, /* 112 */
2,2,2,2,3,2,3,1,1,1,3,1,1,1,2,1, /* 128 */
1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1, /* 144 */
2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2, /* 160 */
1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1, /* 176 */
1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1, /* 192 */
1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1, /* 208 */
1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1, /* 224 */
1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1}; /* 240 */


static ByteCodes		allocByteCodes ();

static void			reallocByteCodes ();


/* These are used to decode special send byte codes. */
static char  *mathMessageName[] = {
  "+",
  "-",
  "<",
  ">",
  "<=",
  ">=",
  "=",
  "~=",
  "*",
  "/",
  "\\",
  "@",
  "bitShift:",
  "//",
  "bitAnd:",
  "bitOr:"
};

/* These are the selectors for special case send bytecodes that the compiler
 * generates.  These are used only to print out the generated byte codes. */
static char  *specialMessageName[] = {
  "at:",
  "at:put:",
  "size",
  "next",
  "nextPut:",
  "atEnd",
  "==",
  "class",
  "blockCopy:",
  "value",
  "value:",
  "do:",
  "new",
  "new:",
  "isNil",
  "notNil"
};

static ByteCodes		currentByteCodes;



ByteCodes
extractByteCodes (byteArrayOOP)
     OOP byteArrayOOP;
{
  ByteCodes result;
  Byte *bytes;
  int  len;
  ByteArray byteArray;

  byteArray = (ByteArray)oopToObj(byteArrayOOP);
  len = oopNumFields(byteArrayOOP);
  result = (ByteCodes) xmalloc(sizeof(struct ByteCodeArray));

  result->base = (Byte *) xmalloc(len);
  result->ptr = result->base + len;
  result->maxLen = len;
  memcpy (result->base, byteArray->bytes, len);
  return (result);
}

void
compileByte (byte)
     Byte byte;
{
  if (currentByteCodes == nil) {
    currentByteCodes = allocByteCodes(BYTE_CODE_CHUNK_SIZE);
  } else {
    if ((currentByteCodes->ptr - currentByteCodes->base)
	>= currentByteCodes->maxLen) {
      reallocByteCodes(currentByteCodes, BYTE_CODE_CHUNK_SIZE);
    }
  }

  *currentByteCodes->ptr++ = byte;
}

void
freeByteCodes (byteCodes)
     ByteCodes byteCodes;
{
  if (byteCodes != nil) {
    xfree(byteCodes->base);
    xfree(byteCodes);
  }
}

void
compileAndFreeByteCodes (byteCodes)
     ByteCodes byteCodes;
{
  compileByteCodes(byteCodes->base, byteCodes->ptr);
  freeByteCodes(byteCodes);
}

/*
 *	ByteCodes getByteCodes()
 *
 * Description
 *
 *	Called when byte code compilation is complete, this routine returns the
 *	set of byte codes that were compiled.  Since compilation is complete,
 *	this routine also resets the internal state of the byte code compiler
 *	in preparation for next time.
 *
 * Outputs
 *
 *	A pointer to a byte code array that contains the bytes for the current
 *	compilation.
 */
ByteCodes
getByteCodes ()
{
  ByteCodes curByteCodes;

  curByteCodes = currentByteCodes;
  currentByteCodes = nil;

  return (curByteCodes);
}


/*
 *	ByteCodes saveByteCodeArray()
 *
 * Description
 *
 *	Called to save the set of byte codes currently being compiled and
 *	prepare for a new compilation of byte codes. The current set of byte
 *	codes being compiled is returned for the caller to keep and to later
 *	use in a restoreByteCodeArray call.
 *
 * Outputs
 *
 *	A byte code array pointer to the current byte codes.
 */
ByteCodes
saveByteCodeArray ()
{
  ByteCodes curByteCodes;

  curByteCodes = currentByteCodes;
  currentByteCodes = nil;

  return (curByteCodes);
}


/*
 *	void restoreByteCodeArray(byteCodes)
 *
 * Description
 *
 *	Restores the internal state of the byte code compiler so that it can
 *	continue compiling byte codes into the byte code array "byteCodes",
 *	which should have been returned at some previous point from
 *	saveByteCodeArray().
 *
 * Inputs
 *
 *	byteCodes:
 *		The byte code array pointer to be restored. Should have come
 *		from a saveByteCodeArray call.
 *
 */
void
restoreByteCodeArray (byteCodes)
     ByteCodes byteCodes;
{
  currentByteCodes = byteCodes;
}

int
byteCodeLength(byteCodes)
ByteCodes byteCodes;
{
  if (byteCodes == nil) {
    return (0);
  }
  return (byteCodes->ptr - byteCodes->base);
}


/*
 *	int currentByteCodeLength()
 *
 * Description
 *
 *	Return the current number of byte codes that have been compiled.
 *
 * Outputs
 *
 *	Number of byte codes present in the current byte code array right now.
 */
int
currentByteCodeLength ()
{
  if (currentByteCodes == nil) {
    return (0);
  }

  return (currentByteCodes->ptr - currentByteCodes->base);
}

void
copyByteCodes (dest, byteCodes)
     Byte *dest;
     ByteCodes byteCodes;
{
  memcpy(dest, byteCodes->base, byteCodeLength(byteCodes));
}

void
truncateByteCodes (here, byteCodes)
     Byte *here;
     ByteCodes byteCodes;
{
  byteCodes->ptr = here;
}



void
printByteCodes (byteCodes, literalVec)
     ByteCodes byteCodes;
     OOP literalVec[];
{
  Byte	*b;
  int  ip;

  if (byteCodes == nil) {
    return;
  }

  for (b = byteCodes->base; b < byteCodes->ptr; b+= byteCodeSize(*b)) {
    ip = b - byteCodes->base;
    printf("%5d:\t", ip);
    printByteCodeName(b, ip, literalVec);
    printf("\n");
  }
  printf("\n");
}

void
printIndexed (byte, literalVec, constOk)
     Byte byte;
     OOP literalVec[];
     mst_Boolean constOk;
{
  switch (byte >> 6) {
  case 0:
    printf("Instance Variable[%d]", byte & 63);
    break;
  case 1:
    printf("Temporary[%d]", byte & 63);
    break;
  case 2:
    printf("Constant[%d]", byte & 63);
    if (literalVec) {
      printf(" = ");
      printObject(literalVec[byte & 63]);
    }
    if (!constOk) {
      printf(" -- INVALID!");
    }
    break;
  case 3:
    printf("Global Variable[%d]", byte & 63);
    if (literalVec) {
      printf(" = ");
      printAssociationKey(literalVec[byte & 63]);
    }
    break;
  }
}

void
printByteCodeName (bp, ip, literalVec)
     Byte bp[];
     int ip;
     OOP literalVec[];
{
  int lit;

  switch (bp[0]) {
  case	0: case	 1: case  2: case  3:
  case	4: case	 5: case  6: case  7:
  case	8: case	 9: case 10: case 11:
  case 12: case 13: case 14: case 15:
    printf("push Instance Variable[%d]", bp[0] & 15);
    break;

  case 16: case 17: case 18: case 19:
  case 20: case 21: case 22: case 23:
  case 24: case 25: case 26: case 27:
  case 28: case 29: case 30: case 31:
    printf("push Temporary[%d]", bp[0] & 15);
    break;

  case 32: case 33: case 34: case 35:
  case 36: case 37: case 38: case 39:
  case 40: case 41: case 42: case 43:
  case 44: case 45: case 46: case 47:
  case 48: case 49: case 50: case 51:
  case 52: case 53: case 54: case 55:
  case 56: case 57: case 58: case 59:
  case 60: case 61: case 62: case 63:
    printf("push Literal[%d]", bp[0] & 31);
    if (literalVec) {
      printf(" = ");
      printObject(literalVec[bp[0] & 31]);
    }
    break;

  case 64: case 65: case 66: case 67:
  case 68: case 69: case 70: case 71:
  case 72: case 73: case 74: case 75:
  case 76: case 77: case 78: case 79:
  case 80: case 81: case 82: case 83:
  case 84: case 85: case 86: case 87:
  case 88: case 89: case 90: case 91:
  case 92: case 93: case 94: case 95:
    printf("push Global Variable[%d]", bp[0] & 31);
    if (literalVec) {
      printf(" = ");
      printAssociationKey(literalVec[bp[0] & 31]);
    }
    break;

  case	96: case  97: case  98: case  99:
  case 100: case 101: case 102: case 103:
    printf("pop and store Instance Variable[%d]", bp[0] & 7);
    break;

  case 104: case 105: case 106: case 107:
  case 108: case 109: case 110: case 111:
    printf("pop and store Temporary[%d]", bp[0] & 7);
    break;

  case 112:
    printf("push self");
    break;

  case 113:
    printf("push true");
    break;

  case 114:
    printf("push false");
    break;

  case 115:
    printf("push nil");
    break;

  case 116:
    printf("push -1");
    break;

  case 117:
    printf("push 0");
    break;

  case 118:
    printf("push 1");
    break;

  case 119:
    printf("push 2");
    break;

  case 120:
    printf("return self");
    break;

  case 121:
    printf("return true");
    break;

  case 122:
    printf("return false");
    break;

  case 123:
    printf("return nil");
    break;

  case 124:
    printf("return explicitly from method");
    break;

  case 125:
    printf("return stack top");
    break;

  case 126:
    switch (bp[1] >> 6) {
    case 0:
      printf("push Literal");
      break;
    case 1:
      printf("push Global Variable");
      break;
    case 2:
      printf("pop and store Global Variable");
      break;
    case 3:
      printf("store Global Variable");
      break;
    }
    printf("[%d]", (bp[1] & 63) * 256 + bp[2]);
    if (literalVec) {
      printf(" = ");
      if (bp[1] >> 6) {
	printAssociationKey(literalVec[(bp[1] & 63) * 256 + bp[2]]);
      } else {
	printObject(literalVec[(bp[1] & 63) * 256 + bp[2]]);
      }
    }
    break;

  case 128:
    printf("push ");
    printIndexed(bp[1], literalVec, true);
    break;

  case 129:
    printf("store ");
    printIndexed(bp[1], literalVec, false);
    break;

  case 130:
    printf("pop and store ");
    printIndexed(bp[1], literalVec, false);
    break;

  case 131:
    printf("send selector %d, %d args", bp[1] & 31, bp[1] >> 5);
    if (literalVec) {
      printf(" = ");
      printSymbol(literalVec[bp[1] & 31]);
    }
    break;

  case 132:
    lit = bp[2] + ((bp[1] & 192) << 2);
    printf("send %sselector %d, %d args",
      bp[1] & extendedSendSuperFlag ? "to super " : "", lit, bp[1] & 31);
    if (literalVec) {
      printf(" = ");
      printSymbol(literalVec[lit]);
    }
    break;

  case 133:
    printf("send to super selector %d, %d args", bp[1] & 31, bp[1] >> 5);
    if (literalVec) {
      printf(" = ");
      printSymbol(literalVec[bp[1] & 31]);
    }
    break;

  case 134:
    switch (bp[1] >> 6) {
    case 0:
      printf("pop and store");
      break;
    case 1:
      printf("push");
      break;
    case 2:
      printf("pop and store");
      break;
    case 3:
      printf("store");
      break;
    }

    printf(" Instance Variable[%d]", (bp[1] & 63) * 256 + bp[2]);
    if ((bp[1] >> 6) == 0) {
      printf(" of new stack top");
    }
    break;

  case 135:
    printf("pop stack top");
    break;

  case 136:
    printf("duplicate stack top");
    break;

  case 137:
    printf("push current context");
    break;

  case 138:
    switch (bp[1] >> 6) {
    case 0:
      printf("(invalid)");
      break;
    case 1:
      printf("push");
      break;
    case 2:
      printf("pop and store");
      break;
    case 3:
      printf("store");
      break;
    }
    printf(" outer var scopes = %d varIndex = %d", bp[2], bp[1] & 63);
    break;

  case 139:
    printf("no operation");
    break;

  case 140:
    printf("set stack top to self");
    break;
  
  case 141:
    printf("set stack top to 1");
    break;
  
  case 142:
    printf("set stack top to ");
    printIndexed(bp[1], literalVec, true);
    break;

  case 143:
    printf("terminate interpreter");
    break;

  case 144: case 145: case 146: case 147:
  case 148: case 149: case 150: case 151:
    printf("jump to %d", (bp[0] & 7) + ip + 1 + 1);
    break;

  case 152: case 153: case 154: case 155:
  case 156: case 157: case 158: case 159:
    printf("jump to %d if false", (bp[0] & 7) + ip + 1 + 1);
    break;

  case 160: case 161: case 162: case 163:
  case 164: case 165: case 166: case 167:
    printf("jump to %d", ((bp[0]&7)-4) * 256 + bp[1] + ip + 2);
    break;

  case 168: case 169: case 170: case 171:
    printf("pop and jump to %d if true", (bp[0]&3) * 256 + bp[1] + ip + 2);
    break;

  case 172: case 173: case 174: case 175:
    printf("pop and jump to %d if false", (bp[0]&3) * 256 + bp[1] + ip + 2);
    break;

  case 176: case 177: case 178: case 179:
  case 180: case 181: case 182: case 183:
  case 184: case 185: case 186: case 187:
  case 188: case 189: case 190: case 191:
    printf("send arithmetic message \"%s\"", mathMessageName[bp[0] & 15]);
    break;

  case 192: case 193: case 194: case 195:
  case 196: case 197: case 198: case 199:
  case 200: case 201: case 202: case 203:
  case 204: case 205: case 206: case 207:
    printf("send special message \"%s\"", specialMessageName[bp[0] & 15]);
    break;

  case 208: case 209: case 210: case 211:
  case 212: case 213: case 214: case 215:
  case 216: case 217: case 218: case 219:
  case 220: case 221: case 222: case 223:
    printf("send selector %d, 0 args", bp[0] & 15);
    if (literalVec) {
      printf(" = ");
      printSymbol(literalVec[bp[0] & 15]);
    }
    break;

  case 224: case 225: case 226: case 227:
  case 228: case 229: case 230: case 231:
  case 232: case 233: case 234: case 235:
  case 236: case 237: case 238: case 239:
    printf("send selector %d, 1 arg", bp[0] & 15);
    if (literalVec) {
      printf(" = ");
      printSymbol(literalVec[bp[0] & 15]);
    }
    break;

  case 240: case 241: case 242: case 243:
  case 244: case 245: case 246: case 247:
  case 248: case 249: case 250: case 251:
  case 252: case 253: case 254: case 255:
    printf("send selector %d, 2 args", bp[0] & 15);
    if (literalVec) {
      printf(" = ");
      printSymbol(literalVec[bp[0] & 15]);
    }
    break;

  default:
    printf("INVALID BYTECODE %d", bp[0]);
    break;
  }
}


void compileByteCodes (from, to)
     register Byte *from, *to;
{
  if (currentByteCodes == nil) {
    currentByteCodes = allocByteCodes(BYTE_CODE_CHUNK_SIZE + (to - from));
  } else {
    register int free;
    free = currentByteCodes->maxLen -
      (currentByteCodes->ptr - currentByteCodes->base);

    if (free < (to - from)) {
      memcpy(currentByteCodes->ptr, from, free);
      currentByteCodes->ptr += free;
      from += free;
      reallocByteCodes(currentByteCodes, BYTE_CODE_CHUNK_SIZE + (to - from));
    }
  }

  memcpy(currentByteCodes->ptr, from, to - from);
  currentByteCodes->ptr += to - from;
}

ByteCodes
allocByteCodes (size)
     int size;
{
  ByteCodes newByteCodes;

  newByteCodes = (ByteCodes)xmalloc(sizeof(struct ByteCodeArray));
  newByteCodes->base = (Byte *)xmalloc(size);
  newByteCodes->ptr = newByteCodes->base;
  newByteCodes->maxLen = size;

  return (newByteCodes);
}

void
reallocByteCodes (byteCodes, delta)
     ByteCodes byteCodes;
     int   delta;
{
#ifndef OPTIMIZE
  if (byteCodes->maxLen != (byteCodes->ptr - byteCodes->base)) {
    errorf("reallocByteCodes called with maxLen != byteCode len");
  }
#endif

  byteCodes->base = (Byte *)xrealloc(byteCodes->base, byteCodes->maxLen + delta);
  byteCodes->ptr  = byteCodes->base + byteCodes->maxLen;
  byteCodes->maxLen += delta;
}

