/******************************** -*- C -*- ****************************
 *
 *	Byte Code Interpreter Module.
 *	This interprets the compiled bytecodes of a method.
 *
 *	$Revision: 1.95.1$
 *	$Date: 2000/12/27 10:45:49$
 *	$Author: pb$
 *
 ***********************************************************************/


/***********************************************************************
 *
 * Copyright 1988-92, 1994-95, 1999, 2000 Free Software Foundation, Inc.
 * Written by Steve Byrne.
 *
 * This file is part of GNU Smalltalk.
 *
 * GNU Smalltalk is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2, or (at your option) any later
 * version.
 *
 * GNU Smalltalk is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * GNU Smalltalk; see the file COPYING.	 If not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 ***********************************************************************/

/*
 * This is basically how the interpreter works:
 * 
 * The interpreter expects to be called in an environment where there
 * already exists a well-defined method context.  The instruction pointer,
 * stored in the global variable "ip", and the stack pointer, stored in the
 * global variable "sp", should be set up to point into the current
 * method and MethodContext.  Other global variables, such as "thisMethod",
 * "self", "temporaries", etc. should also be setup.  See the routine
 * prepareExecutionEnvironment for details.
 * 
 * The interpreter checks to see if any change in its state is required,
 * such as switching to a new process, dealing with an asynchronous signal
 * and printing out the byte codes that are being executed, if that was
 * requested by the user.
 * 
 * After that, the byte code that ip points to is fetched and decoded.
 * Some byte codes perform jumps, which are performed by merely adjusting
 * the value of ip.  Some are message sends, which are described in
 * more detail below.  Some instructions require more than one byte code
 * to perform their work; ip is advanced as needed and the extension
 * byte codes are fetched.  Some byte codes, which push booleans, are
 * usually followed by jump byte codes; in this case, and if JUMP_LOOKAHEAD
 * is defined, the two byte codes are merged for speed.
 * 
 * After dispatching the byte code, the interpreter loops around to
 * execute another byte code.  A particular bytecode signals that the
 * execution of the method is over, and that the interpreter should return
 * to its caller.  This bytecode is never generated by the compiler, it
 * is only present in a private #__terminate method that is generated when
 * bootstrapping.
 *
 * Note that the interpreter is not called recursively to implement message
 * sends.  Rather the state of the interpreter is saved away in the currently
 * executing context, and a new context is created and the global variables
 * such as ip, sp, and temporaries are initialized accordingly.
 *
 * When a message send occurs, the sendMessageInternal routine is invoked.  It
 * determines the class of the receiver, and checks to see if it already has
 * cached the method definition for the given selector and receiver class.
 * If so, that method is used, and if not, the receiver's method dictionary
 * is searched for a method with the proper selector.  If it's not found in
 * that method dictionary, the method dictionary of the classes parent is
 * examined, and on up the hierarchy, until a matching selector is found.
 *
 * If no selector is found, the receiver is sent a #doesNotUnderstand: message
 * to indicate that a matching method could not be found.  The stack is
 * modified, pushing a Message object that embeds information about the
 * original selector and arguments, and sendMessageInternal calls itself
 * recursively to look up #doesNotUnderstand:.  Note that if the object
 * does not understand in turn the #doesNotUnderstand: message, a crash is
 * extremely likely; things like this are however to be expected, since
 * you're really playing a bad game and going against some logical things
 * that the VM assumes for speed.
 *
 * If a method is found, it is examined for some special cases.	 The special
 * cases are primitive return of self, return of an instance variable, return
 * of a literal object, or execution of a primitive method definition.	This
 * latter operation is performed by the executePrimitiveOperation routine.  If
 * the execution of this primitive interpreter fails, the normal message send
 * operation is performed.
 *
 * If the found method is not one of the special cases, or if it is a
 * primitive that failed to execute, a "normal" message send is performed.
 * This basically entails saving away what state the interpreter has, such as
 * the values of ip, and sp, being careful to save their relative locations
 * and not their physical addresses, because one or more garbage collections
 * could occur before the method context is returned to, and the absolute
 * pointers would be invalid.
 *
 * The sendMessage routine then creates a new MethodContext object, makes
 * its parent be the currently executing MethodContext, and sets up
 * the interpreters global variables to reference the new method and
 * new MethodContext.  Once those variables are set, sendMessage returns
 * to the interpreter, which cheerfully begins executing the new method,
 * totally unaware that the method that it was executing has changed.
 *
 * When a method returns, the context that called it is examined to restore
 * the interpreter's global variables to the state that they were in before
 * the callee was invoked.  The values of ip and sp are restored to their
 * absolute address values, and the other global state variables are
 * restored accordingly.  After the state has been restored, the
 * interpreter continues execution, again totally oblivious to the fact
 * that it's not running the same method it was on its previous byte code.
 *
 * Blocks are similarly implemented by sendBlockValue, which is simpler
 * than sendMessageInternal however, because it contains no check for
 * special cases, and no method lookup logic.  Unlike the Blue Book, GNU
 * Smalltalk stores bytecodes for blocks into separate CompiledBlock
 * objects, not into the same CompiledMethods that holds the containing
 * bytecodes.  sendBlockValue expects to find a BlockClosure on the
 * stack, and this BlockClosure object points to the CompiledBlock object
 * to be activated.
 */


#if defined(USE_GCC_DISPATCH) && !defined(PIPELINING)
static void		**globalMonitoredByteCodes, **globalNormalByteCodes;
static void		**dispatchVec;
#endif

#define getContextIP(ctx) 	toInt((ctx)->ipOffset)

#define setThisMethod(method, ipOffset) {				\
  register Method _method = (Method) oopToObj(thisMethod = (method));	\
  methodBase = _method->bytecodes;					\
  literals = oopToObj(_method->literals)->data;				\
  ip = methodBase + (ipOffset);						\
}

#define sendToSuper(sendSelector, sendArgs, dummy) \
    { register OOP _methodClass; \
      _methodClass = getMethodClass(thisMethod); \
      _methodClass = superClass(_methodClass); \
      sendMessageInternal(sendSelector, sendArgs, self, _methodClass); }

#ifdef OPTIMIZE
#define instanceVariable(receiver, index) \
  (oopToObj(receiver)->data[index])

#define storeInstanceVariable(receiver, index, oop) \
  oopToObj(receiver)->data[index] = (oop)

#define inBounds(oop, index) true
#else
#define instanceVariable(receiver, index) \
  (inBounds(receiver, index) ? oopToObj(receiver)->data[index] \
    : (errorf("Index out of bounds %d", index), debug(), nilOOP))

#define storeInstanceVariable(receiver, index, oop) \
{  \
  OOP __storeRecVarOOP = (oop); \
  if (!inBounds(receiver, index)) { \
    errorf("Index out of bounds %d", index); \
    debug(); \
  } \
  oopToObj(receiver)->data[index] = __storeRecVarOOP; \
}

#define inBounds(oop, index) ( \
	((unsigned int)(index)) < (unsigned int)numOOPs(oopToObj(oop)) )
#endif /* OPTIMIZE */

#if REG_AVAILABILITY >= 2 && defined(LOCAL_REGS)
#define receiverVariable(index)		  instanceVariable(selfCache, index)
#define methodTemporary(index) 		  tempCache[index]
#define methodLiteral(index) 		  litCache[index]
#define storeReceiverVariable(index, oop) storeInstanceVariable(selfCache, index, oop)
#define storeMethodTemporary(index, oop)  tempCache[index] = (oop)
#define storeMethodLiteral(index, oop)    litCache[index] = (oop)
#else
#define receiverVariable(index)		  instanceVariable(self, index)
#define methodTemporary(index) 		  temporaries[index]
#define methodLiteral(index) 		  literals[index]
#define storeReceiverVariable(index, oop) storeInstanceVariable(self, index, oop)
#define storeMethodTemporary(index, oop)  temporaries[index] = (oop)
#define storeMethodLiteral(index, oop)    literals[index] = (oop)
#endif

#define methodVariable(index) \
  associationValue(methodLiteral(index))

#define storeMethodVariable(index, oop) \
  setAssociationValue(methodLiteral(index), oop)

/* On entry to this routine, the stack should have the receiver and the
 * arguments pushed on the stack.  We need to get a new context, setup
 * things like the IP, SP, and Temporary pointers, and then return.   Note
 * that this routine DOES NOT invoke the interpreter; it merely sets up a
 * new context so that calling (or, more typically, returning to) the
 * interpreter will operate properly.  This kind of sending is for normal
 * messages only.  Things like sending a "value" message to a block context are
 * handled by primitives which do similar things, but they use information from
 * BlockClosure objects that we don't have available (nor need) here.
 */

void
sendMessageInternal(sendSelector, sendArgs, receiver, methodClass)
     OOP	sendSelector;
     int	sendArgs;
     OOP	receiver;
     OOP	methodClass;	/* the class in which to start the search */
{
  long				hashIndex;
  REGISTER(1, MethodContext	newContext);
  REGISTER(2, MethodCacheEntry	*methodData);
  REGISTER(3, MethodHeader	header);

  /* hash the selector and the class of the receiver together using XOR.
   * Since both are addresses in the oopTable, and since oopTable entries
   * are 2 longs in size, shift over by 3 bits (4 on 64-bit architectures)
   * to remove the useless low order zeros. */

#ifdef PROFBLOCK
  ps.numMessageSends++;
#endif
  sampleCounter++;
  hashIndex = methodCacheHash(sendSelector, methodClass);
  methodData = &methodCache[hashIndex];

  if (methodData->selectorOOP != sendSelector
      || methodData->startingClassOOP != methodClass) {
    /* :-( cache miss )-: */
    if (!lookupMethod (sendSelector, methodData, sendArgs, methodClass)) {
      sendMessage(doesNotUnderstandColonSymbol, 1, false);
      return;
    }
  }

  header = methodData->methodHeader;

#ifndef OPTIMIZE
#ifdef DEBUG_CODE_FLOW
  {
#else /* !DEBUG_CODE_FLOW */
  if (header.numArgs != (unsigned)sendArgs) {
#endif /* !DEBUG_CODE_FLOW */
    OOP receiverClass;
    if (isInt(receiver)) {
      receiverClass = smallIntegerClass;
    } else {
      receiverClass = oopClass(receiver);
    }
    printObject(receiverClass);
    if (receiverClass != methodData->methodClassOOP) {
      printf("(");
      printObject(methodData->methodClassOOP);
      printf(")");
    }
    printf(">>");
    printObject(methodData->selectorOOP);
    printf("\n");
    if (header.numArgs != (unsigned)sendArgs) {
      debug();
      errorf("invalid number of arguments %d, expecting %d", sendArgs,
	     header.numArgs);

      return;
    }
  }
#endif /* !OPTIMIZE */

  if (header.headerFlag) {
    switch (header.headerFlag) {
    case 1:
      /* 1, return the receiver - self is already on the stack...so we leave it */
      selfReturns++;
      return;

    case 2: {
      register long primIndex = header.primitiveIndex;
      /* 2, return instance variable */
      /* replace receiver with the returned instance variable */
      setStackTop(instanceVariable(receiver, primIndex));
      instVarReturns++;
      return;
    }

    case 3: {
      /* 3, return literal constant */
      /* replace receiver with the returned literal constant */
      setStackTop( getMethodLiterals(methodData->methodOOP) [0] );
      literalReturns++;
      return;
    }

    case 4:
      if (!primitiveTable[header.primitiveIndex].primFunc (
	header.primitiveIndex, sendArgs, methodData->methodOOP)) {

	/* primitive succeeded.	 Continue with the parent context */
	return;
      }
      /* primitive failed.  Invoke the normal method.  methodData
       * may be clobbered by a setjmp in executePrimitiveOperation */
      methodData = &methodCache[hashIndex];
      break;

    case 0:	/* only here so that the compiler skips a range check */
    case 5:
    case 6:
    case 7:
    default: break;
    }
  }

#ifdef PROFBLOCK
  ps.stackDepth++;
  if (ps.stackDepth > ps.maxStackDepth) {
    ps.maxStackDepth = ps.stackDepth;
  }
#endif

  /* prepare new state */
  newContext = activateNewContext(header.stackDepth, sendArgs);
  newContext->flags = fromInt(0);
  setThisMethod(methodData->methodOOP, 0);
  self = receiver;

  /* push args and temps, set sp and temporaries */
  prepareContext(newContext, sendArgs, header.numTemps);
}

void
sendMethod(methodOOP)
     OOP	methodOOP;
{
  int sendArgs;
  OOP receiver;
  REGISTER(1, Method	        method);
  REGISTER(2, MethodContext	newContext);
  REGISTER(3, MethodHeader	header);

#ifdef PROFBLOCK
  ps.numMessageSends++;
#endif
  sampleCounter++;

  method = (Method) oopToObj(methodOOP);
  header = method->header;
  sendArgs = header.numArgs;
  receiver = stackAt(sendArgs);

  if (header.headerFlag) {
    switch (header.headerFlag) {
    case 1:
      /* 1, return the receiver - self is already on the stack...so we leave it */
      selfReturns++;
      return;

    case 2: {
      register long primIndex = header.primitiveIndex;
      /* 2, return instance variable */
      /* replace receiver with the returned instance variable */
      setStackTop(instanceVariable(receiver, primIndex));
      instVarReturns++;
      return;
    }

    case 3: {
      /* 3, return literal constant */
      /* replace receiver with the returned literal constant */
      setStackTop( getMethodLiterals(methodOOP) [0] );
      literalReturns++;
      return;
    }

    case 4:
      if (!primitiveTable[header.primitiveIndex].primFunc (
	header.primitiveIndex, sendArgs, methodOOP)) {

	/* primitive succeeded.	 Continue with the parent context */
	return;
      }
      break;

    case 0:	/* only here so that the compiler skips a range check */
    case 5:
    case 6:
    case 7:
    default: break;
    }
  }

#ifdef PROFBLOCK
  ps.stackDepth++;
  if (ps.stackDepth > ps.maxStackDepth) {
    ps.maxStackDepth = ps.stackDepth;
  }
#endif

  /* prepare new state */
  newContext = activateNewContext(header.stackDepth, sendArgs);
  newContext->flags = fromInt(0);
  setThisMethod(methodOOP, 0);
  self = receiver;

  /* push args and temps, set sp and temporaries */
  prepareContext(newContext, sendArgs, header.numTemps);
}


/*
 *	static mst_Boolean sendBlockValue(numArgs)
 *
 * Description
 *
 *	This is the equivalent of sendMessage, but is for blocks.  The block
 *	context that is to the the receiver of the "value" message should be
 *	"numArgs" into the stack.  SP is set to the top of the arguments in the
 *	block context, which have been copied out of the caller's context.
 *
 * Inputs
 *
 *	numArgs:
 *		The number of arguments sent to the block.
 *
 * Outputs
 *
 *	true if failed, false if numArgs matches what the BlockClosure says.
 */
static mst_Boolean
sendBlockValue(numArgs)
     int	numArgs;
{
  OOP				closureOOP;
  REGISTER(1, BlockContext	blockContext);
  REGISTER(2, BlockClosure	closure);
  REGISTER(3, BlockHeader	header);

  closureOOP = stackAt(numArgs);
  closure = (BlockClosure)oopToObj(closureOOP);
  header = ((Block)oopToObj(closure->block))->header;
  if(numArgs != header.numArgs) {
    /* check numArgs asap */
    return (true);
  }

#ifdef PROFBLOCK
  ps.numValues++;
  ps.stackDepth++;
  if (ps.stackDepth > ps.maxStackDepth) {
    ps.maxStackDepth = ps.stackDepth;
  }
#endif

  /* prepare the new state, loading data from the closure */
  /* gc might happen - so reload everything. */
  blockContext = (BlockContext)activateNewContext(header.depth, numArgs);
  closure = (BlockClosure)oopToObj(closureOOP);
  blockContext->outerContext = closure->outerContext;
  self = closure->receiver;
  setThisMethod(closure->block, 0);

  /* push args and temps */
  prepareContext(blockContext, numArgs, header.numTemps);
  return(false);
}

void
validateMethodCacheEntries()
{
}



void
interpret()
{
/******************** REGISTER DECLARATIONS *************************/

#ifdef LOCAL_REGS
# undef  sp
# undef  ip

#if REG_AVAILABILITY == 0
# define localCounter    byteCodeCounter
# define exportRegs()	{ outerSP = sp; outerIP = ip; }
#else
  register int localCounter = 0;
# define exportRegs()	{ outerSP = sp; outerIP = ip; \
  			  byteCodeCounter += localCounter; localCounter = 0; }
#endif

/* If we have a good quantity of registers, activate more caching mechanisms. */
#if  REG_AVAILABILITY >= 2
  register OOP selfCache, *tempCache, *litCache;
  register OOP myNilOOP = nilOOP, myTrueOOP = trueOOP, myFalseOOP = falseOOP;

# define nilOOP myNilOOP
# define trueOOP myTrueOOP
# define falseOOP myFalseOOP
# define importRegs()	{ sp = outerSP; ip = outerIP; \
  			  selfCache = self; tempCache = temporaries; \
  			  litCache = literals; }

#else
  #define importRegs()	{ sp = outerSP; ip = outerIP; }
#endif

  REGISTER(1, IPType	ip);
  REGISTER(2, OOP	*sp);
#else
#  define exportRegs()
#  define importRegs()
#endif /* LOCAL_REGS */


#ifndef USE_GCC_DISPATCH

  /* Most of the difference in speed between switch-based dispatch and
   * label-based dispatch comes from the fact that all bytecodes pass
   * through an if in the former case, while when using labels we can
   * use dispatchVec instead of an if statement.
   *
   * Note that the `goto lab' in BC is completely gratuitous and
   * it's only there to shut up the compiler's warnings. */

#define INTERPRETER		for(;;)
#define MONITOR_CODE		  if (exceptFlag)
#define BYTECODES		  switch(*ip)
#define BC(num, lab)		    goto lab; case num: lab: START
#define NEXT_BC			      break

#else /* defined(USE_GCC_DISPATCH) */
  /* Indirect threaded bytecode interpretation when using GCC.	NormalBytecodes
   * is indexed by bytecode and holds onto the address of the label to jump to
   * to execute that byte code.	 MonitoredByteCodes has all byte codes jump
   * to a common place to check for exceptional conditions, and then jump thru
   * normalByteCodes.  DispatchVec points normally at normalByteCodes, but when
   * there is an exceptional condition, it points at monitoredByteCodes.
   * TrueByteCodes and falseByteCodes are used (if JUMP_LOOKAHEAD is defined)
   * to dispatch conditional jumps immediately following comparisons (see above)
   * without pushing and popping the result of the conditional.
   */

  static void *normalByteCodes[] = {
    /* 1-byte Push bytecodes */
    &&pushRecVar0,    &&pushRecVar1,	&&pushRecVar2,	  &&pushRecVar3,	/* 0 */
    &&pushRecVar4,    &&pushRecVar5,	&&pushRecVar6,	  &&pushRecVar7,	/* 4 */
    &&pushRecVar8,    &&pushRecVar9,	&&pushRecVar10,	  &&pushRecVar11,	/* 8 */
    &&pushRecVar12,   &&pushRecVar13,	&&pushRecVar14,	  &&pushRecVar15,	/* 12 */
    &&pushTemp0,      &&pushTemp1,	&&pushTemp2,	  &&pushTemp3,		/* 16 */
    &&pushTemp4,      &&pushTemp5,	&&pushTemp6,	  &&pushTemp7,		/* 20 */
    &&pushTemp8,      &&pushTemp9,	&&pushTemp10,	  &&pushTemp11,		/* 24 */
    &&pushTemp12,     &&pushTemp13,	&&pushTemp14,	  &&pushTemp15,		/* 28 */
    &&pushLit0,	      &&pushLit1,	&&pushLit2,	  &&pushLit3,		/* 32 */
    &&pushLit4,	      &&pushLit5,	&&pushLit6,	  &&pushLit7,		/* 36 */
    &&pushLit8,	      &&pushLit9,	&&pushLit10,	  &&pushLit11,		/* 40 */
    &&pushLit12,      &&pushLit13,	&&pushLit14,	  &&pushLit15,		/* 44 */
    &&pushLit16,      &&pushLit17,	&&pushLit18,	  &&pushLit19,		/* 48 */
    &&pushLit20,      &&pushLit21,	&&pushLit22,	  &&pushLit23,		/* 52 */
    &&pushLit24,      &&pushLit25,	&&pushLit26,	  &&pushLit27,		/* 56 */
    &&pushLit28,      &&pushLit29,	&&pushLit30,	  &&pushLit31,		/* 60 */
    &&pushVar0,	      &&pushVar1,	&&pushVar2,	  &&pushVar3,		/* 64 */
    &&pushVar4,	      &&pushVar5,	&&pushVar6,	  &&pushVar7,		/* 68 */
    &&pushVar8,	      &&pushVar9,	&&pushVar10,	  &&pushVar11,		/* 72 */
    &&pushVar12,      &&pushVar13,	&&pushVar14,	  &&pushVar15,		/* 76 */
    &&pushVar16,      &&pushVar17,	&&pushVar18,	  &&pushVar19,		/* 80 */
    &&pushVar20,      &&pushVar21,	&&pushVar22,	  &&pushVar23,		/* 84 */
    &&pushVar24,      &&pushVar25,	&&pushVar26,	  &&pushVar27,		/* 88 */
    &&pushVar28,      &&pushVar29,	&&pushVar30,	  &&pushVar31,		/* 92 */

    /* 1-byte Pop/Store bytecodes */
    &&stRecVar0,      &&stRecVar1,	&&stRecVar2,	  &&stRecVar3,		/* 96 */
    &&stRecVar4,      &&stRecVar5,	&&stRecVar6,	  &&stRecVar7,		/* 100 */
    &&stTemp0,	      &&stTemp1,	&&stTemp2,	  &&stTemp3,		/* 104 */
    &&stTemp4,	      &&stTemp5,	&&stTemp6,	  &&stTemp7,		/* 108 */

    /* Special push bytecodes */
    &&pushSelf,	      &&pushTrue,	&&pushFalse,	  &&pushNil,		/* 112 */
    &&pushNeg1,	      &&push0,		&&push1,	  &&push2,		/* 116 */

    /* Return bytecodes */
    &&retSelf,	      &&retTrue,	&&retFalse,	  &&retNil,		/* 120 */
    &&explicitRet,    &&retStackTop,						/* 124 */

    /* All sorts of bytecodes */
    &&bigLiteralOp,   &&unused127,						/* 126 */

    /* Long bytecodes */
    &&pushIndexedVal, &&storeStackTop,	&&popAndStoreTop,			/* 128 */
    &&sendShort,      &&sendLong,	&&supSendShort,	  &&bigInstanceOp,	/* 131 */

    /* More stack bytecodes */
    &&popStack,	      &&dupStack,	&&pushThisContext,			/* 135 */
    &&outerTempOp,    &&nop,		&&topSelf,				/* 138 */
    &&topOne,	      &&topIndexedVal,						/* 141 */

    /* Exit interpreter bytecode */
    &&endExecution,								/* 143 */

    /* Jump bytecodes */
    &&shJmp1,	      &&shJmp2,		&&shJmp3,	  &&shJmp4,		/* 144 */
    &&shJmp5,	      &&shJmp6,		&&shJmp7,	  &&shJmp8,		/* 148 */
    &&shJmp1False,    &&shJmp2False,	&&shJmp3False,	  &&shJmp4False,	/* 152 */
    &&shJmp5False,    &&shJmp6False,	&&shJmp7False,	  &&shJmp8False,	/* 156 */
    &&longJmpNeg4,    &&longJmpNeg3,	&&longJmpNeg2,	  &&longJmpNeg1,	/* 160 */
    &&longJmp0,	      &&longJmp1,	&&longJmp2,	  &&longJmp3,		/* 164 */
    &&popJmpTrue0,    &&popJmpTrue1,	&&popJmpTrue2,	  &&popJmpTrue3,	/* 168 */
    &&popJmpFalse0,   &&popJmpFalse1,	&&popJmpFalse2,	  &&popJmpFalse3,	/* 172 */

    /* Special 1-byte send bytecodes */
    &&sendPlus,	      &&sendMinus,	&&sendLess,	  &&sendGreater,	/* 176 */
    &&sendLessEq,     &&sendGreaterEq,	&&sendEqual,	  &&sendNotEqual,	/* 180 */
    &&sendTimes,      &&sendDivide,	&&sendRemainder,  &&sendAtSign,		/* 184 */
    &&sendBitShift,   &&sendIntDivide,	&&sendBitAnd,	  &&sendBitOr,		/* 188 */
    &&sendAt,	      &&sendAtPut,	&&sendSize,	  &&sendNext,		/* 192 */
    &&sendNextPut,    &&sendAtEnd,	&&sendSameObject, &&sendClass,		/* 196 */
    &&sendBlockCopy,  &&sendValue,	&&sendValueColon, &&sendDoColon,	/* 200 */
    &&sendNew,	      &&sendNewColon,	&&sendIsNil,	  &&sendNotNil,		/* 204 */

    /* General 1-byte send bytecodes */
    &&sendNoArg0,     &&sendNoArg1,	&&sendNoArg2,	  &&sendNoArg3,		/* 208 */
    &&sendNoArg4,     &&sendNoArg5,	&&sendNoArg6,	  &&sendNoArg7,		/* 212 */
    &&sendNoArg8,     &&sendNoArg9,	&&sendNoArg10,	  &&sendNoArg11,	/* 216 */
    &&sendNoArg12,    &&sendNoArg13,	&&sendNoArg14,	  &&sendNoArg15,	/* 220 */
    &&sendOneArg0,    &&sendOneArg1,	&&sendOneArg2,	  &&sendOneArg3,	/* 224 */
    &&sendOneArg4,    &&sendOneArg5,	&&sendOneArg6,	  &&sendOneArg7,	/* 228 */
    &&sendOneArg8,    &&sendOneArg9,	&&sendOneArg10,	  &&sendOneArg11,	/* 232 */
    &&sendOneArg12,   &&sendOneArg13,	&&sendOneArg14,	  &&sendOneArg15,	/* 236 */
    &&sendTwoArg0,    &&sendTwoArg1,	&&sendTwoArg2,	  &&sendTwoArg3,	/* 240 */
    &&sendTwoArg4,    &&sendTwoArg5,	&&sendTwoArg6,	  &&sendTwoArg7,	/* 244 */
    &&sendTwoArg8,    &&sendTwoArg9,	&&sendTwoArg10,	  &&sendTwoArg11,	/* 248 */
    &&sendTwoArg12,   &&sendTwoArg13,	&&sendTwoArg14,	  &&sendTwoArg15	/* 252 */
  };

#define MONITOR_CODE			monitorByteCodes:		\
					  if(exceptFlag)

#define BC(num, lab)			lab: START

#ifdef PIPELINING
  register Byte b1, b2, b3, b4, b5;		/* pre-fetch queue */
  register void *t1, *t2;			/* pre-decode queue */
  BRANCH_REGISTER(t);

#define BYTE1 b1
#define BYTE2 b2


#define LOAD1 (t = t1, t1 = t2, t2 = normalByteCodes[b3],     \
               b1 = b2, b2 = b3, b3 = b4, b4 = b5,            \
               b5 = ip[6], ip++)

#define LOAD2 (t = t2, t1 = normalByteCodes[b3],              \
               t2 = normalByteCodes[b4], b1 = b3, b2 = b4,    \
               b3 = b5, b4 = ip[6], b5 = ip[7], ip += 2)

#define LOAD3 (t = normalByteCodes[b3], t1 = normalByteCodes[b4],	\
               t2 = normalByteCodes[b5], b1 = b4, b2 = b5, b3 = ip[6],  \
               b4 = ip[7], b5 = ip[8], ip += 3)

#define FLUSH         goto monitorByteCodes; }

#define CHECK_NEXT    importRegs();                       \
                      t = normalByteCodes[*ip],           \
                      b1 = ip[1], b2 = ip[2], b3 = ip[3], \
                      b4 = ip[4], b5 = ip[5],             \
                      t1 = t2 = &&monitorByteCodes;       \
                      goto *t;

#define BYTECODES     t = normalByteCodes[*ip],                           \
                      b1 = ip[1], b2 = ip[2], b3 = ip[3],                 \
                      b4 = ip[4], b5 = ip[5],                             \
                      t1 = normalByteCodes[b1], t2 = normalByteCodes[b2]; \
                      goto *t;

#define NEXT_BC       goto *t;

#undef JUMP_LOOKAHEAD

#else /* !PIPELINING */

  static void *monitoredByteCodes[] = {
    [0 ... 255] = &&monitorByteCodes
  };


#define INTERPRETER					\
  globalNormalByteCodes = normalByteCodes;		\
  globalMonitoredByteCodes = monitoredByteCodes;	\
  dispatchVec = normalByteCodes;

#define BYTECODES			  goto *normalByteCodes[*ip];

#define NEXT_BC				  goto *dispatchVec[*ip]

#define LOAD				  REGISTER(3, void *prefetch); \
					  prefetch = dispatchVec[*++ip];
#define PREFETCH			  goto *prefetch; }

/* The default definition does nothing, so it is not valid for us */
#undef SET_EXCEPT_FLAG

#define SET_EXCEPT_FLAG(x)						\
  dispatchVec = x ? globalMonitoredByteCodes : globalNormalByteCodes

#ifdef JUMP_LOOKAHEAD
  static void *trueByteCodes[] = {
    [  0 ... 124] = &&lookaheadFailedTrue,		/* pot-pourri	 */

    &&retTrue,						/* return top	 */

    [126 ... 151] = &&lookaheadFailedTrue,		/* pot-pourri	 */
    [152 ... 159] = &&nop,				/* jump if false */
    [160 ... 167] = &&lookaheadFailedTrue,		/* unconditional */

    &&longJmp0,	      &&longJmp1,	&&longJmp2,	  &&longJmp3,

    [172 ... 175] = &&lookaheadLongJumpNotTaken,	/* jump if false */
    [176 ... 255] = &&lookaheadFailedTrue		/* pot-pourri */
  };

  static void *falseByteCodes[] = {
    [  0 ... 124] = &&lookaheadFailedFalse,		/* pot-pourri	 */

    &&retFalse,						/* return top	 */

    [126 ... 151] = &&lookaheadFailedFalse,		/* pot-pourri	 */

    &&shJmp1,	    &&shJmp2,		&&shJmp3,	  &&shJmp4,
    &&shJmp5,	    &&shJmp6,		&&shJmp7,	  &&shJmp8,

    [160 ... 167] = &&lookaheadFailedFalse,		/* unconditional */
    [168 ... 171] = &&lookaheadLongJumpNotTaken,	/* jump if true */

    &&longJmp0,	    &&longJmp1,		&&longJmp2,	  &&longJmp3,

    [176 ... 255] = &&lookaheadFailedFalse		/* pot-pourri */
  };

#define DO_JUMP_LOOKAHEAD(x) {		\
  if (x) {				\
    popNOOPs(1);			\
    goto *trueByteCodes[*ip];		\
  } else {				\
    popNOOPs(1);			\
    goto *falseByteCodes[*ip];		\
  }					\
}
#endif /* JUMP_LOOKAHEAD */

#endif /* !PIPELINING */
#endif /* USE_GCC_DISPATCH */

/******************** COMMON DEFINITIONS *********************************/

#ifdef OPTIMIZE
# define ILLEGAL(errMsg)
#else
# define ILLEGAL(errMsg)		errorf(errMsg)
#endif

#define EXIT_IF(what)		\
  if (what) {			\
    inInterpreter = false;	\
    ip = nil;			\
    return;			\
  }

#define BRANCH_LABEL(lab)	lab:

#ifdef PROFBLOCK
#define START			localCounter++; bytecodes[*ip]++; {
#else
#define START			localCounter++; {
#endif /* PROFBLOCK */

#define FALL(newLab)		goto newLab; }
#define END			NEXT_BC; }

/******************** DEFAULT DEFINITIONS ********************************/

#ifndef BYTE1
#define BYTE1		ip[1]
#define BYTE2		ip[2]
#define LOAD1		ip++
#define LOAD2		ip += 2
#define LOAD3		ip += 3
#define FLUSH		END
#endif

#define INCR1		ip++
#define INCR2		ip += 2
#define INCR3		ip += 3

#ifndef LOAD
#define LOAD		LOAD1
#define PREFETCH	END
#endif

#ifndef INTERPRETER
#define INTERPRETER
#endif

#ifndef CHECK_NEXT
#define CHECK_NEXT	setExceptFlag(true)
#endif

#ifndef DO_JUMP_LOOKAHEAD
#define DO_JUMP_LOOKAHEAD(x) {		\
  setStackTopBoolean(x);		\
  NEXT_BC;				\
}
#endif

/******************** BODY OF THE INTERPRETER ***********************/

  importRegs();
  EXIT_IF(!ip);

  inInterpreter = true;

  /* this is all we need to do -- since we only pay attention to the
   * execution tracing flag on entry to this routine, and since the first time
   * through we pass through the monitoring code, we can get away with just
   * exceptFlag here.
   */
  exceptFlag = executionTracing;

  INTERPRETER {
    OOP			returnedValue;

    MONITOR_CODE {
      exportRegs();
      if (asyncQueueIndex) {	/* deal with any async signals	*/
	register int	i;
	IntState	oldSigMask;
	oldSigMask = disableInterrupts(); /* block out everything! */
	for (i = 0; i < asyncQueueIndex; i++) {
	  syncSignal(queuedAsyncSignals[i]);
	}
	asyncQueueIndex = 0;
	enableInterrupts(oldSigMask);
      }
      if (!isNil(switchToProcess)) {
	/*exportRegs(); */
	changeProcessContext(switchToProcess);
	importRegs();
	EXIT_IF(!ip);
      }
      if (abortExecution) {
	OOP selectorOOP;
	selectorOOP = internString(abortExecution);
	abortExecution = nil;
	sendMessage(selectorOOP, 0, false);
      }
      setExceptFlag(false);
      if (executionTracing) {
	if (verboseExecTracing) {
	  printf("\t  --> ");
	  printObject(stackTop());
	  printf("\n");
	}
	printf(" %5ld:\t", (long) (ip - methodBase));
	printByteCodeName(ip, ip - methodBase, literals);
	printf("\n");
	CHECK_NEXT;
      }
      importRegs();
    }

    BYTECODES {
    BC(0,  pushRecVar0)
      LOAD;
      pushOOP(receiverVariable(0));
    PREFETCH;
    
    BC(1,  pushRecVar1)
      LOAD;
      pushOOP(receiverVariable(1));
    PREFETCH;
    
    BC(2,  pushRecVar2)
      LOAD;
      pushOOP(receiverVariable(2));
    PREFETCH;
    
    BC(3,  pushRecVar3)
      LOAD;
      pushOOP(receiverVariable(3));
    PREFETCH;
    
    BC(4,  pushRecVar4)
      LOAD;
      pushOOP(receiverVariable(4));
    PREFETCH;
    
    BC(5,  pushRecVar5)
      LOAD;
      pushOOP(receiverVariable(5));
    PREFETCH;
    
    BC(6,  pushRecVar6)
      LOAD;
      pushOOP(receiverVariable(6));
    PREFETCH;
    
    BC(7,  pushRecVar7)
      LOAD;
      pushOOP(receiverVariable(7));
    PREFETCH;
    
    BC(8,  pushRecVar8)
      LOAD;
      pushOOP(receiverVariable(8));
    PREFETCH;
    
    BC(9,  pushRecVar9)
      LOAD;
      pushOOP(receiverVariable(9));
    PREFETCH;
    
    BC(10, pushRecVar10)
      LOAD;
      pushOOP(receiverVariable(10));
    PREFETCH;
    
    BC(11, pushRecVar11)
      LOAD;
      pushOOP(receiverVariable(11));
    PREFETCH;
    
    BC(12, pushRecVar12)
      LOAD;
      pushOOP(receiverVariable(12));
    PREFETCH;
    
    BC(13, pushRecVar13)
      LOAD;
      pushOOP(receiverVariable(13));
    PREFETCH;
    
    BC(14, pushRecVar14)
      LOAD;
      pushOOP(receiverVariable(14));
    PREFETCH;
    
    BC(15, pushRecVar15)
      LOAD;
      pushOOP(receiverVariable(15));
    PREFETCH;


    BC(16, pushTemp0)
      LOAD;
      pushOOP(methodTemporary(0));
    PREFETCH;
    
    BC(17, pushTemp1)
      LOAD;
      pushOOP(methodTemporary(1));
    PREFETCH;
    
    BC(18, pushTemp2)
      LOAD;
      pushOOP(methodTemporary(2));
    PREFETCH;
    
    BC(19, pushTemp3)
      LOAD;
      pushOOP(methodTemporary(3));
    PREFETCH;
    
    BC(20, pushTemp4)
      LOAD;
      pushOOP(methodTemporary(4));
    PREFETCH;
    
    BC(21, pushTemp5)
      LOAD;
      pushOOP(methodTemporary(5));
    PREFETCH;
    
    BC(22, pushTemp6)
      LOAD;
      pushOOP(methodTemporary(6));
    PREFETCH;
    
    BC(23, pushTemp7)
      LOAD;
      pushOOP(methodTemporary(7));
    PREFETCH;
    
    BC(24, pushTemp8)
      LOAD;
      pushOOP(methodTemporary(8));
    PREFETCH;
    
    BC(25, pushTemp9)
      LOAD;
      pushOOP(methodTemporary(9));
    PREFETCH;
    
    BC(26, pushTemp10)
      LOAD;
      pushOOP(methodTemporary(10));
    PREFETCH;
    
    BC(27, pushTemp11)
      LOAD;
      pushOOP(methodTemporary(11));
    PREFETCH;
    
    BC(28, pushTemp12)
      LOAD;
      pushOOP(methodTemporary(12));
    PREFETCH;
    
    BC(29, pushTemp13)
      LOAD;
      pushOOP(methodTemporary(13));
    PREFETCH;
    
    BC(30, pushTemp14)
      LOAD;
      pushOOP(methodTemporary(14));
    PREFETCH;
    
    BC(31, pushTemp15)
      LOAD;
      pushOOP(methodTemporary(15));
    PREFETCH;


    BC(32, pushLit0)
      LOAD;
      pushOOP(methodLiteral(0));
    PREFETCH;
    
    BC(33, pushLit1)
      LOAD;
      pushOOP(methodLiteral(1));
    PREFETCH;
    
    BC(34, pushLit2)
      LOAD;
      pushOOP(methodLiteral(2));
    PREFETCH;
    
    BC(35, pushLit3)
      LOAD;
      pushOOP(methodLiteral(3));
    PREFETCH;
    
    BC(36, pushLit4)
      LOAD;
      pushOOP(methodLiteral(4));
    PREFETCH;
    
    BC(37, pushLit5)
      LOAD;
      pushOOP(methodLiteral(5));
    PREFETCH;
    
    BC(38, pushLit6)
      LOAD;
      pushOOP(methodLiteral(6));
    PREFETCH;
    
    BC(39, pushLit7)
      LOAD;
      pushOOP(methodLiteral(7));
    PREFETCH;
    
    BC(40, pushLit8)
      LOAD;
      pushOOP(methodLiteral(8));
    PREFETCH;
    
    BC(41, pushLit9)
      LOAD;
      pushOOP(methodLiteral(9));
    PREFETCH;
    
    BC(42, pushLit10)
      LOAD;
      pushOOP(methodLiteral(10));
    PREFETCH;
    
    BC(43, pushLit11)
      LOAD;
      pushOOP(methodLiteral(11));
    PREFETCH;
    
    BC(44, pushLit12)
      LOAD;
      pushOOP(methodLiteral(12));
    PREFETCH;
    
    BC(45, pushLit13)
      LOAD;
      pushOOP(methodLiteral(13));
    PREFETCH;
    
    BC(46, pushLit14)
      LOAD;
      pushOOP(methodLiteral(14));
    PREFETCH;
    
    BC(47, pushLit15)
      LOAD;
      pushOOP(methodLiteral(15));
    PREFETCH;
    
    BC(48, pushLit16)
      LOAD;
      pushOOP(methodLiteral(16));
    PREFETCH;
    
    BC(49, pushLit17)
      LOAD;
      pushOOP(methodLiteral(17));
    PREFETCH;
    
    BC(50, pushLit18)
      LOAD;
      pushOOP(methodLiteral(18));
    PREFETCH;
    
    BC(51, pushLit19)
      LOAD;
      pushOOP(methodLiteral(19));
    PREFETCH;
    
    BC(52, pushLit20)
      LOAD;
      pushOOP(methodLiteral(20));
    PREFETCH;
    
    BC(53, pushLit21)
      LOAD;
      pushOOP(methodLiteral(21));
    PREFETCH;
    
    BC(54, pushLit22)
      LOAD;
      pushOOP(methodLiteral(22));
    PREFETCH;
    
    BC(55, pushLit23)
      LOAD;
      pushOOP(methodLiteral(23));
    PREFETCH;
    
    BC(56, pushLit24)
      LOAD;
      pushOOP(methodLiteral(24));
    PREFETCH;
    
    BC(57, pushLit25)
      LOAD;
      pushOOP(methodLiteral(25));
    PREFETCH;
    
    BC(58, pushLit26)
      LOAD;
      pushOOP(methodLiteral(26));
    PREFETCH;
    
    BC(59, pushLit27)
      LOAD;
      pushOOP(methodLiteral(27));
    PREFETCH;
    
    BC(60, pushLit28)
      LOAD;
      pushOOP(methodLiteral(28));
    PREFETCH;
    
    BC(61, pushLit29)
      LOAD;
      pushOOP(methodLiteral(29));
    PREFETCH;
    
    BC(62, pushLit30)
      LOAD;
      pushOOP(methodLiteral(30));
    PREFETCH;
    
    BC(63, pushLit31)
      LOAD;
      pushOOP(methodLiteral(31));
    PREFETCH;


    BC(64, pushVar0) 
      LOAD;
      pushOOP(methodVariable(0));
    PREFETCH;
    
    BC(65, pushVar1) 
      LOAD;
      pushOOP(methodVariable(1));
    PREFETCH;
    
    BC(66, pushVar2) 
      LOAD;
      pushOOP(methodVariable(2));
    PREFETCH;
    
    BC(67, pushVar3) 
      LOAD;
      pushOOP(methodVariable(3));
    PREFETCH;
    
    BC(68, pushVar4) 
      LOAD;
      pushOOP(methodVariable(4));
    PREFETCH;
    
    BC(69, pushVar5) 
      LOAD;
      pushOOP(methodVariable(5));
    PREFETCH;
    
    BC(70, pushVar6) 
      LOAD;
      pushOOP(methodVariable(6));
    PREFETCH;
    
    BC(71, pushVar7) 
      LOAD;
      pushOOP(methodVariable(7));
    PREFETCH;
    
    BC(72, pushVar8) 
      LOAD;
      pushOOP(methodVariable(8));
    PREFETCH;
    
    BC(73, pushVar9) 
      LOAD;
      pushOOP(methodVariable(9));
    PREFETCH;
    
    BC(74, pushVar10)
      LOAD;
      pushOOP(methodVariable(10));
    PREFETCH;
    
    BC(75, pushVar11)
      LOAD;
      pushOOP(methodVariable(11));
    PREFETCH;
    
    BC(76, pushVar12)
      LOAD;
      pushOOP(methodVariable(12));
    PREFETCH;
    
    BC(77, pushVar13)
      LOAD;
      pushOOP(methodVariable(13));
    PREFETCH;
    
    BC(78, pushVar14)
      LOAD;
      pushOOP(methodVariable(14));
    PREFETCH;
    
    BC(79, pushVar15)
      LOAD;
      pushOOP(methodVariable(15));
    PREFETCH;
    
    BC(80, pushVar16)
      LOAD;
      pushOOP(methodVariable(16));
    PREFETCH;
    
    BC(81, pushVar17)
      LOAD;
      pushOOP(methodVariable(17));
    PREFETCH;
    
    BC(82, pushVar18)
      LOAD;
      pushOOP(methodVariable(18));
    PREFETCH;
    
    BC(83, pushVar19)
      LOAD;
      pushOOP(methodVariable(19));
    PREFETCH;
    
    BC(84, pushVar20)
      LOAD;
      pushOOP(methodVariable(20));
    PREFETCH;
    
    BC(85, pushVar21)
      LOAD;
      pushOOP(methodVariable(21));
    PREFETCH;
    
    BC(86, pushVar22)
      LOAD;
      pushOOP(methodVariable(22));
    PREFETCH;
    
    BC(87, pushVar23)
      LOAD;
      pushOOP(methodVariable(23));
    PREFETCH;
    
    BC(88, pushVar24)
      LOAD;
      pushOOP(methodVariable(24));
    PREFETCH;
    
    BC(89, pushVar25)
      LOAD;
      pushOOP(methodVariable(25));
    PREFETCH;
    
    BC(90, pushVar26)
      LOAD;
      pushOOP(methodVariable(26));
    PREFETCH;
    
    BC(91, pushVar27)
      LOAD;
      pushOOP(methodVariable(27));
    PREFETCH;
    
    BC(92, pushVar28)
      LOAD;
      pushOOP(methodVariable(28));
    PREFETCH;
    
    BC(93, pushVar29)
      LOAD;
      pushOOP(methodVariable(29));
    PREFETCH;
    
    BC(94, pushVar30)
      LOAD;
      pushOOP(methodVariable(30));
    PREFETCH;
    
    BC(95, pushVar31)
      LOAD;
      pushOOP(methodVariable(31));
    PREFETCH;


    BC(96,  stRecVar0)
      LOAD;
      storeReceiverVariable(0, popOOP());
    PREFETCH;
    
    BC(97,  stRecVar1)
      LOAD;
      storeReceiverVariable(1, popOOP());
    PREFETCH;
    
    BC(98,  stRecVar2)
      LOAD;
      storeReceiverVariable(2, popOOP());
    PREFETCH;
    
    BC(99,  stRecVar3)
      LOAD;
      storeReceiverVariable(3, popOOP());
    PREFETCH;
    
    BC(100, stRecVar4)
      LOAD;
      storeReceiverVariable(4, popOOP());
    PREFETCH;
    
    BC(101, stRecVar5)
      LOAD;
      storeReceiverVariable(5, popOOP());
    PREFETCH;
    
    BC(102, stRecVar6)
      LOAD;
      storeReceiverVariable(6, popOOP());
    PREFETCH;
    
    BC(103, stRecVar7)
      LOAD;
      storeReceiverVariable(7, popOOP());
    PREFETCH;


    BC(104, stTemp0)
      LOAD;
      storeMethodTemporary(0, popOOP());
    PREFETCH;
    
    BC(105, stTemp1)
      LOAD;
      storeMethodTemporary(1, popOOP());
    PREFETCH;
    
    BC(106, stTemp2)
      LOAD;
      storeMethodTemporary(2, popOOP());
    PREFETCH;
    
    BC(107, stTemp3)
      LOAD;
      storeMethodTemporary(3, popOOP());
    PREFETCH;
    
    BC(108, stTemp4)
      LOAD;
      storeMethodTemporary(4, popOOP());
    PREFETCH;
    
    BC(109, stTemp5)
      LOAD;
      storeMethodTemporary(5, popOOP());
    PREFETCH;
    
    BC(110, stTemp6)
      LOAD;
      storeMethodTemporary(6, popOOP());
    PREFETCH;
    
    BC(111, stTemp7)
      LOAD;
      storeMethodTemporary(7, popOOP());
    PREFETCH;


    BC(112, pushSelf)
      LOAD;
      uncheckedPushOOP(self);	
    PREFETCH;
    
    BC(113, pushTrue)
      LOAD;
      uncheckedPushOOP(trueOOP);
    PREFETCH;
    
    BC(114, pushFalse)
      LOAD;
      uncheckedPushOOP(falseOOP);
    PREFETCH;
    
    BC(115, pushNil)
      LOAD;
      uncheckedPushOOP(nilOOP);	
    PREFETCH;
    
    BC(116, pushNeg1)
      LOAD;
      pushInt(-1);		
    PREFETCH;
    
    BC(117, push0)
      LOAD;
      pushInt(0);		
    PREFETCH;
    
    BC(118, push1)
      LOAD;
      pushInt(1);		
    PREFETCH;
    
    BC(119, push2)
      LOAD;
      pushInt(2);		
    PREFETCH;



    BC(120, retSelf)
      returnedValue = self;
    FALL(doRetStackTop);
    
    BC(121, retTrue)
      returnedValue = trueOOP;
    FALL(doRetStackTop);
    
    BC(122, retFalse)
      returnedValue = falseOOP;
    FALL(doRetStackTop);
    
    BC(123, retNil) 
      returnedValue = nilOOP;
    FALL(doRetStackTop);


    BC(124, explicitRet)		/* return stack top from method */
      returnedValue = stackTop();
      exportRegs();
      unwindMethod();
      importRegs();
      setStackTop(returnedValue);
    FLUSH;

    BC(125, retStackTop)
      returnedValue = stackTop();
    BRANCH_LABEL(doRetStackTop)
      exportRegs();
      unwindContext();
      importRegs();
      setStackTop(returnedValue);
    FLUSH;

    /* 126, not in the blue book, is used for big literals access as follows:
       01111110 yyxxxxxx xxxxxxxx
       on method literal xxx do
	    yy = 00 -> push constant
	    yy = 01 -> push variable
	    yy = 10 -> store variable
	    yy = 11 -> pop/store variable
    */

    BC(126, bigLiteralOp)
      REGISTER(3, unsigned int num);
      num = BYTE1;
      num <<= 8;
      num |= BYTE2;

      switch(num >> 14) {
	case 0:
          LOAD3;
	  pushOOP(methodLiteral(num & 16383));
	  break;
	case 1:
          LOAD3;
	  pushOOP(methodVariable(num & 16383));
	  break;
	case 2:
          LOAD3;
	  storeMethodVariable(num & 16383, popOOP());
	  break;
	case 3: default:
          LOAD3;
	  storeMethodVariable(num & 16383, stackTop());
	  break;
      }
    END;

    BC(127, unused127)
      LOAD1;
      errorf("INVALID BYTECODE 127 USED!!!");
      exportRegs();
      showBacktrace();
      importRegs();
    END;

    BC(128, pushIndexedVal)
      REGISTER(3, Byte		ival2);
      ival2 = BYTE1;
      switch (ival2 >> 6) {
      case 0:
        LOAD2;
	pushOOP(receiverVariable(ival2 & 63));
	break;
      case 1:
        LOAD2;
	pushOOP(methodTemporary(ival2 & 63));
	break;
      case 2:
        LOAD2;
	pushOOP(methodLiteral(ival2 & 63));
	break;
      case 3: default:
        LOAD2;
	pushOOP(methodVariable(ival2 & 63));
	break;
      }
    END;

    BC(129, storeStackTop)
      REGISTER(3, Byte		ival2);
      ival2 = BYTE1;
      switch (ival2 >> 6) {
      case 0:
        LOAD2;
	storeReceiverVariable(ival2 & 63, stackTop());
	break;
      case 1:
        LOAD2;
	storeMethodTemporary(ival2 & 63, stackTop());
	break;
      case 2:
        LOAD2;
	ILLEGAL("Attempt to store into a method constant");
	break;
      case 3: default:
        LOAD2;
	storeMethodVariable(ival2 & 63, stackTop());
      }
    END;

    BC(130, popAndStoreTop)
      REGISTER(3, Byte		ival2);
      ival2 = BYTE1;
      switch (ival2 >> 6) {
      case 0:
        LOAD2;
	storeReceiverVariable(ival2 & 63, popOOP());
	break;
      case 1:
        LOAD2;
	storeMethodTemporary(ival2 & 63, popOOP());
	break;
      case 2:
        LOAD2;
	ILLEGAL("Attempt to store into a method constant");
	break;
      case 3: default:
        LOAD2;
	storeMethodVariable(ival2 & 63, popOOP());
      }
    END;

    BC(131, sendShort)	/* send selector y (xxxyyyyy), x args */
      REGISTER(3, Byte		ival2);
      ival2 = BYTE1;
      INCR2;
      exportRegs();
      sendMessage(methodLiteral(ival2 & 31), ival2 >> 5, false);
      importRegs();
    FLUSH;

    BC(132, sendLong)		/* send selector y (yysxxxxx,yyyyyyyy) x args */
      REGISTER(3, Byte	ival2);			/*     \___to super if s=1    */
      register int	ival3;
      ival2 = BYTE1;		/* the number of args */
      ival3 = BYTE2;		/* the selector */
      ival3 += (ival2 & 192) << 2;
      INCR3;
      exportRegs();
      if (ival2 & 32) {
	sendToSuper(methodLiteral(ival3), ival2 & 31, true);
      } else {
	sendMessage(methodLiteral(ival3), ival2 & 31, false);
      }
      importRegs();
    FLUSH;

    BC(133, supSendShort)	/* send super selector y (xxxyyyyy), x args*/
      REGISTER(3, Byte		ival2);
      ival2 = BYTE1;
      INCR2;
      exportRegs();
      sendToSuper(methodLiteral(ival2 & 31), ival2 >> 5, true);
      importRegs();
    FLUSH;

    BC(134, bigInstanceOp)
      REGISTER(3, unsigned int num);
      num = BYTE1;
      num <<= 8;
      num |= BYTE2;

      switch(num >> 14) {
	case 0: {
          register OOP	value;
          LOAD3;
          value = popOOP();
          storeInstanceVariable(stackTop(), num & 16383, value);
	  break;
	}
	case 1:
          LOAD3;
	  pushOOP(receiverVariable(num & 16383));
	  break;
	case 2:
          LOAD3;
	  storeReceiverVariable(num & 16383, popOOP());
	  break;
	case 3: default:
          LOAD3;
	  storeReceiverVariable(num & 16383, stackTop());
	  break;
      }
    END;

    BC(135, popStack)
      LOAD;
      popNOOPs(1);
    PREFETCH;

    BC(136, dupStack)
      register OOP tempOOP;
      LOAD;
      tempOOP = stackTop();
      pushOOP(tempOOP);
    PREFETCH;

    BC(137, pushThisContext) /* push active context */
      LOAD;
      emptyContextStack();
      pushOOP(thisContextOOP);
#ifdef PROFBLOCK
      ps.numThisContexts++;
#endif
    PREFETCH;

    BC(138, outerTempOp)
	unsigned int numScopes;
	unsigned int varIndex;
	OOP	contextOOP;
	BlockContext context;

	varIndex = BYTE1;
	numScopes = BYTE2;

	context = (BlockContext)oopToObj(thisContextOOP);
	do {
	  contextOOP = context->outerContext;
	  context = (BlockContext)oopToObj(contextOOP);
	} while (--numScopes);

#ifndef OPTIMIZE
	if (context != nil) {	/* we're on the right one */
#endif
	  switch(varIndex >> 6) {
	    case 0:
	      LOAD3;
	      ILLEGAL("Invalid bytecode 138/0");
	      break;
	    case 1:
	      LOAD3;
	      pushOOP(context->contextStack[varIndex & 63]);
	      break;
	    case 2:
	      LOAD3;
	      context->contextStack[varIndex & 63] = popOOP();
	      break;
	    case 3: default:
	      LOAD3;
	      context->contextStack[varIndex & 63] = stackTop();
	      break;
#ifndef OPTIMIZE
	  }
#endif
	}
    END;

    /* :-( Sigh ... have to sacrifice 139 for nop... )-: */
    BC(139, nop)
      LOAD;
    PREFETCH;
    
    BC(140, topSelf)
      LOAD;
      uncheckedSetTop(self);
    PREFETCH;
    
    BC(141, topOne)
      LOAD;
      uncheckedSetTop(fromInt(1));
    PREFETCH;

    BC(142, topIndexedVal)
      REGISTER(3, Byte		ival2);
      ival2 = BYTE1;
      switch (ival2 >> 6) {
      case 0:
	LOAD2;
	setStackTop(receiverVariable(ival2 & 63));
	break;
      case 1:
	LOAD2;
	setStackTop(methodTemporary(ival2 & 63));
	break;
      case 2:
	LOAD2;
	setStackTop(methodLiteral(ival2 & 63));
	break;
      case 3: default:
	LOAD2;
	setStackTop(methodVariable(ival2 & 63));
	break;
      }
    END;

    BC(143, endExecution)
      EXIT_IF(true);
    END;


#define jmpFail \
    unPop(1); \
    exportRegs(); \
    sendMessage(internString("mustBeBoolean"), 0, false); \
    importRegs();

#define jmpLongSigned(highDelta)		\
  REGISTER(3, Byte		ival2);		\
  ival2 = BYTE1;				\
  ip += (long)((highDelta) << 8) + 2 + ival2;	\

    /* Forward jump by 1+1 to 8+1 (+1 is the bytecode itself) */
    BC(144, shJmp1)
      LOAD2;
    END;
    
    BC(145, shJmp2)
      LOAD3;
    END;
    
    BC(146, shJmp3)
      ip += 4;
    FLUSH;
    
    BC(147, shJmp4)
      ip += 5;
    FLUSH;
    
    BC(148, shJmp5)
      ip += 6;
    FLUSH;
    
    BC(149, shJmp6)
      ip += 7;
    FLUSH;
    
    BC(150, shJmp7)
      ip += 8;
    FLUSH;
    
    BC(151, shJmp8)
      ip += 9;
    FLUSH;

    BC(152, shJmp1False)
      REGISTER(3, OOP	tempOOP);
      if ((tempOOP = popOOP()) == falseOOP) { /* jump forward if false */
        LOAD2;
      } else if (tempOOP != trueOOP) {
        LOAD2;
        jmpFail;
      } else {
        LOAD1;
      }
    END;
    
    BC(153, shJmp2False)
      REGISTER(3, OOP	tempOOP);
      if ((tempOOP = popOOP()) == falseOOP) { /* jump forward if false */
        LOAD3;
      } else if (tempOOP != trueOOP) {
        LOAD3;
        jmpFail;
      } else {
        LOAD1;
      }
    END;
    
    BC(154, shJmp3False)
      REGISTER(3, OOP	tempOOP);
      if ((tempOOP = popOOP()) == falseOOP) { /* jump forward if false */
        ip += 4;
      } else if (tempOOP != trueOOP) {
        ip += 4;
        jmpFail;
      } else {
        LOAD1;
        NEXT_BC;
      }
    FLUSH;
    
    BC(155, shJmp4False)
      REGISTER(3, OOP	tempOOP);
      if ((tempOOP = popOOP()) == falseOOP) { /* jump forward if false */
        ip += 5;
      } else if (tempOOP != trueOOP) {
        ip += 5;
        jmpFail;
      } else {
        LOAD1;
        NEXT_BC;
      }
    FLUSH;
    
    BC(156, shJmp5False)
      REGISTER(3, OOP	tempOOP);
      if ((tempOOP = popOOP()) == falseOOP) { /* jump forward if false */
        ip += 6;
      } else if (tempOOP != trueOOP) {
        ip += 6;
        jmpFail;
      } else {
        LOAD1;
        NEXT_BC;
      }
    FLUSH;
    
    BC(157, shJmp6False)
      REGISTER(3, OOP	tempOOP);
      if ((tempOOP = popOOP()) == falseOOP) { /* jump forward if false */
        ip += 7;
      } else if (tempOOP != trueOOP) {
        ip += 7;
        jmpFail;
      } else {
        LOAD1;
        NEXT_BC;
      }
    FLUSH;
    
    BC(158, shJmp7False)
      REGISTER(3, OOP	tempOOP);
      if ((tempOOP = popOOP()) == falseOOP) { /* jump forward if false */
        ip += 8;
      } else if (tempOOP != trueOOP) {
        ip += 8;
        jmpFail;
      } else {
        LOAD1;
        NEXT_BC;
      }
    FLUSH;
    
    BC(159, shJmp8False)
      REGISTER(3, OOP	tempOOP);
      if ((tempOOP = popOOP()) == falseOOP) { /* jump forward if false */
        ip += 9;
      } else if (tempOOP != trueOOP) {
        ip += 9;
        jmpFail;
      } else {
        LOAD1;
        NEXT_BC;
      }
    FLUSH;


    BC(160, longJmpNeg4)
      jmpLongSigned(-4);
    FLUSH;
    
    BC(161, longJmpNeg3)
      jmpLongSigned(-3);
    FLUSH;
    
    BC(162, longJmpNeg2)
      jmpLongSigned(-2);
    FLUSH;
    
    BC(163, longJmpNeg1)
      jmpLongSigned(-1);
    FLUSH;
    
    BC(164, longJmp0)	
      jmpLongSigned(0); 
    FLUSH;
    
    BC(165, longJmp1)	
      jmpLongSigned(1); 
    FLUSH;
    
    BC(166, longJmp2)	
      jmpLongSigned(2); 
    FLUSH;
    
    BC(167, longJmp3)	
      jmpLongSigned(3); 
    FLUSH;


    BC(168, popJmpTrue0)
      REGISTER(3, OOP	tempOOP);
      if ((tempOOP = popOOP()) == trueOOP) {
        jmpLongSigned(0);
      } else if (tempOOP != falseOOP) {
        jmpLongSigned(0);
        jmpFail;
      } else {
        LOAD2;
        NEXT_BC;
      }
    FLUSH;
    
    BC(169, popJmpTrue1)
      REGISTER(3, OOP	tempOOP);
      if ((tempOOP = popOOP()) == trueOOP) {
        jmpLongSigned(1);
      } else if (tempOOP != falseOOP) {
        jmpLongSigned(1);
        jmpFail;
      } else {
        LOAD2;
        NEXT_BC;
      }
    FLUSH;
    
    BC(170, popJmpTrue2)
      REGISTER(3, OOP	tempOOP);
      if ((tempOOP = popOOP()) == trueOOP) {
        jmpLongSigned(2);
      } else if (tempOOP != falseOOP) {
        jmpLongSigned(2);
        jmpFail;
      } else {
        LOAD2;
        NEXT_BC;
      }
    FLUSH;
    
    BC(171, popJmpTrue3)
      REGISTER(3, OOP	tempOOP);
      if ((tempOOP = popOOP()) == trueOOP) {
        jmpLongSigned(3);
      } else if (tempOOP != falseOOP) {
        jmpLongSigned(3);
        jmpFail;
      } else {
        NEXT_BC;
      }
    FLUSH;


    BC(172, popJmpFalse0)
      REGISTER(3, OOP	tempOOP);
      if ((tempOOP = popOOP()) == falseOOP) {
        jmpLongSigned(0);
      } else if (tempOOP != trueOOP) {
        jmpLongSigned(0);
        jmpFail;
      } else {
        LOAD2;
        NEXT_BC;
      }
    FLUSH;
    
    BC(173, popJmpFalse1)
      REGISTER(3, OOP	tempOOP);
      if ((tempOOP = popOOP()) == falseOOP) {
        jmpLongSigned(1);
      } else if (tempOOP != trueOOP) {
        jmpLongSigned(1);
        jmpFail;
      } else {
        LOAD2;
        NEXT_BC;
      }
    FLUSH;
    
    BC(174, popJmpFalse2)
      REGISTER(3, OOP	tempOOP);
      if ((tempOOP = popOOP()) == falseOOP) {
        jmpLongSigned(2);
      } else if (tempOOP != trueOOP) {
        jmpLongSigned(2);
        jmpFail;
      } else {
        LOAD2;
        NEXT_BC;
      }
    FLUSH;
    
    BC(175, popJmpFalse3)
      REGISTER(3, OOP	tempOOP);
      if ((tempOOP = popOOP()) == falseOOP) {
        jmpLongSigned(3);
      } else if (tempOOP != trueOOP) {
        jmpLongSigned(3);
        jmpFail;
      } else {
        LOAD2;
        NEXT_BC;
      }
    FLUSH;

#undef jmpFail
#undef jmpFalse
#undef jmpLongSigned

#ifndef OPEN_CODE_MATH

#define RAW_INT_OP(op, noOverflow)
#define RAW_FLOAT_OP(op)
#define INTERP_BASIC_OP(op, fop)
#define RAW_BOOL_OP(operator)
#define RAW_BOOL_FLOAT_OP(operator)
#define INTERP_BASIC_BOOL(operator)

#else

#define RAW_INT_OP(op, noOverflow)	\
{					\
  register long iarg1, iarg2;		\
  iarg1 = toInt(tempOOP);		\
  iarg2 = toInt(popOOP());		\
  op;					\
  if(noOverflow || !INT_OVERFLOW(iarg1)) { \
    setStackTopInt(iarg1);		\
    NEXT_BC;				\
  } else {				\
    unPop(1);				\
  }					\
}

#define RAW_FLOAT_OP(op)		\
{					\
  OOP	oop2,oopResult;			\
  double farg1, farg2;			\
  oop2 = stackTop();			\
  if (isClass(oop2, floatClass)) {	\
    farg1 = floatOOPValue(tempOOP);	\
    farg2 = floatOOPValue(oop2);	\
    popNOOPs(1);			\
    exportRegs();			\
    oopResult = floatNew(op);		\
    importRegs();			\
    setStackTop(oopResult);		\
    NEXT_BC;				\
  }					\
}


#define INTERP_BASIC_OP(op, fop) {		\
  REGISTER(3, OOP	tempOOP);		\
  tempOOP = stackAt(1);				\
  if (areInts(tempOOP, stackTop())) {		\
    RAW_INT_OP(op, false);			\
  } else if (isClass(tempOOP, floatClass)) {	\
    RAW_FLOAT_OP(fop);				\
  }						\
}

#define RAW_BOOL_OP(operator)						\
  DO_JUMP_LOOKAHEAD(	((long)tempOOP) operator ((long)popOOP()) );

#define RAW_BOOL_FLOAT_OP(operator)	\
{					\
  OOP	oop2;				\
  oop2 = stackTop();			\
  if (isClass(oop2, floatClass)) {	\
    double farg1, farg2;		\
    farg1 = floatOOPValue(tempOOP);	\
    farg2 = floatOOPValue(oop2);	\
    popNOOPs(1);			\
    DO_JUMP_LOOKAHEAD(farg1 operator farg2); \
  }					\
}

#define INTERP_BASIC_BOOL(operator) {	\
  REGISTER(3, OOP	tempOOP);	\
  tempOOP = stackAt(1);			\
  if (areInts(tempOOP, stackTop())) {	\
    RAW_BOOL_OP(operator);		\
  } else if (isClass(tempOOP, floatClass)) {	\
    RAW_BOOL_FLOAT_OP(operator);	\
  }					\
}
#endif

#define MAYBE_PRIMITIVE(cond, symbol, num, numArgs)			\
  if ((cond)) {								\
    exportRegs();							\
    if (!primitiveTable[(num)].primFunc ((num), (numArgs), nil)) {	\
      importRegs();							\
      NEXT_BC;								\
    }									\
  } else {								\
    exportRegs();							\
  }									\
  sendMessage((symbol), (numArgs), false);				\
  importRegs();


    BC(176, sendPlus)
      LOAD1;
      INTERP_BASIC_OP(iarg1 += iarg2, farg1 + farg2);
      exportRegs();
      sendMessage(plusSymbol, 1, false);
      importRegs();
    FLUSH;

    BC(177, sendMinus)
      LOAD1;
      INTERP_BASIC_OP(iarg1 -= iarg2, farg1 - farg2);
      exportRegs();
      sendMessage(minusSymbol, 1, false);
      importRegs();
    FLUSH;

    BC(178, sendLess)
      LOAD1;
      INTERP_BASIC_BOOL(<);
      exportRegs();
      sendMessage(lessThanSymbol, 1, false);
      importRegs();
    FLUSH;

    BC(179, sendGreater)
      LOAD1;
      INTERP_BASIC_BOOL(>);
      exportRegs();
      sendMessage(greaterThanSymbol, 1, false);
      importRegs();
    FLUSH;

    BC(180, sendLessEq)
      LOAD1;
      INTERP_BASIC_BOOL(<=);
      exportRegs();
      sendMessage(lessEqualSymbol, 1, false);
      importRegs();
    FLUSH;

    BC(181, sendGreaterEq)
      LOAD1;
      INTERP_BASIC_BOOL(>=);
      exportRegs();
      sendMessage(greaterEqualSymbol, 1, false);
      importRegs();
    FLUSH;

    BC(182, sendEqual)
      LOAD1;
      INTERP_BASIC_BOOL(==);
      exportRegs();
      sendMessage(equalSymbol, 1, false);
      importRegs();
    FLUSH;

    BC(183, sendNotEqual)
      LOAD1;
      INTERP_BASIC_BOOL(!=);
      exportRegs();
      sendMessage(notEqualSymbol, 1, false);
      importRegs();
    FLUSH;

    BC(184, sendTimes)
      LOAD1;
      INTERP_BASIC_OP(iarg1 = mulWithCheck(iarg1, iarg2), farg1 * farg2);
      exportRegs();
      sendMessage(timesSymbol, 1, false);
      importRegs();
    FLUSH;

    BC(185, sendDivide)
      LOAD1;
      MAYBE_PRIMITIVE(
	isInt(stackAt(1)) && isInt(stackTop()),
	divideSymbol, 10, 1);
    FLUSH;

    BC(186, sendRemainder)
      LOAD1;
      MAYBE_PRIMITIVE(
	isInt(stackAt(1)) && isInt(stackTop()),
	remainderSymbol, 11, 1);
    FLUSH;

    BC(187, sendAtSign)
      INCR1;
      exportRegs();
      sendMessage(atSignSymbol, 1, false);
      importRegs();
    FLUSH;

    BC(188, sendBitShift)
      REGISTER(3, OOP	tempOOP);
      LOAD1;
      tempOOP = stackAt(1);
      if (isInt(tempOOP)) {
	register long	iarg1, iarg2;
	if (isInt(stackTop())) {
	  iarg1 = toInt(tempOOP);
	  iarg2 = toInt(popOOP());
	  if (iarg2 < 0) {
	    setStackTopInt(iarg1 >> -iarg2);
	    NEXT_BC;
	  }
	  if (iarg2 < ST_INT_SIZE) {
	    register long result = iarg1 << iarg2;
	    if ((result >> iarg2) == iarg1) {
	      setStackTopInt(result);
	      NEXT_BC;
	    }
	  }
	}
	unPop(1);
      }
      exportRegs();
      sendMessage(bitShiftColonSymbol, 1, false);
      importRegs();
    FLUSH;

    BC(189, sendIntDivide)
      LOAD1;
      MAYBE_PRIMITIVE(
	isInt(stackAt(1)) && isInt(stackTop()),
	integerDivideSymbol, 12, 1);
    FLUSH;

    BC(190, sendBitAnd)
      REGISTER(3, OOP	tempOOP);
      LOAD1;
      tempOOP = stackAt(1);
      if (areInts(tempOOP, stackTop())) {
	RAW_INT_OP(iarg1 &= iarg2, true);
      }
      exportRegs();
      sendMessage(bitAndColonSymbol, 1, false);
      importRegs();
    FLUSH;

    BC(191, sendBitOr)
      REGISTER(3, OOP	tempOOP);
      LOAD1;
      tempOOP = stackAt(1);
      if (areInts(tempOOP, stackTop())) {
	RAW_INT_OP(iarg1 |= iarg2, true);
      }
      exportRegs();
      sendMessage(bitOrColonSymbol, 1, false);
      importRegs();
    FLUSH;

    BC(192, sendAt)
      REGISTER(3, OOP	tempOOP);
      LOAD1;
      tempOOP = stackAt(1);
      if(isOOP(tempOOP) && atCacheClass == (tempOOP = oopClass(tempOOP))) {
        exportRegs();
	if (!primitiveTable[atCachePrim].primFunc (atCachePrim, 1, nil)) {
	  importRegs();
	  NEXT_BC;
	}
      } else {
        exportRegs();
      }

      /* Not the same class that is in the cache, or the primitive failed --
       * send the message, and modify the cache if the send is resolved to
       * a primitive. */
      lastPrimitive = 0;
      sendMessage(atColonSymbol, 1, false);
      importRegs();
      if (lastPrimitive) {
	atCachePrim = lastPrimitive;
	atCacheClass = tempOOP;
	NEXT_BC;
      }
    FLUSH;

    BC(193, sendAtPut)
      REGISTER(3, OOP	tempOOP);
      LOAD1;
      tempOOP = stackAt(2);
      if(isOOP(tempOOP) && atPutCacheClass == (tempOOP = oopClass(tempOOP))) {
        exportRegs();
	if (!primitiveTable[atPutCachePrim].primFunc (atPutCachePrim, 2, nil)) {
	  importRegs();
	  NEXT_BC;
	}
      } else {
        exportRegs();
      }

      /* Not the same class that is in the cache, or the primitive failed --
       * send the message, and modify the cache if the send is resolved to
       * a primitive. */
      lastPrimitive = 0;
      sendMessage(atColonPutColonSymbol, 2, false);
      importRegs();
      if (lastPrimitive) {
	atPutCachePrim = lastPrimitive;
	atPutCacheClass = tempOOP;
	NEXT_BC;
      }
    FLUSH;

    BC(194, sendSize)
      REGISTER(3, OOP	tempOOP);
      LOAD1;
      tempOOP = stackTop();
      if(isOOP(tempOOP) && sizeCacheClass == (tempOOP = oopClass(tempOOP))) {
        exportRegs();
	if (!primitiveTable[sizeCachePrim].primFunc (sizeCachePrim, 0, nil)) {
	  importRegs();
	  NEXT_BC;
	}
      } else {
        exportRegs();
      }

      /* Not the same class that is in the cache, or the primitive failed --
       * send the message, and modify the cache if the send is resolved to
       * a primitive. */
      lastPrimitive = 0;
      sendMessage(sizeSymbol, 0, false);
      importRegs();
      if (lastPrimitive) {
	sizeCachePrim = lastPrimitive;
	sizeCacheClass = tempOOP;
	NEXT_BC;
      }
    FLUSH;

    BC(195, sendNext)
      INCR1;
      exportRegs();
      sendMessage(nextSymbol, 0, false);
      importRegs();
    FLUSH;

    BC(196, sendNextPut)
      INCR1;
      exportRegs();
      sendMessage(nextPutColonSymbol, 1, false);
      importRegs();
    FLUSH;

    BC(197, sendAtEnd)
      INCR1;
      exportRegs();
      sendMessage(atEndSymbol, 0, false);
      importRegs();
    FLUSH;

    BC(198, sendSameObject)
      REGISTER(3, OOP	tempOOP);
      register OOP oop1;
      LOAD1;
      tempOOP = popOOP();
      oop1 = stackTop();
      DO_JUMP_LOOKAHEAD(oop1 == tempOOP);
    END;

    BC(199, sendClass)
      REGISTER(3, OOP	tempOOP);
      LOAD1;
      tempOOP = stackTop();
      setStackTop(isInt(tempOOP) ? smallIntegerClass : oopClass(tempOOP));
    END;

    BC(200, sendBlockCopy)
      REGISTER(3, OOP	tempOOP);
      LOAD1;
      MAYBE_PRIMITIVE(
	isOOP(tempOOP = stackAt(1)) && oopClass(tempOOP) == blockClosureClass,
	blockCopyColonSymbol, 80, 1);
    FLUSH;

    BC(201, sendValue)
      REGISTER(3, OOP	tempOOP);
      INCR1;
      exportRegs();
      if (isInt(tempOOP = stackTop())
	  || oopClass(tempOOP) != blockClosureClass
	  || sendBlockValue(0)) {
	sendMessage(valueSymbol, 0, false);
      }
      importRegs();
    FLUSH;

    BC(202, sendValueColon)
      REGISTER(3, OOP	tempOOP);
      INCR1;
      exportRegs();
      if (isInt(tempOOP = stackAt(1))
	  || oopClass(tempOOP) != blockClosureClass
	  || sendBlockValue(1)) {
	sendMessage(valueColonSymbol, 1, false);
      }
      importRegs();
    FLUSH;

    BC(203, sendDoColon)
      INCR1;
      exportRegs();
      sendMessage(doColonSymbol, 1, false);
      importRegs();
    FLUSH;

    BC(204, sendNew)
      INCR1;
      exportRegs();
      sendMessage(newSymbol, 0, false);
      importRegs();
    FLUSH;

    BC(205, sendNewColon)
      INCR1;
      exportRegs();
      sendMessage(newColonSymbol, 1, false);
      importRegs();
    FLUSH;

    BC(206, sendIsNil)
      LOAD1;
      DO_JUMP_LOOKAHEAD(stackTop() == nilOOP);
    END;

    BC(207, sendNotNil)
      LOAD1;
      DO_JUMP_LOOKAHEAD(stackTop() != nilOOP);
    END;


    BC(208, sendNoArg0)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(0), 0, false);
      importRegs()
    FLUSH;

    BC(209, sendNoArg1)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(1), 0, false);
      importRegs();
    FLUSH;

    BC(210, sendNoArg2)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(2), 0, false);
      importRegs();
    FLUSH;
    
    BC(211, sendNoArg3)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(3), 0, false);
      importRegs();
    FLUSH;
    
    BC(212, sendNoArg4)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(4), 0, false);
      importRegs();
    FLUSH;
    
    BC(213, sendNoArg5)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(5), 0, false);
      importRegs();
    FLUSH;
    
    BC(214, sendNoArg6)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(6), 0, false);
      importRegs();
    FLUSH;
    
    BC(215, sendNoArg7)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(7), 0, false);
      importRegs();
    FLUSH;
    
    BC(216, sendNoArg8)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(8), 0, false);
      importRegs();
    FLUSH;
    
    BC(217, sendNoArg9)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(9), 0, false);
      importRegs();
    FLUSH;
    
    BC(218, sendNoArg10)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(10), 0, false);
      importRegs();
    FLUSH;
    
    BC(219, sendNoArg11)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(11), 0, false);
      importRegs();
    FLUSH;

    BC(220, sendNoArg12)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(12), 0, false);
      importRegs();
    FLUSH;

    BC(221, sendNoArg13)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(13), 0, false);
      importRegs();
    FLUSH;

    BC(222, sendNoArg14)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(14), 0, false);
      importRegs();
    FLUSH;

    BC(223, sendNoArg15)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(15), 0, false);
      importRegs();
    FLUSH;


    BC(224, sendOneArg0)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(0), 1, false);
      importRegs();
    FLUSH;

    BC(225, sendOneArg1)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(1), 1, false);
      importRegs();
    FLUSH;

    BC(226, sendOneArg2)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(2), 1, false);
      importRegs();
    FLUSH;

    BC(227, sendOneArg3)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(3), 1, false);
      importRegs();
    FLUSH;

    BC(228, sendOneArg4)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(4), 1, false);
      importRegs();
    FLUSH;

    BC(229, sendOneArg5)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(5), 1, false);
      importRegs();
    FLUSH;

    BC(230, sendOneArg6)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(6), 1, false);
      importRegs();
    FLUSH;

    BC(231, sendOneArg7)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(7), 1, false);
      importRegs();
    FLUSH;

    BC(232, sendOneArg8)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(8), 1, false);
      importRegs();
    FLUSH;

    BC(233, sendOneArg9)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(9), 1, false);
      importRegs();
    FLUSH;

    BC(234, sendOneArg10)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(10), 1, false);
      importRegs();
    FLUSH;
    
    BC(235, sendOneArg11)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(11), 1, false);
      importRegs();
    FLUSH;
    
    BC(236, sendOneArg12)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(12), 1, false);
      importRegs();
    FLUSH;
    
    BC(237, sendOneArg13)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(13), 1, false);
      importRegs();
    FLUSH;
    
    BC(238, sendOneArg14)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(14), 1, false);
      importRegs();
    FLUSH;
    
    BC(239, sendOneArg15)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(15), 1, false);
      importRegs();
    FLUSH;


    BC(240, sendTwoArg0)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(0), 2, false);
      importRegs();
    FLUSH;

    BC(241, sendTwoArg1)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(1), 2, false);
      importRegs();
    FLUSH;

    BC(242, sendTwoArg2)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(2), 2, false);
      importRegs();
    FLUSH;

    BC(243, sendTwoArg3)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(3), 2, false);
      importRegs();
    FLUSH;

    BC(244, sendTwoArg4)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(4), 2, false);
      importRegs();
    FLUSH;

    BC(245, sendTwoArg5)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(5), 2, false);
      importRegs();
    FLUSH;

    BC(246, sendTwoArg6)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(6), 2, false);
      importRegs();
    FLUSH;

    BC(247, sendTwoArg7)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(7), 2, false);
      importRegs();
    FLUSH;

    BC(248, sendTwoArg8)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(8), 2, false);
      importRegs();
    FLUSH;

    BC(249, sendTwoArg9)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(9), 2, false);
      importRegs();
    FLUSH;

    BC(250, sendTwoArg10)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(10), 2, false);
      importRegs();
    FLUSH;
    
    BC(251, sendTwoArg11)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(11), 2, false);
      importRegs();
    FLUSH;
    
    BC(252, sendTwoArg12)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(12), 2, false);
      importRegs();
    FLUSH;
    
    BC(253, sendTwoArg13)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(13), 2, false);
      importRegs();
    FLUSH;
    
    BC(254, sendTwoArg14)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(14), 2, false);
      importRegs();
    FLUSH;
    
    BC(255, sendTwoArg15)
      INCR1;
      exportRegs();
      sendMessage(methodLiteral(15), 2, false);
      importRegs();
    FLUSH;

#ifdef JUMP_LOOKAHEAD
    BRANCH_LABEL(lookaheadFailedTrue)
      uncheckedPushOOP(trueOOP);
      NEXT_BC;

    BRANCH_LABEL(lookaheadFailedFalse)
      uncheckedPushOOP(falseOOP);
      NEXT_BC;

    BRANCH_LABEL(lookaheadLongJumpNotTaken)
      LOAD2;
      localCounter++;
      NEXT_BC;
#endif

    } /* BYTECODES */
  } /* INTERPRETER */
} /* interpret() */


/* Always use outer ip/sp outside interpret */
#ifdef LOCAL_REGS
#  define ip		outerIP
#  define sp		outerSP
#  if REG_AVAILABILITY >= 2
#    undef nilOOP
#    undef trueOOP
#    undef falseOOP
#  endif
#  if REG_AVAILABILITY == 0
#    undef localCounter
#  endif
#endif
