/******************************** -*- C -*- ****************************
 *
 *	Binary image save/restore.
 *
 *	$Revision: 1.95.1$
 *	$Date: 2000/12/27 10:45:49$
 *	$Author: pb$
 *
 ***********************************************************************/

/***********************************************************************
 *
 * Copyright 1988-92, 1994-95, 1999, 2000 Free Software Foundation, Inc.
 * Written by Steve Byrne.
 *
 * This file is part of GNU Smalltalk.
 *
 * GNU Smalltalk is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2, or (at your option) any later 
 * version.
 * 
 * GNU Smalltalk is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * GNU Smalltalk; see the file COPYING.  If not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
 *
 ***********************************************************************/


#include "gst.h"
#include "save.h"
#include "comp.h"
#include "interp.h"
#include "dict.h"
#include "sym.h"
#include "oop.h"		/* defines oopTable */
#include "sysdep.h"
#include <stdio.h>

#if STDC_HEADERS
#include <string.h> /* for memcpy */
#include <stdlib.h>
#endif /* STDC_HEADERS */

#define fromBeginning		0 /* symbolic name for file offset modifier */

#define MASK_ENDIANNESS_FLAG 1
#define MASK_LONG_SIZE_FLAG 2

#ifdef WORDS_BIGENDIAN
# define LOCAL_ENDIANNESS_FLAG MASK_ENDIANNESS_FLAG
#else
# define LOCAL_ENDIANNESS_FLAG 0
#endif

#if SIZEOF_LONG == 4
# define LOCAL_LONG_SIZE_FLAG 0
# define byteInvert(x) \
        ((unsigned long int)((((unsigned long int)(x) & 0x000000ffU) << 24) | \
                             (((unsigned long int)(x) & 0x0000ff00U) <<  8) | \
                             (((unsigned long int)(x) & 0x00ff0000U) >>  8) | \
                             (((unsigned long int)(x) & 0xff000000U) >> 24)))
#else
# define LOCAL_LONG_SIZE_FLAG MASK_LONG_SIZE_FLAG
# define byteInvert(x) \
        ((unsigned long int)((((unsigned long int)(x) & 0x00000000000000ffU) << 56) | \
                             (((unsigned long int)(x) & 0x000000000000ff00U) << 40) | \
                             (((unsigned long int)(x) & 0x0000000000ff0000U) << 24) | \
                             (((unsigned long int)(x) & 0x00000000ff000000U) <<  8) | \
                             (((unsigned long int)(x) & 0x000000ff00000000U) >>  8) | \
                             (((unsigned long int)(x) & 0x0000ff0000000000U) >> 24) | \
                             (((unsigned long int)(x) & 0x00ff000000000000U) >> 40) | \
                             (((unsigned long int)(x) & 0xff00000000000000U) >> 56)))
#endif

#define flagChanged(flags, whichFlag)	\
  ((flags ^ LOCAL_##whichFlag) & MASK_##whichFlag)

#define FLAGS_WRITTEN		\
  (LOCAL_ENDIANNESS_FLAG | LOCAL_LONG_SIZE_FLAG)

#define VERSION_REQUIRED	\
  ((ST_MAJOR_VERSION << 16) + (ST_MINOR_VERSION << 8) + ST_EDIT_VERSION)

/*
 * The binary image file has the following format:
 *
 *	header
 *	complete oop table
 *	global oop variable data
 *	object data
 */


typedef struct SaveFileHeaderStruct {
  unsigned char signature[7];	/* 7+1=8 should be enough to align version! */
  unsigned char flags;		/* flags for endianness and sizeof(long) */
  unsigned long version;	/* the Smalltalk version that made this dump */
  unsigned long	objectDataSize;	/* size of object data section in bytes */
  unsigned long	oopTableSize;	/* size of the oop table at dump */
  unsigned long	memSpaceSize;	/* size of the semi spaces at dump time */
} SaveFileHeader;



static void		skipOverHeader(), saveObject(), 
			fixupByteOrder(), restoreInstanceVars(),
			loadOOPTable(), loadNormalOOPs(),
			skipToHeader(),
			saveFileVersion(), loadFileVersion(),
			fixupInstanceVars(), fixupOOPInstanceVars();

static struct OOPStruct	*makeOOPTableToBeSaved();

static int		saveAllObjects(), my_fread();

static mst_Boolean	loadSnapshot();

static mst_Boolean	wrongEndianness;

/* This variable contains the OOP slot index of the highest non-free OOP,
 * excluding the built-in ones (i.e., it will always be < oopTableSize).
 * This is used for optimizing the size of the saved image, and minimizing
 * the load time when restoring the system. */
static int	maxUsedOOPSlot = 0;

/* convert to a relative offset from start of OOP table.  The offset is 0 mod
 * pointer-size, so it still looks like a pointer to the isInt test.  */
#define oopRelative(obj) \
  ( (OOP)((long)(obj) - (long)oopTable) )

/* convert from relative offset to actual oop table address */
#define oopAbsolute(obj) \
  ( (OOP)((long)(obj) + (long)oopTable) )



mst_Boolean
saveToFile(fileName)
     char	*fileName;
{
  FILE			*imageFile;
  unsigned long		objectDataSize;
  mst_Boolean		oldGCState;
  struct OOPStruct	*myOOPTable;

  imageFile = openFile(fileName, "w");
  if (imageFile == NULL) {
    return (false);
  }

  gcFlip();			/* make sure that the world is compact */

  oldGCState = gcOff();

  skipOverHeader(imageFile);

  myOOPTable = makeOOPTableToBeSaved();

  /* save up to the max oop slot in use */
  fwrite(myOOPTable, sizeof(struct OOPStruct), maxUsedOOPSlot + 1, imageFile);
  free(myOOPTable);

#ifdef SNAPSHOT_TRACE
printf("After saving oopt table: %d\n", ftell(imageFile));
#endif /* SNAPSHOT_TRACE */

  /* Remember -- first non-fixed objects, then fixed ones */
  objectDataSize = saveAllObjects(imageFile, false);

#ifdef SNAPSHOT_TRACE
printf("After saving non-fixed objects: %d\n", ftell(imageFile));
#endif /* SNAPSHOT_TRACE */

  saveAllObjects(imageFile, true);

#ifdef SNAPSHOT_TRACE
printf("After saving all objects: %d\n", ftell(imageFile));
#endif /* SNAPSHOT_TRACE */

  skipToHeader(imageFile);
  saveFileVersion(imageFile, objectDataSize);

  fclose(imageFile);

  setGCState(oldGCState);
  return (true);
} 


void
skipOverHeader(imageFile)
     FILE	*imageFile;
{
  fseek(imageFile, sizeof(SaveFileHeader), fromBeginning);
}

/*
 *	static void makeOOPTableToBeSaved(imageFile)
 *
 * Description
 *
 *	Prepares the OOP table to be written to the image file.  We
 *	store the object sizes instead of the pointers, since addresses
 *	will be recalculated upon load.  We also tag byte objects with
 *	F_BYTE, which aids in loading images with the wrong endianness.
 *
 * Outputs
 *
 *	A copy of the OOP table, but with object sizes instead of the addresses.
 *
 */
struct OOPStruct *
makeOOPTableToBeSaved()
{
  OOP			oop;
  struct OOPStruct	*myOOPTable;
  int			i;

  maxUsedOOPSlot = 0;

  for (oop = oopTable; oop < &oopTable[oopTableSize]; oop++) {
    if (oopValid(oop)) {
      maxUsedOOPSlot = oopIndex(oop);
    }
  }

#ifdef SNAPSHOT_TRACE
printf("there are %d free oops out of %d oops, leaving %d\n",
       numFreeOOPs, oopTableSize, oopTableSize - numFreeOOPs);
printf("max used is %d\n", maxUsedOOPSlot);
#endif /* SNAPSHOT_TRACE */

  myOOPTable = malloc(sizeof(struct OOPStruct) * (maxUsedOOPSlot + 1));

  for (i = 0, oop = oopTable; i <= maxUsedOOPSlot; oop++, i++) {
    myOOPTable[i].flags = oop->flags;
    if (oopValid(oop)) {
      if ((oopInstanceSpec(oop) & ISP_INDEXEDVARS) == ISP_ISINDEXABLE) {
	myOOPTable[i].flags |= F_BYTE;
      }

      myOOPTable[i].object = (mst_Object) (oop->object->objSize);
    }  
  }
  return (myOOPTable);
}


int
saveAllObjects(imageFile, fixed)
     FILE	*imageFile;
     mst_Boolean fixed;
{
  long		objectStart, objectEnd;
  OOP		oop;

  objectStart = ftell(imageFile);

  for (oop = oopTable; oop < &oopTable[oopTableSize]; oop++) {
    if (!oopValid(oop)) {
      continue;
    }

    if ( !(oop->flags & F_FIXED) ^ fixed) {
      saveObject(imageFile, oop);
    }
  }

  objectEnd = ftell(imageFile);

  return (objectEnd - objectStart);
}

void
saveObject(imageFile, oop)
     FILE	*imageFile;
     OOP	oop;
{
  mst_Object	object;
  int		numFixed;

  object = oopToObj(oop);
  numFixed = numOOPs(object);

  fixupObject(object, numFixed);
  fwrite(object, sizeof(OOP), object->objSize, imageFile);
  restoreObject(object, numFixed);
}



void
fixupObject(obj, numFixed)
     mst_Object	obj;
     int	numFixed;
{
  OOP		instOOP, classOOP, *i;

  classOOP = obj->objClass;
  obj->objClass = oopRelative(classOOP);

  for (i = obj->data; numFixed; i++, numFixed--) {
    instOOP = *i;
    if (isOOP(instOOP)) {
      *i = oopRelative(instOOP);
    }
  }
}

void
restoreObject(object, numFixed)
     mst_Object	object;
     int	numFixed;
{
  object->objClass = oopAbsolute(object->objClass);
  restoreInstanceVars(object, numFixed);
}

void
restoreInstanceVars(obj, numFixed)
     mst_Object	obj;
     int	numFixed;
{
  OOP		instOOP, *i;

  i = obj->data;
  while(numFixed--) {
    instOOP = *i;
    if (isOOP(instOOP)) {
      *i = oopAbsolute(instOOP);
    }
    i++;
  }
}


void
skipToHeader(imageFile)
     FILE	*imageFile;
{
  rewind(imageFile);
}

void
saveFileVersion(imageFile, objectDataSize)
     FILE	*imageFile;
     unsigned long objectDataSize;
{
  SaveFileHeader header;

  memcpy(header.signature, "GSTIm\0\0", 7);
  header.flags = FLAGS_WRITTEN;
  header.version = VERSION_REQUIRED;
  header.objectDataSize = objectDataSize;
  header.oopTableSize = maxUsedOOPSlot + 1; /* n slots, numbered 0..n-1 */
  header.memSpaceSize = max(memSpace.totalSize, objectDataSize * 2);

  fwrite(&header, sizeof(SaveFileHeader), 1, imageFile);
}



/***********************************************************************
 *
 *	Binary loading routines.
 *
 ***********************************************************************/


mst_Boolean
loadFromFile(fileName)
     char	*fileName;
{
  mst_Boolean	loaded;
  FILE		*imageFile;
  mst_Boolean	oldGCState;

  oldGCState = gcOff();
  imageFile = openFile(fileName, "r");

  loaded = (imageFile != NULL)
     && loadSnapshot(imageFile);

  fclose(imageFile);
  setGCState(oldGCState);
  return (loaded);
}

mst_Boolean
loadSnapshot(imageFile)
     FILE	*imageFile;
{
  SaveFileHeader header;

  loadFileVersion(imageFile, &header);
  if (strcmp(header.signature, "GSTIm\0\0")) return (false);

  if (wrongEndianness = flagChanged(header.flags, ENDIANNESS_FLAG)) {
    header.objectDataSize = byteInvert(header.objectDataSize);
    header.oopTableSize = byteInvert(header.oopTableSize);
    header.memSpaceSize = byteInvert(header.memSpaceSize);
    header.version = byteInvert(header.version);
  }

  /* different sizeof(long) not supported */
  if (flagChanged(header.flags, LONG_SIZE_FLAG)) return (false);

  /* check for version mismatch; if so this image file is invalid */
  if (header.version != VERSION_REQUIRED) return (false);
  
  if (header.memSpaceSize > memSpace.totalSize) {
    growMemoryTo(header.memSpaceSize);
  }

  initOOPTable(max(header.oopTableSize * 2, INITIAL_OOP_TABLE_SIZE));
  loadOOPTable(imageFile, header.oopTableSize);

#ifdef SNAPSHOT_TRACE
  printf("After loading oopt table: %d\n", ftell(imageFile));
#endif /* SNAPSHOT_TRACE */

  loadNormalOOPs(imageFile, header.objectDataSize);

#ifdef SNAPSHOT_TRACE
  printf("After loading objects: %d\n", ftell(imageFile));
#endif /* SNAPSHOT_TRACE */

  fixupInstanceVars();
  refreshOOPFreeList();
  return initDictionaryOnImageLoad();
}

void
loadFileVersion(imageFile, headerp)
     FILE	*imageFile;
     SaveFileHeader *headerp;
{
  fread(headerp, sizeof(SaveFileHeader), 1, imageFile);
}

void
loadOOPTable(imageFile, oldSlotsUsed)
     FILE	*imageFile;
     long 	oldSlotsUsed;
{
  OOP		oop;

  /* load in the valid OOP slots from previous dump */
  my_fread(oopTable, sizeof(struct OOPStruct), oldSlotsUsed, imageFile);
  
  /* mark the remaining ones as available (not a for because of a probable
     bug in GNU C!) */
  for (oop = &oopTable[oldSlotsUsed]; oop < &oopTable[oopTableSize]; oop++) {
    oop->flags = F_FREE;
    oop++;
  }

  /* the fixup gets handled by load normal oops */
}


#ifdef old_code /* Wed Apr 26 21:15:38 1989 */ /* <<<== what dedication...3 days before my wedding -- SBB */
/**/mst_Boolean
/**/loadGlobalOOPs(imageFile)
/**/     FILE	*imageFile;
/**/{
/**/  OOP		**oopPtr, *oopVec;
/**/  long		numGlobalOOPs;
/**/
/**/  for (oopPtr = globalOOPs; *oopPtr; oopPtr++) {
/**/    my_fread(&oop, sizeof(OOP), 1, imageFile);
/**/    **oopPtr = oopAbsolute(oop);
/**/  }
/**/}
#endif /* old_code Wed Apr 26 21:15:38 1989 */

void
loadNormalOOPs(imageFile, objectDataSize)
     FILE	*imageFile;
     unsigned long objectDataSize;
{
  OOP		oop;
  mst_Object	object;
  OOP		*objPtr;

  /* First load the main heap.
   * Not everything must be byte-inverted in this part of the file,
   * so read everything through good old fread and fix later. */

  objPtr = (OOP *) curSpaceAddr();
  fread(objPtr, 1, objectDataSize, imageFile);

  /* Now walk the oop table.  Start fixing the byte order, and get
   * to the end of the image file by loading fixed objects. */

  numFreeOOPs = 0;
  for (oop = oopTable; oop < &oopTable[oopTableSize]; oop++) {
    if (!oopValid(oop)) {
      numFreeOOPs++;
      oop->flags = F_FREE;	/* just free */
      continue;
    }

    if (oop->flags & F_FIXED) {
      /* oops... this still is to be read... */
      long size;
      size = (long) oop->object;

      object = (mst_Object) xmalloc( size * SIZEOF_LONG);
      fread(object, SIZEOF_LONG, size, imageFile);
    } else {
      object = (mst_Object) objPtr;
    }

    oop->object = object;
    if (wrongEndianness) {
      fixupByteOrder(object, (oop->flags & F_BYTE)
	  ? sizeof(struct OOPStruct) / SIZEOF_LONG
	  : byteInvert(oop->object->objSize));
    }
    object->objClass = oopAbsolute(object->objClass);

    if (!(oop->flags & F_FIXED)) {
      objPtr += object->objSize;
    }
  }

  /* numOOPs requires access to the instance spec in the class objects. So
   * we start by fixing the endianness of NON-BYTE objects (including
   * classes!), for which we can do without numOOPs, then do another pass
   * here and fix the byte objects using the now correct class objects. */
  for (oop = oopTable; oop < &oopTable[oopTableSize]; oop++) {
    if (wrongEndianness && (oop->flags & F_BYTE)) {
      object = oopToObj(oop);
      fixupByteOrder(object->data, numOOPs(object));
    }
    
    /* Remove flags that are invalid after an image has been loaded. */
    oop->flags &= ~F_RUNTIME;
  }


  setSpaceInfo(objectDataSize);
}



void
fixupInstanceVars()
{
  OOP		oop;

  for (oop = oopTable; oop < &oopTable[oopTableSize]; oop++) {
    if (oopValid(oop)) {
      fixupOOPInstanceVars(oop);
    }
  }
}

void
fixupOOPInstanceVars(oop)
     OOP	oop;
{
  int		numFixed;
  mst_Object	object;

  object = oopToObj(oop);
  numFixed = numOOPs(object);
  restoreInstanceVars(object, numFixed);
}

void
fixupByteOrder(p, size)
     unsigned long int	*p;
     int		size;
{
  for (; size--; p++) {
    *p = byteInvert(*p);
  }
}

int
my_fread(p, size, n, file)
     unsigned long int	*p;
     int		size;
     int		n;
     FILE		*file;
{
  int result;

  result = fread(p, size, n, file);
  if (wrongEndianness) {
    fixupByteOrder(p, result / sizeof(long));
  }
  return (result);
}
