"======================================================================
|
|   Smalltalk TCP/IP sockets - Stream hierarchy
|
|   $Revision: 1.95.1$
|   $Date: 2000/12/27 10:45:49$
|   $Author: pb$
|
 ======================================================================"


"======================================================================
|
| Copyright 1988-92, 1994-95, 1999, 2000 Free Software Foundation, Inc.
| Written by Paolo Bonzini.
|
| This file is part of the GNU Smalltalk class library.
|
| The GNU Smalltalk class library is free software; you can redistribute it
| and/or modify it under the terms of the GNU Lesser General Public License
| as published by the Free Software Foundation; either version 2.1, or (at
| your option) any later version.
| 
| The GNU Smalltalk class library is distributed in the hope that it will be
| useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
| MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser
| General Public License for more details.
| 
| You should have received a copy of the GNU Lesser General Public License
| along with the GNU Smalltalk class library; see the file COPYING.LESSER.
| If not, write to the Free Software Foundation, 59 Temple Place - Suite
| 330, Boston, MA 02111-1307, USA.  
|
 ======================================================================"


Stream subclass: #AbstractSocket
	  instanceVariableNames: 'impl'
	  classVariableNames: 'CheckPeriod Timeout'
	  poolDictionaries: ''
	  category: 'Sockets-Streams'
!

AbstractSocket class instanceVariableNames: 'defaultImplementationClass'!

AbstractSocket subclass: #DatagramSocket
	instanceVariableNames: ''
	classVariableNames: 'DefaultBufferSize DefaultRawImplementationClass'
	poolDictionaries: ''
	category: 'Sockets-Streams'
!

DatagramSocket subclass: #MulticastSocket
	instanceVariableNames: ''
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Sockets-Streams'
!

AbstractSocket subclass: #ServerSocket
	  instanceVariableNames: ''
	  classVariableNames: ''
	  poolDictionaries: ''
	  category: 'Sockets-Streams'
!

AbstractSocket subclass: #Socket
	  instanceVariableNames: 'lookahead readBuffer writeBuffer outOfBand'
	  classVariableNames: 'Ports ReadBufferSize WriteBufferSize'
	  poolDictionaries: ''
	  category: 'Sockets-Streams'
!

!AbstractSocket class methodsFor: 'timed-out operations'!

checkPeriod
    "Answer the period that is to elapse between socket polls if data
     data is not ready and the connection is still open (in milliseconds)"
    ^CheckPeriod
!

checkPeriod: anInteger
    "Set the period that is to elapse between socket polls if data
     data is not ready and the connection is still open (in milliseconds)"
    CheckPeriod := anInteger truncated
!

timeout
    "Answer the period that is to elapse between the request for (yet
     unavailable) data and the moment when the connection is considered dead
     (in milliseconds)"
    ^Timeout
!

timeout: anInteger
    "Set the period that is to elapse between the request for (yet
     unavailable) data and the moment when the connection is considered
     dead (in milliseconds)"
    Timeout := anInteger truncated
! !

!AbstractSocket class methodsFor: 'accessing'!

defaultImplementationClass
    "Answer the class that, by default, is used to map between the
     receiver's protocol and a low-level C interface."
    ^defaultImplementationClass
!

defaultImplementationClass: aClass
    "Set which class will be used by default to map between the
     receiver's protocol and a low-level C interface."
    defaultImplementationClass := aClass
! !

!AbstractSocket class methodsFor: 'instance creation'!

new: implementation
    "Answer a new instance of the receiver, using as the underlying
     layer the object passed as the `implementation' parameter; the
     object is probably going to be some kind of AbstractSocketImpl."
    ^super new initialize: implementation
!

new
    "Answer a new instance of the receiver, using a new instance of
     the defaultImplementationClass as the underlying implementation
     layer."
    ^self new: self defaultImplementationClass new
! !

!AbstractSocket methodsFor: 'accessing'!

address
    "Answer an IP address that is of common interest (this can be either
     the local or the remote address, according to the definition in the
     subclass)."
    self subclassResponsibility
!

available
    "Answer whether there is data available on the socket."
    ^self implementation canRead
!

close
    "Close the socket represented by the receiver."
    self flush.
    self implementation close.
!

flush
    "Flush any buffers used by the receiver."
!

isOpen
    "Answer whether the connection between the receiver and the remote
     endpoint is still alive."
    self implementation isNil ifFalse: [ ^false ].
    ^self implementation isOpen
!

localAddress
    "Answer the local IP address of the socket."
    self implementation isNil ifTrue: [ self error: 'socket not connected' ].
    ^self implementation localAddress
!

localPort
    "Answer the local IP port of the socket."
    self implementation isNil ifTrue: [ self error: 'socket not connected' ].
    ^self implementation localPort
!

port
    "Answer an IP port that is of common interest (this can be the port for 
     either the local or remote endpoint, according to the definitions in the
     subclass"
    self subclassResponsibility
!

remoteAddress
    "Answer the IP address of the socket's remote endpoint."
    self implementation isNil ifTrue: [ self error: 'socket not connected' ].
    ^self implementation remoteAddress
!

remotePort
    "Answer the IP port of the socket's remote endpoint."
    self implementation isNil ifTrue: [ self error: 'socket not connected' ].
    ^self implementation remotePort
! !

!AbstractSocket methodsFor: 'printing'!

printOn: aStream
    aStream
	print: self class;
	nextPut: $[;
	print: self address;
	nextPut: $: ;
	print: self port;
	nextPutAll: ']'
! !

!AbstractSocket methodsFor: 'private'!

implementation
    ^impl
!

initialize: implementation
    impl := implementation.
!

waitUntil: aBlock then: resultBlock onTimeoutDo: timeoutBlock

    Timeout // CheckPeriod timesRepeat: [
	aBlock value ifTrue: [ ^resultBlock value ].
	(Delay forMilliseconds: CheckPeriod) wait
    ].
    self close.
    ^timeoutBlock value.
! !

!AbstractSocket methodsFor: 'stream protocol'!

atEnd
    "By default, answer whether the connection is still open."
    ^self isOpen
!

next
    "Read another character from the socket, failing if the connection is
     dead."
    ^self implementation next
!

nextPut: char
    "Write `char' to the socket, failing if the connection is dead.  The
     SIGPIPE signal is automatically caught and ignored by the system."
    ^self implementation nextPut: char
! !

!DatagramSocket class methodsFor: 'accessing'!

defaultRawImplementationClass
    ^DefaultRawImplementationClass
!

defaultRawImplementationClass: size
    DefaultRawImplementationClass := size
!

defaultBufferSize
    ^DefaultBufferSize
!

defaultBufferSize: size
    DefaultBufferSize := size
! !

!DatagramSocket class methodsFor: 'initialization'!

initialize
    DatagramSocket
	defaultImplementationClass: UDPSocketImpl;
	defaultRawImplementationClass: ICMPSocketImpl;
	defaultBufferSize: 128
! !

!DatagramSocket class methodsFor: 'instance creation'!

raw
    "Create a new raw socket, providing access to low-level network protocols
     and interfaces. Ordinary user programs usually have no need to use this
     style."
    ^self new: DefaultRawImplementationClass new
!

new
    "Answer a new datagram socket (by default an UDP socket), without
     a specified local address and port."
    ^self local: nil port: 0
!

port: localPort
    "Create a new socket and bind it to the local host on the given port."
    ^self
	remote: nil
	port: 0
	local: nil
	port: localPort
!

local: ipAddressOrString port: remotePort
    "Create a new socket and bind it to the given host (passed as a
     String to be resolved or as an IPAddress), on the given port."
    ^self
	remote: nil
	port: 0
	local: ipAddressOrString
	port: remotePort
!

remote: ipAddressOrString port: remotePort local: ipAddress port: localPort
    "Create a new socket and bind it to the given host (passed as a
     String to be resolved or as an IPAddress), and to the given remotePort.
     The default destination for the datagrams will be ipAddressOrString
     (if not nil), on the remotePort port."
    ^super new
	remote: ipAddressOrString
	port: remotePort
	local: ipAddress
	port: localPort
! !

!DatagramSocket methodsFor: 'accessing'!

address
    "Answer the local address."
    ^self localAddress
!

bufferSize
    ^self implementation bufferSize
!

bufferSize: size
    self implementation bufferSize: size
!

nextPut: aDatagram
    "Send the given datagram on the socket."
    self
	waitUntil: [ self implementation canRead ]
	then: [ self implementation nextPut: aDatagram. aDatagram ]
	onTimeoutDo: [ nil ]
!

port
    "Answer the local port."
    ^self localPort
!

peek
    "Peek for a datagram on the socket and answer it."
    ^self
	waitUntil: [ self implementation canRead ]
	then: [ self implementation peek ]
	onTimeoutDo: [ nil ]
!

peek: datagram
    "Peek for a datagram on the socket, store it in `datagram', and
     answer the datagram itself."
    ^self
	waitUntil: [ self implementation canRead ]
	then: [ self implementation peek: datagram. true ]
	onTimeoutDo: [ false ]
!

receive: datagram
    "Read a datagram from the socket, store it in `datagram', and
     answer the datagram itself."
    ^self
	waitUntil: [ self implementation canRead ]
	then: [ self implementation receive: datagram. true ]
	onTimeoutDo: [ false ]
! !


!DatagramSocket methodsFor: 'direct operations'!

nextFrom: ipAddress port: port
    "Answer the next datagram from the given address and port."
    self
	waitUntil: [ self implementation canRead ]
	then: [ self implementation nextFrom: ipAddress port: port ]
	onTimeoutDo: [ nil ]
! !

!DatagramSocket methodsFor: 'private'!

create
    self implementation create
!

remote: remoteAddress port: remotePort local: ipAddress port: localPort
    | remoteAddr |
    remoteAddr := remoteAddress isString
	ifTrue: [ self implementation addressClass byName: remoteAddress ]
	ifFalse: [ remoteAddress ].

    self create.
    self implementation
        bufferSize: self class defaultBufferSize;
        connectTo: remoteAddr;
        port: remotePort;

	bindTo: (ipAddress isNil
	    ifTrue: [ self implementation addressClass anyLocalAddress ]
	    ifFalse: [ ipAddress ])
	port: localPort.
! !


!MulticastSocket methodsFor: 'instance creation'!

interface
    ^impl ipMulticastIf
!

interface: ipAddress
    impl ipMulticastIf: ipAddress
!

join: ipAddress
    "Join the multicast socket at the given IP address"
    impl join: ipAddress
!

leave: ipAddress
    "Leave the multicast socket at the given IP address"
    impl leave: ipAddress
!

nextPut: packet timeToLive: timeToLive
    "Send the datagram with a specific TTL (time-to-live)"
    | oldTTL |
    oldTTL := impl timeToLive.
    impl timeToLive: timeToLive.
    self nextPut: packet.
    impl timeToLive: oldTTL
!

timeToLive
    "Answer the socket's datagrams' default time-to-live"
    ^impl timeToLive
!

timeToLive: newTTL
    "Set the default time-to-live for the socket's datagrams"
    impl timeToLive: newTTL
! !

!MulticastSocket methodsFor: 'private'!

create
    impl create.
    impl soReuseAddr: -1.
! !


!ServerSocket class methodsFor: 'instance creation'!

initialize
    ServerSocket defaultImplementationClass: TCPSocketImpl
! !

!ServerSocket class methodsFor: 'instance creation'!

defaultQueueSize
    "Answer the default length of the queue for pending connections.  When
     the queue fills, new clients attempting to connect fail until the server
     is send #accept to accept a connection from the queue."
    ^50
!

queueSize: backlog
    "Answer a new ServerSocket serving on any local address and port, with a
     pending connections queue of the given length."
    ^self
        port: 0
        queueSize: backlog
        bindTo: nil
!

queueSize: backlog bindTo: ipAddress
    "Answer a new ServerSocket serving on the given local address,
     and on any port, with a pending connections queue of the given length."
    ^self
        port: 0
        queueSize: backlog
        bindTo: ipAddress
!

port: anInteger
    "Answer a new ServerSocket serving on any local address, on the given
     port, with a pending connections queue of the default length."
    ^self
        port: anInteger
        queueSize: self defaultQueueSize
        bindTo: nil
!

port: anInteger queueSize: backlog
    "Answer a new ServerSocket serving on any local address, on the given
     port, with a pending connections queue of the given length."
    ^self
        port: anInteger
        queueSize: backlog
        bindTo: nil
!

port: anInteger bindTo: ipAddress
    "Answer a new ServerSocket serving on the given address and port,
     with a pending connections queue of the default length."
    ^self
        port: anInteger
        queueSize: self defaultQueueSize
        bindTo: ipAddress
!

port: anInteger queueSize: backlog bindTo: ipAddress
    "Answer a new ServerSocket serving on the given address and port,
     and with a pending connections queue of the given length."
    ^self new
	port: anInteger queueSize: backlog bindTo: ipAddress
! !

!ServerSocket methodsFor: 'accessing'!

address
    "Answer the local address"
    ^self localAddress
!

port
    "Answer the local port (the port that the passive socket is listening on)."
    ^self localPort
!

accept
    "Accept a new connection and create a new instance of Socket if there is
     one, else answer nil."
    self available ifFalse: [ ^nil ].	"Make it non-blocking"
    ^self primAccept: Socket
!

accept: socketClass
    "Accept a new connection and create a new instance of socketClass if
     there is one, else answer nil.  This is usually needed only to create
     DatagramSockets."

    self available ifFalse: [ ^nil ].	"Make it non-blocking"
    ^self primAccept: socketClass
!

primAccept: socketClass
    "Accept a new connection and create a new instance of Socket if there is
     one, else fail."
    | implClass newImpl |
    implClass := socketClass defaultImplementationClass.
    newImpl := self implementation accept: implClass.
    ^socketClass new: newImpl
! !

!ServerSocket methodsFor: 'initializing'!

port: anInteger queueSize: backlog bindTo: ipAddress

    | localAddr |
    localAddr := ipAddress isNil
	ifTrue: [ ipAddress ]
	ifFalse: [ self implementation addressClass unknownAddress ].

    self implementation
	create;

	bindTo: (ipAddress isNil
	    ifTrue: [ self implementation addressClass anyLocalAddress ]
	    ifFalse: [ ipAddress ])
	port: anInteger;

	listen: backlog
! !


!Socket class methodsFor: 'well known ports'!

portEcho	   "Answer the port on which the ECHO service listens"	  ^7!
portDiscard	   "Answer the port on which the DISCARD service listens" ^9!
portSystat	   "Answer the port on which the SYSTAT service listens"  ^11!
portDayTime	   "Answer the port on which the TOD service listens"     ^13!
portNetStat	   "Answer the port on which the NETSTAT service listens" ^15!
portFTP		   "Answer the port on which the FTP daemon listens"      ^21!
portTelnet	   "Answer the port on which the TELNET daemon listens"   ^23!
portSMTP	   "Answer the port on which the SMTP daemon listens"     ^25!
portTimeServer     "Answer the port on which the time server listens"     ^37!
portDNS		   "Answer the port on which the DNS listens"             ^42!
portWhois	   "Answer the port on which the WHOIS daemon listens"    ^43!
portGopher	   "Answer the port on which the Gopher daemon listens"   ^70!
portFinger	   "Answer the port on which the finger daemon listens"   ^79!
portHTTP	   "Answer the port on which the http daemon listens"     ^80!
portPOP3	   "Answer the port on which the pop3 daemon listens"     ^110!
portNNTP	   "Answer the port on which the nntp daemon listens"     ^119!
portExecServer     "Answer the port on which the exec server listens"     ^512!
portLoginServer    "Answer the port on which the rlogin daemon listens"   ^513!
portCmdServer      "Answer the port on which the rsh daemon listens"      ^514!
portKerberosIV	   "Answer the port for Kerberos IV authentications"      ^750!
portReserved       "Answer the last port reserved to privileged processes"^1023!

defaultPortAt: protocol
    "Answer the port that is used (by default) for the given service (high
     level protocol)"
    ^Ports at: protocol
!

defaultPortAt: protocol ifAbsent: port
    "Answer the port that is used (by default) for the given service (high
     level protocol), or the specified port if none is registered."
    ^Ports at: protocol ifAbsent: port
!

defaultPortAt: protocol put: port
    "Associate the given port to the service specified by `protocol'."
    ^Ports at: protocol put: port
!

initialize
    "Initialize the receiver's defaults"
    self defaultImplementationClass: TCPSocketImpl.
    self readBufferSize: 1024; writeBufferSize: 256.
    Ports := Dictionary new
	at: 'ftp' put: 21;
	at: 'telnet' put: 23;
	at: 'smtp' put: 25;
	at: 'dns' put: 42;
	at: 'whois' put: 43;
	at: 'finger' put: 79;
	at: 'http' put: 80;
	at: 'pop3' put: 110;
	at: 'nntp' put: 119;
	at: 'kerberos4' put: 750;
	yourself
! !

!Socket class methodsFor: 'accessing'!

readBufferSize
    "Answer the size of the read buffer for newly-created sockets"
    ^ReadBufferSize
!

readBufferSize: anInteger
    "Set the size of the read buffer for newly-created sockets"
    ReadBufferSize := anInteger
!

writeBufferSize
    "Answer the size of the write buffer for newly-created sockets"
    ^WriteBufferSize
!

writeBufferSize: anInteger
    "Set the size of the write buffer for newly-created sockets"
    WriteBufferSize := anInteger
! !

!Socket class methodsFor: 'instance creation'!

remote: ipAddressOrString port: remotePort
    "Create a new socket and connect to the given host (passed as a
     String to be resolved or as an IPAddress), and to the given port."
    ^self
	remote: ipAddressOrString
	port: remotePort
	local: nil
	port: 0
!

remote: ipAddressOrString port: remotePort local: ipAddress port: localPort
    "Create a new socket and connect to the given host (passed as a
     String to be resolved or as an IPAddress), and to the given remotePort.
     Then bind it to the local address passed in ipAddress, on the localPort
     port; if the former is nil, any local address will do, and if the latter
     is 0, any local port will do."
    ^self new
	remote: ipAddressOrString
	port: remotePort
	local: ipAddress
	port: localPort
! !

!Socket methodsFor: 'accessing'!

address
    "Answer the address of the remote endpoint"
    ^self remoteAddress
!

port
    "Answer the port of the remote endpoint"
    ^self remotePort
!

soLinger
    "Answer the number of seconds that the socket is allowed to wait
     if it promises reliable delivery but has unacknowledged/untransmitted
     packets when it is closed, or nil if those packets are left to their
     destiny or discarded."
    ^self implementation soLinger
!

soLinger: linger
    "Set the number of seconds that the socket is allowed to wait
     if it promises reliable delivery but has unacknowledged/untransmitted
     packets when it is closed."
    ^self implementation soLinger: linger
!

soLingerOff
    "Specify that, even if the socket promises reliable delivery, any
     packets that are unacknowledged/untransmitted when it is closed
     are to be left to their destiny or discarded."
    ^self implementation soLinger: nil
!

species
    ^String
!

soReuseAddr
    "Answer whether another socket can be bound the same local address as this
     one.  If you enable this option, you can actually have two sockets with the
     same Internet port number; but the system won't allow you to use the two
     identically-named sockets in a way that would confuse the Internet.  The
     reason for this option is that some higher-level Internet protocols,
     including FTP, require you to keep reusing the same socket number."
    ^self implementation soReuseAddr
!

soReuseAddr: aBoolean
    "Set whether another socket can be bound the same local address as this
     one."
    ^self implementation soReuseAddr: aBoolean
! !

!Socket methodsFor: 'printing'!

printOn: aStream
    aStream
	print: self class;
	nextPutAll: '[local ';
	print: self localAddress;
	nextPut: $: ;
	print: self localPort;
	nextPutAll: ', remote ';
	print: self remoteAddress;
	nextPut: $: ;
	print: self remotePort;
	nextPut: $]
! !

!Socket methodsFor: 'private'!

remote: ipAddressOrString port: remotePort local: ipAddress port: localPort
    | remoteAddr |
    self implementation create.

    ipAddress isNil ifFalse: [
	self implementation bindTo: ipAddress port: localPort
    ].

    remoteAddr := ipAddressOrString isString
	ifTrue: [ self implementation addressClass byName: ipAddressOrString ]
	ifFalse: [ ipAddressOrString ].

    self implementation connectTo: remoteAddr port: remotePort
!

species
    ^String
! !

!Socket methodsFor: 'stream protocol'!

atEnd
    "Answer whether more data is available on the socket"
    ^self peek isNil
!

available
    "Answer whether more data is available in the socket's read buffer
     or from the operating system."
    ^(self hasReadBuffer and: [ self readBuffer notEmpty ])
	or: [ super available ]
!

bufferContents
    "Answer the current contents of the read buffer"
    | result |
    result := self readBuffer bufferContents.
    lookahead isNil ifFalse: [
	result := lookahead asString, result.
	lookahead := nil.
    ].
    ^result
!
    
close
    "Flush and close the socket."
    super close.
    self deleteBuffers
!

fill
    "Fill the read buffer with data read from the socket"
    self readBuffer fill
!

flush
    "Flush the write buffer to the operating system"
    self writeBuffer flush
!

isPeerAlive
    "Answer whether the connection with the peer remote machine is still
     valid."
    ^self readBuffer notNil
!

next
    "Read a byte from the socket.  This might yield control to other
     Smalltalk Processes."
    | result |
    lookahead isNil ifTrue: [ ^self readBuffer next ].
    result := lookahead.
    lookahead := nil.
    ^result
!

next: count
    "Read `count' bytes from the socket.  This might yield control to other
     Smalltalk Processes."
    | result |
    lookahead isNil ifTrue: [ ^self readBuffer next: count ].
    result := (String with: lookahead), (self readBuffer next: count - 1).
    lookahead := nil.
    ^result
!

nextPut: char
    "Write a character to the socket; this acts as a bit-bucket when
     the socket is closed.  This might yield control to other
     Smalltalk Processes."
    self writeBuffer isNil ifTrue: [ ^self ].
    self writeBuffer nextPut: char
!

nextPutAll: aString
    "Write aString to the socket; this acts as a bit-bucket when
     the socket is closed.  This might yield control to other
     Smalltalk Processes."
    self writeBuffer isNil ifTrue: [ ^self ].
    self writeBuffer nextPutAll: aString
!

peek
    "Read a byte from the socket, without advancing the buffer; answer
     nil if no more data is available.  This might yield control to other
     Smalltalk Processes."
    lookahead isNil ifTrue: [
	self readBuffer isNil ifTrue: [ ^nil ].
	self readBuffer atEnd ifTrue: [ ^nil ].
	lookahead := self readBuffer next ].
    ^lookahead
!

peekFor: anObject
    "Read a byte from the socket, advancing the buffer only if it matches
     anObject; answer whether they did match or not.  This might yield
     control to other Smalltalk Processes."
    lookahead isNil ifTrue: [
	self readBuffer isNil ifTrue: [ ^false ].
	self readBuffer atEnd ifTrue: [ ^false ].
	lookahead := self readBuffer next ].
    ^lookahead = anObject
        ifTrue: [ lookahead := nil. true ]
        ifFalse: [ false ]
!

readBufferSize: size
    "Create a new read buffer of the given size (which is only
     possible before the first read or if the current buffer is
     empty)."
    readBuffer isNil ifTrue: [ ^self ].
    (self hasReadBuffer and: [ readBuffer notEmpty ])
	ifTrue: [ self error: 'read buffer must be empty before changing its size' ].

    readBuffer := self newReadBuffer: size.
!

writeBufferSize: size
    "Create a new write buffer of the given size, flushing the
     old one is needed.  This might yield control to other 
     Smalltalk Processes."
    writeBuffer isNil ifTrue: [ ^self ].
    self hasWriteBuffer ifTrue: [ writeBuffer flush ].
    writeBuffer := self newWriteBuffer: size.
! !

!Socket methodsFor: 'private - buffering'!

deleteBuffers
    readBuffer := writeBuffer := nil.
!

noBufferFlag
    "Value that means `lazily initialize the readBuffer and writeBuffer'."
    ^0
!

hasReadBuffer
    ^readBuffer ~~ self noBufferFlag
!

hasWriteBuffer
    ^writeBuffer ~~ self noBufferFlag
!

initialize: implementation
    super initialize: implementation.
    readBuffer := writeBuffer := self noBufferFlag.
!

newReadBuffer: size
    ^(ReadBuffer on: (String new: size))
	fillBlock: [ :data :size || alive num |
	    alive := true.
	    self
		waitUntil: [
		    self implementation canRead 
			ifFalse: [ (alive := self implementation isOpen) not ]
			ifTrue: [ (num := self implementation read: data numBytes: size) > 0 ]
                ]
		then: [
		    alive ifTrue: [ num ] ifFalse: [ self deleteBuffers. 0 ]
		]
		onTimeoutDo: [ self implementation close. self deleteBuffers. 0 ]
	]
!

newWriteBuffer: size
    ^(WriteBuffer on: (String new: size))
	flushBlock: [ :data :size || alive |
	    alive := true.
	    self
		waitUntil: [
		    self implementation canWrite or: [
			(alive := self implementation isOpen) not ] ]
		then: [
		    alive := alive and: [
			(self implementation write: data numBytes: size) > -1 ].

		    alive ifFalse: [ self deleteBuffers ]
		]
		onTimeoutDo: [ self implementation close. self deleteBuffers ]
	]
!

readBuffer
    readBuffer == self noBufferFlag
	ifTrue: [ readBuffer := self newReadBuffer: ReadBufferSize ].
    ^readBuffer
!

writeBuffer
    writeBuffer == self noBufferFlag
	ifTrue: [ writeBuffer := self newWriteBuffer: WriteBufferSize ].
    ^writeBuffer
! !

!Socket methodsFor: 'out-of-band data'!

outOfBand
    "Return a datagram socket to be used for receiving out-of-band data
     on the receiver."
    outOfBand isNil ifTrue: [
	outOfBand :=
	    DatagramSocket new: self implementation outOfBandImplClass.
	self implementation outOfBandInitialize: outOfBand implementation
    ].
    ^outOfBand
! !


MulticastSocket defaultImplementationClass: UDPSocketImpl!
AbstractSocket timeout: 30000; checkPeriod: 1000!
