"======================================================================
|
|   Generic web-server framework
|
|   $Revision: 1.95.1$
|   $Date: 2000/12/27 10:45:49$
|   $Author: pb$
|
 ======================================================================"


"======================================================================
|
| Copyright 1988-92, 1994-95, 1999, 2000 Free Software Foundation, Inc.
| Written by Paolo Bonzini
|
| This file is part of GNU Smalltalk.
|
| GNU Smalltalk is free software; you can redistribute it and/or modify it
| under the terms of the GNU General Public License as published by the Free
| Software Foundation; either version 2, or (at your option) any later version.
|
| GNU Smalltalk is distributed in the hope that it will be useful, but WITHOUT
| ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
| FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
| details.
|
| You should have received a copy of the GNU General Public License along with
| GNU Smalltalk; see the file COPYING.	If not, write to the Free Software
| Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
|
 ======================================================================"

Stream subclass:  #CrLfStream
	instanceVariableNames: 'stream readStatus eatLf '
	classVariableNames: 'Cr Lf '
	poolDictionaries: ''
	category: 'Web-Framework'!

CrLfStream comment:
'A CrLfStream acts as a pipe which transforms incoming data into LF-separated
lines, and outgoing data into CRLF-separated lines.'!

Object subclass:  #NetworkThread
	instanceVariableNames: 'process socket priority '
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Sockets-Serving framework'!

NetworkThread comment:
'A NetworkThread runs a process attached to a specified socket.'!

NetworkThread subclass:  #NetworkServer
	instanceVariableNames: 'port process'
	classVariableNames: 'Servers'
	poolDictionaries: ''
	category: 'Sockets-Serving framework'!

NetworkServer comment:
'A NetworkServer keeps a socket listening on a port, and dispatches incoming
requests to NetworkSession objects.'!

NetworkThread subclass:  #NetworkSession
	instanceVariableNames: 'server'
	classVariableNames: ''
	poolDictionaries: ''
	category: 'Sockets-Serving framework'!

NetworkSession comment:
'NetworkSessions divide a session in separate requests and provide 
exception handling for those.'!

!NetworkThread class methodsFor: 'instance creation'!

new
    ^super new initialize! !

!NetworkThread methodsFor: 'private'!

newProcess
    ^[ self run ] forkAt: priority
! 

socket
    ^socket
!

createSocket
    self subclassResponsibility
! !

!NetworkThread methodsFor: 'printing'!

printOn: aStream
    aStream print: self class; nextPut: $:.
    self isServing ifFalse: [ ^aStream nextPutAll: 'idle' ].
    aStream print: self socket port
! !

!NetworkThread methodsFor: 'running'!

run
    self subclassResponsibility
! !

!NetworkThread methodsFor: 'initialize-release'!

initialize
    priority := self defaultPriority!

defaultPriority
    ^Processor userSchedulingPriority!

release
    self stop.
    super release! !

!NetworkThread methodsFor: 'serving'!

isRunning
    ^process notNil!

start
    self isRunning ifTrue: [^self].
    socket := self createSocket.
    process := self newProcess!

stop
    self isRunning
	ifTrue:
	    [process terminate.
	    process := nil.
	    socket close.
	    socket := nil.
	    ]! !


!NetworkServer class methodsFor: 'accessing'!

at: port
    | server |
    Servers isNil ifTrue: [ Servers := Dictionary new ].
    ^Servers at: port ifAbsentPut: [ self new ].
!

terminateServer: port
    Servers isNil ifTrue: [ ^self ].
    (Servers includesKey: port) ifTrue: [ 
	(Servers at: port) release.
	Servers removeKey: port.
    ]
!

initializeServer: port
    | server |
    server := self at: port.
    server isRunning ifFalse: [server startOn: port].
    ^server
! !

!NetworkServer methodsFor: 'accessing'!

port
    ^port!

port: anObject
    self stop.
    port := anObject!

priority
    ^priority!

priority: anInteger
    priority := anInteger.
    self isRunning ifTrue: [process priority: priority]!

startOn: aPortNumber
    self port: aPortNumber.
    self start
! !

!NetworkServer methodsFor: 'abstract'!

newSession
    self subclassResponsibility
!

respondTo: aRequest
    self subclassResponsibility
! !

!NetworkServer methodsFor: 'private'!

defaultPriority
    ^Processor lowIOPriority!

run
    | clientConnection delay |

	Processor activeProcess name: 'listen'.
	delay := Delay forMilliseconds: 20.
	[
	    [ socket available ] whileFalse: [ delay wait ].
	    self newSession
		server: self;
		start
	] repeat!

createSocket
    ^ServerSocket port: port
! !

!NetworkSession methodsFor: 'private'!

createSocket
    ^server socket accept
!

stop
    super stop.
    Servers removeKey: self port ifAbsent: [].
!

run
    | req time |
    Processor activeProcess name: 'connection'.
    [
	[
	    req := self next.
	    time := Time millisecondsToRun: [
		self server respondTo: req.
		req release
	    ]
	]   on: ExAll
	    do: [ :ex |
		"Ignore errors due to bad communication lines."
		self socket isPeerAlive ifFalse: [ ex return ].
		ex pass
	    ].

	self log: req time: time.

	self socket isPeerAlive
    ] whileTrue
! !

!NetworkSession methodsFor: 'accessing'!

server
    ^server
!

server: aServer
    server := aServer
! !

!NetworkSession methodsFor: 'abstract'!

next
    self subclassResponsibility
!

log: request time: milliseconds
! !

!CrLfStream class methodsFor: 'instance creation'!

on: aStream
    Cr := Character cr.
    Lf := Character nl.
    ^self new on: aStream! !

!CrLfStream methodsFor: 'initializing'!

on: aStream
    stream := aStream.
    eatLf := false.
    readStatus := #none! !

!CrLfStream methodsFor: 'stream'!

atEnd
    ^stream atEnd and: [ readStatus == #none ]!

close
    stream close!

flush
    stream flush!

next
    | result |
    readStatus == #none ifFalse: [
	readStatus == Cr ifTrue: [ stream peekFor: Lf ].
	readStatus := #none.
	^Lf ].

    result := stream next.
    ^(result == Cr or: [ result == Lf ])
	ifTrue: [ readStatus := result. Cr ]
	ifFalse: [ result ]!

nextPut: aCharacter
    eatLf
	ifTrue: [
	    eatLf := false.
	    aCharacter == Lf ifTrue: [ ^self ] ]
	ifFalse: [
	    aCharacter == Lf ifTrue: [
		stream nextPut: Cr; nextPut: Lf.
		^self ] ].

    stream nextPut: aCharacter.
    aCharacter == Cr ifTrue: [
	stream nextPut: Lf.
	eatLf := true.
    ]!

peek
    | result |
    readStatus == #none ifFalse: [
	readStatus == Cr ifTrue: [ stream peekFor: Lf ].
	readStatus := Lf. "peek for LF just once"
	^Lf ].

    result := stream peek.
    ^result == Lf
	ifTrue: [ Cr ]
	ifFalse: [ result ]!

peekFor: aCharacter
    | result success |
    readStatus == #none ifFalse: [
	readStatus == Cr ifTrue: [ stream peekFor: Lf ].
	success := aCharacter == Lf.
	readStatus := success ifTrue: [ #none ] ifFalse: [ Lf ]. "peek for LF just once"
	^success
    ].
    result := stream peek.
    (result == Cr or: [ result == Lf ]) ifTrue: [
	success := aCharacter == Cr.
	success ifTrue: [ readStatus := stream next ].
	^success
    ].
    success := aCharacter == result.
    success ifTrue: [ stream next ].
    ^success!

species
    ^stream species! !
