/******************************** -*- C -*- ****************************
 *
 *	C Callin facility
 *
 *	This module provides the routines necessary to allow C code to
 *	invoke Smalltalk messages on objects.
 *
 *
 ***********************************************************************/


/***********************************************************************
 *
 * Copyright 1988,89,90,91,92,94,95,99,2000,2001,2002
 * Free Software Foundation, Inc.
 * Written by Steve Byrne.
 *
 * This file is part of GNU Smalltalk.
 *
 * GNU Smalltalk is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2, or (at your option) any later 
 * version.
 * 
 * GNU Smalltalk is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * GNU Smalltalk; see the file COPYING.  If not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
 *
 ***********************************************************************/


#include "gst.h"
#include "gstpriv.h"
#include "rbtrees.h"

#include <stdio.h>
#if STDC_HEADERS
#include <stdlib.h>
#endif /* STDC_HEADERS */

#include <stdarg.h>

#include <math.h>

#ifndef NAN
#define NAN (0.0 / 0.0)
#endif

typedef struct oop_registry
{
  rb_node_t rb;
  OOP oop;
  int usage;
}
oop_registry;

typedef struct oop_array_registry
{
  rb_node_t rb;
  OOP **first;
  OOP **last;
  int usage;
}
oop_array_registry;


/* The registry of OOPs which have been passed to C code.  Implemented
   as a red-black tree.  The registry is examined at GC time to ensure
   that OOPs that C code knows about don't go away. */
static oop_registry *oop_registry_root;
static oop_array_registry *oop_array_registry_root;

VMProxy gst_interpreter_proxy = {
  NULL, NULL, NULL,

  _gst_msg_send, _gst_vmsg_send, _gst_nvmsg_send, _gst_str_msg_send,
  _gst_msg_sendf,
  _gst_eval_expr, _gst_eval_code,

  _gst_object_alloc, _gst_basic_size,

  _gst_define_cfunc, _gst_register_oop, _gst_unregister_oop,

/* Convert C datatypes to Smalltalk types */

  _gst_id_to_oop, _gst_int_to_oop, _gst_float_to_oop, _gst_bool_to_oop,
  _gst_char_to_oop, _gst_class_name_to_oop,
  _gst_string_to_oop, _gst_byte_array_to_oop, _gst_symbol_to_oop,
  _gst_c_object_to_oop, _gst_type_name_to_oop, _gst_set_c_object,

/* Convert Smalltalk datatypes to C data types */

  _gst_oop_to_c, _gst_oop_to_id, _gst_oop_to_int, _gst_oop_to_float,
  _gst_oop_to_bool, _gst_oop_to_char,
  _gst_oopto_string, _gst_oop_to_byte_array, _gst_oop_to_cobject,

/* Smalltalk process support */
  _gst_async_signal, _gst_sync_wait, _gst_async_signal_and_unregister,
};

OOP
_gst_msg_send (OOP receiver,
	       OOP selector,
	       ...)
{
  va_list args;
  OOP anArg, result;
  int numArgs;

  va_start (args, selector);

  if (!_gst_smalltalk_initialized)
    gst_init_smalltalk ();

  /* These objects don't need to be incubated because they are being
     pushed onto the Smalltalk stack which will make them visible to
     the GC */
  _gst_prepare_execution_environment ();
  PUSH_OOP (receiver);
  for (numArgs = 0; (anArg = va_arg (args, OOP)) != NULL; numArgs++)
    PUSH_OOP (anArg);

  if (numArgs != _gst_selector_num_args (selector))
    result = _gst_nil_oop;
  else
    {
      SEND_MESSAGE (selector, numArgs, false);
      _gst_interpret ();
      result = POP_OOP ();
    }
  _gst_finish_execution_environment ();

  return (result);
}

OOP
_gst_vmsg_send (OOP receiver,
		OOP selector,
		OOP * args)
{
  OOP anArg, result;
  int numArgs;

  if (!_gst_smalltalk_initialized)
    gst_init_smalltalk ();

  /* These objects don't need to be incubated because they are being
     pushed onto the Smalltalk stack which will make them visible to
     the GC */
  _gst_prepare_execution_environment ();
  PUSH_OOP (receiver);
  for (numArgs = 0; (anArg = *args++) != NULL; numArgs++)
    PUSH_OOP (anArg);

  if (numArgs != _gst_selector_num_args (selector))
    result = _gst_nil_oop;
  else
    {
      SEND_MESSAGE (selector, numArgs, false);
      _gst_interpret ();
      result = POP_OOP ();
    }
  _gst_finish_execution_environment ();

  return (result);
}

OOP
_gst_nvmsg_send (OOP receiver,
		 OOP selector,
		 OOP * args,
		 int nargs)
{
  OOP result;
  int numArgs;

  if (!_gst_smalltalk_initialized)
    gst_init_smalltalk ();

  if (nargs != _gst_selector_num_args (selector))
    return (_gst_nil_oop);

  /* These objects don't need to be incubated because they are being
     pushed onto the Smalltalk stack which will make them visible to
     the GC */
  _gst_finish_execution_environment ();
  _gst_prepare_execution_environment ();
  PUSH_OOP (receiver);
  for (numArgs = 0; nargs--; numArgs++)
    PUSH_OOP (*args++);

  SEND_MESSAGE (selector, numArgs, false);
  _gst_interpret ();
  result = POP_OOP ();
  _gst_finish_execution_environment ();

  return (result);
}

OOP
_gst_str_msg_send (OOP receiver, 
		   char *sel,
		   ...)
{
  va_list args;
  OOP selector, anArg, result;
  int numArgs;
  inc_ptr incPtr;

  va_start (args, sel);

  if (!_gst_smalltalk_initialized)
    gst_init_smalltalk ();

  selector = _gst_intern_string (sel);

  /* It's possible (there is an existing case) that in SEND_MESSAGE a
     GC can occur, and selector is not protected pb: but isn't the
     selector a gst_symbol, and hence in the root set? */
  incPtr = INC_SAVE_POINTER ();
  INC_ADD_OOP (selector);

  _gst_prepare_execution_environment ();
  PUSH_OOP (receiver);
  for (numArgs = 0; (anArg = va_arg (args, OOP)) != NULL; numArgs++)
    PUSH_OOP (anArg);

  if (numArgs != _gst_selector_num_args (selector))
    result = _gst_nil_oop;
  else
    {
      SEND_MESSAGE (selector, numArgs, false);
      _gst_interpret ();
      result = POP_OOP ();
    }
  _gst_finish_execution_environment ();

  INC_RESTORE_POINTER (incPtr);
  return (result);
}

/* Use like printf */
void
_gst_msg_sendf (PTR resultPtr, 
		char *fmt, 
		...)
{
  va_list args;
  OOP selector, anArg, result;
  int numArgs;
  char *fp, *s, selectorBuf[256];
  inc_ptr incPtr;

  va_start (args, fmt);

  if (!_gst_smalltalk_initialized)
    gst_init_smalltalk ();

  incPtr = INC_SAVE_POINTER ();
  _gst_prepare_execution_environment ();

  numArgs = -1;
  for (s = selectorBuf, fp = &fmt[2]; *fp; fp++)
    {
      if (*fp == '%')
	{
	  fp++;
	  numArgs++;
	  switch (*fp)
	    {
	    case 'i':
	      PUSH_INT (va_arg (args, long));
	      break;

	    case 'f':
	      anArg = floatd_new (va_arg (args, double));
	      PUSH_OOP (anArg);
	      break;

	    case 'F':
	      anArg = floatq_new (va_arg (args, long double));
	      PUSH_OOP (anArg);
	      break;

	    case 'b':
	      if (va_arg (args, int))
		  PUSH_OOP (_gst_true_oop);
	      else
		PUSH_OOP (_gst_false_oop);
	      break;

	    case 'c':
	      anArg = CHAR_OOP_AT ((char) va_arg (args, int));
	      PUSH_OOP (anArg);
	      break;

	    case 'C':
	      anArg = COBJECT_NEW (va_arg (args, PTR));
	      PUSH_OOP (anArg);
	      break;

	    case 's':
	      anArg = _gst_string_new (va_arg (args, char *));
	      PUSH_OOP (anArg);
	      break;

	    case 'S':
	      anArg = _gst_intern_string (va_arg (args, char *));
	      PUSH_OOP (anArg);
	      break;

	    case 'o':
	      anArg = va_arg (args, OOP);
	      PUSH_OOP (anArg);
	      break;

	    case 't':		/* type string, followed by a void * */
	      {
		OOP ctype;
		ctype = _gst_type_name_to_oop (va_arg (args, char *));
		INC_ADD_OOP (ctype);

		anArg =
		  _gst_c_object_new_typed (va_arg (args, PTR), ctype);
		PUSH_OOP (anArg);
	      }
	      break;


	    case 'T':		/* existing type instance, and a void * 
				 */
	      {
		OOP ctype;
		ctype = va_arg (args, OOP);
		anArg =
		  _gst_c_object_new_typed (va_arg (args, PTR), ctype);
		PUSH_OOP (anArg);
	      }
	      break;

	    case '%':
	      *s++ = '%';
	      numArgs--;
	      break;
	    }
	}
      else if (*fp != ' ' && *fp != '\t')
	*s++ = *fp;
    }

  *s = '\0';

  selector = _gst_intern_string (selectorBuf);

  INC_ADD_OOP (selector);	/* not automatically protected! */
  /* pb: but isn't the selector a gst_symbol, and hence in the root
     set? */

  if (numArgs != _gst_selector_num_args (selector))
    result = _gst_nil_oop;
  else
    {
      SEND_MESSAGE (selector, numArgs, false);
      _gst_interpret ();
      result = POP_OOP ();
    }
  _gst_finish_execution_environment ();

  if (resultPtr)
    {
      switch (fmt[1])
	{
	case 'i':
	  *(int *) resultPtr = IS_NIL (result) ? 0 : TO_INT (result);
	  break;

	case 'c':
	  *(char *) resultPtr =
	    IS_NIL (result) ? 0 : CHAR_OOP_VALUE (result);
	  break;

	case 'C':
	  /* !!! Fix this -- it is ugly, but OS/2 compilers don't like
	     it without */
	  *(PTR *) resultPtr =
	    IS_NIL (result) ? NULL : COBJECT_VALUE (result);
	  break;

	case 's':
	  *(char **) resultPtr =
	    IS_NIL (result) ? NULL : (char *) _gst_to_cstring (result);
	  break;

	case 'b':
	  *(int *) resultPtr =
	    IS_NIL (result) ? false : (result == _gst_true_oop);
	  break;

	case 'f':
	  *(double *) resultPtr =
	    IS_NIL (result) ? 0.0 : _gst_oop_to_float (result);
	  break;

	case 'v':		/* don't care about the result */
	  break;		/* "v" for "void" */

	case '?':
	  *(long *) resultPtr = _gst_oop_to_c (result);
	  break;

	case 'o':
	default:
	  *(OOP *) resultPtr = result;
	  break;
	}
    }

  INC_RESTORE_POINTER (incPtr);
}

OOP
_gst_type_name_to_oop (char *name)
{
  OOP result;
  char buf[300];

  sprintf (buf, "^%s!", name);

  result = _gst_eval_expr (buf);
  return (result);
}


void
_gst_eval_code (char *str)
{
  if (!_gst_smalltalk_initialized)
    gst_init_smalltalk ();

  _gst_compile_code = true;
  _gst_push_cstring (str);
  _gst_parse_stream ();
  _gst_pop_stream (false);
}


OOP
_gst_eval_expr (char *str)
{
  OOP result;

  if (!_gst_smalltalk_initialized)
    gst_init_smalltalk ();

  _gst_compile_code = false;
  _gst_push_cstring (str);
  _gst_parse_stream ();
  _gst_pop_stream (false);
  result = _gst_last_returned_value;

  return (result);
}

OOP
_gst_object_alloc (OOP class_oop,
		   int size)
{
  OOP oop;

  if (CLASS_IS_INDEXABLE (class_oop))
    instantiate_with (class_oop, (unsigned long) size, &oop);
  else
    instantiate (class_oop, &oop);

  INC_ADD_OOP (oop);
  return oop;
}

int
_gst_basic_size (OOP oop)
{
  return (NUM_INDEXABLE_FIELDS (oop));
}


/***********************************************************************
 *
 *	Conversion *to* Smalltalk datatypes routines
 *
 ***********************************************************************/

OOP
_gst_class_name_to_oop (char *name)
{
  OOP result, key;

  key = _gst_symbol_to_oop (name);	/* this inits Smalltalk */
  result = dictionary_at (_gst_smalltalk_dictionary, key);
  return (result);
}


OOP
_gst_int_to_oop (long int i)
{
  if (!_gst_smalltalk_initialized)
    gst_init_smalltalk ();

  return (FROM_INT (i));
}

OOP
_gst_id_to_oop (long int i)
{
  if (!_gst_smalltalk_initialized)
    gst_init_smalltalk ();

  return (OOP_AT (i));
}

OOP
_gst_float_to_oop (double f)
{
  return (INC_ADD_OOP (floatd_new (f)));
}

OOP
_gst_bool_to_oop (int b)
{
  if (!_gst_smalltalk_initialized)
    gst_init_smalltalk ();

  if (b)
    return (_gst_true_oop);
  else
    return (_gst_false_oop);
}


OOP
_gst_char_to_oop (char c)
{
  if (!_gst_smalltalk_initialized)
    gst_init_smalltalk ();

  return (CHAR_OOP_AT (c));
}


/* !!! Add in byteArray support sometime soon */

OOP
_gst_string_to_oop (char *str)
{
  if (!_gst_smalltalk_initialized)
    gst_init_smalltalk ();

  if (str == NULL)
    return (_gst_nil_oop);
  else
    return (INC_ADD_OOP (_gst_string_new (str)));
}

OOP
_gst_byte_array_to_oop (char *str,
			int n)
{
  if (!_gst_smalltalk_initialized)
    gst_init_smalltalk ();

  if (str == NULL)
    return (_gst_nil_oop);
  else
    return (INC_ADD_OOP (_gst_byte_array_new (str, n)));
}

OOP
_gst_symbol_to_oop (char *str)
{
  if (!_gst_smalltalk_initialized)
    gst_init_smalltalk ();

  if (str == NULL)
    return (_gst_nil_oop);
  else
    /* Symbols don't get freed, so the new OOP doesn't need to be
       registered */
    return (_gst_intern_string (str));
}

OOP
_gst_c_object_to_oop (PTR co)
{
  if (!_gst_smalltalk_initialized)
    gst_init_smalltalk ();

  if (co == NULL)
    return (_gst_nil_oop);
  else
    return (INC_ADD_OOP (COBJECT_NEW (co)));
}

void
_gst_set_c_object (OOP oop, PTR co)
{
  if (!_gst_smalltalk_initialized)
    gst_init_smalltalk ();

  SET_COBJECT_VALUE(oop, co);
}

OOP
_gst_c_object_to_typed_oop (PTR co,
			    OOP typeOOP)
{
  if (!_gst_smalltalk_initialized)
    gst_init_smalltalk ();

  if (co == NULL)
    return (_gst_nil_oop);
  else
    return (INC_ADD_OOP (_gst_c_object_new_typed (co, typeOOP)));
}


/***********************************************************************
 *
 *	Conversion *from* Smalltalk datatypes routines
 *
 ***********************************************************************/

/* ### need a type inquiry routine */

long
_gst_oop_to_c (OOP oop)
{
  if (!_gst_smalltalk_initialized)
    gst_init_smalltalk ();

  if (IS_INT (oop))
    return (TO_INT (oop));

  else if (OOP_CLASS (oop) == _gst_true_class
	   || OOP_CLASS (oop) == _gst_false_class)
    return (oop == _gst_true_oop);

  else if (OOP_CLASS (oop) == _gst_char_class)
    return (CHAR_OOP_VALUE (oop));

  else if (IS_NIL (oop))
    return (0);

  else if (is_a_kind_of (OOP_CLASS (oop), _gst_c_object_class))
    return ((long) COBJECT_VALUE (oop));

  else
    return (0);
}

long
_gst_oop_to_int (OOP oop)
{
  if (!_gst_smalltalk_initialized)
    gst_init_smalltalk ();

  return (TO_INT (oop));
}

long
_gst_oop_to_id (OOP oop)
{
  if (!_gst_smalltalk_initialized)
    gst_init_smalltalk ();

  return (OOP_INDEX (oop));
}

double
_gst_oop_to_float (OOP oop)
{
  if (!_gst_smalltalk_initialized)
    gst_init_smalltalk ();

  if (IS_CLASS (oop, _gst_floatd_class))
    return (FLOATD_OOP_VALUE (oop));
  else if (IS_CLASS (oop, _gst_floate_class))
    return (FLOATE_OOP_VALUE (oop));
  else if (IS_CLASS (oop, _gst_floatq_class))
    return (FLOATQ_OOP_VALUE (oop));
  else
    return 0.0 / 0.0;
}

int
_gst_oop_to_bool (OOP oop)
{
  if (!_gst_smalltalk_initialized)
    gst_init_smalltalk ();

  return (oop == _gst_true_oop);
}

char
_gst_oop_to_char (OOP oop)
{
  if (!_gst_smalltalk_initialized)
    gst_init_smalltalk ();

  return (CHAR_OOP_VALUE (oop));
}

char *
_gst_oopto_string (OOP oop)
{
  if (!_gst_smalltalk_initialized)
    gst_init_smalltalk ();

  if (IS_NIL (oop))
    return (NULL);
  else
    return ((char *) _gst_to_cstring (oop));
}

char *
_gst_oop_to_byte_array (OOP oop)
{
  if (!_gst_smalltalk_initialized)
    gst_init_smalltalk ();

  if (IS_NIL (oop))
    return (NULL);
  else
    return ((char *) _gst_to_byte_array (oop));
}

PTR
_gst_oop_to_cobject (OOP oop)
{
  if (!_gst_smalltalk_initialized)
    gst_init_smalltalk ();

  if (IS_NIL (oop))
    return (NULL);
  else
    return (COBJECT_VALUE (oop));
}



/***********************************************************************
 *
 *	Registry bookkeeping routines
 *
 ***********************************************************************/

OOP
_gst_register_oop (OOP oop)
{
  rb_node_t **p = (rb_node_t **) &oop_registry_root;
  oop_registry *node;
  oop_registry *entry = NULL;

  if (!oop)
    return (oop);

  while (*p)
    {
      entry = (oop_registry *) *p;

      if (oop < entry->oop)
	p = &(*p)->rb_left;
      else if (oop > entry->oop)
	p = &(*p)->rb_right;
      else
	{
	  entry->usage++;
	  return (oop);
	}
    }

  node = (oop_registry *) xmalloc(sizeof(oop_registry));
  node->rb.rb_parent = (rb_node_t *) entry;
  node->rb.rb_left = node->rb.rb_right = NULL;
  node->usage = 1;
  node->oop = oop;
  *p = &(node->rb);

  rb_rebalance(&node->rb, (rb_node_t **) &oop_registry_root);
  return (oop);
}

void
_gst_unregister_oop (OOP oop)
{
  oop_registry *entry = oop_registry_root;

  /* Speed things up, this will never be in the registry (but we allow
     it to simplify client code). */
  if (!oop)
    return;

  while (entry)
    {
      if (entry->oop == oop) 
	{
	  if (!--entry->usage)
	    {
	      rb_erase (&entry->rb, (rb_node_t **) &oop_registry_root);
	      xfree (entry);
	    }
	  break;
	}
      
      entry = (oop_registry *) 
	(oop < entry->oop ? entry->rb.rb_left : entry->rb.rb_right);
    }
}


void
_gst_register_oop_array (OOP **first, OOP **last)
{
  rb_node_t **p = (rb_node_t **) &oop_array_registry_root;
  oop_array_registry *node;
  oop_array_registry *entry = NULL;

  while (*p)
    {
      entry = (oop_array_registry *) *p;

      if (first < entry->first)
	p = &(*p)->rb_left;
      else if (first > entry->first)
	p = &(*p)->rb_right;
      else
	entry->usage++;
    }

  node = (oop_array_registry *) xmalloc(sizeof(oop_array_registry));
  node->rb.rb_parent = (rb_node_t *) entry;
  node->rb.rb_left = node->rb.rb_right = NULL;
  node->usage = 1;
  node->first = first;
  node->last = last;
  *p = &(node->rb);

  rb_rebalance(&node->rb, (rb_node_t **) &oop_array_registry_root);
}

void
_gst_unregister_oop_array (OOP **first)
{
  oop_array_registry *entry = oop_array_registry_root;

  while (entry)
    {
      if (entry->first == first) 
	{
	  if (!--entry->usage)
	    {
	      rb_erase (&entry->rb, (rb_node_t **) &oop_array_registry_root);
	      xfree (entry);
	    }
	  break;
	}
      
      entry = (oop_array_registry *) 
	(first < entry->first ? entry->rb.rb_left : entry->rb.rb_right);
    }
}


void
_gst_mark_registered_oops (void)
{
  rb_node_t *node;
  rb_traverse_t t;

  /* Walk the OOP registry... */
  for (node = rb_first(&(oop_registry_root->rb), &t); 
       node; node = rb_next(&t))
    {
      oop_registry *k = (oop_registry *) node;
      MAYBE_MARK_OOP (k->oop);
    }

  /* ...and then the OOP-array registry. */
  for (node = rb_first(&(oop_array_registry_root->rb), &t); 
       node; node = rb_next(&t))
    {
      oop_array_registry *k = (oop_array_registry *) node;

      /* Dereference the pointers in the tree to obtain where the array
	 lies. */
      OOP *first = *(k->first);
      OOP *last = *(k->last);
      MARK_OOP_RANGE (first, last);
    }
}

void
_gst_init_vmproxy (void)
{
  gst_interpreter_proxy.nilOOP = _gst_nil_oop;
  gst_interpreter_proxy.trueOOP = _gst_true_oop;
  gst_interpreter_proxy.falseOOP = _gst_false_oop;
}
