/******************************** -*- C -*- ****************************
 *
 *	sbrk-like behavior for separate mmap'ed regions
 *
 *
 ***********************************************************************/

/***********************************************************************
 *
 * Copyright 2000, 2001, 2002 Software Foundation, Inc.
 * Written by Paolo Bonzini (redisorganization of GNU mmalloc).
 *
 * This file is part of GNU Smalltalk.
 *
 * GNU Smalltalk is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2, or (at your option) any later 
 * version.
 * 
 * GNU Smalltalk is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * GNU Smalltalk; see the file COPYING.  If not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
 ***********************************************************************/


#include "gst.h"
#include "gstpriv.h"

#include <signal.h>
#include <setjmp.h>
#include <sys/types.h>
#include <errno.h>
#include <fcntl.h>		/* dpx/2 needs this after sys/types.h */
#include <string.h>

#ifdef WIN32
#include <windows.h>
#else
#include <sys/mman.h>
#endif

#ifdef HAVE_UNISTD_H
#  include <unistd.h>
#endif

#ifdef HAVE_STDDEF_H
#  include <stddef.h>
#endif

#ifdef HAVE_LIMITS_H
#  include <limits.h>
#else
#  ifndef CHAR_BIT
#    define CHAR_BIT 8
#  endif
#endif

/* Dirk Sondermann (d.sondermann@freenet.de) says some of the possible
   positions for the heap are not good.  Define this to try them.  */
/* #define TRY_DANGEROUS_POSITIONS */

/* Internal structure that defines the format of the heap descriptor.
   This gets written to the base address of the region that we are
   managing. */
struct heap
{
  size_t areasize;

  /* The base address of the memory region for this malloc heap.  This
     is the location where the bookkeeping data for mmap and for malloc
     begins. */
  char *base;

  /* The current location in the memory region for this malloc heap
     which represents the end of memory in use. */
  char *breakval;

  /* The end of the current memory region for this malloc heap.  This
     is the first location past the end of mapped memory. */
  char *top;
};

/* This is the internal function for heap_sbrk which receives a struct
   heap instead of the pointer to the base location available to
   clients. */
static PTR heap_sbrk_internal (struct heap *hdp,
			       int size);

/* Cache pagesize-1 for the current host machine.  Note that if the
   host does not readily provide a getpagesize() function, we need to
   emulate it elsewhere, not clutter up this file with lots of kluges
   to try to figure it out. */
static size_t pageround, pagesize;
#ifndef HAVE_GETPAGESIZE
extern int getpagesize ();
#endif

#define PAGE_ALIGN(addr) ((PTR) (((long)(addr) + pageround) & ~pageround))

/* We allocate extra pages for the heap descriptor and answer an
   address that is HEAP_DELTA bytes past the actual beginning of the
   allocation. */
#define HEAP_DELTA	 ((long) PAGE_ALIGN(sizeof (struct heap)))

/* Wrappers for Windows and POSIX */
#ifdef WIN32

#define _heap_reserve(hdp, size) \
  ((hdp) -> base = VirtualAlloc(NULL, (size), MEM_RESERVE, PAGE_NOACCESS))

#define _heap_release(hdp) \
  VirtualFree((hdp) -> base, (hdp) -> areasize, MEM_RELEASE)

#define _heap_map_in(hdp, size) \
  VirtualAlloc ((hdp) -> top, (size), MEM_COMMIT, PAGE_EXECUTE_READWRITE)

#define _heap_map_out(hdp, size) \
  VirtualFree(PAGE_ALIGN ((hdp) -> breakval), (size), MEM_DECOMMIT)

#else /* !WIN32 */

static PTR find_heap_base (void);

#ifndef MAP_FAILED
#define MAP_FAILED ((char *) -1)
#endif

static char *baseaddr;

#define _heap_reserve(hdp, size) \
  ( ((hdp) -> base = baseaddr, baseaddr += (size), 1) )

#define _heap_release(hdp) \
  ( ((baseaddr == (hdp) -> base + (hdp) -> areasize) \
      ? baseaddr = (hdp) -> base : 0))

#if defined(MAP_ANONYMOUS) && !defined(MAP_ANON)
#define MAP_ANON MAP_ANONYMOUS
#endif

#ifdef MAP_ANON
#define anon_mmap(where, size, flags)				\
  mmap ((where), (size), PROT_READ | PROT_WRITE | PROT_EXEC,	\
	(flags) | MAP_PRIVATE | MAP_ANON, -1, 0)

#else /* !MAP_ANON */

/* Open file descriptor for the file to which malloc heaps is mapped
 * (/dev/zero). */
static int fd;

#define anon_mmap(where, size, flags)				\
  mmap ((where), (size), PROT_READ | PROT_WRITE | PROT_EXEC,	\
	(flags) | MAP_PRIVATE, fd, 0)

#endif /* !MAP_ANON */

/* Map in enough more memory to satisfy the request. */
#define _heap_map_in(hdp, size) \
  anon_mmap ((hdp) -> top, (size), MAP_FIXED)

#define _heap_map_out(hdp, size) \
  munmap (PAGE_ALIGN ((hdp) -> breakval), (size))

#endif /* !WIN32 */


heap
_gst_heap_create (int size)
{
  struct heap mtemp;
  struct heap *hdp;
  char *mbase;

  if (!pageround)
    {
      pagesize = getpagesize ();
      pageround = pagesize - 1;

#ifndef WIN32
#ifndef MAP_ANON
      /* Open the file descriptor we need for anonymous mmaps */
      fd = open ("/dev/zero", O_RDWR);
#endif
      baseaddr = find_heap_base ();
#endif
    }

  /* We start off with the heap descriptor allocated on the stack,
     until we build it up enough to call heap_sbrk_internal() to
     allocate the first page of the region and copy it there.  Ensure
     that it is zero'd and then initialize the fields that we know
     values for. */

  hdp = &mtemp;
  memzero ((char *) hdp, sizeof (mtemp));
  hdp->areasize = size;

  if (!_heap_reserve (hdp, size))
    {
      errno = ENOMEM;
      return (NULL);
    }

  hdp->breakval = hdp->top = hdp->base;

  /* Now try to map in the first page, copy the heap descriptor
     structure there, and arrange to return a pointer to this new copy. 
     If the mapping fails, then close the file descriptor if it was
     opened by us, and arrange to return a NULL. */

  if ((mbase = heap_sbrk_internal (hdp, sizeof (mtemp))) != NULL)
    {
      memcpy (mbase, hdp, sizeof (mtemp));
      hdp = (struct heap *) (mbase + HEAP_DELTA);
    }
  else
    {
      _heap_release (hdp);
      hdp = NULL;
    }

  return ((heap) hdp);
}

heap
_gst_heap_destroy (heap hd)
{
  struct heap mtemp;

  if (hd != NULL)
    {
      mtemp = *(struct heap *) (hd - HEAP_DELTA);

      /* Now unmap all the pages associated with this region by asking
         for a negative increment equal to the current size of the
         region. */

      if ((heap_sbrk_internal (&mtemp, mtemp.base - mtemp.top)) == NULL)
	/* Update the original heap descriptor with any changes */
	*(struct heap *) (hd - HEAP_DELTA) = mtemp;
      else
	{
	  _heap_release (&mtemp);
	  hd = NULL;
	}
    }

  return (hd);
}

PTR
_gst_heap_sbrk (heap hd,
		size_t size)
{
  struct heap *hdp;

  hdp = (struct heap *) (hd - HEAP_DELTA);
  return heap_sbrk_internal (hdp, size);
}

PTR
heap_sbrk_internal (struct heap * hdp,
		    int size)
{
  char *result = NULL;
  size_t mapbytes;		/* Number of bytes to map */
  char *moveto;			/* Address where we wish to move "break 
				   value" to */
  char *mapto;			/* Address we actually mapped to */

  if (size == 0)
    /* Just return the current "break" value. */
    result = hdp->breakval;

  else if (size < 0)
    {
      /* We are deallocating memory.  If the amount requested would
         cause us to try to deallocate back past the base of the mmap'd 
         region then do nothing, and return NULL.  Otherwise,
         deallocate the memory and return the old break value. */
      if (hdp->breakval + size >= hdp->base)
	{
	  result = (PTR) hdp->breakval;
	  hdp->breakval += size;
	  moveto = PAGE_ALIGN (hdp->breakval);
	  _heap_map_out (hdp, (size_t) (hdp->top - moveto));
	  hdp->top = moveto;
	}
    }
  else if (hdp->breakval + size > hdp->top)
    {
      moveto = PAGE_ALIGN (hdp->breakval + size);
      mapbytes = moveto - hdp->top;
      mapto = _heap_map_in (hdp, mapbytes);
      if (mapto == hdp->top)
	{
	  hdp->top = moveto;
	  result = (PTR) hdp->breakval;
	  hdp->breakval += size;
	}
      else
	errno = ENOMEM;
    }
  else
    {
      result = (PTR) hdp->breakval;
      hdp->breakval += size;
    }

  return (result);
}

/* This is hairy and a hack.  We have to find a place for our heaps... */

#ifndef WIN32

/* This signal handler is used if it is the only means to decide if
   a page is mapped into memory.  We intercept SIGSEGV and SIGBUS
   and we decide that the heap can be allocated at a given location
   only if we receive one of the signals.

   The reason why msync is used if available, is that it makes
   debugging more painful to have SIGSEGV's as part of the normal
   initialization sequence. */

#ifndef HAVE_MSYNC
static RETSIGTYPE not_mapped (int sig);
static jmp_buf already_mapped;

RETSIGTYPE
not_mapped (sig)
     int sig;
{
  _gst_set_signal_handler (sig, not_mapped);
  longjmp (already_mapped, 1);
}
#endif

PTR
find_heap_base (void)
{
  volatile char *mmapGuess, *higher, *lower;	/* reference addresses */
  volatile char *first = NULL, *second;	/* probed addresses */
  volatile const int *step;

  static const int steps[] = {
    true, 256, 256, 0,		/* try 256 Mb after the higher address */
    true, 128, 256, 0,		/* try 128 Mb after the higher address */
    true, 64, 256, 0,		/* try 64 Mb after the higher address */
#if TRY_DANGEROUS_POSITIONS
    true, -256, -256, 640,	/* try 256 Mb before the higher address 
				 */
    true, -128, -256, 512,	/* try 128 Mb before the higher address 
				 */
    true, -64, -256, 448,	/* try 64 Mb before the higher address */
#endif
    false, 256, 256, 512,	/* try 256 Mb after the lower address */
    false, 128, 256, 384,	/* try 128 Mb after the lower address */
    false, 64, 256, 320,	/* try 64 Mb after the lower address */
    true, 32, 256, 0,		/* try 32 Mb after the higher address */
    true, 32, 128, 0,		/* again, for a smaller heap */
#if TRY_DANGEROUS_POSITIONS
    true, -32, -256, 416,	/* try 32 Mb before the higher address */
    true, -32, -128, 288,	/* again, for a smaller heap */
#endif
    false, 64, 128, 192,	/* this has a smaller heap too */
    -1
  };

#ifndef HAVE_MSYNC
  volatile int test, *testPtr = &test;
  volatile SigHandler oldSegvHandler, oldBusHandler;

  /* Placate GNU C's warnings about clobbered variables */
  testPtr = (volatile int *) &higher;
  testPtr = (volatile int *) &lower;
  testPtr = (volatile int *) &first;
  testPtr = (volatile int *) &step;
  testPtr = (volatile int *) &test;

  /* Install custom signal handlers to detect failed memory accesses */
#ifdef SIGBUS
  oldBusHandler = _gst_set_signal_handler (SIGBUS, not_mapped);
#endif
  oldSegvHandler = _gst_set_signal_handler (SIGSEGV, not_mapped);
#endif /* !HAVE_MSYNC */

  /* Get two reference addresses which we will base ourselves on */
  mmapGuess = higher = anon_mmap (NULL, pagesize, 0);
  lower = sbrk (0);

  if (higher < lower)
    {
      higher = lower;
      lower = mmapGuess;
    }

  /* Now try each of the possibilities... */
  for (step = steps; *step > -1; step += 4)
    {
      if (higher > lower + (step[3] << 20))
	{
	  first =
	    PAGE_ALIGN ((step[0] ? higher : lower) + (step[1] << 20));
	  second = PAGE_ALIGN (first + (step[2] << 20));

	  /* Try reading the two locations */
#ifdef HAVE_MSYNC
	  if (msync ((char *) first, pagesize, MS_SYNC) == 0)
	    continue;

	  if (msync ((char *) second, pagesize, MS_SYNC) == 0)
	    continue;
#else
	  if (setjmp (already_mapped) == 0)
	    {
	      *testPtr = *first;
	      continue;
	    }
	  if (setjmp (already_mapped) == 0)
	    {
	      *testPtr = *second;
	      continue;
	    }
#endif

	  /* Try mmap-ing them */
	  if (anon_mmap ((char *) first, pagesize, MAP_FIXED) ==
	      MAP_FAILED)
	    continue;

	  munmap ((char *) first, pagesize);
	  if (anon_mmap ((char *) second, pagesize, MAP_FIXED) ==
	      MAP_FAILED)
	    continue;

	  /* Were not readable and could be mmap-ed.  We're done. */
	  munmap ((char *) second, pagesize);
	  break;
	}
    }

  /* Restore things... */
#ifndef HAVE_MSYNC
#ifdef SIGBUS
  _gst_set_signal_handler (SIGBUS, oldBusHandler);
#endif
  _gst_set_signal_handler (SIGSEGV, oldSegvHandler);
#endif /* !HAVE_MSYNC */

  munmap ((char *) mmapGuess, pagesize);

  if (first == NULL || *step == -1)
    {
      _gst_errorf ("Could not find a place for the Smalltalk heaps.");
      return (NULL);
    }
  else
    return (PTR) (first);
}
#endif
