/******************************** -*- C -*- ****************************
 *
 *	The Smalltalk Virtual Machine in itself.
 *
 *	This, together with oop.c, is the `bridge' between Smalltalk and
 *	the underlying machine
 *
 *
 ***********************************************************************/

/***********************************************************************
 *
 * Copyright 1988,89,90,91,92,94,95,99,2000,2001,2002
 * Free Software Foundation, Inc.
 * Written by Steve Byrne.
 *
 * This file is part of GNU Smalltalk.
 *
 * GNU Smalltalk is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2, or (at your option) any later
 * version.
 *
 * GNU Smalltalk is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * GNU Smalltalk; see the file COPYING.	 If not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 ***********************************************************************/

#include "gst.h"
#include "gstpriv.h"

#include <errno.h>
#include <math.h>
#include <stdio.h>
#include <signal.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <setjmp.h>

#ifdef STDC_HEADERS
#include <string.h>
#include <stdlib.h>
#endif /* STDC_HEADERS */
#ifdef HAVE_IO_H
#include <io.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif



/* The local regs concept hopes, by caching the values of IP and SP in
   local register variables, to increase performance.  You only need
   to export the variables when calling out to routines that might
   change them and that create objects.  This is because creating
   objects may trigger the GC, which can change the values of IP and
   SP (since they point into the object space).  It's easy to deal
   with that, however, it's just a matter of importing and exporting
   the registers at the correct places: for example stack operations
   are innocuous, while message sends can result in a GC (because
   stack chunks are exhausted or because primitive #new is invoked),
   so they export the registers and import them (possibly with their
   value changed by the GC) after the send.  I'm leaving the code to
   deal with them as local registers conditionally compiled in so that
   you can disable it easily if necessary; however this seems quite
   improbable except for debugging purposes. */
#define LOCAL_REGS

#ifdef HAVE_GOTO_VOID_P

/* New-style dispatching obtains a 30/40% speed boost over standard
   switch-statement dispatching.  It works by replacing the switch
   statement with a `computed goto'.  The checks for asynchronous
   events (semaphore signals, timers, etc.) are skipped because code
   that handles these events modifies the interpreter's status so that
   the next `computed goto' jumps to where the event is passed to
   Smalltalk.  The default is to use new-style dispatch with GNU C;
   comment this to use old-style dispatching with GCC too. */
#define USE_GCC_DISPATCH
#endif /* HAVE_GOTO_VOID_P */

/* By "hard wiring" the definitions of the special math operators
   (bytecodes 176-191), we get a performance boost of more than 50%.
   Yes, it means that we cannot redefine + et al for SmallInteger and
   Float, but I think the trade is worth it.  Besides, the Blue Book
   does it. */
#define OPEN_CODE_MATH

/* jump lookahead uses special machinery after open-coded boolean
   selectors (<, =, >, <=, >=, ~= for Integers and Floats; ==, IS_NIL
   and notNil for all objects) that executes conditional jump
   bytecodes without pushing and popping the result of the comparison.
   This catches the common "a < b ifTrue: [ ... ]" and "[ ... a < b ]
   whileTrue: [ ... ]" patterns, as well as code generated for
   #to:do:, #timesRepeat: and #to:by:do:.  Jump lookahead only
   applies to the GCC-based bytecode interpreter (that is,
   USE_GCC_DISPATCH defined, USE_JIT_TRANSLATION not defined). */
#define JUMP_LOOKAHEAD

/* Pipelining uses separate fetch-decode-execute stages, which is a
   nice choice for VLIW machines.  It also enables more aggressive
   caching of global variables.  It is currently enabled for the IA-64
   only, because it is a win only where we would have had lots of
   unused instruction scheduling slots and an awful lot of registers.
   It disables jump lookahead because jump lookahead interferes with
   the pre-decoding mechanism of the pipelined interpreter.  */
#if defined(USE_GCC_DISPATCH) && REG_AVAILABILITY == 3
#define PIPELINING
#endif

/* Answer the quantum assigned to each Smalltalk process (in
   milliseconds) before it is preempted.  Setting this to zero
   disables preemption until gst_processor_scheduler>>#timeSlice: is
   set. */
#define DEFAULT_PREEMPTION_TIMESLICE 40

/* Used to handle the case when the user types a ^C while executing
   callout code.  If STACK_JMPBUFS is defined, the C callout primitive
   saves the old jmp_buf on the stacks and uses a new one; if it is
   not defined, a ^C will immediately jump outside ALL the callouts.
   The former behavior is usually cleaner, so I define it. */
#define STACK_JMPBUFS

/* This symbol does not control execution speed.  Instead, it causes
   SEND_MESSAGE to print every message that is ever sent in the
   SmallInteger(Object)>>#printString form.  Can be useful to find out
   the last method sent before an error, if the context stack is
   trashed when the debugger gets control and printing a backtrace is
   impossible. */
/* #define DEBUG_CODE_FLOW */

/* The method cache is a hash table used to cache the most commonly
   used methods.  Its size is determined by this preprocessor
   constant.  It is currently 2048, a mostly random choice; you can
   modify it, but be sure it is a power of two.  Additionally,
   separately from this, the interpreter caches the last primitive
   numbers used for sends of #at:, #at:put: and #size, in an attempt
   to speed up these messages for Arrays, Strings, and ByteArrays. */
#define METHOD_CACHE_SIZE		(1 << 11)

/* Length of the queue of Semaphores to be signaled at the next
   sequence point. */
#define ASYNC_QUEUE_SIZE		100


/* CompiledMethod cache (see descriptions in interp-bc.inl and
   interp-jit.inl) */
typedef struct method_cache_entry
{
  OOP selectorOOP;
  OOP startingClassOOP;
  OOP methodOOP;
  OOP methodClassOOP;
  method_header methodHeader;
#ifdef USE_JIT_TRANSLATION
  OOP receiverClass;
  PTR nativeCode;
  PTR dummy;			/* 32 bytes are usually a sweet spot */
#endif
}
method_cache_entry;

typedef struct async_queue_entry
{
  OOP sem;
  mst_Boolean unregister;
}
async_queue_entry;

/* This type hides the implementation of the jmp_buf type.  The
   original reason was that if jmp_buf is implemented as an array,
   taking its address caused the compiler to warn about taking the
   address of an array, and there was no way to tell at compile time
   whether this is going to be a problem.  Now I built the jmpBuf
   chain in the structure (a provision for possible future changes) so
   the structure is needed anyway.  */
typedef struct interp_jmp_buf
{
  jmp_buf jmpBuf;
  struct interp_jmp_buf *old;
}
interp_jmp_buf;



/* If this is true, for each byte code that is executed, we print on
   stdout the byte index within the current gst_compiled_method and a
   decoded interpretation of the byte code. */
mst_Boolean _gst_execution_tracing;

/* When this is true, and an interrupt occurs (such as SIGSEGV),
   Smalltalk will terminate itself by making a core dump (normally it
   produces a backtrace). */
mst_Boolean _gst_make_core_file = false;

/* When true, this indicates that there is no top level loop for
   control to return to, so it causes the system to exit. */
mst_Boolean _gst_non_interactive = true;

/* The table of functions that implement the primitives. */
static prim_table_entry _gst_primitive_table[NUM_PRIMITIVES];

/* Some performance counters from the interpreter: these
   count the number of special returns. */
unsigned long _gst_literal_returns, _gst_inst_var_returns;
unsigned long _gst_self_returns;

/* The number of primitives executed. */
unsigned long _gst_primitives_executed;

/* The number of bytecodes executed. */
unsigned long _gst_bytecode_counter;

/* The number of method cache misses */
unsigned long _gst_cache_misses;

/* The number of cache lookups - either hits or misses */
unsigned long _gst_sample_counter;

/* More performance counters when the machine is instrumented. */
#ifdef PROFBLOCK
struct prof_struct ps;
static long bytecodes[256];
static long primitives[1024];
#endif


#ifdef USE_JIT_TRANSLATION
#undef PROFBLOCK
#define method_base		0
char *native_ip;
#else /* plain bytecode interpreter */
static ip_type method_base;
#endif

/* The virtual machine's stack and instruction pointers. */
OOP *sp;
ip_type ip;

/* Global state
   The following variables constitute the interpreter's state:

   ip -- the real memory address of the next byte code to be executed.

   sp -- the real memory address of the stack that's stored in the
   currently executing block or method context.

   _gst_this_method -- a gst_compiled_method or gst_compiled_block
   that is the currently executing method.

   _gst_this_context_oop -- a gst_block_context or gst_method_context
   that indicates the context that the interpreter is currently
   running in.

   _gst_temporaries -- physical address of the base of the method
   temporary variables.  Typically a small number of bytes (multiple
   of 4 since it points to OOPs) lower than sp.

   _gst_literals -- physical address of the base of the method
   literals.

   _gst_self -- an OOP that is the current receiver of the current
   message.  */

OOP *_gst_temporaries, *_gst_literals;
OOP _gst_self;
OOP _gst_this_context_oop;
OOP _gst_this_method;

/* Answer whether we are in the interpreter or in application code. */
static mst_Boolean in_interpreter = false;

/* CompiledMethod cache which memoizes the methods and some more
   information for each class->selector pairs. */
static method_cache_entry method_cache[METHOD_CACHE_SIZE] CACHELINE_ALIGNED;

/* The number of the last primitive called. */
static int last_primitive;

/* A special cache that tries to skip method lookup when #at:, #at:put
   and #size are implemented by a class through a primitive, and is
   repeatedly sent to instances of the same class.  Since this is a
   mini-inline cache it makes no sense when JIT translation is
   enabled. */
#ifndef USE_JIT_TRANSLATION
static OOP at_cache_class, at_put_cache_class, size_cache_class;
static int at_cache_prim, at_put_cache_prim, size_cache_prim;
#endif

/* Queue for async (outside the interpreter) semaphore signals */
static volatile int async_queue_index;
static volatile async_queue_entry queued_async_signals[ASYNC_QUEUE_SIZE]
  CACHELINE_ALIGNED;

/* When not NULL, this causes the byte code interpreter to immediately
   send the message whose selector is here to the current stack
   top. */
volatile char *_gst_abort_execution = NULL;

/* Set to true when some special action must be done at the next
   sequence point. */
volatile mst_Boolean _gst_except_flag;

/* Set to non-nil if a process must preempt the current one. */
static volatile OOP switch_to_process;

/* Set to true if it is time to switch process in a round-robin
   time-sharing fashion. */
static volatile mst_Boolean time_to_preempt;

/* When this is true, it means that the system is executing external C
   code, which can be used by the ^C handler to know whether it should
   longjmp to the end of the C callout primitive in
   _gst_execute_primitive_operation. */
static mst_Boolean in_ccode = false;

/* Used to bail out of a C callout and back to the interpreter. */
static interp_jmp_buf base_callout_jmp_buf, *c_callout_jmp_buf =
  &base_callout_jmp_buf;

/* when this flag is on and execution tracing is in effect, the top of
   the stack is printed as well as the byte code */
static mst_Boolean verbose_exec_tracing = false;

/* Locates in the ProcessorScheduler's process lists and returns the
   highest priority process different from the current process.  */
static OOP highest_priority_process (void) FN_PURE;

/* Remove the head of the given list (a Semaphore is a subclass of
   LinkedList) and answer it. */
static OOP remove_first_link (OOP semaphoreOOP);

/* Add PROCESSOOP as the head of the given list (a Semaphore is a
   subclass of LinkedList) and answer it. */
static void add_first_link (OOP semaphoreOOP,
			   OOP processOOP);

/* Add PROCESSOOP as the tail of the given list (a Semaphore is a
   subclass of LinkedList) and answer it. */
static void add_last_link (OOP semaphoreOOP,
			   OOP processOOP);

/* Answer the highest priority process different from the current one.
   Answer nil if there is no other process than the current one.
   Create a new process that terminates execution if there is no
   runnable process (which should never be because there is always the
   idle process). */
static OOP next_scheduled_process (void);

/* Sets flags so that the interpreter starts returning immediately from
   whatever byte codes it's executing.  It returns via a normal message
   send of the unary selector MSG, so that the world is in a consistent 
   state when it's done. */
static void stop_executing (char *msg);

/* Set a timer at the end of which we'll preempt the current process. */
static void set_preemption_timer (void);

/* Put the given process to sleep by rotating the list of processes for
   PROCESSOOP's priority (i.e. it was the head of the list and becomes
   the tail). */
static void sleep_process (OOP processOOP);

/* Sets flags so that the interpreter switches to PROCESSOOP at the
   next sequence point.  Unless PROCESSOOP is already active, in which
   case nothing happens, the process is made the head of the list of
   processes for PROCESSOOP's priority. */
static void activate_process (OOP processOOP);

/* Save the virtual machine's state into the suspended Process and
   ContextPart objects, and load them from NEWPROCESS and from
   NEWPROCESS's suspendedContext.  The Processor (the only instance
   of ProcessorScheduler is also updated accordingly.  */
static void change_process_context (OOP newProcess);

/* Mark the semaphores attached to the process system (asynchronous
   events, the signal queue, and if any the process which we'll
   switch to at the next sequence point).  */
static void mark_semaphore_oops (void);

/* Signal the given SEMAPHOREOOP and if processes were queued on it
   resume the one that has waited for the longest time and is still
   alive. */
static void sync_signal (OOP semaphoreOOP);

/* Resume execution of PROCESSOOP.  If it must preempt the currently
   running process, put to sleep the active process and activate 
   PROCESSOOP instead; if it must not, make it the head of the
   process list for its priority, so that it will be picked once
   higher priority processes all go to sleep. 

   If PROCESSOOP is terminating, answer false.  If PROCESSOOP can
   be restarted or at least put back in the process list for its
   priority, answer true.  */
static mst_Boolean resume_process (OOP processOOP);

/* Answer whether PROCESSOOP is ready to execute (neither terminating,
   nor suspended, nor waiting on a semaphore).  */
static mst_Boolean is_process_ready (OOP processOOP) FN_PURE;

/* Answer whether any processes are queued in the PROCESSLISTOOP
   (which can be a LinkedList or a Semaphore).  */
static inline mst_Boolean is_empty (OOP processListOOP) FN_PURE;

/* Answer whether the processs is terminating, that is, it does not
   have an execution context to resume execution from. */
static inline mst_Boolean is_process_terminating (OOP processOOP) FN_PURE;

/* Answer the active process (that is, the executing process or, if
   any, the process that is scheduled to start execution at the next
   sequence point. */
static inline OOP get_active_process (void) FN_PURE;

/* Create a new Semaphore OOP and return it. */
static inline OOP semaphore_new (void);

/* This is the equivalent of SEND_MESSAGE, but is for blocks.  The
   block context that is to the the receiver of the "value" message
   should be the NUMARGS-th into the stack.  SP is set to the top of
   the arguments in the block context, which have been copied out of
   the caller's context. 

   On failure return true, on success (i.e. if NUMARGS matches what
   the BlockClosure says) return false. */
static mst_Boolean send_block_value (int numArgs);

/* This is a kind of simplified _gst_send_message_internal that,
   instead of setting up a context for a particular receiver, stores
   information on the lookup into METHODDATA.  Unlike
   _gst_send_message_internal, this function is generic and valid for
   both the interpreter and the JIT compiler. */
static mst_Boolean lookup_method (OOP sendSelector,
				  method_cache_entry * methodData,
				  int sendArgs,
				  OOP method_class);

/* This tenures context objects from the stack to the context pools
   (see below for a description). */
static void empty_context_stack (void);

/* This allocates a new context pool, eventually triggering a GC once
   no more pools are available.  */
static gst_method_context alloc_new_chunk (void);

/* This allocates a context object which is SIZE words big from
   a pool, allocating one if the current pool is full. */
static inline gst_method_context alloc_stack_context (int size);

/* This frees the most recently allocated stack from the current
   context pool.  It is called when unwinding. */
static inline void dealloc_stack_context (gst_context_part context);

/* This allocates a new context of SIZE, prepares an OOP for it
   (taking it from the LIFO_CONTEXTS arrays that is defined below),
   and pops SENDARGS arguments from the current context.  Only the
   parentContext field of the newly-allocated context is initialized,
   because the other fields can be desumed from the execution state:
   these other fields instead are filled in the parent context since
   the execution state will soon be overwritten. */
static inline gst_method_context activate_new_context (int size,
						       int sendArgs);

/* Push the ARGS topmost words below the stack pointer, and then TEMPS
   nil objects, onto the stack of CONTEXT.  */
static inline void prepare_context (gst_context_part context,
				    int args,
				    int temps);

/* Return from the current context and restore the virtual machine's
   status (ip, sp, _gst_this_method, _gst_self, ...). */
static void unwind_context (void);

/* Used to help minimize the number of primitives used to control the
   various debugging flags, this routine maps the variable's INDEX to the
   address of a boolean debug flag, which it returns. */
static inline mst_Boolean *bool_addr_index (int index) FN_PURE;

/* Check whether it is true that sending SENDSELECTOR to RECEIVER
   accepts NUMARGS arguments.  Note that the RECEIVER is only used to
   do a quick check in the method cache before examining the selector
   itself; in other words, true is returned even if a message is not
   understood by the receiver, provided that NUMARGS matches the
   number of arguments expected by the selector (1 if binary, else the
   number of colons).  If you don't know a receiver you can just pass
   _gst_nil_oop or directly call _gst_selector_num_args. */
static inline mst_Boolean check_send_correctness (OOP receiver,
						  OOP sendSelector,
						  int numArgs) FN_PURE;

/* Unwind the contexts up until the caller of the method that
   created the block context, no matter how many levels of message
   sending are between where we currently are and the context that
   we are going to return from.

   Note that unwind_method is only called inside `dirty' (or `full')
   block closures, hence the context we return from can be found by
   following OUTERCONTEXT links starting from the currently executing
   context, and until we reach a MethodContext. */
static mst_Boolean unwind_method (void);

/* Unwind up to context returnContextOOP, carefully examining the
   method call stack.  That is, we examine each context and we only
   deallocate those that, during their execution, did not create a
   block context; the others need to be marked as returned.  We
   continue up the call chain until we finally reach methodContextOOP
   or an unwind method.  In this case the non-unwind contexts between
   the unwind method and the returnContextOOP must be removed from the
   chain. */
static void unwind_to (OOP returnContextOOP);

/* Arrange things so that all the non-unwinding contexts up to
   returnContextOOP aren't executed.  For block contexts this can
   be done simply by removing them from the chain, but method
   context must stay there so that we can do non-local returns
   from them!  For this reason, method contexts are flagged as
   disabled and unwind_context takes care of skipping them when
   doing a local return.  */
static void disable_non_unwind_contexts (OOP returnContextOOP);

/* Called to handle signals that are not passed to the Smalltalk
   program, such as interrupts or segmentation violation.  In the
   latter case, try to show a method invocation backtrace if possibly,
   otherwise try to show where the system was in the file it was
   processing when the error occurred. */
static RETSIGTYPE interrupt_handler (int sig);

/* Called to preempt the current process after a specified amount
   of time has been spent in the GNU Smalltalk interpreter. */
#ifdef ENABLE_PREEMPTION
static RETSIGTYPE preempt_smalltalk_process (int sig);
#endif

/* Pick a process that is the highest-priority process different from
   the currently executing one, and schedule it for execution after
   the first sequence points. */
#define ACTIVE_PROCESS_YIELD() \
  activate_process(next_scheduled_process());

/* Answer an OOP for a Smalltalk object of class Array, holding the
   different process lists for each priority. */
#define GET_PROCESS_LISTS() \
  (((gst_processor_scheduler)OOP_TO_OBJ(_gst_processor_oop))->processLists)

/* Tell the interpreter that special actions are needed as soon as a
   sequence point is reached.  */
#define SET_EXCEPT_FLAG(x) do {						\
  _gst_except_flag = (x);						\
  DO_SET_EXCEPT_FLAG(x);						\
} while(0)

#define DO_SET_EXCEPT_FLAG(x)	/* might be redefined by interp-*.inl */

/* Answer an hash value for a send of the SENDSELECTOR message, when
   the CompiledMethod is found in class METHODCLASS.  */
#define METHOD_CACHE_HASH(sendSelector, methodClass)			 \
    (( ((long)(sendSelector)) ^ ((long)(methodClass)) >> (LONG_SHIFT+1)) \
      & (METHOD_CACHE_SIZE - 1))

/* Answer whether CONTEXT is a MethodContext.  This happens whenever
   we have some SmallInteger flags (and not the pointer to the outer
   context) in the last instance variable.  */
#define CONTEXT_FLAGS(context) \
  ( ((gst_method_context)(context)) ->flags)

/* Context management
 
   The contexts make up a linked list.  Their structure is:
                                               
      +-----------------------------------+
      | parentContext			  |
      +-----------------------------------+	THESE ARE CONTEXT'S
      | misc. information		  |	FIXED INSTANCE VARIABLES
      | ...				  |
      +-----------------------------------+-------------------------------
      | args				  |
      | ...				  |	THESE ARE THE CONTEXT'S
      +-----------------------------------+	INDEXED INSTANCE VARIABLES
      | temps				  |
      | ...				  |
      +-----------------------------------+
      | stack				  |
      | ...				  |
      +-----------------------------------+
 
   The space labeled "misc. information" is initialized when
   thisContext is pushed or when the method becomes the parent context
   of a newly activated context.  It contains, among other things, the
   pointer to the CompiledMethod or CompiledBlock for the context.
   That's comparable to leaf procedure optimization in RISC
   processors.
 
   Contexts are special in that they are not created immediately in
   the main heap.  Instead they have three life phases:

   a) their OOPs are allocated on a stack, and their object data is
   allocated outside of the main heap.  This state lasts until the
   context returns (in which case the OOP can be reused) or until a
   reference to the context is made (in which case we swiftly move all
   the OOPs to the OOP table, leaving the object data outside the
   heap).

   b) their OOPs are allocated in the main OOP table, their object
   data still resides outside of the main heap.  Unlike the main heap,
   this area grows more slowly, but like the main heap, a GC is
   triggered when it's full.  Upon GC, most context objects (which are
   generated by `full' or `dirty' blocks) that could not be discarded
   when they were returned from are reclaimed, and the others are
   tenured, moving them to the main heap.

   c) their OOPs are allocated in the main OOP table, their object
   data stays in the main heap.  And in this state they will remain
   until they become garbage and are reclaimed.  */

/* I made CHUNK_SIZE a nice power of two.  Allocate 64KB at a time,
   never use more than 3 MB; anyway these are here so behavior can be
   fine tuned.  MAX_LIFO_DEPTH is enough to have room for an entire
   stack chunk and avoid testing for overflows in lifo_contexts. */
#define CHUNK_SIZE			16384
#define MAX_CHUNKS_IN_MEMORY		48
#define MAX_LIFO_DEPTH			(CHUNK_SIZE / CTX_SIZE(0))

/* CHUNK points to an item of CHUNKS.  CUR_CHUNK_BEGIN is equal
   to *CHUNK (i.e. points to the base of the current chunk) and
   CUR_CHUNK_END is equal to CUR_CHUNK_BEGIN + CHUNK_SIZE. */
static gst_context_part cur_chunk_begin = NULL, cur_chunk_end = NULL;
static gst_context_part chunks[MAX_CHUNKS_IN_MEMORY] CACHELINE_ALIGNED;
static gst_context_part *chunk = chunks - 1;

/* These are used for OOP's allocated in a LIFO manner.  A context is
   kept on this stack as long as it generates only clean blocks, as
   long as it resides in the same chunk as the newest object created,
   and as long as no context switches happen since the time the
   process was created.  FREE_LIFO_CONTEXT points to just after the
   top of the stack. */
static struct OOP lifo_contexts[MAX_LIFO_DEPTH] CACHELINE_ALIGNED;
static OOP free_lifo_context = lifo_contexts;

/* Include `plug-in' modules for the appropriate interpreter.
 
   A plug-in must define
   - _gst_send_message_internal
   - _gst_send_method
   - send_block_value
   - _gst_interpret
   - GET_CONTEXT_IP
   - SET_THIS_METHOD
   - _gst_validate_method_cache_entries
   - any others that are needed by the particular implementation (e.g.
     lookup_native_ip for the JIT plugin)
 
   They are included rather than linked to for speed (they need access
   to lots of inlines and macros).  The same holds for prims.inl,
   which defines _gst_execute_primitive_operation.  */

#include "prims.inl"

#ifdef USE_JIT_TRANSLATION
#include "interp-jit.inl"
#else
#include "interp-bc.inl"
#endif



void
_gst_empty_context_pool (void)
{
  cur_chunk_begin = cur_chunk_end = NULL;
  chunk = chunks - 1;
}

void
empty_context_stack (void)
{
  OOP contextOOP, last, oop;
  gst_method_context context;

  /* printf("[[[[ Gosh, not lifo anymore! (free = %p, base = %p)\n",
     free_lifo_context, lifo_contexts); */
  if (free_lifo_context != lifo_contexts)
    {
      free_lifo_context = contextOOP = lifo_contexts;
      last = _gst_this_context_oop;
      context = (gst_method_context) OOP_TO_OBJ (contextOOP);

      for (;;)
	{
	  oop = alloc_oop (context);
	  oop->flags = F_POOLED | F_CONTEXT;

	  /* Fill the object's uninitialized fields. */
	  context->objClass = CONTEXT_FLAGS (context) & MCF_IS_METHOD_CONTEXT
	    ? _gst_method_context_class : _gst_block_context_class;

#ifndef USE_JIT_TRANSLATION
	  /* This field is unused without the JIT compiler, but it must 
	     be initialized when a context becomes a fully formed
	     Smalltalk object.  We do that here.  Note that we need the 
	     field so that the same image is usable with or without the 
	     JIT compiler. */
	  context->native_ip = DUMMY_NATIVE_IP;
#endif

	  /* The last context is not referenced anywhere, so we're done 
	     with it. */
	  if (contextOOP++ == last)
	    break;

	  /* Else we redirect its sender field to the main OOP table */
	  context = (gst_method_context) OOP_TO_OBJ (contextOOP);
	  context->parentContext = oop;
	}
      _gst_this_context_oop = oop;
    }
  else
    {
      if (IS_NIL (_gst_this_context_oop))
	return;

      context = (gst_method_context) OOP_TO_OBJ (_gst_this_context_oop);
    }

  /* When a context gets out of the context stack it must be a fully
     formed Smalltalk object.  These fields were left uninitialized in
     _gst_send_message_internal and send_block_value -- set them here. */
  context->method = _gst_this_method;
  context->receiver = _gst_self;
  context->spOffset = FROM_INT (sp - context->contextStack);
  context->ipOffset = FROM_INT (ip - method_base);

  /* Even if the JIT is active, the current context might have no
     attached native_ip -- in fact it has one only if we are being
     called from activate_new_context -- so we have to `invent'
     one. We test for a valid native_ip first, though; this test must
     have no false positives, i.e. it won't ever overwrite a valid
     native_ip, and won't leave a bogus OOP for the native_ip. */
  if (!IS_INT (context->native_ip))
    context->native_ip = DUMMY_NATIVE_IP;
}

gst_method_context
alloc_new_chunk (void)
{
  gst_method_context newContext;

  if UNCOMMON (++chunk >= &chunks[MAX_CHUNKS_IN_MEMORY])
    {
      /* No more chunks available - GC */
      _gst_minor_scavenge ();
      ++chunk;
    }
  else
    empty_context_stack ();

  if UNCOMMON (!(newContext = (gst_method_context) * chunk))
    {
      /* Allocate memory only the first time we're using the chunk.
         _gst_empty_context_pool resets the status but doesn't free
         the memory. */
      cur_chunk_begin = *chunk = (gst_context_part)
	xmalloc (SIZE_TO_BYTES (CHUNK_SIZE));

      newContext = (gst_method_context) cur_chunk_begin;
    }
  else
    cur_chunk_begin = *chunk;

  cur_chunk_end = (gst_context_part) (
    ((char *) cur_chunk_begin) + SIZE_TO_BYTES(CHUNK_SIZE));
  return (newContext);
}

gst_method_context
alloc_stack_context (int size)
{
  gst_method_context newContext;

#ifdef PROFBLOCK
  ps.numMethodAllocs++;
#endif

  size = CTX_SIZE (size);
  newContext = (gst_method_context) cur_chunk_begin;
  cur_chunk_begin += size;
  if UNCOMMON (cur_chunk_begin >= cur_chunk_end)
    {
      /* Not enough room in the current chunk */
      newContext = alloc_new_chunk ();
      cur_chunk_begin += size;
    }

  newContext->objSize = FROM_INT (size);
  return (newContext);
}

gst_method_context
activate_new_context (int size,
		      int sendArgs)
{
  OOP oop;
  gst_method_context newContext;
  gst_method_context thisContext;

#ifndef OPTIMIZE
  if (IS_NIL (_gst_this_context_oop))
    {
      printf ("Somebody forgot _gst_prepare_execution_environment!\n");
      _gst_debug ();
    }
#endif

  /* We cannot overflow lifo_contexts, because it is designed to
     contain all of the contexts in a chunk, and we empty lifo_contexts 
     when we exhaust a chunk.  So we can get the oop the easy way. */
  newContext = alloc_stack_context (size);
  oop = free_lifo_context++;

  /* printf("[[[[ Context (size %d) allocated at %p (oop = %p)\n",
     size, newContext, oop); */
  SET_OOP_OBJECT (oop, newContext);
  newContext->parentContext = _gst_this_context_oop;

  /* save old context information */
  /* leave sp pointing to receiver, which is replaced on return with
     value */
  thisContext = (gst_method_context) OOP_TO_OBJ (_gst_this_context_oop);
  thisContext->method = _gst_this_method;
  thisContext->receiver = _gst_self;
  thisContext->spOffset =
    FROM_INT ((sp - thisContext->contextStack) - sendArgs);
  thisContext->ipOffset = FROM_INT (ip - method_base);

  _gst_this_context_oop = oop;

  return (newContext);
}

void
dealloc_stack_context (gst_context_part context)
{
#ifndef OPTIMIZE
  if (free_lifo_context == lifo_contexts
      || (OOP_TO_OBJ (free_lifo_context - 1) != (mst_Object) context))
    {
      _gst_errorf ("Deallocating a non-LIFO context!!!");
      _gst_debug ();
    }
#endif

  cur_chunk_begin = context;
  free_lifo_context--;

#ifdef PROFBLOCK
  ps.numMethodFrees++;
#endif
}

void
prepare_context (gst_context_part context,
		 int args,
		 int temps)
{
  REGISTER (1, OOP * mySP);
  _gst_temporaries = mySP = context->contextStack;
  if (args)
    {
      REGISTER (2, unsigned long num);
      REGISTER (3, OOP * src);
      num = args;
      src = &sp[1 - num];

#define UNROLL_OP(n) mySP[n] = src[n]
#define UNROLL_ADV(n) mySP += n, src += n
      UNROLL_BY_8 (num);
#undef UNROLL_OP
#undef UNROLL_ADV

      mySP += num;
    }
  mySP = nil_fill (mySP, temps);
  sp = --mySP;
}

mst_Boolean
lookup_method (OOP sendSelector,
	       method_cache_entry * methodData,
	       int sendArgs,
	       OOP method_class)
{
  inc_ptr inc;
  long i;
  OOP argsArrayOOP, receiverClass;
  mst_Object argsArray;

#ifdef PROFBLOCK
  if (methodData->selectorOOP != NULL)
    ps.numCacheCollisions++;
#endif

  receiverClass = method_class;
  for (; !IS_NIL (method_class);
       method_class = SUPERCLASS (method_class))
    {
      OOP methodOOP =
	_gst_find_class_method (method_class, sendSelector);
      if (!IS_NIL (methodOOP))
	{
	  methodData->startingClassOOP = receiverClass;
	  methodData->selectorOOP = sendSelector;
	  methodData->methodOOP = methodOOP;
	  methodData->methodClassOOP = method_class;
	  methodData->methodHeader = GET_METHOD_HEADER (methodOOP);
	  _gst_cache_misses++;
	  return (true);
	}
    }


  inc = INC_SAVE_POINTER ();
  argsArray = new_instance_with (_gst_array_class, sendArgs, &argsArrayOOP);
  INC_ADD_OOP (argsArrayOOP);
  for (i = 0; i < sendArgs; i++)
    argsArray->data[i] = STACK_AT (sendArgs - i - 1);

  POP_N_OOPS (sendArgs);
  PUSH_OOP (_gst_message_new_args (sendSelector, argsArrayOOP));
  INC_RESTORE_POINTER (inc);
  return (false);
}

mst_Boolean
check_send_correctness (OOP receiver,
			OOP sendSelector,
			int numArgs)
{
  long hashIndex;
  method_cache_entry *methodData;
  OOP receiverClass;

  receiverClass =
    IS_INT (receiver) ? _gst_small_integer_class : OOP_CLASS (receiver);
  hashIndex = METHOD_CACHE_HASH (sendSelector, receiverClass);
  methodData = &method_cache[hashIndex];

  if (methodData->selectorOOP == sendSelector
      && methodData->startingClassOOP == receiverClass)
    return (methodData->methodHeader.numArgs == numArgs);
  else
    return (_gst_selector_num_args (sendSelector) == numArgs);
}

void
unwind_context (void)
{
  gst_method_context oldContext, newContext;
  OOP oldContextOOP, newContextOOP;
  int numLifoContexts;

#ifdef PROFBLOCK
  ps.stack_depth--;
#endif
  newContextOOP = _gst_this_context_oop;
  newContext = (gst_method_context) OOP_TO_OBJ (newContextOOP);
  numLifoContexts = free_lifo_context - lifo_contexts;

  do
    {
      oldContextOOP = newContextOOP;
      oldContext = newContext;

      /* Descend in the chain... */
      newContextOOP = oldContext->parentContext;

      if COMMON (numLifoContexts > 0)
	{
	  dealloc_stack_context ((gst_context_part) oldContext);
	  numLifoContexts--;
	}

      else
	/* This context cannot be deallocated in a LIFO way.  We must
	   keep it around so that the blocks it created can reference
	   arguments and temporaries in it. Method contexts, however,
	   need to be marked as non-returnable so that attempts to
	   return from them to an undefined place will lose; doing
	   that for block contexts too, we skip a test and are also
	   able to garbage collect more context objects.  */
	oldContext->parentContext = _gst_nil_oop;

      newContext = (gst_method_context) OOP_TO_OBJ (newContextOOP);
    }
  while UNCOMMON (CONTEXT_FLAGS (newContext) 
		  == (MCF_IS_METHOD_CONTEXT | MCF_IS_DISABLED_CONTEXT));

  /* Clear the bit so that we return here just once.
     This makes this absurd snippet work:

	^[ [ 12 ] ensure: [ ^34 ] ] ensure: [ 56 ]!

     If it were not for this statement, the inner #ensure:
     would resume after the ^34 block exited, and would answer
     12 (the result of the evaluation of the receiver of the
     inner #ensure:).  

     HACK ALERT!!  This is actually valid only for method contexts
     but I carefully put the modified bits in the low bits so that
     they are already zero for block contexts.  */
  CONTEXT_FLAGS (newContext) &= ~(MCF_IS_DISABLED_CONTEXT |
				  MCF_IS_UNWIND_CONTEXT);

  _gst_this_context_oop = newContextOOP;
  _gst_temporaries = newContext->contextStack;
  sp = newContext->contextStack + TO_INT (newContext->spOffset);
  _gst_self = newContext->receiver;

  SET_THIS_METHOD (newContext->method, GET_CONTEXT_IP (newContext));
}



mst_Boolean
unwind_method (void)
{
  OOP oldContextOOP, newContextOOP;
  gst_block_context newContext;

  /* We're executing in a block context and an explicit return is
     encountered.  This means that we are to return from the caller of
     the method that created the block context, no matter how many
     levels of message sending are between where we currently are and
     our parent method context. */

  newContext = (gst_block_context) OOP_TO_OBJ (_gst_this_context_oop);
  do
    {
      newContextOOP = newContext->outerContext;
      newContext = (gst_block_context) OOP_TO_OBJ (newContextOOP);
    }
  while UNCOMMON (!(CONTEXT_FLAGS (newContext) & MCF_IS_METHOD_CONTEXT));

  /* test for block return in a dead method */
  if UNCOMMON (IS_NIL (newContext->parentContext))
    {
      /* We are to create a reference to thisContext, so empty the
         stack. */
      empty_context_stack ();
      oldContextOOP = _gst_this_context_oop;

      /* Just unwind to the caller, and prepare to send a message to
         the context */
      unwind_context ();
      SET_STACKTOP (oldContextOOP);

      return (false);
    }

  unwind_to (newContext->parentContext);
  return (true);
}


void
unwind_to (OOP returnContextOOP)
{
  OOP oldContextOOP, newContextOOP;
  gst_method_context oldContext, newContext;

  empty_context_stack ();

  newContextOOP = _gst_this_context_oop;
  newContext = (gst_method_context) OOP_TO_OBJ (newContextOOP);

  while (newContextOOP != returnContextOOP)
    {
#ifdef PROFBLOCK
      ps.stack_depth--;
#endif
      oldContextOOP = newContextOOP;
      oldContext = newContext;

      /* Descend in the chain... */
      newContextOOP = oldContext->parentContext;
      newContext = (gst_method_context) OOP_TO_OBJ (newContextOOP);

      /* Check if we got to an unwinding context (#ensure:). */
      if UNCOMMON (CONTEXT_FLAGS (newContext) & MCF_IS_UNWIND_CONTEXT)
        {
	  _gst_this_context_oop = oldContextOOP;

	  /* _gst_this_context_oop is the context above the
	     one we return to.   We only unwind up to the #ensure:
	     context.  */
	  disable_non_unwind_contexts (returnContextOOP);

	  unwind_context ();
	  return;
	}

      /* This context cannot be deallocated in a LIFO way.  We must
         keep it around so that the blocks it created can reference
         arguments and temporaries in it. Method contexts, however,
         need to be marked as non-returnable so that attempts to
         return from them to an undefined place will lose; doing
         that for block contexts too, we skip a test and are also
         able to garbage collect more context objects.  */
      oldContext->parentContext = _gst_nil_oop;
    }

  /* Clear the bit so that we return here just once.
     This makes this absurd snippet work:

        ^[ [ 12 ] ensure: [ ^34 ] ] ensure: [ 56 ]!

     If it were not for this statement, the inner #ensure:
     would resume after the ^34 block exited, and would answer
     12 (the result of the evaluation of the receiver of the
     inner #ensure:).

     HACK ALERT!!  This is actually valid only for method contexts
     but I carefully put the modified bits in the low bits so that
     they are already zero for block contexts.  */
  CONTEXT_FLAGS (newContext) &= ~(MCF_IS_DISABLED_CONTEXT |
                                  MCF_IS_UNWIND_CONTEXT);

  _gst_this_context_oop = newContextOOP;
  _gst_temporaries = newContext->contextStack;
  sp = newContext->contextStack + TO_INT (newContext->spOffset);
  _gst_self = newContext->receiver;

  SET_THIS_METHOD (newContext->method, GET_CONTEXT_IP (newContext));
}

void
disable_non_unwind_contexts (OOP returnContextOOP)
{
  OOP oldContextOOP, newContextOOP, *chain;
  gst_method_context oldContext, newContext;

  newContextOOP = _gst_this_context_oop;
  newContext = (gst_method_context) OOP_TO_OBJ (newContextOOP);
  chain = &newContext->parentContext;

  for (;;)
    {
      oldContextOOP = newContextOOP;
      oldContext = newContext;

      /* Descend in the chain... */
      newContextOOP = oldContext->parentContext;
      newContext = (gst_method_context) OOP_TO_OBJ (newContextOOP);

      if (!(CONTEXT_FLAGS (oldContext) & MCF_IS_METHOD_CONTEXT))
        /* This context cannot be deallocated in a LIFO way.  Setting
	   its parent context field to nil makes us able to garbage
	   collect more context objects.  */
        oldContext->parentContext = _gst_nil_oop;

      if (newContextOOP == returnContextOOP)
	{
	  *chain = newContextOOP;
	  chain = &newContext->parentContext;
	  break;
	}

      if (CONTEXT_FLAGS (newContext) & MCF_IS_METHOD_CONTEXT)
	{
	  CONTEXT_FLAGS (newContext) |= MCF_IS_DISABLED_CONTEXT;
	  *chain = newContextOOP;
	  chain = &newContext->parentContext;
	}
    }

  /* Skip any disabled methods.  */
  while UNCOMMON (CONTEXT_FLAGS (newContext)
                  == (MCF_IS_METHOD_CONTEXT | MCF_IS_DISABLED_CONTEXT))
    {
      oldContextOOP = newContextOOP;
      oldContext = newContext;

      /* Descend in the chain... */
      newContextOOP = oldContext->parentContext;
      newContext = (gst_method_context) OOP_TO_OBJ (newContextOOP);

      /* This context cannot be deallocated in a LIFO way.  We must
         keep it around so that the blocks it created can reference
         arguments and temporaries in it. Method contexts, however,
         need to be marked as non-returnable so that attempts to
         return from them to an undefined place will lose; doing
         that for block contexts too, we skip a test and are also
         able to garbage collect more context objects.  */
      oldContext->parentContext = _gst_nil_oop;
    }

  *chain = newContext->parentContext;
}


void
change_process_context (OOP newProcess)
{
  gst_method_context thisContext;
  OOP processOOP;
  gst_process process;
  gst_processor_scheduler processor;

  switch_to_process = _gst_nil_oop;

  /* save old context information */
  if (ip)
    empty_context_stack ();

  processor = (gst_processor_scheduler) OOP_TO_OBJ (_gst_processor_oop);
  processOOP = processor->activeProcess;
  if (processOOP != newProcess && !is_process_terminating (processOOP))
    {
      process = (gst_process) OOP_TO_OBJ (processOOP);
      process->suspendedContext = _gst_this_context_oop;
    }

  processor->activeProcess = newProcess;
  process = (gst_process) OOP_TO_OBJ (newProcess);

  _gst_this_context_oop = process->suspendedContext;
  thisContext = (gst_method_context) OOP_TO_OBJ (_gst_this_context_oop);

  SET_THIS_METHOD (thisContext->method, GET_CONTEXT_IP (thisContext));
  sp = thisContext->contextStack + TO_INT (thisContext->spOffset);

  _gst_temporaries = thisContext->contextStack;
  _gst_self = thisContext->receiver;
}



OOP
get_active_process (void)
{
  gst_processor_scheduler processor;

  if (!IS_NIL (switch_to_process))
    return (switch_to_process);
  else
    {
      processor =
	(gst_processor_scheduler) OOP_TO_OBJ (_gst_processor_oop);
      return (processor->activeProcess);
    }
}

void
add_first_link (OOP semaphoreOOP,
		OOP processOOP)
{
  gst_semaphore sem;
  gst_process process, lastProcess;
  OOP lastProcessOOP;

  process = (gst_process) OOP_TO_OBJ (processOOP);
  if (!IS_NIL (process->myList))
    {
      sem = (gst_semaphore) OOP_TO_OBJ (process->myList);
      if (sem->firstLink == processOOP)
	{
	  sem->firstLink = process->nextLink;
	  if (sem->lastLink == processOOP)
	    {
	      /* It was the only process in the list */
	      sem->lastLink = _gst_nil_oop;
	    }
	}
      else if (sem->lastLink == processOOP)
	{
	  /* Find the new last link */
	  lastProcessOOP = sem->firstLink;
	  lastProcess = (gst_process) OOP_TO_OBJ (lastProcessOOP);
	  while (lastProcess->nextLink != processOOP)
	    {
	      lastProcessOOP = lastProcess->nextLink;
	      lastProcess = (gst_process) OOP_TO_OBJ (lastProcessOOP);
	    }
	  sem->lastLink = lastProcessOOP;
	  lastProcess->nextLink = _gst_nil_oop;
	}
    }

  sem = (gst_semaphore) OOP_TO_OBJ (semaphoreOOP);
  process->myList = semaphoreOOP;
  process->nextLink = sem->firstLink;

  sem->firstLink = processOOP;
  if (IS_NIL (sem->lastLink))
    sem->lastLink = processOOP;
}

void
add_last_link (OOP semaphoreOOP,
	       OOP processOOP)
{
  gst_semaphore sem;
  gst_process process, lastProcess;
  OOP lastProcessOOP;

  process = (gst_process) OOP_TO_OBJ (processOOP);
  if (!IS_NIL (process->myList))
    {
      sem = (gst_semaphore) OOP_TO_OBJ (process->myList);
      if (sem->firstLink == processOOP)
	{
	  sem->firstLink = process->nextLink;
	  if (sem->lastLink == processOOP)
	    {
	      /* It was the only process in the list */
	      sem->lastLink = _gst_nil_oop;
	    }
	}
      else if (sem->lastLink == processOOP)
	{
	  /* Find the new last link */
	  lastProcessOOP = sem->firstLink;
	  lastProcess = (gst_process) OOP_TO_OBJ (lastProcessOOP);
	  while (lastProcess->nextLink != processOOP)
	    {
	      lastProcessOOP = lastProcess->nextLink;
	      lastProcess = (gst_process) OOP_TO_OBJ (lastProcessOOP);
	    }
	  sem->lastLink = lastProcessOOP;
	  lastProcess->nextLink = _gst_nil_oop;
	}
    }

  sem = (gst_semaphore) OOP_TO_OBJ (semaphoreOOP);
  process->myList = semaphoreOOP;
  process->nextLink = _gst_nil_oop;

  if (IS_NIL (sem->lastLink))
    sem->firstLink = sem->lastLink = processOOP;
  else
    {
      lastProcessOOP = sem->lastLink;
      lastProcess = (gst_process) OOP_TO_OBJ (lastProcessOOP);
      lastProcess->nextLink = processOOP;
      sem->lastLink = processOOP;
    }
}

mst_Boolean
is_empty (OOP processListOOP)
{
  gst_semaphore processList;

  processList = (gst_semaphore) OOP_TO_OBJ (processListOOP);
  return (IS_NIL (processList->firstLink));
}


void
sync_signal (OOP semaphoreOOP)
{
  gst_semaphore sem;
  OOP freedOOP;

  sem = (gst_semaphore) OOP_TO_OBJ (semaphoreOOP);
  do
    {
      if (is_empty (semaphoreOOP))
	{
	  sem->signals = INCR_INT (sem->signals);
	  break;
	}
      freedOOP = remove_first_link (semaphoreOOP);

      /* If they terminated this process, well, try another */
    }
  while (!resume_process (freedOOP));
}

void
_gst_async_signal (OOP semaphoreOOP)
{
  int_state oldSigMask;

  oldSigMask = _gst_disable_interrupts ();	/* block out
						   everything! */
  queued_async_signals[async_queue_index].sem = semaphoreOOP;
  queued_async_signals[async_queue_index++].unregister = false;
  
  SET_EXCEPT_FLAG (true);
  _gst_enable_interrupts (oldSigMask);
}

void
_gst_async_signal_and_unregister (OOP semaphoreOOP)
{
  int_state oldSigMask;

  oldSigMask = _gst_disable_interrupts ();	/* block out
						   everything! */
  queued_async_signals[async_queue_index].sem = semaphoreOOP;
  queued_async_signals[async_queue_index++].unregister = true;
  
  SET_EXCEPT_FLAG (true);
  _gst_enable_interrupts (oldSigMask);
}

void
_gst_sync_wait (OOP semaphoreOOP)
{
  gst_semaphore sem;
  int_state oldSigMask;

  oldSigMask = _gst_disable_interrupts ();	/* block out
						   everything! */
  sem = (gst_semaphore) OOP_TO_OBJ (semaphoreOOP);
  if (TO_INT (sem->signals) <= 0)
    {
      /* have to suspend, move this to the end of the list */
      add_last_link (semaphoreOOP, get_active_process ());
      ACTIVE_PROCESS_YIELD ();
    }
  else
    sem->signals = DECR_INT (sem->signals);

  _gst_enable_interrupts (oldSigMask);
}

OOP
remove_first_link (OOP semaphoreOOP)
{
  gst_semaphore sem;
  gst_process process;
  OOP processOOP;

  sem = (gst_semaphore) OOP_TO_OBJ (semaphoreOOP);
  processOOP = sem->firstLink;
  process = (gst_process) OOP_TO_OBJ (processOOP);

  sem = (gst_semaphore) OOP_TO_OBJ (semaphoreOOP);
  sem->firstLink = process->nextLink;
  if (IS_NIL (sem->firstLink))
    sem->lastLink = _gst_nil_oop;

  /* Unlink the process from any list it was in! */
  process->myList = _gst_nil_oop;
  process->nextLink = _gst_nil_oop;
  return (processOOP);
}

mst_Boolean
resume_process (OOP processOOP)
{
  int priority;
  OOP activeOOP;
  OOP processLists;
  OOP processList;
  gst_process process, active;

  activeOOP = get_active_process ();
  active = (gst_process) OOP_TO_OBJ (activeOOP);
  process = (gst_process) OOP_TO_OBJ (processOOP);

  if (process == active)
    return (true);

  if (is_process_terminating (processOOP))
    {
      /* The process was terminated - nothing to resume, fail */
      return (false);
    }

  priority = TO_INT (process->priority);
  processLists = GET_PROCESS_LISTS ();
  processList = ARRAY_AT (processLists, priority);

  if (priority >= TO_INT (active->priority))
    {
      /* We're resuming a process with a *equal or higher* priority, so sleep
         the current one and activate the new one */
      sleep_process (activeOOP);
      activate_process (processOOP);
    }
  else
    {
      /* this process has a lower priority than the active one, so the
         policy is that it doesn't preempt the currently running one.
         Anyway, it must be the first in its priority queue - so don't
         put it to sleep. */
      add_first_link (processList, processOOP);
    }

  /* printf("%#O", ((gst_process)(processOOP->object)) ->name ); */
  return (true);
}

void
activate_process (OOP processOOP)
{
  gst_process process;
  int priority;
  OOP processLists;
  OOP processList;

  if (IS_NIL (processOOP))
    return;

  if (processOOP != get_active_process ())
    {
      process = (gst_process) OOP_TO_OBJ (processOOP);
      priority = TO_INT (process->priority);
      processLists = GET_PROCESS_LISTS ();
      processList = ARRAY_AT (processLists, priority);
      add_first_link (processList, processOOP);
    }

  SET_EXCEPT_FLAG (true);
  switch_to_process = processOOP;
}

#ifdef ENABLE_PREEMPTION
RETSIGTYPE
preempt_smalltalk_process (int sig)
{
  time_to_preempt = true;
  SET_EXCEPT_FLAG (true);
}
#endif

mst_Boolean
is_process_terminating (OOP processOOP)
{
  gst_process process;

  process = (gst_process) OOP_TO_OBJ (processOOP);
  return (IS_NIL (process->suspendedContext));
}

mst_Boolean
is_process_ready (OOP processOOP)
{
  gst_process process;
  int priority;
  OOP processLists;
  OOP processList;

  process = (gst_process) OOP_TO_OBJ (processOOP);
  priority = TO_INT (process->priority);
  processLists = GET_PROCESS_LISTS ();
  processList = ARRAY_AT (processLists, priority);

  /* check if process is in the priority queue */
  return (process->myList == processList);
}

void
sleep_process (OOP processOOP)
{
  gst_process process;
  int priority;
  OOP processLists;
  OOP processList;

  process = (gst_process) OOP_TO_OBJ (processOOP);
  priority = TO_INT (process->priority);
  processLists = GET_PROCESS_LISTS ();
  processList = ARRAY_AT (processLists, priority);

  /* add process to end of priority queue */
  add_last_link (processList, processOOP);
}


OOP
highest_priority_process (void)
{
  OOP processLists, processList;
  int priority;
  OOP processOOP;

  processLists = GET_PROCESS_LISTS ();
  priority = NUM_OOPS (OOP_TO_OBJ (processLists));
  for (; priority > 0; priority--)
    {
      processList = ARRAY_AT (processLists, priority);
      if (!is_empty (processList))
	{
	  processOOP = remove_first_link (processList);
	  if (processOOP == get_active_process ())
	    {
	      /* The current process has yielded control, i.e. it has
	         been moved to the end of its list - but if there's
	         only one element it is still looks like the highest
	         one, and we must discard it */
	      /* printf("Current process discarded"); */
	      add_last_link (processList, processOOP);
	    }
	  else
	    return (processOOP);
	}
    }
  return (_gst_nil_oop);
}

OOP
next_scheduled_process (void)
{
  OOP processLists, processList;
  OOP processOOP;
  gst_processor_scheduler processor;
  gst_process process;
  gst_method_context dummyContext;
  int priority;

  processOOP = highest_priority_process ();

  if (!IS_NIL (processOOP))
    return (processOOP);

  if (is_process_ready (get_active_process ()))
    return (_gst_nil_oop);

  /* instead of returning _gst_nil_oop, let's return a newly created
     initial process and see what happens 10-Oct-93 14:17:48 -- didn't
     work -- system hung

     pb -- Let's instead return the current process, modifying it so
     that it stops the Smalltalk interpreter. */

  /* _gst_print_process_state(); */
  /* _gst_init_process_system(); *//* reset things */

  /* now make a dummy context to run with. */
  empty_context_stack ();
  dummyContext = alloc_stack_context (4);
  dummyContext->parentContext = _gst_nil_oop;
  dummyContext->method = _gst_get_termination_method ();
  dummyContext->flags = MCF_IS_METHOD_CONTEXT
	 | MCF_IS_EXECUTION_ENVIRONMENT
	 | MCF_IS_UNWIND_CONTEXT;
  dummyContext->receiver = _gst_nil_oop;
  dummyContext->ipOffset = FROM_INT (0);
  dummyContext->spOffset = FROM_INT (-1);

#ifdef USE_JIT_TRANSLATION
  dummyContext->native_ip = FROM_INT ((char *) _gst_return_from_native_code);
#else
  dummyContext->native_ip = DUMMY_NATIVE_IP;	/* See
						   empty_context_stack */
#endif

  processor = (gst_processor_scheduler) OOP_TO_OBJ (_gst_processor_oop);
  process = (gst_process) OOP_TO_OBJ (processor->activeProcess);
  priority = TO_INT (process->priority);
  processLists = GET_PROCESS_LISTS ();
  processList = ARRAY_AT (processLists, priority);

  process->suspendedContext = alloc_oop (dummyContext);
  process->myList = processList;

  /* stop_executing("noRunnableProcess"); */

  return (processor->activeProcess);
}

/* Mainly for being invoked from a debugger */
void
_gst_print_process_state (void)
{
  OOP processLists, processListOOP, processOOP;
  int priority;
  gst_semaphore processList;
  gst_process process;

  processLists = GET_PROCESS_LISTS ();
  priority = NUM_OOPS (OOP_TO_OBJ (processLists));
  for (; priority > 0; priority--)
    {
      printf ("  Priority %d: ", priority);
      processListOOP = ARRAY_AT (processLists, priority);
      processList = (gst_semaphore) OOP_TO_OBJ (processListOOP);

      printf ("First %p last %p ", processList->firstLink,
	      processList->lastLink);

      for (processOOP = processList->firstLink; !IS_NIL (processOOP);
	   processOOP = process->nextLink)
	{
	  process = (gst_process) OOP_TO_OBJ (processOOP);
	  printf ("\n    <Proc %p prio: %ld next %p context %p> ",
		  processOOP, TO_INT (process->priority),
		  process->nextLink, process->suspendedContext);
	}


      printf ("\n");
    }
}

OOP
semaphore_new (void)
{
  gst_semaphore sem;
  OOP semaphoreOOP;

  sem = (gst_semaphore) instantiate (_gst_semaphore_class, &semaphoreOOP);
  sem->signals = FROM_INT (0);

  return (semaphoreOOP);
}

/* runs before every evaluation (_gst_execute_statements) and before GC turned on.
   Note that we don't use the incubator because _gst_processor_oop is a global. */
void
_gst_init_process_system (void)
{
  OOP processListsOOP;
  int i;
  gst_processor_scheduler processor;
  gst_process initialProcess;
  OOP initialProcessOOP, initialProcessListOOP;

  processor = (gst_processor_scheduler) OOP_TO_OBJ (_gst_processor_oop);
  if (IS_NIL (processor->processLists))
    {
      mst_Object processLists;

      processLists = instantiate_with (_gst_array_class, NUM_PRIORITIES,
				       &processor->processLists);

      for (i = 0; i < NUM_PRIORITIES; i++)
	processLists->data[i] = semaphore_new ();
    }

  /* No process is active -- so highest_priority_process() need not
     worry about discarding an active process. */
  processor->activeProcess = _gst_nil_oop;

  initialProcessOOP = highest_priority_process ();
  if (IS_NIL (initialProcessOOP))
    {
      initialProcess = (gst_process)
	instantiate (_gst_process_class, &initialProcessOOP);

      initialProcess->priority = FROM_INT (4);	/* userSchedulingPriority */
      initialProcess->name = _gst_string_to_oop ("initial process");

      /* This is quite a problem. The initialProcess has undoubtedly a
         suspended context -- the #_gst_execute_statements context --
         but it hasn't been created yet. But suspendedContext must not
         be nil, otherwise change_process_context will think that it
         belongs to a terminated process. No problem, we just set it to 
         a bogus value. I chose this Integer because it is likely to
         cause a SIGSEGV/SIGBUS if change_process_context behaves
         differently from what we think -- i.e. if it uses the
         suspendedContext read from the suspended process to do
         something more interesting than comparing it to nil . */

      initialProcess->suspendedContext = FROM_INT (0);
    }

  if (IS_NIL (processor->processTimeslice))
    processor->processTimeslice =
      _gst_int_to_oop (DEFAULT_PREEMPTION_TIMESLICE);

  processListsOOP = processor->processLists;
  initialProcessListOOP = ARRAY_AT (processListsOOP, 4);
  add_first_link (initialProcessListOOP, initialProcessOOP);
  /* initialProcessOOP now is in the root set */

  processor->activeProcess = initialProcessOOP;
  switch_to_process = _gst_nil_oop;
  set_preemption_timer ();
}




mst_Boolean *
bool_addr_index (int index)
{
  switch (index)
    {
    case 0:
      return (&_gst_declare_tracing);
    case 1:
      return (&_gst_execution_tracing);
    case 2:
      return (&verbose_exec_tracing);
    case 3:
      return (&_gst_gc_message);
    default:
      return (NULL);		/* index out of range, signal the error 
				 */
    }
}



void
_gst_init_interpreter (void)
{
  unsigned int i;

#ifdef USE_JIT_TRANSLATION
  init_translator ();
  ip = 0;
#else
  ip = NULL;
#endif

  _gst_this_context_oop = _gst_nil_oop;
  async_queue_index = 0;

  for (i = 0; i < MAX_LIFO_DEPTH; i++)
    lifo_contexts[i].flags = F_POOLED | F_CONTEXT;

  _gst_init_async_events ();
  _gst_init_process_system ();
}

void
_gst_prepare_execution_environment (void)
{
  gst_method_context newContext;

  empty_context_stack ();

  /* now make a dummy context to run with */
  /* the first +1 accounts for the receiver (which must be pushed on
     this context too); the second +1 accounts for any needed extra
     space, just to be sure */
  newContext =
    alloc_stack_context (((MAX_NUM_ARGS + 1) >> DEPTH_SCALE) + 1);
  newContext->objClass = _gst_method_context_class;
  newContext->parentContext = _gst_this_context_oop;
  newContext->flags = MCF_IS_METHOD_CONTEXT
	 | MCF_IS_EXECUTION_ENVIRONMENT
	 | MCF_IS_UNWIND_CONTEXT;

#ifdef USE_JIT_TRANSLATION
  newContext->native_ip = FROM_INT ((char *) _gst_return_from_native_code);
  SET_THIS_METHOD (_gst_get_termination_method (),
		   GET_CONTEXT_IP (newContext));
#else
  newContext->native_ip = DUMMY_NATIVE_IP;	/* See
						   empty_context_stack */
  SET_THIS_METHOD (_gst_get_termination_method (), 0);
#endif

  sp = newContext->contextStack - 1;
  _gst_temporaries = newContext->contextStack;
  _gst_self = _gst_nil_oop;
  _gst_this_context_oop = alloc_oop (newContext);
  _gst_this_context_oop->flags = F_POOLED | F_CONTEXT;

  _gst_invalidate_method_cache ();
}

void
set_preemption_timer (void)
{
#ifdef ENABLE_PREEMPTION
  gst_processor_scheduler processor;
  int timeSlice;

  processor = (gst_processor_scheduler) OOP_TO_OBJ (_gst_processor_oop);
  timeSlice = TO_INT (processor->processTimeslice);

  time_to_preempt = false;
  if (timeSlice > 0)
    _gst_signal_after (timeSlice, preempt_smalltalk_process,
		       TIMER_PROCESS);
#endif
}

OOP
_gst_finish_execution_environment (void)
{
  gst_block_context oldContext;
  OOP oldContextOOP, returnedValue;

  returnedValue = STACKTOP ();
  oldContextOOP = _gst_this_context_oop;
  oldContext = (gst_block_context) OOP_TO_OBJ (oldContextOOP);
  oldContextOOP = oldContext->parentContext;

  if (free_lifo_context > lifo_contexts)
    dealloc_stack_context ((gst_context_part) oldContext);

  oldContext = (gst_block_context) OOP_TO_OBJ (oldContextOOP);
  _gst_this_context_oop = oldContextOOP;

  if (!IS_NIL (_gst_this_context_oop))
    {
      gst_method_context thisContext;

      /* restore old context information */
      thisContext =
	(gst_method_context) OOP_TO_OBJ (_gst_this_context_oop);
      SET_THIS_METHOD (thisContext->method,
		       GET_CONTEXT_IP (thisContext));
      sp = thisContext->contextStack + TO_INT (thisContext->spOffset);
      _gst_temporaries = thisContext->contextStack;
      _gst_self = thisContext->receiver;
    }
  else
    {
      /* restore dummy context information */
#if USE_JIT_TRANSLATION
      ip = 0;
#else
      ip = NULL;
#endif
      _gst_literals = _gst_temporaries = NULL;
      _gst_self = _gst_this_method = _gst_nil_oop;
    }
  return (returnedValue);
}

void
_gst_invalidate_method_cache (void)
{
  int i;

  _gst_cache_misses = _gst_sample_counter = 0;

  for (i = 0; i < METHOD_CACHE_SIZE; i++)
    {
      method_cache[i].selectorOOP = NULL;
#ifdef USE_JIT_TRANSLATION
      method_cache[i].receiverClass = NULL;
#endif
    }

#ifndef USE_JIT_TRANSLATION
  at_cache_class = at_put_cache_class = size_cache_class = NULL;
#endif
}

#ifdef PROFBLOCK
void
init_bytecode_counter (void)
{
  int i;

#ifndef CANNOT_COUNT_BYTECODES
  for (i = 0; i < 256; i++)
    bytecodes[i] = 0;
#endif
  for (i = 0; i < 1024; i++)
    primitives[i] = 0;
}

void
print_bytecode_counts (void)
{
  int i;

#ifndef CANNOT_COUNT_BYTECODES
  for (i = 0; i < 256; i++)
    {
      if (bytecodes[i])
	printf ("Byte code %d = %d\n", i, bytecodes[i]);
    }
#endif

  printf ("\n---> primitives:\n");
  for (i = 0; i < 1024; i++)
    {
      if (primitives[i])
	{
	  printf ("Primitive %d = %d\n", i, primitives[i]);
	}
    }

}
#endif /* PROFBLOCK */





void
_gst_mark_processor_registers (void)
{
  mark_semaphore_oops ();

  /* Get everything into the main OOP table first. */
  empty_context_stack ();
  MAYBE_MARK_OOP (_gst_this_context_oop);

  /* everything else is pointed to by _gst_this_context_oop, either
     directly or indirectly, or has been marked when scanning the 
     registered roots. */
}

void
_gst_fixup_object_pointers (void)
{
  gst_method_context thisContext;

  if (ip)
    {
      thisContext =
	(gst_method_context) OOP_TO_OBJ (_gst_this_context_oop);
#ifdef DEBUG_FIXUP
      fflush (stderr);
      printf
	("\nF sp %x %d    ip %x %d	_gst_this_method %x  thisContext %x",
	 sp, sp - thisContext->contextStack, ip, ip - method_base,
	 _gst_this_method->object, thisContext);
      fflush (stdout);
#endif
      thisContext->method = _gst_this_method;
      thisContext->receiver = _gst_self;
      thisContext->spOffset = FROM_INT (sp - thisContext->contextStack);
      thisContext->ipOffset = FROM_INT (ip - method_base);
    }
}

void
_gst_restore_object_pointers (void)
{
  gst_context_part thisContext;

  /* !!! The objects can move after the growing or compact phase. But,
     all this information is re-computable, so we pick up
     _gst_this_method to adjust the ip and _gst_literals accordingly,
     and we also pick up the context to adjust sp and the temps
     accordingly. */

  if (ip)
    {
      thisContext =
	(gst_context_part) OOP_TO_OBJ (_gst_this_context_oop);
      _gst_temporaries = thisContext->contextStack;

#ifndef OPTIMIZE		/* Mon Jul 3 01:21:06 1995 */
      /* these should not be necessary */
      if (_gst_this_method != thisContext->method)
	{
	  _gst_debug ();
	  printf ("$$$$$$$$$$$$$$$$$$$ GOT ONE!!!!\n");
	  printf ("this method %O\n", _gst_this_method);
	  printf ("this context %O\n", thisContext->receiver);
	  _gst_this_method = thisContext->method;
	}
      if (_gst_self != thisContext->receiver)
	{
	  _gst_debug ();
	  printf ("$$$$$$$$$$$$$$$$$$$ GOT ONE!!!!\n");
	  printf ("self %O\n", _gst_self);
	  printf ("this context %O\n", thisContext->receiver);
	  _gst_self = thisContext->receiver;
	}
#endif /* OPTIMIZE Mon Jul 3 01:21:06 1995 */

      SET_THIS_METHOD (_gst_this_method, GET_CONTEXT_IP (thisContext));
      sp = TO_INT (thisContext->spOffset) + thisContext->contextStack;

#ifdef DEBUG_FIXUP
      fflush (stderr);
      printf
	("\nR sp %x %d    ip %x %d	_gst_this_method %x  thisContext %x\n",
	 sp, sp - thisContext->contextStack, ip, ip - method_base,
	 _gst_this_method->object, thisContext);
      fflush (stdout);
#endif
    }

  SET_EXCEPT_FLAG (true);	/* force to import registers */
}

void
mark_semaphore_oops (void)
{
  int i;
  int_state oldSigMask;

  oldSigMask = _gst_disable_interrupts ();	/* block out
						   everything! */

  for (i = 0; i < async_queue_index; i++)
    MAYBE_MARK_OOP (queued_async_signals[i].sem);

  /* there does seem to be a window where this is not valid */
  MAYBE_MARK_OOP (switch_to_process);

  _gst_enable_interrupts (oldSigMask);
}




void
_gst_init_signals (void)
{
  if (!_gst_make_core_file)
    {
#ifdef SIGBUS
      _gst_set_signal_handler (SIGBUS, interrupt_handler);
#endif
      _gst_set_signal_handler (SIGSEGV, interrupt_handler);
    }
  _gst_set_signal_handler (SIGINT, interrupt_handler);
  _gst_set_signal_handler (SIGFPE, interrupt_handler);
  _gst_set_signal_handler (SIGUSR1, interrupt_handler);
}


void
stop_executing (char *msg)
{
  _gst_abort_execution = msg;
  SET_EXCEPT_FLAG (true);
  if (in_ccode)
    longjmp (c_callout_jmp_buf->jmpBuf, 1);	/* throw out from C
						   code */
}


RETSIGTYPE
interrupt_handler (int sig)
{
  _gst_set_signal_handler (sig, SIG_DFL);

  switch (sig)
    {
    case SIGUSR1:
      if (!_gst_gc_running && ip)
	_gst_show_backtrace();

      _gst_set_signal_handler (sig, interrupt_handler);
      return;

    case SIGFPE:
      if (in_interpreter)
	_gst_set_signal_handler (sig, interrupt_handler);
      else
	kill (getpid (), sig);

      return;

    case SIGINT:
      if (_gst_non_interactive)
	{
	  printf ("Signal %d, exiting...\n", sig);
	  if (!_gst_gc_running && ip)
	    _gst_show_backtrace ();

	  exit (1);
	}
      else
	{
	  _gst_set_signal_handler (sig, interrupt_handler);
	  stop_executing ("userInterrupt");
	  return;
	}

#ifdef SIGBUS
    case SIGBUS:
      _gst_errorf ("Bus error");
      break;
#endif

    case SIGSEGV:
      _gst_errorf ("Segmentation violation");
      break;

    default:
      _gst_errorf ("Unknown signal caught: %d", sig);
    }

  _gst_debug ();
  if (!_gst_gc_running && ip && in_interpreter)
    {
      /* Avoid recursive signals */
      in_interpreter = false;
      _gst_show_backtrace ();
    }
  else
    _gst_errorf ("Error occurred while not in byte code interpreter!!");

  kill (getpid (), sig);
}

void
_gst_show_backtrace (void)
{
  OOP contextOOP;
  gst_method_context context;
  gst_compiled_block block;
  gst_compiled_method method;
  gst_method_info methodInfo;

  empty_context_stack ();
  for (contextOOP = _gst_this_context_oop; !IS_NIL (contextOOP);
       contextOOP = context->parentContext)
    {
      context = (gst_method_context) OOP_TO_OBJ (contextOOP);
      if (CONTEXT_FLAGS (context) 
	  == (MCF_IS_METHOD_CONTEXT | MCF_IS_DISABLED_CONTEXT))
	continue;

#if 0
      printf ("(OOP %p)", context->method);
#endif
      if (CONTEXT_FLAGS (context) & MCF_IS_METHOD_CONTEXT)
	{
	  OOP receiver, receiverClass;

          if (CONTEXT_FLAGS (context) & MCF_IS_EXECUTION_ENVIRONMENT)
	    {
	      if (IS_NIL(context->parentContext))
	        printf ("<bottom>\n");
	      else
	        printf ("<call-in>\n");
	      continue;
	    }

          if (CONTEXT_FLAGS (context) & MCF_IS_UNWIND_CONTEXT)
	    printf ("<unwind> ");

	  /* a method context */
	  method = (gst_compiled_method) OOP_TO_OBJ (context->method);
	  methodInfo =
	    (gst_method_info) OOP_TO_OBJ (method->descriptor);
	  receiver = context->receiver;
	  if (IS_INT (receiver))
	    receiverClass = _gst_small_integer_class;

	  else
	    receiverClass = OOP_CLASS (receiver);

	  if (receiverClass == methodInfo->class)
	    printf ("%O", receiverClass);
	  else
	    printf ("%O(%O)", receiverClass, methodInfo->class);
	}
      else
	{
	  /* a block context */
	  block = (gst_compiled_block) OOP_TO_OBJ (context->method);
	  method = (gst_compiled_method) OOP_TO_OBJ (block->method);
	  methodInfo =
	    (gst_method_info) OOP_TO_OBJ (method->descriptor);

	  printf ("[] in %O", methodInfo->class);
	}
      printf (">>%O\n", methodInfo->selector);
    }
}

void
_gst_show_stack_contents (void)
{
  gst_method_context context;
  OOP *walk;
  mst_Boolean first;

  if (IS_NIL (_gst_this_context_oop))
    return;

  context = (gst_method_context) OOP_TO_OBJ (_gst_this_context_oop);
  for (first = true, walk = context->contextStack;
       walk <= sp; first = false, walk++)
    {
      if (!first)
	printf (", ");

      printf ("%O", *walk);
    }
  printf ("\n\n");
}
