/******************************** -*- C -*- ****************************
 *
 *	Object Table maintenance module.
 *
 *
 ***********************************************************************/

/***********************************************************************
 *
 * Copyright 1988,89,90,91,92,94,95,99,2000,2001,2002
 * Free Software Foundation, Inc.
 * Written by Steve Byrne.
 *
 * This file is part of GNU Smalltalk.
 *
 * GNU Smalltalk is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2, or (at your option) any later 
 * version.
 * 
 * GNU Smalltalk is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * GNU Smalltalk; see the file COPYING.  If not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  
 *
 ***********************************************************************/


#include "gst.h"
#include "gstpriv.h"

#include <stdio.h>
#if STDC_HEADERS
#include <stdlib.h>
#include <string.h>
#endif /* STDC_HEADERS */

/* Size of the object semi spaces, in bytes */
#define	K		1024

/* you can increase this value if you need more space, and it won't hurt
 * performance *if* your machine has enough physical memory (otherwise, you
 * thrash the pager) */
#define INIT_MEM_SPACE_SIZE           (256 * K * SIZEOF_LONG)
#define NEW_GENERATION_SIZE           (INIT_MEM_SPACE_SIZE / 8)


#define INIT_NUM_INCUBATOR_OOPS   50	/* SWAG */
#define INCUBATOR_CHUNK_SIZE	  20	/* SWAG */


/* Define this flag to turn on debugging code for OOP table management */
/* #define OOP_DEBUGGING */

/* Turn on the mark bit in OBJ */
#define MARK_OBJECT(obj) \
  (PTR)(((long)(obj)) | 1)

/* Turn off the mark bit in OBJ */
#define UNMARK_OBJECT(obj) \
  (PTR)(((long)(obj)) & ~1)

/* Answer whether the class field in OBJ has the mark bit set. */
#define IS_MARKED(obj) \
  IS_OBJ_MARKED((obj)->objClass)

/* Answer whether the pointer OBJ has the mark bit set. */
#define IS_OBJ_MARKED(obj) \
  ((long)(obj) & 1)



/* These are the memory areas within which we mantain object data. */
heap _gst_oop_heap, _gst_object_heap;

/* These are the real OOPS for nil, true, and false */
OOP _gst_nil_oop, _gst_true_oop, _gst_false_oop;

/* The OOP table.  This contains a pointer to the object, and some flag
   bits indicating whether the object is read-only, reachable and/or pooled.
   Some of the bits indicate the difference between the allocated length
   (stored in the object itself), and the real length, because variable
   byte objects may not be an even multiple of sizeof(PTR). */
struct OOP *_gst_oop_table = NULL, *_gst_all_oops_table;

/* These are the pointer to the first free OOP (they constitute a linked
   list whose "next" pointer is stored in the OBJECT field of the OOP)
   and a pointer to the last used OOP; OOPs following this are not 
   considered. */
OOP _gst_first_free_oop, _gst_last_used_oop;

/* This is the number of OOPs in the free list and in the full OOP
   table.  _gst_num_free_oops is only correct after a GC!*/
int _gst_num_free_oops, _gst_oop_table_size;

/* This is true to show a message whenever a GC happens. */
mst_Boolean _gst_gc_message = true;

/* This is true in the middle of a GC. */
mst_Boolean _gst_gc_running = false;


/* Objects that are at least this big (in bytes) are allocated outside
   the main heap, hoping to provide more locality of reference between
   small objects. */
int _gst_big_object_threshold = 2 * K;

/* If there is this much space used after a _gst_scavenge, we need to
   grow the object heap by _gst_space_grow_rate % next time we
   _gst_scavenge, so that the storage gets copied into the new, larger
   area. */
int _gst_grow_threshold_percent = 80;

/* Grow the object heap by this percentage when the amount of space
   used exceeds _gst_grow_threshold_percent. */
int _gst_space_grow_rate = 30;

/* This vector holds the storage for all the Character objects in the
   system.  Since all character objects are unique, we pre-allocate
   space for 256 of them, and treat them as special built-ins when
   doing garbage collection.*/
gst_char_object _gst_char_object_table[NUM_CHAR_OBJECTS];

/* This is "nil" object in the system.  That is, the single instance
   of the UndefinedObject class, which is called "nil". */
struct gst_nil_object _gst_nil_object;

/* These represent the two boolean objects in the system, true and
   false.  This is the object storage for those two objects.  
   false == &_gst_boolean_objects[0],
   true == &_gst_boolean_objects[1] */
struct gst_boolean_object _gst_boolean_objects[2];

/* This variable represents information about the memory space.
   _gst_mem_space holds the required information: basically the
   pointer to the base and top of the space, and the pointers into it
   for allocation and copying. */
struct memory_space _gst_mem_space CACHELINE_ALIGNED;

/* These variables hold onto the object incubator's state */
OOP *_gst_inc_oop_base_ptr, *_gst_inc_oopptr, *_gst_inc_oop_end_ptr;


/* Allocates a table for OOPs of SIZE bytes, and store pointers to the
   builtin OOPs into _gst_nil_oop et al. */
static void alloc_ooptable (long int size);

/* The mark phase of the GC. */
static inline void mark_oops (void);

/* The sweep phase of the GC, starting at BOTTOM and ending at
   the top of the allocated space. */
static inline void sweep_oops (char *bottom);

/* Mark the OOPs that are part of the root set because they are
   finalizable (they will always live for another iteration) */
static inline void mark_finalizable_oops (void);

/* Performs special checks on weak and finalizable objects.  When one
   of the objects pointed to by a weak object have no other
   references, the slot of the weak object is replaced by a nil.  When
   a finalizable object has no references outside weak objects,
   prepare_for_sweep() marks it in a buffer.  This routine (which is
   executed after the sweep) calls back the finalize method. */
static inline void check_objects_after_sweep (void);

/* Walks the instance variables of OOP and nils out those that are not
   surviving the garbage collection.  Called by preare_for_sweep.  */
static inline void check_weak_refs (OOP oop);

/* Iterate through the OOP table.  Copy pooled, marked objects to the
   main heap space (starting at address dest), free the others. */
static inline void sweep_pooled_contexts (char *dest);

/* Iterate through the OOP table.  On marked OOPs, store the pointer
   to the object's class into the OOP and the OOP pointer where the
   object class usually is.  This allows the sweep phase to change the
   OOPs as it moves objects in memory.  Also mark specially objects to
   be finalized.  Along the way, it counts the number of bytes to be
   allocated for reachable pooled objects, and returns that value.

   Even if not reachable, objects whose object data resides below
   BOTTOM survive. */
static unsigned long prepare_for_sweep (PTR bottom);


void
_gst_init_ooptable (long int size)
{
  int i;

  _gst_oop_heap = NULL;
  for (i = MAX_OOP_TABLE_SIZE; i && !_gst_oop_heap; i >>= 1)
    _gst_oop_heap = _gst_heap_create (i * sizeof (struct OOP));

  if (!_gst_oop_heap)
    nomemory (true);

  alloc_ooptable (size);

  _gst_nil_oop->flags = F_READONLY | F_REACHABLE;
  _gst_nil_oop->object = (mst_Object) & _gst_nil_object;
  _gst_nil_object.objSize =
    FROM_INT (ROUNDED_WORDS (sizeof (struct gst_nil_object)));

  _gst_true_oop->flags = F_READONLY | F_REACHABLE;
  _gst_true_oop->object = (mst_Object) & _gst_boolean_objects[0];
  _gst_false_oop->flags = F_READONLY | F_REACHABLE;
  _gst_false_oop->object = (mst_Object) & _gst_boolean_objects[1];
  _gst_boolean_objects[0].objSize =
    FROM_INT (ROUNDED_WORDS (sizeof (struct gst_boolean_object)));
  _gst_boolean_objects[1].objSize =
    FROM_INT (ROUNDED_WORDS (sizeof (struct gst_boolean_object)));
  _gst_boolean_objects[0].booleanValue = _gst_true_oop;
  _gst_boolean_objects[1].booleanValue = _gst_false_oop;

  for (i = 0; i < NUM_CHAR_OBJECTS; i++)
    {
      _gst_char_object_table[i].objSize =
	FROM_INT (ROUNDED_WORDS (sizeof (_gst_char_object_table[i])));
      _gst_char_object_table[i].charVal = (char) i;
      _gst_oop_table[i + CHAR_OBJECT_BASE].object =
	(mst_Object) & _gst_char_object_table[i];
      _gst_oop_table[i + CHAR_OBJECT_BASE].flags =
	F_READONLY | F_REACHABLE;
    }
}


void
alloc_ooptable (long int size)
{
  long i;
  long bytes;

  _gst_oop_table_size = size;
  bytes = (size - FIRST_OOP_INDEX) * sizeof (struct OOP);
  _gst_all_oops_table =
    (struct OOP *) _gst_heap_sbrk (_gst_oop_heap, bytes);
  if (!_gst_all_oops_table)
    nomemory (true);

  _gst_oop_table = &_gst_all_oops_table[-FIRST_OOP_INDEX];
  _gst_nil_oop = &_gst_oop_table[nilOOPIndex];
  _gst_true_oop = &_gst_oop_table[trueOOPIndex];
  _gst_false_oop = &_gst_oop_table[falseOOPIndex];

  _gst_first_free_oop = &_gst_oop_table[size - 1];
  _gst_last_used_oop = _gst_oop_table;
  _gst_num_free_oops = size;
  _gst_first_free_oop->object = NULL;
  _gst_first_free_oop->flags = F_FREE;

  i = size - 1;
  while (_gst_first_free_oop != &_gst_oop_table[0])
    {
      _gst_first_free_oop = &_gst_oop_table[--i];
      _gst_first_free_oop->object =
	(mst_Object) & _gst_oop_table[i + 1];
      _gst_first_free_oop->flags = F_FREE;
    }
}

mst_Boolean
_gst_realloc_ooptable (long int newSize)
{
  long bytes;
  OOP oop;

  bytes = (newSize - _gst_oop_table_size) * sizeof (struct OOP);
  if (bytes < 0)
    return (true);

  if (!_gst_heap_sbrk (_gst_oop_heap, bytes))
    {
      /* try to recover.  Note that we cannot move the OOP table like
         we do with the object data. */
      nomemory (false);
      return (false);
    }

  /* mark the new OOPs as available */
  for (oop = &_gst_oop_table[_gst_oop_table_size];
       oop < &_gst_oop_table[newSize]; oop++)
    oop->flags = F_FREE;

  _gst_first_free_oop = &_gst_oop_table[newSize - 1];
  _gst_oop_table_size = newSize;
  _gst_refresh_oop_free_list ();
  return (true);
}

void
_gst_init_builtin_objects_classes (void)
{
  int i;

  _gst_nil_object.objClass = _gst_undefined_object_class;
  _gst_boolean_objects[0].objClass = _gst_true_class;
  _gst_boolean_objects[1].objClass = _gst_false_class;

  for (i = 0; i < NUM_CHAR_OBJECTS; i++)
    _gst_char_object_table[i].objClass = _gst_char_class;
}

void
_gst_fixup_metaclass_objects (void)
{
  OOP oop;
  mst_Object object;

  for (oop = _gst_oop_table; oop <= _gst_last_used_oop; oop++)
    {
      object = OOP_TO_OBJ (oop);
      if (!(oop->flags & F_FREE) && IS_NIL (object->objClass))
	object->objClass = _gst_metaclass_class;
    }
}


OOP
_gst_find_an_instance (OOP class_oop)
{
  OOP oop;

  for (oop = _gst_oop_table; oop <= _gst_last_used_oop; oop++)
    {
      if (!(oop->flags & F_FREE) && (OOP_CLASS (oop) == class_oop))
	return (oop);
    }

  return (_gst_nil_oop);
}


void
_gst_swap_objects (OOP oop1,
		   OOP oop2)
{
  struct OOP tempOOP;

  tempOOP = *oop2;		/* note structure assignment going on
				   here */
  *oop2 = *oop1;
  *oop1 = tempOOP;
}

mst_Boolean
_gst_oop_index_valid (long int index)
{
  return (index >= FIRST_OOP_INDEX && index <= _gst_oop_table_size);
}


mst_Object
_gst_alloc_obj (long unsigned int size,
		OOP *p_oop)
{
  char *newAllocPtr;
  mst_Object p_instance;

  size = ROUNDED_BYTES (size);
  newAllocPtr = _gst_mem_space.allocPtr + size;

#if 0
  if UNCOMMON (size >= _gst_big_object_threshold)
    {
      if UNCOMMON (newAllocPtr >= _gst_mem_space.maxPtr)
	{
	  /* not enough room currently, try to make some more */
	  int percent;
	  percent = (_gst_mem_space.newAllocPtr - _gst_mem_space.space)
	    * 100 / _gst_mem_space.totalSize;

	  if (percent >= _gst_grow_threshold_percent)
	    _gst_scavenge ();
	  else
	    _gst_minor_scavenge ();

	  newAllocPtr = _gst_mem_space.allocPtr + size;
	}

      /* If the object is big enough, we put it outside the heap and
	 point the oop to some malloc-ed memory.  However, we diminish
	 maxPtr, so that one cannot make the system allocate memory
	 without ever GCing by allocating big objects.  This is also
	 the reason why even above this threshold we check if we'd
	 better start a collection. */

      p_instance = (mst_Object) xmalloc (size);
      if (p_instance)
	{
	  _gst_mem_space.maxPtr -= size;
	  *p_oop = alloc_oop (p_instance);
	  (*p_oop)->flags |= F_FIXED;
	  return p_instance;
	}

      /* If allocation failed outside the main heap, try allocating
	 inside it.  Maybe that will succeed.  */
    }
#endif

  /* We don't want to have allocPtr pointing to the wrong thing during
     GC, so we use a local var to hold its new value */
  if UNCOMMON (newAllocPtr >= _gst_mem_space.maxPtr)
    {
      if (!_gst_gc_running)
	{
	  /* not enough room currently, try to make some more */
	  int percent;
	  percent = (_gst_mem_space.newAllocPtr - _gst_mem_space.space)
	    * 100 / _gst_mem_space.totalSize;

	  if (percent >= _gst_grow_threshold_percent)
	    _gst_scavenge ();
	  else
	    _gst_minor_scavenge ();

	  newAllocPtr = _gst_mem_space.allocPtr + size;
	}

      if UNCOMMON (newAllocPtr >= _gst_mem_space.maxPtr)
	{
	  /* uh oh, still too big -- we need to grow memory */
	  unsigned long spaceInUse;
	  unsigned long spaceNeeded;

	  spaceInUse = _gst_mem_space.allocPtr - _gst_mem_space.space;

	  /* we need what we already are using, plus some breathing
	     room */
	  spaceNeeded = size + spaceInUse;
	  if (_gst_space_grow_rate > 0)
	    spaceNeeded += spaceNeeded / 100 * _gst_space_grow_rate;

	  if (!_gst_grow_memory_to (spaceNeeded))
	    {
	      /* !!! do something more reasonable in the future */
	      _gst_errorf ("Cannot recover, exiting...");
	      exit (1);
	    }
	  newAllocPtr = _gst_mem_space.allocPtr + size;
	}
    }

  p_instance = (mst_Object) _gst_mem_space.allocPtr;
  _gst_mem_space.allocPtr = newAllocPtr;
  *p_oop = alloc_oop (p_instance);
  return p_instance;
}

mst_Object _gst_alloc_words (long size)
{
  mst_Object result;
  char *newAllocPtr;

  newAllocPtr = _gst_mem_space.allocPtr + SIZE_TO_BYTES (size);
  if (newAllocPtr >= _gst_mem_space.maxPtr)
    {
      /* uh oh, too big -- we need to grow memory */
      unsigned long spaceInUse;
      unsigned long spaceNeeded;

      spaceInUse = _gst_mem_space.allocPtr - _gst_mem_space.space;

      /* we need what we already are using, plus some breathing
	 room */
      spaceNeeded = SIZE_TO_BYTES (size) + spaceInUse;
      if (_gst_space_grow_rate > 0)
	spaceNeeded += spaceNeeded / 100 * _gst_space_grow_rate;

      if (!_gst_grow_memory_to (spaceNeeded))
	{
	  /* !!! do something more reasonable in the future */
	  _gst_errorf ("Cannot recover, exiting...");
	  exit (1);
	}
      newAllocPtr = _gst_mem_space.allocPtr + size;
    }

  result = (mst_Object) _gst_mem_space.allocPtr;
  result->objSize = FROM_INT (size);
  _gst_mem_space.allocPtr = newAllocPtr;
  return result;
}

mst_Boolean
_gst_make_fixed_oop (OOP oop)
{
  mst_Object newObj;
  int size = TO_INT(oop->object->objSize) * SIZEOF_LONG;

  if (((char *) oop->object) + size != _gst_mem_space.allocPtr)
    /* Can only make fixed the last created object. */
    return (false);

  newObj = (mst_Object) xmalloc (size * SIZEOF_LONG);
  memcpy (newObj, oop->object, size);

  /* Un-allocate the object from the heap and point the oop to the
     malloc-ed memory.  However, we diminish maxPtr, so that one cannot
     make the system allocate memory without ever GCing by allocating
     many small fixed objects. */

  _gst_mem_space.maxPtr -= _gst_mem_space.allocPtr - (char *) oop->object;
  _gst_mem_space.allocPtr = (char *) oop->object;
  oop->object = newObj;
  oop->flags |= F_FIXED;
  return (true);
}



void
_gst_init_mem (void)
{
  int i;

  _gst_object_heap = NULL;
  for (i = MAX_OBJECT_DATA_SIZE; i && !_gst_object_heap; i >>= 1)
    _gst_object_heap = _gst_heap_create (i);

  if (!_gst_object_heap)
    nomemory (true);

  _gst_mem_space.totalSize = INIT_MEM_SPACE_SIZE;
  _gst_mem_space.space = (char *) _gst_heap_sbrk (_gst_object_heap,
						  _gst_mem_space.
						  totalSize +
						  NEW_GENERATION_SIZE);

  if (!_gst_mem_space.space)
    nomemory (true);

  _gst_mem_space.allocPtr = _gst_mem_space.space;
  _gst_mem_space.newAllocPtr = _gst_mem_space.space;
  _gst_mem_space.maxPtr =
    _gst_mem_space.space + _gst_mem_space.totalSize;

  _gst_inc_init_registry ();
}


mst_Object
_gst_cur_space_addr (void)
{
  return ((mst_Object) _gst_mem_space.space);
}

void
_gst_set_space_info (long int size)
{
  _gst_mem_space.newAllocPtr = _gst_mem_space.allocPtr =
    _gst_mem_space.space + size;
}


mst_Boolean
_gst_grow_to (long unsigned int spaceSize)
{
  _gst_scavenge ();
  return _gst_grow_memory_to (spaceSize);
}


mst_Boolean
_gst_grow_memory_to (long unsigned int spaceSize)
{
  long spaceDelta;
  long bytes;
  char *newSpacePtr;
  OOP oop;
  heap newHeap;

  if (spaceSize <= _gst_mem_space.totalSize)
    return (true);

  if (spaceSize > MAX_OBJECT_DATA_SIZE - NEW_GENERATION_SIZE)
    {
      if (_gst_mem_space.totalSize >=
	  MAX_OBJECT_DATA_SIZE - NEW_GENERATION_SIZE)
	return (false);
      else
	spaceSize = MAX_OBJECT_DATA_SIZE - NEW_GENERATION_SIZE;
    }

  /* Do the real work... */
  bytes = spaceSize + NEW_GENERATION_SIZE;
  if (_gst_heap_sbrk
      (_gst_object_heap, spaceSize - _gst_mem_space.totalSize))
    {
      newSpacePtr = _gst_mem_space.space;
      newHeap = _gst_object_heap;
    }
  else
    {
      int i;

      /* Try to move to a newly allocated heap */
      newHeap = NULL;
      for (i = MAX_OBJECT_DATA_SIZE; i > bytes && !newHeap; i >>= 1)
	newHeap = _gst_heap_create (i);

      if (!newHeap || !(newSpacePtr = _gst_heap_sbrk (newHeap, bytes)))
	{
	  nomemory (false);
	  spaceSize = _gst_mem_space.totalSize;
	  if (newHeap)
	    _gst_heap_destroy (newHeap);

	  return (false);
	}
    }

#ifdef GROW_DEBUG
  printf ("old = %8x, new = %8x, delta = %8x\n", _gst_mem_space.space,
	  newSpacePtr, spaceDelta);
#endif

  _gst_fixup_object_pointers ();

  spaceDelta = newSpacePtr - _gst_mem_space.space;
  if (spaceDelta)
    {
      memcpy (newSpacePtr, _gst_mem_space.space,
	      spaceSize + NEW_GENERATION_SIZE);
      _gst_heap_destroy (_gst_object_heap);
      _gst_object_heap = newHeap;
    }

  _gst_mem_space.space = newSpacePtr;
  _gst_mem_space.totalSize = spaceSize;
  _gst_mem_space.allocPtr += spaceDelta;
  _gst_mem_space.newAllocPtr += spaceDelta;
  _gst_mem_space.maxPtr =
    _gst_mem_space.space + _gst_mem_space.totalSize;

  if (spaceDelta && _gst_oop_table)
    {
      /* Fix up the OOP table pointers to objects */
      for (oop = _gst_oop_table; oop <= _gst_last_used_oop; oop++)
	{
	  if (!(oop->flags & (F_FREE | F_POOLED)))
	    {
#ifdef GROW_DEBUG
	      PTR obj;
	      printf ("old = %8x, ", obj = (PTR) oop->object);
#endif
	      oop->object =
		(mst_Object) (((char *) oop->object) + spaceDelta);
#ifdef GROW_DEBUG
	      printf ("new = %x, delta = %x\n", oop->object,
		      ((PTR) oop->object) - obj);
#endif
	    }
	}
    }

  _gst_restore_object_pointers ();
  return (true);
}



void
_gst_scavenge (void)
{
  int lastPercent;

#ifdef PROFBLOCK
  ps.numMajorGCs++;
#endif

  if (_gst_gc_message && !_gst_regression_testing)
    {
      /* print the first part of this message before we finish
         scanning oop table for live ones, so that the delay caused by
         this scanning is apparent.  Note the use of stderr for the
         printed message.  The idea here was that generated output
         could be treated as Smalltalk code, HTML or whatever else you
         want without harm. */
      fflush (stdout);
      fprintf (stderr, "\"Major scavenging... ");
      fflush (stderr);
    }

  /* In alloc_oop, we don't want to worry about going below the
     low-water threshold, so we massage _gst_num_free_oops.  The real
     number of free OOPs will be computed in prepare_for_sweep. */
  _gst_num_free_oops += LOW_WATER_OOP_THRESHOLD;

  _gst_gc_running = true;
  _gst_fixup_object_pointers ();
  mark_oops ();
  sweep_oops (_gst_mem_space.space);
  _gst_restore_object_pointers ();
  _gst_gc_running = false;

  /* At this point, storage in memory is compacted and contiguous, so
     we can examine how much memory we have left, and decide if we need 
     to increase memory some more. */
  lastPercent = (_gst_mem_space.allocPtr - _gst_mem_space.space) * 100
    / _gst_mem_space.totalSize;

  if (lastPercent > _gst_grow_threshold_percent)
    {
      /* with bad grow rates, can undergrow.  Takes care of it */
      unsigned long newSpaceSize;

      newSpaceSize =
	_gst_mem_space.totalSize / 100 * (100 + _gst_space_grow_rate);
      newSpaceSize &= ~(sizeof (long) - 1);	/* round to word
						   boundary */
      if (newSpaceSize > _gst_mem_space.totalSize)
	_gst_grow_memory_to (newSpaceSize);
    }

  if (_gst_gc_message && !_gst_regression_testing)
    {
      fprintf (stderr, "done, used space = %i%%\"\n", lastPercent);
      fflush (stderr);
    }

  check_objects_after_sweep ();
#if 0
  _gst_invoke_hook ("scavenge");
#endif

}

void
_gst_minor_scavenge (void)
{
  int lastPercent;

#ifdef PROFBLOCK
  ps.numMinorGCs++;
#endif

  if (_gst_gc_message && !_gst_regression_testing)
    {
      /* print the first part of this message before we finish
	 scanning oop table for live ones, so that the delay caused by
	 this scanning is apparent.  note the use of stderr for the
	 printed message.  The idea here was that generated output
	 could be treated as Smalltalk code, HTML or whatever else you
	 want without harm. */
      fflush (stdout);
      fprintf (stderr, "\"Minor scavenging...");
      fflush (stderr);
    }

  /* In alloc_oop, we don't want to worry about going below the
     low-water threshold, so we massage _gst_num_free_oops.  The real
     number of free OOPs will be computed in prepare_for_sweep. */
  _gst_num_free_oops += LOW_WATER_OOP_THRESHOLD;

  _gst_gc_running = true;
  _gst_fixup_object_pointers ();
  mark_oops ();
  sweep_oops (_gst_mem_space.newAllocPtr);
  _gst_restore_object_pointers ();
  _gst_gc_running = false;

  lastPercent = (_gst_mem_space.allocPtr - _gst_mem_space.space) * 100
    / _gst_mem_space.totalSize;

  if (_gst_gc_message && !_gst_regression_testing)
    {
      fprintf (stderr, " done, used space = %i%%\"\n", lastPercent);
      fflush (stderr);
    }

  check_objects_after_sweep ();
#if 0
  _gst_invoke_hook ("minorScavenge");
#endif
}


void
check_objects_after_sweep (void)
{
  OOP *pOOP;
  long size;

  /* Make a local copy of the buffer */
  size = _gst_buffer_size ();
  pOOP = alloca (size);
  _gst_copy_buffer (pOOP);
  size /= SIZEOF_CHAR_P;

  while (size--)
    _gst_msg_send (*pOOP++, _gst_finalize_symbol, NULL);
}

void
check_weak_refs (OOP oop)
{
  mst_Object object;
  OOP *field;
  int numFields;

  object = OOP_TO_OBJ (oop);
  for (field = object->data, numFields = NUM_OOPS (object); numFields;
       field++, numFields--)
    {
      if (IS_INT (*field))
	continue;
      if (*field <= oop)
	{
	  /* Not yet scanned by prepare_for_sweep */
	  if (!IS_OOP_MARKED (*field))
	    *field = _gst_nil_oop;
	}
      else
	{
	  /* Already scanned by prepare_for_sweep */
	  if (IS_OOP_FREE (*field))
	    *field = _gst_nil_oop;
	}
    }
}

void
mark_oops (void)
{
  _gst_mark_registered_oops ();
  _gst_mark_processor_registers ();
  mark_finalizable_oops ();
}

#define TAIL_MARK_OOP(newOOP) BEGIN_MACRO { \
  oop = (newOOP); \
  continue;		/* tail recurse!!! */ \
} END_MACRO

#define TAIL_MARK_OOPRANGE(firstOOP, oopAtEnd) BEGIN_MACRO { \
  curOOP = (OOP *)(firstOOP); \
  atEndOOP = (OOP *)(oopAtEnd); \
  oop = NULL; \
  continue; \
} END_MACRO

void
_gst_mark_an_oop_internal (OOP oop,
			   OOP * curOOP,
			   OOP * atEndOOP)
{
  for (;;)
    {
      if (!oop)
	{			/* in the loop! */
#ifndef OPTIMIZE
	  mst_Object obj = (mst_Object) (curOOP - 1);	/* for
							   debugging */
#endif
	iterationLoop:
	  /* in a loop, do next iteration */
	  oop = *curOOP;
	  curOOP++;
	  if (IS_OOP (oop))
	    {
#ifndef OPTIMIZE
	      if (!IS_OOP_ADDR (oop))
		{
		  printf
		    ("Error! Invalid OOP %p was found inside %p!\n",
		     oop, obj);
		  _gst_debug ();
		}
	      else
#endif
	      if (!IS_OOP_MARKED (oop))
		{
		  if COMMON (curOOP < atEndOOP)
		    {
		      _gst_mark_an_oop_internal (oop, NULL, NULL);
		      goto iterationLoop;
		    }
		  else
		    /* On the last object in the set, reuse the
		       current invocation. oop is valid, so we go to
		       the single-object case */
		    continue;
		}
	    }
	  /* We reach this point if the object isn't to be marked.  The 
	     code above contains a continue to tail recurse, so we
	     cannot put the loop in a do...while and a goto is
	     necessary here.  Speed is a requirement, so I'm doing it. */
	  if (curOOP < atEndOOP)
	    goto iterationLoop;
	}
      else
	{			/* just starting with this oop */
	  OOP objClass;
	  mst_Object object;
	  unsigned long size;

#ifndef OPTIMIZE
	  if (IS_OOP_FREE (oop))
	    {
	      printf ("Error! Free OOP %p is being marked!\n", oop);
	      _gst_debug ();
	      break;
	    }
	  if (!IS_OOP_OUTSIDE_HEAP (oop)
	      && !IS_OBJ_ADDR (OOP_TO_OBJ (oop)))
	    {
	      printf
		("Error! OOP at %p points to invalid object data at %p!\n",
		 oop, oop->object);
	      _gst_debug ();
	      break;
	    }
#endif
	  /* see if the object has pointers, set up to copy them if so. 
	   */
	  oop->flags |= F_REACHABLE;
	  object = OOP_TO_OBJ (oop);
	  objClass = object->objClass;
	  if UNCOMMON (oop->flags & F_CONTEXT)
	    {
	      gst_method_context ctx;
	      long methodSP;
	      ctx = (gst_method_context) object;
	      methodSP = TO_INT (ctx->spOffset);
	      ctx->method->flags |= F_XLAT_REACHABLE;
	      /* printf("setting up for loop on context %x, sp = %d\n", 
	         ctx, methodSP); */
	      TAIL_MARK_OOPRANGE (&ctx->objClass,
				  ctx->contextStack + methodSP + 1);

	    }
	  else if UNCOMMON (oop->flags & F_WEAK)
	    {
	      /* In general, there will be many instances of a class,
		 but only the first time will it be unmarked.  So I'm
		 marking this as uncommon. */
	      if UNCOMMON (!IS_OOP_MARKED (objClass))
		{
		  TAIL_MARK_OOP (objClass);
		}
	    }
	  else
	    {
	      size = NUM_OOPS (object);
	      if COMMON (size)
		{
		  TAIL_MARK_OOPRANGE (&object->objClass,
				      object->data + size);
		}
	      else if UNCOMMON (!IS_OOP_MARKED (objClass))
		{
		  TAIL_MARK_OOP (objClass);
		}
	    }
	}
      /* This point is reached if and only if nothing has to be marked
         anymore in the current iteration. So exit. */
      break;
    }				/* for(;;) */
}

void
_gst_refresh_oop_free_list (void)
{
  OOP newFirstFree, oop;

  /* The free list will be reconstructed */
  newFirstFree = NULL;
  _gst_num_free_oops = _gst_oop_table_size;

  /* Proceed backwards so that the first free OOPs are at the head of
     the free list.  This minimizes the amount of space used by the OOP 
     table in a saved image. */

  for (oop = &_gst_oop_table[_gst_oop_table_size - 1];
       oop >= _gst_oop_table; oop--)
    {
      if (oop->flags & F_FREE)
	{
	  oop->object = (mst_Object) newFirstFree;
	  newFirstFree = oop;
	}
      else
	{
	  _gst_last_used_oop = oop;
	  _gst_num_free_oops--;
	  break;
	}
    }
  while (--oop >= _gst_oop_table)
    {
      if (oop->flags & F_FREE)
	{
	  oop->object = (mst_Object) newFirstFree;
	  newFirstFree = oop;
	}
      else
	_gst_num_free_oops--;
    }
  _gst_first_free_oop = newFirstFree;
}

unsigned long
prepare_for_sweep (PTR bottom)
{
  OOP newFirstFree, oop;
  mst_Object object;
  unsigned long pooledSize;

  /* The free list will be reconstructed, but the unused OOPs are
     unchanged */
  newFirstFree = _gst_first_free_oop;
  _gst_num_free_oops = _gst_oop_table_size;
  pooledSize = 0;

  _gst_reset_buffer ();

  /* Proceed backwards so that the first free OOPs are at the head of
     the free list.  This minimizes the amount of space used by the
     OOP table in a saved image. */

  for (oop = _gst_last_used_oop; oop >= _gst_oop_table; oop--)
    {
      if (oop->flags & F_WEAK)
	check_weak_refs (oop);

      /* The code below has a serious bug: sometimes the GC will sweep
	 away a method that is on the stack frame!  For this reason
         we are not garbage collecting unused native code yet. */
#if defined(USE_JIT_TRANSLATION) && buggy_code
      if (oop->flags & F_XLAT)
	{
	  if (oop->flags & F_XLAT_REACHABLE)
	    /* Reachable, and referenced by active contexts.  Keep it 
	       around. */
	    oop->flags &= ~F_XLAT_2NDCHANCE;
	  else
	    {
	      /* Reachable, but not referenced by active contexts.  We
	         give it a second chance... */
	      if (oop->flags & F_XLAT_2NDCHANCE)
	        release_native_code (oop);

	      oop->flags ^= F_XLAT_2NDCHANCE;
	    }
	}
#endif

      if (oop->flags & (F_FINALIZE | F_REACHABLE))
	{
	  if (oop->flags & F_POOLED)
	    /* Reachable pooled object.  Get it to the heap. */
	    pooledSize += TO_INT (OOP_TO_OBJ (oop)->objSize);

	  else if (oop->flags & F_FIXED)
	    /* Reachable FixedSpace object -- do nothing. */
	    ;

	  else if (((PTR) oop->object) >= bottom)
	    {
	      /* Object reachable after mark phase.  Reverse the
	         pointer */
	      object = OOP_TO_OBJ (oop);
	      oop->object = (mst_Object) object->objClass;
	      object->objClass = MARK_OBJECT (oop);
	    }

	  if (!(oop->flags & F_REACHABLE))
	    {
	      /* Object is *going* to be finalized, but it was not
	         yet.  We found a weak reference to it, so we mark it
	         so that finalization will occur soon after the end of
	         the sweep pass.  In the meanwhile, we let it survive
	         and decide the object's fate at the next GC pass */
	      oop->flags ^= F_FINALIZE;
	      _gst_add_buf_pointer (oop);
	    }
	  oop->flags &= ~F_REACHABLE | F_XLAT_REACHABLE;
	  _gst_num_free_oops--;

	}
      else if (!(oop->flags & F_FREE))
	{
	  /* Object not marked and not already freed.  Add to OOP free
	     list */

#ifdef USE_JIT_TRANSLATION
	  if (oop->flags & F_XLAT)
	    if (!(oop->flags & F_REACHABLE))
	      /* Unreachable, always free the native code.  Note
		 that being unreachable implies that
		 F_XLAT_REACHABLE is not set. Also note that it is
		 *not* optional to free the code in this case -- and
		 I'm not talking about memory leaks: a different
		 method could use the same OOP as this one and the
		 old method one would be executed instead of the new
		 one! */
	      release_native_code (oop);
#endif

	  if (oop->flags & F_FIXED)
	    /* Unreachable fixed (malloc-ed) object.  Free it. */
	    xfree (oop->object);

	  oop->flags = F_FREE;
	  oop->object = (mst_Object) newFirstFree;
	  newFirstFree = oop;
	  if (oop == _gst_last_used_oop)
	    _gst_last_used_oop--;
	}
    }

  _gst_first_free_oop = newFirstFree;

#ifdef USE_JIT_TRANSLATION
  /* Go and really free the blocks associated to garbage collected
     native code. */
  free_released_native_code ();
#endif

  return (pooledSize);
}

void
sweep_pooled_contexts (char *dest)
{
  OOP oop;
  unsigned long size;

  for (oop = _gst_last_used_oop; oop >= _gst_oop_table; oop--)
    {
      if (oop->flags & F_POOLED)
	{
	  size = SIZE_TO_BYTES (TO_INT (OOP_TO_OBJ (oop)->objSize));
	  memcpy (dest, oop->object, size);
	  oop->object = (mst_Object) dest;
	  oop->flags &= ~(F_POOLED | F_REACHABLE);
	  dest += size;
	}
    }
  _gst_empty_context_pool ();
}

/* #define SWEEP_DEBUG */
static inline void
sweep_oops (char *bottom)
{
  char *from, *fromStart, *to;
  unsigned long chunkSize, pooledSize;
  mst_Object object;
  OOP curClass, oop;

  pooledSize = prepare_for_sweep (bottom);

  /* Algorithm: 
     initialize: 
     * Start at beginning of allocated space. 
     * Skip over the initial contiguous range of marked object, unmarking
       as you go.

     loop:
     * skip over the contiguous range of unmarked objects, leaving 
       "to" where it is and advancing "from".
     * if "to" passes the end of allocated storage, we are done.
     * set "fromStart" to "from", and skip over the next contiguous
       range of marked objects, advancing "from". 
     * copy the range ["fromStart".."from") to "to".  
     * advance "to" to right after the newly copied area. */

  from = to = bottom;

  while (to < _gst_mem_space.allocPtr)
    {
      object = (mst_Object) to;
      if (!IS_MARKED (object))
	{
	  /* found the end of the contiguous range */
	  break;
	}
      /* unmark this dude */
      oop = UNMARK_OBJECT (object->objClass);
      curClass = (OOP) oop->object;
      oop->object = object;
      object->objClass = curClass;

      to += SIZE_TO_BYTES (TO_INT (object->objSize));
    }

#ifdef SWEEP_DEBUG
  printf ("skipped %d bytes of contig alloc %x space %x max %x\n",
	  to - _gst_mem_space.space, _gst_mem_space.allocPtr,
	  _gst_mem_space.space, _gst_mem_space.maxPtr);
#endif /* SWEEP_DEBUG */

  /* we've skipped over the marked initial set of objects, for which no 
     move is necessary.  Now begin the main execution loop */

  from = to;
  while (from < _gst_mem_space.allocPtr)
    {
      fromStart = from;		/* debugging only */
      while (from < _gst_mem_space.allocPtr)
	{
	  object = (mst_Object) from;
	  if (IS_MARKED (object))
	    /* found a non-free chunk */
	    break;

	  /* skip over the free memory */
	  from += SIZE_TO_BYTES (TO_INT (object->objSize));
	}

#ifdef SWEEP_DEBUG
      printf ("skipped free range %x .. %x %d bytes\n", fromStart, from,
	      from - fromStart);
#endif /* SWEEP_DEBUG */

      if (from >= _gst_mem_space.allocPtr)
	{
#ifdef SWEEP_DEBUG
	  printf ("hit end of memory\n");
#endif /* SWEEP_DEBUG */
	  break;		/* we've hit the end of active memory */
	}

      fromStart = from;
      /* span the next in-use contiguous chunk of objects */
      while (from < _gst_mem_space.allocPtr)
	{
	  object = (mst_Object) from;
	  if (!IS_MARKED (object))
	    {
	      /* found a free chunk */
	      break;
	    }

	  /* unmark this dude & tell the oop where the object *will be* 
	   */
	  oop = UNMARK_OBJECT (object->objClass);
	  curClass = (OOP) oop->object;
	  oop->object =
	    (mst_Object) (to + ((char *) object - fromStart));
	  object->objClass = curClass;

	  /* skip over the object */
	  from += SIZE_TO_BYTES (TO_INT (object->objSize));
	}


      /* copy the bytes down */
      chunkSize = from - fromStart;
#ifdef SWEEP_DEBUG
      printf ("copying range %x .. %x to %x, %d bytes\n",
	      fromStart, from, to, chunkSize);
#endif /* SWEEP_DEBUG */

      memcpy (to, fromStart, chunkSize);
      to += chunkSize;
    }

  _gst_mem_space.allocPtr = to;

  object = _gst_alloc_words (pooledSize);
  sweep_pooled_contexts ((char *) object);
  _gst_mem_space.newAllocPtr = _gst_mem_space.allocPtr;

  /* maxPtr is lowered when a big object is allocated.  Make it correct
     again. */
  _gst_mem_space.maxPtr =
    _gst_mem_space.space + _gst_mem_space.totalSize;
}

void
mark_finalizable_oops (void)
{
  OOP oop;

  for (oop = _gst_oop_table; oop <= _gst_last_used_oop; oop++)
    {
      /* A finalizable object will always survive this scavenging, even if 
         it is not reachable.  We mark the objects that they refer to,
         so they survive too, but keep the finalizable object marked as
         unreachable. */

      if ((oop->flags & (F_FINALIZE | F_REACHABLE)) == F_FINALIZE)
	{
	  _gst_mark_an_oop_internal (oop, NULL, NULL);
	  oop->flags ^= F_REACHABLE;
	}
    }
}




/***********************************************************************
 *
 * Incubator support routines
 *
 ***********************************************************************/

void
_gst_inc_init_registry (void)
{
  _gst_inc_oop_base_ptr =
    (OOP *) xmalloc (INIT_NUM_INCUBATOR_OOPS * sizeof (OOP *));
  _gst_inc_oopptr = _gst_inc_oop_base_ptr;
  _gst_inc_oop_end_ptr =
    _gst_inc_oop_base_ptr + INIT_NUM_INCUBATOR_OOPS;

  /* Make the incubated objects part of the root set */
  _gst_register_oop_array (&_gst_inc_oop_base_ptr, &_gst_inc_oopptr);
}

void
_gst_inc_grow_registry (void)
{
  OOP *oldBase;
  unsigned long oldPtrOffset;
  unsigned long oldRegistrySize, newRegistrySize;

  oldBase = _gst_inc_oop_base_ptr;
  oldPtrOffset = _gst_inc_oopptr - _gst_inc_oop_base_ptr;
  oldRegistrySize = _gst_inc_oop_end_ptr - _gst_inc_oop_base_ptr;

  newRegistrySize = oldRegistrySize + INCUBATOR_CHUNK_SIZE;

  _gst_inc_oop_base_ptr =
    (OOP *) xrealloc (_gst_inc_oop_base_ptr,
		      newRegistrySize * sizeof (OOP *));
  _gst_inc_oopptr = _gst_inc_oop_base_ptr + oldPtrOffset;
  _gst_inc_oop_end_ptr = _gst_inc_oop_base_ptr + newRegistrySize;
}
