/*  -*- Mode: C -*-  */

/* format.c ---  */

/* Author:	       Gary V. Vaughan <gary@oranda.demon.co.uk>
 * Maintainer:	       Gary V. Vaughan <gary@oranda.demon.co.uk>
 * Created:	       Thu Apr 22 23:13:34 1999
 * Last Modified:      Wed Jun 26 12:36:31 2002
 *            by:      Paolo Bonzini <bonzini@gnu.org>
 * ---------------------------------------------------------------------
 * @(#) $Id$
 * ---------------------------------------------------------------------
 */

/* Copyright (C) 1999, 2002 Gary V. Vaughan */

/* This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * As a special exception to the GNU General Public License, if you
 * distribute this file as part of a program that also links with and
 * uses the libopts library from AutoGen, you may include it under
 * the same distribution terms used by the libopts library.
 */

/* Code: */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <float.h>
#include <math.h>

#include "snprintfv/snprintfv.h"
#include "snprintfv/mem.h"

#if 0
/* libltdl requires this */
#define snv_default_spec_table	format_LTX_snv_default_spec_table
#endif

/* This is where the parsing of FORMAT strings is handled:

   Each of these functions should inspect PPARSER for parser
   state information;  update PPARSER as necessary based on
   the state discovered;  possibly put some characters in STREAM, in
   which case that number of characters must be returned.  If the
   handler detects that parsing (of the current specifier) is complete,
   then it must set pparser->state to SNV_STATE_END.  The library will then
   copy characters from the format string to STREAM until another unescaped
   SNV_CHAR_SPEC is detected when the handlers will be called again. */
   
static int number_handler PARAMS((snv_parser * const pparser,
                        	  STREAM *stream,
                        	  const struct spec_entry *spec));
static int flag_handler PARAMS((snv_parser * const pparser,
                        	  STREAM *stream,
                        	  const struct spec_entry *spec));
static int modifier_handler PARAMS((snv_parser * const pparser,
                        	  STREAM *stream,
                        	  const struct spec_entry *spec));
static int char_specifier_handler PARAMS((snv_parser * const pparser,
                        	  STREAM *stream,
                        	  const struct spec_entry *spec));
static int float_specifier_handler PARAMS((snv_parser * const pparser,
                        	  STREAM *stream,
                        	  const struct spec_entry *spec));
static int number_specifier_handler PARAMS((snv_parser * const pparser,
                        	  STREAM *stream,
                        	  const struct spec_entry *spec));
static int pointer_specifier_handler PARAMS((snv_parser * const pparser,
                        	  STREAM *stream,
                        	  const struct spec_entry *spec));
static int string_specifier_handler PARAMS((snv_parser * const pparser,
                        	  STREAM *stream,
                        	  const struct spec_entry *spec));

spec_entry snv_default_spec_table[] =
{
    /* ch  type		function */
    { ' ', FALSE, 0,		flag_handler },
    { '#', FALSE, 0,		flag_handler },
    { '+', FALSE, 0,		flag_handler },
    { '-', FALSE, 0,		flag_handler },
    { '\'',FALSE, 0,		flag_handler },
    { '*', FALSE, PA_INT,	number_handler },
    { '.', FALSE, 0,		number_handler },
    { '0', FALSE, 0,		flag_handler },
    { '1', FALSE, 0,		number_handler },
    { '2', FALSE, 0,		number_handler },
    { '3', FALSE, 0,		number_handler },
    { '4', FALSE, 0,		number_handler },
    { '5', FALSE, 0,		number_handler },
    { '6', FALSE, 0,		number_handler },
    { '7', FALSE, 0,		number_handler },
    { '8', FALSE, 0,		number_handler },
    { '9', FALSE, 0,		number_handler },
    { 'c', TRUE,  PA_CHAR,	char_specifier_handler },
    { 'd', TRUE,  PA_INT,	number_specifier_handler, (snv_pointer)10 },
    { 'e', TRUE,  PA_DOUBLE,	float_specifier_handler, (snv_pointer)6 },
    { 'E', TRUE,  PA_DOUBLE,	float_specifier_handler, (snv_pointer)6 },
    { 'f', TRUE,  PA_DOUBLE,	float_specifier_handler, (snv_pointer)6 },
    { 'F', TRUE,  PA_DOUBLE,	float_specifier_handler, (snv_pointer)6 },
    { 'g', TRUE,  PA_DOUBLE,	float_specifier_handler, (snv_pointer)1 },
    { 'G', TRUE,  PA_DOUBLE,	float_specifier_handler, (snv_pointer)1 },
    { 'h', TRUE,  0,		modifier_handler },
    { 'i', TRUE,  PA_INT,	number_specifier_handler, (snv_pointer)10 },
    { 'j', TRUE,  0,		modifier_handler },
    { 'l', TRUE,  0,		modifier_handler },
    { 'L', TRUE,  0,		modifier_handler },
    { 'o', TRUE,  PA_INT | PA_FLAG_UNSIGNED,
				number_specifier_handler, (snv_pointer)8 },
    { 'p', TRUE,  PA_POINTER,   pointer_specifier_handler, (snv_pointer)16 },
    { 'q', TRUE,  0,		modifier_handler },
    { 's', TRUE,  PA_STRING,	string_specifier_handler },
    { 't', TRUE,  0,		modifier_handler },
    { 'u', TRUE,  PA_INT | PA_FLAG_UNSIGNED,
				number_specifier_handler, (snv_pointer)10 },
    { 'x', TRUE,  PA_INT | PA_FLAG_UNSIGNED,
				number_specifier_handler, (snv_pointer)16 },
    { 'X', TRUE,  PA_INT | PA_FLAG_UNSIGNED,
				number_specifier_handler, (snv_pointer)16 },
    { 'z', TRUE,  0,		modifier_handler },
    { '\0',FALSE, PA_LAST,	NULL }
};

static void	fetch_width_and_prec PARAMS((snv_parser *pparser,
				    int default_prec));
static intmax_t	fetch_intmax PARAMS((snv_parser *pparser,
				    int index));
static uintmax_t fetch_uintmax PARAMS((snv_parser *pparser,
				    int index));
static double	fetch_double PARAMS((snv_parser *pparser,
				    int index));

/* Helper functions to print doubles */
static inline double ipow2 PARAMS((double base, 
				    int n));
static inline double ipow PARAMS((double base,
				    int n));

static int	print_float PARAMS((snv_parser *pparser,
				    char *buf,
				    double n));


static uintmax_t
fetch_uintmax (pparser, index)
    snv_parser *pparser;
    int index;
{
    /* parser read '*', and should get value from argument vector */
    if (pparser->is_long)
    {
	/* A long may be larger than a char*, so we stored the address
	   of the data to avoid losing information in the cast. */
	return (uintmax_t) *(unsigned long*)pparser->argv[index];
    }
    else if (pparser->is_long_double)
    {
	/* A long long may be larger than a char*, so we stored the address
	   of the data to avoid losing information in the cast. */
	return *(uintmax_t*)pparser->argv[index];
    }
    else
    {
	/* Any other integral type can fit in a pointer, so we stored
	   the data directly in the word. */
	return (uintmax_t)SNV_POINTER_TO_UINT(pparser->argv[index]);
    }
}

static intmax_t
fetch_intmax (pparser, index)
    snv_parser *pparser;
    int index;
{
    /* parser read '*', and should get value from argument vector */
    if (pparser->is_long)
    {
	/* A long may be larger than a char*, so we stored the address
	   of the data to avoid losing information in the cast. */
	return (intmax_t) *(long*)pparser->argv[index];
    }
    else if (pparser->is_long_double)
    {
	/* A long long may be larger than a char*, so we stored the address
	   of the data to avoid losing information in the cast. */
	return *(intmax_t*)pparser->argv[index];
    }
    else
    {
	/* Any other integral type can fit in a pointer, so we stored
	   the data directly in the word. */
	return (intmax_t)SNV_POINTER_TO_INT(pparser->argv[index]);
    }
}

static double
fetch_double (pparser, index)
    snv_parser *pparser;
    int index;
{
    if (pparser->is_long_double)
	return (double) *(long double*)pparser->argv[index];
    else
	return *(double*)pparser->argv[index];
}

static void
fetch_width_and_prec (pparser, default_prec)
    snv_parser *pparser;
    int default_prec;
{
    if (pparser->width == INT_MIN)
    {
	if (pparser->argv != NULL)
	{
	    /* An int fits in a pointer, so we stored
	       the data directly in the word. */
	    pparser->width = SNV_POINTER_TO_INT(pparser->argv[pparser->argindex]);

	    /* Check for a negative width. */
	    if (pparser->width < 0)
	    {
		pparser->width = -pparser->width;
		pparser->left = !pparser->left;
	    }
	}

	pparser->argindex++;
    }
    
    if (pparser->prec == -1)
    {
	/* parser didn't read an explicit value */
	pparser->prec = default_prec;
    }

    if (pparser->prec == INT_MIN)
    {
	if (pparser->argv == NULL)
	{
	    pparser->prec = default_prec;
	}
	else
	{
	    /* An int fits in a pointer, so we stored
	       the data directly in the word. */
	    pparser->prec = SNV_POINTER_TO_INT(pparser->argv[pparser->argindex]);
	}

	pparser->argindex++;
    }
}


#ifndef HAVE_LDEXP
/* A couple of replacement math functions...
   Emulate frexp and ldexp with logb and scalb */

static inline double
frexp (dbl, exp_ptr)
    double dbl;
    int *exp_ptr;
{
  int exponent = 1 + (int) logb(dbl);
  *exp_ptr = exponent;
  return scalb (dbl, -exponent);
}

static inline double
ldexp (dbl, exponent)
    double dbl;
    int exp_ptr;
{
  return scalb (dbl, exponent);
}
#endif

#ifndef HAVE_COPYSIGN
/* If we do not have signbit, assume zero is unsigned (too risky to
   assume we have infinities, which would allow to test with
   (x < 0.0 && 1.0 / x < 0.0).  */
copysign (x, y)
    double x, y;
{
  return (x < 0.0 ^ y < 0.0) ? x * -1.0 : x;
}
#endif

static inline double
ipow2 (base, n)
    double base;
    int n;
{
    int k = 1;
    double result = 1.0;
    while (n)
    {
        if (n & k)
        {
            result *= base;
	    n ^= k;
	}
	base *= base;
        k <<= 1;
    }
    return result;
}

static inline double
ipow (base, n)
    double base;
    int n;
{
    if (n < 0)
        return ipow2 (1.0 / base, -n);
    else
        return ipow2 (base, n);
}

int
print_float (pparser, buf, n)
    snv_parser *pparser;
    char *buf;
    double n;
{
    /* Print value of n in a buffer in the given base.
       Based upon the algorithm outlined in:
         Robert G. Burger and R. Kent Dybvig
         Printing Floating Point Numbers Quickly and Accurately
         ACM SIGPLAN 1996 Conference on Programming Language Design and Implementation
         June 1996.
     
       This version performs all calculations with doubles. */

    const int printed_bits = DBL_MANT_DIG - 3;
    const double base = 10.0;

    int exp, base_exp, d, zeros_dropped = 0, dec_point_count, prec;
    boolean fixed_format, drop_zeros, tc1, tc2;
    double m, m_plus, m_minus, significand, scale;
    char exp_char, *p = buf;
	
    /* Parse the specifier and adapt our printing format to it */
    exp_char = pparser->spec < 'a' ? 'E' : 'e';
    prec = pparser->prec;
    switch (pparser->spec)
    {
        case 'e': case 'E':
	    drop_zeros = FALSE;
	    fixed_format = FALSE;
	    break;
	    
	case 'f': case 'F':
	    drop_zeros = FALSE;
	    fixed_format = TRUE;
	    break;
	    
        case 'g': case 'G':
	    prec = MAX(1, prec);
	    drop_zeros = !pparser->alt;
	    fixed_format = TRUE;
	    break;
	    
	default:
	    abort();
    }
   
    /* Do the special cases: nans, infinities, zero, and negative numbers. */ 
    if (n != n)
    {
        /* Not-a-numbers are printed as a simple string. */
	*p++ = pparser->spec < 'a' ? 'N' : 'n';
	*p++ = pparser->spec < 'a' ? 'A' : 'a';
	*p++ = pparser->spec < 'a' ? 'N' : 'n';
        return p - buf;
    }

    /* Zero and infinity also can have a sign in front of them. */
    if (copysign(1.0, n) < 0.0)
    {
	n = -1.0 * n;
	*p++ = '-';
    }
    else if (pparser->space)
    {
	*p++ = ' ';
    }
    else if (pparser->showsign)
    {
	*p++ = '+';
    }

    if (n == 0.0)
    {
        /* Zero is printed as either '0' or '0.00000...' */
        *p++ = '0';
        if (drop_zeros || (prec == 0 && !pparser->alt))
	{
	    return p - buf;
	}
	*p++ = '.';
	memset (p, '0', prec);
	p += prec;
	return p - buf;
    }

    if ((n - n) != (n - n))
    {
        /* Infinities are printed as a simple string. */
	*p++ = pparser->spec < 'a' ? 'I' : 'i';
	*p++ = pparser->spec < 'a' ? 'N' : 'n';
	*p++ = pparser->spec < 'a' ? 'F' : 'f';
        return p - buf;
    }

    /* Do the black magic required to initialize the loop. */
    significand = frexp(n, &exp);
    base_exp = ceil (exp / log(base) * 6.9314718036912381649E-1 - 1.0E-10);
   
    /* The algorithm repeatedly strips off the leading digit
       of the number, prints it, and multiplies by 10 to get
       the next one.  However to avoid overflows when we have
       big numbers or denormals (whose reciprocal is +Inf) we
       have sometimes to work on scaled versions of n.  This
       scaling factor goes into m.

       m_plus and m_minus track the indetermination in the bits
       of n that we print (not n/m in this case) so that we have
       a precise measure of when we have printed a number that
       is equivalent to n.  */

    m = 1.0;
    if (exp >= 0)
    {
	m_plus = ldexp (1.0, exp - printed_bits);
	m_minus = (significand != 1.0) ? m_plus : m_plus / 2.0;
    }
    else
    {
	n = ldexp (n, printed_bits);
	m = ldexp (m, printed_bits);
	m_minus = ldexp (1.0, MAX(exp, DBL_MIN_EXP - 3));
	m_plus = (exp == DBL_MIN_EXP - 2 || significand != 1.0) 
		   ? m_minus : m_minus * 2.0;
    }
    
    if (base_exp >= 0)
    {
	if (exp == DBL_MAX_EXP - 1)
	{
	    /* Scale down to prevent overflow to Infinity during conversion */
	    n /= base;
	    m /= base;
	    m_plus /= base;
	    m_minus /= base;
	}
    } else {
	if (exp < DBL_MIN_EXP - 2)
	{
	    /* Scale up to prevent denorm reciprocals overflowing to Infinity */
	    d = ceil (53 / log(base) * 6.9314718036912381649E-1 - 1.0E-10);
	    scale = ipow (base, d);
	    n *= scale;
	    m *= scale;
	    m_plus *= scale;
	    m_minus *= scale;
	}
    }
    
    /* Modify m so that n/m < 0.1 (the first printed digit is n/m,
       and we remove the equality sign later). */
    m *= ipow (base, base_exp);

    /* Now reduce it to 1 <= n/m < 10. */
    while (n + m_plus < m)
    {
        m /= base;
        base_exp--;
    }
    base_exp++;
   
    if (pparser->spec == 'g' || pparser->spec == 'G')
    { 
	fixed_format = base_exp >= -4 && base_exp <= prec;

	/* For %g/%G, the precision specifies the number of printed
	   significant digits, not the number of decimal digits.  */
	prec -= fixed_format ? base_exp : 1;
    }

    if (fixed_format)
    {
    	/* Not in scientific notation.  Print 0.00000's if needed */
        dec_point_count = base_exp;
	if (dec_point_count <= 0)
	{
	    *p++ = '0';
	}
	if (dec_point_count < 0)
	{
	    *p++ = '.';
	    memset (p, '0', -dec_point_count);
	    p += MIN(prec, -dec_point_count);
	}
    }
    else
    {
        dec_point_count = 1;
    }

    /* Round the last printed digits.  If the precision is high enough,
       the RHS is too small to actually change n, so we still need to
       check for rounding in the loop below.  */
    n += m * 0.5 * ipow (base, 1 - dec_point_count - prec);

    do
    {
	/* Extract the most significant digit and test the
	   two termination conditions corresponding to underflow. */
        d = floor (n / m);
	n -= d * m;
	
	tc1 = n < m_minus;
	tc2 = n + m_plus >= m;
	if (tc2 && (!tc1 || n * 2.0 >= m))
        {
	    /* Check whether printing the correct value requires us
	       to round towards +infinity. */
            d++;
        }

	if (drop_zeros && d == 0 && dec_point_count <= 0)
	{
    	    /* Keep a count of trailing zeros after the decimal point. */
	    zeros_dropped++;
	}
	else
	{
    	    /* Write zeros that we thought were trailing. */
	    if (dec_point_count == -zeros_dropped)
	    {
	        *p++ = '.';
	    }

	    while (zeros_dropped--)
	    {
	        *p++ = '0';
	    }

	    zeros_dropped = 0;
	    *p++ = d + '0';
	}

	n *= base;
	m_plus *= base;
	m_minus *= base;
    }
    /* Exit when we underflow n, or when we wrote all the
       decimal places we were asked for. */
    while (!tc1 && !tc2 && --dec_point_count > -prec);

    /* Write trailing zeros *before* the decimal point. */
    if (dec_point_count > 0)
    {
	memset (p, '0', dec_point_count);
	p += dec_point_count;
	dec_point_count = 0;
    }

    /* Don't put the decimal point if there are no trailing
       zeros, unless %# was given. */
    if (dec_point_count == 0 && (prec > 0 || pparser->alt))
    {
	*p++ = '.';
    }

    /* If asked for, write trailing zeros *after* the decimal point. */
    if (!drop_zeros && prec + dec_point_count > 0)
    {
	memset (p, '0', prec + dec_point_count);
	p += prec + dec_point_count;
    }

    /* Print the exponent now. */    
    if (!fixed_format)
    {
	char x[10], *q = x;
        *p++ = exp_char;
	base_exp--;
	if (base_exp < 0)
	{
            *p++ = '-';
	    base_exp = -base_exp;
	}
	else
	{
	    *p++ = '+';
	}

	if (base_exp < 10)
	{
	    /* At least two digits in the exponent please. */
	    *p++ = '0';
	}

	do
	{
	    *q++ = base_exp % 10 + '0';
	    base_exp /= 10;
	}
	while (base_exp != 0);

	/* Copy to the main buffer in reverse order */
	while (q > x)
	{
	    *p++ = *--q;
	}
    }

    return p - buf;
}

static int
flag_handler (pparser, stream, spec)
    snv_parser * const pparser;
    STREAM *stream;
    const struct spec_entry *spec;
{
    return_val_if_fail(pparser != NULL, SNV_ERROR);

    if (!(pparser->state & (SNV_STATE_BEGIN|SNV_STATE_FLAG)))
    {
	PARSER_ERROR(pparser, "invalid specifier");
	return -1;
    }
    
    pparser->state = SNV_STATE_FLAG;
    
    while (pparser->state & SNV_STATE_FLAG)
    {
	switch (*pparser->format)
	{
	case '#':
	    pparser->alt = TRUE;
	    pparser->format++;
	    break;

	case '0':
	    /* We never store allocated storage under "pad", so we can
	       ignore any returned stale data. */
	    pparser->pad = '0';
	    pparser->format++;
	    break;

	case '-':
	    pparser->left = TRUE;
	    pparser->format++;
	    break;
	    
	case ' ':
	    pparser->space = TRUE;
	    pparser->format++;
	    break;

	case '+':
	    pparser->showsign = TRUE;
	    pparser->format++;
	    break;

	case '\'':
	    pparser->group = TRUE;
	    pparser->format++;
	    break;

	default:
	    pparser->state = ~(SNV_STATE_BEGIN|SNV_STATE_FLAG);
	    break;
	}
    }

    /* Return the number of characters emitted. */
    return 0;
}

static int
number_handler (pparser, stream, spec)
    snv_parser * const pparser;
    STREAM *stream;
    const struct spec_entry *spec;
{
    char *pEnd = NULL;
    int *pInfo = NULL;
    unsigned long value; /* we use strtoul, and cast back to int. */

    return_val_if_fail(pparser != NULL, SNV_ERROR);

    /* If we are looking at a ``.'', then this is a precision parameter. */
    if (*pparser->format == '.')
    {
	if (!(pparser->state & (SNV_STATE_PRECISION|SNV_STATE_BEGIN)))
	{
	    PARSER_ERROR(pparser, "invalid specifier");
	    return -1;
	}
	
	pInfo = &pparser->prec;
	pparser->format++;

	/* Acceptable states for the next token. */
	pparser->state = SNV_STATE_MODIFIER|SNV_STATE_SPECIFIER;
    }

    /* Parse the number (or optionally a ``*''). */
    value = strtoul(pparser->format, &pEnd, 10);
    if (pEnd != NULL && pEnd > pparser->format)
    {
	pparser->format = pEnd;
    }
    else if (*pparser->format == '*')
    {
	value = (unsigned long)INT_MIN;
	pparser->format++;

	/* If the caller wants type info we will have an address at which
	   to store that info. */
	if (pparser->argv == NULL)
	{
	    pparser->argtype(pparser, PA_INT, pparser->argindex);
	    pparser->argindex++;
	}
    }
    else
    {
	PARSER_ERROR(pparser, "invalid specifier");
	return -1;
    }
    
    /* If the next character is a dollar,
       then the number is an argument index. */
    if (*pparser->format == '$')
    {
	if (!(pparser->state & SNV_STATE_BEGIN))
	{
	    PARSER_ERROR(pparser, "invalid specifier");
	    return -1;
	}
	
	if (value == (unsigned long)INT_MIN)
	{
	    /* Reject ``*'' for an argument index. */
	    PARSER_ERROR(pparser, "invalid specifier");
	    return -1;
	}

	pInfo = &pparser->dollar;
	value--;		/* pparser->argv indices are zero-based */
	    
	/* Acceptable states for the next token. */
	pparser->state = ~SNV_STATE_BEGIN;
	pparser->format++;
    }

    if (pInfo == NULL)
    {
	/* Implies we must have read a width specification. */
	if (!(pparser->state & (SNV_STATE_BEGIN|SNV_STATE_WIDTH)))
	{
	    PARSER_ERROR(pparser, "invalid specifier");
	    return -1;
	}
	
	pInfo = &pparser->width;
	
	/* Acceptable states for the next token. */
	pparser->state = ~(SNV_STATE_BEGIN|SNV_STATE_FLAG|SNV_STATE_WIDTH);
    }

    if ((value != (unsigned long)INT_MIN) && (value > INT_MAX))
    {
	PARSER_ERROR(pparser, "out of range");
	return -1;
    }
    
    *pInfo = value;
    
    /* Return the number of characters emitted. */
    return 0;
}

static int
modifier_handler (pparser, stream, spec)
    snv_parser * const pparser;
    STREAM *stream;
    const struct spec_entry *spec;
{
    return_val_if_fail(pparser != NULL, SNV_ERROR);

    /* Check for valid pre-state. */
    if (!(pparser->state & (SNV_STATE_BEGIN|SNV_STATE_MODIFIER)))
    {
	PARSER_ERROR(pparser, "out of range");
	return -1;
    }
    
    while (pparser->state != SNV_STATE_SPECIFIER)
    {
	switch (*pparser->format)
	{
	case 'h':
	    pparser->is_char = pparser->is_short;
	    pparser->is_short = !pparser->is_short;
	    pparser->format++;
	    break;

	case 'z':
	case 't':
	    if (sizeof (long) > sizeof (char *))
	    {
		pparser->is_long_double = TRUE;
	    }
	    else
	    {
		pparser->is_long = TRUE;
	    }
            break;

	case 'l':
            if (*++pparser->format != 'l')
            {
	        pparser->is_long = TRUE;
                break;
            }
	    /*NOBREAK*/
	case 'j':
	case 'q':
	case 'L':
	    pparser->is_long_double = TRUE;
	    pparser->format++;
	    break;

	default:
	    pparser->state = SNV_STATE_SPECIFIER;
	    break;
	}
    }
    
    /* Return the number of characters emitted. */
    return 0;
}

static int
char_specifier_handler (pparser, stream, spec)
    snv_parser * const pparser;
    STREAM *stream;
    const struct spec_entry *spec;
{
    int count_or_errorcode = SNV_OK;
    char ch = '\0';

    return_val_if_fail(pparser != NULL, SNV_ERROR);

    /* Check for valid pre-state. */
    if (!(pparser->state & (SNV_STATE_BEGIN|SNV_STATE_SPECIFIER)))
    {
	PARSER_ERROR(pparser, "out of range");
	return -1;
    }
    
    /* Set post-state. */
    pparser->state = SNV_STATE_END;

    /* Advance the format pointer past the parsed character. */
    pparser->format++;
    
    /* If the caller wants type info we will have an address at which
       to store that info. */

    /* Read these now to advance the argument pointer appropriately */
    fetch_width_and_prec(pparser, 0);
    return_val_if_fail(pparser->prec >= 0, SNV_ERROR);

    if (pparser->dollar == -1)
    {
	pparser->dollar = pparser->argindex++;
    }
    if (pparser->argv == NULL)
    {
	pparser->argtype(pparser, PA_CHAR, pparser->dollar);
	return 1;
    }

    /* Extract the correct argument from the arg vector. */
    ch = (char)SNV_POINTER_TO_UINT(pparser->argv[pparser->dollar]);
    
    /* Left pad to the width if the supplied argument is less than
     * the width specifier.
     */
    if ((pparser->width > 1) && (pparser->left))
    {
	int padwidth = pparser->width - 1;

	while ((count_or_errorcode >= 0)
	       && (count_or_errorcode < padwidth))
	{
	    SNV_EMIT(pparser->pad, stream, count_or_errorcode);
	}
    }

    /* Emit the character argument.
     */
    SNV_EMIT(ch, stream, count_or_errorcode);

    /* Right pad to the width if we still didn't reach the specified
     * width and the left justify flag was set.
     */
    if ((count_or_errorcode < pparser->width) && (pparser->left))
    {
	while ((count_or_errorcode >= 0) && (count_or_errorcode < pparser->width))
	{
	    /* we always left justify with spaces */
	    SNV_EMIT(' ', stream, count_or_errorcode);
	}
    }

    /* Return the number of characters emitted. */
    return count_or_errorcode;
}

static int
float_specifier_handler (pparser, stream, spec)
    snv_parser * const pparser;
    STREAM *stream;
    const struct spec_entry *spec;
{
    double value = 0.0;
    int len, count_or_errorcode = SNV_OK, type = PA_DOUBLE;
    char buffer[DBL_MAX_10_EXP+20], *p = buffer;

    return_val_if_fail(pparser != NULL, SNV_ERROR);

    /* Check for valid pre-state. */
    if (!(pparser->state & (SNV_STATE_BEGIN|SNV_STATE_SPECIFIER)))
    {
	PARSER_ERROR(pparser, "out of range");
	return -1;
    }

    /* Set post-state. */
    pparser->state = SNV_STATE_END;

    /* Advance the format pointer past the parsed character. */
    pparser->format++;
    
    if (pparser->argv == NULL)
    {
        /* Read these now to advance the argument pointer appropriately */
        fetch_width_and_prec(pparser, (int) spec->user);
        return_val_if_fail(pparser->prec >= 0, SNV_ERROR);

        if (pparser->dollar == -1)
        {
	    pparser->dollar = pparser->argindex++;
        }
        {
	    pparser->argtype(pparser, type, pparser->dollar);
	    return 1;
        }
    }

    /* Read these now to advance the argument pointer appropriately */
    fetch_width_and_prec(pparser, (int) spec->user);
    return_val_if_fail(pparser->prec >= 0, SNV_ERROR);

    if (pparser->dollar == -1)
    {
	pparser->dollar = pparser->argindex++;
    }
    if (pparser->argv == NULL)
    {
	pparser->argtype(pparser, type, pparser->dollar);
	return 1;
    }

    /* Extract the correct argument from the arg vector. */
    value = fetch_double(pparser, pparser->dollar);
    
    /* Convert the number into a string. */
    len = print_float (pparser, buffer, value);

    pparser->width -= len;

    /* Left pad to the remaining width if the supplied argument is less
     * than the width specifier, and the padding character is ' '.
     */
    if (pparser->pad == ' ' && !pparser->left)
    {
	while ((count_or_errorcode >= 0)
	       && (pparser->width-- > 0))
	{
	    SNV_EMIT(pparser->pad, stream, count_or_errorcode);
	}
    }

    /* Display any sign character. */
    if (count_or_errorcode >= 0)
    {
	if (*p == '+' || *p == '-' || *p == ' ')
	{
	    SNV_EMIT(*p++, stream, count_or_errorcode);
	    len--;
	}
    }

    /* Left pad to the remaining width if the supplied argument is less
     * than the width specifier, and the padding character is not ' '.
     */
    if (pparser->pad != ' ' && !pparser->left)
    {
	while ((count_or_errorcode >= 0)
	       && (pparser->width-- > 0))
	{
	    SNV_EMIT(pparser->pad, stream, count_or_errorcode);
	}
    }

    /* Fill the stream buffer with as many characters from the number
     * buffer as possible without overflowing.
     */
    while ((count_or_errorcode >= 0) && (len-- > 0))
    {
	SNV_EMIT(*p++, stream, count_or_errorcode);
    }

    /* Right pad to the width if we still didn't reach the specified
     * width and the left justify flag was set.
     */
    if (pparser->left)
    {
	while ((count_or_errorcode >= 0) && (pparser->width-- > 0))
	{
	    /* we always left justify with spaces */
	    SNV_EMIT(' ', stream, count_or_errorcode);
	}
    }

    /* Return the number of characters emitted. */
    return count_or_errorcode;
}

static int
number_specifier_handler (pparser, stream, spec)
    snv_parser * const pparser;
    STREAM *stream;
    const struct spec_entry *spec;
{
    static const char digits_lower[] = "0123456789abcdefghijklmnopqrstuvwxyz";
    static const char digits_upper[] = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ";
    const char *digits;

    unsigned base = (unsigned) spec->user;
    uintmax_t value = 0L, mask = UINT_MAX;
    int type, count_or_errorcode = SNV_OK;
    char buffer[256], *p, *end;
    boolean is_negative = FALSE;

    return_val_if_fail(pparser != NULL, SNV_ERROR);

    /* Check for valid pre-state. */
    if (!(pparser->state & (SNV_STATE_BEGIN|SNV_STATE_SPECIFIER)))
    {
	PARSER_ERROR(pparser, "out of range");
	return -1;
    }
    
    /* Set post-state. */
    pparser->state = SNV_STATE_END;

    /* Advance the format pointer past the parsed character. */
    pparser->format++;
    
    /* Upper or lower-case hex conversion? */
    digits = ((spec->spec >= 'a') && (spec->spec <= 'z'))
	   ? digits_lower : digits_upper;

    /* Ensure that the type flags are propogated to the caller. */
    type = spec->type;
    if ((type & PA_TYPE_MASK) == PA_POINTER)
    {
	type |= PA_FLAG_UNSIGNED;

	/* Set mask to all 1's */
	mask |= !mask;

	if (sizeof (mask) > sizeof (char *))
	{
	    /* Keep the rightmost 8*sizeof(char *) bits.  */
	    mask ^= mask << (8 * sizeof (char *));
	}
    }
    if (pparser->is_char)
    {
	type |= PA_FLAG_CHAR;
	mask = UCHAR_MAX;
    }
    if (pparser->is_short)
    {
	type |= PA_FLAG_SHORT;
	mask = USHRT_MAX;
    }
    if (pparser->is_long)
    {
	type |= PA_FLAG_LONG;
	mask = ULONG_MAX;
    }
    if (pparser->is_long_double)
    {
	type |= PA_FLAG_LONG;
	mask = ~(intmax_t)0;
    }

    /* If the caller wants type info we will have an address at which
       to store that info. */

    /* Read these now to advance the argument pointer appropriately */
    fetch_width_and_prec(pparser, 0);
    return_val_if_fail(pparser->prec >= 0, SNV_ERROR);

    if (pparser->dollar == -1)
    {
	pparser->dollar = pparser->argindex++;
    }
    if (pparser->argv == NULL)
    {
	pparser->argtype(pparser, type, pparser->dollar);
	return 1;
    }

    /* Extract the correct argument from the arg vector. */
    if (type & PA_FLAG_UNSIGNED)
    {
    	value = fetch_uintmax(pparser, pparser->dollar);
	is_negative = FALSE;
    }
    else
    {
        long svalue = 0L;
	svalue = fetch_intmax(pparser, pparser->dollar);
        is_negative = (svalue < 0);
        value = (uintmax_t)ABS(svalue);
    }
   
    value &= mask;
 
    /* Convert the number into a string. */
    p = end = &buffer[sizeof(buffer) - 1];

    if (value == 0)
    {
	*p-- = '0';
    }
    else while (value > 0)
    {
	*p-- = digits[value % base];
	value /= base;
    }
    pparser->width -= end - p;
    pparser->prec  -= end - p;

    /* Octal numbers have a leading zero in alterate form. */
    if (pparser->alt
	&& base == 8 && pparser->prec <= 0 && *p != '0')
    {
	*p-- = '0';
	--pparser->width;
    }

    /* Left pad with zeros to make up the precision. */
    if (pparser->prec > 0)
    {
	pparser->width -= pparser->prec;
	while (pparser->prec-- > 0)
	{
	    *p-- = '0';
	}
    }
    
    /* Reserve room for leading `0x' for hexadecimal. */
    if (pparser->alt && base == 16)
    {
	pparser->width -= 2;
    }
    
    /* Reserve room for a sign character. */
    if (is_negative || pparser->showsign || pparser->space)
    {
	--pparser->width;
    }

    /* Left pad to the remaining width if the supplied argument is less
     * than the width specifier, and the padding character is ' '.
     */
    if (pparser->pad == ' ' && !pparser->left)
    {
	while ((count_or_errorcode >= 0)
	       && (pparser->width-- > 0))
	{
	    SNV_EMIT(pparser->pad, stream, count_or_errorcode);
	}
    }

    /* Display any sign character. */
    if (count_or_errorcode >= 0)
    {
	if (is_negative)
	{
	    SNV_EMIT('-', stream, count_or_errorcode);
	}
	else if (pparser->showsign)
	{
	    SNV_EMIT('+', stream, count_or_errorcode);
	}
	else if (pparser->space)
	{
	    SNV_EMIT(' ', stream, count_or_errorcode);
	}
    }

    /* Display `0x' for alternate hexadecimal specifier. */
    if ((count_or_errorcode >= 0) && (base == 16)
	&& pparser->alt)
    {
	SNV_EMIT('0', stream, count_or_errorcode);
	SNV_EMIT(digits['X'-'A'+10], stream, count_or_errorcode);
    }

    /* Left pad to the remaining width if the supplied argument is less
     * than the width specifier, and the padding character is not ' '.
     */
    if (pparser->pad != ' ' && !pparser->left)
    {
	while ((count_or_errorcode >= 0)
	       && (pparser->width-- > 0))
	{
	    SNV_EMIT(pparser->pad, stream, count_or_errorcode);
	}
    }

    /* Fill the stream buffer with as many characters from the number
     * buffer as possible without overflowing.
     */
    while ((count_or_errorcode >= 0) && (++p < &buffer[sizeof(buffer)]))
    {
	SNV_EMIT(*p, stream, count_or_errorcode);
    }

    /* Right pad to the width if we still didn't reach the specified
     * width and the left justify flag was set.
     */
    if (pparser->left)
    {
	while ((count_or_errorcode >= 0) && (pparser->width-- > 0))
	{
	    /* we always left justify with spaces */
	    SNV_EMIT(' ', stream, count_or_errorcode);
	}
    }

    /* Return the number of characters emitted. */
    return count_or_errorcode;
}

static int
pointer_specifier_handler (pparser, stream, spec)
    snv_parser * const pparser;
    STREAM *stream;
    const struct spec_entry *spec;
{
    int count_or_errorcode = SNV_OK;

    return_val_if_fail(pparser != NULL, SNV_ERROR);

    /* Check for valid pre-state. */
    if (!(pparser->state & (SNV_STATE_BEGIN|SNV_STATE_SPECIFIER)))
    {
	PARSER_ERROR(pparser, "out of range");
	return -1;
    }

    /* Always print 0x. */
    pparser->alt = 1;
   
    /* Read these now to advance the argument pointer appropriately */
    fetch_width_and_prec(pparser, 0);
    return_val_if_fail(pparser->prec >= 0, SNV_ERROR);

    if (pparser->dollar == -1)
    {
        pparser->dollar = pparser->argindex++;
    }

    /* Use the standard routine for numbers for the printing call,
       if the pointer is not NULL.  */
    if ((pparser->argv != NULL)
	&& (pparser->argv[pparser->dollar] != NULL))
    {
	return number_specifier_handler (pparser, stream, spec);
    }

    /* Set post-state. */
    pparser->state = SNV_STATE_END;

    /* Advance the format pointer past the parsed character. */
    pparser->format++;

    if (pparser->argv == NULL)
    {
        /* If the caller wants type info we will have an address at which
           to store that info. */

        pparser->format++;
        pparser->argtype(pparser, spec->type, pparser->dollar);
        return 1;
    }

    /* Print a NULL pointer as (nil), appropriately padded.  */
    if ((pparser->width > 5) && (!pparser->left))
    {
        int padwidth = pparser->width - 5;
        while ((count_or_errorcode >= 0)
               && (count_or_errorcode < padwidth))
        {
            SNV_EMIT(pparser->pad, stream, count_or_errorcode);
        }
    }

    SNV_EMIT('(', stream, count_or_errorcode);
    SNV_EMIT('n', stream, count_or_errorcode);
    SNV_EMIT('i', stream, count_or_errorcode);
    SNV_EMIT('l', stream, count_or_errorcode);
    SNV_EMIT(')', stream, count_or_errorcode);

    if ((pparser->width > 5) && (pparser->left))
    {
        while ((count_or_errorcode >= 0)
               && (count_or_errorcode < pparser->width))
        {
	    /* we always left justify with spaces */
            SNV_EMIT(' ', stream, count_or_errorcode);
        }
    }

    return count_or_errorcode;
}

static int
string_specifier_handler (pparser, stream, spec)
    snv_parser * const pparser;
    STREAM *stream;
    const struct spec_entry *spec;
{
    int len = 0, count_or_errorcode = SNV_OK;
    char *p = NULL;

    return_val_if_fail(pparser != NULL, SNV_ERROR);

    /* Check for valid pre-state. */
    if (!(pparser->state & (SNV_STATE_BEGIN|SNV_STATE_SPECIFIER)))
    {
	PARSER_ERROR(pparser, "out of range");
	return -1;
    }

    /* Set post-state. */
    pparser->state = SNV_STATE_END;

    /* Advance the format pointer past the parsed character. */
    pparser->format++;

    /* If the caller wants type info we will have an address at which
       to store that info. */

    /* Read these now to advance the argument pointer appropriately */
    fetch_width_and_prec(pparser, 0);
    return_val_if_fail(pparser->prec >= 0, SNV_ERROR);

    if (pparser->dollar == -1)
    {
	pparser->dollar = pparser->argindex++;
    }
    if (pparser->argv == NULL)
    {
	pparser->argtype(pparser, PA_STRING, pparser->dollar);
	return 1;
    }

    /* Extract the correct argument from the arg vector. */
    p = (char*)pparser->argv[pparser->dollar];
    
    /* Left pad to the width if the supplied argument is less than
     * the width specifier.
     */
    if (p != NULL)
    {
	len = strlen (p);
	if (pparser->prec && pparser->prec < len)
	    len = pparser->prec;
    }
    
    if ((len < pparser->width) && (!pparser->left))
    {
	int padwidth = pparser->width - len;
	while ((count_or_errorcode >= 0)
	       && (count_or_errorcode < padwidth))
	{
	    SNV_EMIT(pparser->pad, stream, count_or_errorcode);
	}
    }

    /* Fill the buffer with as many characters from the format argument
     * as possible without overflowing or exceeding the precision.
     */
    if ((count_or_errorcode >= 0) && (p != NULL))
    {
	int mark = count_or_errorcode;
	while ((count_or_errorcode >= 0) && *p != '\0'
	       && ((pparser->prec == 0) || (count_or_errorcode - mark < len)))
	{
	    SNV_EMIT(*p++, stream, count_or_errorcode);
	}
    }

    /* Right pad to the width if we still didn't reach the specified
     * width and the left justify flag was set.
     */
    if ((count_or_errorcode < pparser->width) && pparser->left)
    {
	while ((count_or_errorcode >= 0) && (count_or_errorcode < pparser->width))
	{
	    /* we always left justify with spaces */
	    SNV_EMIT(' ', stream, count_or_errorcode);
	}
    }

    /* Return the number of characters emitted. */
    return count_or_errorcode;
}

/**
 * snv_generic_specifier_handler:
 * @pparser: the current state information for the format
 * string parser.  The fields in the @pparser type can (and should)
 * be modified by functions of this type to implement the fsm which
 * parses format strings.
 * @stream: the stream (possibly a struct %printfv_stream appropriately
 * cast) on which to write output.
 * @spec: the spec_entry for the character that has just been read
 * by the parser, and can be used so that multiple specifiers are
 * implemented by a single handler.
 *
 * An example implementation of an handler, used to provide easy
 * access to justification, width and precision options.
 *
 * Return value:
 * The number of arguments eaten if %pparser->argv is %NULL, otherwise
 * the number of characters output.
 **/
int
snv_generic_specifier_handler (pparser, stream, spec)
    snv_parser * const pparser;
    STREAM *stream;
    const struct spec_entry *spec;
{
    int len = 0, count_or_errorcode = SNV_OK;
    char *p = NULL;

    /* Used to interface to the custom function. */
    STREAM out;
    filament *fil;
    printfv_function *user_func = (printfv_function *) spec->user;

    return_val_if_fail(pparser != NULL, SNV_ERROR);

    /* Check for valid pre-state. */
    if (!(pparser->state & (SNV_STATE_BEGIN|SNV_STATE_SPECIFIER)))
    {
	PARSER_ERROR(pparser, "out of range");
	return -1;
    }

    /* Set post-state. */
    pparser->state = SNV_STATE_END;

    /* Advance the format pointer past the parsed character. */
    pparser->format++;

    /* If the caller wants type info we will have an address at which
       to store that info. */

    /* Read these now to advance the argument pointer appropriately */
    fetch_width_and_prec(pparser, 0);
    return_val_if_fail(pparser->prec >= 0, SNV_ERROR);

    if (pparser->dollar == -1)
    {
	pparser->dollar = pparser->argindex++;
    }
    if (pparser->argv == NULL)
    {
	pparser->argtype(pparser, spec->type, pparser->dollar);
	return 1;
    }

    /* Print to a stream using a user-supplied function. */
    fil = filnew(NULL, 0);
    stream_init(&out, (stream_gpointer)fil, SNV_UNLIMITED, NULL, filputc);
    user_func (pparser, &out, spec);
    len = fillen(fil);
    p = fildelete(fil);
    
    /* Left pad to the width if the supplied argument is less than
     * the width specifier.
     */
    if (p != NULL && pparser->prec && pparser->prec < len)
    {
	len = pparser->prec;
    }
    
    if ((len < pparser->width) && (!pparser->left))
    {
	int padwidth = pparser->width - len;
	while ((count_or_errorcode >= 0)
	       && (count_or_errorcode < padwidth))
	{
	    SNV_EMIT(pparser->pad, stream, count_or_errorcode);
	}
    }

    /* Fill the buffer with as many characters from the format argument
     * as possible without overflowing or exceeding the precision.
     */
    if ((count_or_errorcode >= 0) && (p != NULL))
    {
	int mark = count_or_errorcode;
	while ((count_or_errorcode >= 0) && *p != '\0'
	       && ((pparser->prec == 0) || (count_or_errorcode - mark < len)))
	{
	    SNV_EMIT(*p++, stream, count_or_errorcode);
	}
    }

    /* Right pad to the width if we still didn't reach the specified
     * width and the left justify flag was set.
     */
    if ((count_or_errorcode < pparser->width) && pparser->left)
    {
	while ((count_or_errorcode >= 0) && (count_or_errorcode < pparser->width))
	{
	    /* we always left justify with spaces */
	    SNV_EMIT(' ', stream, count_or_errorcode);
	}
    }

    /* Return the number of characters emitted. */
    return count_or_errorcode;
}

/* format.c ends here */
