/*  -*- Mode: C -*-  */

/* snprintfv.c --- printf clone for argv arrays */

/* Author:	       Gary V. Vaughan <gary@oranda.demon.co.uk>
 * Maintainer:	       Gary V. Vaughan <gary@oranda.demon.co.uk>
 * Created:	       Fri Nov 13 11:23:23 1998
 * Last Modified:      Wed Jun 26 12:36:31 2002
 *            by:      Paolo Bonzini <bonzini@gnu.org>
 * ---------------------------------------------------------------------
 * @(#) $Id$
 * ---------------------------------------------------------------------
 */

/* Copyright (C) 1998, 1999, 2000, 2002 Gary V. Vaughan */

/* This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * As a special exception to the GNU General Public License, if you
 * distribute this file as part of a program that also links with and
 * uses the libopts library from AutoGen, you may include it under
 * the same distribution terms used by the libopts library.
 */

/* Code: */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <unistd.h>		/* for the write(2) call */

#define COMPILING_SNPRINTFV_C
#include "snprintfv.h"
#undef COMPILING_SNPRINTFV_C

#include "filament.h"
#include "stream.h"
#include "mem.h"

#define EOS			'\0'
#define SNV_CHAR_SPEC		'%'
#define SNV_ESC_SPEC		'\\'

/* We deliberately don't prototype the malloc functions;  they are cast
   to match the function pointers we expose to avoid compiler warnings
   from mismatched prototypes (if we find a host implementation.

   Not also that if this file is compiled -DWITH_DMALLOC, the inclusion
   in mem.h will cause the malloc references below to be redirected
   correctly. */
snv_pointer (*snv_malloc) PARAMS((size_t count)) 
		= (snv_pointer(*)PARAMS((size_t)))malloc;

snv_pointer (*snv_realloc) PARAMS((snv_pointer old, size_t count)) 
		= (snv_pointer(*)PARAMS((snv_pointer, size_t)))realloc;

void (*snv_free) PARAMS((snv_pointer old))
		= (void(*)PARAMS((snv_pointer old)))free;
char* (*snv_strdup) PARAMS((const char *str))
		= (char*(*)PARAMS((const char*)))strdup;


/* Functions to manage mapping of spec chars to handlers. */
static	unsigned spec_hash	PARAMS((unsigned spec));
static	void	spec_init	PARAMS((void));
static	spec_entry *spec_lookup	PARAMS((unsigned spec));
static	void	spec_insert 	PARAMS((spec_entry *pentry));

/* Convert a variadic va_list to an arg vector. */
static snv_constpointer *va_list_to_argv PARAMS((const char *format,
				   va_list ap));

/* FIXME:  We are assuming an ASCII character set where all the
           printable characters are between SPACE and DEL. */
#define ASCII_DEL	(int)''
#define ASCII_SPACE	(int)' '

/* TODO:  This is not thread-safe.  Change the API to pass the spec_table
          in as the first parameter to the functions which use it. */
static spec_entry *spec_table[ASCII_DEL - ASCII_SPACE];

static inline unsigned
spec_hash (spec)
    unsigned spec;
{
    return (spec & ASCII_DEL) - ASCII_SPACE;
}

/* Register all of the functions in INIT_SPEC_TABLE. */
static void
spec_init PARAMS((void))
{
    static boolean is_init = FALSE;
    
    if (!is_init)
    {
	extern spec_entry snv_default_spec_table[];
	unsigned index = 0;

	memset(spec_table, 0, sizeof(spec_table));
	while (snv_default_spec_table[index].spec != EOS)
	{
	    unsigned hash = spec_hash(snv_default_spec_table[index].spec);
	    spec_table[hash] = snv_default_spec_table + index;
	    index++;
	}

	is_init = TRUE;
    }
}

/* Insert PENTRY, a new handler, into SPEC_TABLE. */
static inline void
spec_insert (pentry)
    spec_entry *pentry;
{
    unsigned hash = spec_hash(pentry->spec);
    spec_init();
    spec_table[hash] = pentry;
}

/* Lookup and return the SPEC_TABLE entry for SPEC. */
static inline spec_entry*
spec_lookup (spec)
    unsigned spec;
{
    unsigned hash = spec_hash(spec);
    spec_init();
    return spec_table[hash];
}

/**
 * snv_register_specifier: snprintfv.h
 * @spec: the character which will trigger @func, cast to an unsigned int.
 * @type: the type tag which will be returned by the format parser.
 * @func: the handler function.
 * @user: an arbitrary pointer which is passed to @func through a %spec_entry struct.
 * 
 * Register @func to be called when @spec is encountered in a format
 * string.
 * 
 * Return value:
 * Returns %SNV_OK if @func was successfully registered, %SNV_ERROR
 * otherwise.
 **/
int
snv_register_specifier (spec, type, func, user)
    unsigned spec;
    int type;
    printfv_function *func;
    snv_pointer user;
{
    spec_entry *new, *old;
    old = spec_lookup (spec);
    if (old && !old->overridable)
    {
        return SNV_ERROR;
    }

    new              =  snv_new(spec_entry, 1);
    new->spec        = spec;
    new->overridable = TRUE;
    new->type        = type;
    new->func        = func;
    new->user        = user;

    spec_insert(new);

    return SNV_OK;
}

static void
argtype_maybe_store (pparser, type, argindex)
    snv_parser * const pparser;
    int type;
    int argindex;
{
    pparser->argc = MAX(pparser->argc, 1+ argindex);
    if (1+argindex < pparser->argsize)
    {
        pparser->argtypes[argindex] = type;
    }
}

static void
argtype_renew (pparser, type, argindex)
    snv_parser * const pparser;
    int type;
    int argindex;
{
    /* This helper function will ensure that PARGTYPES has been allocated
       enough elements to accomodate ARGINDEX by reallocating if necessary.
       PARGTYPES can be a pointer to a NULL pointer when calling this
       function.
 
       FIXME: Don't blow your load so early!  systematically extend
              pargtypes and check for type mismatches on multiple
	      parameter uses. */
    
    pparser->argc = pparser->argsize = MAX(pparser->argc, 1+ argindex);
    
    if (pparser->argtypes == NULL)
    {
	/* No elements have been added to the type vector yet,
	   if the first element added is not element zero (i.e. from
	   a dollar specifier) we may need to add several elements. */
	pparser->argtypes = snv_new(int, pparser->argc);
    }
    else
    {
	pparser->argtypes = snv_renew(int, pparser->argtypes, pparser->argc);
    }

    pparser->argtypes[argindex] = type;
}


/* (re)initialise the memory used by PPARSER. */
static inline snv_parser*
parser_init (pparser, format, argv)
    snv_parser *pparser;
    const char *format;
    snv_constpointer * const argv;
{
    memset (pparser, 0, sizeof(snv_parser));
    pparser->format = format;
    pparser->argv = argv;
}

static inline snv_parser*
parser_reset (pparser)
    snv_parser *pparser;
{
    pparser->is_long_double = pparser->is_char = pparser->is_short =
    pparser->is_long = pparser->alt = pparser->space = pparser->left =
    pparser->showsign = pparser->group = pparser->wide = pparser->extra =
    pparser->dollar = pparser->width = pparser->spec = 0;

    pparser->state = SNV_STATE_BEGIN;
    pparser->prec = -1;
    pparser->dollar = -1;
    pparser->pad = ' ';

    return pparser;
}

/**
 * parser_error: snprintfv.h
 * @pparser: pointer to the current parser state.
 * @error_message: new error message to append to @pparser.
 * 
 * The contents of @error_message are appended to the @pparser internal
 * error string, so it is safe to pass static strings or recycle the
 * original when this function returns.
 * 
 * Return value:
 * The address of the full accumulated error message in @pparser is
 * returned.
 **/
char*
parser_error (pparser, error_message)
    snv_parser *pparser;
    const char *error_message;
{
    if (pparser->error == NULL)
    {
	pparser->error = filnew(NULL, 0);
    }

    return filcat(pparser->error, error_message);
}


static snv_constpointer*
va_list_to_argv (format, ap)
    const char *format;
    va_list ap;
{
    snv_constpointer *args = NULL;
    snv_parser parser;

    return_val_if_fail (format != NULL, NULL);

    parser_init(&parser, format, NULL);
    parser.argtype = argtype_renew;
    while (!(*parser.format == EOS || parser.count < 0))
    {
	int ch = (int)*parser.format++;

	switch (ch)
	{
	case SNV_CHAR_SPEC:
	    if (*parser.format != SNV_CHAR_SPEC)
	    {
		/* We found the start of a format specifier! */

		parser_reset(&parser);
		while (parser.count >= 0
		       && parser.state != SNV_STATE_END)
		{
		    /* Until we fill the stream (or ger some other
		       exception) or one of the handlers tells us
		       we have reached the end of the specifier... */
		    spec_entry *data;
		    int status;

		    /* ...lookup the handler associated with the char
		       we are looking at in the format string... */
		    data = spec_lookup(*parser.format);
		    if (data == NULL)
		    {
			PARSER_ERROR(&parser, "unregistered specifier");
			return NULL;
		    }

		    /* ...and call the relevant handler. Notice that there
		       is no stream specified (i.e. we don't want output)
		       and that we have specified somewere to store the
		       argtypes. */
		    parser.spec = *parser.format;
		    parser.extra = 0;
		    status = (*data->func)(&parser, NULL, data);
		    
		    parser.count = (status < 0) ? status
			: parser.count + status;

		}
#if 0
		parser_delete(parser);
#endif
		break;
	    }
	    /* An escaped CHAR_SPEC: ignore it (by falling through). */
	    ++parser.format;
	    /*NOBREAK*/

	default: /* Just a character: ignore it. */
	    break;
	}
    }

    if (parser.argc > 0)
    {
	int index = 0;

	args = snv_new(snv_constpointer, parser.argc);
	while (index < parser.argc
	       && parser.count >= 0)
	{
	    int argtype = parser.argtypes[index];

	    /* Scan the format string to find the type of the argument
	       so we can cast it and store it correctly.

	       Note that according to the ISO C standards, standard
	       type promotion takes place on any variadic arguments as
	       they are aligned on the call stack, and so it is these
	       promoted types that we must extract with the va_arg()
	       macro, or the alignment gets all messed up.

	       Thanks to Robert Lipe <robertlipe@usa.net> for explaining all
	       this to me. */
	    switch(argtype & (PA_TYPE_MASK | PA_FLAG_PTR))
	    {
	    case PA_INT:
		if (argtype & PA_FLAG_LONG)
		{
		    args[index] = (snv_constpointer)snv_new(long, 1);
		    *(long*)args[index] = va_arg(ap, long);
		}
		else if (argtype & PA_FLAG_LONG_LONG)
		{
		    args[index] = (snv_constpointer)snv_new(intmax_t, 1);
		    *(intmax_t*)args[index] = va_arg(ap, intmax_t);
		}
		else
		{
		    args[index] = SNV_INT_TO_POINTER(va_arg(ap, int));
		}
		break;
	    case PA_FLOAT:
	    case PA_DOUBLE:
		if (argtype & PA_FLAG_LONG_DOUBLE)
		{
		    args[index] = (snv_constpointer)snv_new(long double, 1);
		    *(long double*)args[index] = va_arg(ap, long double);
		}
		else
		{
		    args[index] = (snv_constpointer)snv_new(double, 1);
		    *(double*)args[index] = va_arg(ap, double);
		}
		break;
	    case PA_WCHAR:
	    case PA_CHAR:
		args[index] =
		    (snv_constpointer)SNV_INT_TO_POINTER(va_arg(ap, int));
		break;

	    default:
		/* This is a pointer (either a PA_POINTER or a PA_FLAG_PTR
		   or something > PA_LAST). */
		args[index] = va_arg(ap, snv_constpointer);
		break;
	    }
	    index++;
	}
    }
    
    /* We don't save this type information, because it is already encoded
       in the format string, and when the handlers use the arg vector they
       will use the format string to work out type information (as they did
       when called in the first pass to give us this information). */
    if (parser.argtypes != NULL)
    {
	snv_delete(parser.argtypes);
    }
    
    if (parser.error)
    {
	snv_delete(fildelete(parser.error));
    }
    
    return args;
}


/**
 * parse_printf_format: snprintfv.h
 * @format: a % delimited format string.
 * @n: the size of the @argtypes vector
 * @argtypes: a vector of ints, to be filled with the argument types from @format
 * 
 * Returns information about the number and types of
 * arguments expected by the template string @format.
 * The argument @n specifies the number of elements in the array
 * @argtypes.  This is the maximum number of elements that
 * the function will try to write.
 *
 * Return value:
 * The total number of arguments required by @format.  If this
 * number is greater than @n, then the information returned
 * describes only the first @n arguments.  If you want information
 * about additional arguments, allocate a bigger array and call
 * this function again. If there is an error, then %SNV_ERROR
 * is returned instead.
 **/
size_t
parse_printf_format (format, n, argtypes)
    const char *format;
    int n;
    int *argtypes;
{
    snv_parser parser;

    return_val_if_fail (format != NULL, -1);

    parser_init(&parser, format, NULL);
    parser.argtype = argtype_maybe_store;
    parser.argtypes = argtypes;
    parser.argsize = n;

    while (!(*parser.format == EOS || parser.count < 0))
    {
	int ch = (int)*parser.format++;

	switch (ch)
	{
	case SNV_CHAR_SPEC:
	    if (*parser.format != SNV_CHAR_SPEC)
	    {
		/* We found the start of a format specifier! */

		parser_reset(&parser);
		while (parser.count >= 0
		       && parser.state != SNV_STATE_END)
		{
		    /* Until we fill the stream (or get some other
		       exception) or one of the handlers tells us
		       we have reached the end of the specifier... */
		    spec_entry *data;
		    int status;

		    /* ...lookup the handler associated with the char
		       we are looking at in the format string... */
		    data = spec_lookup(*parser.format);
		    if (data == NULL)
		    {
			PARSER_ERROR(&parser, "unregistered specifier");
			return -1;
		    }

		    /* ...and call the relevant handler. Notice that there
		       is no stream specified (i.e. we don't want output)
		       and that we have specified somewere to store the
		       argtypes. */
		    parser.spec = *parser.format;
		    parser.extra = 0;
		    status = (*data->func)(&parser, NULL, data);
		    
		    parser.count = (status < 0) ? status
			: parser.count + status;

		}
#if 0
		parser_delete(parser);
#endif
		break;
	    }
	    /* An escaped CHAR_SPEC: ignore it (by falling through). */
	    ++parser.format;
	    /*NOBREAK*/

	default: /* Just a character: ignore it. */
	    break;
	}
    }

    if (parser.error)
    {
	snv_delete(fildelete(parser.error));
    }
    
    return parser.count < 0 ? parser.count : parser.argc;
}

/**
 * stream_printfv: snprintfv.h
 * @stream: an initialised stream structure.
 * @format: a % delimited format string.
 * @args: a vector of argument addresses to match @format.
 * 
 * Format the elements of @args according to @format, and write
 * the results to @stream.  If @stream is %NULL, only count the
 * number of characters needed to output the format.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
int
stream_printfv (stream, format, args)
    STREAM *stream;
    const char *format;
    snv_constpointer const args[];
{
    snv_parser parser;

    /* This is the parser driver.

       Here we scan through the format string and move bytes into the
       stream and call handlers based on the parser state. */

    parser_init(&parser, format, args);

    /* Keep going until the format string runs out! */
    while (!(*parser.format == EOS || parser.count < 0))
    {
	int ch = (int)*parser.format++;

	switch (ch)
	{
	case SNV_CHAR_SPEC:
	    if (*parser.format != SNV_CHAR_SPEC)
	    {
		/* We found the start of a format specifier! */

		parser_reset(&parser);
		while (parser.count >= 0
		       && parser.state != SNV_STATE_END)
		{
		    /* Until we fill the stream (or get some other
		       exception) or one of the handlers tells us
		       we have reached the end of the specifier... */
		    spec_entry *data;
		    int status;

		    /* ...lookup the handler associated with the char
		       we are looking at in the format string... */
		    data = spec_lookup(*parser.format);
		    if (data == NULL)
		    {
			PARSER_ERROR(&parser, "unregistered specifier");
			return -1;
		    }

		    /* ...and call the relevant handler. */
		    parser.spec = *parser.format;
		    parser.extra = 0;
		    status = (*data->func)(&parser, stream, data);
		
		    parser.count = (status < 0) ? status
			: parser.count + status;

		}
#if 0
		parser_delete(parser);
#endif
		break;
	    }
	    /* An escaped CHAR_SPEC: copy it (by falling through). */
	    ++parser.format;
	    /*NOBREAK*/

	default: /* Just a character: copy it. */
	{
	    SNV_EMIT(ch, stream, parser.count);
	    break;
	}
	}
    }

    if (parser.error)
    {
	snv_delete(fildelete(parser.error));
    }
    
    return parser.count;
}

/**
 * stream_vprintf: snprintfv.h
 * @stream: an initialised stream structure.
 * @format: a % delimited format string.
 * @ap: a varargs/stdargs va_list.
 * 
 * Format the elements of @ap according to @format, and write
 * the results to @stream.  If @stream is %NULL, only count the
 * number of characters needed to output the format.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
int
stream_vprintf (stream, format, ap)
    STREAM *stream;
    const char *format;
    va_list ap;
{
    snv_constpointer *args = va_list_to_argv(format, ap);
    int count_or_errorcode = stream_printfv(stream, format, args);

    /* FIXME: leaks memory for pointers to malloced mem. */
    if (args) snv_delete(args);
    return count_or_errorcode;
}

/**
 * stream_printf: snprintfv.h
 * @stream: an initialised stream structure.
 * @format: a % delimited format string.
 * @va_alist: a varargs/stdargs va_list.
 * 
 * Format the elements of @va_alist according to @format, and write
 * the results to @stream.  If @stream is %NULL, only count the
 * number of characters needed to output the format.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
int
#ifdef SNV_USING_STDARG_H
stream_printf (STREAM *stream, const char *format, ...)
#else    
stream_printf (stream, format, va_alist)
    STREAM *stream;
    const char *format;
    va_dcl
#endif
{
    int count_or_errorcode;
    va_list ap;

    VA_START(ap, format);
    count_or_errorcode = stream_vprintf(stream, format, ap);
    VA_END(ap);
    
    return count_or_errorcode;
}


/*  Finally... the main API implementation: */

/**
 * fdputc: snprintfv.h
 * @ch: A single character to be added to @stream.
 * @stream: The stream in which to write @ch.
 * 
 * A stream_put_function function for use in putting characters
 * into STREAMs holding a file descriptor.
 * 
 * Return value:
 * The value of @ch that has been put in @stream, or -1 in case of
 * an error (errno will be set to indicate the type of error).
 **/
int
fdputc (ch, stream)
    int ch;
    STREAM *stream;
{
    static char buf[2] = "c";
    *buf = (char)ch;
    return write(SNV_POINTER_TO_INT(stream->stream), buf, 1) ? ch : -1;
}

/**
 * snv_dprintf: snprintfv.h
 * @fd: an open file descriptor.
 * @format: a % delimited format string.
 * @va_alist: a varargs/stdargs va_list.
 * 
 * Format the elements of @va_alist according to @format, and write
 * the results to the file descriptor @fd.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
int
#ifdef SNV_USING_STDARG_H
snv_dprintf (int fd, const char *format, ...)
#else
snv_dprintf (fd, format, va_alist)
    int fd;
    const char *format;
    va_dcl
#endif
{
    int count_or_errorcode;
    va_list ap;

    VA_START(ap, format);
    count_or_errorcode = snv_vdprintf(fd, format, ap);
    VA_END(ap);
    
    return count_or_errorcode;
}

/**
 * snv_vdprintf: snprintfv.h
 * @fd: an open file descriptor.
 * @format: a % delimited format string.
 * @ap: a varargs/stdargs va_list.
 * 
 * Format the elements of @ap according to @format, and write
 * the results to the file descriptor @fd.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
int
snv_vdprintf (fd, format, ap)
    int fd;
    const char *format;
    va_list ap;
{
    STREAM out;
    stream_init(&out, (stream_gpointer)SNV_INT_TO_POINTER(fd),
		SNV_UNLIMITED, NULL, fdputc);

    return stream_vprintf(&out, format, ap);
}    

/**
 * dprintfv: snprintfv.h
 * @fd: an open file descriptor.
 * @format: a % delimited format string.
 * @args: a vector of argument addresses to match @format.
 * 
 * Format the elements of @args according to @format, and write
 * the results to file descriptor @fd.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
int
dprintfv (fd, format, args)
    int fd;
    const char *format;
    snv_constpointer const args[];
{
    STREAM out;			/* allocated in this stack frame */
    stream_init(&out, (stream_gpointer)SNV_INT_TO_POINTER(fd),
		SNV_UNLIMITED, NULL, fdputc);
    return stream_printfv(&out, format, args);
}
    

/**
 * fileputc: snprintfv.h
 * @ch: A single character to be added to @stream.
 * @stream: The stream in which to write @ch.
 * 
 * A stream_put_function function for use in putting characters
 * into STREAMs holding a FILE*.
 * 
 * Return value: 
 * The value of @ch that has been put in @stream.
 **/
int
fileputc (ch, stream)
    int ch;
    STREAM *stream;
{
    return fputc(ch, (FILE*)stream->stream);
}

/**
 * snv_printf: snprintfv.h
 * @format: a % delimited format string.
 * @va_alist: a varargs/stdargs va_list.
 * 
 * Format the elements of @va_alist according to @format, and write
 * the results to the standard output stream.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
int
#ifdef SNV_USING_STDARG_H
snv_printf (const char *format, ...)
#else
snv_printf (format, va_alist)
    const char *format;
    va_dcl
#endif
{
    int count_or_errorcode;
    va_list ap;

    VA_START(ap, format);
    count_or_errorcode = snv_vprintf(format, ap);
    VA_END(ap);
    
    return count_or_errorcode;
}

/**
 * snv_vprintf: snprintfv.h
 * @format: a % delimited format string.
 * @ap: a varargs/stdargs va_list.
 * 
 * Format the elements of @ap according to @format, and write
 * the results to the standard output stream.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
int
snv_vprintf (format, ap)
    const char *format;
    va_list ap;
{
    STREAM out;
    stream_init(&out, (stream_gpointer)stdout, SNV_UNLIMITED, NULL,
		fileputc);

    return stream_vprintf(&out, format, ap);
}

/**
 * printfv: snprintfv.h
 * @format: a % delimited format string.
 * @args: a vector of argument addresses to match @format.
 * 
 * Format the elements of @args according to the string @format,
 * and write the result to the standard output stream.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
int
printfv (format, args)
    const char *format;
    snv_constpointer const args[];
{
    STREAM out;			/* allocated in this stack frame */
    stream_init(&out, (stream_gpointer)stdout, SNV_UNLIMITED, NULL,
		fileputc);
    return stream_printfv(&out, format, args);
}

/**
 * snv_fprintf: snprintfv.h
 * @file: a stdio.h FILE* stream.
 * @format: a % delimited format string.
 * @va_alist: a varargs/stdargs va_list.
 * 
 * Format the elements of @va_alist according to @format, and write
 * the results to the @file stream.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
int
#ifdef SNV_USING_STDARG_H
snv_fprintf (FILE *file, const char *format, ...)
#else
snv_fprintf (file, format, va_alist)
    FILE *file;
    const char *format;
    va_dcl
#endif
{
    int count_or_errorcode;
    va_list ap;

    VA_START(ap, format);
    count_or_errorcode = snv_vfprintf(file, format, ap);
    VA_END(ap);
    
    return count_or_errorcode;
}

/**
 * snv_vfprintf: snprintfv.h
 * @file: a stdio.h FILE* stream.
 * @format: a % delimited format string.
 * @ap: a varargs/stdargs va_list.
 * 
 * Format the elements of @ap according to @format, and write
 * the results to the @file stream.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
int
snv_vfprintf (file, format, ap)
    FILE *file;
    const char *format;
    va_list ap;
{
    STREAM out;
    stream_init(&out, (stream_gpointer)file, SNV_UNLIMITED, NULL, fileputc);

    return stream_vprintf(&out, format, ap);
}    

/**
 * fprintfv: snprintfv.h
 * @file: a stdio.h FILE* stream.
 * @format: a % delimited format string.
 * @args: a vector of argument addresses to match @format.
 * 
 * Format the elements of @args according to @format, and write
 * the results to @file.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
int
fprintfv (file, format, args)
    FILE *file;
    const char *format;
    snv_constpointer const args[];
{
    STREAM out;			/* allocated in this stack frame */
    stream_init(&out, (stream_gpointer)file, SNV_UNLIMITED, NULL,
		fileputc);
    return stream_printfv(&out, format, args);
}


/**
 * bufputc: snprintfv.h
 * @ch: A single character to be added to @stream.
 * @stream: The stream in which to write @ch.
 * 
 * A stream_put_function function for use in putting characters
 * into STREAMs holding a char buffer.
 * 
 * Return value:
 * The value of @ch that has been put in @stream.
 **/
int
bufputc (ch, stream)
    int ch;
    STREAM *stream;
{
    char **ppbuffer = (char**)stream->stream;
    **ppbuffer = (char)ch;
    (*ppbuffer)++;
    return ch;
}

/**
 * snv_sprintf: snprintfv.h
 * @buffer: a preallocated char* buffer.
 * @format: a % delimited format string.
 * @va_alist: a varargs/stdargs va_list.
 * 
 * Format the elements of @va_alist according to @format, and write
 * the results to the string @buffer.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
int
#ifdef SNV_USING_STDARG_H
snv_sprintf (char buffer[], const char *format, ...)
#else
snv_sprintf (buffer, format, va_alist)
    char buffer[];
    const char *format;
    va_dcl
#endif
{
    int count_or_errorcode;
    va_list ap;

    VA_START(ap, format);
    count_or_errorcode = snv_vsprintf(buffer, format, ap);
    VA_END(ap);

    return count_or_errorcode;
}

/**
 * snv_vsprintf: snprintfv.h
 * @buffer: a preallocated char* buffer.
 * @format: a % delimited format string.
 * @ap: a varargs/stdargs va_list.
 * 
 * Format the elements of @ap according to @format, and write
 * the results to the string @buffer.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
int
snv_vsprintf (buffer, format, ap)
    char buffer[];
    const char *format;
    va_list ap;
{
    int count_or_errorcode;
    STREAM out;
    stream_init(&out, (stream_gpointer)&buffer, SNV_UNLIMITED, NULL, bufputc);
    count_or_errorcode = stream_vprintf(&out, format, ap);

    /* Bah.  This is messy because we need to explicity put an end-of-string
       marker at the end of the real output. */
    if (count_or_errorcode >= 0)
    {
        /* Terminate with an EOS without incrementing the counter. */
        count_or_errorcode = (stream_put(EOS, &out) != EOF)
	                   ? count_or_errorcode : EOF;
    }

    return count_or_errorcode;
}    

/**
 * sprintfv: snprintfv.h
 * @buffer: a preallocated char* buffer.
 * @format: a % delimited format string.
 * @args: a vector of argument addresses to match @format.
 * 
 * Format the elements of @args according to @format, and write
 * the results to the string @buffer.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
int
sprintfv (buffer, format, args)
    char buffer[];
    const char *format;
    snv_constpointer const args[];
{
    int count_or_errorcode;
    STREAM out;			/* allocated in this stack frame */
    stream_init(&out, (stream_gpointer)&buffer, SNV_UNLIMITED, NULL, bufputc);
    count_or_errorcode = stream_printfv(&out, format, args);

    /* Bah.  This is messy because we need to explicity put an end-of-string
       marker at the end of the real output. */
    if (count_or_errorcode >= 0)
    {
        /* Terminate with an EOS without incrementing the counter. */
        count_or_errorcode = (stream_put(EOS, &out) != EOF)
	                   ? count_or_errorcode : EOF;
    }

    return count_or_errorcode;
}

/**
 * snv_snprintf: snprintfv.h
 * @buffer: a preallocated char* buffer.
 * @limit: the maximum number of characters to write into @buffer.
 * @format: a % delimited format string.
 * @va_alist: a varargs/stdargs va_list.
 * 
 * Format the elements of @va_alist according to @format, and write
 * the results to the string @buffer, truncating the formatted string
 * if it reaches @limit characters in length.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
int
#ifdef SNV_USING_STDARG_H
snv_snprintf (char buffer[], unsigned long limit, const char *format, ...)
#else
snv_snprintf (buffer, limit, format, va_alist)
    char buffer[];
    unsigned long limit;
    const char *format;
    va_dcl
#endif
{
    int count_or_errorcode;
    va_list ap;

    VA_START(ap, format);
    count_or_errorcode = snv_vsnprintf(buffer, limit, format, ap);
    VA_END(ap);

    return count_or_errorcode;
}

/**
 * snv_vsnprintf: snprintfv.h
 * @buffer: a preallocated char* buffer.
 * @limit: the maximum number of characters to write into @buffer.
 * @format: a % delimited format string.
 * @ap: a varargs/stdargs va_list.
 * 
 * Format the elements of @ap according to @format, and write
 * the results to the string @buffer, truncating the formatted string
 * if it reaches @limit characters in length.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
int
snv_vsnprintf (buffer, limit, format, ap)
    char buffer[];
    unsigned long limit;
    const char *format;
    va_list ap;
{
    int count_or_errorcode;
    STREAM out;
    stream_init(&out, (stream_gpointer)&buffer, limit, NULL, bufputc);
    count_or_errorcode = stream_vprintf(&out, format, ap);

    /* Bah.  This is messy because we need to explicity put an end-of-string
       marker at the end of the real output. */
    if (count_or_errorcode >= 0)
    {
        /* Terminate with an EOS without incrementing the counter. */
        count_or_errorcode = (stream_put(EOS, &out) != EOF)
	                   ? count_or_errorcode : EOF;
    }

    return count_or_errorcode;
}    

/**
 * snprintfv: snprintfv.h
 * @buffer: a preallocated char* buffer.
 * @limit: the maximum number of characters to write into @buffer.
 * @format: a % delimited format string.
 * @args: a vector of argument addresses to match @format.
 * 
 * Format the elements of @args according to @format, and write
 * the results to the string @buffer, truncating the formatted string
 * if it reaches @limit characters in length.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
int
snprintfv (buffer, limit, format, args)
    char buffer[];
    unsigned long limit;
    const char *format;
    snv_constpointer const args[];
{
    int count_or_errorcode;
    STREAM out;
    stream_init(&out, (stream_gpointer)&buffer, limit, NULL, bufputc);
    count_or_errorcode = stream_printfv(&out, format, args);

    /* Bah.  This is messy because we need to explicity put an end-of-string
       marker at the end of the real output. */
    if (count_or_errorcode >= 0)
    {
        /* Terminate with an EOS without incrementing the counter. */
        count_or_errorcode = (stream_put(EOS, &out) != EOF)
	                   ? count_or_errorcode : EOF;
    }

    return count_or_errorcode;
}


/**
 * filputc: snprintfv.h
 * @ch: A single character to be added to @stream.
 * @stream: The stream in which to write @ch.
 * 
 * A stream_put_function function for use in putting characters
 * into STREAMs holding a filament*.
 * 
 * Return value: 
 * The value of @ch that has been put in @stream.
 **/
int
filputc (ch, stream)
    int ch;
    STREAM *stream;
{
    return filccat((filament*)stream->stream, ch), ch;
}

/**
 * snv_asprintf: snprintfv.h
 * @result: the address of a char * variable.
 * @format: a % delimited format string.
 * @va_alist: a varargs/stdargs va_list.
 * 
 * Format the elements of @va_alist according to @format, and write
 * the results to an internally allocated buffer whose address is
 * stored in @result (and should be freed by the caller) unless
 * there is an error.
 *
 * Yes, this interface is cumbersome and totally useless.  It would
 * have been better to simply return the allocated address, but
 * it turns out that somebody wasn't thinking much when adding 
 * asprintf to libiberty a few years ago.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
int
#ifdef SNV_USING_STDARG_H
snv_asprintf (char **result, const char *format, ...)
#else
snv_asprintf (result, format, va_alist)    
    char **result;
    const char *format;
    va_dcl
#endif
{
    int count;
    va_list ap;

    VA_START(ap, format);
    count = snv_vasprintf(result, format, ap);
    VA_END(ap);

    return count;
}

/**
 * snv_vasprintf: snprintfv.h
 * @result: the address of a char * variable.
 * @format: a % delimited format string.
 * @ap: a varargs/stdargs va_list.
 * 
 * Format the elements of @ap according to @format, and write
 * the results to an internally allocated buffer whose address is
 * stored in @result (and should be freed by the caller) unless
 * there is an error.
 * 
 * Above moaning for asprintf applies here too.
 *
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
int
snv_vasprintf (result, format, ap)
    char **result;
    const char *format;
    va_list ap;
{
    int count_or_errorcode;
    char *base;
    STREAM out;
    filament *fil = filnew(NULL, 0);
    stream_init(&out, (stream_gpointer)fil, SNV_UNLIMITED, NULL, filputc);
    count_or_errorcode = stream_vprintf(&out, format, ap);

    /* Be careful to recycle this -- it is not allocated on the stack
       like the stream which holds its address */
    base = fildelete(fil);

    *result = (count_or_errorcode < 0) ? NULL : base;
    return count_or_errorcode;
}

/**
 * asprintfv: snprintfv.h
 * @result: the address of a char * variable.
 * @format: a % delimited format string.
 * @args: a vector of argument addresses to match @format.
 * 
 * Format the elements of @args according to @format, and write
 * the results to an internally allocated buffer whose address is
 * stored in @result (and should be freed by the caller) unless
 * there is an error.
 * 
 * Above moaning for asprintf applies here too.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
int
asprintfv (result, format, args)
    char **result;
    const char *format;
    snv_constpointer const args[];
{
    int count_or_errorcode;
    char *base;
    STREAM out;
    filament *fil = filnew(NULL, 0);
    stream_init(&out, (stream_gpointer)fil, SNV_UNLIMITED, NULL, filputc);
    count_or_errorcode = stream_printfv(&out, format, args);

    /* Be careful to recycle this -- it is not allocated on the stack
       like the stream which holds its address */
    base = fildelete(fil);

    *result = (count_or_errorcode < 0) ? NULL : base;
    return count_or_errorcode;
}

/* snprintfv.c ends here */
