/*  -*- Mode: C -*-  */

/* snprintfv.h --- printf clone for argv arrays */

/* Author:	       Gary V. Vaughan <gary@oranda.demon.co.uk>
 * Maintainer:	       Gary V. Vaughan <gary@oranda.demon.co.uk>
 * Created:	       Fri Nov 13 16:51:38 1998
 * Last Modified:      Wed Jun 26 12:36:31 2002
 *            by:      Paolo Bonzini <bonzini@gnu.org>
 * ---------------------------------------------------------------------
 * @(#) $Id$
 * ---------------------------------------------------------------------
 */

/* Copyright (C) 1998, 1999, 2002 Gary V. Vaughan */

/* This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * As a special exception to the GNU General Public License, if you
 * distribute this file as part of a program that also links with and
 * uses the libopts library from AutoGen, you may include it under
 * the same distribution terms used by the libopts library.
 */

/* Code: */

#ifndef SNPRINTFV_SNPRINTFV_H
#define SNPRINTFV_SNPRINTFV_H 1

#include <snprintfv/compat.h>
#include <snprintfv/filament.h>
#include <snprintfv/stream.h>

#ifdef WITH_DMALLOC
#  include <dmalloc.h>
#endif

#ifdef __cplusplus
extern "C" {
#endif

/* The type of each element in the spec_table exported by the fsm. */
struct spec_entry;

typedef enum {
    SNV_ERROR = -1,
    SNV_OK
} snv_status;

/* Basic states required by the parser.  On initialisation the parser
   will be in SNV_STATE_BEGIN, and tokens will be parsed by the registered
   functions until the parser reached SNV_STATE_END. */
#define SNV_STATE_BEGIN		1
#define SNV_STATE_END		0

/* States needed to support:
   %[<number>$]<flags>[<width>|\*][.<precision>|\*]<modifiers><specifier> */
#define SNV_STATE_FLAG          (1 << 1)
#define SNV_STATE_WIDTH         (1 << 2)
#define SNV_STATE_PRECISION     (1 << 3)
#define SNV_STATE_MODIFIER      (1 << 4)
#define SNV_STATE_SPECIFIER     (1 << 5)

/* First state available to the user */
#define SNV_STATE_USER_FIRST    (1 << 8)

/* Mask for states available to the user */
#define SNV_STATE_USER_MASK     ~(SNV_STATE_USER_FIRST - 1)

typedef struct printf_info {
    int count;			/* accumulated count, or SNV_ERROR */
    int state;			/* one of the defines above */
    filament *error;		/* accumulated error details */


   /* Argument-type decyphering callback.
      Called to set the type of the ARGINDEX-th parameter to TYPE. */

    void (*argtype) PARAMS((struct printf_info * const pparser,
			    int type, 
			    int argindex));

    const char *format;		/* pointer to format string */
    int argc;			/* number of arguments used by format */
    int argindex;		/* number of arguments used so far */
    snv_constpointer const *argv; /* pointer to argument vector */

    int *argtypes;		/* internal: pointer to array of argtypes */
    int argsize;		/* internal: size of array of argtypes */

    int dollar;			/* standard parser state, as in glibc */
    int prec;			/* from this field on, as in glibc */
    int width;
    unsigned int extra;		/* for custom handlers */

    char spec;
    char pad;
    unsigned is_long_double : 1;
    unsigned is_char : 1;
    unsigned is_short : 1;
    unsigned is_long : 1;
    unsigned alt : 1;
    unsigned space : 1;
    unsigned left : 1;
    unsigned showsign : 1;
    unsigned group : 1;
    unsigned wide : 1;
} snv_parser;

#ifdef __STDC__
#  define PARSER_ERROR(pp, str)                                         \
            parser_error(pp, "file " __FILE__ ": line "                 \
                         STR(__LINE__) ERROR_FUNCTION ": " str "\n")
#else
   /* Yuck!  K&R doesn't handle string concatenation in macros like
      we use above, so we have to pass the responsibility through
      to runtime in this case. */
#  ifndef KNR_PARSE_ERROR_BUFSIZ
#  define KNR_PARSE_ERROR_BUFSIZ 4096
#  endif /* KNR_PARSE_ERROR_BUFSIZ */
#  define PARSER_ERROR(pp, str)         SNV_STMT_START{                 \
            char m_tmpstr1[KNR_PARSE_ERROR_BUFSIZ];                     \
            size_t m_tmplen2 = KNR_PARSE_ERROR_BUFSIZ -2; /* 2->\n\0 */ \
            strcpy(m_tmpstr1, "file ");                                 \
            strcat(m_tmpstr1, __FILE__);                                \
            strcat(m_tmpstr1, ": line ");                               \
            strcat(m_tmpstr1, STR(__LINE__));                           \
            strcat(m_tmpstr1, ERROR_FUNCTION);                          \
            m_tmplen2 -= strlen(m_tmpstr1);                             \
            strncat(m_tmpstr1, m_tmplen2, str);                         \
            strcat(m_tmpstr1, "\n");                                    \
            parser_error(pp, m_tmpstr1);                                \
                                        }SNV_STMT_END
#endif

/**
 * printfv_function:
 * @pparser: the current state information for the format string parser.  
 * @stream: the stream on which to write output.
 * @spec: the spec_entry for the character that has just been read
 *
 * Type of a printf specifier-handler function.
 *
 * The fields in the @pparser type can (and should)
 * be modified by functions of this type to implement the fsm which
 * parses format strings.
 *
 * @spec can be used so that multiple specifiers are
 * implemented by a single handler.
 *
 * The core library may call a registered printfv_function twice for
 * each place in %pparser->format that warrants it.  The first time is
 * when the number and types of arguments required by the format string
 * is being determined so that %pparser->argv can be built.  For this first call,
 * @stream and %pparser->argv will be %NULL, and the function should call the
 * %pparser->argtype callback to fill a suitable array with information
 * on the argument types.  This first call is only performed if the
 * non-vector API calls are used, because a vector has to be built for
 * the parser engine.
 *
 * @pparser will be reset for the next call (it is good practice to
 * minimise the use of @pparser to tracking the fsm state if a first
 * pass call is made) where %pparser->argv will be set.  It is this pass which
 * should put characters in @stream, always using the %SNV_EMIT macro
 * so that @stream being %NULL and buffer overflows can be detected and
 * handled.
 *
 * The function should return the number of characters written,
 * or a negative argument for errors.  In the case of an error, arbitrary
 * strings can be stored in the parser for output by the engine using the
 * parser_error function. 
 *
 * As an alternative, pass %snv_generic_specifier_handler as the handler
 * function, and your function as the user data.  The generic
 * handler will call this function just once to print the argument on a
 * filament, and then proceed to format the resulting string as if %s
 * was specified, but retaining all the special formatting options
 * including left and right justifying, padding, width and precision, etc.
 * In practice, the only options under your control are %space, %showsign
 * and %alt.  %pparser->argv[pparser->dollar] is the pointer to the
 * argument you are interested in.
 *
 * Return value:
 * The number of arguments eaten if %pparser->argv is %NULL, otherwise
 * the number of characters output.
 **/
typedef int	printfv_function PARAMS((snv_parser * const pparser, STREAM *stream, const struct spec_entry *spec));

/* The type of a parser element data deletion function. */
typedef void parser_delete_func PARAMS((snv_pointer address));

/**
 * SNV_UNLIMITED:
 * Used to denote that there is no upper limit to the number of characters
 * that can safely be written to a stream.
 **/
#define SNV_UNLIMITED           ~(0L)

/**
 * spec_entry:
 * @spec: the specifier character that was matched
 * @overridable: usually %TRUE, for internal use
 * @type: only a hint in general, but not when snv_generic_specifier_handler
 * is used, when this is the type of the argument eaten by the specifier.
 * @func: the handler function for the specifier
 * @user: the user data for the specifier, accessible to the handler function
 */
typedef struct spec_entry {
    int spec;
    int overridable;
    int type; 
    printfv_function *func;
    snv_pointer user;
} spec_entry;

/* Codes to determine basic types.

   These values cover all the standard format specifications.
   Users can add new values after PA_LAST for their own types.  */

enum
{
  PA_INT,		/* int */
  PA_CHAR,		/* int, cast to char */
  PA_WCHAR,		/* wide char */
  PA_STRING,		/* const char *, a '\0'-terminated string */
  PA_WSTRING,		/* const wchar_t *, wide character string */
  PA_POINTER,		/* void * */
  PA_FLOAT,		/* float */
  PA_DOUBLE,		/* double */
  PA_LAST
};

/* Flag bits that can be set in a type. */
#define PA_TYPE_MASK		0x00ff
#define	PA_FLAG_MASK		~SNV_TYPE_MASK

#define	PA_FLAG_LONG_LONG	(1 << 8)
#define	PA_FLAG_LONG_DOUBLE	PA_FLAG_LONG_LONG
#define	PA_FLAG_LONG		(1 << 9)
#define	PA_FLAG_SHORT		(1 << 10)
#define PA_FLAG_UNSIGNED	(1 << 11)
#define	PA_FLAG_CHAR		(1 << 12)
#define	PA_FLAG_PTR		(1 << 13)

/**
 * SNV_EMIT:
 * @ch: the character to be printed
 * @stream: the stream on which to print
 * @count: a variable to be updated with the count of printed
 * characters
 *
 * Maintain the count while putting @ch in @stream, also be careful about
 * handling %NULL stream if the handler is being called purely to count
 * output size.
 **/
#define SNV_EMIT(ch, stream, count)     SNV_STMT_START { \
    if (stream) \
    { \
	if (count >= 0) \
	{ \
	    int m_status = stream_put((ch), stream); \
	    count = m_status < 0 ? m_status : count + m_status; \
	} \
    } \
    else \
    { \
        (void)(ch); \
	count++; \
    } \
} SNV_STMT_END

/**
 * snv_generic_specifier_handler:
 * @pparser: the current state information for the format
 * string parser.  The fields in the @pparser type can (and should)
 * be modified by functions of this type to implement the fsm which
 * parses format strings.
 * @stream: the stream (possibly a struct printfv_stream appropriately
 * cast) on which to write output.
 * @spec: the spec_entry for the character that has just been read
 * by the parser, and can be used so that multiple specifiers are
 * implemented by a single handler.
 *
 * An example implementation of an handler, used to provide easy
 * access to justification, width and precision options.
 *
 * Return value:
 * The number of arguments eaten if %pparser->argv is %NULL, otherwise
 * the number of characters output.
 **/
extern int	snv_generic_specifier_handler PARAMS((snv_parser * const pparser, STREAM *stream, const struct spec_entry *spec));


/**
 * snv_register_specifier:  
 * @spec: the character which will trigger @func, cast to an unsigned int.
 * @type: the type tag which will be returned by the format parser.
 * @func: the handler function.
 * @user: an arbitrary pointer which is passed to @func through a %spec_entry struct.
 * 
 * Register @func to be called when @spec is encountered in a format
 * string.
 * 
 * Return value:
 * Returns %SNV_OK if @func was successfully registered, %SNV_ERROR
 * otherwise.
 **/
extern int snv_register_specifier PARAMS((unsigned spec, int type, printfv_function *func, snv_pointer user));

/**
 * parser_data_get:  
 * @pparser: pointer to the current parser state.
 * @key: the storage key for the parser state data in question.
 * 
 * This function will attempt to retrieve data previously stored in
 * @pparser under @key.
 * 
 * Return value: 
 * If there is no data associated with @key, %NULL is returned, otherwise
 * the address of the stored data is returned (so that 0 valued data can
 * be distinguished from a %NULL return).
 **/ 
extern snv_pointer* parser_data_get PARAMS((snv_parser *pparser, const char *key));

/**
 * parser_data_set:  
 * @pparser: pointer to the current parser state.
 * @key: the storage key for the parser state data in question.
 * @data: the data to be associated with @key in @pparser.
 * @delete_func: a function which will correctly free @data in due course.
 * 
 * This function will store the arbitrary stae information, @data, in
 * @pparser so that it can be shared with the other handler functions
 * which implement the format string parsing fsm.  @data is stored with
 * an associated @key for later retrieval.  @key is copied into @pparser.
 * so it is safe to pass static strings or recycle the original.  Any
 * stale @data already stored against @key in @pparser is passed to the
 * stale @delete_func for recycling, before the new @data and @delete_func
 * are used to overwrite the stale values.
 **/
extern void parser_data_set PARAMS((snv_parser *pparser, const char *key, snv_pointer data, parser_delete_func *delete_func));

/**
 * parser_error:  
 * @pparser: pointer to the current parser state.
 * @error_message: new error message to append to @pparser.
 * 
 * The contents of @error_message are appended to the @pparser internal
 * error string, so it is safe to pass static strings or recycle the
 * original when this function returns.
 * 
 * Return value:
 * The address of the full accumulated error message in @pparser is
 * returned.
 **/
extern char* parser_error PARAMS((snv_parser *pparser, const char *error_message));

/**
 * parse_printf_format:  
 * @format: a % delimited format string.
 * @n: the size of the @argtypes vector
 * @argtypes: a vector of ints, to be filled with the argument types from @format
 * 
 * Returns information about the number and types of
 * arguments expected by the template string @format.
 * The argument @n specifies the number of elements in the array
 * @argtypes.  This is the maximum number of elements that
 * the function will try to write.
 *
 * Return value:
 * The total number of arguments required by @format.  If this
 * number is greater than @n, then the information returned
 * describes only the first @n arguments.  If you want information
 * about additional arguments, allocate a bigger array and call
 * this function again. If there is an error, then %SNV_ERROR
 * is returned instead.
 **/
extern size_t parse_printf_format PARAMS((const char *format, int n, int *argtypes));

/**
 * stream_printfv:  
 * @stream: an initialised stream structure.
 * @format: a % delimited format string.
 * @args: a vector of argument addresses to match @format.
 * 
 * Format the elements of @args according to @format, and write
 * the results to @stream.  If @stream is %NULL, only count the
 * number of characters needed to output the format.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
extern int stream_printfv PARAMS((STREAM *stream, const char *format, snv_constpointer const args[]));

/**
 * stream_vprintf:  
 * @stream: an initialised stream structure.
 * @format: a % delimited format string.
 * @ap: a varargs/stdargs va_list.
 * 
 * Format the elements of @ap according to @format, and write
 * the results to @stream.  If @stream is %NULL, only count the
 * number of characters needed to output the format.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
extern int stream_vprintf PARAMS((STREAM *stream, const char *format, va_list ap));

/**
 * stream_printf:  
 * @stream: an initialised stream structure.
 * @format: a % delimited format string.
 * @va_alist: a varargs/stdargs va_list.
 * 
 * Format the elements of @va_alist according to @format, and write
 * the results to @stream.  If @stream is %NULL, only count the
 * number of characters needed to output the format.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
extern int stream_printf SNV_GNUC_PRINTF((STREAM *stream, const char *format, ...), 2, 3);


/**
 * fdputc:  
 * @ch: A single character to be added to @stream.
 * @stream: The stream in which to write @ch.
 * 
 * A stream_put_function function for use in putting characters
 * into STREAMs holding a file descriptor.
 * 
 * Return value:
 * The value of @ch that has been put in @stream, or -1 in case of
 * an error (errno will be set to indicate the type of error).
 **/
extern int fdputc PARAMS((int ch, STREAM *stream));

/**
 * snv_dprintf:  
 * @fd: an open file descriptor.
 * @format: a % delimited format string.
 * @va_alist: a varargs/stdargs va_list.
 * 
 * Format the elements of @va_alist according to @format, and write
 * the results to the file descriptor @fd.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
extern int snv_dprintf SNV_GNUC_PRINTF((int fd, const char *format, ...), 2, 3);


/**
 * snv_vdprintf:  
 * @fd: an open file descriptor.
 * @format: a % delimited format string.
 * @ap: a varargs/stdargs va_list.
 * 
 * Format the elements of @ap according to @format, and write
 * the results to the file descriptor @fd.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
extern int snv_vdprintf PARAMS((int fd, const char *format, va_list ap));

/**
 * dprintfv:  
 * @fd: an open file descriptor.
 * @format: a % delimited format string.
 * @args: a vector of argument addresses to match @format.
 * 
 * Format the elements of @args according to @format, and write
 * the results to file descriptor @fd.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
extern int dprintfv PARAMS((int fd, const char *format, snv_constpointer const args[]));

/**
 * fileputc:  
 * @ch: A single character to be added to @stream.
 * @stream: The stream in which to write @ch.
 * 
 * A stream_put_function function for use in putting characters
 * into STREAMs holding a FILE*.
 * 
 * Return value: 
 * The value of @ch that has been put in @stream.
 **/
extern int fileputc PARAMS((int ch, STREAM *stream));

/**
 * snv_printf:  
 * @format: a % delimited format string.
 * @va_alist: a varargs/stdargs va_list.
 * 
 * Format the elements of @va_alist according to @format, and write
 * the results to the standard output stream.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
extern int snv_printf SNV_GNUC_PRINTF((const char *format, ...), 1, 2);


/**
 * snv_vprintf:  
 * @format: a % delimited format string.
 * @ap: a varargs/stdargs va_list.
 * 
 * Format the elements of @ap according to @format, and write
 * the results to the standard output stream.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
extern int snv_vprintf PARAMS((const char *format, va_list ap));

/**
 * printfv:  
 * @format: a % delimited format string.
 * @args: a vector of argument addresses to match @format.
 * 
 * Format the elements of @args according to the string @format,
 * and write the result to the standard output stream.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
extern int printfv PARAMS((const char *format, snv_constpointer const args[]));

/**
 * snv_fprintf:  
 * @file: a stdio.h FILE* stream.
 * @format: a % delimited format string.
 * @va_alist: a varargs/stdargs va_list.
 * 
 * Format the elements of @va_alist according to @format, and write
 * the results to the @file stream.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
extern int snv_fprintf SNV_GNUC_PRINTF((FILE *file, const char *format, ...), 2, 3);


/**
 * snv_vfprintf:  
 * @file: a stdio.h FILE* stream.
 * @format: a % delimited format string.
 * @ap: a varargs/stdargs va_list.
 * 
 * Format the elements of @ap according to @format, and write
 * the results to the @file stream.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
extern int snv_vfprintf PARAMS((FILE *file, const char *format, va_list ap));

/**
 * fprintfv:  
 * @file: a stdio.h FILE* stream.
 * @format: a % delimited format string.
 * @args: a vector of argument addresses to match @format.
 * 
 * Format the elements of @args according to @format, and write
 * the results to @file.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
extern int fprintfv PARAMS((FILE *file, const char *format, snv_constpointer const args[]));

/**
 * bufputc:  
 * @ch: A single character to be added to @stream.
 * @stream: The stream in which to write @ch.
 * 
 * A stream_put_function function for use in putting characters
 * into STREAMs holding a char buffer.
 * 
 * Return value:
 * The value of @ch that has been put in @stream.
 **/
extern int bufputc PARAMS((int ch, STREAM *stream));

/**
 * snv_sprintf:  
 * @buffer: a preallocated char* buffer.
 * @format: a % delimited format string.
 * @va_alist: a varargs/stdargs va_list.
 * 
 * Format the elements of @va_alist according to @format, and write
 * the results to the string @buffer.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
extern int snv_sprintf SNV_GNUC_PRINTF((char buffer[], const char *format, ...), 2, 3);


/**
 * snv_vsprintf:  
 * @buffer: a preallocated char* buffer.
 * @format: a % delimited format string.
 * @ap: a varargs/stdargs va_list.
 * 
 * Format the elements of @ap according to @format, and write
 * the results to the string @buffer.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
extern int snv_vsprintf PARAMS((char buffer[], const char *format, va_list ap));

/**
 * sprintfv:  
 * @buffer: a preallocated char* buffer.
 * @format: a % delimited format string.
 * @args: a vector of argument addresses to match @format.
 * 
 * Format the elements of @args according to @format, and write
 * the results to the string @buffer.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
extern int sprintfv PARAMS((char buffer[], const char *format, snv_constpointer const args[]));

/**
 * snv_snprintf:  
 * @buffer: a preallocated char* buffer.
 * @limit: the maximum number of characters to write into @buffer.
 * @format: a % delimited format string.
 * @va_alist: a varargs/stdargs va_list.
 * 
 * Format the elements of @va_alist according to @format, and write
 * the results to the string @buffer, truncating the formatted string
 * if it reaches @limit characters in length.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
extern int snv_snprintf SNV_GNUC_PRINTF((char buffer[], unsigned long limit, const char *format, ...), 3, 4);


/**
 * snv_vsnprintf:  
 * @buffer: a preallocated char* buffer.
 * @limit: the maximum number of characters to write into @buffer.
 * @format: a % delimited format string.
 * @ap: a varargs/stdargs va_list.
 * 
 * Format the elements of @ap according to @format, and write
 * the results to the string @buffer, truncating the formatted string
 * if it reaches @limit characters in length.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
extern int snv_vsnprintf PARAMS((char buffer[], unsigned long limit, const char *format, va_list ap));

/**
 * snprintfv:  
 * @buffer: a preallocated char* buffer.
 * @limit: the maximum number of characters to write into @buffer.
 * @format: a % delimited format string.
 * @args: a vector of argument addresses to match @format.
 * 
 * Format the elements of @args according to @format, and write
 * the results to the string @buffer, truncating the formatted string
 * if it reaches @limit characters in length.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
extern int snprintfv PARAMS((char buffer[], unsigned long limit, const char *format, snv_constpointer const args[]));

/**
 * filputc:  
 * @ch: A single character to be added to @stream.
 * @stream: The stream in which to write @ch.
 * 
 * A stream_put_function function for use in putting characters
 * into STREAMs holding a filament*.
 * 
 * Return value: 
 * The value of @ch that has been put in @stream.
 **/
extern int filputc PARAMS((int ch, STREAM *stream));

/**
 * snv_asprintf:  
 * @result: the address of a char * variable.
 * @format: a % delimited format string.
 * @va_alist: a varargs/stdargs va_list.
 * 
 * Format the elements of @va_alist according to @format, and write
 * the results to an internally allocated buffer whose address is
 * stored in @result (and should be freed by the caller) unless
 * there is an error.
 *
 * Yes, this interface is cumbersome and totally useless.  It would
 * have been better to simply return the allocated address, but
 * it turns out that somebody wasn't thinking much when adding 
 * asprintf to libiberty a few years ago.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
extern int snv_asprintf SNV_GNUC_PRINTF((char **result, const char *format, ...), 2, 3);


/**
 * snv_vasprintf:  
 * @result: the address of a char * variable.
 * @format: a % delimited format string.
 * @ap: a varargs/stdargs va_list.
 * 
 * Format the elements of @ap according to @format, and write
 * the results to an internally allocated buffer whose address is
 * stored in @result (and should be freed by the caller) unless
 * there is an error.
 * 
 * Above moaning for asprintf applies here too.
 *
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
extern int snv_vasprintf PARAMS((char **result, const char *format, va_list ap));

/**
 * asprintfv:  
 * @result: the address of a char * variable.
 * @format: a % delimited format string.
 * @args: a vector of argument addresses to match @format.
 * 
 * Format the elements of @args according to @format, and write
 * the results to an internally allocated buffer whose address is
 * stored in @result (and should be freed by the caller) unless
 * there is an error.
 * 
 * Above moaning for asprintf applies here too.
 * 
 * Return value:
 * The number of characters written is returned, unless there is
 * an error, when %SNV_ERROR is returned.
 **/
extern int asprintfv PARAMS((char **result, const char *format, snv_constpointer const args[]));


/* If you don't want to use snprintfv functions for *all* of your string
   formatting API, then define COMPILING_SNPRINTFV_C and use the snv_
   prefix for the entry points below. */
#ifndef COMPILING_SNPRINTFV_C
#define printf          snv_printf
#define vprintf         snv_vprintf
#define dprintf         snv_dprintf
#define vdprintf        snv_vdprintf
#define fprintf         snv_fprintf
#define vfprintf        snv_vfprintf
#define sprintf         snv_sprintf
#define vsprintf        snv_vsprintf
#define snprintf        snv_snprintf
#define vsnprintf       snv_vsnprintf
#define asprintf        snv_asprintf
#define vasprintf       snv_vasprintf
#endif /* !COMPILING_SNPRINTFV_C */

#ifdef __cplusplus
}
#endif

#endif /* SNPRINTFV_SNPRINTFV_H */

/* snprintfv.h ends here */
