#define __SP_EXPORT_C__

/*
 * PNG export dialog
 *
 * Authors:
 *   Lauris Kaplinski <lauris@kaplinski.com>
 *
 * Copyright (C) 1999-2002 Authors
 * Copyright (C) 2001-2002 Ximian, Inc.
 *
 * Released under GNU GPL, read the file 'COPYING' for more information
 */

#include <config.h>

#include <string.h>
#include <glib.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>
#include <gtk/gtksignal.h>
#include <gtk/gtkwindow.h>
#include <gtk/gtkvbox.h>
#include <gtk/gtkhbox.h>
#include <gtk/gtkframe.h>
#include <gtk/gtktable.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkspinbutton.h>
#include <gtk/gtktogglebutton.h>
#include <gtk/gtkhseparator.h>
#include <libgnomeui/gnome-file-entry.h>

#include "helper/unit-menu.h"
#include "sodipodi.h"
#include "document.h"
#include "desktop-handles.h"
#include "sp-item.h"
#include "selection.h"
#include "file.h"
#include "export.h"

#define SP_EXPORT_MIN_SIZE 16.0

static void sp_export_area_toggled (GtkToggleButton *tb, GtkObject *base);
static void sp_export_export_clicked (GtkButton *button, GtkObject *base);

static void sp_export_area_x_value_changed (GtkAdjustment *adj, GtkObject *base);
static void sp_export_area_y_value_changed (GtkAdjustment *adj, GtkObject *base);
static void sp_export_area_width_value_changed (GtkAdjustment *adj, GtkObject *base);
static void sp_export_area_height_value_changed (GtkAdjustment *adj, GtkObject *base);
static void sp_export_bitmap_width_value_changed (GtkAdjustment *adj, GtkObject *base);
static void sp_export_xdpi_value_changed (GtkAdjustment *adj, GtkObject *base);

static void sp_export_set_area (GtkObject *base, float x0, float y0, float x1, float y1);
static void sp_export_value_set (GtkObject *base, const unsigned char *key, float val);
static void sp_export_value_set_pt (GtkObject *base, const unsigned char *key, float val);
static float sp_export_value_get (GtkObject *base, const unsigned char *key);
static float sp_export_value_get_pt (GtkObject *base, const unsigned char *key);

static GtkWidget *dlg = NULL;

static void
sp_export_dialog_destroy (GtkObject *object, gpointer data)
{
	dlg = NULL;
}

static void
sp_export_spinbutton_new (GtkTable *t, int x, int y, const unsigned char *ll, const unsigned char *lr,
			  GtkAdjustment *a, int step, int digits, unsigned int sensitive)
{
	GtkWidget *l, *sb;
	int pos;

	pos = 0;

	if (ll) {
		l = gtk_label_new (ll);
		gtk_misc_set_alignment (GTK_MISC (l), 1.0, 0.5);
		gtk_table_attach (t, l, x + pos, x + pos + 1, y, y + 1, 0, 0, 0, 0);
		gtk_widget_set_sensitive (l, sensitive);
		pos += 1;
	}

	sb = gtk_spin_button_new (a, step, digits);
	gtk_table_attach (GTK_TABLE (t), sb, x + pos, x + pos + 1, y, y + 1, 0, 0, 0, 0);
	gtk_widget_set_usize (sb, 64, -1);
	gtk_widget_set_sensitive (sb, sensitive);
	pos += 1;

	if (lr) {
		l = gtk_label_new (lr);
		gtk_misc_set_alignment (GTK_MISC (l), 0.0, 0.5);
		gtk_table_attach (t, l, x + pos, x + pos + 1, y, y + 1, 0, 0, 0, 0);
		gtk_widget_set_sensitive (l, sensitive);
		pos += 1;
	}
}

void
sp_export_dialog (void)
{
	if (!dlg) {
		GtkWidget *vb, *f, *t, *hb, *us, *l, *fe, *hs, *b;
		GtkObject *a;

		dlg = gtk_window_new (GTK_WINDOW_TOPLEVEL);
		gtk_window_set_title (GTK_WINDOW (dlg), _("Export bitmap"));
		gtk_signal_connect (GTK_OBJECT (dlg), "destroy", GTK_SIGNAL_FUNC (sp_export_dialog_destroy), NULL);

		vb = gtk_vbox_new (FALSE, 4);
		gtk_widget_show (vb);
		gtk_container_set_border_width (GTK_CONTAINER (vb), 4);
		gtk_container_add (GTK_CONTAINER (dlg), vb);

		/* Export area frame */
		f = gtk_frame_new (_("Export area"));
		gtk_box_pack_start (GTK_BOX (vb), f, FALSE, FALSE, 0);
		t = gtk_table_new (3, 6, FALSE);
		gtk_table_set_row_spacings (GTK_TABLE (t), 4);
		gtk_table_set_col_spacings (GTK_TABLE (t), 4);
		gtk_container_set_border_width (GTK_CONTAINER (t), 4);
		gtk_container_add (GTK_CONTAINER (f), t);

		hb = gtk_hbox_new (FALSE, 0);
		gtk_table_attach (GTK_TABLE (t), hb, 0, 6, 0, 1, GTK_EXPAND | GTK_FILL, 0, 0, 0);

		b = gtk_toggle_button_new_with_label (_("Page"));
		gtk_object_set_data (GTK_OBJECT (b), "key", "page");
		gtk_object_set_data (GTK_OBJECT (dlg), "page", b);
		gtk_box_pack_start (GTK_BOX (hb), b, FALSE, FALSE, 0);
		gtk_signal_connect (GTK_OBJECT (b), "clicked", GTK_SIGNAL_FUNC (sp_export_area_toggled), dlg);
		b = gtk_toggle_button_new_with_label (_("Drawing"));
		gtk_object_set_data (GTK_OBJECT (b), "key", "drawing");
		gtk_object_set_data (GTK_OBJECT (dlg), "drawing", b);
		gtk_box_pack_start (GTK_BOX (hb), b, FALSE, FALSE, 0);
		gtk_signal_connect (GTK_OBJECT (b), "clicked", GTK_SIGNAL_FUNC (sp_export_area_toggled), dlg);
		b = gtk_toggle_button_new_with_label (_("Selection"));
		gtk_object_set_data (GTK_OBJECT (b), "key", "selection");
		gtk_object_set_data (GTK_OBJECT (dlg), "selection", b);
		gtk_box_pack_start (GTK_BOX (hb), b, FALSE, FALSE, 0);
		gtk_signal_connect (GTK_OBJECT (b), "clicked", GTK_SIGNAL_FUNC (sp_export_area_toggled), dlg);

		us = sp_unit_selector_new (SP_UNIT_ABSOLUTE);
		gtk_box_pack_end (GTK_BOX (hb), us, FALSE, FALSE, 0);
		l = gtk_label_new (_("Units:"));
		gtk_box_pack_end (GTK_BOX (hb), l, FALSE, FALSE, 0);
		gtk_object_set_data (GTK_OBJECT (dlg), "units", us);

		a = gtk_adjustment_new (0.0, -10000.0, 10000.0, 0.1, 1.0, 1.0);
		gtk_object_set_data (GTK_OBJECT (a), "key", "x0");
		gtk_object_set_data (GTK_OBJECT (dlg), "x0", a);
		sp_unit_selector_add_adjustment (SP_UNIT_SELECTOR (us), GTK_ADJUSTMENT (a));
		sp_export_spinbutton_new (GTK_TABLE (t), 0, 1, _("x0:"), NULL, GTK_ADJUSTMENT (a), 1.0, 2, 1);
		gtk_signal_connect (GTK_OBJECT (a), "value_changed", GTK_SIGNAL_FUNC (sp_export_area_x_value_changed), dlg);

		a = gtk_adjustment_new (0.0, -10000.0, 10000.0, 0.1, 1.0, 1.0);
		gtk_object_set_data (GTK_OBJECT (a), "key", "x1");
		gtk_object_set_data (GTK_OBJECT (dlg), "x1", a);
		sp_unit_selector_add_adjustment (SP_UNIT_SELECTOR (us), GTK_ADJUSTMENT (a));
		sp_export_spinbutton_new (GTK_TABLE (t), 2, 1, _("x1:"), NULL, GTK_ADJUSTMENT (a), 1.0, 2, 1);
		gtk_signal_connect (GTK_OBJECT (a), "value_changed", GTK_SIGNAL_FUNC (sp_export_area_x_value_changed), dlg);

		a = gtk_adjustment_new (0.0, -10000.0, 10000.0, 0.1, 1.0, 1.0);
		gtk_object_set_data (GTK_OBJECT (dlg), "width", a);
		sp_unit_selector_add_adjustment (SP_UNIT_SELECTOR (us), GTK_ADJUSTMENT (a));
		sp_export_spinbutton_new (GTK_TABLE (t), 4, 1, _("Width:"), NULL, GTK_ADJUSTMENT (a), 1.0, 2, 1);
		gtk_signal_connect (GTK_OBJECT (a), "value_changed", GTK_SIGNAL_FUNC (sp_export_area_width_value_changed), dlg);

		a = gtk_adjustment_new (0.0, -10000.0, 10000.0, 0.1, 1.0, 1.0);
		gtk_object_set_data (GTK_OBJECT (a), "key", "y0");
		gtk_object_set_data (GTK_OBJECT (dlg), "y0", a);
		sp_unit_selector_add_adjustment (SP_UNIT_SELECTOR (us), GTK_ADJUSTMENT (a));
		sp_export_spinbutton_new (GTK_TABLE (t), 0, 2, _("y0:"), NULL, GTK_ADJUSTMENT (a), 1.0, 2, 1);
		gtk_signal_connect (GTK_OBJECT (a), "value_changed", GTK_SIGNAL_FUNC (sp_export_area_y_value_changed), dlg);

		a = gtk_adjustment_new (0.0, -10000.0, 10000.0, 0.1, 1.0, 1.0);
		gtk_object_set_data (GTK_OBJECT (a), "key", "y1");
		gtk_object_set_data (GTK_OBJECT (dlg), "y1", a);
		sp_unit_selector_add_adjustment (SP_UNIT_SELECTOR (us), GTK_ADJUSTMENT (a));
		sp_export_spinbutton_new (GTK_TABLE (t), 2, 2, _("y1:"), NULL, GTK_ADJUSTMENT (a), 1.0, 2, 1);
		gtk_signal_connect (GTK_OBJECT (a), "value_changed", GTK_SIGNAL_FUNC (sp_export_area_y_value_changed), dlg);

		a = gtk_adjustment_new (0.0, -10000.0, 10000.0, 0.1, 1.0, 1.0);
		gtk_object_set_data (GTK_OBJECT (dlg), "height", a);
		sp_unit_selector_add_adjustment (SP_UNIT_SELECTOR (us), GTK_ADJUSTMENT (a));
		sp_export_spinbutton_new (GTK_TABLE (t), 4, 2, _("Height:"), NULL, GTK_ADJUSTMENT (a), 1.0, 2, 1);
		gtk_signal_connect (GTK_OBJECT (a), "value_changed", GTK_SIGNAL_FUNC (sp_export_area_height_value_changed), dlg);

		gtk_widget_show_all (f);

		/* Bitmap size frame */
		f = gtk_frame_new (_("Bitmap size"));
		gtk_box_pack_start (GTK_BOX (vb), f, FALSE, FALSE, 0);
		t = gtk_table_new (2, 5, FALSE);
		gtk_table_set_row_spacings (GTK_TABLE (t), 4);
		gtk_table_set_col_spacings (GTK_TABLE (t), 4);
		gtk_container_set_border_width (GTK_CONTAINER (t), 4);
		gtk_container_add (GTK_CONTAINER (f), t);

		a = gtk_adjustment_new (16.0, -10000.0, 10000.0, 1, 10.0, 10.0);
		sp_export_spinbutton_new (GTK_TABLE (t), 0, 0, _("Width:"), _("pixels"), GTK_ADJUSTMENT (a), 1.0, 0, 1);
		gtk_object_set_data (GTK_OBJECT (dlg), "bmwidth", a);
		gtk_signal_connect (GTK_OBJECT (a), "value_changed", GTK_SIGNAL_FUNC (sp_export_bitmap_width_value_changed), dlg);

		a = gtk_adjustment_new (72.0, 1.0, 9600.0, 0.1, 1.0, 1.0);
		sp_export_spinbutton_new (GTK_TABLE (t), 3, 0, NULL, _("dpi"), GTK_ADJUSTMENT (a), 1.0, 2, 1);
		gtk_object_set_data (GTK_OBJECT (dlg), "xdpi", a);
		gtk_signal_connect (GTK_OBJECT (a), "value_changed", GTK_SIGNAL_FUNC (sp_export_xdpi_value_changed), dlg);

		a = gtk_adjustment_new (16.0, -10000.0, 10000.0, 1, 10.0, 10.0);
		sp_export_spinbutton_new (GTK_TABLE (t), 0, 1, _("Height:"), _("pixels"), GTK_ADJUSTMENT (a), 1.0, 0, 0);
		gtk_object_set_data (GTK_OBJECT (dlg), "bmheight", a);

		a = gtk_adjustment_new (72.0, 1.0, 9600.0, 0.1, 1.0, 1.0);
		sp_export_spinbutton_new (GTK_TABLE (t), 3, 1, NULL, _("dpi"), GTK_ADJUSTMENT (a), 1.0, 2, 0);
		gtk_object_set_data (GTK_OBJECT (dlg), "ydpi", a);

		gtk_widget_show_all (f);

		/* File entry */
		fe = gnome_file_entry_new ("export", _("Export png file"));
		gtk_widget_show (fe);
		gtk_box_pack_start (GTK_BOX (vb), fe, FALSE, FALSE, 0);
		gtk_object_set_data (GTK_OBJECT (dlg), "filename", fe);

		/* Buttons */
		hb = gtk_hbox_new (FALSE, 0);
		gtk_widget_show (hb);
		gtk_box_pack_end (GTK_BOX (vb), hb, FALSE, FALSE, 0);

		b = gtk_button_new_with_label (_("Export"));
		gtk_widget_show (b);
		gtk_box_pack_end (GTK_BOX (hb), b, FALSE, FALSE, 0);
		gtk_signal_connect (GTK_OBJECT (b), "clicked", GTK_SIGNAL_FUNC (sp_export_export_clicked), dlg);

		hs = gtk_hseparator_new ();
		gtk_widget_show (hs);
		gtk_box_pack_end (GTK_BOX (vb), hs, FALSE, FALSE, 0);
	}

	gtk_widget_show (dlg);
}

static void
sp_export_area_toggled (GtkToggleButton *tb, GtkObject *base)
{
	if (gtk_toggle_button_get_active (tb)) {
		const unsigned char *key;
		key = gtk_object_get_data (GTK_OBJECT (tb), "key");
		if (strcmp (key, "page")) {
			gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (gtk_object_get_data (base, "page")), FALSE);
		}
		if (strcmp (key, "drawing")) {
			gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (gtk_object_get_data (base, "drawing")), FALSE);
		}
		if (strcmp (key, "selection")) {
			gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (gtk_object_get_data (base, "selection")), FALSE);
		}
		gtk_object_set_data (base, "area", (gpointer) key);
		if (SP_ACTIVE_DESKTOP) {
			SPDocument *doc;
			ArtDRect bbox;
			doc = SP_DT_DOCUMENT (SP_ACTIVE_DESKTOP);
			if (!strcmp (key, "page")) {
				bbox.x0 = 0.0;
				bbox.y0 = 0.0;
				bbox.x1 = sp_document_width (doc);
				bbox.y1 = sp_document_height (doc);
			} else if (!strcmp (key, "drawing")) {
				sp_item_bbox_desktop (SP_ITEM (SP_DOCUMENT_ROOT (doc)), &bbox);
			} else {
				sp_selection_bbox (SP_DT_SELECTION (SP_ACTIVE_DESKTOP), &bbox);
			}
			sp_export_set_area (base, bbox.x0, bbox.y0, bbox.x1, bbox.y1);
		}
	}
}

static void
sp_export_export_clicked (GtkButton *button, GtkObject *base)
{
	GtkWidget *fe;
	const unsigned char *filename;
	float x0, y0, x1, y1, width, height;

	if (!SP_ACTIVE_DESKTOP) return;

	fe = gtk_object_get_data (base, "filename");
	filename = gnome_file_entry_get_full_path (GNOME_FILE_ENTRY (fe), FALSE);

	x0 = sp_export_value_get_pt (base, "x0");
	y0 = sp_export_value_get_pt (base, "y0");
	x1 = sp_export_value_get_pt (base, "x1");
	y1 = sp_export_value_get_pt (base, "y1");
	width = sp_export_value_get (base, "bmwidth");
	height = sp_export_value_get (base, "bmheight");

	sp_export_png_file (SP_DT_DOCUMENT (SP_ACTIVE_DESKTOP), filename, x0, y0, x1, y1, width, height, 0x00000000);
}

static void
sp_export_area_x_value_changed (GtkAdjustment *adj, GtkObject *base)
{
	float x0, x1, xdpi, width;

	if (gtk_object_get_data (base, "update")) return;
	if (sp_unit_selector_update_test (gtk_object_get_data (base, "units"))) return;
	gtk_object_set_data (base, "update", GUINT_TO_POINTER (TRUE));

	x0 = sp_export_value_get_pt (base, "x0");
	x1 = sp_export_value_get_pt (base, "x1");
	xdpi = sp_export_value_get (base, "xdpi");

	width = (x1 - x0) * xdpi / 72.0;

	if (width < SP_EXPORT_MIN_SIZE) {
		const unsigned char *key;
		width = SP_EXPORT_MIN_SIZE;
		key = gtk_object_get_data (GTK_OBJECT (adj), "key");
		if (!strcmp (key, "x0")) {
			x1 = x0 + width * 72.0 / xdpi;
			sp_export_value_set_pt (base, "x1", x1);
		} else {
			x0 = x1 - width * 72.0 / xdpi;
			sp_export_value_set_pt (base, "x0", x0);
		}
	}

	sp_export_value_set_pt (base, "width", x1 - x0);
	sp_export_value_set (base, "bmwidth", width);

	gtk_object_set_data (base, "update", GUINT_TO_POINTER (FALSE));
}

static void
sp_export_area_y_value_changed (GtkAdjustment *adj, GtkObject *base)
{
	float y0, y1, ydpi, height;

	if (gtk_object_get_data (base, "update")) return;
	if (sp_unit_selector_update_test (gtk_object_get_data (base, "units"))) return;
	gtk_object_set_data (base, "update", GUINT_TO_POINTER (TRUE));

	y0 = sp_export_value_get_pt (base, "y0");
	y1 = sp_export_value_get_pt (base, "y1");
	ydpi = sp_export_value_get (base, "ydpi");

	height = (y1 - y0) * ydpi / 72.0;

	if (height < SP_EXPORT_MIN_SIZE) {
		const unsigned char *key;
		height = SP_EXPORT_MIN_SIZE;
		key = gtk_object_get_data (GTK_OBJECT (adj), "key");
		if (!strcmp (key, "y0")) {
			y1 = y0 + height * 72.0 / ydpi;
			sp_export_value_set_pt (base, "y1", y1);
		} else {
			y0 = y1 - height * 72.0 / ydpi;
			sp_export_value_set_pt (base, "y0", y0);
		}
	}

	sp_export_value_set_pt (base, "height", y1 - y0);
	sp_export_value_set (base, "bmheight", height);

	gtk_object_set_data (base, "update", GUINT_TO_POINTER (FALSE));
}

static void
sp_export_area_width_value_changed (GtkAdjustment *adj, GtkObject *base)
{
	float x0, x1, xdpi, width, bmwidth;

	if (gtk_object_get_data (base, "update")) return;
	if (sp_unit_selector_update_test (gtk_object_get_data (base, "units"))) return;
	gtk_object_set_data (base, "update", GUINT_TO_POINTER (TRUE));

	x0 = sp_export_value_get_pt (base, "x0");
	x1 = sp_export_value_get_pt (base, "x1");
	xdpi = sp_export_value_get (base, "xdpi");
	width = sp_export_value_get_pt (base, "width");
	bmwidth = width * xdpi / 72.0;

	if (bmwidth < SP_EXPORT_MIN_SIZE) {
		bmwidth = SP_EXPORT_MIN_SIZE;
		width = bmwidth * 72.0 / xdpi;
		sp_export_value_set_pt (base, "width", width);
	}

	sp_export_value_set_pt (base, "x1", x0 + width);
	sp_export_value_set (base, "bmwidth", bmwidth);

	gtk_object_set_data (base, "update", GUINT_TO_POINTER (FALSE));
}

static void
sp_export_area_height_value_changed (GtkAdjustment *adj, GtkObject *base)
{
	float y0, y1, ydpi, height, bmheight;

	if (gtk_object_get_data (base, "update")) return;
	if (sp_unit_selector_update_test (gtk_object_get_data (base, "units"))) return;
	gtk_object_set_data (base, "update", GUINT_TO_POINTER (TRUE));

	y0 = sp_export_value_get_pt (base, "y0");
	y1 = sp_export_value_get_pt (base, "y1");
	ydpi = sp_export_value_get (base, "ydpi");
	height = sp_export_value_get_pt (base, "height");
	bmheight = height * ydpi / 72.0;

	if (bmheight < SP_EXPORT_MIN_SIZE) {
		bmheight = SP_EXPORT_MIN_SIZE;
		height = bmheight * 72.0 / ydpi;
		sp_export_value_set_pt (base, "height", height);
	}

	sp_export_value_set_pt (base, "y1", y0 + height);
	sp_export_value_set (base, "bmheight", bmheight);

	gtk_object_set_data (base, "update", GUINT_TO_POINTER (FALSE));
}

static void
sp_export_set_image_y (GtkObject *base)
{
	float y0, y1, xdpi;

	y0 = sp_export_value_get_pt (base, "y0");
	y1 = sp_export_value_get_pt (base, "y1");
	xdpi = sp_export_value_get (base, "xdpi");

	sp_export_value_set (base, "ydpi", xdpi);
	sp_export_value_set (base, "bmheight", (y1 - y0) * xdpi / 72.0);
}

static void
sp_export_bitmap_width_value_changed (GtkAdjustment *adj, GtkObject *base)
{
	float x0, x1, bmwidth, xdpi;

	if (gtk_object_get_data (base, "update")) return;
	if (sp_unit_selector_update_test (gtk_object_get_data (base, "units"))) return;
	gtk_object_set_data (base, "update", GUINT_TO_POINTER (TRUE));

	x0 = sp_export_value_get_pt (base, "x0");
	x1 = sp_export_value_get_pt (base, "x1");
	bmwidth = sp_export_value_get (base, "bmwidth");

	if (bmwidth < SP_EXPORT_MIN_SIZE) {
		bmwidth = SP_EXPORT_MIN_SIZE;
		sp_export_value_set (base, "bmwidth", bmwidth);
	}

	xdpi = bmwidth * 72.0 / (x1 - x0);
	sp_export_value_set (base, "xdpi", xdpi);

	sp_export_set_image_y (base);

	gtk_object_set_data (base, "update", GUINT_TO_POINTER (FALSE));
}

void
sp_export_xdpi_value_changed (GtkAdjustment *adj, GtkObject *base)
{
	float x0, x1, xdpi, bmwidth;

	if (gtk_object_get_data (base, "update")) return;
	if (sp_unit_selector_update_test (gtk_object_get_data (base, "units"))) return;
	gtk_object_set_data (base, "update", GUINT_TO_POINTER (TRUE));

	x0 = sp_export_value_get_pt (base, "x0");
	x1 = sp_export_value_get_pt (base, "x1");
	xdpi = sp_export_value_get (base, "xdpi");

	bmwidth = (x1 - x0) * xdpi / 72.0;

	if (bmwidth < SP_EXPORT_MIN_SIZE) {
		bmwidth = SP_EXPORT_MIN_SIZE;
		xdpi = bmwidth * 72.0 / (x1 - x0);
		sp_export_value_set (base, "xdpi", xdpi);
	}

	sp_export_value_set (base, "bmwidth", bmwidth);

	sp_export_set_image_y (base);

	gtk_object_set_data (base, "update", GUINT_TO_POINTER (FALSE));
}

static void
sp_export_set_area (GtkObject *base, float x0, float y0, float x1, float y1)
{
	sp_export_value_set_pt (base, "x1", x1);
	sp_export_value_set_pt (base, "y1", y1);
	sp_export_value_set_pt (base, "x0", x0);
	sp_export_value_set_pt (base, "y0", y0);
}

static void
sp_export_value_set (GtkObject *base, const unsigned char *key, float val)
{
	GtkAdjustment *adj;

	adj = gtk_object_get_data (base, key);

	gtk_adjustment_set_value (adj, val);
}

static void
sp_export_value_set_pt (GtkObject *base, const unsigned char *key, float val)
{
	GtkAdjustment *adj;
	const SPUnit *unit;

	adj = gtk_object_get_data (base, key);

	unit = sp_unit_selector_get_unit (gtk_object_get_data (base, "units"));

	gtk_adjustment_set_value (adj, sp_points_get_units (val, unit));
}

static float
sp_export_value_get (GtkObject *base, const unsigned char *key)
{
	GtkAdjustment *adj;

	adj = gtk_object_get_data (base, key);

	return adj->value;
}

static float
sp_export_value_get_pt (GtkObject *base, const unsigned char *key)
{
	GtkAdjustment *adj;
	const SPUnit *unit;

	adj = gtk_object_get_data (base, key);

	unit = sp_unit_selector_get_unit (gtk_object_get_data (base, "units"));

	return sp_units_get_points (adj->value, unit);
}
