#define __SP_OBJECT_C__

/*
 * Abstract base class for all nodes
 *
 * Authors:
 *   Lauris Kaplinski <lauris@kaplinski.com>
 *
 * Copyright (C) 1999-2002 authors
 * Copyright (C) 2001-2002 Ximian, Inc.
 *
 * Released under GNU GPL, read the file 'COPYING' for more information
 */

#include <stdlib.h>
#include <string.h>
#include <gtk/gtksignal.h>
#include "xml/repr-private.h"
#include "document.h"
#include "style.h"
#include "sp-object-repr.h"
#include "sp-object.h"

static void sp_object_class_init (SPObjectClass * klass);
static void sp_object_init (SPObject * object);
static void sp_object_destroy (GtkObject * object);

static void sp_object_build (SPObject * object, SPDocument * document, SPRepr * repr);
static void sp_object_release (SPObject *object);

static void sp_object_read_attr (SPObject * object, const gchar * key);
static SPRepr *sp_object_private_write (SPObject *object, SPRepr *repr, guint flags);

/* Real handlers of repr signals */

static gboolean sp_object_repr_change_attr (SPRepr *repr, const guchar *key, const guchar *oldval, const guchar *newval, gpointer data);
static void sp_object_repr_attr_changed (SPRepr *repr, const guchar *key, const guchar *oldval, const guchar *newval, gpointer data);

static void sp_object_repr_content_changed (SPRepr *repr, const guchar *oldcontent, const guchar *newcontent, gpointer data);

static void sp_object_repr_child_added (SPRepr *repr, SPRepr *child, SPRepr *ref, gpointer data);
static gboolean sp_object_repr_remove_child (SPRepr *repr, SPRepr *child, SPRepr *ref, gpointer data);

static void sp_object_repr_order_changed (SPRepr *repr, SPRepr *child, SPRepr *old, SPRepr *new, gpointer data);

static gchar * sp_object_get_unique_id (SPObject * object, const gchar * defid);

enum {RELEASE, MODIFIED, LAST_SIGNAL};

SPReprEventVector object_event_vector = {
	NULL, /* Destroy */
	NULL, /* Add child */
	sp_object_repr_child_added,
	sp_object_repr_remove_child,
	NULL, /* Child removed */
	sp_object_repr_change_attr,
	sp_object_repr_attr_changed,
	NULL, /* Change content */
	sp_object_repr_content_changed,
	NULL, /* change_order */
	sp_object_repr_order_changed
};

static GtkObjectClass *parent_class;
static guint object_signals[LAST_SIGNAL] = {0};

GtkType
sp_object_get_type (void)
{
	static GtkType object_type = 0;
	if (!object_type) {
		GtkTypeInfo object_info = {
			"SPObject",
			sizeof (SPObject),
			sizeof (SPObjectClass),
			(GtkClassInitFunc) sp_object_class_init,
			(GtkObjectInitFunc) sp_object_init,
			NULL, /* reserved_1 */
			NULL, /* reserved_2 */
			(GtkClassInitFunc) NULL
		};
		object_type = gtk_type_unique (gtk_object_get_type (), &object_info);
	}
	return object_type;
}

static void
sp_object_class_init (SPObjectClass * klass)
{
	GtkObjectClass * gtk_object_class;

	gtk_object_class = (GtkObjectClass *) klass;

	parent_class = gtk_type_class (gtk_object_get_type ());

	object_signals[RELEASE] =  gtk_signal_new ("release",
						   GTK_RUN_FIRST,
						   gtk_object_class->type,
						   GTK_SIGNAL_OFFSET (SPObjectClass, release),
						   gtk_marshal_NONE__NONE,
						   GTK_TYPE_NONE, 0);
	object_signals[MODIFIED] = gtk_signal_new ("modified",
						   GTK_RUN_FIRST,
						   gtk_object_class->type,
						   GTK_SIGNAL_OFFSET (SPObjectClass, modified),
						   gtk_marshal_NONE__UINT,
						   GTK_TYPE_NONE, 1, GTK_TYPE_UINT);
	gtk_object_class_add_signals (gtk_object_class, object_signals, LAST_SIGNAL);

	gtk_object_class->destroy = sp_object_destroy;

	klass->build = sp_object_build;
	klass->release = sp_object_release;
	klass->read_attr = sp_object_read_attr;
	klass->write = sp_object_private_write;
}

static void
sp_object_init (SPObject * object)
{
	object->hrefcount = 0;
	object->document = NULL;
	object->parent = object->next = NULL;
	object->repr = NULL;
	object->id = NULL;
	object->style = NULL;
	object->title = NULL;
	object->description = NULL;
}

static void
sp_object_destroy (GtkObject * object)
{
	SPObject *spobject;

	spobject = (SPObject *) object;

	/* fixme: This is here temporarily */
	/* fixme: I f everything is ported to ::release, we can remove destroy at all */
	if (spobject->document) {
		sp_object_invoke_release (spobject);
	}

	if (((GtkObjectClass *) (parent_class))->destroy)
		(* ((GtkObjectClass *) (parent_class))->destroy) (object);
}

/*
 * Refcounting
 *
 * Owner is here for debug reasons, you can set it to NULL safely
 * Ref should return object, NULL is error, unref return always NULL
 */

SPObject *
sp_object_ref (SPObject *object, SPObject *owner)
{
	g_return_val_if_fail (object != NULL, NULL);
	g_return_val_if_fail (SP_IS_OBJECT (object), NULL);
	g_return_val_if_fail (!owner || SP_IS_OBJECT (owner), NULL);

	gtk_object_ref (GTK_OBJECT (object));

	return object;
}

SPObject *
sp_object_unref (SPObject *object, SPObject *owner)
{
	g_return_val_if_fail (object != NULL, NULL);
	g_return_val_if_fail (SP_IS_OBJECT (object), NULL);
	g_return_val_if_fail (!owner || SP_IS_OBJECT (owner), NULL);

	gtk_object_unref (GTK_OBJECT (object));

	return NULL;
}

SPObject *
sp_object_href (SPObject *object, gpointer owner)
{
	g_return_val_if_fail (object != NULL, NULL);
	g_return_val_if_fail (SP_IS_OBJECT (object), NULL);

	object->hrefcount++;

	return object;
}

SPObject *
sp_object_hunref (SPObject *object, gpointer owner)
{
	g_return_val_if_fail (object != NULL, NULL);
	g_return_val_if_fail (SP_IS_OBJECT (object), NULL);
	g_return_val_if_fail (object->hrefcount > 0, NULL);

	object->hrefcount--;

	return NULL;
}

/*
 * Attaching/detaching
 */

SPObject *
sp_object_attach_reref (SPObject *parent, SPObject *object, SPObject *next)
{
	g_return_val_if_fail (parent != NULL, NULL);
	g_return_val_if_fail (SP_IS_OBJECT (parent), NULL);
	g_return_val_if_fail (object != NULL, NULL);
	g_return_val_if_fail (SP_IS_OBJECT (object), NULL);
	g_return_val_if_fail (!next || SP_IS_OBJECT (next), NULL);
	g_return_val_if_fail (!object->parent, NULL);
	g_return_val_if_fail (!object->next, NULL);

	sp_object_ref (object, parent);
	gtk_object_unref (GTK_OBJECT (object));
	object->parent = parent;
	object->next = next;

	return object;
}

SPObject *
sp_object_detach (SPObject *parent, SPObject *object)
{
	SPObject *next;

	g_return_val_if_fail (parent != NULL, NULL);
	g_return_val_if_fail (SP_IS_OBJECT (parent), NULL);
	g_return_val_if_fail (object != NULL, NULL);
	g_return_val_if_fail (SP_IS_OBJECT (object), NULL);
	g_return_val_if_fail (object->parent == parent, NULL);

	next = object->next;
	object->parent = NULL;
	object->next = NULL;

	sp_object_invoke_release (object);

	return next;
}

SPObject *
sp_object_detach_unref (SPObject *parent, SPObject *object)
{
	SPObject *next;

	g_return_val_if_fail (parent != NULL, NULL);
	g_return_val_if_fail (SP_IS_OBJECT (parent), NULL);
	g_return_val_if_fail (object != NULL, NULL);
	g_return_val_if_fail (SP_IS_OBJECT (object), NULL);
	g_return_val_if_fail (object->parent == parent, NULL);

	next = object->next;
	object->parent = NULL;
	object->next = NULL;

	sp_object_invoke_release (object);

	sp_object_unref (object, parent);

	return next;
}

/*
 * SPObject specific build method
 */

static void
sp_object_build (SPObject * object, SPDocument * document, SPRepr * repr)
{
	/* Nothing specific here */
}

static void
sp_object_release (SPObject * object)
{
	/* Nothing specific here */
}

void
sp_object_invoke_build (SPObject * object, SPDocument * document, SPRepr * repr, gboolean cloned)
{
	const gchar * id;
	gchar * realid;
	gboolean ret;

	g_assert (object != NULL);
	g_assert (SP_IS_OBJECT (object));
	g_assert (document != NULL);
	g_assert (SP_IS_DOCUMENT (document));
	g_assert (repr != NULL);

	g_assert (object->document == NULL);
	g_assert (object->repr == NULL);
	g_assert (object->id == NULL);

	/* Bookkeeping */

	object->document = document;
	object->repr = repr;
	sp_repr_ref (repr);
	if (cloned) GTK_OBJECT_SET_FLAGS (object, SP_OBJECT_CLONED_FLAG);

	/* If we are not cloned, force unique id */
	if (!SP_OBJECT_IS_CLONED (object)) {
		id = sp_repr_attr (repr, "id");
		realid = sp_object_get_unique_id (object, id);
		g_assert (realid != NULL);
		sp_document_def_id (document, realid, object);
		object->id = realid;
		/* Redefine ID, if required */
		if ((id == NULL) || (strcmp (id, realid) != 0)) {
			ret = sp_repr_set_attr (repr, "id", realid);
			if (!ret) {
				g_error ("Cannot change id %s -> %s - probably there is stale ref", id, realid);
			}
		}
	} else {
		g_assert (object->id == NULL);
	}

	/* Invoke derived methods, if any */

	if (((SPObjectClass *)(((GtkObject *) object)->klass))->build)
		(*((SPObjectClass *)(((GtkObject *) object)->klass))->build) (object, document, repr);

	/* Signalling (should be connected AFTER processing derived methods */

	sp_repr_add_listener (repr, &object_event_vector, object);
}

void
sp_object_invoke_release (SPObject *object)
{
	g_assert (object != NULL);
	g_assert (SP_IS_OBJECT (object));

	/* Parent refcount us, so there shouldn't be any */
	g_assert (!object->parent);
	g_assert (!object->next);
	g_assert (object->document);
	g_assert (object->repr);

	gtk_signal_emit (GTK_OBJECT (object), object_signals[RELEASE]);

	/* href holders HAVE to release it in signal handler */
	g_assert (object->hrefcount == 0);

	if (object->style) {
		object->style = sp_style_unref (object->style);
	}

	if (!SP_OBJECT_IS_CLONED (object)) {
		g_assert (object->id);
		sp_document_undef_id (object->document, object->id);
		g_free (object->id);
		object->id = NULL;
	} else {
		g_assert (!object->id);
	}

	sp_repr_remove_listener_by_data (object->repr, object);
	sp_repr_unref (object->repr);

	object->document = NULL;
	object->repr = NULL;
}

static void
sp_object_repr_child_added (SPRepr *repr, SPRepr *child, SPRepr *ref, gpointer data)
{
	SPObject * object; 

	object = SP_OBJECT (data);

	if (((SPObjectClass *)(((GtkObject *) object)->klass))->child_added)
		(*((SPObjectClass *)(((GtkObject *) object)->klass))->child_added) (object, child, ref);

	sp_document_child_added (object->document, object, child, ref);
}

static gboolean
sp_object_repr_remove_child (SPRepr *repr, SPRepr *child, SPRepr *ref, gpointer data)
{
	SPObject * object;

	object = SP_OBJECT (data);

	if (((SPObjectClass *)(((GtkObject *) object)->klass))->remove_child)
		(* ((SPObjectClass *)(((GtkObject *) object)->klass))->remove_child) (object, child);

	sp_document_child_removed (object->document, object, child, ref);

	return TRUE;
}

/* fixme: */

static void
sp_object_repr_order_changed (SPRepr * repr, SPRepr * child, SPRepr * old, SPRepr * new, gpointer data)
{
	SPObject * object;

	object = SP_OBJECT (data);

	if (((SPObjectClass *) (((GtkObject *) object)->klass))->order_changed)
		(* ((SPObjectClass *)(((GtkObject *) object)->klass))->order_changed) (object, child, old, new);

	sp_document_order_changed (object->document, object, child, old, new);
}

static void
sp_object_read_attr (SPObject * object, const gchar * key)
{
	const gchar * id;

	g_assert (SP_IS_DOCUMENT (object->document));
	/* fixme: rething that cloning issue */
	g_assert (SP_OBJECT_IS_CLONED (object) || object->id != NULL);
	g_assert (key != NULL);

	if (strcmp (key, "id") == 0) {
		if (!SP_OBJECT_IS_CLONED (object)) {
			id = sp_repr_attr (object->repr, "id");
			g_assert (id != NULL);
			if (strcmp (id, object->id) == 0) return;
			g_assert (!sp_document_lookup_id (object->document, id));
			sp_document_undef_id (object->document, object->id);
			g_free (object->id);
			object->id = g_strdup (id);
			sp_document_def_id (object->document, object->id, object);
		} else {
			g_warning ("ID of cloned object changed, so document is out of sync");
		}
	}
}

void
sp_object_invoke_read_attr (SPObject * object, const gchar * key)
{
	g_assert (object != NULL);
	g_assert (SP_IS_OBJECT (object));
	g_assert (key != NULL);

	g_assert (SP_IS_DOCUMENT (object->document));
	g_assert (object->repr != NULL);
	/* fixme: rething that cloning issue */
	g_assert (SP_OBJECT_IS_CLONED (object) || object->id != NULL);

	if (((SPObjectClass *)(((GtkObject *) object)->klass))->read_attr)
		(*((SPObjectClass *)(((GtkObject *) object)->klass))->read_attr) (object, key);
}

static gboolean
sp_object_repr_change_attr (SPRepr *repr, const guchar *key, const guchar *oldval, const guchar *newval, gpointer data)
{
	SPObject * object;
	gpointer defid;

	object = SP_OBJECT (data);

	if (strcmp (key, "id") == 0) {
		defid = sp_document_lookup_id (object->document, newval);
		if (defid == object) return TRUE;
		if (defid) return FALSE;
	}

	return TRUE;
}

static void
sp_object_repr_attr_changed (SPRepr *repr, const guchar *key, const guchar *oldval, const guchar *newval, gpointer data)
{
	SPObject * object;

	object = SP_OBJECT (data);

	sp_object_invoke_read_attr (object, key);

	sp_document_attr_changed (object->document, object, key, oldval, newval);
}

static void
sp_object_repr_content_changed (SPRepr *repr, const guchar *oldcontent, const guchar *newcontent, gpointer data)
{
	SPObject * object;

	object = SP_OBJECT (data);

	if (((SPObjectClass *)(((GtkObject *) object)->klass))->read_content)
		(*((SPObjectClass *)(((GtkObject *) object)->klass))->read_content) (object);

	sp_document_content_changed (object->document, object, oldcontent, newcontent);
}

void
sp_object_invoke_forall (SPObject *object, SPObjectMethod func, gpointer data)
{
	SPRepr *repr, *child;

	g_return_if_fail (object != NULL);
	g_return_if_fail (SP_IS_OBJECT (object));
	g_return_if_fail (func != NULL);

	func (object, data);

#if 0

	if (((SPObjectClass *) (((GtkObject *) object)->klass))->forall)
		((SPObjectClass *) (((GtkObject *) object)->klass))->forall (object, func, data);
#else
	repr = SP_OBJECT_REPR (object);
	for (child = repr->children; child != NULL; child = child->next) {
		const unsigned char *id;
		SPObject *cho;
		id = sp_repr_attr (child, "id");
		cho = sp_document_lookup_id (SP_OBJECT_DOCUMENT (object), id);
		if (cho) sp_object_invoke_forall (cho, func, data);
	}
#endif
}

static SPRepr *
sp_object_private_write (SPObject *object, SPRepr *repr, guint flags)
{
	if (!repr && (flags & SP_OBJECT_WRITE_BUILD)) {
		repr = sp_repr_duplicate (SP_OBJECT_REPR (object));
	} else {
		sp_repr_set_attr (repr, "id", object->id);
	}

	return repr;
}

SPRepr *
sp_object_invoke_write (SPObject *object, SPRepr *repr, guint flags)
{
	g_return_val_if_fail (object != NULL, NULL);
	g_return_val_if_fail (SP_IS_OBJECT (object), NULL);

	if (((SPObjectClass *) (((GtkObject *) object)->klass))->write) {
		if (!(flags & SP_OBJECT_WRITE_BUILD) && !repr) {
			repr = SP_OBJECT_REPR (object);
		}
		return ((SPObjectClass *) (((GtkObject *) object)->klass))->write (object, repr, flags);
	} else {
		g_warning ("Class %s does not implement ::write", gtk_type_name (GTK_OBJECT_TYPE (object)));
		if (!repr) {
			if (flags & SP_OBJECT_WRITE_BUILD) {
				repr = sp_repr_duplicate (SP_OBJECT_REPR (object));
			}
			/* fixme: else probably error (Lauris) */
		} else {
			sp_repr_merge (repr, SP_OBJECT_REPR (object), "id");
		}
		return repr;
	}
}

/* Modification */

void
sp_object_request_modified (SPObject *object, guint flags)
{
	gboolean propagate;

	g_return_if_fail (object != NULL);
	g_return_if_fail (SP_IS_OBJECT (object));
	g_return_if_fail (!(flags & SP_OBJECT_PARENT_MODIFIED_FLAG));
	g_return_if_fail ((flags & SP_OBJECT_MODIFIED_FLAG) || (flags & SP_OBJECT_CHILD_MODIFIED_FLAG));
	g_return_if_fail (!((flags & SP_OBJECT_MODIFIED_FLAG) && (flags & SP_OBJECT_CHILD_MODIFIED_FLAG)));

	/* Check for propagate before we set any flags */
	/* Propagate means, that object is not passed through by modification request cascade yet */
	propagate = (!(SP_OBJECT_FLAGS (object) & (SP_OBJECT_MODIFIED_FLAG | SP_OBJECT_CHILD_MODIFIED_FLAG)));

	/* Just set object flags safe even if some have been set before */
	SP_OBJECT_SET_FLAGS (object, flags);

	if (propagate) {
		if (object->parent) {
			sp_object_request_modified (object->parent, SP_OBJECT_CHILD_MODIFIED_FLAG);
		} else {
			sp_document_request_modified (object->document);
		}
	}
}

void
sp_object_modified (SPObject *object, guint flags)
{
	g_return_if_fail (object != NULL);
	g_return_if_fail (SP_IS_OBJECT (object));
	g_return_if_fail (!(flags & ~SP_OBJECT_MODIFIED_CASCADE));

	flags |= (GTK_OBJECT_FLAGS (object) & SP_OBJECT_MODIFIED_STATE);

	g_return_if_fail (flags != 0);

	/* Merge style if we have good reasons to think that parent style is changed */
	/* I am not sure, whether we should check only propagated flag */
	/* We are currently assuming, that style parsing is done immediately */
	/* I think this is correct (Lauris) */
	if (flags & SP_OBJECT_STYLE_MODIFIED_FLAG) {
		if (object->style && object->parent) {
			sp_style_merge_from_parent (object->style, object->parent->style);
		}
	}

	gtk_object_ref (GTK_OBJECT (object));
	gtk_signal_emit (GTK_OBJECT (object), object_signals[MODIFIED], flags);
	gtk_object_unref (GTK_OBJECT (object));

	/* If style is modified, invoke style_modified virtual method */
	/* It is pure convenience, and should be used with caution */
	/* The cascade is created solely by modified method plus appropriate flag */
	/* Also, it merely signals, that actual style object has changed */
	if (flags & SP_OBJECT_STYLE_MODIFIED_FLAG) {
		if (((SPObjectClass *)(((GtkObject *) object)->klass))->style_modified)
			(*((SPObjectClass *)(((GtkObject *) object)->klass))->style_modified) (object, flags);
	}

	/*
	 * fixme: I am not sure - it was before class method, but moved it here
	 */

	SP_OBJECT_UNSET_FLAGS (object, SP_OBJECT_MODIFIED_STATE);
}

/* Sequence */
gint
sp_object_sequence (SPObject *object, gint seq)
{
	if (((SPObjectClass *)(((GtkObject *) object)->klass))->sequence)
		return (*((SPObjectClass *)(((GtkObject *) object)->klass))->sequence) (object, seq);

	return seq + 1;
}

const guchar *
sp_object_getAttribute (SPObject *object, const guchar *key, SPException *ex)
{
	g_return_val_if_fail (object != NULL, NULL);
	g_return_val_if_fail (SP_IS_OBJECT (object), NULL);
	g_return_val_if_fail (key != NULL, NULL);
	g_return_val_if_fail (*key != '\0', NULL);

	/* If exception is not clear, return */
	if (!SP_EXCEPTION_IS_OK (ex)) return NULL;

	return (const guchar *) sp_repr_attr (object->repr, key);
}

void
sp_object_setAttribute (SPObject *object, const guchar *key, const guchar *value, SPException *ex)
{
	g_return_if_fail (object != NULL);
	g_return_if_fail (SP_IS_OBJECT (object));
	g_return_if_fail (key != NULL);
	g_return_if_fail (*key != '\0');
	g_return_if_fail (value != NULL);

	/* If exception is not clear, return */
	g_return_if_fail (SP_EXCEPTION_IS_OK (ex));

	if (!sp_repr_set_attr (object->repr, key, value)) {
		ex->code = SP_NO_MODIFICATION_ALLOWED_ERR;
	}
}

void
sp_object_removeAttribute (SPObject *object, const guchar *key, SPException *ex)
{
	g_return_if_fail (object != NULL);
	g_return_if_fail (SP_IS_OBJECT (object));
	g_return_if_fail (key != NULL);
	g_return_if_fail (*key != '\0');

	/* If exception is not clear, return */
	g_return_if_fail (SP_EXCEPTION_IS_OK (ex));

	if (!sp_repr_set_attr (object->repr, key, NULL)) {
		ex->code = SP_NO_MODIFICATION_ALLOWED_ERR;
	}
}

/* Helper */

static gchar *
sp_object_get_unique_id (SPObject * object, const gchar * id)
{
	static gint count = 0;
	const gchar * name;
	gchar * realid;
	gchar * b;
	gint len;

	g_assert (SP_IS_OBJECT (object));
	g_assert (SP_IS_DOCUMENT (object->document));

	count++;

	name = sp_repr_name (object->repr);
	g_assert (name != NULL);
	len = strlen (name) + 17;
	b = alloca (len);
	g_assert (b != NULL);
	realid = NULL;

	if (id != NULL) {
		if (sp_document_lookup_id (object->document, id) == NULL) {
			realid = g_strdup (id);
			g_assert (realid != NULL);
		}
	}

	while (realid == NULL) {
		g_snprintf (b, len, "%s%d", name, count);
		if (sp_document_lookup_id (object->document, b) == NULL) {
			realid = g_strdup (b);
			g_assert (realid != NULL);
		} else {
			count++;
		}
	}

	return realid;
}

/* Style */

const guchar *
sp_object_get_style_property (SPObject *object, const gchar *key, const gchar *def)
{
	const gchar *style;
	const guchar *val;

	g_return_val_if_fail (object != NULL, NULL);
	g_return_val_if_fail (SP_IS_OBJECT (object), NULL);
	g_return_val_if_fail (key != NULL, NULL);

	style = sp_repr_attr (object->repr, "style");
	if (style) {
		gchar *p;
		gint len;
		len = strlen (key);
		for (p = strstr (style, key); p; p = strstr (style, key)) {
			p += len;
			while ((*p <= ' ') && *p) p++;
			if (*p++ != ':') break;
			while ((*p <= ' ') && *p) p++;
			if (*p) return p;
		}
	}
	val = sp_repr_attr (object->repr, key);
	if (val) return val;
	if (object->parent) {
		return sp_object_get_style_property (object->parent, key, def);
	}

	return def;
}

