;; Spelling correction interface for Emacs.

(defvar stava-command "stava -k"
  "*Command to run the spell program.")

(defvar stava-filter nil
  "*Filter function to process text before passing it to spell program.
This function might remove text-processor commands.
nil means don't alter the text before checking it.")

(defun stava-buffer ()
  "Check spelling of every word in the buffer.
For each incorrect word, you are asked for the correct spelling
and then put into a query-replace to fix some or all occurrences.
If you do not want to change a word, just give the same word
as its \"correct\" spelling; then the query replace is skipped."
  (interactive)
  (stava-region (point-min) (point-max) "bufferten"))

(defun stava-ord ()
  "Check spelling of word at or before point.
If it is not correct, ask user for the correct spelling
and query-replace the entire buffer to substitute it."
  (interactive)
  (let (beg end spell-filter)
    (save-excursion
     (if (not (looking-at "\\<"))
	 (backward-1-swedish-word))
     (setq beg (point))
     (forward-1-swedish-word)
     (setq end (point)))
    (stava-region beg end (buffer-substring beg end))))

(defun stava-region (start end &optional description)
  "Like stava-buffer but applies only to region.
Used in a program, applies from START to END.
DESCRIPTION is an optional string naming the unit being checked:
for example, \"word\"."
  (interactive "r")
  (let ((filter stava-filter)
	(buf (get-buffer-create " *temp*")))
    (save-excursion
     (set-buffer buf)
     (widen)
     (erase-buffer))
    (message "Stavningskontroll av `%s'..." (or description "regionen"))
    (if (and (null filter) (= ?\n (char-after (1- end))))
	(if (string= "stava" stava-command)
	    (call-process-region start end "stava" nil buf)
	  (call-process-region start end shell-file-name
			       nil buf nil "-c" stava-command))
      (let ((oldbuf (current-buffer)))
	(save-excursion
	 (set-buffer buf)
	 (insert-buffer-substring oldbuf start end)
	 (or (bolp) (insert ?\n))
	 (if filter (funcall filter))
	 (if (string= "stava" stava-command)
	     (call-process-region (point-min) (point-max) "stava" t buf)
	   (call-process-region (point-min) (point-max) shell-file-name
				t buf nil "-c" stava-command)))))
    (message "Stavningskontroll av `%s'...%s"
	     (or description "region")
	     (if (save-excursion
		  (set-buffer buf)
		  (> (buffer-size) 0))
		 "felaktigt"
	       "korrekt"))
    (let (word newword
	  (case-fold-search t)
	  (case-replace t))
      (while (save-excursion
	      (set-buffer buf)
	      (> (buffer-size) 0))
	(save-excursion
	 (set-buffer buf)
	 (goto-char (point-min))
	 (setq word (downcase
		     (buffer-substring (point)
				       (progn (end-of-line) (point)))))
	 (forward-char 1)
	 (delete-region (point-min) (point))
	 (setq newword
	       (read-input (concat "`" word
				   "' oknt ord; ge det ord du vill ha istllet: ")
			   word))
	 (flush-lines (concat "^" (regexp-quote word) "$")))
	(if (not (equal word newword))
	    (progn
	     (goto-char (point-min))
	     (query-replace-regexp (concat "\\b" (regexp-quote word) "\\b")
				   newword)))))))


(defun stava-string (string)
  "Check spelling of string supplied as argument."
  (interactive "sGe strngen: ")
  (let ((buf (get-buffer-create " *temp*")))
    (save-excursion
     (set-buffer buf)
     (widen)
     (erase-buffer)
     (insert string "\n")
     (if (string= "stava" stava-command)
	 (call-process-region (point-min) (point-max) "stava"
			      t t)
       (call-process-region (point-min) (point-max) shell-file-name
			    t t nil "-c" stava-command))
     (if (= 0 (buffer-size))
	 (message "%s ; r korrekt" string)
       (goto-char (point-min))
       (while (search-forward "\n" nil t)
	 (replace-match " "))
       (message "%sfelaktigt" (buffer-substring 1 (point-max)))))))

(defun forward-1-swedish-word ()
  (interactive)
  (forward-word 1)
  (let ((ch (char-after (point))))
    (if (or (char-equal ?} ch)
	    (char-equal ?{ ch)
            (char-equal ?\| ch)
            (char-equal ?] ch)
            (char-equal ?[ ch)
            (char-equal ?\\ ch)
            (char-equal ?- ch))
	(progn
	  (forward-char 1)
	  (forward-1-swedish-word)))))



(defun backward-1-swedish-word ()
  (interactive)
  (forward-word -1)
  (let ((ch (char-after (1- (point)))))
    (if (or (char-equal ?} ch)
	    (char-equal ?{ ch)
            (char-equal ?\| ch)
            (char-equal ?] ch)
            (char-equal ?[ ch)
            (char-equal ?\\ ch)
            (char-equal ?- ch))
	(progn
	  (forward-char -1)
	  (backward-1-swedish-word)))))
