
/******************************************************************************
* MODULE     : glue.gen.cc
* DESCRIPTION: Glue for linking TeXmacs commands to guile
* COPYRIGHT  : (C) 1999  Joris van der Hoeven
*******************************************************************************
* This software falls under the GNU general public license and comes WITHOUT
* ANY WARRANTY WHATSOEVER. See the file $TEXMACS_PATH/LICENSE for more details.
* If you don't have this file, write to the Free Software Foundation, Inc.,
* 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
******************************************************************************/

#include <server.gen.h>
#include <connect.gen.h>
#include <file.gen.h>
#include <dir.gen.h>

#module code_glue
#import server
#import connect
#import file
#import dir
#include <string.h>
#include <guile/gh.h>
#include <libguile.h>
#define SCM_ARG8 8
#define SCM_ARG9 9

#ifdef DOTS_OK
typedef SCM (*FN)(...);
#else
typedef SCM (*FN)();
#endif

extern void initialize_glue_basic ();
extern void initialize_glue_editor ();
extern void initialize_glue_server ();
extern void thomas_test (string s);

/******************************************************************************
* Booleans
******************************************************************************/

#define SCM_ASSERT_BOOL(flag,arg,rout) \
  SCM_ASSERT (gh_boolean_p (flag), flag, arg, rout)

static SCM
bool_to_scm (bool flag) {
  return gh_bool2scm (flag);
}

static bool
scm_to_bool (SCM flag) {
  return gh_scm2bool (flag);
}

/******************************************************************************
* Integers
******************************************************************************/

#define SCM_ASSERT_INT(i,arg,rout) \
  SCM_ASSERT (SCM_INUMP (i), i, arg, rout);

/*static*/
SCM
int_to_scm (int i) {
  return gh_long2scm ((long) i);
}

static int
scm_to_int (SCM i) {
  return (int) gh_scm2long (i);
}

/******************************************************************************
* Strings
******************************************************************************/

#define SCM_ASSERT_STRING(s,arg,rout) \
  SCM_ASSERT (SCM_NIMP (s) && SCM_STRINGP (s), s, arg, rout)

static SCM
string_to_scm (string s) {
  char* _s= as_charp (s);
  SCM r= gh_str2scm (_s, strlen (_s));
  delete[] _s;
  return r;
}

string
scm_to_string (SCM s) {
  guile_str_size_t len_r;
  char* _r= gh_scm2newstr (s, &len_r);
  string r= _r;
  free (_r);
  return r;
}

static SCM
symbol_to_scm (string s) {
  char* _s= as_charp (s);
  SCM r= gh_symbol2scm (_s);
  delete[] _s;
  return r;
}

static string
scm_to_symbol (SCM s) {
  guile_str_size_t len_r;
  char* _r= gh_symbol2newstr (s, &len_r);
  string r= _r;
  free (_r);
  return r;
}

/******************************************************************************
* Trees
******************************************************************************/

static long tree_tag;

#define SCM_ASSERT_TREE(t,arg,rout) \
  SCM_ASSERT ((SCM_NIMP (t) && (((long) SCM_CAR (t)) == tree_tag)), \
              t, arg, rout)

static SCM
tree_to_scm (tree t) {
  SCM tree_smob;
  SCM_NEWCELL (tree_smob);
  SCM_SETCDR (tree_smob, (SCM) ((void*) (new tree (t))));
  SCM_SETCAR (tree_smob, tree_tag);
  return tree_smob;
}

static tree
scm_to_tree (SCM tree_smob) {
  return *((tree*) SCM_CDR (tree_smob));
}

static SCM
mark_tree (SCM tree_smob) {
  (void) tree_smob;
  return SCM_BOOL_F;
}

static scm_sizet
free_tree (SCM tree_smob) {
  tree *ptr = (tree *) SCM_CDR (tree_smob);
  delete ptr;
  return sizeof (tree); // should be replaced by total size of the tree
}

static int
print_tree (SCM tree_smob, SCM port, scm_print_state *pstate) {
  (void) pstate;
  tree   t= scm_to_tree (tree_smob);
  string s= "<tree " * tree_to_texmacs (t) * ">";
  scm_display (string_to_scm (s), port);
  return 1;
}

static SCM
cmp_tree (SCM t1, SCM t2) {
  return gh_bool2scm (scm_to_tree (t1) == scm_to_tree (t2));
}

tree
coerce_string_tree (string s) {
  return s;
}

string
coerce_tree_string (tree t) {
  return as_string (t);
}

/******************************************************************************
* Scheme trees
******************************************************************************/

#define SCM_ASSERT_SCHEME_TREE(p,arg,rout)

/*static*/
SCM
scheme_tree_to_scm (scheme_tree t) {
  if (is_atomic (t)) {
    string s= t->label;
    if (s == "#t") return SCM_BOOL_T;
    if (s == "#f") return SCM_BOOL_F;
    if (is_int (s)) return int_to_scm (as_int (s));
    if ((N(s)>=2) && (s[0]=='\42') && (s[N(s)-1]='\42'))
      return string_to_scm (s (1, N(s)-1));
    return symbol_to_scm (t->label);
  }
  else {
    int i;
    SCM p= gh_list (SCM_UNDEFINED);
    for (i=N(t)-1; i>=0; i--)
      p= gh_cons (scheme_tree_to_scm (t[i]), p);
    return p;
  }
}

scheme_tree
scm_to_scheme_tree (SCM p) {
  if (gh_list_p (p)) {
    tree t (TUPLE);
    while (!gh_null_p (p)) {
      t << scm_to_scheme_tree (gh_car (p));
      p= gh_cdr (p);
    }
    return t;
  }
  if (gh_symbol_p (p)) {
    string s= scm_to_symbol (p);
    if (s == "quote") return "'"; else return s;
  }
  if (gh_string_p (p)) return "\"" * scm_to_string (p) * "\"";
  if (SCM_INUMP (p)) return as_string (scm_to_int (p));
  if (gh_boolean_p (p)) return (scm_to_bool (p)? string ("#t"): string ("#f"));
  return "?";
}

/******************************************************************************
* TeXmacs trees
******************************************************************************/

#define texmacs_tree tree
#define SCM_ASSERT_TEXMACS_TREE SCM_ASSERT_TREE
#define texmacs_tree_to_scm tree_to_scm
#define scm_to_texmacs_tree scm_to_tree

/******************************************************************************
* Initialization
******************************************************************************/

#ifdef SCM_NEWSMOB

void
initialize_glue () {
  tree_tag= scm_make_smob_type ("tree", 0);
  scm_set_smob_mark (tree_tag, mark_tree);
  scm_set_smob_free (tree_tag, free_tree);
  scm_set_smob_print (tree_tag, print_tree);
  scm_set_smob_equalp (tree_tag, cmp_tree);
  initialize_glue_basic ();
  initialize_glue_editor ();
  initialize_glue_server ();
}

#else

scm_smobfuns tree_smob_funcs = {
  mark_tree, free_tree, print_tree, cmp_tree
};

void
initialize_glue () {
  tree_tag= scm_newsmob (&tree_smob_funcs);
  initialize_glue_basic ();
  initialize_glue_editor ();
  initialize_glue_server ();
}

#endif

#endmodule // code_glue
