
/******************************************************************************
* MODULE     : tm_data.gen.cc
* DESCRIPTION: Buffer management for TeXmacs server
* COPYRIGHT  : (C) 1999  Joris van der Hoeven
*******************************************************************************
* This software falls under the GNU general public license and comes WITHOUT
* ANY WARRANTY WHATSOEVER. See the file $TEXMACS_PATH/LICENSE for more details.
* If you don't have this file, write to the Free Software Foundation, Inc.,
* 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
******************************************************************************/

#include <tm_data.gen.h>
#include <file.gen.h>
#include <array.gen.cc>
#include <iterator.gen.cc>

#module code_tm_data
#import tm_data
#import file
#define no_tree_converter<pointer>
#import code_array (pointer)
#import code_hashmap_iterator (string, tree)

string the_init_buffer_file= "$TEXMACS_HOME_PATH/progs/Init-buffer.scm";
int    max_undo_depth= 100; // should actually be part of tm_data_rep

/******************************************************************************
* Constructor and destructor
******************************************************************************/

tm_data_rep::tm_data_rep () {}
tm_data_rep::~tm_data_rep () {}

/******************************************************************************
* Low level functions for maintaining the buffer menu
******************************************************************************/

int
tm_data_rep::find_buffer (string name) {
  int i;
  for (i=0; i<N(bufs); i++)
    if (((tm_buffer) bufs[i])->name == name)
      return i;
  return -1;
}

string
tm_data_rep::new_menu_name (string name) {
  int i, j;
  for (i=N(name)-1; i>=0; i--)
    if (name[i]=='/') break;
  name= name (i+1, N(name));

  for (j=1; TRUE; j++) {
    bool flag= TRUE;
    string ret (name);
    if (j>1) ret= name * " (" * as_string (j) * ")";
    for (i=0; i<N(bufs); i++)
      if (((tm_buffer) bufs[i])->abbr==ret) flag= FALSE;
    if (flag) return ret;
  }
}

void
tm_data_rep::update_menu () {
  int i;
  string s ("(set! buffer-menu '(");
  for (i=0; i<N(bufs); i++) {
    if (i>0) s << " ";
    s << "(\"" << ((tm_buffer) bufs[i])->abbr << "\" ";
    s << "(switch-to-buffer \"" * ((tm_buffer) bufs[i])->name * "\"))";
  }
  s << "))";
  exec (s);
}

void
tm_data_rep::menu_insert_buffer (tm_buffer buf) {
  bufs << ((pointer) buf);
  update_menu ();
}

void
tm_data_rep::menu_delete_buffer (tm_buffer buf) {
  int i, nr, n=N(bufs);
  for (nr=0; nr<n; nr++)
    if (bufs[nr] == ((pointer) buf)) break;
  if (nr==N(bufs)) return;

  for (i=nr; i<(n-1); i++) bufs[i]= bufs[i+1];
  bufs->resize (n-1);
  update_menu();
}

void
tm_data_rep::menu_focus_buffer (tm_buffer buf) {
  int i, nr;
  for (nr=0; nr<N(bufs); nr++)
    if (bufs[nr] == ((pointer) buf)) break;
  if (nr==N(bufs)) return;

  for (i=nr; i>=1; i--) bufs[i]= bufs[i-1];
  bufs[0]= buf;
  update_menu ();
}

/******************************************************************************
* Low level buffer routines
******************************************************************************/

tm_buffer
tm_data_rep::new_buffer (string name) {
  int nr= find_buffer (name);
  if (nr != -1) return (tm_buffer) bufs[nr];
  tm_buffer buf= new tm_buffer_rep (name);
  buf->abbr= new_menu_name (name);
  menu_insert_buffer (buf);
  return buf;
}

tm_buffer
tm_data_rep::new_buffer (string name, tree doc) {
  int nr= find_buffer (name);
  if (nr != -1) return (tm_buffer) bufs[nr];
  tm_buffer buf= new_buffer (name);
  buf->t      = extract (doc, "body");
  buf->project= extract (doc, "project");
  buf->style  = extract (doc, "style");
  buf->init   = hashmap<string,tree> (UNINIT, extract (doc, "initial"));
  buf->fin    = hashmap<string,tree> (UNINIT, extract (doc, "final"));
  buf->ref    = hashmap<string,tree> (UNINIT, extract (doc, "references"));
  buf->aux    = hashmap<string,tree> (UNINIT, extract (doc, "auxiliary"));
  if (buf->project != "") {
    string prj_name= get_relative_file_name (name, as_string (buf->project));
    buf->prj= load_passive_buffer (prj_name);
  }
  return buf;
}

void
tm_data_rep::revert_buffer (string name, tree doc) {
  int i, nr= find_buffer (name);
  if (nr == -1) return;
  tm_buffer buf= (tm_buffer) bufs[nr];
  buf->project= extract (doc, "project");
  buf->style  = extract (doc, "style");
  buf->init   = hashmap<string,tree> (UNINIT, extract (doc, "initial"));
  buf->fin    = hashmap<string,tree> (UNINIT, extract (doc, "final"));
  buf->ref    = hashmap<string,tree> (UNINIT, extract (doc, "references"));
  buf->aux    = hashmap<string,tree> (UNINIT, extract (doc, "auxiliary"));
  if (N(buf->vws)==0) buf->t= extract (doc, "body");
  else for (i=0; i<N(buf->vws); i++) {
    tm_view vw= (tm_view) buf->vws[i];
    if (i==0) vw->ed->assign (path(), extract (doc, "body"));
    vw->ed->set_style (buf->style);
    vw->ed->set_init  (buf->init);
    vw->ed->set_fin   (buf->fin);
    vw->ed->notify_change (THE_DECORATIONS);
  }
  buf->mark_undo_block ();
  buf->need_save= buf->need_autosave= FALSE;
  buf->last_save= buf->last_autosave= buf->undo_depth- 1;
}

void
tm_data_rep::delete_buffer (tm_buffer buf) {
  int i;
  menu_delete_buffer (buf);
  for (i=0; i<N(buf->vws); i++)
    delete_view ((tm_view) buf->vws[i]);
  delete buf;
}

void
tm_data_rep::set_name_buffer (string s) {
  int i;
  tm_buffer buf= get_buffer ();
  if (buf->name == s) return;
  buf->name= s;
  buf->abbr= new_menu_name (s);
  update_menu ();
  for (i=0; i<N(buf->vws); i++) {
    tm_view vw2= (tm_view) buf->vws[i];
    if (vw2->win != NULL)
      vw2->win->win->set_name (buf->abbr);
  }
}

string
tm_data_rep::get_name_buffer () {
  tm_buffer buf= get_buffer ();
  return buf->name;
}

/******************************************************************************
* Low level view routines
******************************************************************************/

tm_view
tm_data_rep::new_view (string name) {
  // cout << "Creating new view\n";

  tm_buffer buf= new_buffer (name);
  editor    ed = new_editor (get_server (), get_display (), buf);
  tm_view   vw = new tm_view_rep (buf, ed);
  buf->vws << vw;

  ed->set_style (buf->style);
  ed->set_init (buf->init);
  ed->set_fin (buf->fin);
  ed->set_page_parameters ();
  ed->add_init (buf->init);
  ed->notify_change (THE_DECORATIONS);
  ed->notify_change (THE_AUTOMATIC_SIZE);

  tm_view temp_vw= get_view (FALSE);
  set_view (vw);
  if (file_exists (".", the_init_buffer_file))
    exec_file (".", the_init_buffer_file);
  else exec_file ("$TEXMACS_PATH/progs", "Init-buffer.scm");
  set_view (temp_vw);

  // cout << "View created\n";
  return vw;
}

tm_view
tm_data_rep::get_passive_view (tm_buffer buf) {
  int i;
  for (i=0; i<N(buf->vws); i++)
    if (((tm_view) buf->vws[i])->win == NULL)
      return (tm_view) buf->vws[i];
  return new_view (buf->name);
}

void
tm_data_rep::delete_view (tm_view vw) {
  tm_buffer buf= vw->buf;
  int i, j, n= N(buf->vws);
  for (i=0; i<n; i++)
    if (buf->vws[i] == vw) {
      array<tm_view> a (n-1);
      for (j=0; j<n-1; j++)
	if (j<i) a[j]= buf->vws[j];
	else a[j]= buf->vws[j+1];
      buf->vws= a;
    }
  delete vw;
}

void
tm_data_rep::attach_view (tm_window win, tm_view vw) {
  // cout << "Attach view " << vw->buf->name << "\n";
  vw->win= win;
  win->wid->a[0]->a[2]->a[0]->a[0]= vw->ed;
  if (win->wid->attached ()) {
    vw->ed->resume ();
    win->wid->set_window_name (vw->buf->abbr);
    win->wid->a[0] << emit_update ();
  }
  // cout << "View attached\n";
}

void
tm_data_rep::detach_view (tm_view vw) {
  // cout << "Detach view " << vw->buf->name << "\n";
  tm_window win= vw->win;
  if (win == NULL) return;
  vw->win= NULL;
  win->wid->a[0]->a[2]->a[0]->a[0]= glue_widget (get_display ());
  if (win->wid->attached ()) {
    vw->ed->suspend ();
    vw->ed << emit_attach_window (NULL);
    win->wid->set_window_name ("TeXmacs");
    win->wid->a[0] << emit_update ();
  }
  // cout << "View detached\n";
}

/******************************************************************************
* Low level window routines
******************************************************************************/

tm_window
tm_data_rep::new_window (display dis, bool map_flag) {
  tm_window win= new tm_window_rep (new tm_widget_rep (this, dis));
  if (map_flag) win->win->map ();
  return win;
}

bool
tm_data_rep::delete_view_from_window (tm_window win) {
  int i, j;
  for (i=0; i<N(bufs); i++) {
    tm_buffer buf= (tm_buffer) bufs[i];
    for (j=0; j<N(buf->vws); j++) {
      tm_view vw= (tm_view) buf->vws[j];
      if (vw->win == win) {
	detach_view (vw);
	delete_view (vw);
	return TRUE;
      }
    }
  }
  return FALSE;
}

void
tm_data_rep::delete_window (tm_window win) {
  while (delete_view_from_window (win));
  win->win->unmap ();
  delete win->win;
  delete win;
}

/******************************************************************************
* Other subroutines
******************************************************************************/

void
tm_data_rep::new_buffer_in_this_window (string name, tree doc) {
  int nr= find_buffer (name);
  if (nr != -1) switch_to_buffer (nr);
  else {
    (void) new_buffer (name, doc);
    switch_to_buffer (name);
  }
}

void
tm_data_rep::new_buffer_in_new_window (string name, tree doc) {
  tm_window win= new_window (get_display ());
  tm_buffer buf= new_buffer (name, doc);
  tm_view   vw = get_passive_view (buf);
  attach_view (win, vw);
  set_view (vw);
  menu_focus_buffer (buf);
}

/******************************************************************************
* Exported routines
******************************************************************************/

void
tm_data_rep::new_buffer () {
  int i;
  for (i=1; TRUE; i++) {
    string name= "no name " * as_string (i);
    if (i==1) name= "no name";
    if (find_buffer (name) != -1) continue;
    new_buffer_in_this_window (name, tree (DOCUMENT));
    return;
  }
}

void
tm_data_rep::switch_to_buffer (int nr) {
  // cout << "Switching to buffer " << nr << "\n";
  tm_window win    = get_window ();
  tm_buffer buf    = (tm_buffer) (bufs[nr]);
  tm_view   old_vw = get_view ();
  tm_view   new_vw = get_passive_view (buf);
  detach_view (old_vw);
  attach_view (win, new_vw);
  set_view (new_vw);
  menu_focus_buffer (buf);
  project_update_menu ();
  // cout << "Switched to buffer " << nr << "\n";
}

void
tm_data_rep::switch_to_buffer (string name) {
  int nr= find_buffer (name);
  if (nr == -1) {
    load_passive_buffer (name);
    nr= find_buffer (name);
  }
  if (nr != -1) switch_to_buffer (nr);
}

void
tm_data_rep::revert_buffer () {
  tm_buffer buf= get_buffer ();
  tree doc= file_to_tree ("", buf->name, buf->fm);
  if (doc == "error") set_message ("Error: file not found", "revert buffer");
  else revert_buffer (buf->name, doc);
}

void
tm_data_rep::kill_buffer () {
  int i, nr;
  if (N(bufs) <= 1) quit();
  tm_buffer buf= get_buffer();
  for (nr=0; nr<N(bufs); nr++) if (buf == bufs[nr]) break;
  if (nr == N(bufs))
    fatal_error ("Invalid situation", "tm_data_rep::kill_buffer");

  for (i=0; i<N(buf->vws); i++) {
    tm_view old_vw= (tm_view) buf->vws[i];
    if (old_vw->win != NULL) {
      tm_window win = old_vw->win;
      tm_view new_vw= get_passive_view ((tm_buffer) bufs[1]);
      detach_view (old_vw);
      attach_view (win, new_vw);
      if (get_view () == old_vw) set_view (new_vw);
    }
  }

  delete_buffer (buf);
}

void
tm_data_rep::open_window () {
  int i;
  for (i=1; TRUE; i++) {
    string name= "no name " * as_string (i);
    if (i==1) name= "no name";
    if (find_buffer (name) != -1) continue;
    new_buffer_in_new_window (name, tree (DOCUMENT));
    return;
  }
}

void
tm_data_rep::clone_window () {
  tm_window win= new_window (get_display ());
  tm_buffer buf= get_buffer ();
  tm_view   vw = get_passive_view (buf);
  attach_view (win, vw);
  set_view (vw);
  menu_focus_buffer (buf);
}

void
tm_data_rep::kill_window () {
  int i, j;
  tm_window win    = get_window ();
  for (i=0; i<N(bufs); i++) {
    tm_buffer buf= (tm_buffer) bufs[i];
    for (j=0; j<N(buf->vws); j++) {
      tm_view vw= (tm_view) buf->vws[j];
      if ((vw->win != NULL) && (vw->win != win)) {
	set_view (vw);
	menu_focus_buffer (vw->buf);
	delete_window (win);
	return;
      }
    }
  }
  quit ();
}

void
tm_data_rep::set_max_undo_depth (int i) {
  if (i <  0) i= -1;
  if (i == 0) i= 1;
  max_undo_depth= i;
}

int
tm_data_rep::get_max_undo_depth () {
  return max_undo_depth;
}

bool
tm_data_rep::no_bufs () {
  return N(bufs) == 0;
}

/******************************************************************************
* Projects
******************************************************************************/

void
tm_data_rep::project_attach (string prj_name) {
  int i;
  tm_buffer buf= get_buffer ();
  buf->project= prj_name;
  buf->need_save= buf->need_autosave= TRUE;
  for (i=0; i<N(buf->vws); i++) {
    tm_view vw= (tm_view) buf->vws[i];
    vw->ed->notify_change (THE_DECORATIONS);
  }
  if (prj_name == "") buf->prj= NULL;
  else {
    string full_name= get_relative_file_name (buf->name, prj_name);
    buf->prj= load_passive_buffer (full_name);
    project_update_menu ();
  }
}

bool
tm_data_rep::project_attached () {
  tm_buffer buf= get_buffer ();
  return buf->project != "";
}

void
tm_data_rep::project_update_menu () {
  tm_buffer buf= get_buffer ();
  if (buf->prj == NULL) return;
  string s ("(set! project-buffer-menu '(");
  s << "(\"" << buf->prj->abbr << "\" ";
  s << "(switch-to-buffer \"" * buf->prj->name * "\"))";

  tree t= buf->prj->t;
  int i, j, n= N(t);
  for (i=0; i<n; i++)
    if (is_func (t[i], INCLUDE, 1) && is_atomic (t[i][0])) {
      string name=
	get_relative_file_name (buf->prj->name, as_string (t[i][0]), FALSE);
      for (j=N(name)-1; j>=0; j--)
	if (name[j]=='/') break;
      s << " (\"" << name (j+1, N(name)) << "\" ";
      s << "(switch-to-buffer \"" * name * "\"))";
    }

  s << "))";
  exec (s);
}

/******************************************************************************
* Undo/redo routines for buffers as well as tests for (auto)saves
******************************************************************************/

void
tm_buffer_rep::mark_undo_block () {
  if ((undo == "nil") || (undo[0] != "")) {
    undo= tree (BACKUP, "", undo);
    undo_depth++;
  }
}

void
tm_buffer_rep::mark_redo_block () {
  if ((redo == "nil") || (redo[0] != "")) {
    redo= tree (BACKUP, "", redo);
    exdo= tree (BACKUP, "", exdo);
    redo_depth++;
  }
}

void
tm_buffer_rep::unmark_undo_block () {
  if ((undo != "nil") && undo[0] == "") {
    undo= undo[1];
    undo_depth--;
  }
}

void
tm_buffer_rep::unmark_redo_block () {
  if ((redo != "nil") && redo[0] == "") {
    redo= redo[1];
    exdo= exdo[1];
    redo_depth--;
  }
}

void
tm_buffer_rep::redo_to_undo () {
  if (redo == "nil") mark_undo_block ();
  else {
    tree re= redo[0];
    tree ex= exdo[0];
    redo= redo[1];
    exdo= exdo[1];
    if (re == "") redo_depth--;
    if (ex == "") mark_undo_block ();
    else undo= tree (BACKUP, ex, undo);
    redo_to_undo ();
    if (re == "") mark_undo_block ();
    else undo= tree (BACKUP, re, undo);
  }
}

void
tm_buffer_rep::truncate_undos (int nr) {
  int i;
  tree rev= "nil";
  for (i=0; i<nr; i++) {
    while (undo[0] != "") {
      rev= tree (BACKUP, undo[0], rev);
      undo= undo[1];
    }
    rev= tree (BACKUP, undo[0], rev);
    undo= undo[1];
  }

  undo= "nil";
  for (i=0; i<nr; i++) {
    undo= tree (BACKUP, rev[0], undo);
    rev= rev[1];
    while ((rev != "nil") && (rev[0] != "")) {
      undo= tree (BACKUP, rev[0], undo);
      rev= rev[1];
    }
  }

  int del= undo_depth- nr;
  undo_depth    -= del;
  last_save     -= del;
  last_autosave -= del;
}

bool
tm_buffer_rep::needs_to_be_saved () {
  if (need_save) return TRUE;
  if ((undo == "nil") || (undo[0] != ""))
    return (last_save != undo_depth);
  else return (last_save != (undo_depth-1));
}

bool
tm_buffer_rep::needs_to_be_autosaved () {
  if (!needs_to_be_saved ()) return FALSE;
  if (need_autosave) return TRUE;
  if ((undo == "nil") || (undo[0] != ""))
    return (last_autosave != undo_depth);
  else return (last_autosave != (undo_depth-1));
}

#endmodule // code_tm_data
