
/******************************************************************************
* MODULE     : tm_server.gen.cc
* DESCRIPTION: The TeXmacs server
* COPYRIGHT  : (C) 1999  Joris van der Hoeven
*******************************************************************************
* This software falls under the GNU general public license and comes WITHOUT
* ANY WARRANTY WHATSOEVER. See the file $TEXMACS_PATH/LICENSE for more details.
* If you don't have this file, write to the Free Software Foundation, Inc.,
* 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
******************************************************************************/

#include <tm_server.gen.h>
#include <analyze.gen.h>
#include <connect.gen.h>

#module code_tm_server
#import tm_server
#import analyze
#import connect

void initialize_guile ();
extern bool (*guile_eval_routine) (string);
extern string printing_dpi;
extern string printing_cmd;
extern string printing_on;

/******************************************************************************
* Texmacs server constructor and destructor
******************************************************************************/

string  the_init_file= "$TEXMACS_HOME_PATH/progs/Init.scm";
server* the_server   = NULL;

void
texmacs_interpose_handler () {
  if (the_server != NULL)
    (*the_server)->interpose_handler ();
}

void
texmacs_wait_handler (string message, string arg, int level) {
  (void) level;
  if (the_server != NULL)
    (*the_server)->wait_handler (message, arg);
}

bool
texmacs_guile_eval_routine (string s) {
  if (the_server == NULL) return TRUE;
  else {
    (*the_server)->exec_delayed (string_to_scheme_tree (s));
    return FALSE;
  }
}

server
get_server () {
  if (the_server == NULL)
    fatal_error ("TeXmacs server not yet started", "get_server");
  return *the_server;
}

server_rep::server_rep () {}
server_rep::~server_rep () {}

tm_server_rep::tm_server_rep (display dis2):
  dis (dis2), vw (NULL), banner_nr (-1)
{
  the_server= new server (this);
  initialize_guile ();
  set_interpose_handler (texmacs_interpose_handler);
  set_wait_handler (texmacs_wait_handler);
  guile_eval_routine= texmacs_guile_eval_routine;
  out_lan= dis->get_output_language ();
  if (file_exists (".", the_init_file)) exec_file (".", the_init_file);
  else exec_file ("$TEXMACS_PATH/progs", "Init.scm");
#os gnu-linux
  return; // in order to avoid segmentation faults
#os powerpc-gnu-linux
  return; // in order to avoid segmentation faults
#os all
}

void
tm_server_rep::advance_banner () {
  banner_nr++;
  if (get_editor()->et != tree (DOCUMENT, "")) banner_nr=5;
  if (banner_nr < 4) {
    static string banner[4]= {
      "Welcome to GNU TeXmacs",
      "GNU TeXmacs falls under the GNU general public license",
      "GNU TeXmacs comes without any form of legal warranty",
      "More information about GNU TeXmacs can be found in the help/about menu"
    };
    set_message (banner[banner_nr], "GNU TeXmacs " * TEXMACS_VERSION);
    dis->delayed_message (get_meta(), "banner", 2500);
  }

  else if (banner_nr == 4) set_message ("", "");
}

tm_server_rep::~tm_server_rep () {}
server::server (display dis): rep (new tm_server_rep (dis)) {}

/******************************************************************************
* Get and set objects associated to server
******************************************************************************/

server_rep*
tm_server_rep::get_server () {
  return this;
}

display
tm_server_rep::get_display () {
  return dis;
}

tm_view
tm_server_rep::get_view (bool must_be_valid) {
  if (must_be_valid && (vw==NULL))
    fatal_error ("No active view", "tm_server_rep::get_meta");
  return vw;
}

void
tm_server_rep::set_view (tm_view vw2) {
  vw= vw2;
}

tm_buffer
tm_server_rep::get_buffer () {
  tm_view vw= get_view ();
  return vw->buf;
}

editor
tm_server_rep::get_editor () {
  tm_view vw= get_view ();
  // cout << "Get editor" << vw->ed << "\n";
  return vw->ed;
}

tm_window
tm_server_rep::get_window () {
  tm_view vw= get_view ();
  if (vw->win==NULL)
    fatal_error ("No window attached to view", "tm_server_rep::get_meta");
  return vw->win;
}

tm_widget
tm_server_rep::get_meta () {
  tm_window win= get_window ();
  return win->wid;
}

/******************************************************************************
* Routines concerning the widget
******************************************************************************/

void
tm_server_rep::get_visible (SI& x1, SI& y1, SI& x2, SI& y2) {
  widget meta= (widget) get_meta ();
  meta["canvas"] << ::get_visible (x1, y1, x2, y2);
}

void
tm_server_rep::scroll_where (SI& x, SI& y) {
  widget meta= (widget) get_meta ();
  meta["canvas"] << get_coord2 ("scroll position", x, y);
}

void
tm_server_rep::scroll_to (SI x, SI y) {
  widget meta= (widget) get_meta ();
  meta["canvas"] << set_scroll_pos (x, y);
}

void
tm_server_rep::set_extents (SI x1, SI y1, SI x2, SI y2) {
  widget meta= (widget) get_meta ();
  meta["canvas"] << ::set_extents (x1, y1, x2, y2);
}

void
tm_server_rep::set_left_footer (string s) {
  if ((vw == NULL) || (vw->win == NULL)) return;
  get_meta()->set_left_footer (s);
}

void
tm_server_rep::set_right_footer (string s) {
  if ((vw == NULL) || (vw->win == NULL)) return;
  get_meta()->set_right_footer (s);
}

void
tm_server_rep::set_message (string left, string right) {
  get_editor()->set_message (left, right);
}

void
tm_server_rep::interactive (string name, string& s, command call_back) {
  get_meta()->interactive (name, s, call_back);
}

/******************************************************************************
* Miscellaneous routines
******************************************************************************/

void
tm_server_rep::interpose_handler () {
  listen_to_pipes ();
  exec_pending_commands ();

  int i,j;
  for (i=0; i<N(bufs); i++) {
    tm_buffer buf= (tm_buffer) bufs[i];
    for (j=0; j<N(buf->vws); j++) {
      tm_view vw= (tm_view) buf->vws[j];
      if (vw->win != NULL) vw->ed->process_extern_input ();
    }
    for (j=0; j<N(buf->vws); j++) {
      tm_view vw= (tm_view) buf->vws[j];
      if (vw->win != NULL) vw->ed->apply_changes ();
    }
  }
}

void
tm_server_rep::wait_handler (string message, string arg) {
  dis->set_wait_indicator (message, arg);
}

void
tm_server_rep::set_script_status (int i) {
  script_status= i;
}

void
tm_server_rep::focus_on_editor (editor ed) {
  int i,j;
  for (i=0; i<N(bufs); i++) {
    tm_buffer buf= (tm_buffer) bufs[i];
    for (j=0; j<N(buf->vws); j++) {
      tm_view vw= (tm_view) buf->vws[j];
      if (vw->ed == ed) {
	set_view (vw);
	return;
      }
    }
  }
  fatal_error ("Invalid situation", "tm_server_rep::focus_on_editor");
}

void
tm_server_rep::set_printing_command (string cmd) {
  printing_cmd= cmd;
}

void
tm_server_rep::set_printer_page_type (string type) {
  printing_on= type;
}

void
tm_server_rep::set_printer_dpi (string dpi) {
  printing_dpi= dpi;
}

void
tm_server_rep::postscript_gc (string which) {
  dis->postscript_gc (which);
  int i,j;
  for (i=0; i<N(bufs); i++) {
    tm_buffer buf= (tm_buffer) bufs[i];
    for (j=0; j<N(buf->vws); j++) {
      tm_view vw= (tm_view) buf->vws[j];
      vw->ed->typeset_invalidate_all ();
    }
  }  
}

void
tm_server_rep::inclusions_gc (string which) {
  (void) which;
  reset_inclusions ();
  int i,j;
  for (i=0; i<N(bufs); i++) {
    tm_buffer buf= (tm_buffer) bufs[i];
    for (j=0; j<N(buf->vws); j++) {
      tm_view vw= (tm_view) buf->vws[j];
      vw->ed->typeset_invalidate_all ();
    }
  }  
}

bool
tm_server_rep::is_yes (string s) {
  s= locase_all (s);
  return
    (s == "yes") || (s == "y") ||
    (s == "oui") || (s == "o") ||
    (s == "ja") || (s == "j") ||
    (s == "si") || (s == "s") ||
    (s == "sim") || (s == "s") ||
    (s == "tak") || (s == "t");
}

void
tm_server_rep::quit () {
  connection_stop_all ();
  exit (0);
}

/******************************************************************************
* Extern packages
******************************************************************************/

void
tm_server_rep::declare (string name, string lib, string symb, string init) {
  package_declare (name, lib, symb, init);
}

tree
tm_server_rep::evaluate (string name, string session, tree expr) {
  if (!package_declared (name)) {
    set_message ("Package#'" * name * "'#not declared",
		 "Evaluate#'" * name * "'#expression");
    return "";
  }
  if (!package_installed (name)) {
    string r= package_install (name);
    set_message (r, "Install#'" * name * "'#package");
    if (!package_installed (name)) return "";
  }
  return package_evaluate (name, session, expr);
}

string
tm_server_rep::execute (string name, string session, string expr) {
  if (!package_declared (name)) {
    set_message ("Package#'" * name * "'#not declared",
		 "Evaluate#'" * name * "'#expression");
    return "";
  }
  if (!package_installed (name)) {
    string r= package_install (name);
    set_message (r, "Install#'" * name * "'#package");
    if (!package_installed (name)) return "";
  }
  return package_execute (name, session, expr);
}

/******************************************************************************
* System commands
******************************************************************************/

void
tm_server_rep::shell (string s) {
  system (s);
}

#endmodule // code_tm_server
