#
# TCL Library for tkcvs
#

#
# $Id: workdir.tcl,v 1.62 2000/10/11 05:26:43 dorothyr Exp $
#
# Current working directory display.  Handles all of the functions
# concerned with navigating about the current directory on the main
# window.
#

proc workdir_setup {} {
  global cwd
  global module_dir
  global cvscfg
  global current_tagname
  global feedback
  global logclass

  gen_log:log T "ENTER"
  set cwd [pwd]
  set pid [pid]

  # Window manager stuff
  wm title . "TkCVS"
  wm iconname . "TkCVS"
  wm iconbitmap . @$cvscfg(bitmapdir)/tkcvs48.xbm
  wm minsize . 5 10

  workdir_menus

  #
  # Top section - where we are, where the module is
  #
  frame .top -relief groove -border 2 -width 480
  frame .top.cwd
  frame .top.root
  frame .top.dtag
  pack .top -side top -fill x

  pack .top.cwd -side top -fill x -expand yes
  label .top.cwd.lcwd -text "Current Directory" \
     -anchor w -width 18 -font $cvscfg(guifont)
  entry .top.cwd.tcwd -textvariable cwd
  pack .top.cwd.lcwd -side left -fill x
  pack .top.cwd.tcwd -side left -fill x -expand yes

  label .top.root.lmodule -text "Module Location" \
     -anchor w -width 18 -font $cvscfg(guifont)
  label .top.root.tmodule -textvariable module_dir \
     -anchor w -font $cvscfg(guifont)

  # Make the Module Browser button prominent
  button .bmodbrowse -image Modules \
     -command { modbrowse_run $cvscfg(cvsroot) }
  pack .bmodbrowse -in .top \
     -side right -anchor ne -ipadx 1 -ipady 1

  pack .top.root -side top -fill x -expand yes
  pack .top.root.lmodule -side left -fill x
  pack .top.root.tmodule -side left -fill x -expand yes

  pack .top.dtag -side top -fill x -expand yes
  label .top.dtag.ltagname -text "Directory Tag" \
     -anchor w -width 18 -font $cvscfg(guifont)
  label .top.dtag.ttagname -textvariable current_tagname \
     -anchor w -font $cvscfg(guifont)
  pack .top.dtag.ltagname -side left -fill x
  pack .top.dtag.ttagname -side left -fill x -expand yes

  bind .top.cwd.tcwd <Return> {change_dir "$cwd"}

  # Pack the bottom before the middle so it doesnt disappear if
  # the window is resized smaller
  frame .bottom -relief groove -border 2 -height 128
  frame .bottom.filters
  pack .bottom -side bottom -fill x
  pack .bottom.filters -side top -fill x

  label .bottom.filters.showlbl -text "Show:" -anchor w -font $cvscfg(guifont)
  entry .bottom.filters.showentry -textvariable cvscfg(file_filter) -width 12
  label .bottom.filters.hidelbl -text "   Hide:" -anchor w \
     -font $cvscfg(guifont)
  entry .bottom.filters.hideentry -width 12 \
     -textvariable cvscfg(default_ignore_filter)
  label .bottom.filters.space -text "    " -anchor w -font $cvscfg(guifont)
  button .bottom.filters.cleanbutton -text "Clean:" -font $cvscfg(guifont) \
     -command workdir_cleanup
  entry .bottom.filters.cleanentry -width 12 \
     -textvariable cvscfg(clean_these)
  bind .bottom.filters.showentry <Return> {setup_dir}
  bind .bottom.filters.hideentry <Return> {
     set cvscfg(default_ignore_filter) [.bottom.filters.hideentry get]
     setup_dir}
  bind .bottom.filters.cleanentry <Return> {workdir_cleanup}
  pack .bottom.filters.showlbl -side left
  pack .bottom.filters.showentry -side left
  pack .bottom.filters.hidelbl -side left
  pack .bottom.filters.hideentry -side left
  pack .bottom.filters.space -side left
  pack .bottom.filters.cleanbutton -side left -ipadx 0 -ipady 0
  pack .bottom.filters.cleanentry -side left


  frame .bottom.buttons
  frame .bottom.buttons.dirfuncs -relief groove -bd 2
  frame .bottom.buttons.cvsfuncs -relief groove -bd 2
  frame .bottom.buttons.modfuncs -relief groove -bd 2
  pack .bottom.buttons -side top -fill x -expand yes
  pack .bottom.buttons.dirfuncs -side left -fill x -expand yes
  pack .bottom.buttons.cvsfuncs -side left -fill x -expand yes
  pack .bottom.buttons.modfuncs -side left -fill x -expand yes

  #
  # Action buttons along the bottom of the screen.
  #
  button .bedit_files -image Fileedit \
     -command { workdir_edit_file [workdir_list_files] }
  button .bdelete_file -image Delete \
     -command { workdir_delete_file [workdir_list_files] }
  button .bclear -image Clear \
     -command { .main.file_list.list select clear 0 end }
  button .brefresh -image Refresh \
     -command setup_dir
  button .bconflict -image Conflict \
     -command { cvs_merge_conflict [workdir_list_files] }

  button .blogfile -image Branches \
     -command { cvs_logcanvas [workdir_list_files] }
  button .btag -image Tag \
     -command file_tag_dialog
  button .badd_files -image Add \
     -command { add_dialog [workdir_list_files] }
  button .bremove -image Remove \
     -command { subtract_dialog [workdir_list_files] }
  button .bdiff -image Diff \
     -command { cvs_diff [workdir_list_files] }
  button .bcheckin -image Checkin \
     -command commit_run
  button .bupdate -image Checkout \
     -command { cvs_update "BASE" "Normal" \
        "Remove" "No" " " [workdir_list_files] }
  button .beditflag_files -image Edit \
     -command { cvs_edit [workdir_list_files] }
  button .buneditflag_files -image Unedit \
     -command { cvs_unedit [workdir_list_files] }
  button .bquit -text "Quit" -font $cvscfg(guifont) \
     -command exit_cleanup

  pack .bedit_files \
       .bdelete_file \
       .bclear \
       .brefresh \
    -in .bottom.buttons.dirfuncs \
    -side left -ipadx 1 -ipady 1 -fill x -expand 1

  pack .blogfile \
       .bdiff \
       .badd_files \
       .bremove \
       .bconflict \
     -in .bottom.buttons.cvsfuncs \
     -side left -ipadx 1 -ipady 1 -fill x -expand 1

  pack .bcheckin \
       .bupdate \
       .btag \
     -in .bottom.buttons.modfuncs \
     -side left -ipadx 1 -ipady 1 -fill x -expand 1
  if {$cvscfg(econtrol)} {
    pack .beditflag_files \
         .buneditflag_files \
       -in .bottom.buttons.modfuncs \
       -side left -ipadx 1 -ipady 1 -fill x -expand 1
    }
  pack .bquit \
     -in .bottom.buttons.modfuncs \
     -side right -ipadx 0 -ipady 0 -fill x -expand 1

  set_tooltips .bedit_files \
     {"Edit the selected files"}
  set_tooltips .bdelete_file \
     {"Delete the selected files from the current directory"}
  set_tooltips .bclear \
     {"Unselect all files"}
  set_tooltips .brefresh \
     {"Re-read the current directory"}
  set_tooltips .bconflict \
     {"Merge Conflicts using TkDiff"}
  set_tooltips .blogfile \
     {"See the revision log and branches of the selected files"}
  set_tooltips .badd_files \
     {"Add the selected files to the repository"}
  set_tooltips .btag \
     {"Tag the selected files"}
  set_tooltips .bremove \
     {"Remove the selected files from the repository"}
  set_tooltips .bcheckin \
     {"Check in (commit) the selected files to the repository"}
  set_tooltips .bupdate \
     {"Update (checkout, patch) the selected files from the repository"}
  if {$cvscfg(econtrol)} {
    set_tooltips .beditflag_files \
       {"Set the Edit flag on the selected files"}
    set_tooltips .buneditflag_files \
       {"Reset the Edit flag on the selected files"}
  }
  set_tooltips .bdiff \
     {"Compare the selected files with the repository version"}
  set_tooltips .bmodbrowse \
     {"Open the Module Browser"}
  set_tooltips .bquit \
     {"Exit from TkCVS"}


  #
  # Entry widget to be used for feedback
  #
  set feedback(cvs) [entry .bottom.feedback -width 55 -state disabled]
  pack .bottom.feedback -side bottom -fill x -expand yes

  #
  # The listboxes - where the files, their status and tag are listed
  #
  frame .main
  pack .main -side bottom -fill both -expand yes

  # Try to make sure the window isn't taller than the screen
  set screenHeight [winfo vrootheight .]
  set linespace_gui [font metrics $cvscfg(guifont) -linespace]
  set linespace_list [font metrics $cvscfg(listboxfont) -linespace]
  set mainreq [expr {$cvscfg(y_size) * $linespace_list}]
  set bottomreq [winfo reqheight .bottom]
  set widgetreq [expr {$bottomreq * 2}]
  set totalreq [expr {$widgetreq + $mainreq}]
  gen_log:log D "Total $totalreq, $screenHeight available"
  if {$totalreq > $screenHeight} {
    # Leave a few pixels for the window frame
    set avail [expr {$screenHeight - $widgetreq - 20}]
    gen_log:log D "($avail / $linespace_list)"
    set cvscfg(y_size) [expr {$avail / $linespace_list}]
    gen_log:log D "reducing y_size to $cvscfg(y_size) lines"
  }

  frame .main.file_list
  frame .main.status_list
  frame .main.tag_list
  frame .main.edit_list

  listbox .main.file_list.list -yscroll {workdir_scroll_file_list} \
    -relief sunken -height $cvscfg(y_size) -width 0 -setgrid yes \
    -selectmode extended -font $cvscfg(listboxfont)
  listbox .main.status_list.list -yscroll {workdir_scroll_status_list} \
    -relief sunken -height $cvscfg(y_size) -width 0 -setgrid yes \
    -font $cvscfg(listboxfont)
  listbox  .main.tag_list.list -yscroll {workdir_scroll_tag_list} \
    -relief sunken -height $cvscfg(y_size) -width 0 -setgrid yes \
    -font $cvscfg(listboxfont)
  listbox  .main.edit_list.list -yscroll {workdir_scroll_edit_list} \
    -relief sunken -height $cvscfg(y_size) -width 0 -setgrid yes \
    -font $cvscfg(listboxfont)
  scrollbar .main.scroll -command {workdir_scroll_scrollbar} -relief sunken

  frame .main.file_list.head -relief raised -bd 1
  frame .main.status_list.head -relief raised -bd 1
  frame .main.tag_list.head -relief raised -bd 1
  frame .main.edit_list.head -relief raised -bd 1

  label .main.file_list.head.lbl -text "file" -font $cvscfg(guifont)
  label .main.status_list.head.lbl -text "status" -font $cvscfg(guifont)
  label .main.tag_list.head.lbl -text "tag" -font $cvscfg(guifont)
  label .main.edit_list.head.lbl -text "editors" -font $cvscfg(guifont)

  button .main.file_list.head.sbut -relief flat -highlightthickness 0 -image arh_dn
  button .main.status_list.head.sbut -relief flat -highlightthickness 0 -image arr_dn
  button .main.tag_list.head.sbut -relief flat -highlightthickness 0 -image arr_dn
  button .main.edit_list.head.sbut -relief flat -highlightthickness 0 -image arr_dn

  pack .main.scroll -side right -fill both -expand no
  pack .main.file_list -side left -fill both -expand yes
  pack .main.status_list -side left -fill both -expand yes
  pack .main.tag_list -side left -fill both -expand yes
  if {$cvscfg(econtrol)} {
    pack .main.edit_list -side left -fill both -expand yes
  }

  pack .main.file_list.head -side top -fill x -expand no
  pack .main.status_list.head -side top -fill x -expand no
  pack .main.tag_list.head -side top -fill x -expand no
  pack .main.file_list.head.lbl -side left -fill x -expand yes
  pack .main.status_list.head.lbl -side left -fill x -expand yes
  pack .main.tag_list.head.lbl -side left -fill x -expand yes
  pack .main.file_list.head.sbut -side right
  pack .main.status_list.head.sbut -side right
  pack .main.tag_list.head.sbut -side right
  if {$cvscfg(econtrol)} {
    pack .main.edit_list.head -side top -fill x -expand no
    pack .main.edit_list.head.lbl -side left -fill x -expand yes
    pack .main.edit_list.head.sbut -side right
  }

  pack .main.file_list.list -side top -fill both -expand yes
  pack .main.status_list.list -side top -fill both -expand yes
  pack .main.tag_list.list -side top -fill both -expand yes
  if {$cvscfg(econtrol)} {
    pack .main.edit_list.list -side top -fill both -expand yes
  }

  bind .main.file_list.head.sbut <ButtonPress-1> { toggle_col 0 }
  bind .main.file_list.head.sbut <ButtonPress-2> { sort_by_col 0 inc }
  bind .main.file_list.head.sbut <ButtonPress-3> { sort_by_col 0 dec }
  bind .main.status_list.head.sbut <ButtonPress-1> { toggle_col 1 }
  bind .main.status_list.head.sbut <ButtonPress-2> { sort_by_col 1 inc }
  bind .main.status_list.head.sbut <ButtonPress-3> { sort_by_col 1 dec }
  bind .main.tag_list.head.sbut <ButtonPress-1> { toggle_col 2 }
  bind .main.tag_list.head.sbut <ButtonPress-2> { sort_by_col 2 inc }
  bind .main.tag_list.head.sbut <ButtonPress-3> { sort_by_col 2 dec }
  bind .main.edit_list.head.sbut <ButtonPress-1> { toggle_col 3 }
  bind .main.edit_list.head.sbut <ButtonPress-2> { sort_by_col 3 inc }
  bind .main.edit_list.head.sbut <ButtonPress-3> { sort_by_col 3 dec }

  bind .main.file_list.list  <Double-Button-1> \
     { workdir_edit_file [workdir_list_files] }
  bind .main.file_list.list  <Button-2>         { nop }
  bind .main.file_list.list  <ButtonRelease-3>  { nop }
  bind .main.status_list.list <Double-Button-1> { nop }
  bind .main.status_list.list <ButtonRelease-1> { nop }
  bind .main.status_list.list <1>               { nop }
  bind .main.status_list.list <2>               { nop }
  bind .main.status_list.list <Any-B1-Motion>   { nop }
  bind .main.status_list.list <Any-B2-Motion>   { nop }
  bind .main.status_list.list <Any-B3-Motion>   { nop }


  # Do the directory listing.  We go to change_dir so that the original
  # working directory gets in the Go menu
  change_dir "[pwd]"
  gen_log:log T "LEAVE"
}

proc workdir_menus {} {
  global cvscfg
  global usermenu
  global cvsmenu

  gen_log:log T "ENTER"
  set startdir "[pwd]"

  frame .menubar -relief raised -bd 2
  pack .menubar -side top -fill x

  #
  # Create the Menu bar
  #
  menubutton .menubar.file -text File -underline 0 \
     -menu .menubar.file.m -font $cvscfg(guifont)
  menubutton .menubar.reports -text Reports -underline 0 \
     -menu .menubar.reports.m -font $cvscfg(guifont)
  menubutton .menubar.options -text Options -underline 0 \
     -menu .menubar.options.m -font $cvscfg(guifont)
  menubutton .menubar.help -text Help -underline 0 \
     -menu .menubar.help.m -font $cvscfg(guifont)
  pack .menubar.file \
       .menubar.reports \
       .menubar.options \
     -side left
  if { [info exists cvsmenu] || [info exists usermenu] } {
    menubutton .menubar.user -text "User Defined" -underline 0 \
       -menu .menubar.user.m -font $cvscfg(guifont)
    pack .menubar.user -side left
    gen_log:log T "Adding user defined menu"
  }
  menubutton .menubar.goto -text "Go" -underline 0 \
     -menu .menubar.goto.m -font $cvscfg(guifont)
  pack .menubar.goto -side left
  pack .menubar.help -side right

  #
  # Create the Menus
  #
  menu .menubar.file.m -font $cvscfg(guifont)
  .menubar.file.m add command -label "Open" -underline 0 \
     -command { workdir_edit_file [workdir_list_files] }
  .menubar.file.m add command -label "Print" -underline 0 \
     -command { workdir_print_file  [workdir_list_files ] }
  .menubar.file.m add separator
  .menubar.file.m add command -label "Browse Modules" -underline 0 \
     -command { modbrowse_run $cvscfg(cvsroot) }
  .menubar.file.m add command -label "Cleanup" -underline 0 \
     -command workdir_cleanup
  .menubar.file.m add separator
  .menubar.file.m add command -label "Check Out (Update)" -underline 11 \
     -command { \
        cvs_update {BASE} {Normal} {Remove} {No} { } [workdir_list_files] }
  .menubar.file.m add command -label "Check In (Commit)" -underline 6 \
     -command commit_run
  .menubar.file.m add command -label "Add Files" -underline 0 \
     -command { add_dialog [workdir_list_files] }
  .menubar.file.m add command -label "Remove Files" -underline 0 \
     -command { subtract_dialog [workdir_list_files] }
  .menubar.file.m add command -label "Set Edit Flag (Edit)" -underline 15 \
     -command { cvs_edit [workdir_list_files] }
  .menubar.file.m add command -label "Reset Edit Flag (Unedit)" -underline 11 \
     -command { cvs_unedit [workdir_list_files] }
  .menubar.file.m add command -label "Tag Files" -underline 0 \
     -command file_tag_dialog
  .menubar.file.m add command -label "Log Browse" -underline 0 \
     -command { cvs_logcanvas [workdir_list_files] }
  .menubar.file.m add command -label "Merge Conflict" -underline 0 \
     -command { cvs_merge_conflict [workdir_list_files] }
  .menubar.file.m add separator
  .menubar.file.m add command -label "Shell window" -underline 0 \
     -command {eval exec $cvscfg(shell) >& $cvscfg(null) &}
  .menubar.file.m add separator
  .menubar.file.m add command -label Exit -underline 1 \
     -command exit

  menu .menubar.reports.m -font $cvscfg(guifont)
  .menubar.reports.m add command -label "Check Directory" -underline 0 \
     -command cvs_check
  .menubar.reports.m add command -label "CVS status" -underline 4 \
     -command { cvs_status [workdir_list_files] }
  .menubar.reports.m add command -label "Sticky status" -underline 4 \
     -command { cvs_tag_status [workdir_list_files] }
  .menubar.reports.m add command -label "CVS diff" -underline 4 \
     -command { cvs_diff [workdir_list_files] }
  .menubar.reports.m add command -label "CVS log" -underline 4 \
     -command { cvs_log [workdir_list_files] }
  .menubar.reports.m add command -label "CVS annotate" -underline 4 \
     -command { cvs_annotate [workdir_list_files] }


  menu .menubar.options.m -font $cvscfg(guifont)
  .menubar.options.m add command -label "Checkout with Options" -underline 0 \
     -command update_run
  .menubar.options.m add separator
  .menubar.options.m add checkbutton -label "Show hidden files" \
     -variable cvscfg(allfiles) -onvalue true -offvalue false \
     -command setup_dir
  .menubar.options.m add checkbutton -label "Automatic directory status" \
     -variable cvscfg(auto_status) -onvalue true -offvalue false
  .menubar.options.m add checkbutton -label "Confirmation Dialogs" \
     -variable cvscfg(confirm_prompt) -onvalue true -offvalue false
  .menubar.options.m add checkbutton -label "Editors Column" \
     -variable cvscfg(econtrol) -onvalue true -offvalue false \
     -command econtrol_toggle
  .menubar.options.m add separator
  .menubar.options.m add checkbutton -label "Report Check recursively" \
     -variable cvscfg(checkrecursive) -onvalue {} -offvalue -l
  .menubar.options.m add checkbutton -label "Report Status Recursively" \
     -variable cvscfg(recurse) -onvalue true -offvalue false
  .menubar.options.m add cascade -label "CVS Status Detail" \
     -menu .menubar.options.m.report_detail
  .menubar.options.m add cascade -label "CVS Log Detail" \
     -menu .menubar.options.m.logfile_detail
  .menubar.options.m add separator
  .menubar.options.m add checkbutton -label "Tracing On/Off" \
     -variable cvscfg(logging) -onvalue true -offvalue false \
     -command log_toggle
  .menubar.options.m add cascade -label "Trace Level" \
     -menu .menubar.options.m.trace_level
  .menubar.options.m add separator
  .menubar.options.m add command -label "Save Options" -underline 0 \
     -command save_options

  menu .menubar.options.m.trace_level -font $cvscfg(guifont)
  .menubar.options.m.trace_level add checkbutton -label "CVS commands (C)" \
     -variable logclass(C) -onvalue "C" -offvalue "" \
     -command gen_log:changeclass
  .menubar.options.m.trace_level add checkbutton -label "File creation/deletion (F)" \
     -variable logclass(F) -onvalue F -offvalue "" \
     -command gen_log:changeclass
  .menubar.options.m.trace_level add checkbutton -label "Function entry/exit (T)" \
     -variable logclass(T) -onvalue T -offvalue "" \
     -command gen_log:changeclass
  .menubar.options.m.trace_level add checkbutton -label "Debugging (D)" \
     -variable logclass(D) -onvalue D -offvalue "" \
     -command gen_log:changeclass

  menu .menubar.options.m.report_detail -font $cvscfg(guifont)
  .menubar.options.m.report_detail add radiobutton -label "Verbose" \
     -variable cvscfg(rdetail) -value "verbose"
  .menubar.options.m.report_detail add radiobutton -label "Summary" \
     -variable cvscfg(rdetail) -value "summary"
  .menubar.options.m.report_detail add radiobutton -label "Terse" \
     -variable cvscfg(rdetail) -value "terse"

  menu .menubar.options.m.logfile_detail -font $cvscfg(guifont)
  .menubar.options.m.logfile_detail add radiobutton -label "Latest" \
     -variable cvscfg(ldetail) -value "latest"
  .menubar.options.m.logfile_detail add radiobutton -label "Summary" \
     -variable cvscfg(ldetail) -value "summary"
  .menubar.options.m.logfile_detail add radiobutton -label "Verbose" \
     -variable cvscfg(ldetail) -value "verbose"

  menu .menubar.goto.m -font $cvscfg(guifont)
  .menubar.goto.m add command -label "Home" \
     -command {change_dir ~}
  .menubar.goto.m add command -label $startdir \
     -command [format {change_dir "%s"} $startdir]

  menu .menubar.help.m -font $cvscfg(guifont)
  .menubar.help.m add command -label "About TkCVS" -underline 0 \
     -command aboutbox
  .menubar.help.m add command -label "About CVS" -underline 6 \
     -command cvs_version
  .menubar.help.m add command -label "About Wish" -underline 6 \
     -command wish_version
  .menubar.help.m add separator
  .menubar.help.m add command -label "Working with TkCVS" \
     -command working_with_tkcvs
  .menubar.help.m add command -label "Checking out Modules" \
     -command checking_out_modules
  .menubar.help.m add command -label "Exporting" \
     -command exporting
  .menubar.help.m add command -label "Tagging and Branching" \
     -command tagging_and_branching
  .menubar.help.m add command -label "Importing New Modules" \
     -command importing_new_modules
  .menubar.help.m add separator
  .menubar.help.m add command -label "Current Directory Display" \
     -command current_directory
  .menubar.help.m add command -label "Buttons" \
     -command buttons_help
  .menubar.help.m add separator
  .menubar.help.m add command -label "Module Browser" \
     -command module_browser
  .menubar.help.m add command -label "File Browser" \
     -command file_browser
  .menubar.help.m add command -label "Log Browser" \
     -command log_browser
  .menubar.help.m add separator
  .menubar.help.m add command -label "Configuration Files" \
     -command configuration_files
  .menubar.help.m add command -label "Environment Variables" \
     -command environment_variables
  .menubar.help.m add command -label "Command Line Options" \
     -command cli_options
  .menubar.help.m add command -label "User Defined Menu" \
     -command user_defined_menu
  .menubar.help.m add command -label "CVS modules File" \
     -command cvs_modules_file

  #
  # Add user commands to the menu.
  #
  if { [info exists cvsmenu] || [info exists usermenu] } {
    menu .menubar.user.m -font $cvscfg(guifont)
  }
  if {[info exists cvsmenu]} {
    foreach item [array names cvsmenu] {
      .menubar.user.m add command -label $item \
         -command "eval cvs_usercmd $cvsmenu($item) \[workdir_list_files\]"
    }
  }
  if {[info exists usermenu]} {
    .menubar.user.m add separator
    foreach item [array names usermenu] {
      .menubar.user.m add command -label $item \
         -command "eval cvs_anycmd $usermenu($item) \[workdir_list_files\]"
    }
  }
  gen_log:log T "LEAVE"
}

proc workdir_list_files {} {
  global cvscfg

  gen_log:log T "ENTER"
  set getlist {}
  foreach item [.main.file_list.list curselection] {
    set itemstring [.main.file_list.list get $item]
    gen_log:log D "selection is $itemstring"
    if {[string match "no file *" $itemstring]} {
      set lst [split $itemstring]
      set itemstring [lrange $lst 2 end]
    }
    lappend getlist $itemstring
    gen_log:log D "$getlist"
  }
  if {[llength $getlist] > 0} {
    gen_log:log D "LEAVE -- return getlist ($getlist)"
    return $getlist
  } else {
    gen_log:log D "LEAVE -- cvscfg(thisdir) ($cvscfg(thisdir))"
    return "$cvscfg(thisdir)"
  }
}

proc workdir_edit_command {file} {
  global cvscfg

  gen_log:log T "ENTER ($file)"
  if {[info exists cvscfg(editors)]} {
    foreach {editor pattern} $cvscfg(editors) {
      if {[string match $pattern $file]} {
        return "$editor \"$file\""
      }
    }
  }
  return "$cvscfg(editor) $cvscfg(editorargs) \"$file\""
}

proc workdir_edit_file {args} {
  global cvscfg
  global cwd
  global feedback

  gen_log:log T "ENTER ($args)"

  set filelist [join $args]
  if {$filelist == $cvscfg(thisdir)} {
    cvsfail "Please select some files to edit first!"
    return
  }

  feedback_cvs $feedback(cvs) "Building scroll list, please wait!"
  gen_log:log D "$filelist"
  foreach file $filelist {
    if {[file isdirectory $file]} {
      change_dir "$file"
    } else {
      feedback_cvs $feedback(cvs) "Starting editor, please wait!"
      if {[file isfile $file]} {
        gen_log:log D " [workdir_edit_command $file]"
        eval exec [workdir_edit_command $file] >& $cvscfg(null) &
      } else {
        gen_log:log D "$file is not a plain file"
      }
    }
  }
  feedback_cvs $feedback(cvs) ""
  gen_log:log T "LEAVE"
}

proc workdir_status_list_files {} {
  foreach item [.main.status_list.list curselection] {
    if [info exists getlist] {
      lappend getlist [.main.file_list.list get $item]
    } else {
      set getlist [.main.file_list.list get $item]
    }
  }

  if [info exists getlist] {
    set cur_select [.main.status_list.list curselection]
    set start_pos [ lindex $cur_select 0 ]
    set end_pos   [ expr [ llength $cur_select ] + $start_pos - 1 ]
    .main.main.file_list.list select set $start_pos $end_pos
    return $getlist
  } else {
    set cur_select [.main.status_list.list curselection]
    return {}
  }
}

proc workdir_status_list_file {yposition} {
  set cur_select [.main.status_list.list nearest $yposition]
  return $cur_select
}

#------------------------------------------------------
# Update the "Go" menu for directories we can go to
# new_dir - the directory we're going to
# doPwd   - tells whether the directory path has
#           been specified  1 means relative to cwd
#                           0 means fully path specified
#-------------------------------------------------------
proc update_go {new_dir} {
  global cvscfg
  global dirlist
  global maxdirs
  global dirlen

  gen_log:log T "ENTER ($new_dir)"
  if {$new_dir == $cvscfg(thisdir)} { return }
  if {$new_dir == "~" } { return }
  if {$new_dir == ".."} {
    set new_dir [file dirname [pwd]]
  }
  if {[file pathtype $new_dir] == "relative"} {
    # Get full pathname of directory
    set new_dir [format {%s/%s} [pwd] $new_dir]
  }

  #puts $new_dir
  # Check if already in Go list
  set dirlocation  [lsearch -exact $dirlist $new_dir]

  # Move a directory already in the list to the top of the list
  if {$dirlocation != -1} {
    set dirlist [lreplace $dirlist $dirlocation $dirlocation ]
    set dirlist [linsert $dirlist 0 $new_dir]
  } else {
    set dirlist [linsert $dirlist 0 $new_dir]
  }
  set dirlen  [llength $dirlist]

  # Truncate end of directory list if we have too many directories
  if {$dirlen > $maxdirs} {
    set dirlen [incr dirlen -1]
    set dirlist [lreplace $dirlist $dirlen $dirlen ]
  }

  # Destroy old menu selections for "Go"
  destroy .menubar.goto.m
  menu .menubar.goto.m -font $cvscfg(guifont)
  .menubar.goto.m add command -label "Home" \
     -command {change_dir ~}

  # Rebuild menu selections for "Go" with new dirlist
  for {set i 0} {$i < $dirlen} {incr i 1} {
    set tmpdir [lindex $dirlist $i]
    .menubar.goto.m add command -label $tmpdir \
       -command [format {change_dir "%s"} $tmpdir]
  }
  gen_log:log T "LEAVE"
}

proc change_dir {new_dir} {
  global cwd

  gen_log:log T "ENTER ($new_dir)"
  update_go $new_dir
  set cwd $new_dir
  setup_dir

  gen_log:log T "LEAVE"
}


# I modified this a lot to support the status listbox and marked canvas.
# I cringe at the size of the procedure -- it needs to be broken into smaller
# ones badly.
# -sj

proc setup_dir { } {
  #
  # Call this when entering a directory.  It puts all of the file names
  # in the listbox, and reads the CVS or directory.
  #
  global cwd
  global module_dir
  global incvs
  global cvscfg
  global current_tagname

  gen_log:log T "ENTER"

  if { ! [winfo exists .] } {
    workdir_setup
  } else {
    .main.file_list.list delete 0 end
    .main.status_list.list delete 0 end
    .main.tag_list.list delete 0 end
    .main.edit_list.list delete 0 end
  }
  set module_dir "Not in the repository"
  set current_tagname "No directory tag"

  if {![file isdirectory $cwd]} {
    gen_log:log D "$cwd is not a directory"
    gen_log:log T "LEAVE -- $cwd is not a directory"
    return
  }

  cd $cwd
  gen_log:log F "CD [pwd]"
  set cwd [pwd]

  cvsroot_check
  gen_log:log D " incvs = $incvs"

  set cvscfg(ignore_file_filter) $cvscfg(default_ignore_filter)

  if { [ file exists ".cvsignore" ] } {
    set fileId [ open ".cvsignore" "r" ]
    while { [ eof $fileId ] == 0 } {
      gets $fileId line
      append cvscfg(ignore_file_filter) " $line"
    }
    close $fileId
  }

  set filelist [ getFiles ]

  # Select from those files only the ones we want (e.g., no CVS dirs)
  set j 0
  foreach i $filelist {
    if { [ isCmDirectory $i ] } {
      if {$i == "CVS"} {
        read_cvs_dir $cwd/$i
      } else {
        nop
      }
    } else {
      .main.file_list.list insert end $i
      # count actual number of visible elements (not showing CM directories)
      set j [expr {$j + 1}]
    }
  }

  if {! $incvs} {
    set module_dir "Not a CVS directory."
    set current_tagname "Not a CVS directory."
    # unpack the status listbox and scrollbar from the screen
    pack forget .main.status_list.list .main.tag_list.list .main.edit_list.list
    pack forget .main.status_list.head .main.tag_list.head .main.edit_list.head
    pack forget .main.status_list .main.tag_list .main.edit_list
  } else {
    pack .main.status_list -side left -fill both -expand yes
    pack .main.tag_list -side left -fill both -expand yes
    if {$cvscfg(econtrol)} {
      pack .main.edit_list -side left -fill both -expand yes
    }
    pack .main.status_list.head -side top -fill x -expand no
    pack .main.tag_list.head -side top -fill x -expand no
    if {$cvscfg(econtrol)} {
      pack .main.edit_list.head -side top -fill x -expand no
    }
    pack .main.status_list.list -side top -fill both -expand yes
    pack .main.tag_list.list -side top -fill both -expand yes
    if {$cvscfg(econtrol)} {
      pack .main.edit_list.list -side top -fill both -expand yes
    }
    setup_columns
  }

  gen_log:log T "LEAVE"
}


# Assumes .file_list has a list, and .status_list and .tag_list are empty
# and it will modify all three listboxes
proc setup_columns { } {
  global incvs
  global cvs
  global cvscfg
  global cvsglb

  gen_log:log T "ENTER"
  if {! $incvs} {
    cvs_notincvs
    return 1
  }

  busy_start .main

  # gets cvs status in current directory only, pulling out lines that include
  # Status: or Sticky Tag:, putting each file's info (name, status, and tag)
  # into a list, which is an element in the overall list.
  set commandline "$cvs -n -q -l status -l $cvscfg(thisdir)"
  gen_log:log C "$commandline"
  catch {eval "exec $commandline"} raw_log

  set log_lines [split $raw_log "\n"]
  foreach logline $log_lines {
    if {[string match "File:*" $logline]} {
      # Note: versions of tcl < 8.1 dont have advanced regular expressions
      regsub -all {[ \t]+} $logline " " logline
      set line [split [string trim $logline]]
      gen_log:log D "$line"
      set elems [llength $line]
      set statindex [lsearch -exact $line "Status:"]
      set filename [lrange $line 1 [expr {$statindex - 1}]]
      set status($filename) [lrange $line [expr {$statindex + 1}] end]
      if {[string match "no file*" $filename]} {
         set savstat $status($filename)
         unset status($filename)
         regsub {^no file } $filename "" filename
         set status($filename) $savstat
         set nofile($filename) 1
      }
    } elseif {[string match "*Working revision:*" $logline]} {
      regsub -all {[ \t]+} $logline " " logline
      set line [split [string trim $logline]]
      #gen_log:log D "$line"
      set elems [llength $line]
      set workingrev($filename) [lindex $line 2]
    } elseif {[string match "*Sticky Tag:*" $logline]} {
      regsub -all {[ \t]+} $logline " " logline
      set line [split [string trim $logline]]
      #gen_log:log D "$line"
      set elems [llength $line]
      set stickytag($filename) [lrange $line 2 end]
      if { $stickytag($filename) == "(none)" } {
        set stickytag($filename) "(main)"
      }
    }
  }

  if {$cvscfg(econtrol)} {
    gen_log:log C "$cvs -n -q -l editors -l $cvscfg(thisdir) "
    catch {eval "exec $cvs -n -q -l editors -l $cvscfg(thisdir)"} raw_log2

    set log_lines [split $raw_log2 "\n"]
    foreach logline $log_lines {
      regsub -all {[ \t]+} $logline " " logline
      set line [split $logline]
      gen_log:log D "$line"
      set elems [llength $line]
      set editoridx [expr {$elems - 9}]
      set nameidx [expr {$editoridx - 1}]
      # if there is no filename, then use the one from last time
      if [regexp {^ } $line] {
        set editor($filename) [lrange $line 0 0]/$editor($filename)
      } else {
        set filename [lrange $line 0 $nameidx]
        set editor($filename) [lrange $line $editoridx $editoridx]
      }
    }
  }

  set givenfiles [.main.file_list.list get 0 end]

  # There may be files that CVS doesnt know about
  foreach givenfilename $givenfiles {
    # Match log entries to "given" files
    if {! [info exists status($givenfilename)] } {
      if {[file isdirectory $givenfilename]} {
        if {[file isdirectory [file join $givenfilename "CVS"]]} {
          .main.status_list.list insert end "<directory:CVS>"
        } elseif {[file isdirectory [file join $givenfilename CVS.adm]]} {
          .main.status_list.list insert end "<directory:oldCVS>"
        } elseif {[file isdirectory [file join $givenfilename RCS]]} {
          .main.status_list.list insert end "<directory:RCS>"
        } elseif {[file isdirectory [file join $givenfilename SCCS]]} {
          .main.status_list.list insert end "<directory:SCCS>"
        } else {
          .main.status_list.list insert end "<directory>"
        }
	.main.tag_list.list insert end " "
        .main.edit_list.list insert end " "
      } else {
        .main.status_list.list insert end "  ?"
        .main.tag_list.list insert end " "
        .main.edit_list.list insert end " "
      }
    } else {
      .main.status_list.list insert end "$status($givenfilename)"
      .main.tag_list.list insert end \
        "$workingrev($givenfilename)   on   $stickytag($givenfilename)"
      if [info exists editor($givenfilename)] {
       .main.edit_list.list insert end "$editor($givenfilename)"
      } else {
       .main.edit_list.list insert end " "
      }
    }
  }

  # There may be files that CVS knows about, but they've been removed
  foreach cvsfile [array names nofile] {
    set idx 0
    foreach givenfile [.main.file_list.list get 0 end] {
      set order [concat $givenfile $cvsfile]
      set sorted [lsort -ascii $order]
      if {[lindex $order 0] != [lindex $sorted 0]} {
        set newfile($idx) $cvsfile
        break
      }
      incr idx
    }
  }
  set boost 0
  foreach i [lsort -integer [array names newfile]] {
    set newpos [expr {$i + $boost}]
    # When you insert one, the indices jump ahead - hence the boost
    .main.file_list.list insert $newpos "no file $newfile($i)"
    .main.status_list.list insert $newpos "$status($newfile($i))"
    .main.tag_list.list insert $newpos \
      "$workingrev($newfile($i))  on  $stickytag($newfile($i))"
    .main.edit_list.list insert end " "
    incr boost
  }

  sort_by_col [lindex $cvsglb(sort_pref) 0] [lindex $cvsglb(sort_pref) 1]
  busy_done .main

  gen_log:log T "LEAVE"
}

# Re-sort the workdir lists
proc sort_by_col {col sense} {
  global cvscfg
  global cvsglb

  gen_log:log T "ENTER ($col $sense)"
  set cvsglb(sort_pref) [list $col $sense]

  set col0 [.main.file_list.list get 0 end]
  set col1 [.main.status_list.list get 0 end]
  set col2 [.main.tag_list.list get 0 end]
  set col3 [.main.edit_list.list get 0 end]

  set arr(0) .main.file_list.head.sbut
  set arr(1) .main.status_list.head.sbut
  set arr(2) .main.tag_list.head.sbut
  set arr(3) .main.edit_list.head.sbut

  # Clear them
  set x {}
  foreach i $col0 j $col1 k $col2 l $col3 {
    #gen_log:log D "append   [list $i $j $k $l]"
    lappend x [list $i $j $k $l]
  }
  # Reset the button arrows
  foreach i [list $arr(0) $arr(1) $arr(2) $arr(3)] {
    $i configure -image arr_dn
  }

  if {$sense == "dec"} {
    gen_log:log D "sort column $col decreasing"
    $arr($col) configure -image arh_up
    set y [lsort -decreasing -index $col $x]
  } else {
    gen_log:log D "sort column $col increasing"
    $arr($col) configure -image arh_dn
    set y [lsort -index $col $x]
  }

  .main.file_list.list delete 0 end
  .main.status_list.list delete 0 end
  .main.tag_list.list delete 0 end
  .main.edit_list.list delete 0 end

  foreach i $y {
    .main.file_list.list insert end [lindex $i 0]
    .main.status_list.list insert end [lindex $i 1]
    .main.tag_list.list insert end [lindex $i 2]
    .main.edit_list.list insert end [lindex $i 3]
  }
   gen_log:log T "LEAVE"
}

proc toggle_col {col} {
  global cvscfg
  global cvsglb

  gen_log:log T "ENTER ($col)"
  set cur_col [lindex $cvsglb(sort_pref) 0]
  set cur_sense [lindex $cvsglb(sort_pref) 1]

  # if it's the currently sorted column, reverse the direction.
  if {$col == $cur_col} {
    if {$cur_sense == "inc"} {
      set sense "dec"
    } else {
      set sense "inc"
    }
  # Otherwise, sort it by the global preference
  } else {
    set sense "inc"
  }

  sort_by_col $col $sense
  gen_log:log D "sort column $col $sense"

  gen_log:log T "LEAVE"
}

proc read_cvs_dir {dirname} {
#
# Reads a CVS directory
#
  global module_dir
  global cvscfg
  global current_tagname

  gen_log:log T "ENTER ($dirname)"
  if {[file isdirectory $dirname]} {
    if {[file isfile [file join $dirname Repository]]} {
      set f [open [file join $dirname Repository] r]
      gets $f module_dir
      close $f
      gen_log:log D "  MODULE $module_dir"
      if {[file isfile [file join $dirname Root]]} {
        set f [open [file join $dirname Root] r]
        gets $f cvscfg(cvsroot)
        close $f
        gen_log:log D "  CVSROOT $cvscfg(cvsroot)"
      }
      if {[file isfile [file join $dirname Tag]]} {
        set f [open [file join $dirname Tag] r]
        gets $f current_tagname
        close $f
        # T = branch tag, N = non-branch, D = sticky date
        #set current_tagname [string trimleft $current_tagname "TND"]
        set current_tagname [string range $current_tagname 1 end]
        gen_log:log D "  BRANCH TAG $current_tagname"
      }
    } else {
      cvsfail "Repository file not found in $dirname"
    }
  } else {
    cvsfail "$dirname is not a directory"
  }
  gen_log:log T "LEAVE"
}

proc workdir_scroll_file_list {first last} {
# support middle-button drag scrolling
  global cvscfg

  .main.scroll set $first $last
  .main.status_list.list yview moveto $first
  .main.tag_list.list yview moveto $first
  if {$cvscfg(econtrol)} {
    .main.edit_list.list yview moveto $first
  }
}

proc workdir_scroll_status_list {first last} {
# support middle-button drag scrolling
  global cvscfg

  .main.scroll set $first $last
  .main.file_list.list yview moveto $first
  .main.tag_list.list yview moveto $first
  if {$cvscfg(econtrol)} {
    .main.edit_list.list yview moveto $first
  }
}

proc workdir_scroll_tag_list {first last} {
# support middle-button drag scrolling
  global cvscfg

  .main.scroll set $first $last
  .main.file_list.list yview moveto $first
  .main.status_list.list yview moveto $first
  if {$cvscfg(econtrol)} {
    .main.edit_list.list yview moveto $first
  }
}

proc workdir_scroll_edit_list {first last} {
# support middle-button drag scrolling

  .main.scroll set $first $last
  .main.file_list.list yview moveto $first
  .main.status_list.list yview moveto $first
  .main.edit_list.list yview moveto $first
}

proc workdir_scroll_scrollbar {args} {
# To support scrolling all listboxes simultaneously
  global cvscfg

  eval ".main.file_list.list     yview $args"
  eval ".main.status_list.list   yview $args"
  eval ".main.tag_list.list      yview $args"
  if {$cvscfg(econtrol)} {
    eval ".main.edit_list.list     yview $args"
  }
}

proc workdir_cleanup {} {
  global cvscfg

  gen_log:log T "ENTER"
  set list [ split $cvscfg(clean_these) " " ]
  if { [ are_you_sure "You are about to delete:\n" $list] == 1 } {
    foreach item $list {
      gen_log:log D "item $item"
      if { $item != "" } {
        set rmitem [glob -nocomplain $item]
        if {$rmitem != ""} {
          gen_log:log F "DELETE $rmitem"
          eval file delete -force -- $rmitem
        }
      } else {
        nop
      }
    }
    setup_dir
  }
  gen_log:log T "LEAVE"
}

proc workdir_delete_file {args} {
  global cvscfg

  gen_log:log T "ENTER ($args)"

  set filelist [join $args]
  if {$filelist == $cvscfg(thisdir)} {
    cvsfail "Please select some files to delete first!"
    return
  }

  if { [ are_you_sure "This will delete these files from your local, working directory:\n" $filelist ] == 1 } {
    gen_log:log F "DELETE $filelist"
    eval file delete -force -- $filelist
    setup_dir
  }
  gen_log:log T "LEAVE"
}

proc econtrol_toggle { } {
#
# Turn display of "Editors" column on and off
#
  global cvscfg

  if {$cvscfg(econtrol)} {
    if {! [winfo viewable .beditflag_files]} {
      pack .beditflag_files .buneditflag_files \
        -in .bottom.buttons.modfuncs \
               -side left -ipadx 1 -ipady 1 -fill x -expand 1
      pack .main.edit_list.head -side top -fill x -expand no
      pack .main.edit_list.head.lbl -side left -fill x -expand yes
      pack .main.edit_list.head.sbut -side right
    }
  } else {
    if {[winfo viewable .beditflag_files]} {
      pack forget .main.edit_list
      pack forget .beditflag_files .buneditflag_files
    }
  }

  setup_dir
}

proc are_you_sure {mess args} {
#
# General posting message
#
  global cvscfg

  gen_log:log T "ENTER ($mess $args)"

  set filelist [join $args]
  if {$cvscfg(confirm_prompt)} {
    append mess "\n"
    set indent "      "

    foreach item $filelist {
      if { $item != {} } {
        append mess " $indent"
        append mess " $item\n"
      }
    }
    append mess "\nAre you sure?"
    if {[cvsconfirm $mess] == 1} {
      gen_log:log T "LEAVE 0"
      return 0
    }
  }
  gen_log:log T "LEAVE 1"
  return 1
}

proc busy_start {w} {

  foreach widget [winfo children $w] {
    catch {$widget config -cursor watch}
  }
  update idletasks
}

proc busy_done {w} {

  foreach widget [winfo children $w] {
    catch {$widget config -cursor ""}
  }
}

proc workdir_print_file {args} {
  global cvscfg

  gen_log:log T "ENTER ($args)"

  set filelist [join $args]
  if {$filelist == $cvscfg(thisdir)} {
    cvsfail "Please select some files to print first!"
    return
  }

  set mess "This will print these files:\n\n"
  foreach file $filelist {
    append mess "   $file\n"
  }
  append mess "\nUsing $cvscfg(print_cmd)\n"
  append mess "\nAre you sure?"
  if {[cvsconfirm $mess] == 0} {
    set final_result ""
    foreach file $filelist {
      gen_log:log F "$cvscfg(print_cmd) \"$file\""
      catch { eval exec $cvscfg(print_cmd) \"$file\" } file_result
      if { $file_result != "" } {
        set final_result "$final_result\n$file_result"
      }
    }
    if { $final_result != "" } {
      view_output "Print" $final_result
    }
  }
  gen_log:log T "LEAVE"
}

proc cvsroot_check {} {
  global cvsglb
  global cvscfg
  global incvs
  global env

  gen_log:log T "ENTER"
  set incvs 0
  if {[file isfile [file join . CVS Root]]} {
    set f [open [file join . CVS Root] r]
    gets $f root
    close $f
    set incvs 1
    set cvscfg(cvsroot) $root
  }
  gen_log:log T " cvsroot: $cvscfg(cvsroot)"
  if {[string match "*:*" $cvscfg(cvsroot)]} {
    set cvsglb(remote) 1
  } else {
    set cvsglb(remote) 0
  }
  gen_log:log T " remote=$cvsglb(remote)"
  gen_log:log T "LEAVE"
}

proc nop {} {}

proc disabled {} {
  cvsok "Command disabled."
}

proc isCmDirectory { file } {
  switch -- $file  {
    "CVS"  -
    "RCS"  - 
    "SCCS" { set value 1 }
    default { set value 0 } 
  }
  return $value
}

# Get the files in the current working directory.  Use the file_filter
# values Add hidden files if desired by the user.  Sort them to match
# the ordering that will be returned by cvs commands (this matches the
# default ls ordering.).
proc getFiles { } {
  global cvscfg

  gen_log:log T "ENTER"
  set filelist ""
    
  # make sure the file filter is at least set to "*".
  if { $cvscfg(file_filter) == "" } {
    set cvscfg(file_filter) "*"
  }

  # get the initial file list, including hidden if requested
  if {$cvscfg(allfiles)} {
    # get hidden as well
    foreach item $cvscfg(file_filter) {
      set filelist [ concat [ glob -nocomplain .$item $item ] $filelist ]
    }
  } else {
    foreach item $cvscfg(file_filter) {
      set filelist [ concat [ glob -nocomplain $item ] $filelist ]
    }
  }
  #gen_log:log D "filelist ($filelist)"

  # ignore files if requested
  if { $cvscfg(ignore_file_filter) != "" } {
    foreach item $cvscfg(ignore_file_filter) {
      # for each pattern
      if { $item != "*" } {
        # if not "*"
        while { [set idx [lsearch $filelist $item]] != -1 } {
          # for each occurence, delete
          catch { set filelist [ lreplace $filelist $idx $idx ] }
        }
      }
    }
  }

  # make sure "." is always in the list for 'cd' purposes
  if { ( [ lsearch -exact $filelist "." ] == -1 ) } {
    set filelist [ concat "." $filelist ]
  }
    
  # make sure ".." is always in the list for 'cd' purposes
  if { ( [ lsearch -exact $filelist ".." ] == -1 ) } {
    set filelist [ concat ".." $filelist ]
  }
    
  # sort it
  set filelist [ lsort $filelist ]
    
  # if this directory is under CVS and CVS is not in the list, add it. Its
  # presence is needed for later processing
  if { ( [ file exists "CVS" ] ) && 
       ( [ lsearch -exact $filelist "CVS" ] == -1 ) } {
    #puts "********* added CVS"
    catch { set filelist [ concat "CVS" $filelist ] }
  }

  set cvscfg(ignore_filter) $cvscfg(default_ignore_filter)
  gen_log:log T "return ($filelist)"
  return $filelist
}

proc feedback_cvs { e message } {
#
# This code is adapted from the text "Practical Programming in
# Tcl and Tk", by Brent B. Welch (see page 209)
# An entry widget is used because it won't change size
# base on the message length, and it can be scrolled by
# dragging with button 2.
# Author: Eugene Lee, Aerospace Corporation, 9/6/95
#
  global feedback
  global cvscfg

  $e config -state normal
  $e delete 0 end
  $e insert 0 $message
  # Leave the entry in a read-only state
  $e config -state disabled

  # Force a disable update
  update idletasks
}

proc log_toggle { } {
  global cvscfg

  if {$cvscfg(logging)} {
    gen_log:init
  } else {
    gen_log:quit
  }
}

proc exit_cleanup { } {
  global cvscfg
  
  set pid [pid]
  gen_log:log F "DELETE $cvscfg(tmpdir)/cvstmpdir.$pid"
  catch [file delete -force [file join $cvscfg(tmpdir) cvstmpdir.$pid]]
  destroy .
  exit
}

proc save_options { } {
#
# Save the options which are configurable from the GUI
#
  global cvscfg

  gen_log:log T "ENTER"

  # There are two kinds of options we can set
  set BOOLopts { allfiles auto_status confirm_prompt econtrol \
                 checkrecursive recurse logging }
  set STRGopts { file_filter ignore_file_filter clean_these \
                 rdetail ldetail log_classes }
  # Save the list so we can keep track of what we've done
  set BOOLset $BOOLopts
  set STRGset $STRGopts

  set optfile [file join ~ .tkcvs]
  set bakfile [file join ~ .tkcvs.bak]
  # Save the old .tkcvs file
  gen_log:log F "MOVE $optfile $bakfile"
  file rename -force $optfile $bakfile

  gen_log:log F "OPEN $bakfile"
  set fi [open $bakfile r]
  gen_log:log F "OPEN $optfile"
  set fo [open $optfile w]
  
  while { [eof $fi] == 0 } {
    gets $fi line
    set match 0
    if {[regexp {^#} $line]} {
      # Don't try to scan comments.
      #gen_log:log D "PASSING \"$line\""
      puts $fo "$line"
      continue
    } else {
      foreach opt $BOOLopts {
        if {[string match "*set *cvscfg($opt)*" $line]} {
          # Print it and remove it from the list
          gen_log:log D "REPLACING $line  with  set cvscfg($opt) $cvscfg($opt)"
          puts $fo "set cvscfg($opt) $cvscfg($opt)"
          set idx [lsearch $BOOLset $opt]
          set BOOLset [lreplace $BOOLset $idx $idx]
          set match 1
          break
        }
      }
      foreach opt $STRGopts {
        if {[string match "*set *cvscfg($opt)*" $line]} {
          # Print it and remove it from the list
          gen_log:log D "REPLACING $line  with  set cvscfg($opt) \"$cvscfg($opt)\""
          puts $fo "set cvscfg($opt) \"$cvscfg($opt)\""
          set idx [lsearch $STRGset $opt]
          set STRGset [lreplace $STRGset $idx $idx]
          set match 1
          break
        }
      }
      if {$match == 0} {
        # We didn't do a replacement
        gen_log:log D "PASSING \"$line\""
        puts $fo "$line"
      }
    }
  }
  close $fi

  # Print what's left over
  foreach opt $BOOLset {
    gen_log:log D "ADDING cvscfg($opt) $cvscfg($opt)"
    puts $fo "set cvscfg($opt) $cvscfg($opt)"
  }

  foreach opt $STRGset {
    gen_log:log D "ADDING cvscfg($opt) \"$cvscfg($opt)\""
    puts $fo "set cvscfg($opt) \"$cvscfg($opt)\""
  }

  close $fo
  gen_log:log T "LEAVE"
}
