/*
 * WBXML Lib, the WBXML Library.
 * Copyright (C) 2002-2003  Aymerick Jhanne
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License (version 2.1) as published by the Free Software Foundation.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * LGPL v2.1: http://www.gnu.org/licenses/lgpl.txt
 *
 * Author Contact: libwbxml@jehanne.org
 * WBXML Lib home: http://libwbxml.jehanne.org
 */
 
/**
 * @file wbxml2xml_clb.c
 * @ingroup wbxml2xml
 *
 * @author Aymerick Jhanne <libwbxml@jehanne.org>
 * @date 03/02/22
 *
 * @brief WBXML to XML Converter Callbacks (Callbacks for WBXML Parser)
 */

#include <string.h> /* For strlen() */

#include "wbxml_conv.h"
#include "wbxml2xml_clb.h"


/** New Line */
#define WBXML2XML_NEW_LINE "\n"

/** <?xml version=\"1.0\"?> */
#define WBXML2XML_XML_HEADER "<?xml version=\"1.0\"?>"

/** <!DOCTYPE */
#define WBXML2XML_XML_DOCTYPE "<!DOCTYPE "

/**  PUBLIC " */
#define WBXML2XML_XML_PUBLIC " PUBLIC \""

/** " " */
#define WBXML2XML_XML_DTD "\" \""

#define WBXML2XML_XML_DTD_INDENT "\"\n\t\t\t\""

/** "> */
#define WBXML2XML_XML_END_DTD "\">"

/** Memory Error */
#define WBXML2XML_ERROR_MEM { \
        wbxml2xml_ctx->error = WBXML_ERROR_NOT_ENOUGH_MEMORY; \
        return; \
    }


/* Private Functions Prototypes */
static WB_BOOL add_new_line(WBXMLBuffer *buff);


/************************************
 * Public Functions
 */

void wbxml2xml_clb_start_document(void *ctx, WB_LONG charset, const WBXMLLangEntry *lang)
{
    wbxml2xmlCtx *wbxml2xml_ctx = (wbxml2xmlCtx *) ctx;

    /* Write to Buffer */
    if (wbxml2xml_ctx->xml != NULL) 
    {
        /* <?xml version=\"1.0\"?> */
        if (!wbxml_buffer_append_cstr(wbxml2xml_ctx->xml, WBXML2XML_XML_HEADER))
            WBXML2XML_ERROR_MEM

        /* New Line */
        if (wbxml2xml_ctx->set_indent) {
            if (!add_new_line(wbxml2xml_ctx->xml))
                WBXML2XML_ERROR_MEM
        }

        /* <!DOCTYPE */
        if (!wbxml_buffer_append_cstr(wbxml2xml_ctx->xml, WBXML2XML_XML_DOCTYPE))
            WBXML2XML_ERROR_MEM

        /* Root Element */
        if (!wbxml_buffer_append_cstr(wbxml2xml_ctx->xml, lang->publicID->xmlRootElt))
            WBXML2XML_ERROR_MEM

        /*  PUBLIC " */
        if (!wbxml_buffer_append_cstr(wbxml2xml_ctx->xml, WBXML2XML_XML_PUBLIC))
            WBXML2XML_ERROR_MEM

        /* Public ID */
        if (!wbxml_buffer_append_cstr(wbxml2xml_ctx->xml, lang->publicID->xmlPublicID))
            WBXML2XML_ERROR_MEM

        /* DTD */
        if (wbxml2xml_ctx->set_indent) {
            if (!wbxml_buffer_append_cstr(wbxml2xml_ctx->xml, WBXML2XML_XML_DTD_INDENT))
                WBXML2XML_ERROR_MEM
        }
        else {
            if (!wbxml_buffer_append_cstr(wbxml2xml_ctx->xml, WBXML2XML_XML_DTD))
                WBXML2XML_ERROR_MEM
        }

        /* DTD */
        if (!wbxml_buffer_append_cstr(wbxml2xml_ctx->xml, lang->publicID->xmlDTD))
            WBXML2XML_ERROR_MEM

        /* "> */
        if (!wbxml_buffer_append_cstr(wbxml2xml_ctx->xml, WBXML2XML_XML_END_DTD))
            WBXML2XML_ERROR_MEM

        /* New Line */
        if (wbxml2xml_ctx->set_indent) {
            if (!add_new_line(wbxml2xml_ctx->xml))
                WBXML2XML_ERROR_MEM
        }
    }
}


void wbxml2xml_clb_end_document(void *ctx)
{
    wbxml2xmlCtx *wbxml2xml_ctx = (wbxml2xmlCtx *) ctx;

    /* Check if we had a fatal error */
    if (wbxml2xml_ctx->error != WBXML_OK)
        return;

    /* Write New Line to Buffer */
    if ((wbxml2xml_ctx->xml != NULL) && (wbxml2xml_ctx->set_indent)) {
        if (!add_new_line(wbxml2xml_ctx->xml))
            WBXML2XML_ERROR_MEM
    }
}


void wbxml2xml_clb_start_element(void *ctx, WBXMLTag *element, WBXMLAttribute **atts, WB_BOOL empty)
{    
    wbxml2xmlCtx *wbxml2xml_ctx = (wbxml2xmlCtx *) ctx;
    WB_ULONG i = 0, j = 0;

    /* Check if we had a fatal error */
    if (wbxml2xml_ctx->error != WBXML_OK)
        return;

    /* Write to Buffer */
    if (wbxml2xml_ctx->xml != NULL) 
    {
        /* Indent Start Element */
        if (wbxml2xml_ctx->set_indent) {            
            for (j=0; j<(wbxml2xml_ctx->indent); j++) {
                if (!wbxml_buffer_append_char(wbxml2xml_ctx->xml, ' '))
                    WBXML2XML_ERROR_MEM
            }
        }

        /* Write Start Element */
        if (!wbxml_buffer_append_char(wbxml2xml_ctx->xml, '<'))
            WBXML2XML_ERROR_MEM

        if (!wbxml_buffer_append_cstr(wbxml2xml_ctx->xml, wbxml_tag_get_xml_name(element)))
            WBXML2XML_ERROR_MEM

        /* Write Attributes */
        if (atts != NULL) {
            while (atts[i] != NULL)
            {
                /* Write Attribute Name */
                if (!wbxml_buffer_append_char(wbxml2xml_ctx->xml, ' '))
                    WBXML2XML_ERROR_MEM

                if (!wbxml_buffer_append_cstr(wbxml2xml_ctx->xml, wbxml_attribute_get_xml_name(atts[i])))
                    WBXML2XML_ERROR_MEM
                    
                /* Write Attriubute Value */
                if (!wbxml_buffer_append_cstr(wbxml2xml_ctx->xml, "=\""))
                    WBXML2XML_ERROR_MEM

                if (!wbxml_buffer_append_cstr(wbxml2xml_ctx->xml, wbxml_attribute_get_xml_value(atts[i])))
                    WBXML2XML_ERROR_MEM

                if (!wbxml_buffer_append_char(wbxml2xml_ctx->xml, '"'))
                    WBXML2XML_ERROR_MEM

                i++;
            }        
        }

        if (empty) {
            if (!wbxml_buffer_append_cstr(wbxml2xml_ctx->xml, "/>"))
                WBXML2XML_ERROR_MEM

            /* New Line */
            if (wbxml2xml_ctx->set_indent) {
                if (!add_new_line(wbxml2xml_ctx->xml))
                    WBXML2XML_ERROR_MEM
            }
        }
        else {
            if (!wbxml_buffer_append_char(wbxml2xml_ctx->xml, '>'))
                WBXML2XML_ERROR_MEM

            /* New Line */
            if (wbxml2xml_ctx->set_indent) {
                if (!add_new_line(wbxml2xml_ctx->xml))
                    WBXML2XML_ERROR_MEM

                wbxml2xml_ctx->indent++;
            }
        }
    }
}


void wbxml2xml_clb_end_element(void *ctx, WBXMLTag *element, WB_BOOL empty)
{
    wbxml2xmlCtx *wbxml2xml_ctx = (wbxml2xmlCtx *) ctx;
    WB_ULONG j = 0;

    /* Check if we had a fatal error */
    if (wbxml2xml_ctx->error != WBXML_OK)
        return;

    /* Write to Buffer */
    if (wbxml2xml_ctx->xml != NULL) 
    {
        if (!empty) {
            if (wbxml2xml_ctx->set_indent) {
                /* Add a LF if there were content in this element */
                if (wbxml2xml_ctx->content) {
                    if (!add_new_line(wbxml2xml_ctx->xml))
                        WBXML2XML_ERROR_MEM
                }

                wbxml2xml_ctx->indent--;

                /* Indent End Element */
                for (j=0; j<wbxml2xml_ctx->indent; j++) {
                    if (!wbxml_buffer_append_char(wbxml2xml_ctx->xml, ' '))
                        WBXML2XML_ERROR_MEM
                }
            }

            /* End Element */
            if (!wbxml_buffer_append_cstr(wbxml2xml_ctx->xml, "</"))
                WBXML2XML_ERROR_MEM

            if (!wbxml_buffer_append_cstr(wbxml2xml_ctx->xml, wbxml_tag_get_xml_name(element)))
                WBXML2XML_ERROR_MEM

            if (!wbxml_buffer_append_char(wbxml2xml_ctx->xml, '>'))
                WBXML2XML_ERROR_MEM

            /* New Line */
            if (wbxml2xml_ctx->set_indent) {
                if (!add_new_line(wbxml2xml_ctx->xml))
                    WBXML2XML_ERROR_MEM
            }

        }

        wbxml2xml_ctx->content = FALSE;
    }
}


void wbxml2xml_clb_characters(void *ctx, WB_UTINY *ch, WB_ULONG start, WB_ULONG length)
{
    wbxml2xmlCtx *wbxml2xml_ctx = (wbxml2xmlCtx *) ctx;
    WB_ULONG j = 0;

    /* Check if we had a fatal error */
    if (wbxml2xml_ctx->error != WBXML_OK)
        return;

    /* Write to Buffer */
    if (wbxml2xml_ctx->xml != NULL) 
    {
        if ((!wbxml2xml_ctx->content) && (wbxml2xml_ctx->set_indent)) 
        {
            /* Indent Content (only indent in first Content Callback) */
            for (j=0; j<wbxml2xml_ctx->indent; j++) {
                if (!wbxml_buffer_append_char(wbxml2xml_ctx->xml, ' '))
                    WBXML2XML_ERROR_MEM
            }
        }

        /* Write Content */
        for(j=start; j<length; j++) {
            if (!wbxml_buffer_append_char(wbxml2xml_ctx->xml, ch[j]))
                WBXML2XML_ERROR_MEM
        }
	    
        wbxml2xml_ctx->content = TRUE;
    }
}


void wbxml2xml_clb_pi(void *ctx, const WB_UTINY *target, WB_UTINY *data)
{
}


/************************************
 * Private Functions
 */

/**
 * @brief Append a New Line to buffer
 * @param buff The buffer
 * @return TRUE if added, FALSE otherwise
 */
static WB_BOOL add_new_line(WBXMLBuffer *buff)
{
    if (buff == NULL)
        return FALSE;

    return wbxml_buffer_append_data(buff, WBXML2XML_NEW_LINE, WBXML_STRLEN(WBXML2XML_NEW_LINE));
}
