/*
 * WBXML Lib, the WBXML Library.
 * Copyright (C) 2002-2003  Aymerick Jhanne
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License (version 2.1) as published by the Free Software Foundation.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * LGPL v2.1: http://www.gnu.org/licenses/lgpl.txt
 *
 * Author Contact: libwbxml@jehanne.org
 * WBXML Lib home: http://libwbxml.jehanne.org
 */
 
/**
 * @file wbxml2xml.c
 * @ingroup wbxml2xml
 *
 * @author Aymerick Jhanne <libwbxml@jehanne.org>
 * @date 03/02/23
 *
 * @brief WBXML to XML Converter
 *
 * @note There are two ways to launch wbxml2xml():
 *          - Tree Mode: The default one (preferred). In this mode, we only used wbxml_parser_parse_to_tree() and
 *                       wbxml_encoder_encode_to_xml() functions. So we use the internal WBXML Tree representation.
 *          - Direct Mode: Using internal Callbacks to generate XML (should NOT be used). In this mode we use our
 *                         defaults WBXML Parser Callbacks (wbxml2xml_clb.[h|c]) to generate XML.
 */

#include <string.h> /* For memcpy() */

#include "wbxml_conv.h"
#include "wbxml2xml_clb.h"


/** Malloc Block for producing XML Buffer */
#define WBXML2XML_XML_MALLOC_BLOCK 500


/************************************
 * Public Functions
 */

WBXML_DECLARE(WBXMLError) wbxml2xml_real(WB_UTINY *wbxml, WB_ULONG wbxml_len, WB_UTINY **xml, WBXML2XMLParameters *params, WB_BOOL tree)
{
    WBXMLParser *wbxml_parser = NULL;
    WBXMLEncoder *wbxml_encoder = NULL;
    WBXMLTree *wbxml_tree = NULL;
    WB_ULONG error_index = 0;
    wbxml2xmlCtx wbxml2xml_ctx; /**< The Parser Context */
    WBXMLError ret = WBXML_OK;
    WBXMLContentHandler wbxml2xml_handler = 
        {
            wbxml2xml_clb_start_document,
            wbxml2xml_clb_end_document,
            wbxml2xml_clb_start_element,
            wbxml2xml_clb_end_element,
            wbxml2xml_clb_characters,
            wbxml2xml_clb_pi
        };


    /* Check Parameters */
    if ((wbxml == NULL) || (wbxml_len == 0) || (xml == NULL))
        return WBXML_ERROR_BAD_PARAMETER;

    *xml = NULL;

    if (!tree) 
    {
        /****************************************************************
         *  First Method: Direct Mode
         *  We use our own WBXML Parser Callbacks to construct XML
         */

        WBXML_ERROR((WBXML_CONV, "wbxml2xml: Normal Mode"));

        /* Create WBXML Parser */
        wbxml_parser = wbxml_parser_create();
        if (wbxml_parser == NULL)
            return WBXML_ERROR_NOT_ENOUGH_MEMORY;

        /* Init context */
        wbxml2xml_ctx.set_indent = TRUE;
        wbxml2xml_ctx.indent = 0;
        wbxml2xml_ctx.error = WBXML_OK;
        wbxml2xml_ctx.content = FALSE;

        if ((wbxml2xml_ctx.xml = wbxml_buffer_create("", 0, WBXML2XML_XML_MALLOC_BLOCK)) == NULL) {
            wbxml_parser_destroy(wbxml_parser);
            return WBXML_ERROR_NOT_ENOUGH_MEMORY;
        }

        /* Initialize WBXML Parser */
        wbxml_parser_set_user_data(wbxml_parser, &wbxml2xml_ctx);
        wbxml_parser_set_content_handler(wbxml_parser, &wbxml2xml_handler);

        /* Parse WBXML document */
        ret = wbxml_parser_parse(wbxml_parser, wbxml, wbxml_len);
        if ((ret != WBXML_OK) || (wbxml2xml_ctx.error != WBXML_OK))
        {
            if (ret != WBXML_OK) {
                error_index = wbxml_parser_get_current_byte_index(wbxml_parser);
                WBXML_ERROR((WBXML_CONV, "wbxml2xml convertion failed at %ld - Token %x - %s",
                                            error_index,
                                            wbxml[error_index],
                                            wbxml_errors_string(ret)));
            }
            else {
                WBXML_ERROR((WBXML_CONV, "wbxml2xml convertion failed : %s", wbxml_errors_string(wbxml2xml_ctx.error)));
            }
        }
        else {
            /* Copy result Buffer */
            *xml = (WB_UTINY *) wbxml_malloc((wbxml_buffer_len(wbxml2xml_ctx.xml) + 1) * sizeof(WB_UTINY));
            if (*xml == NULL) {
                wbxml_parser_destroy(wbxml_parser);
                wbxml_buffer_destroy(wbxml2xml_ctx.xml);
                return WBXML_ERROR_NOT_ENOUGH_MEMORY;
            }

            /* Copy XML Document */
            memcpy(*xml, wbxml_buffer_get_cstr(wbxml2xml_ctx.xml), wbxml_buffer_len(wbxml2xml_ctx.xml));
            (*xml)[wbxml_buffer_len(wbxml2xml_ctx.xml)] = '\0';
        }

        /* Clean-up */
        wbxml_parser_destroy(wbxml_parser);
        wbxml_buffer_destroy(wbxml2xml_ctx.xml);

        if (ret != WBXML_OK)
            return ret;
        else
            return wbxml2xml_ctx.error;
    }
    else {
        /****************************************************************
         *  Second Method: Tree Mode
         *  We use internals WBXML Parser to construct WBXML Tree, then the WBXML Encoder
         *  to encode the final XML Document.
         */

        WBXML_ERROR((WBXML_CONV, "wbxml2xml: Tree Mode"));

        /* Parse WBXML to WBXML Tree */
        ret = wbxml_parser_parse_to_tree(wbxml, wbxml_len, &wbxml_tree);
        if (ret != WBXML_OK) {
            WBXML_ERROR((WBXML_CONV, "wbxml2xml convertion failed - WBXML Parser Error: %s",
                                     wbxml_errors_string(ret)));

            return ret;
        }
        else {
            /* Create WBXML Encoder */
            if ((wbxml_encoder = wbxml_encoder_create()) == NULL) {
                wbxml_tree_destroy(wbxml_tree);
                return WBXML_ERROR_NOT_ENOUGH_MEMORY;
            }

            /* Set the WBXML Tree to encode */
            wbxml_encoder_set_tree(wbxml_encoder, wbxml_tree);

            if (params == NULL) {
                /* Default Values */

                /* Set XML Generation Type */
                wbxml_encoder_set_xml_gen_type(wbxml_encoder, WBXML_ENCODER_XML_GEN_INDENT);

                /* Set Indent */
                wbxml_encoder_set_indent(wbxml_encoder, 0);

                /* Skip Ignorable Whitespaces */
                wbxml_encoder_set_ignore_empty_text(wbxml_encoder, TRUE);
                wbxml_encoder_set_remove_text_blanks(wbxml_encoder, TRUE);
            }
            else {
                /* Set XML Generation Type */
                wbxml_encoder_set_xml_gen_type(wbxml_encoder, params->gen_type);

                /* Set Indent */
                if (params->gen_type == WBXML_ENCODER_XML_GEN_INDENT)
                    wbxml_encoder_set_indent(wbxml_encoder, params->indent);

                /* Ignorable Whitespaces */
                if (params->keep_ignorable_ws) {
                    wbxml_encoder_set_ignore_empty_text(wbxml_encoder, FALSE);
                    wbxml_encoder_set_remove_text_blanks(wbxml_encoder, FALSE);
                }
                else {
                    wbxml_encoder_set_ignore_empty_text(wbxml_encoder, TRUE);
                    wbxml_encoder_set_remove_text_blanks(wbxml_encoder, TRUE);
                }
            }

            /* Encode WBXML Tree to XML */
            ret = wbxml_encoder_encode_to_xml(wbxml_encoder, xml);
            if (ret != WBXML_OK) {
                WBXML_ERROR((WBXML_CONV, "wbxml2xml convertion failed - WBXML Encoder Error: %s",
                                         wbxml_errors_string(ret)));
            }

            /* Clean-up */
            wbxml_encoder_destroy(wbxml_encoder);
            wbxml_tree_destroy(wbxml_tree);
            return ret;
        }
    }
}
