/*
 * WBXML Lib, the WBXML Library.
 * Copyright (C) 2002-2003  Aymerick Jhanne
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License (version 2.1) as published by the Free Software Foundation.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * LGPL v2.1: http://www.gnu.org/licenses/lgpl.txt
 *
 * Author Contact: libwbxml@jehanne.org
 * WBXML Lib home: http://libwbxml.jehanne.org
 */
 
/**
 * @file wbxml_conv.h
 * @ingroup wbxml_conv
 *
 * @author Aymerick Jhanne <libwbxml@jehanne.org>
 * @date 03/02/23
 *
 * @brief WBXML Convertion Library
 */

#ifndef WBXML_CONV_H
#define WBXML_CONV_H

#include "wbxml.h"  /* libwbxml2 */


#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/** Is Tree Mode the default WBXML2XML Convertion Mode ? */
#define WBXML_CONV_WBXML2XML_DEFAULT_TREE_MODE TRUE

/** @addtogroup wbxml_conv 
 *  @{ 
 */

/** Parameters for xml2wbxml() function  */
typedef struct WBXML2XMLParameters_s {
    WBXMLEncoderXMLGenType gen_type;    /**< WBXML_ENCODER_XML_GEN_COMPACT | WBXML_ENCODER_XML_GEN_INDENT | WBXML_ENCODER_XML_GEN_CANONICAL (Default: WBXML_ENCODER_XML_GEN_INDENT) */
    WB_UTINY indent;                    /**< Indentation Delta, when using WBXML_ENCODER_XML_GEN_INDENT Generation Type (Default: 0) */
    WB_BOOL keep_ignorable_ws;          /**< Keep Ignorable Whitespaces (Default: FALSE) */
} WBXML2XMLParameters;

/** Parameters for wbxml2xml() function  */
typedef struct XML2WBXMLParameters_s {
    WB_BOOL keep_ignorable_ws;  /**< Keep Ignorable Whitespaces (Default: FALSE) */
    WB_BOOL use_strtbl;         /**< Generate String Table (Default: TRUE) */
} XML2WBXMLParameters;


/**
 * @brief Convert WBXML to XML
 * @param wbxml [in] WBXML Document to convert
 * @param wbxml_len [in] Length of WBXML Document
 * @param xml [out] Resulting XML Document
 * @param params [in] Parameters (if NULL, default values are used)
 * @param tree_mode [in] Do we use Tree Mode ?
 * @return WBXML_OK if convertion succeeded, an Error Code otherwise
 * @note There are two ways to launch wbxml2xml():
 *          - Tree Mode: The default one (preferred)
 *          - Direct Mode: Using internal Callbacks to generate XML (should NOT be used)
 * @warning Use the wbxml2xml() maccro instead of calling this function directly.
 */
WBXML_DECLARE(WBXMLError) wbxml2xml_real(WB_UTINY *wbxml, WB_ULONG wbxml_len, WB_UTINY **xml, WBXML2XMLParameters *params, WB_BOOL tree_mode);

/** Wrapper arround wbxml2xml_real() with default Convertion Mode */
#define wbxml2xml(a,b,c,d) wbxml2xml_real(a,b,c,d,WBXML_CONV_WBXML2XML_DEFAULT_TREE_MODE)

/**
 * @brief Convert XML to WBXML
 * @param wbxml [in] XML Document to convert
 * @param wbxml_len [out] Resulting WBXML Document
 * @param xml [out] Length of resulting WBXML Document
 * @param params [in] Parameters (if NULL, default values are used)
 * @return WBXML_OK if convertion succeeded, an Error Code otherwise
 */
WBXML_DECLARE(WBXMLError) xml2wbxml(WB_UTINY *xml, WB_UTINY **wbxml, WB_ULONG *wbxml_len, XML2WBXMLParameters *params);

/** @} */

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* WBXML_CONV_H */
