/*
 * WBXML Lib, the WBXML Library.
 * Copyright (C) 2002-2003  Aymerick Jhanne
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License (version 2.1) as published by the Free Software Foundation.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * LGPL v2.1: http://www.gnu.org/licenses/lgpl.txt
 *
 * Author Contact: libwbxml@jehanne.org
 * WBXML Lib home: http://libwbxml.jehanne.org
 */
 
/**
 * @file xml2wbxml.c
 * @ingroup xml2wbxml
 *
 * @author Aymerick Jhanne <libwbxml@jehanne.org>
 * @date 03/03/11
 *
 * @brief XML to WBXML Converter
 *
 */

#include <string.h> /* For strlen() */

#include "wbxml_conv.h"
#include "xml2wbxml_clb.h"


/************************************
 * Public Functions
 */

WBXML_DECLARE(WBXMLError) xml2wbxml(WB_UTINY *xml, WB_UTINY **wbxml, WB_ULONG *wbxml_len, XML2WBXMLParameters *params)
{
    WBXMLEncoder *wbxml_encoder = NULL;
    XML_Parser xml_parser;
    xml2wbxmlCtx xml2wbxml_ctx; /**< The Parser Context */
    WBXMLError ret = WBXML_OK;

    /* Check Parameters */
    if ((xml == NULL) || (wbxml == NULL) || (wbxml_len == NULL))
        return WBXML_ERROR_BAD_PARAMETER;

    *wbxml = NULL;
    *wbxml_len = 0;
    
    /* Create XML Parser */
    if ((xml_parser = XML_ParserCreate(NULL)) == NULL)
        return WBXML_ERROR_NOT_ENOUGH_MEMORY;
    
    /* Init User Data */
    xml2wbxml_ctx.current = NULL;
    xml2wbxml_ctx.error = WBXML_OK;

    if ((xml2wbxml_ctx.tree = wbxml_tree_create()) == NULL) {
        XML_ParserFree(xml_parser);
        return WBXML_ERROR_NOT_ENOUGH_MEMORY;
    }
    
    /* Set Handlers Callbacks */
    XML_SetStartDoctypeDeclHandler(xml_parser, xml2wbxml_clb_doctype_decl);
    XML_SetElementHandler(xml_parser, xml2wbxml_clb_start_element, xml2wbxml_clb_end_element);
	XML_SetCdataSectionHandler(xml_parser, xml2wbxml_clb_start_cdata, xml2wbxml_clb_end_cdata);
	XML_SetProcessingInstructionHandler(xml_parser , xml2wbxml_clb_pi);
    XML_SetCharacterDataHandler(xml_parser, xml2wbxml_clb_characters);
    XML_SetUserData(xml_parser, (void*)&xml2wbxml_ctx);

    /* Parse the XML Document to WBXML Tree */
    if (XML_Parse(xml_parser, xml, WBXML_STRLEN(xml), TRUE) == 0)
    {
        enum XML_Error err_code = XML_GetErrorCode(xml_parser);
        int line_num = XML_GetCurrentLineNumber(xml_parser);
        int col_num = XML_GetCurrentColumnNumber(xml_parser);
        long byte_idx = XML_GetCurrentByteIndex(xml_parser);
        int byte_total = XML_GetCurrentByteCount(xml_parser);
        const XML_LChar *error_str = XML_ErrorString(err_code);
        
        WBXML_ERROR((WBXML_CONV, "xml2wbxml convertion failed - expat error %i\n"
            "\tdescription: %s\n"
            "\tline: %i\n"
            "\tcolumn: %i\n"
            "\tbyte index: %i\n"
            "\ttotal bytes: %i\n%s",
            err_code, error_str, line_num, 
            col_num, byte_idx, byte_total, xml));

        /* Free XML Parser */
        XML_ParserFree(xml_parser);
        wbxml_tree_destroy(xml2wbxml_ctx.tree);

        return WBXML_ERROR_XML_PARSING_FAILED;
    }

    /* Free XML Parser */
    XML_ParserFree(xml_parser);

    /* Encode WBXML Tree to WBXML Document */
    if ((wbxml_encoder = wbxml_encoder_create()) == NULL) {
        wbxml_tree_destroy(xml2wbxml_ctx.tree);
        return WBXML_ERROR_NOT_ENOUGH_MEMORY;
    }
    
    /* Set the WBXML Tree to encode */
    wbxml_encoder_set_tree(wbxml_encoder, xml2wbxml_ctx.tree);

    if (params == NULL) {
        /* Default Parameters */

        /* Ignores "Empty Text" Nodes */
        wbxml_encoder_set_ignore_empty_text(wbxml_encoder, TRUE);

        /* Remove leading and trailing whitespaces in "Text Nodes" */
        wbxml_encoder_set_remove_text_blanks(wbxml_encoder, TRUE);

        /* Use String Table */
        wbxml_encoder_set_use_strtbl(wbxml_encoder, TRUE);
    }
    else {
        /* Keep Ignorable Whitespaces ? */
        if (!params->keep_ignorable_ws) {
            /* Ignores "Empty Text" Nodes */
            wbxml_encoder_set_ignore_empty_text(wbxml_encoder, TRUE);

            /* Remove leading and trailing whitespaces in "Text Nodes" */
            wbxml_encoder_set_remove_text_blanks(wbxml_encoder, TRUE);
        }

        /* String Table */
        wbxml_encoder_set_use_strtbl(wbxml_encoder, params->use_strtbl);
    }

    /* Encode WBXML */
    ret = wbxml_encoder_encode(wbxml_encoder, wbxml, wbxml_len);

    /* Clean-up */
    wbxml_tree_destroy(xml2wbxml_ctx.tree);
    wbxml_encoder_destroy(wbxml_encoder);

    return ret;
}
