/*
 * WBXML Lib, the WBXML Library.
 * Copyright (C) 2002-2003  Aymerick Jhanne
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License (version 2.1) as published by the Free Software Foundation.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * LGPL v2.1: http://www.gnu.org/licenses/lgpl.txt
 *
 * Author Contact: libwbxml@jehanne.org
 * WBXML Lib home: http://libwbxml.jehanne.org
 */
 
/**
 * @file xml2wbxml_clb.c
 * @ingroup xml2wbxml
 *
 * @author Aymerick Jhanne <libwbxml@jehanne.org>
 * @date 03/03/11
 *
 * @brief XML to WBXML Converter Callbacks (Callbacks for Expat)
 */

#include <string.h> /* For strlen() */

#include "wbxml_conv.h"
#include "xml2wbxml_clb.h"


/* Private Functions Prototypes */
static WBXMLTreeAttribute *construct_attribute_list(const WBXMLLangEntry *lang_table, const XML_Char **attrs);
static void add_node_to_tree(WBXMLTreeNode *node, xml2wbxmlCtx *ctx);


/************************************
 * Public Functions
 */

void xml2wbxml_clb_doctype_decl(void *ctx, const XML_Char *doctypeName, 
                                const XML_Char *sysid, const XML_Char *pubid, 
                                int has_internal_subset)
{    
    xml2wbxmlCtx *xml2wbxml_ctx = (xml2wbxmlCtx *) ctx;
    const WBXMLLangEntry *lang_table = NULL;

    /* Search for Language Table, given the XML Public ID and System ID */
    lang_table = wbxml_tables_search_table(wbxml_tables_get_main(), 
                                           (const WB_UTINY *) pubid, 
                                           (const WB_UTINY *) sysid, 
                                           NULL);

    if (lang_table != NULL) {
        /* Ho Yeah ! We got it ! */
        xml2wbxml_ctx->tree->lang = lang_table;
    }
    else {
        /* We will try to find the Language Table, given the Root Element */
        WBXML_WARNING((WBXML_CONV, "Language Table NOT found, given the XML Public ID and System ID"));
    }
}


void xml2wbxml_clb_start_element(void *ctx, const XML_Char *localName, const XML_Char **attrs)
{
    xml2wbxmlCtx *xml2wbxml_ctx = (xml2wbxmlCtx *) ctx;
    const WBXMLLangEntry *lang_table = NULL;
    const WBXMLTagEntry *tag_entry = NULL;
    WBXMLTreeNode *node = NULL;

    if (xml2wbxml_ctx->error != WBXML_OK)
        return;

    if (xml2wbxml_ctx->current == NULL) {
        /* This is the Root Element */
        if (xml2wbxml_ctx->tree->lang == NULL) {
            /* Language Table not already found: Search again */
            lang_table = wbxml_tables_search_table(wbxml_tables_get_main(), 
                                                   NULL, 
                                                   NULL, 
                                                   (const WB_UTINY *) localName);
        
            if (lang_table == NULL) {
                /* Damn, this is an unknown language for us... */
                xml2wbxml_ctx->error = WBXML_ERROR_UNKNOWN_XML_LANGUAGE;
                return;
            }
            else {
                /* Well, we hope this was the Language we are searching for.. let's try with it :| */
                xml2wbxml_ctx->tree->lang = lang_table;
            }
        }
    }

    /* Create a new Node */
    if ((node = wbxml_tree_node_create(WBXML_TREE_ELEMENT_NODE)) == NULL) {
        xml2wbxml_ctx->error = WBXML_ERROR_NOT_ENOUGH_MEMORY;
        return;
    }

    /* Search for XML Tag Name in Table */
    if ((tag_entry = wbxml_tables_get_tag_from_xml(xml2wbxml_ctx->tree->lang, (const WB_UTINY *) localName)) != NULL)
        node->name = wbxml_tag_create_token(tag_entry);
    else
        node->name = wbxml_tag_create_literal((WB_UTINY *) localName);

    if (node->name == NULL) {
        wbxml_tree_node_destroy(node);
        xml2wbxml_ctx->error = WBXML_ERROR_NOT_ENOUGH_MEMORY;
        return;
    }

    /* Set Attributes */
    if ((attrs != NULL) && (*attrs != NULL))
    {
        if ((node->attrs = construct_attribute_list(xml2wbxml_ctx->tree->lang, attrs)) == NULL) {
            wbxml_tree_node_destroy(node);
            xml2wbxml_ctx->error = WBXML_ERROR_NOT_ENOUGH_MEMORY;
            return;
        }
    }

    /* Add this Node to Tree  */
    add_node_to_tree(node, xml2wbxml_ctx);
}


void xml2wbxml_clb_end_element(void *ctx, const XML_Char *localName)
{
    xml2wbxmlCtx *xml2wbxml_ctx = (xml2wbxmlCtx *) ctx;

    if (xml2wbxml_ctx->error != WBXML_OK)
        return;

    if (xml2wbxml_ctx->current == NULL) {
        xml2wbxml_ctx->error = WBXML_ERROR_INTERNAL;
        return;
    }

    if (xml2wbxml_ctx->current->parent == NULL) {
        /* This must be the Root Element */
        if (xml2wbxml_ctx->current != xml2wbxml_ctx->tree->root) {
            xml2wbxml_ctx->error = WBXML_ERROR_INTERNAL;
        }
    }
    else {
        /* Go back one step upper in the tree */
        xml2wbxml_ctx->current = xml2wbxml_ctx->current->parent;
    }
}


void xml2wbxml_clb_start_cdata(void *ctx)
{
    xml2wbxmlCtx *xml2wbxml_ctx = (xml2wbxmlCtx *) ctx;

    if (xml2wbxml_ctx->error != WBXML_OK)
        return;

    /** @todo xml2wbxml_clb_start_cdata() */
}


void xml2wbxml_clb_end_cdata(void *ctx)
{
    xml2wbxmlCtx *xml2wbxml_ctx = (xml2wbxmlCtx *) ctx;

    if (xml2wbxml_ctx->error != WBXML_OK)
        return;

    /** @todo xml2wbxml_clb_end_cdata() */
}


void xml2wbxml_clb_characters(void *ctx, const XML_Char *ch, int len)
{
    xml2wbxmlCtx *xml2wbxml_ctx = (xml2wbxmlCtx *) ctx;
    WBXMLTreeNode *node = NULL;

    if (xml2wbxml_ctx->error != WBXML_OK)
        return;

    /* Create a new Node */
    if ((node = wbxml_tree_node_create(WBXML_TREE_TEXT_NODE)) == NULL) {
        xml2wbxml_ctx->error = WBXML_ERROR_NOT_ENOUGH_MEMORY;
        return;
    }

    /* Set Content */
    if ((node->content = wbxml_buffer_create_real(ch, len, len)) == NULL) {
        wbxml_tree_node_destroy(node);
        xml2wbxml_ctx->error = WBXML_ERROR_NOT_ENOUGH_MEMORY;
        return;
    }

    /* Add this Node to Tree  */
    add_node_to_tree(node, xml2wbxml_ctx);

    /* Go back one step upper in the tree */
    xml2wbxml_ctx->current = xml2wbxml_ctx->current->parent;
}


void xml2wbxml_clb_pi(void *ctx, const XML_Char *target, const XML_Char *data)
{
    xml2wbxmlCtx *xml2wbxml_ctx = (xml2wbxmlCtx *) ctx;

    if (xml2wbxml_ctx->error != WBXML_OK)
        return;

    /** @todo wbxml2xml_clb_pi() */
}


/***************************************************
 *    Private Functions
 */


/**
 * @brief Construct a Tree Attribute List
 * @param lang_table The Language Table we are dealing with
 * @param attrs The Attribute Table to Duplicate
 * @return The Attribute List, or NULL if Error
 */
static WBXMLTreeAttribute *construct_attribute_list(const WBXMLLangEntry *lang_table, const XML_Char **attrs)
{
    WBXMLTreeAttribute *attr = NULL, *first = NULL, *curr = NULL;
    const WBXMLAttrEntry *attr_entry = NULL;
    const WB_UTINY **p = (const WB_UTINY **) attrs;

    if ((lang_table == NULL) || (attrs == NULL))
        return NULL;

    while (p && *p) {
        /* Create Tree Attribute */
        if ((attr = wbxml_tree_attribute_create()) == NULL) {
            wbxml_tree_attribute_destroy(first);
            return NULL;
        }

        /* Keep the first one somewhere */
        if (p == (const WB_UTINY **) attrs)
            first = attr;

        /* Link it to previous Attribute */
        if (curr != NULL)
            curr->next = attr;

        /* Create Attribute */
        if ((attr->attr = wbxml_attribute_create()) == NULL) {
            wbxml_tree_attribute_destroy(first);
            return NULL;
        }

        /* Set Attribute Name */
        if ((attr_entry = wbxml_tables_get_attr_from_xml(lang_table, (WB_UTINY *) *p, (WB_UTINY *) *(p+1), NULL)) != NULL)
            attr->attr->name = wbxml_attribute_name_create_token(attr_entry);
        else
            attr->attr->name = wbxml_attribute_name_create_literal((WB_UTINY*) *p);

        if (attr->attr->name == NULL) {
            wbxml_tree_attribute_destroy(first);
            return NULL;
        }

        /* Set Attribute Value */
        attr->attr->value = wbxml_buffer_create_real(*(p+1), WBXML_STRLEN(*(p+1)), WBXML_STRLEN(*(p+1)));
        if (attr->attr->value == NULL) {
            wbxml_tree_attribute_destroy(first);
            return NULL;
        }
        
        /* Go to next Attribute */
        curr = attr;

        p += 2;
    }

    return first;
}


/**
 * @brief Add a Node to Tree
 * @param node The Node to add
 * @param ctx The Tree Context
 */
static void add_node_to_tree(WBXMLTreeNode *node, xml2wbxmlCtx *ctx)
{
    WBXMLTreeNode *parent = NULL, *tmp = NULL;

    /* This is the new Current Node */
    parent = ctx->current;
    node->parent = parent;    
    ctx->current = node;

    /* Check if this is the Root Element */
    if (parent != NULL) {
		/* This is not the Root Element... search for previous sibbling element */
		if (parent->children != NULL) {
            /* Add this Node to end of Sibbling Node list of Parent */
			tmp = parent->children;
			
			while (tmp->next != NULL)
				tmp = tmp->next;
			
			node->prev = tmp;
			tmp->next = node;
		}
		else {
			/* No previous sibbling element */
			parent->children = node;
		}
    }
    else {
        /* This is the Root Element */
        ctx->tree->root = node;
    }
}
