/*
 * WBXML Lib, the WBXML Library.
 * Copyright (C) 2002-2003  Aymerick Jhanne
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License (version 2.1) as published by the Free Software Foundation.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * LGPL v2.1: http://www.gnu.org/licenses/lgpl.txt
 *
 * Author Contact: libwbxml@jehanne.org
 * WBXML Lib home: http://libwbxml.jehanne.org
 */
 
/**
 * @file wbxml2xml_tool.c
 * @ingroup wbxml2xml_tool
 *
 * @author Aymerick Jhanne <libwbxml@jehanne.org>
 * @date 03/02/22
 *
 * @brief WBXML to XML Converter
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <memory.h>

#include "getopt.h"

#include "wbxml_conv.h" /* libwbxml2_conv */


#define INPUT_BUFFER_SIZE 1000


static void help(void) {
    printf("Usage: \n");
    printf("  wbxml2xml -i -o output.xml input.wbxml\n\n");
    printf("Options: \n");
    printf("    -o output.xml : output file\n");
    printf("    -m X (Generation mode - Default: 1) with:\n");
    printf("       0: Compact Generation\n");
    printf("       1: Indent Generation\n");
    printf("       2: Canonical Generation\n");
    printf("    -i X (Indent delta when using mode '1' - Default: 1)\n");
    printf("    -k (Keep Ignorable Whitespaces - Default: FALSE)\n");
}


WB_LONG main(WB_LONG argc, WB_TINY **argv)
{
    WB_UTINY *wbxml = NULL, *output = NULL, *xml = NULL;
    FILE *input_file = NULL, *output_file = NULL;
    WB_LONG count = 0, wbxml_size = 0, total = 0;
    WB_TINY opt;
    WBXMLError ret = WBXML_OK;
    WB_UTINY input_buffer[INPUT_BUFFER_SIZE + 1];
    WBXML2XMLParameters params;

    /* Init Default Parameters */    
    params.gen_type = WBXML_ENCODER_XML_GEN_INDENT;
    params.indent = 1;
    params.keep_ignorable_ws = FALSE;

    while ((opt = getopt(argc, argv, "kh?o:m:i:")) != EOF) 
    {
        switch (opt) {
        case 'k':
            params.keep_ignorable_ws = TRUE;
            break;
        case 'i':
            params.indent = atoi(optarg);
            break;
        case 'm':
            switch (atoi(optarg)) {
            case 0:
                params.gen_type = WBXML_ENCODER_XML_GEN_COMPACT;
                break;
            case 1:
                params.gen_type = WBXML_ENCODER_XML_GEN_INDENT;
                break;
            case 2:
                params.gen_type = WBXML_ENCODER_XML_GEN_CANONICAL;
                break;
            default:
                params.gen_type = WBXML_ENCODER_XML_GEN_INDENT;
            }
            break;
        case 'o':
            output = optarg;
            break;
        case 'h':
        case '?':
        default:
            help();
            return 0;
        }
    }

    if (optind >= argc) {
        printf("Missing arguments\n");
        help();
        return 0;
    }

#ifdef WBXML_USE_LEAKTRACKER
    lt_init_mem();
    lt_log_open_file("wbxml2xml.log");
#endif

    /**********************************
     *  Read the WBXML Document
     */

    /* Open WBXML document */
    input_file = fopen(argv[optind], "rb");
    if (input_file == NULL) {
        printf("Failed to open %s\n", argv[optind]);
        goto clean_up;
    }

    /* Read WBXML document */
    while(!feof(input_file))    {
        count = fread(input_buffer, sizeof(WB_UTINY), INPUT_BUFFER_SIZE, input_file);
        if (ferror(input_file))      {
            printf("Error while reading from file %s\n", argv[optind]);
            fclose(input_file);
            if (wbxml != NULL)
                wbxml_free(wbxml);
            goto clean_up;
        }

        total += count;
        wbxml = wbxml_realloc(wbxml, total);
        if (wbxml == NULL) {
            printf("Not enought memory\n");
            fclose(input_file);
            if (wbxml != NULL)
                wbxml_free(wbxml);
            goto clean_up;
        }

        memcpy(wbxml + wbxml_size, input_buffer, count);
        wbxml_size += count;
    }

    fclose(input_file);

    /* Convert WBXML document */
    ret = wbxml2xml(wbxml, wbxml_size, &xml, &params);
    if (ret != WBXML_OK) {
        printf("wbxml2xml failed: %s\n", wbxml_errors_string(ret));
    }
    else {
        printf("wbxml2xml succeded: \n%s\n", xml);

        if (output != NULL) {
            /* Open Output File */
            output_file = fopen(output, "w");
            if (output_file == NULL) {
                printf("Failed to open output file: %s\n", output);
            }
            else {
                /* Write to Output File */
                fprintf(output_file, xml);
                fclose(output_file);
            }
        }

        /* Clean-up */
        wbxml_free(xml);
    }

    wbxml_free(wbxml);

clean_up:

#ifdef WBXML_USE_LEAKTRACKER
    lt_check_leaks();
    lt_shutdown_mem();
    lt_log_close_file();
#endif

    return 0;
}
