/*
 * WBXML Lib, the WBXML Library.
 * Copyright (C) 2002-2003  Aymerick Jhanne
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License (version 2.1) as published by the Free Software Foundation.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * LGPL v2.1: http://www.gnu.org/licenses/lgpl.txt
 *
 * Author Contact: libwbxml@jehanne.org
 * WBXML Lib home: http://libwbxml.jehanne.org
 */
 
/**
 * @file xml2wbxml_tool.c
 * @ingroup xml2wbxml_tool
 *
 * @author Aymerick Jhanne <libwbxml@jehanne.org>
 * @date 03/03/11
 *
 * @brief XML to WBXML Converter
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <memory.h>

#include "getopt.h"

#include "wbxml_conv.h" /* libwbxml2_conv */


#define INPUT_BUFFER_SIZE 5000


static void help(void) {
    printf("Usage: \n");
    printf("  xml2wbxml -o output.wbxml input.xml\n\n");
    printf("Options: \n");
    printf("    -o output.wbxml : output file\n");
    printf("    -k : keep ignorable whitespaces (Default: ignore)\n");
    printf("    -n : do NOT generate String Table (Default: generate)\n");
}


WB_LONG main(WB_LONG argc, WB_TINY **argv)
{
    WB_UTINY *wbxml = NULL, *output = NULL, *xml = NULL;
    FILE *input_file = NULL, *output_file = NULL;
    WB_LONG count = 0, xml_size = 0, total = 0, wbxml_len = 0;
    WB_TINY opt;
    WBXMLError ret = WBXML_OK;
    WB_UTINY input_buffer[INPUT_BUFFER_SIZE + 1];
    XML2WBXMLParameters params;

    /* Init Default Parameters */    
    params.use_strtbl = TRUE;
    params.keep_ignorable_ws = FALSE;

    while ((opt = getopt(argc, argv, "nkh?o:")) != EOF) 
    {
        switch (opt) {
        case 'n':
            params.use_strtbl = FALSE;
            break;
        case 'k':
            params.keep_ignorable_ws = TRUE;
            break;
        case 'o':
            output = optarg;
            break;
        case 'h':
        case '?':
        default:
            help();
            return 0;
        }
    }

    if (optind >= argc) {
        printf("Missing arguments\n");
        help();
        return 0;
    }

#ifdef WBXML_USE_LEAKTRACKER
    lt_init_mem();
    lt_log_open_file("xml2wbxml.log");
#endif

    /**********************************
     *  Read the XML Document
     */

    /* Open XML document */
    input_file = fopen(argv[optind], "r");
    if (input_file == NULL) {
        printf("Failed to open %s\n", argv[optind]);
        goto clean_up;
    }

    /* Read XML document */
    while(!feof(input_file))    {
        count = fread(input_buffer, sizeof(WB_UTINY), INPUT_BUFFER_SIZE, input_file);
        if (ferror(input_file))      {
            printf("Error while reading from file %s\n", argv[1]);
            fclose(input_file);
            if (xml != NULL)
                wbxml_free(xml);
            goto clean_up;
        }

        total += count;
        xml = wbxml_realloc(xml, total + 1);
        if (xml == NULL) {
            printf("Not enought memory\n");
            fclose(input_file);
            goto clean_up;
        }

        memcpy(xml + xml_size, input_buffer, count);
        xml_size += count;
    }

    fclose(input_file);

    xml[xml_size] = '\0';

    /* Convert XML document */
    ret = xml2wbxml(xml, &wbxml, &wbxml_len, &params);
    if (ret != WBXML_OK) {
        printf("xml2wbxml failed: %s\n", wbxml_errors_string(ret));
    }
    else {
        printf("xml2wbxml succeded\n");

        if (output != NULL) {
            /* Open Output File */
            output_file = fopen(output, "wb");
            if (output_file == NULL) {
                printf("Failed to open output file: %s\n", output);
            }
            else {
                /* Write to Output File */
                if (fwrite(wbxml, sizeof(WB_UTINY), (WB_ULONG) wbxml_len, output_file) < (WB_ULONG) wbxml_len)
                    printf("Error while writing to file: %s\n", output);
                else
                    printf("Written %u bytes to file: %s\n", wbxml_len, output);

                fclose(output_file);
            }
        }

        /* Clean-up */
        if (wbxml != NULL)
            wbxml_free(wbxml);
    }

    if (xml != NULL)
        wbxml_free(xml);

clean_up:

#ifdef WBXML_USE_LEAKTRACKER
    lt_check_leaks();
    lt_shutdown_mem();
    lt_log_close_file();
#endif

    return 0;
}
