/*
 * (c) Copyright 1992 by Panagiotis Tsirigotis
 * (c) Sections Copyright 1998-2001 by Rob Braun
 * All rights reserved.  The file named COPYRIGHT specifies the terms 
 * and conditions for redistribution.
 */


#include "config.h"
#include <syslog.h>
#include <time.h>
#include <signal.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <stdlib.h>

#ifdef HAVE_STDINT_H
#include <stdint.h>
#endif

#include "str.h"
#include "access.h"
#ifdef LIBWRAP
#include <tcpd.h>
int deny_severity = LOG_INFO;
int allow_severity = LOG_INFO;
#endif

#ifdef HAVE_LOADAVG
#include "xgetloadavg.h"
#endif

#include "msg.h"
#include "addr.h"
#include "sconf.h"
#include "log.h"	
#include "state.h"
#include "sconst.h"
#include "main.h"	/* for ps */
#include "timex.h"
#include "xconfig.h"
#include "xtimer.h"

#if !defined(NAME_MAX)
      #ifdef FILENAME_MAX
      #define NAME_MAX FILENAME_MAX
      #else
      #define NAME_MAX 256
      #endif
#endif

/*
 * This is the globals for the Sensor. The Sensor will add the incoming IP
 * address to the global_no_access table for whatever the configured time is.
 */
pset_h global_no_access = NULL;      /* global no_access listn   */
pset_h global_no_access_time = NULL;   /* time of the infraction   */


struct name_value access_code_names[] =
   {
      { "address",                  (int) AC_ADDRESS         },
      { "time",                     (int) AC_TIME            },
      { "fork",                     (int) AC_FORK            },
      { "service_limit",            (int) AC_SERVICE_LIMIT   },
      { "per_source_limit",         (int) AC_PER_SOURCE_LIMIT},
      { "process_limit",            (int) AC_PROCESS_LIMIT   },
      { "libwrap",                  (int) AC_LIBWRAP         },
      { "load",                     (int) AC_LOAD            },
      { "connections per second",   (int) AC_CPS             },
      { CHAR_NULL,                  1                        },
      { "UNKNOWN",                  0                        }
   } ;

/* This is called by the flags processor */
static void cps_service_restart(void)
{
   int i;
   time_t nowtime;
   const char *func = "cps_service_restart";

   nowtime = time(NULL);
   for( i=0; i < pset_count( SERVICES(ps) ); i++ ) {
      struct service *sp;
      struct service_config *scp;

      sp = pset_pointer( SERVICES(ps), i);
      scp = SVC_CONF( sp );

      if( sp->svc_state == SVC_DISABLED ) {
         if ( scp->sc_time_reenable <= nowtime ) {
            /* re-enable the service */
            if( svc_activate(sp) == OK ) {
               msg(LOG_ERR, func,
               "Activating service %s", scp->sc_name);
            } else {
               msg(LOG_ERR, func,
               "Error activating service %s", 
               scp->sc_name) ;
            } /* else */
         }
      }
   } /* for */
}

/*
 * Returns OK if the IP address in sinp is acceptable to the access control
 * lists of the specified service.
 */
static status_e remote_address_check(const struct service *sp, const union xsockaddr *sinp)
{
   /*
    * of means only_from, na means no_access
    */
   const char *func = "remote_address_check";
   bool_int   of_matched  = FALSE;
   bool_int   na_matched  = FALSE;
   union xsockaddr     addr;
   int len;

   if (sinp == NULL )
      return FAILED;

   len = sizeof(*sinp);
   memcpy(&addr, sinp, len);

   if ( SC_SENSOR(sp->svc_conf) )
   {   /* They hit a sensor...   */
      if (sp->svc_conf->sc_deny_time != 0)   /* 0 simply logs it   */
      {
         if ( pset_count( global_no_access ) < MAX_GLOBAL_NO_ACCESS)
         {
            int item_matched = addrlist_match( global_no_access, SA(&addr) );
            if ( item_matched == 0)
            {   /* no match...adding to the list   */
               char *dup_addr = new_string(xaddrname( sinp ) );

               if (dup_addr == NULL )
                  return FAILED;

               if (addrlist_add(global_no_access, dup_addr) == FAILED)
                  msg(LOG_ERR, func,
                  "Failed adding %s to the global_no_access list", dup_addr);
               else
               {
                  time_t nowtime;
                  char time_buf[40], *tmp;

                  msg(LOG_CRIT, func,
                     "Adding %s to the global_no_access list for %d minutes",
                  dup_addr, sp->svc_conf->sc_deny_time);
                  nowtime = time(NULL);

                  if (sp->svc_conf->sc_deny_time == -1)
                     strcpy(time_buf, "-1");
                  else
                     strx_nprint(time_buf, 38, "%ld", 
                        (time_t)nowtime+(60*sp->svc_conf->sc_deny_time));
                  tmp = new_string(time_buf);
                  if (tmp != NULL)
                  {
                     if (pset_add(global_no_access_time, tmp) == NULL)
                     {
                        msg(LOG_ERR, func,
                           "Failed adding %s to the global_no_access_time list. global_no_access list is broken, xinetd needs restarting.", dup_addr);
                        /* ideally, we should rollback the previous addr addition.   */
                     }
                  }
               }
               free(dup_addr);
            }
            else
            {   
               /* Here again, eh?...update time stamp. Note: We are in 
                * parent's context.   
                */
               char *exp_time;
               time_t stored_time;

               item_matched--; /* Is # plus 1, to even get here must be >= 1 */
               exp_time = pset_pointer( global_no_access_time, item_matched ) ;
               if (exp_time == NULL)
                  return FAILED;

               if( parse_base10(exp_time, (int *)&stored_time) ) 
               {  /* if never let them off, bypass */
                  if (stored_time != -1)
                  {
                     time_t nowtime, new_time;

                     nowtime = time(NULL);
                     new_time = (time_t)nowtime+(60*sp->svc_conf->sc_deny_time);
                     if (difftime(new_time, (time_t)stored_time) > 0.0)
                     {   /* new_time is longer save it   */
                        char time_buf[40], *new_exp_time;

                        strx_nprint(time_buf, 38, "%ld", (long)new_time);
                        new_exp_time = new_string(time_buf);
                        if( new_exp_time )
                        {
                           free(exp_time);
                           global_no_access_time->ptrs[ (unsigned)item_matched ] = new_exp_time;
                        }
                     }
                  }
               }
            }
         }
         else
         {
            msg(LOG_ERR, func, "Maximum global_no_access count reached.");
         }
      }
      return ( FAILED );
   }
   else if ( (global_no_access) && pset_count( global_no_access ) )
   {   
      /* They hit a real server...note, this is likely to be a child process. */
      if (addrlist_match( global_no_access, SA(&addr) ))
         return ( FAILED );
   }

   if ( sp->svc_no_access != NULL )
      na_matched = addrlist_match( sp->svc_no_access, SA(&addr));

   if ( sp->svc_only_from != NULL )
      of_matched = addrlist_match( sp->svc_only_from, SA(&addr));

   /*
    * Check if the specified address is in both lists
    */
   if ( na_matched && of_matched )
   {
      /*
       * The greater match wins.
       * If the matches are equal, this is an error in the service entry
       * and we cannot allow a server to start.
       * We do not disable the service entry (not our job).
       */
      if ( na_matched == of_matched )
         msg( LOG_ERR, func,
"Service=%s: only_from list and no_access list match equally the address %s",
            SVC_ID( sp ), 
            xaddrname( sinp ) ) ;
      return( ( of_matched > na_matched ) ? OK : FAILED ) ;
   }

   if ( sp->svc_no_access != NULL && (na_matched != 0) )
      return( FAILED ) ;
   if ( sp->svc_only_from != NULL && (of_matched == 0) )
      return( FAILED ) ;
   /*
    * If no lists were specified, the default is to allow starting a server
    */
   return( OK ) ;
}

void scrub_global_access_lists( void )
{
   unsigned count;

   if( global_no_access == NULL )
      count = 0;
   else
      count = pset_count( global_no_access );
   if (count )
   {   
      int found_one = 0;
      unsigned u;
      time_t nowtime = time(NULL);

      for (u=0; u < count; u++)
      {
         char *exp_time;
         time_t stored_time;
         exp_time = pset_pointer( global_no_access_time, u ) ;
         stored_time = atol(exp_time);
         if (stored_time == -1)   /* never let them off   */
            continue;
         if (difftime(nowtime, (time_t)stored_time) > 0.0)
         {
            __pset_pointer ptr;

            pset_pointer(global_no_access, u) = NULL;

            ptr = global_no_access_time->ptrs[ u ];
            free(ptr);
            pset_pointer(global_no_access_time, u ) = NULL;
            found_one = 1;
         }
      }
      if (found_one)
      {
         pset_compact( global_no_access );
         pset_compact( global_no_access_time );
      }
   }
}

/*
 * mp is the mask pointer, t is the check type
 */
#define CHECK( mp, t )      ( ( (mp) == NULL ) || M_IS_SET( *(mp), t ) )

/*
 * Perform the access controls specified by check_mask.
 * If check_mask is NULL, perform all access controls
 */
access_e access_control( struct service *sp, 
                         const connection_s *cp, 
                         const mask_t *check_mask )
{
   struct service_config   *scp = SVC_CONF( sp ) ;

   /* make sure it's not one of the special pseudo services */
   if( (strncmp(SC_NAME( scp ), INTERCEPT_SERVICE_NAME, sizeof(INTERCEPT_SERVICE_NAME)) == 0) || (strncmp(SC_NAME( scp ), LOG_SERVICE_NAME, sizeof(LOG_SERVICE_NAME)) == 0) ) {
      return (AC_OK);
   }

   /* This has to be before the TCP_WRAPPERS stuff to make sure that
      the sensor gets a chance to see the address */
   if ( CHECK( check_mask, CF_ADDRESS ) &&
         remote_address_check( sp, conn_xaddress( cp ) ) == FAILED )
      return( AC_ADDRESS ) ;

   if( ! SC_NOLIBWRAP( scp ) ) 
   { /* LIBWRAP code block */
#ifdef LIBWRAP
   struct request_info req;
   char *server = NULL;

   /* get the server name to pass to libwrap */
   if( SC_NAMEINARGS( scp ) )
      server = strrchr( scp->sc_server_argv[0], '/' );
   else {
      if( scp->sc_server == NULL ) {
         /* probably an internal server, use the service id instead */
         server = scp->sc_id;
         server--;  /* nasty.  we increment it later... */
      } else {
         server = strrchr( scp->sc_server, '/' );
      }
   }

   /* If this is a redirection, go by the service name,
    * since the server name will be bogus.
    */
   if( scp->sc_redir_addr != NULL ) {
      server = scp->sc_name;
      server--; /* nasty but ok. */
   }

   if( server == NULL )
      server = scp->sc_server_argv[0];
   else
      server++;

   request_init(&req, RQ_DAEMON, server, RQ_FILE, cp->co_descriptor, 0);
   fromhost(&req);
   if (!hosts_access(&req)) {
      msg(deny_severity, "access_control", "libwrap refused connection to %s from %s", SC_ID(scp), conn_addrstr(cp));
      return(AC_LIBWRAP);
   }
#endif
   } /* LIBWRAP code block */

   return( AC_OK ) ;
}


/* Do the "light weight" access control here */
access_e parent_access_control( struct service *sp, const connection_s *cp )
{
   struct service_config *scp = SVC_CONF( sp ) ;
   int u, n;
   time_t nowtime;

   /* make sure it's not one of the special pseudo services */
   if( (strncmp(SC_NAME( scp ), INTERCEPT_SERVICE_NAME, sizeof(INTERCEPT_SERVICE_NAME)) == 0) || (strncmp(SC_NAME( scp ), LOG_SERVICE_NAME, sizeof(LOG_SERVICE_NAME)) == 0) ) 
      return (AC_OK);

   /* CPS handler */
   if( scp->sc_time_conn_max != 0 ) {
      int time_diff;
      nowtime = time(NULL);
      time_diff = nowtime - scp->sc_time_limit ;

      if( scp->sc_time_conn == 0 ) {
         scp->sc_time_conn++;
         scp->sc_time_limit = nowtime;
      } else if( time_diff < scp->sc_time_conn_max ) {
         scp->sc_time_conn++;
         if( time_diff == 0 ) time_diff = 1;
         if( scp->sc_time_conn/time_diff > scp->sc_time_conn_max ) {
            svc_deactivate(sp);
            msg(LOG_ERR, "xinetd", "Deactivating service %s due to excessive incoming connections.  Restarting in %d seconds.", scp->sc_name, (int)scp->sc_time_wait);
            nowtime = time(NULL);
            scp->sc_time_reenable = nowtime + scp->sc_time_wait;
            xtimer_add(cps_service_restart, scp->sc_time_wait);
            return(AC_CPS);
         }
      } else {
         scp->sc_time_limit = nowtime;
         scp->sc_time_conn = 1;
      }
   }

#ifdef HAVE_LOADAVG
   if ( scp->sc_max_load != 0 ) {
      if ( xgetloadavg() >= scp->sc_max_load ) {
         msg(LOG_ERR, "xinetd", 
            "refused connect from %s due to excessive load", 
            conn_addrstr(cp));
         return( AC_LOAD );
      }
   }
#endif

   if ( SC_ACCESS_TIMES( scp ) != NULL && 
         ! ti_current_time_check( SC_ACCESS_TIMES( scp ) ) )
      return( AC_TIME ) ;

   if ( SVC_RUNNING_SERVERS( sp ) >= SC_INSTANCES( scp ) )
      return( AC_SERVICE_LIMIT ) ;

   if( scp->sc_per_source != UNLIMITED ) {
      if ( conn_xaddress(cp) != NULL ) {
         n = 0 ;
         for ( u = 0 ; u < pset_count( SERVERS( ps ) ) ; u++ ) {
            struct server *serp = NULL;
            connection_s *cop = NULL;
            serp = SERP( pset_pointer( SERVERS( ps ), u ) ) ;
            if ( (SERVER_SERVICE( serp ) == sp) &&
               ( cop = SERVER_CONNECTION( serp ) ) ) {

               if ( SC_IPV6( scp ) && IN6_ARE_ADDR_EQUAL( &(cop->co_remote_address.sa_in6.sin6_addr), &((conn_xaddress(cp))->sa_in6.sin6_addr)) )
                  n++;
               if ( SC_IPV4( scp ) && (cop->co_remote_address.sa_in.sin_addr.s_addr == (conn_xaddress(cp))->sa_in.sin_addr.s_addr) )
                  n++;
            }
         }

         if ( n >= scp->sc_per_source )
            return( AC_PER_SOURCE_LIMIT ) ;
      }
   }

   if ( ps.ros.process_limit ) {
      unsigned processes_to_create = SC_IS_INTERCEPTED( scp ) ? 2 : 1 ;

      if ( pset_count( SERVERS( ps ) ) + processes_to_create > 
         ps.ros.process_limit ) {
         return( AC_PROCESS_LIMIT ) ;
      }
   }

   return (AC_OK);
}
