/*
 * (c) Copyright 1992 by Panagiotis Tsirigotis
 * (c) Sections Copyright 1998-2001 by Rob Braun
 * All rights reserved.  The file named COPYRIGHT specifies the terms 
 * and conditions for redistribution.
 */

#include "config.h"
#include <sys/types.h>
#include <syslog.h>
#include <string.h>
#include <netinet/in.h>
#include <stdlib.h>

#ifdef HAVE_RPC_RPC_H
#include <rpc/rpc.h>
#endif

#ifdef HAVE_RPC_RPCENT_H
#include <rpc/rpcent.h>
#endif

#ifdef HAVE_NETDB_H
#include <netdb.h>
#endif

#include "str.h"
#include "sio.h"
#include "confparse.h"
#include "msg.h"
#include "xconfig.h"
#include "parse.h"
#include "special.h"
#include "sconst.h"
#include "env.h"
#include "sconf.h"
#include "sensor.h"

/*
 * Pset iterator used by functions in this file.
 * It lives only when get_configuration is called (i.e. it is created and
 * destroyed each time). This is because the pset it is iterating on
 * changes.
 */
static psi_h iter ;

static status_e fix_server_argv( struct service_config *scp )
{
   char *server_name ;
   const char *func = "fix_server_argv" ;

   if( scp->sc_server == NULL )
      return( FAILED );

   if( SC_NAMEINARGS( scp ) ) {
      if( !SC_SPECIFIED(scp, A_SERVER_ARGS ) ){
         msg( LOG_ERR, func, 
              "Must specify server args if using NAMEINARGS flag");
         return( FAILED );
      }

      return ( OK );
   }

   /*
    * Check if the user specified any server arguments.
    * If not, then the server_argv has not been allocated yet,
    * so malloc it (size 2)
    * Put in argv[ 0 ] the last component of the server pathname
    */
   if ( ! SC_SPECIFIED( scp, A_SERVER_ARGS ) )
   {
      scp->sc_server_argv = (char **) malloc( 2 * sizeof( char * ) ) ;
      if ( scp->sc_server_argv == NULL )
      {
         out_of_memory( func ) ;
         return( FAILED ) ;
      }
      scp->sc_server_argv[ 0 ] = NULL ;
      scp->sc_server_argv[ 1 ] = NULL ;
      SC_PRESENT( scp, A_SERVER_ARGS ) ;
   }

   /*
    * Determine server name
    */
   server_name = strrchr( scp->sc_server, '/' ) ;
   if ( server_name == NULL )
      server_name = scp->sc_server ;
   else
      server_name++ ;      /* skip the '/' */

   /*
    * Place it in argv[ 0 ]
    */
   scp->sc_server_argv[ 0 ] = new_string( server_name ) ;
   if ( scp->sc_server_argv[ 0 ] == NULL )
   {
      out_of_memory( func ) ;
      return( FAILED ) ;
   }
   return( OK ) ;
}



#define USE_DEFAULT( scp, def, attr_id )   \
         ( ! SC_SPECIFIED( scp, attr_id ) && SC_SPECIFIED( def, attr_id ) )

/*
 * Fill the service configuration with attributes that were not
 * explicitly specified. These can be:
 *      1) implied attributes (like the server name in argv[0])
 *      2) attributes from 'defaults' so that we won't need to check
 *         'defaults' anymore.
 *      3) default values (like the service instance limit)
 */
static status_e attr_fill( struct service_config *scp, 
                            struct service_config *def )
{
   const char *func = "attr_fill" ;

   if ( ! SC_IS_INTERNAL( scp ) && fix_server_argv( scp ) == FAILED )
      return( FAILED ) ;

   /* 
    * FIXME: Should all these set SPECIFY or PRESENT ? 
    * PRESENT makes more sense. Also, these sb documented on manpage. -SG
    */
   if ( ! SC_SPECIFIED( scp, A_INSTANCES ) )
   {
      scp->sc_instances = SC_SPECIFIED( def, A_INSTANCES ) ? def->sc_instances
                                                     : DEFAULT_INSTANCE_LIMIT ;
      SC_PRESENT( scp, A_INSTANCES ) ;
   }

   if ( (! SC_SPECIFIED( scp, A_UMASK )) && SC_SPECIFIED( def, A_UMASK ) ) 
   {
      scp->sc_umask = def->sc_umask;
      SC_SPECIFY( scp, A_UMASK );
   }

   if ( ! SC_SPECIFIED( scp, A_PER_SOURCE ) )
   {
      scp->sc_per_source = SC_SPECIFIED( def, A_PER_SOURCE ) ? def->sc_per_source : DEFAULT_INSTANCE_LIMIT ;
      SC_SPECIFY( scp, A_PER_SOURCE ) ;
   }

   if ( ! SC_SPECIFIED( scp, A_GROUPS ) )
   {
      scp->sc_groups = SC_SPECIFIED( def, A_GROUPS ) ? def->sc_groups : NO;
      SC_SPECIFY( scp, A_GROUPS );
   }

   if ( ! SC_SPECIFIED( scp, A_CPS ) ) 
   {
      scp->sc_time_conn_max = SC_SPECIFIED( def, A_CPS ) ? def->sc_time_conn_max : DEFAULT_LOOP_RATE;
      scp->sc_time_wait = SC_SPECIFIED( def, A_CPS ) ? def->sc_time_wait : DEFAULT_LOOP_TIME;
      scp->sc_time_reenable = 0;
   }

   if ( ! SC_SPECIFIED( scp, A_MAX_LOAD ) ) {
      scp->sc_max_load = SC_SPECIFIED( def, A_MAX_LOAD ) ? def->sc_max_load : 0 ;
      SC_SPECIFY( scp, A_MAX_LOAD ) ;
   }

   if ( ! SC_SPECIFIED( scp, A_BIND ) ) {
      scp->sc_bind_addr = SC_SPECIFIED( def, A_BIND ) ? def->sc_bind_addr : 0 ;
      if (scp->sc_bind_addr != 0)
         SC_SPECIFY( scp, A_BIND ) ;
   }

   if ( ! SC_SPECIFIED( scp, A_V6ONLY ) ) {
      scp->sc_v6only = SC_SPECIFIED( def, A_V6ONLY ) ? def->sc_v6only : NO;
      SC_SPECIFY( scp, A_V6ONLY );
   }

   if ( ! SC_SPECIFIED( scp, A_DENY_TIME ) )
   {
      scp->sc_deny_time = SC_SPECIFIED( def, A_DENY_TIME ) ? def->sc_deny_time  : 0 ;
      SC_SPECIFY( scp, A_DENY_TIME ) ;
   }

   if ( (! SC_IPV4( scp )) && (!SC_IPV6( scp )) )
   {
#ifdef INET6
      /*
       * If bind is specified, check the address and see what family is
       * available. If not, then use default.
       */
      if ( ! SC_SPECIFIED( scp, A_BIND ) ) 
         M_SET(scp->sc_xflags, SF_IPV6);
      else
      {
	  if ( SAIN6(scp->sc_bind_addr)->sin6_family == AF_INET )
             M_SET(scp->sc_xflags, SF_IPV4);
	  else
             M_SET(scp->sc_xflags, SF_IPV6);
      }
#else
      M_SET(scp->sc_xflags, SF_IPV4);
#endif
   }

   /* This should be removed if sock_stream is ever something other than TCP,
    * or sock_dgram is ever something other than UDP.
    */
   if ( (! SC_SPECIFIED( scp, A_PROTOCOL )) && 
	( SC_SPECIFIED( scp, A_SOCKET_TYPE ) ) )
   {
      struct protoent *pep ;

      if( scp->sc_socket_type == SOCK_STREAM ) {
         if( (pep = getprotobyname( "tcp" )) != NULL ) {
            scp->sc_protocol.name = new_string ( "tcp" );
            if( scp->sc_protocol.name == NULL )
               return( FAILED );
            scp->sc_protocol.value = pep->p_proto ;
            SC_SPECIFY(scp, A_PROTOCOL);
         }
      }

      if( scp->sc_socket_type == SOCK_DGRAM ) {
         if( (pep = getprotobyname( "udp" )) != NULL ) {
            scp->sc_protocol.name = new_string ( "udp" );
            if( scp->sc_protocol.name == NULL )
               return( FAILED );
            scp->sc_protocol.value = pep->p_proto ;
            SC_SPECIFY(scp, A_PROTOCOL);
         }
      }
   }
   if ( ( SC_SPECIFIED( scp, A_PROTOCOL )) && 
        (! SC_SPECIFIED( scp, A_SOCKET_TYPE ) ) )
   {
      if( (scp->sc_protocol.name != NULL) && EQ("tcp", scp->sc_protocol.name) )
      {
            scp->sc_socket_type = SOCK_STREAM;
            SC_SPECIFY(scp, A_SOCKET_TYPE);
      }

      if( (scp->sc_protocol.name != NULL) && EQ("udp", scp->sc_protocol.name) )
      {
            scp->sc_socket_type = SOCK_DGRAM;
            SC_SPECIFY(scp, A_SOCKET_TYPE);
      }
   }

   if ( USE_DEFAULT( scp, def, A_LOG_ON_SUCCESS ) )
   {
      scp->sc_log_on_success = def->sc_log_on_success ;
      SC_SPECIFY( scp, A_LOG_ON_SUCCESS ) ;
   }

   if ( USE_DEFAULT( scp, def, A_LOG_ON_FAILURE ) )
   {
      scp->sc_log_on_failure = def->sc_log_on_failure ;
      SC_SPECIFY( scp, A_LOG_ON_FAILURE ) ;
   }

   if ( USE_DEFAULT( scp, def, A_LOG_TYPE ) )
   {
      struct log *dlp = SC_LOG( def ) ;
      struct log *slp = SC_LOG( scp ) ;

      switch ( LOG_GET_TYPE( dlp ) )
      {
         case L_NONE:
            LOG_SET_TYPE( slp, L_NONE ) ;
            break ;
         
         case L_SYSLOG:
            *slp = *dlp ;
            break ;
         
         case L_FILE:
            LOG_SET_TYPE( slp, L_COMMON_FILE ) ;
            break ;

         default:
            msg( LOG_ERR, func,
                        "bad log type: %d", (int) LOG_GET_TYPE( dlp ) ) ;
            return( FAILED ) ;
      }
      SC_SPECIFY( scp, A_LOG_TYPE ) ;
   }
   if ( setup_environ( scp, def ) == FAILED )
      return( FAILED ) ;
   return( OK ) ;
}


static void remove_disabled_services( struct configuration *confp )
{
   pset_h disabled_services ;
   pset_h enabled_services ;
   struct service_config *scp ;
   struct service_config *defaults = confp->cnf_defaults ;

   if( SC_SPECIFIED( defaults, A_ENABLED ) ) {
      enabled_services = defaults->sc_enabled ;
      

      /* Mark all the services disabled */
      for ( scp = SCP( psi_start( iter ) ) ; scp ; scp = SCP( psi_next(iter) ) )
         SC_DISABLE( scp );

      /* Enable the selected services */
      for ( scp = SCP( psi_start( iter ) ) ; scp ; scp = SCP( psi_next(iter) ) )
      {
         register char *sid = SC_ID( scp ) ;
         register unsigned u ;

         for ( u = 0 ; u < pset_count( enabled_services ) ; u++ ) {
            if ( EQ( sid, (char *) pset_pointer( enabled_services, u ) ) ) {
               SC_ENABLE( scp );
               break;
            }
         }
      }
   }

   /* Remove any services that are left marked disabled */
   for ( scp = SCP( psi_start( iter ) ) ; scp ; scp = SCP( psi_next(iter)) ){
      if( SC_IS_DISABLED( scp ) ) {
         msg(LOG_DEBUG, "remove_disabled_services", "removing %s", scp->sc_name);
         SC_DISABLE( scp );
         sc_free(scp);
         psi_remove(iter);
      }
   }

   if ( ! SC_SPECIFIED( defaults, A_DISABLED ) )
      return ;
   
   disabled_services = defaults->sc_disabled ;

   for ( scp = SCP( psi_start( iter ) ) ; scp ; scp = SCP( psi_next( iter ) ) )
   {
      register char *sid = SC_ID( scp ) ;
      register unsigned u ;

      for ( u = 0 ; u < pset_count( disabled_services ) ; u++ )
         if ( EQ( sid, (char *) pset_pointer( disabled_services, u ) ) )
         {
            sc_free( scp ) ;
            psi_remove( iter ) ;
            break ;
         }
   }
}


/*
 * Check if all required attributes have been specified
 */
static status_e attr_check( struct service_config *scp )
{
   mask_t         necessary_and_specified ;
   mask_t         necessary_and_missing ;
   mask_t         must_specify = NECESSARY_ATTRS ;
   int            attr_id ;
   char          *attr_name ;
   const char    *func = "attr_check" ;

   /*
    * Determine what attributes must be specified
    */
   if ( ! SC_IS_INTERNAL( scp ) )
      M_OR( must_specify, must_specify, NECESSARY_ATTRS_EXTERNAL ) ;

   if ( SC_IS_UNLISTED( scp ) ) {
      M_OR( must_specify, must_specify, NECESSARY_ATTRS_UNLISTED ) ;
   }
   else if (!SC_IS_INTERNAL( scp )){
      M_OR( must_specify, must_specify, NECESSARY_ATTRS_LISTED ) ;
   }

   if ( SC_IS_RPC( scp ) ) {
      if ( SC_IS_UNLISTED( scp ) ) {
         M_OR( must_specify, must_specify, NECESSARY_ATTRS_RPC_UNLISTED ) ;
         M_CLEAR( must_specify, A_PORT ) ;
      } else {
         M_OR( must_specify, must_specify, NECESSARY_ATTRS_RPC_LISTED ) ;
      }
   }

   if( SC_IPV4( scp ) && SC_IPV6( scp ) ) {
      msg( LOG_ERR, func, 
         "Service %s specified as both IPv4 and IPv6 - DISABLING", 
	 SC_NAME(scp));
      return FAILED ;
   }

   if( (scp->sc_redir_addr != NULL) ) {
#define DEFAULT_SERVER "/bin/true"
       scp->sc_server = new_string( DEFAULT_SERVER );
       SC_SPECIFY(scp, A_SERVER);
   }

   /*
    * Check if all necessary attributes have been specified
    *
    * NOTE: None of the necessary attributes can belong to "defaults"
    *         This is why we use the sc_attributes_specified mask instead
    *         of the sc_attributes_present mask.
    */

   M_AND( necessary_and_specified,
                  scp->sc_specified_attributes, must_specify ) ;
   M_XOR( necessary_and_missing, necessary_and_specified, must_specify ) ;

   if ( M_ARE_ALL_CLEAR( necessary_and_missing) )
   {
      return OK ;
   }

   /*
    * Print names of missing attributes
    */
   for ( attr_id = 0 ; attr_id < SERVICE_ATTRIBUTES ; attr_id++ )
      if ( M_IS_SET( necessary_and_missing, attr_id ) && 
                  ( attr_name = attr_name_lookup( attr_id ) ) != NULL )
      {
         msg( LOG_ERR, func,
            "Service %s missing attribute %s - DISABLING", 
	    scp->sc_id, attr_name ) ;
      }
   return FAILED ;
}


/*
 * Perform validity checks on the specified entry
 *
 * Also does the following:
 *      1. If this is an internal service, it finds the function that
 *         implements it
 *      2. For RPC services, it finds the program number
 *      3. For non-RPC services, it finds the port number.
 */
static status_e check_entry( struct service_config *scp )
{
   const char *func = "check_entry" ;

   if ( attr_check( scp ) == FAILED )
      return( FAILED ) ;

   /*
    * Currently, we cannot intercept:
    *      1) internal services
    *      2) multi-threaded services
    * We clear the INTERCEPT flag without disabling the service.
    */
   if ( SC_IS_INTERCEPTED( scp ) )
   {
      if ( SC_IS_INTERNAL( scp ) )
      {
         msg( LOG_ERR, func,
            "Internal services cannot be intercepted: %s ", scp->sc_id ) ;
         M_CLEAR( scp->sc_xflags, SF_INTERCEPT ) ;
      }
      if ( scp->sc_wait == NO )
      {
         msg( LOG_ERR, func,
            "Multi-threaded services cannot be intercepted: %s", scp->sc_id ) ;
         M_CLEAR( scp->sc_xflags, SF_INTERCEPT ) ;
      }
   }
   
   /* Steer the lost sheep home */
   if ( SC_SENSOR( scp ) )
      M_SET( scp->sc_type, ST_INTERNAL );

   if ( SC_IS_INTERNAL( scp ) )
   {   /* If SENSOR flagged redirect to internal builtin function. */ 
      if ( SC_SENSOR( scp ) )
      {
	 init_sensor();
         scp->sc_builtin =
            builtin_find( "sensor", scp->sc_socket_type );
      }
      else
         scp->sc_builtin =
            builtin_find( scp->sc_name, scp->sc_socket_type );
      if (scp->sc_builtin == NULL )
         return( FAILED ) ;
   }

   /*
    * If this is an unlisted service we are done.
    * attr_check() guaranteed that we have all the information we need
    * to start the service.
    */
   if ( SC_IS_UNLISTED( scp ) )
      return( OK ) ;

/* #ifndef NO_RPC */
#if defined(HAVE_RPC_RPCENT_H) || defined(HAVE_NETDB_H)
   if ( SC_IS_RPC( scp ) )
   {
      struct rpcent *rep = (struct rpcent *)getrpcbyname( scp->sc_name ) ;

      if ( rep == NULL )
      {
         msg( LOG_ERR, func, "unknown RPC service: %s", scp->sc_name ) ;
         return( FAILED ) ;
      }
      SC_RPCDATA( scp )->rd_program_number = rep->r_number ;
   }
   else
#endif   /* ! NO_RPC */
   {
      struct servent *sep ;
      uint16_t service_port ;

      if( SC_SPECIFIED( scp, A_PROTOCOL) && SC_SPECIFIED( scp, A_PORT ) )
         return(OK);
      
      /*
       * Check if a protocol was specified.
       * If so, verify it is a proper protocol for the given service.
       * If no protocol was specified, find the protocol for the service
       * and its number.
       */
      if ( SC_SPECIFIED( scp, A_PROTOCOL ) )
      {
         sep = getservbyname( scp->sc_name, scp->sc_protocol.name ) ;
         if ( (sep == NULL) )
         {
            msg( LOG_ERR, func, 
               "service/protocol combination not in /etc/services: %s/%s",
                  scp->sc_name, scp->sc_protocol.name ) ;
            return( FAILED ) ;
         }
      }
      else
      {
         struct protoent *pep ;
         char *protoname=CHAR_NULL;

         if ( SC_SPECIFIED( scp, A_SOCKET_TYPE ) ) {
            if (SC_SOCKET_TYPE(scp) == SOCK_DGRAM) {
               protoname="udp";
            } else if ((SC_SOCKET_TYPE(scp) == SOCK_STREAM)) {
               protoname="tcp";
            }
         }

         sep = getservbyname( scp->sc_name, protoname ) ;
         if ( sep == NULL )
         {
            msg( LOG_ERR, func, "Unknown service: %s/%s", scp->sc_name, protoname ? protoname : "(any protocol)" ) ;
            return( FAILED ) ;
         }
         scp->sc_protocol.name = new_string( sep->s_proto ) ;
         if ( scp->sc_protocol.name == NULL )
         {
            out_of_memory( func ) ;
            return( FAILED ) ;
         }
         pep = getprotobyname( sep->s_proto ) ;
         if ( pep == NULL )
         {
            msg( LOG_ERR, func,
               "Protocol '%s' for service '%s' is not in /etc/protocols",
                  sep->s_proto, scp->sc_name ) ;
            return( FAILED ) ;
         }
         scp->sc_protocol.value = pep->p_proto ;
      }

      service_port = sep->s_port ;      /* s_port is in network-byte-order */

      /*
       * If a port was specified, it must be the right one
       */
      if ( SC_SPECIFIED( scp, A_PORT ) && 
                                 scp->sc_port != ntohs( service_port ) )
      {
         msg( LOG_ERR, func, "Service %s expects port %d, not %d",
                        scp->sc_name, ntohs( service_port ), scp->sc_port ) ;
         return( FAILED ) ;
      }
      scp->sc_port = ntohs( service_port ) ;
   }

   if ( SC_NAMEINARGS(scp) )
   {
      if (SC_IS_INTERNAL( scp ) )
      {
         msg( LOG_ERR, func, 
              "Service %s is INTERNAL and has NAMEINARGS flag set", 
	      scp->sc_name );
         return FAILED;
      }
      else if (!SC_SPECIFIED( scp, A_SERVER_ARGS) )
      {
         msg( LOG_ERR, func, 
              "Service %s has NAMEINARGS flag set and no server args", 
	      scp->sc_name );
         return FAILED;
      }
   }
   return( OK ) ;
}

/*
 * Get a configuration from the specified file.
 */
static status_e get_conf( int fd, struct configuration *confp )
{
   parse_conf_file( fd, confp ) ;
   parse_end() ;
   return( OK ) ;
}



#define CHECK_AND_CLEAR( scp, mask, mask_name )                               \
   if ( M_IS_SET( mask, LO_USERID ) )                                         \
   {                                                                          \
      msg( LOG_WARNING, func,                                                 \
      "%s service: clearing USERID option from %s", scp->sc_id, mask_name ) ; \
      M_CLEAR( mask, LO_USERID ) ;                                            \
   }

/*
 * Get a configuration by reading the configuration file.
 */
status_e cnf_get( struct configuration *confp )
{
   int config_fd ;
   struct service_config *scp ;
   const char *func = "get_configuration" ;

   if ( cnf_init( confp, &config_fd, &iter ) == FAILED )
      return( FAILED ) ;

   else if ( get_conf( config_fd, confp ) == FAILED )
   {
      Sclose( config_fd ) ;
      cnf_free( confp ) ;
      psi_destroy( iter ) ;
      return( FAILED ) ;
   }

   Sclose( config_fd ) ;

   remove_disabled_services( confp ) ;

   for ( scp = SCP( psi_start( iter ) ) ; scp ; scp = SCP( psi_next( iter ) ) )
   {
      /*
       * Fill the service configuration from the defaults.
       * We do this so that we don't have to look at the defaults any more.
       */
      if ( attr_fill( scp, confp->cnf_defaults ) == FAILED )
      {
         sc_free( scp ) ;
         psi_remove( iter ) ;
         continue ;
      }

      if ( check_entry( scp ) == FAILED )
      {
         sc_free( scp ) ;
         psi_remove( iter ) ;
         continue ;
      }

      /*
       * If the INTERCEPT flag is set, change this service to an internal 
       * service using the special INTERCEPT builtin.
       */
      if ( SC_IS_INTERCEPTED( scp ) )
      {
         const builtin_s *bp ;

         bp = spec_find( INTERCEPT_SERVICE_NAME, scp->sc_socket_type ) ;
         if ( bp == NULL )
         {
            msg( LOG_ERR, func, "removing service %s", SC_ID( scp ) ) ;
            sc_free( scp ) ;
            psi_remove( iter ) ;
            continue ;
         }

         scp->sc_builtin = bp ;
         M_SET( scp->sc_type, ST_INTERNAL ) ;
      }

      /*
       * Clear the USERID flag for the identity service because
       * it may lead to loops (for example, remote xinetd issues request,
       * local xinetd issues request to remote xinetd etc.)
       * We identify the identity service by its (protocol,port) combination.
       */
      if ( scp->sc_port == IDENTITY_SERVICE_PORT && 
                                       scp->sc_protocol.value == IPPROTO_TCP )
      {
         CHECK_AND_CLEAR( scp, scp->sc_log_on_success, "log_on_success" ) ;
         CHECK_AND_CLEAR( scp, scp->sc_log_on_failure, "log_on_failure" ) ;
      }
   }

   psi_destroy( iter ) ;

   if ( debug.on && debug.fd != -1 )
      cnf_dump( confp, debug.fd ) ;

   endservent() ;
   endprotoent() ;
#ifndef NO_RPC
   endrpcent() ;
#endif
   return( OK ) ;
}

