/*
 *   cdda - CD Digital Audio support
 *
 *   Copyright (C) 1993-2002  Ti Kan
 *   E-mail: xmcd@amb.org
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 *   IBM AIX ATAPI/IDE CDDA read ioctl support
 */
#ifndef LINT
static char *_aixread_c_ident_ = "@(#)aixread.c	7.9 02/05/09";
#endif

#include "common_d/appenv.h"
#include "common_d/util.h"
#include "cdda_d/cdda.h"

#if defined(CDDA_RD_AIX) && defined(CDDA_SYSVIPC)

#include <sys/ide.h>
#include <sys/idecdrom.h>
#include <sys/scdisk.h>
#include <sys/cdrom.h>
#include "cdda_d/sysvipc.h"
#include "cdda_d/sem.h"
#include "cdda_d/aixread.h"

extern appdata_t	app_data;
extern FILE		*errfp;
extern void		*cdda_shmaddr;

STATIC int		aix_rsemid,	/* Semaphores identifier */
			aix_rfd;	/* CD device file descriptor */
STATIC cd_state_t	*aix_rcd;	/* CD state pointer */
STATIC bool_t		aix_rcddamode = FALSE;
					/* Set to CDDA mode */


/*
 * aixcd_read_fillbuf
 *	Before beginning to play we fill the data buffer.
 *
 * Args:
 *	fd - device file descriptor
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
STATIC bool_t
aixcd_read_fillbuf(int fd)
{
	byte_t			*start,
				*end,
				*p;
	int			i,
				nframes;
	struct mode_form_op	m;
	struct ide_rdwrt	cdda;

	/* Get lock */
	cdda_waitsem(aix_rsemid, LOCK);

	/* Initialize */
	start = &aix_rcd->cdb->data[aix_rcd->cds->olap_bytes];

	/* Set CDDA mode */
	(void) memset(&m, 0, sizeof(m));
	m.action = CD_CHG_MODE;
	m.cd_mode_form = CD_DA;
	if (ioctl(fd, DK_CD_MODE, &m) < 0) {
		(void) sprintf(aix_rcd->i->msgbuf,
			"aixcd_read_fillbuf: DK_CD_MODE ioctl failed: %s",
			strerror(errno));
		DBGPRN(DBG_DEVIO)(errfp, "%s\n", aix_rcd->i->msgbuf);
		return FALSE;
	}

	DBGPRN(DBG_DEVIO)(errfp,
		"\nSent DK_CD_MODE ioctl: mode=CD_CHG_MODE\n");

	aix_rcddamode = TRUE;

	/* Set up for read */
	(void) memset(&cdda, 0, sizeof(cdda));
	nframes = aix_rcd->cds->chunk_blocks + aix_rcd->cds->olap_blocks;
	cdda.logical_blk_addr = aix_rcd->i->start_lba;
	cdda.data_length = CDDA_BLKSZ * nframes;
	cdda.buffer = (char *) start;
	cdda.timeout_value = 5;

	cdda_rd_setcurtrk(aix_rcd, (int) cdda.logical_blk_addr);

	/* Fill up our buffer */
	i = 0;

	while (i < aix_rcd->cds->buffer_chunks && !aix_rcd->i->cdda_done) {
		/*
		 * If we are passing the end of the play segment we have to
		 * take special action. We read in up to end_lba and zero out
		 * the buffer. This is equivalent to rounding to the nearest
		 * aix_rcd->cds->chunk_bytes. We also set the cdda_done flag.
		 */
		if ((cdda.logical_blk_addr + aix_rcd->cds->chunk_blocks) >=
		    aix_rcd->i->end_lba) {
			nframes = aix_rcd->i->end_lba - cdda.logical_blk_addr;
			cdda.data_length = CDDA_BLKSZ * nframes;
			(void) memset(
				aix_rcd->cdb->data,
				0,
				aix_rcd->cds->chunk_bytes +
				(aix_rcd->cds->olap_bytes << 1)
			);
			aix_rcd->i->cdda_done = 1;
		}

		/* Read audio from cd using ioctl method */
		if (ioctl(fd, CDIORDSE, &cdda) < 0) {
			(void) sprintf(aix_rcd->i->msgbuf,
					"aixcd_read_fillbuf: CDIORDSE ioctl "
					"failed: %s",
					strerror(errno));
			DBGPRN(DBG_DEVIO)(errfp, "%s\n", aix_rcd->i->msgbuf);
			cdda_postsem(aix_rsemid, LOCK);
			return FALSE;
		}

		DBGPRN(DBG_DEVIO)(errfp,
			"\nSent CDIORDSE ioctl: blk=%d len=%d\n",
			(int) cdda.logical_blk_addr,
			(int) cdda.data_length);

		/* Do jitter correction */
		if (i == 0)
			p = &aix_rcd->cdb->data[aix_rcd->cds->olap_bytes];
		else
			p = cdda_rd_corrjitter(aix_rcd);

		/* Data end */
		end = p + aix_rcd->cds->chunk_bytes;

		/* Reinitialize cd_olap */
		if (aix_rcd->i->jitter) {
			(void) memcpy(
				aix_rcd->cdb->olap,
				end - aix_rcd->cds->search_bytes,
				aix_rcd->cds->search_bytes << 1
			);
		}

		/* Copy in */
		(void) memcpy(
			&aix_rcd->cdb->b[
			    aix_rcd->cds->chunk_bytes * aix_rcd->cdb->nextin
			],
			p,
			aix_rcd->cds->chunk_bytes
		);

		/* Update pointers */
		aix_rcd->cdb->nextin++;
		aix_rcd->cdb->nextin %= aix_rcd->cds->buffer_chunks;
		aix_rcd->cdb->occupied++;

		cdda.logical_blk_addr += ((end - start) / CDDA_BLKSZ);
		/* Check if we need to round up */
		if (((end - start) % CDDA_BLKSZ) >= (CDDA_BLKSZ >> 1))
			cdda.logical_blk_addr++;

		cdda_rd_setcurtrk(aix_rcd, (int) cdda.logical_blk_addr);

		i++;

		/* Update debug level */
		app_data.debug = aix_rcd->i->debug;
	}

	/* No room available now */
	cdda_waitsem(aix_rsemid, ROOM);

	/* Signal data available */
	cdda_postsem(aix_rsemid, DATA);

	/* Release lock */
	cdda_postsem(aix_rsemid, LOCK);

	return TRUE;
}


/*
 * aixcd_read_cont
 *	Function responsible for reading from the cd,
 *	correcting for jitter and writing to the buffer.
 *
 * Args:
 *	fd - device file descriptor
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
STATIC bool_t
aixcd_read_cont(int fd)
{
	byte_t			*start,
				*end,
				*p;
	int			nframes;
	struct ide_rdwrt	cdda;

	/* Initialize */
	start = &aix_rcd->cdb->data[aix_rcd->cds->olap_bytes];

	/* Set up for read */
	(void) memset(&cdda, 0, sizeof(cdda));
	nframes = aix_rcd->cds->chunk_blocks + aix_rcd->cds->olap_blocks;
	cdda.logical_blk_addr = aix_rcd->i->start_lba;
	cdda.data_length = CDDA_BLKSZ * nframes;
	cdda.buffer = (char *) start;
	cdda.timeout_value = 5;

	cdda_rd_setcurtrk(aix_rcd, (int) cdda.logical_blk_addr);

	/* While not stopped or finished */
	while (aix_rcd->i->state != CDSTAT_COMPLETED &&
	       !aix_rcd->i->cdda_done) {
		/*
		 * If we are passing the end of the play segment we have to
		 * take special action. We read in up to end_lba and zero out
		 * the buffer. This is equivalent to rounding to the nearest
		 * aix_rcd->cds->chunk_bytes. We also set the cdda_done flag.
		 */
		if ((cdda.logical_blk_addr + aix_rcd->cds->chunk_blocks) >=
		    aix_rcd->i->end_lba) {
			nframes = aix_rcd->i->end_lba - cdda.logical_blk_addr;
			cdda.data_length = CDDA_BLKSZ * nframes;
			(void) memset(
				aix_rcd->cdb->data,
				0,
				aix_rcd->cds->chunk_bytes +
				(aix_rcd->cds->olap_bytes << 1)
			);
			aix_rcd->i->cdda_done = 1;
		}

		/* Read audio from cd using ioctl method */
		if (ioctl(fd, CDIORDSE, &cdda) < 0) {
			(void) sprintf(aix_rcd->i->msgbuf,
					"aixcd_read_cont: CDIORDSE ioctl "
					"failed: %s",
					strerror(errno));
			DBGPRN(DBG_DEVIO)(errfp, "%s\n", aix_rcd->i->msgbuf);
			return FALSE;
		}

		DBGPRN(DBG_DEVIO)(errfp,
			"\nSent CDIORDSE ioctl: blk=%d len=%d\n",
			(int) cdda.logical_blk_addr,
			(int) cdda.data_length);

		/* Do jitter correction */
		p = cdda_rd_corrjitter(aix_rcd);

		/* Data end */
		end = p + aix_rcd->cds->chunk_bytes;

		/* Reinitialize cd_olap */
		if (aix_rcd->i->jitter) {
			(void) memcpy(
				aix_rcd->cdb->olap,
				end - aix_rcd->cds->search_bytes,
				aix_rcd->cds->search_bytes << 1
			);
		}

		/* Get lock */
		cdda_waitsem(aix_rsemid, LOCK);

		/* Wait until there is room */
		while (aix_rcd->cdb->occupied >= aix_rcd->cds->buffer_chunks &&
		       aix_rcd->i->state != CDSTAT_COMPLETED) {
			cdda_postsem(aix_rsemid, LOCK);
			cdda_waitsem(aix_rsemid, ROOM);
			cdda_waitsem(aix_rsemid, LOCK);
		}

		/* Break if completed */
		if (aix_rcd->i->state == CDSTAT_COMPLETED) {
			cdda_postsem(aix_rsemid, LOCK);
			break;
		}

		/* Copy to free location */
		(void) memcpy(
			&aix_rcd->cdb->b[
			    aix_rcd->cds->chunk_bytes * aix_rcd->cdb->nextin
			],
			p,
			aix_rcd->cds->chunk_bytes
		);

		/* Update pointers */
		aix_rcd->cdb->nextin++;
		aix_rcd->cdb->nextin %= aix_rcd->cds->buffer_chunks;
		aix_rcd->cdb->occupied++;

		cdda.logical_blk_addr += ((end - start) / CDDA_BLKSZ);
		/* Check if we need to round up */
		if (((end - start) % CDDA_BLKSZ) >= (CDDA_BLKSZ >> 1))
			cdda.logical_blk_addr++;

		cdda_rd_setcurtrk(aix_rcd, (int) cdda.logical_blk_addr);

		/* Signal data available */
		cdda_postsem(aix_rsemid, DATA);

		/* Update debug level */
		app_data.debug = aix_rcd->i->debug;

		/* Release lock */
		cdda_postsem(aix_rsemid, LOCK);
	}

	/* Wake up writer */
	cdda_postsem(aix_rsemid, DATA);

	return TRUE;
}


/*
 * aixcd_cleanup
 *	Detaches shared memory.
 *
 * Args:
 *	killwriter - whether to kill the writer process
 *
 * Return:
 *	Nothing.
 */
STATIC void
aixcd_cleanup(bool_t killwriter)
{
	struct mode_form_op	m;

	DBGPRN(DBG_DEVIO)(errfp,
			  "\naixcd_cleanup: Cleaning up reader pid=%d\n",
			  (int) getpid());

	if (aix_rcddamode) {
		/* Restore CDROM mode */
		(void) memset(&m, 0, sizeof(m));
		m.action = CD_CHG_MODE;
		m.cd_mode_form = CD_MODE1;
		if (ioctl(aix_rfd, DK_CD_MODE, &m) < 0) {
			DBGPRN(DBG_DEVIO)(errfp,
				"aixcd_read: DK_CD_MODE ioctl failed: %s\n",
				strerror(errno));
		}

		DBGPRN(DBG_DEVIO)(errfp,
			"\nSent DK_CD_MODE ioctl: mode=CD_MODE1\n");

		aix_rcddamode = FALSE;
	}

	(void) sleep(1);

	if (aix_rcd != NULL) {
		if (killwriter && aix_rcd->i != NULL &&
		    aix_rcd->i->writer > 0) {
			(void) kill(aix_rcd->i->writer, SIGTERM);
			aix_rcd->i->writer = 0;
			aix_rcd->i->state = CDSTAT_COMPLETED;
		}

		MEM_FREE(aix_rcd);
		aix_rcd = NULL;
	}
}


/*
 * aixcd_onterm
 *	SIGTERM signal handler - gracefully exit
 *
 * Args:
 *	None.
 *
 * Return:
 *	Nothing.
 */
/*ARGSUSED*/
STATIC void
aixcd_onterm(int signo)
{
	aixcd_cleanup(FALSE);
	exit(1);
}


/*
 * aix_readinit
 *	Pre-playback support check function
 *
 * Args:
 *	None.
 *
 * Return:
 *	Bitmask of supported features
 */
word32_t
aix_readinit(void)
{
	return CDDA_READAUDIO;
}


/*
 * aix_read
 *	Attaches to shared memory and semaphores. Continuously reads
 *	data from the cd and places into shared memory.
 *
 * Args:
 *	fd - device file descriptor
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
bool_t
aix_read(int fd)
{
	struct sigaction	action_term;

	/* Check configuration */
	if (app_data.di_method != 3 /* DI_AIXIOC */) {
		(void) fprintf(errfp,
			"aix_read: Inconsistent deviceInterfaceMethod "
			"and cddaReadMethod parameters.  Aborting.\n");
		return FALSE;
	}

	/* Save file descriptor */
	aix_rfd = fd;

	/* Set some signal behaviors */
	(void) signal(SIGINT, SIG_IGN);
	(void) signal(SIGQUIT, SIG_IGN);

	action_term.sa_handler = aixcd_onterm;
	sigemptyset(&action_term.sa_mask);
	action_term.sa_flags = 0;
	if (sigaction(SIGTERM, &action_term, NULL) < 0) {
		(void) fprintf(errfp,
				"aix_read: sigaction failed (SIGTERM): %s\n",
				strerror(errno));
		return FALSE;
	}

	/* Allocate mempry */
	aix_rcd = (cd_state_t *) MEM_ALLOC("cd_state_t", sizeof(cd_state_t));
	if (aix_rcd == NULL) {
		(void) fprintf(errfp, "aix_read: out of memory\n");
		return FALSE;
	}

	(void) memset(aix_rcd, 0, sizeof(cd_state_t));

	/* Initialize cd pointers to point into shared memory */
	if ((aix_rsemid = cdda_rw_initipc(aix_rcd)) < 0) {
		aixcd_cleanup(TRUE);
		return FALSE;
	}

	DBGPRN(DBG_DEVIO)(errfp,
			  "\naixcd_read: Reading CDDA: "
			  "chunk_blks=%d, olap_blks=%d\n",
			  aix_rcd->cds->chunk_blocks,
			  aix_rcd->cds->olap_blocks);

	/* Fill data buffer */
	if (!aixcd_read_fillbuf(fd)) {
		aixcd_cleanup(TRUE);
		return FALSE;
	}

	/* Keep reading */
	if (!aixcd_read_cont(fd)) {
		aixcd_cleanup(TRUE);
		return FALSE;
	}

	aixcd_cleanup(FALSE);
	return TRUE;
}

#endif	/* CDDA_RD_AIX CDDA_SYSVIPC */

