/*
 *   cdda - CD Digital Audio support
 *
 *   Copyright (C) 1993-2002  Ti Kan
 *   E-mail: xmcd@amb.org
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#ifndef LINT
static char *_cdda_c_ident_ = "@(#)cdda.c	7.41 02/04/24";
#endif

#include "common_d/appenv.h"
#include "common_d/util.h"
#include "cdda_d/cdda.h"
#include "cdda_d/sysvipc.h"


extern appdata_t	app_data;
extern FILE		*errfp;

cdda_client_t		*cdda_clinfo;		/* Client information struct */

STATIC bool_t		cdda_initted = FALSE;	/* Initialization done flag */


/*
 * Cdda method call table
 * This array must correspond to the CDDA_xxx definitions found in the
 * cdda.h file, and the index is specified by the cddaMethod parameter.
 */
struct {
	word32_t (*capab)(void);
	bool_t	 (*init)(curstat_t *);
	void	 (*halt)(int, curstat_t *);
	bool_t	 (*play)(int, curstat_t *, int, int);
	bool_t	 (*pause_resume)(int, curstat_t *, bool_t);
	bool_t	 (*stop)(int, curstat_t *);
	int	 (*vol)(int, curstat_t *, int, bool_t);
	bool_t	 (*chroute)(int, curstat_t *);
	bool_t	 (*getstatus)(int, curstat_t *, cdstat_t *);
	void	 (*debug)(word32_t);
} cdda_calltbl[] = {
	{ NULL, NULL, NULL, NULL, NULL,
	  NULL, NULL, NULL, NULL, NULL },	/* CDDA_NONE */
	{ cdda_sysvipc_capab,
	  cdda_sysvipc_init,
	  cdda_sysvipc_halt,
	  cdda_sysvipc_play,
	  cdda_sysvipc_pause_resume,
	  cdda_sysvipc_stop,
	  cdda_sysvipc_vol,
	  cdda_sysvipc_chroute,
	  cdda_sysvipc_getstatus,
	  cdda_sysvipc_debug }			/* CDDA_SYSVIPC */
};


/**********************
 *  Public functions  *
 **********************/


/*
 * cdda_capab
 *	Query configured CDDA subsystem capabilities
 *
 * Args:
 *	None.
 *
 * Return:
 *	Bitmask of supported features
 */
word32_t
cdda_capab(void)
{
	word32_t	ret,
			(*func)(void);

	if (app_data.cdda_method == CDDA_NONE)
		return 0;

	if (app_data.cdda_method < CDDA_NONE ||
	    app_data.cdda_method >= CDDA_METHODS) {
		CDDA_WARNING(app_data.str_cddainit_fail);
		DBGPRN(DBG_DEVIO)(errfp, "Warning: %s\n",
				  app_data.str_cddainit_fail);
		return 0;
	}

	func = cdda_calltbl[app_data.cdda_method].capab;
	if (func != NULL)
		ret = (*func)();
	else {
		CDDA_WARNING(app_data.str_cddainit_fail);
		DBGPRN(DBG_DEVIO)(errfp, "Warning: %s\n",
				  app_data.str_cddainit_fail);
		ret = 0;
	}

	return (ret);
}


/*
 * cdda_init
 *	Initialize the CDDA subsystem.
 *
 * Args:
 *	s   - Pointer to the curstat_t structure.
 *	clp - Pointer to cdda subsystem client registration info structure.
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
bool_t
cdda_init(curstat_t *s, cdda_client_t *clp)
{
	bool_t	ret,
		(*func)(curstat_t *);

	if (cdda_initted)
		return TRUE;

	if (app_data.cdda_method == CDDA_NONE)
		return FALSE;

	cdda_clinfo = clp;

	if (app_data.cdda_method < CDDA_NONE ||
	    app_data.cdda_method >= CDDA_METHODS) {
		CDDA_WARNING(app_data.str_cddainit_fail);
		DBGPRN(DBG_DEVIO)(errfp, "Warning: %s\n",
				  app_data.str_cddainit_fail);
		return FALSE;
	}

	func = cdda_calltbl[app_data.cdda_method].init;
	if (func != NULL)
		ret = (*func)(s);
	else {
		CDDA_WARNING(app_data.str_cddainit_fail);
		DBGPRN(DBG_DEVIO)(errfp, "Warning: %s\n",
				  app_data.str_cddainit_fail);
		ret = FALSE;
	}

	cdda_initted = ret;
	return ret;
}


/*
 * cdda_halt
 *	Shuts down the CDDA subsystem.
 *
 * Args:
 *	fd - Device file descriptor
 *	s  - Pointer to the curstat_t structure
 *
 * Return:
 *	Nothing.
 */
void
cdda_halt(int fd, curstat_t *s)
{
	void	(*func)(int, curstat_t *);

	if (!cdda_initted)
		return;

	func = cdda_calltbl[app_data.cdda_method].halt;
	if (func != NULL)
		(*func)(fd, s);

	cdda_initted = FALSE;
}


/*
 * cdda_play
 *	Start playing.  No error checking is done on the output file path
 *	or pipe program strings here.  The caller should pre-check these.
 *
 * Args:
 *	fd - Device file descriptor
 *	s  - Pointer to the curstat_t structure
 *	start_lba - Start logical block address
 *	end_lba   - End logical block address
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
bool_t
cdda_play(int fd, curstat_t *s, int start_lba, int end_lba)
{
	bool_t	(*func)(int, curstat_t *, int, int);

	if (!cdda_initted)
		return FALSE;

	func = cdda_calltbl[app_data.cdda_method].play;
	if (func != NULL)
		return (*func)(fd, s, start_lba, end_lba);

	return FALSE;
}


/*
 * cdda_pause_resume
 *	Pause/resume playback
 *
 * Args:
 *	fd     - Device file descriptor
 *	s      - Pointer to the curstat_t structure
 *	resume - Whether to resume playback
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
bool_t
cdda_pause_resume(int fd, curstat_t *s, bool_t resume)
{
	bool_t	(*func)(int, curstat_t *, bool_t);

	if (!cdda_initted)
		return FALSE;

	func = cdda_calltbl[app_data.cdda_method].pause_resume;
	if (func != NULL)
		return (*func)(fd, s, resume);

	return FALSE;
}


/*
 * cdda_stop
 *	Stop playback
 *
 * Args:
 *	fd - Device file descriptor
 *	s  - Pointer to the curstat_t structure
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
bool_t
cdda_stop(int fd, curstat_t *s)
{
	bool_t	(*func)(int, curstat_t *);

	if (!cdda_initted)
		return FALSE;

	func = cdda_calltbl[app_data.cdda_method].stop;
	if (func != NULL)
		return (*func)(fd, s);

	return FALSE;
}


/*
 * cdda_vol
 *	Change volume setting
 *
 * Args:
 *	fd    - Device file descriptor
 *	s     - Pointer to the curstat_t structure
 *	vol   - Desired volume level
 *	query - Whether querying or setting the volume
 *
 * Return:
 *	The volume level
 */
int
cdda_vol(int fd, curstat_t *s, int vol, bool_t query)
{
	int	(*func)(int, curstat_t *, int, bool_t);

	if (!cdda_initted)
		return FALSE;

	func = cdda_calltbl[app_data.cdda_method].vol;
	if (func != NULL)
		return (*func)(fd, s, vol, query);

	return -1;
}


/*
 * cdda_chroute
 *	Change channel routing setting
 *
 * Args:
 *	fd - Device file descriptor
 *	s  - Pointer to the curstat_t structure
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
bool_t
cdda_chroute(int fd, curstat_t *s)
{
	bool_t	(*func)(int, curstat_t *);

	if (!cdda_initted)
		return FALSE;

	func = cdda_calltbl[app_data.cdda_method].chroute;
	if (func != NULL)
		return (*func)(fd, s);

	return FALSE;
}


/*
 * cdda_getstatus
 *	Get playback status
 *
 * Args:
 *	fd - Device file descriptor
 *	s  - Pointer to the curstat_t structure
 *	sp - cdstat_t return info structure
 *
 * Return:
 *	FALSE - failure
 *	TRUE  - success
 */
bool_t
cdda_getstatus(int fd, curstat_t *s, cdstat_t *sp)
{
	bool_t	(*func)(int, curstat_t *, cdstat_t *);

	if (!cdda_initted)
		return FALSE;

	func = cdda_calltbl[app_data.cdda_method].getstatus;
	if (func != NULL)
		return (*func)(fd, s, sp);

	return FALSE;
}


/*
 * cdda_debug
 *	Debug level change notification function
 *
 * Args:
 *	lev - The new debug level
 *
 * Return:
 *	Nothing.
 */
void
cdda_debug(word32_t lev)
{
	void	(*func)(word32_t);

	if (!cdda_initted)
		return;

	func = cdda_calltbl[app_data.cdda_method].debug;
	if (func != NULL)
		(*func)(lev);
}


