/*
 *   cdda - CD Digital Audio support
 *
 *   Copyright (C) 1993-2002  Ti Kan
 *   E-mail: xmcd@amb.org
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#ifndef __CDDA_H__
#define __CDDA_H__

#ifndef LINT
static char *_cdda_h_ident_ = "@(#)cdda.h	7.52 02/04/06";
#endif


/* CDDA methods */
#define CDDA_METHODS	2	/* Total number of CDDA methods */

#define CDDA_NONE	0	/* Don't use CDDA method */
#if defined(SVR4) || defined(SYSV) || defined(linux) || defined(sun) || \
    defined(_AIX) || defined(sgi) || defined(__hpux) || \
    defined(__FreeBSD__) || defined(__OpenBSD__) || defined(__NetBSD__) || \
    defined(__bsdi__) || (defined(__osf__) && defined(__alpha))
#define CDDA_SYSVIPC	1	/* SYSV IPC method */
#endif


/* CDDA read methods */
#define CDDA_RD_METHODS	6	/* Total number of CDDA read methods */

#define CDDA_RD_NONE	0	/* Don't use CDDA read method */
#define CDDA_RD_SCSIPT	1	/* SCSI pass-through method */
#if defined(SVR4) && defined(sun)
#define CDDA_RD_SOL	2	/* Sun Solaris ioctl method */
#endif
#ifdef linux
#define CDDA_RD_LX	3	/* Linux ioctl method */
#endif
#if defined(__FreeBSD__) || defined(__OpenBSD__) || defined(__NetBSD__)
#define CDDA_RD_FB	4	/* FreeBSD/OpenBSD/NetBSD ioctl method */
#endif
#if defined(_AIX) && defined(AIX_IDE)
#define CDDA_RD_AIX	5	/* AIX ATAPI/IDE ioctl method */
#endif


/* CDDA write methods */
#define CDDA_WR_METHODS	6	/* Total number of CDDA write methods */

#define CDDA_WR_NONE	0	/* Don't use CDDA write method */
#define CDDA_WR_OSS	1	/* OSS & Linux sound driver method */
#if defined(SVR4) && defined(sun)
#define CDDA_WR_SOL	2	/* Sun Solaris audio driver method */
#endif
#ifdef sgi
#define CDDA_WR_IRIX	3	/* SGI Irix audio driver method */
				/* PLACE HOLDER: not yet implemented */
#endif
#ifdef __hpux
#define CDDA_WR_HPUX	4	/* HP-UX audio driver method */
				/* PLACE HOLDER: not yet implemented */
#endif
#define CDDA_WR_FILE	5	/* File-only method: No audio playback */


/* Basic CDDA read unit */
#define	CDDA_BLKSZ		2352

/* We define the default CDDA read size to be 75 blocks
 * (equating to 1 sec of audio data). We also define the
 * upper and lower bounds on this value.  Note that different
 * platforms' CDROM drivers may impose different limits which
 * require that this be externally tuned via the cddaReadChunkBlocks
 * parameter.
 */
#define	DEF_CDDA_CHUNK_BLKS	8			/* 8 blocks */
#define MIN_CDDA_CHUNK_BLKS	1			/* 1 block */
#define MAX_CDDA_CHUNK_BLKS	(3 * FRAME_PER_SEC)	/* 3 seconds */

/* CDDA data is queued up in memory. Here we define (in
 * seconds) the amount of queued data. This is not a
 * modifiable resource.
 */
#define	DEF_CDDA_BUFFER_SECS	5

/* During jitter correction we search through DEFAULT_SEARCH_BLOCKS
 * for a match.  This is not a modifiable resource.
 */
#define	DEF_CDDA_SRCH_BLKS	2

/* In order to perform jitter correction we need to
 * read more CDDA data than we "need". Below we define
 * the amount of extra data required. So each CDDA
 * read will comprise CHUNK_BLOCKS + OLAP_BLOCKS.
 */
#define	DEF_CDDA_OLAP_BLKS	(DEF_CDDA_SRCH_BLKS * 2)

/* We have found a match when DEFAULT_STR_LENGTH bytes are
 * identical in the data and the overlap.
 */
#define	DEF_CDDA_STR_LEN	20

/* Maximum number of interrupts before system call fails */
#define	CDDA_INTR_MAX		200

/* CDDA supported features bitmask values, returned by cdda_capab()
 * These should match the similar CAPAB_xxx defines in libdi_d/libdi.h
 */
#define CDDA_READAUDIO		0x02
#define CDDA_WRITEDEV		0x10
#define CDDA_WRITEFILE		0x20
#define CDDA_WRITEPIPE		0x40


/* Buffer for audio data transfer */
typedef struct cd_buffer {
	unsigned int	occupied;
	unsigned int	nextin;
	unsigned int	nextout;
	byte_t		*olap;
	byte_t		*data;
	byte_t		*b;
} cd_buffer_t;


/* Various sizes */
typedef struct cd_size {
	/* Chunk */
	unsigned int	chunk_blocks;
	unsigned int	chunk_bytes;

	/* Buffer */
	unsigned int	buffer_chunks;
	unsigned int	buffer_blocks;
	unsigned int	buffer_bytes;

	/* Overlap */
	unsigned int	olap_blocks;
	unsigned int	olap_bytes;

	/* Searching */
	unsigned int	search_blocks;
	unsigned int	search_bytes;
	unsigned int	str_length;

	/* Overall size */
	size_t		size;
} cd_size_t;


/* CD info */
typedef struct cd_info {
	int		state;			/* CDSTAT_PLAYING, etc. */
	int		chroute;		/* CHROUTE_NORMAL etc. */
	int		vol_taper;		/* Volume control taper */
	int		vol;			/* Current volume level */
	int		vol_left;		/* Current left vol % */
	int		vol_right;		/* Current right vol % */
	int		jitter;			/* 1 = on, 0 = off */
	int		cdda_done;		/* 1 = yes, 0 = no */
	word32_t	debug;			/* Debug level */

	unsigned int	start_lba;		/* Start position */
	unsigned int	end_lba;		/* End position */
	unsigned int	frm_played;		/* Frames written */
	unsigned int	trk_idx;		/* Current track index */
	unsigned int	trk_len;		/* Track length in bytes */

	pid_t		reader;			/* Reader process id */
	pid_t		writer;			/* Writer process id */

	char		msgbuf[ERR_BUF_SZ];	/* Message buffer */
} cd_info_t;


/* CD state structure - pointers to the size, info and buffer areas */
typedef struct cd_state {
	cd_size_t	*cds;			/* Sizes */
	cd_info_t	*i;			/* Info */
	cd_buffer_t	*cdb;			/* Data buffer */
} cd_state_t ;


/* Message dialog macros */
#define CDDA_FATAL(msg)		{					\
	if (cdda_clinfo != NULL && cdda_clinfo->fatal_msg != NULL)	\
		cdda_clinfo->fatal_msg(app_data.str_fatal, (msg));	\
	else {								\
		(void) fprintf(errfp, "Fatal: %s\n", (msg));		\
		exit(1);						\
	}								\
}
#define CDDA_WARNING(msg)	{					\
	if (cdda_clinfo != NULL && cdda_clinfo->warning_msg != NULL)	\
		cdda_clinfo->warning_msg(app_data.str_warning, (msg));	\
	else								\
		(void) fprintf(errfp, "Warning: %s\n", (msg));		\
}
#define CDDA_INFO(msg)		{					\
	if (cdda_clinfo != NULL && cdda_clinfo->info_msg != NULL)	\
		cdda_clinfo->info_msg(app_data.str_info, (msg));	\
	else								\
		(void) fprintf(errfp, "Info: %s\n", (msg));		\
}


/* Client registration structure */
typedef struct {
	curstat_t *	(*curstat_addr)(void);
	void		(*fatal_msg)(char *, char *);
	void		(*warning_msg)(char *, char *);
	void		(*info_msg)(char *, char *);
} cdda_client_t;




/* Function prototypes */
extern word32_t	cdda_capab(void);
extern bool_t	cdda_init(curstat_t *, cdda_client_t *);
extern void	cdda_halt(int, curstat_t *);
extern bool_t	cdda_play(int, curstat_t *, int, int);
extern bool_t	cdda_pause_resume(int, curstat_t *, bool_t);
extern bool_t	cdda_stop(int, curstat_t *);
extern int	cdda_vol(int, curstat_t *, int, bool_t);
extern bool_t	cdda_chroute(int, curstat_t *);
extern bool_t	cdda_getstatus(int, curstat_t *, cdstat_t *);
extern void	cdda_debug(word32_t);

#endif	/* __CDDA_H__ */

